/* valatrystatement.vala
 *
 * Copyright (C) 2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valatrystatement.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valablock.h>
#include <vala/valacatchclause.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>

struct _ValaTryStatementPrivate {
	ValaBlock* _body;
	ValaBlock* _finally_body;
	GeeList* catch_clauses;
};
#define VALA_TRY_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_TRY_STATEMENT, ValaTryStatementPrivate))
enum  {
	VALA_TRY_STATEMENT_DUMMY_PROPERTY,
	VALA_TRY_STATEMENT_BODY,
	VALA_TRY_STATEMENT_FINALLY_BODY
};
static void vala_try_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_try_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_try_statement_parent_class = NULL;
static ValaStatementIface* vala_try_statement_vala_statement_parent_iface = NULL;
static void vala_try_statement_dispose (GObject * obj);


/**
 * Creates a new try statement.
 *
 * @param body             body of the try statement
 * @param finally_body     body of the optional finally clause
 * @param source_reference reference to source code
 * @return                 newly created try statement
 */
ValaTryStatement* vala_try_statement_new (ValaBlock* body, ValaBlock* finally_body, ValaSourceReference* source_reference) {
	ValaTryStatement * self;
	g_return_val_if_fail (VALA_IS_BLOCK (body), NULL);
	g_return_val_if_fail (finally_body == NULL || VALA_IS_BLOCK (finally_body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_TRY_STATEMENT, 0, NULL);
	vala_try_statement_set_body (self, body);
	vala_try_statement_set_finally_body (self, finally_body);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends the specified clause to the list of catch clauses.
 *
 * @param clause a catch clause
 */
void vala_try_statement_add_catch_clause (ValaTryStatement* self, ValaCatchClause* clause) {
	g_return_if_fail (VALA_IS_TRY_STATEMENT (self));
	g_return_if_fail (VALA_IS_CATCH_CLAUSE (clause));
	gee_collection_add (GEE_COLLECTION (self->priv->catch_clauses), clause);
}


/**
 * Returns a copy of the list of catch clauses.
 *
 * @return list of catch clauses
 */
GeeCollection* vala_try_statement_get_catch_clauses (ValaTryStatement* self) {
	g_return_val_if_fail (VALA_IS_TRY_STATEMENT (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_CATCH_CLAUSE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->catch_clauses)));
}


static void vala_try_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaTryStatement * self;
	self = VALA_TRY_STATEMENT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_try_statement (visitor, self);
}


static void vala_try_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaTryStatement * self;
	self = VALA_TRY_STATEMENT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_try_statement_get_body (self)), visitor);
	{
		GeeList* clause_collection;
		GeeIterator* clause_it;
		clause_collection = self->priv->catch_clauses;
		clause_it = gee_iterable_iterator (GEE_ITERABLE (clause_collection));
		while (gee_iterator_next (clause_it)) {
			ValaCatchClause* clause;
			clause = gee_iterator_get (clause_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (clause), visitor);
				(clause == NULL ? NULL : (clause = (g_object_unref (clause), NULL)));
			}
		}
		(clause_it == NULL ? NULL : (clause_it = (g_object_unref (clause_it), NULL)));
	}
	if (vala_try_statement_get_finally_body (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_try_statement_get_finally_body (self)), visitor);
	}
}


ValaBlock* vala_try_statement_get_body (ValaTryStatement* self) {
	g_return_val_if_fail (VALA_IS_TRY_STATEMENT (self), NULL);
	return self->priv->_body;
}


void vala_try_statement_set_body (ValaTryStatement* self, ValaBlock* value) {
	ValaBlock* _tmp1;
	g_return_if_fail (VALA_IS_TRY_STATEMENT (self));
	_tmp1 = NULL;
	self->priv->_body = (_tmp1 = g_object_ref (value), (self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL))), _tmp1);
}


ValaBlock* vala_try_statement_get_finally_body (ValaTryStatement* self) {
	g_return_val_if_fail (VALA_IS_TRY_STATEMENT (self), NULL);
	return self->priv->_finally_body;
}


void vala_try_statement_set_finally_body (ValaTryStatement* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (VALA_IS_TRY_STATEMENT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_finally_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_finally_body == NULL ? NULL : (self->priv->_finally_body = (g_object_unref (self->priv->_finally_body), NULL))), _tmp2);
}


static void vala_try_statement_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaTryStatement * self;
	self = VALA_TRY_STATEMENT (object);
	switch (property_id) {
		case VALA_TRY_STATEMENT_BODY:
		g_value_set_object (value, vala_try_statement_get_body (self));
		break;
		case VALA_TRY_STATEMENT_FINALLY_BODY:
		g_value_set_object (value, vala_try_statement_get_finally_body (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_try_statement_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaTryStatement * self;
	self = VALA_TRY_STATEMENT (object);
	switch (property_id) {
		case VALA_TRY_STATEMENT_BODY:
		vala_try_statement_set_body (self, g_value_get_object (value));
		break;
		case VALA_TRY_STATEMENT_FINALLY_BODY:
		vala_try_statement_set_finally_body (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_try_statement_class_init (ValaTryStatementClass * klass) {
	vala_try_statement_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaTryStatementPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_try_statement_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_try_statement_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_try_statement_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_try_statement_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_try_statement_real_accept_children;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TRY_STATEMENT_BODY, g_param_spec_object ("body", "body", "body", VALA_TYPE_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TRY_STATEMENT_FINALLY_BODY, g_param_spec_object ("finally-body", "finally-body", "finally-body", VALA_TYPE_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_try_statement_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_try_statement_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_try_statement_init (ValaTryStatement * self) {
	self->priv = VALA_TRY_STATEMENT_GET_PRIVATE (self);
	self->priv->catch_clauses = GEE_LIST (gee_array_list_new (VALA_TYPE_CATCH_CLAUSE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
}


static void vala_try_statement_dispose (GObject * obj) {
	ValaTryStatement * self;
	self = VALA_TRY_STATEMENT (obj);
	(self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL)));
	(self->priv->_finally_body == NULL ? NULL : (self->priv->_finally_body = (g_object_unref (self->priv->_finally_body), NULL)));
	(self->priv->catch_clauses == NULL ? NULL : (self->priv->catch_clauses = (g_object_unref (self->priv->catch_clauses), NULL)));
	G_OBJECT_CLASS (vala_try_statement_parent_class)->dispose (obj);
}


GType vala_try_statement_get_type (void) {
	static GType vala_try_statement_type_id = 0;
	if (G_UNLIKELY (vala_try_statement_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaTryStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_try_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaTryStatement), 0, (GInstanceInitFunc) vala_try_statement_init };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_try_statement_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_try_statement_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaTryStatement", &g_define_type_info, 0);
		g_type_add_interface_static (vala_try_statement_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_try_statement_type_id;
}




