/* valapropertyaccessor.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valapropertyaccessor.h>
#include <vala/valaproperty.h>
#include <vala/valablock.h>
#include <vala/valabasicblock.h>
#include <vala/valaformalparameter.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>

struct _ValaPropertyAccessorPrivate {
	ValaProperty* _prop;
	gboolean _readable;
	gboolean _writable;
	gboolean _construction;
	ValaSymbolAccessibility _access;
	ValaBlock* _body;
	ValaBasicBlock* _entry_block;
	ValaBasicBlock* _exit_block;
	ValaFormalParameter* _value_parameter;
};
#define VALA_PROPERTY_ACCESSOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessorPrivate))
enum  {
	VALA_PROPERTY_ACCESSOR_DUMMY_PROPERTY,
	VALA_PROPERTY_ACCESSOR_PROP,
	VALA_PROPERTY_ACCESSOR_READABLE,
	VALA_PROPERTY_ACCESSOR_WRITABLE,
	VALA_PROPERTY_ACCESSOR_CONSTRUCTION,
	VALA_PROPERTY_ACCESSOR_ACCESS,
	VALA_PROPERTY_ACCESSOR_BODY,
	VALA_PROPERTY_ACCESSOR_ENTRY_BLOCK,
	VALA_PROPERTY_ACCESSOR_EXIT_BLOCK,
	VALA_PROPERTY_ACCESSOR_VALUE_PARAMETER
};
static void vala_property_accessor_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_property_accessor_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_property_accessor_parent_class = NULL;
static void vala_property_accessor_dispose (GObject * obj);


/**
 * Creates a new property accessor.
 *
 * @param readable     true if get accessor, false otherwise
 * @param writable     true if set accessor, false otherwise
 * @param construction true if construct accessor, false otherwise
 * @param body         accessor body
 * @param source       reference to source code
 * @return             newly created property accessor
 */
ValaPropertyAccessor* vala_property_accessor_new (gboolean readable, gboolean writable, gboolean construction, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaPropertyAccessor * self;
	g_return_val_if_fail (body == NULL || VALA_IS_BLOCK (body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_PROPERTY_ACCESSOR, 0, NULL);
	vala_property_accessor_set_readable (self, readable);
	vala_property_accessor_set_writable (self, writable);
	vala_property_accessor_set_construction (self, construction);
	vala_property_accessor_set_body (self, body);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


static void vala_property_accessor_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaPropertyAccessor * self;
	self = VALA_PROPERTY_ACCESSOR (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_property_accessor (visitor, self);
}


static void vala_property_accessor_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaPropertyAccessor * self;
	self = VALA_PROPERTY_ACCESSOR (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	if (vala_property_accessor_get_body (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_property_accessor_get_body (self)), visitor);
	}
}


ValaProperty* vala_property_accessor_get_prop (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), NULL);
	return self->priv->_prop;
}


void vala_property_accessor_set_prop (ValaPropertyAccessor* self, ValaProperty* value) {
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	self->priv->_prop = value;
}


gboolean vala_property_accessor_get_readable (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), FALSE);
	return self->priv->_readable;
}


void vala_property_accessor_set_readable (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	self->priv->_readable = value;
}


gboolean vala_property_accessor_get_writable (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), FALSE);
	return self->priv->_writable;
}


void vala_property_accessor_set_writable (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	self->priv->_writable = value;
}


gboolean vala_property_accessor_get_construction (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), FALSE);
	return self->priv->_construction;
}


void vala_property_accessor_set_construction (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	self->priv->_construction = value;
}


ValaSymbolAccessibility vala_property_accessor_get_access (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), 0);
	return self->priv->_access;
}


void vala_property_accessor_set_access (ValaPropertyAccessor* self, ValaSymbolAccessibility value) {
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	self->priv->_access = value;
}


ValaBlock* vala_property_accessor_get_body (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), NULL);
	return self->priv->_body;
}


void vala_property_accessor_set_body (ValaPropertyAccessor* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL))), _tmp2);
}


ValaBasicBlock* vala_property_accessor_get_entry_block (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), NULL);
	return self->priv->_entry_block;
}


void vala_property_accessor_set_entry_block (ValaPropertyAccessor* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_entry_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_entry_block == NULL ? NULL : (self->priv->_entry_block = (g_object_unref (self->priv->_entry_block), NULL))), _tmp2);
}


ValaBasicBlock* vala_property_accessor_get_exit_block (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), NULL);
	return self->priv->_exit_block;
}


void vala_property_accessor_set_exit_block (ValaPropertyAccessor* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_exit_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_exit_block == NULL ? NULL : (self->priv->_exit_block = (g_object_unref (self->priv->_exit_block), NULL))), _tmp2);
}


ValaFormalParameter* vala_property_accessor_get_value_parameter (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), NULL);
	return self->priv->_value_parameter;
}


void vala_property_accessor_set_value_parameter (ValaPropertyAccessor* self, ValaFormalParameter* value) {
	ValaFormalParameter* _tmp2;
	ValaFormalParameter* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_value_parameter == NULL ? NULL : (self->priv->_value_parameter = (g_object_unref (self->priv->_value_parameter), NULL))), _tmp2);
}


static void vala_property_accessor_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaPropertyAccessor * self;
	self = VALA_PROPERTY_ACCESSOR (object);
	switch (property_id) {
		case VALA_PROPERTY_ACCESSOR_PROP:
		g_value_set_object (value, vala_property_accessor_get_prop (self));
		break;
		case VALA_PROPERTY_ACCESSOR_READABLE:
		g_value_set_boolean (value, vala_property_accessor_get_readable (self));
		break;
		case VALA_PROPERTY_ACCESSOR_WRITABLE:
		g_value_set_boolean (value, vala_property_accessor_get_writable (self));
		break;
		case VALA_PROPERTY_ACCESSOR_CONSTRUCTION:
		g_value_set_boolean (value, vala_property_accessor_get_construction (self));
		break;
		case VALA_PROPERTY_ACCESSOR_ACCESS:
		g_value_set_int (value, vala_property_accessor_get_access (self));
		break;
		case VALA_PROPERTY_ACCESSOR_BODY:
		g_value_set_object (value, vala_property_accessor_get_body (self));
		break;
		case VALA_PROPERTY_ACCESSOR_ENTRY_BLOCK:
		g_value_set_object (value, vala_property_accessor_get_entry_block (self));
		break;
		case VALA_PROPERTY_ACCESSOR_EXIT_BLOCK:
		g_value_set_object (value, vala_property_accessor_get_exit_block (self));
		break;
		case VALA_PROPERTY_ACCESSOR_VALUE_PARAMETER:
		g_value_set_object (value, vala_property_accessor_get_value_parameter (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_property_accessor_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaPropertyAccessor * self;
	self = VALA_PROPERTY_ACCESSOR (object);
	switch (property_id) {
		case VALA_PROPERTY_ACCESSOR_PROP:
		vala_property_accessor_set_prop (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_ACCESSOR_READABLE:
		vala_property_accessor_set_readable (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_ACCESSOR_WRITABLE:
		vala_property_accessor_set_writable (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_ACCESSOR_CONSTRUCTION:
		vala_property_accessor_set_construction (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_ACCESSOR_ACCESS:
		vala_property_accessor_set_access (self, g_value_get_int (value));
		break;
		case VALA_PROPERTY_ACCESSOR_BODY:
		vala_property_accessor_set_body (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_ACCESSOR_ENTRY_BLOCK:
		vala_property_accessor_set_entry_block (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_ACCESSOR_EXIT_BLOCK:
		vala_property_accessor_set_exit_block (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_ACCESSOR_VALUE_PARAMETER:
		vala_property_accessor_set_value_parameter (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_property_accessor_class_init (ValaPropertyAccessorClass * klass) {
	vala_property_accessor_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaPropertyAccessorPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_property_accessor_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_property_accessor_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_property_accessor_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_property_accessor_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_property_accessor_real_accept_children;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_ACCESSOR_PROP, g_param_spec_object ("prop", "prop", "prop", VALA_TYPE_PROPERTY, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_ACCESSOR_READABLE, g_param_spec_boolean ("readable", "readable", "readable", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_ACCESSOR_WRITABLE, g_param_spec_boolean ("writable", "writable", "writable", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_ACCESSOR_CONSTRUCTION, g_param_spec_boolean ("construction", "construction", "construction", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_ACCESSOR_ACCESS, g_param_spec_int ("access", "access", "access", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_ACCESSOR_BODY, g_param_spec_object ("body", "body", "body", VALA_TYPE_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_ACCESSOR_ENTRY_BLOCK, g_param_spec_object ("entry-block", "entry-block", "entry-block", VALA_TYPE_BASIC_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_ACCESSOR_EXIT_BLOCK, g_param_spec_object ("exit-block", "exit-block", "exit-block", VALA_TYPE_BASIC_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_ACCESSOR_VALUE_PARAMETER, g_param_spec_object ("value-parameter", "value-parameter", "value-parameter", VALA_TYPE_FORMAL_PARAMETER, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_property_accessor_init (ValaPropertyAccessor * self) {
	self->priv = VALA_PROPERTY_ACCESSOR_GET_PRIVATE (self);
}


static void vala_property_accessor_dispose (GObject * obj) {
	ValaPropertyAccessor * self;
	self = VALA_PROPERTY_ACCESSOR (obj);
	(self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL)));
	(self->priv->_entry_block == NULL ? NULL : (self->priv->_entry_block = (g_object_unref (self->priv->_entry_block), NULL)));
	(self->priv->_exit_block == NULL ? NULL : (self->priv->_exit_block = (g_object_unref (self->priv->_exit_block), NULL)));
	(self->priv->_value_parameter == NULL ? NULL : (self->priv->_value_parameter = (g_object_unref (self->priv->_value_parameter), NULL)));
	G_OBJECT_CLASS (vala_property_accessor_parent_class)->dispose (obj);
}


GType vala_property_accessor_get_type (void) {
	static GType vala_property_accessor_type_id = 0;
	if (G_UNLIKELY (vala_property_accessor_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaPropertyAccessorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_property_accessor_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaPropertyAccessor), 0, (GInstanceInitFunc) vala_property_accessor_init };
		vala_property_accessor_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaPropertyAccessor", &g_define_type_info, 0);
	}
	return vala_property_accessor_type_id;
}




