/* valanamedargument.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valanamedargument.h>
#include <vala/valaexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaNamedArgumentPrivate {
	char* _name;
	ValaExpression* _argument;
};

#define VALA_NAMED_ARGUMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_NAMED_ARGUMENT, ValaNamedArgumentPrivate))
enum  {
	VALA_NAMED_ARGUMENT_DUMMY_PROPERTY
};
static void vala_named_argument_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_named_argument_parent_class = NULL;
static void vala_named_argument_finalize (ValaCodeNode* obj);



/**
 * Creates a new named argument.
 *
 * @param name   property name
 * @param arg    property value expression
 * @param source reference to source code
 * @return       newly created named argument
 */
ValaNamedArgument* vala_named_argument_construct (GType object_type, const char* name, ValaExpression* argument, ValaSourceReference* source_reference) {
	ValaNamedArgument* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (argument), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = ((ValaNamedArgument*) (g_type_create_instance (object_type)));
	vala_named_argument_set_name (self, name);
	vala_named_argument_set_argument (self, argument);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


ValaNamedArgument* vala_named_argument_new (const char* name, ValaExpression* argument, ValaSourceReference* source_reference) {
	return vala_named_argument_construct (VALA_TYPE_NAMED_ARGUMENT, name, argument, source_reference);
}


static void vala_named_argument_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaNamedArgument * self;
	self = VALA_NAMED_ARGUMENT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (self->priv->_argument), visitor);
	vala_code_visitor_visit_named_argument (visitor, self);
}


const char* vala_named_argument_get_name (ValaNamedArgument* self) {
	g_return_val_if_fail (VALA_IS_NAMED_ARGUMENT (self), NULL);
	return self->priv->_name;
}


void vala_named_argument_set_name (ValaNamedArgument* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_NAMED_ARGUMENT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_name = (g_free (self->priv->_name), NULL)), _tmp2);
}


ValaExpression* vala_named_argument_get_argument (ValaNamedArgument* self) {
	g_return_val_if_fail (VALA_IS_NAMED_ARGUMENT (self), NULL);
	return self->priv->_argument;
}


void vala_named_argument_set_argument (ValaNamedArgument* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_NAMED_ARGUMENT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_argument = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_argument == NULL ? NULL : (self->priv->_argument = (vala_code_node_unref (self->priv->_argument), NULL))), _tmp2);
}


static void vala_named_argument_class_init (ValaNamedArgumentClass * klass) {
	vala_named_argument_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_named_argument_finalize;
	g_type_class_add_private (klass, sizeof (ValaNamedArgumentPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_named_argument_real_accept;
}


static void vala_named_argument_instance_init (ValaNamedArgument * self) {
	self->priv = VALA_NAMED_ARGUMENT_GET_PRIVATE (self);
}


static void vala_named_argument_finalize (ValaCodeNode* obj) {
	ValaNamedArgument * self;
	self = VALA_NAMED_ARGUMENT (obj);
	self->priv->_name = (g_free (self->priv->_name), NULL);
	(self->priv->_argument == NULL ? NULL : (self->priv->_argument = (vala_code_node_unref (self->priv->_argument), NULL)));
	VALA_CODE_NODE_CLASS (vala_named_argument_parent_class)->finalize (obj);
}


GType vala_named_argument_get_type (void) {
	static GType vala_named_argument_type_id = 0;
	if (vala_named_argument_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaNamedArgumentClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_named_argument_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaNamedArgument), 0, (GInstanceInitFunc) vala_named_argument_instance_init, NULL };
		vala_named_argument_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaNamedArgument", &g_define_type_info, 0);
	}
	return vala_named_argument_type_id;
}




