/* valaswitchsection.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaswitchsection.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valaswitchlabel.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valacodevisitor.h>
#include <vala/valastatement.h>




struct _ValaSwitchSectionPrivate {
	GeeList* labels;
};

#define VALA_SWITCH_SECTION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SWITCH_SECTION, ValaSwitchSectionPrivate))
enum  {
	VALA_SWITCH_SECTION_DUMMY_PROPERTY
};
static void vala_switch_section_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_switch_section_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_switch_section_parent_class = NULL;
static void vala_switch_section_finalize (ValaCodeNode* obj);



/**
 * Creates a new switch section.
 *
 * @param source_reference reference to source code
 * @return                 newly created switch section
 */
ValaSwitchSection* vala_switch_section_construct (GType object_type, ValaSourceReference* source_reference) {
	ValaSwitchSection* self;
	g_return_val_if_fail (VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = ((ValaSwitchSection*) (g_type_create_instance (object_type)));
	vala_symbol_init (VALA_SYMBOL (VALA_BLOCK (self)));
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


ValaSwitchSection* vala_switch_section_new (ValaSourceReference* source_reference) {
	return vala_switch_section_construct (VALA_TYPE_SWITCH_SECTION, source_reference);
}


/**
 * Appends the specified label to the list of switch labels.
 *
 * @param label a switch label
 */
void vala_switch_section_add_label (ValaSwitchSection* self, ValaSwitchLabel* label) {
	g_return_if_fail (VALA_IS_SWITCH_SECTION (self));
	g_return_if_fail (VALA_IS_SWITCH_LABEL (label));
	gee_collection_add (GEE_COLLECTION (self->priv->labels), label);
}


/**
 * Returns a copy of the list of switch labels.
 *
 * @return switch label list
 */
GeeList* vala_switch_section_get_labels (ValaSwitchSection* self) {
	g_return_val_if_fail (VALA_IS_SWITCH_SECTION (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_SWITCH_LABEL, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->labels));
}


gboolean vala_switch_section_has_default_label (ValaSwitchSection* self) {
	g_return_val_if_fail (VALA_IS_SWITCH_SECTION (self), FALSE);
	{
		GeeList* label_collection;
		int label_it;
		label_collection = self->priv->labels;
		for (label_it = 0; label_it < gee_collection_get_size (GEE_COLLECTION (label_collection)); label_it = label_it + 1) {
			ValaSwitchLabel* label;
			label = ((ValaSwitchLabel*) (gee_list_get (GEE_LIST (label_collection), label_it)));
			{
				if (vala_switch_label_get_expression (label) == NULL) {
					gboolean _tmp0;
					return (_tmp0 = TRUE, (label == NULL ? NULL : (label = (vala_code_node_unref (label), NULL))), _tmp0);
				}
				(label == NULL ? NULL : (label = (vala_code_node_unref (label), NULL)));
			}
		}
	}
	return FALSE;
}


static void vala_switch_section_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaSwitchSection * self;
	self = VALA_SWITCH_SECTION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_switch_section (visitor, self);
}


static void vala_switch_section_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaSwitchSection * self;
	self = VALA_SWITCH_SECTION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* label_collection;
		int label_it;
		label_collection = self->priv->labels;
		for (label_it = 0; label_it < gee_collection_get_size (GEE_COLLECTION (label_collection)); label_it = label_it + 1) {
			ValaSwitchLabel* label;
			label = ((ValaSwitchLabel*) (gee_list_get (GEE_LIST (label_collection), label_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (label), visitor);
				(label == NULL ? NULL : (label = (vala_code_node_unref (label), NULL)));
			}
		}
	}
	{
		GeeList* st_collection;
		int st_it;
		st_collection = vala_block_get_statements (VALA_BLOCK (self));
		for (st_it = 0; st_it < gee_collection_get_size (GEE_COLLECTION (st_collection)); st_it = st_it + 1) {
			ValaStatement* st;
			st = ((ValaStatement*) (gee_list_get (GEE_LIST (st_collection), st_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (st), visitor);
				(st == NULL ? NULL : (st = (vala_code_node_unref (st), NULL)));
			}
		}
		(st_collection == NULL ? NULL : (st_collection = (gee_collection_object_unref (st_collection), NULL)));
	}
}


static void vala_switch_section_class_init (ValaSwitchSectionClass * klass) {
	vala_switch_section_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_switch_section_finalize;
	g_type_class_add_private (klass, sizeof (ValaSwitchSectionPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_switch_section_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_switch_section_real_accept_children;
}


static void vala_switch_section_instance_init (ValaSwitchSection * self) {
	self->priv = VALA_SWITCH_SECTION_GET_PRIVATE (self);
	self->priv->labels = GEE_LIST (gee_array_list_new (VALA_TYPE_SWITCH_LABEL, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal));
}


static void vala_switch_section_finalize (ValaCodeNode* obj) {
	ValaSwitchSection * self;
	self = VALA_SWITCH_SECTION (obj);
	(self->priv->labels == NULL ? NULL : (self->priv->labels = (gee_collection_object_unref (self->priv->labels), NULL)));
	VALA_CODE_NODE_CLASS (vala_switch_section_parent_class)->finalize (obj);
}


GType vala_switch_section_get_type (void) {
	static GType vala_switch_section_type_id = 0;
	if (vala_switch_section_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSwitchSectionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_switch_section_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSwitchSection), 0, (GInstanceInitFunc) vala_switch_section_instance_init, NULL };
		vala_switch_section_type_id = g_type_register_static (VALA_TYPE_BLOCK, "ValaSwitchSection", &g_define_type_info, 0);
	}
	return vala_switch_section_type_id;
}




