/* valatypecheck.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valatypecheck.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaTypeCheckPrivate {
	ValaExpression* _expression;
	ValaDataType* _data_type;
};

#define VALA_TYPECHECK_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_TYPECHECK, ValaTypeCheckPrivate))
enum  {
	VALA_TYPECHECK_DUMMY_PROPERTY
};
static void vala_typecheck_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_typecheck_real_is_pure (ValaExpression* base);
static void vala_typecheck_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gpointer vala_typecheck_parent_class = NULL;
static void vala_typecheck_finalize (ValaCodeNode* obj);



/**
 * Creates a new type check expression.
 *
 * @param expr   an expression
 * @param type   a data type
 * @param source reference to source code
 * @return       newly created type check expression
 */
ValaTypeCheck* vala_typecheck_construct (GType object_type, ValaExpression* expr, ValaDataType* type, ValaSourceReference* source) {
	ValaTypeCheck* self;
	g_return_val_if_fail (VALA_IS_EXPRESSION (expr), NULL);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (type), NULL);
	g_return_val_if_fail (VALA_IS_SOURCE_REFERENCE (source), NULL);
	self = ((ValaTypeCheck*) (g_type_create_instance (object_type)));
	vala_typecheck_set_expression (self, expr);
	vala_typecheck_set_type_reference (self, type);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	return self;
}


ValaTypeCheck* vala_typecheck_new (ValaExpression* expr, ValaDataType* type, ValaSourceReference* source) {
	return vala_typecheck_construct (VALA_TYPE_TYPECHECK, expr, type, source);
}


static void vala_typecheck_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaTypeCheck * self;
	self = VALA_TYPECHECK (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (self->priv->_expression), visitor);
	vala_code_node_accept (VALA_CODE_NODE (vala_typecheck_get_type_reference (self)), visitor);
	vala_code_visitor_visit_type_check (visitor, self);
	vala_code_visitor_visit_expression (visitor, VALA_EXPRESSION (self));
}


static gboolean vala_typecheck_real_is_pure (ValaExpression* base) {
	ValaTypeCheck * self;
	self = VALA_TYPECHECK (base);
	return vala_expression_is_pure (self->priv->_expression);
}


static void vala_typecheck_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaTypeCheck * self;
	self = VALA_TYPECHECK (base);
	g_return_if_fail (VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (VALA_IS_DATA_TYPE (new_type));
	if (vala_typecheck_get_type_reference (self) == old_type) {
		vala_typecheck_set_type_reference (self, new_type);
	}
}


ValaExpression* vala_typecheck_get_expression (ValaTypeCheck* self) {
	g_return_val_if_fail (VALA_IS_TYPECHECK (self), NULL);
	return self->priv->_expression;
}


void vala_typecheck_set_expression (ValaTypeCheck* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_TYPECHECK (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_expression == NULL ? NULL : (self->priv->_expression = (vala_code_node_unref (self->priv->_expression), NULL))), _tmp2);
}


ValaDataType* vala_typecheck_get_type_reference (ValaTypeCheck* self) {
	g_return_val_if_fail (VALA_IS_TYPECHECK (self), NULL);
	return self->priv->_data_type;
}


void vala_typecheck_set_type_reference (ValaTypeCheck* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_TYPECHECK (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_data_type), VALA_CODE_NODE (self));
}


static void vala_typecheck_class_init (ValaTypeCheckClass * klass) {
	vala_typecheck_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_typecheck_finalize;
	g_type_class_add_private (klass, sizeof (ValaTypeCheckPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_typecheck_real_accept;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_typecheck_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_typecheck_real_replace_type;
}


static void vala_typecheck_instance_init (ValaTypeCheck * self) {
	self->priv = VALA_TYPECHECK_GET_PRIVATE (self);
}


static void vala_typecheck_finalize (ValaCodeNode* obj) {
	ValaTypeCheck * self;
	self = VALA_TYPECHECK (obj);
	(self->priv->_expression == NULL ? NULL : (self->priv->_expression = (vala_code_node_unref (self->priv->_expression), NULL)));
	(self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL)));
	VALA_CODE_NODE_CLASS (vala_typecheck_parent_class)->finalize (obj);
}


GType vala_typecheck_get_type (void) {
	static GType vala_typecheck_type_id = 0;
	if (vala_typecheck_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaTypeCheckClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_typecheck_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaTypeCheck), 0, (GInstanceInitFunc) vala_typecheck_instance_init, NULL };
		vala_typecheck_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaTypeCheck", &g_define_type_info, 0);
	}
	return vala_typecheck_type_id;
}




