/* valaccodeblock.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <ccode/valaccodeblock.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/collection.h>
#include <ccode/valaccodelabel.h>
#include <ccode/valaccodecasestatement.h>
#include <ccode/valaccodereturnstatement.h>
#include <ccode/valaccodegotostatement.h>
#include <ccode/valaccodecontinuestatement.h>
#include <ccode/valaccodebreakstatement.h>




struct _ValaCCodeBlockPrivate {
	gboolean _suppress_newline;
	GeeList* statements;
};

#define VALA_CCODE_BLOCK_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_BLOCK, ValaCCodeBlockPrivate))
enum  {
	VALA_CCODE_BLOCK_DUMMY_PROPERTY
};
static void vala_ccode_block_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer);
static gpointer vala_ccode_block_parent_class = NULL;
static void vala_ccode_block_finalize (ValaCCodeNode* obj);



/**
 * Prepend the specified statement to the list of statements.
 */
void vala_ccode_block_prepend_statement (ValaCCodeBlock* self, ValaCCodeNode* statement) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (statement != NULL);
	gee_list_insert (self->priv->statements, 0, statement);
}


/**
 * Append the specified statement to the list of statements.
 */
void vala_ccode_block_add_statement (ValaCCodeBlock* self, ValaCCodeNode* statement) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (statement != NULL);
	/* allow generic nodes to include comments */
	gee_collection_add (((GeeCollection*) (self->priv->statements)), statement);
}


static void vala_ccode_block_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer) {
	ValaCCodeBlock * self;
	ValaCCodeNode* last_statement;
	self = ((ValaCCodeBlock*) (base));
	g_return_if_fail (writer != NULL);
	/* the last reachable statement*/
	last_statement = NULL;
	vala_ccode_writer_write_begin_block (writer);
	{
		GeeList* statement_collection;
		int statement_it;
		statement_collection = self->priv->statements;
		for (statement_it = 0; statement_it < gee_collection_get_size (GEE_COLLECTION (statement_collection)); statement_it = statement_it + 1) {
			ValaCCodeNode* statement;
			statement = ((ValaCCodeNode*) (gee_list_get (GEE_LIST (statement_collection), statement_it)));
			{
				vala_ccode_node_write_declaration (statement, writer);
				/* determine last reachable statement*/
				if (VALA_IS_CCODE_LABEL (statement) || VALA_IS_CCODE_CASE_STATEMENT (statement)) {
					ValaCCodeNode* _tmp0;
					_tmp0 = NULL;
					last_statement = (_tmp0 = NULL, (last_statement == NULL ? NULL : (last_statement = (vala_ccode_node_unref (last_statement), NULL))), _tmp0);
				} else {
					if (VALA_IS_CCODE_RETURN_STATEMENT (statement) || VALA_IS_CCODE_GOTO_STATEMENT (statement) || VALA_IS_CCODE_CONTINUE_STATEMENT (statement) || VALA_IS_CCODE_BREAK_STATEMENT (statement)) {
						ValaCCodeNode* _tmp2;
						ValaCCodeNode* _tmp1;
						_tmp2 = NULL;
						_tmp1 = NULL;
						last_statement = (_tmp2 = (_tmp1 = statement, (_tmp1 == NULL ? NULL : vala_ccode_node_ref (_tmp1))), (last_statement == NULL ? NULL : (last_statement = (vala_ccode_node_unref (last_statement), NULL))), _tmp2);
					}
				}
				(statement == NULL ? NULL : (statement = (vala_ccode_node_unref (statement), NULL)));
			}
		}
	}
	{
		GeeList* statement_collection;
		int statement_it;
		statement_collection = self->priv->statements;
		for (statement_it = 0; statement_it < gee_collection_get_size (GEE_COLLECTION (statement_collection)); statement_it = statement_it + 1) {
			ValaCCodeNode* statement;
			statement = ((ValaCCodeNode*) (gee_list_get (GEE_LIST (statement_collection), statement_it)));
			{
				vala_ccode_node_write (statement, writer);
				/* only output reachable code*/
				if (statement == last_statement) {
					(statement == NULL ? NULL : (statement = (vala_ccode_node_unref (statement), NULL)));
					break;
				}
				(statement == NULL ? NULL : (statement = (vala_ccode_node_unref (statement), NULL)));
			}
		}
	}
	vala_ccode_writer_write_end_block (writer);
	if (!self->priv->_suppress_newline) {
		vala_ccode_writer_write_newline (writer);
	}
	(last_statement == NULL ? NULL : (last_statement = (vala_ccode_node_unref (last_statement), NULL)));
}


/**
 * Represents a C code block.
 */
ValaCCodeBlock* vala_ccode_block_construct (GType object_type) {
	ValaCCodeBlock* self;
	self = ((ValaCCodeBlock*) (g_type_create_instance (object_type)));
	return self;
}


ValaCCodeBlock* vala_ccode_block_new (void) {
	return vala_ccode_block_construct (VALA_TYPE_CCODE_BLOCK);
}


gboolean vala_ccode_block_get_suppress_newline (ValaCCodeBlock* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_suppress_newline;
}


void vala_ccode_block_set_suppress_newline (ValaCCodeBlock* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_suppress_newline = value;
}


static void vala_ccode_block_class_init (ValaCCodeBlockClass * klass) {
	vala_ccode_block_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_NODE_CLASS (klass)->finalize = vala_ccode_block_finalize;
	g_type_class_add_private (klass, sizeof (ValaCCodeBlockPrivate));
	VALA_CCODE_NODE_CLASS (klass)->write = vala_ccode_block_real_write;
}


static void vala_ccode_block_instance_init (ValaCCodeBlock * self) {
	self->priv = VALA_CCODE_BLOCK_GET_PRIVATE (self);
	self->priv->statements = ((GeeList*) (gee_array_list_new (VALA_TYPE_CCODE_NODE, ((GBoxedCopyFunc) (vala_ccode_node_ref)), vala_ccode_node_unref, g_direct_equal)));
}


static void vala_ccode_block_finalize (ValaCCodeNode* obj) {
	ValaCCodeBlock * self;
	self = VALA_CCODE_BLOCK (obj);
	(self->priv->statements == NULL ? NULL : (self->priv->statements = (gee_collection_object_unref (self->priv->statements), NULL)));
	VALA_CCODE_NODE_CLASS (vala_ccode_block_parent_class)->finalize (obj);
}


GType vala_ccode_block_get_type (void) {
	static GType vala_ccode_block_type_id = 0;
	if (vala_ccode_block_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeBlockClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_block_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeBlock), 0, (GInstanceInitFunc) vala_ccode_block_instance_init, NULL };
		vala_ccode_block_type_id = g_type_register_static (VALA_TYPE_CCODE_STATEMENT, "ValaCCodeBlock", &g_define_type_info, 0);
	}
	return vala_ccode_block_type_id;
}




