/* valacfgbuilder.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valacfgbuilder.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/collection.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valabasicblock.h>
#include <vala/valavoidtype.h>
#include <vala/valareport.h>
#include <vala/valacodenode.h>
#include <vala/valasourcereference.h>
#include <vala/valalocalvariable.h>
#include <vala/valainvocationexpression.h>
#include <vala/valamemberaccess.h>
#include <vala/valaexpression.h>
#include <vala/valaswitchsection.h>
#include <vala/valastatement.h>
#include <vala/valacatchclause.h>
#include <vala/valadatatype.h>
#include <vala/valaerrorcode.h>
#include <gobject/gvaluecollector.h>


#define VALA_CFG_BUILDER_TYPE_JUMP_TARGET (vala_cfg_builder_jump_target_get_type ())
#define VALA_CFG_BUILDER_JUMP_TARGET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_CFG_BUILDER_TYPE_JUMP_TARGET, ValaCFGBuilderJumpTarget))
#define VALA_CFG_BUILDER_JUMP_TARGET_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_CFG_BUILDER_TYPE_JUMP_TARGET, ValaCFGBuilderJumpTargetClass))
#define VALA_CFG_BUILDER_IS_JUMP_TARGET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_CFG_BUILDER_TYPE_JUMP_TARGET))
#define VALA_CFG_BUILDER_IS_JUMP_TARGET_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_CFG_BUILDER_TYPE_JUMP_TARGET))
#define VALA_CFG_BUILDER_JUMP_TARGET_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_CFG_BUILDER_TYPE_JUMP_TARGET, ValaCFGBuilderJumpTargetClass))

typedef struct _ValaCFGBuilderJumpTarget ValaCFGBuilderJumpTarget;
typedef struct _ValaCFGBuilderJumpTargetClass ValaCFGBuilderJumpTargetClass;
typedef struct _ValaCFGBuilderJumpTargetPrivate ValaCFGBuilderJumpTargetPrivate;
typedef struct _ValaCFGBuilderParamSpecJumpTarget ValaCFGBuilderParamSpecJumpTarget;

struct _ValaCFGBuilderJumpTarget {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCFGBuilderJumpTargetPrivate * priv;
};

struct _ValaCFGBuilderJumpTargetClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCFGBuilderJumpTarget *self);
};

struct _ValaCFGBuilderParamSpecJumpTarget {
	GParamSpec parent_instance;
};



struct _ValaCFGBuilderPrivate {
	ValaCodeContext* context;
	ValaBasicBlock* current_block;
	gboolean unreachable_reported;
	GeeList* jump_stack;
};

#define VALA_CFG_BUILDER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CFG_BUILDER, ValaCFGBuilderPrivate))
enum  {
	VALA_CFG_BUILDER_DUMMY_PROPERTY
};
static void vala_cfg_builder_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static void vala_cfg_builder_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_cfg_builder_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_cfg_builder_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_cfg_builder_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_cfg_builder_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed);
static void vala_cfg_builder_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_cfg_builder_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_cfg_builder_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_cfg_builder_real_visit_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_cfg_builder_real_visit_declaration_statement (ValaCodeVisitor* base, ValaDeclarationStatement* stmt);
static void vala_cfg_builder_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static void vala_cfg_builder_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt);
static void vala_cfg_builder_real_visit_switch_statement (ValaCodeVisitor* base, ValaSwitchStatement* stmt);
static void vala_cfg_builder_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt);
static void vala_cfg_builder_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt);
static void vala_cfg_builder_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt);
static void vala_cfg_builder_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_cfg_builder_real_visit_break_statement (ValaCodeVisitor* base, ValaBreakStatement* stmt);
static void vala_cfg_builder_real_visit_continue_statement (ValaCodeVisitor* base, ValaContinueStatement* stmt);
static void vala_cfg_builder_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_cfg_builder_handle_errors (ValaCFGBuilder* self, ValaCodeNode* node);
static void vala_cfg_builder_real_visit_yield_statement (ValaCodeVisitor* base, ValaYieldStatement* stmt);
static void vala_cfg_builder_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt);
static void vala_cfg_builder_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt);
static void vala_cfg_builder_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt);
static gboolean vala_cfg_builder_unreachable (ValaCFGBuilder* self, ValaCodeNode* node);
struct _ValaCFGBuilderJumpTargetPrivate {
	gboolean _is_break_target;
	gboolean _is_continue_target;
	gboolean _is_return_target;
	gboolean _is_error_target;
	ValaErrorDomain* _error_domain;
	ValaErrorCode* _error_code;
	gboolean _is_finally_clause;
	ValaBasicBlock* _basic_block;
	ValaBasicBlock* _last_block;
	ValaCatchClause* _catch_clause;
};

#define VALA_CFG_BUILDER_JUMP_TARGET_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_CFG_BUILDER_TYPE_JUMP_TARGET, ValaCFGBuilderJumpTargetPrivate))
enum  {
	VALA_CFG_BUILDER_JUMP_TARGET_DUMMY_PROPERTY
};
static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_construct_break_target (GType object_type, ValaBasicBlock* basic_block);
static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_new_break_target (ValaBasicBlock* basic_block);
static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_construct_continue_target (GType object_type, ValaBasicBlock* basic_block);
static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_new_continue_target (ValaBasicBlock* basic_block);
static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_construct_return_target (GType object_type, ValaBasicBlock* basic_block);
static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_new_return_target (ValaBasicBlock* basic_block);
static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_construct_error_target (GType object_type, ValaBasicBlock* basic_block, ValaCatchClause* catch_clause, ValaErrorDomain* error_domain, ValaErrorCode* error_code);
static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_new_error_target (ValaBasicBlock* basic_block, ValaCatchClause* catch_clause, ValaErrorDomain* error_domain, ValaErrorCode* error_code);
static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_construct_finally_clause (GType object_type, ValaBasicBlock* basic_block, ValaBasicBlock* last_block);
static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_new_finally_clause (ValaBasicBlock* basic_block, ValaBasicBlock* last_block);
static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_construct (GType object_type);
static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_new (void);
static gboolean vala_cfg_builder_jump_target_get_is_break_target (ValaCFGBuilderJumpTarget* self);
static void vala_cfg_builder_jump_target_set_is_break_target (ValaCFGBuilderJumpTarget* self, gboolean value);
static gboolean vala_cfg_builder_jump_target_get_is_continue_target (ValaCFGBuilderJumpTarget* self);
static void vala_cfg_builder_jump_target_set_is_continue_target (ValaCFGBuilderJumpTarget* self, gboolean value);
static gboolean vala_cfg_builder_jump_target_get_is_return_target (ValaCFGBuilderJumpTarget* self);
static void vala_cfg_builder_jump_target_set_is_return_target (ValaCFGBuilderJumpTarget* self, gboolean value);
static gboolean vala_cfg_builder_jump_target_get_is_error_target (ValaCFGBuilderJumpTarget* self);
static void vala_cfg_builder_jump_target_set_is_error_target (ValaCFGBuilderJumpTarget* self, gboolean value);
static ValaErrorDomain* vala_cfg_builder_jump_target_get_error_domain (ValaCFGBuilderJumpTarget* self);
static void vala_cfg_builder_jump_target_set_error_domain (ValaCFGBuilderJumpTarget* self, ValaErrorDomain* value);
static ValaErrorCode* vala_cfg_builder_jump_target_get_error_code (ValaCFGBuilderJumpTarget* self);
static void vala_cfg_builder_jump_target_set_error_code (ValaCFGBuilderJumpTarget* self, ValaErrorCode* value);
static gboolean vala_cfg_builder_jump_target_get_is_finally_clause (ValaCFGBuilderJumpTarget* self);
static void vala_cfg_builder_jump_target_set_is_finally_clause (ValaCFGBuilderJumpTarget* self, gboolean value);
static ValaBasicBlock* vala_cfg_builder_jump_target_get_basic_block (ValaCFGBuilderJumpTarget* self);
static void vala_cfg_builder_jump_target_set_basic_block (ValaCFGBuilderJumpTarget* self, ValaBasicBlock* value);
static ValaBasicBlock* vala_cfg_builder_jump_target_get_last_block (ValaCFGBuilderJumpTarget* self);
static void vala_cfg_builder_jump_target_set_last_block (ValaCFGBuilderJumpTarget* self, ValaBasicBlock* value);
static ValaCatchClause* vala_cfg_builder_jump_target_get_catch_clause (ValaCFGBuilderJumpTarget* self);
static void vala_cfg_builder_jump_target_set_catch_clause (ValaCFGBuilderJumpTarget* self, ValaCatchClause* value);
static GParamSpec* vala_cfg_builder_param_spec_jump_target (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
static gpointer vala_cfg_builder_value_get_jump_target (const GValue* value);
static void vala_cfg_builder_value_set_jump_target (GValue* value, gpointer v_object);
static gpointer vala_cfg_builder_jump_target_parent_class = NULL;
static void vala_cfg_builder_jump_target_finalize (ValaCFGBuilderJumpTarget* obj);
static GType vala_cfg_builder_jump_target_get_type (void);
static gpointer vala_cfg_builder_jump_target_ref (gpointer instance);
static void vala_cfg_builder_jump_target_unref (gpointer instance);
static gpointer vala_cfg_builder_parent_class = NULL;
static void vala_cfg_builder_finalize (ValaCodeVisitor* obj);



ValaCFGBuilder* vala_cfg_builder_construct (GType object_type) {
	ValaCFGBuilder* self;
	self = ((ValaCFGBuilder*) (g_type_create_instance (object_type)));
	return self;
}


ValaCFGBuilder* vala_cfg_builder_new (void) {
	return vala_cfg_builder_construct (VALA_TYPE_CFG_BUILDER);
}


/**
 * Build control flow graph in the specified context.
 *
 * @param context a code context
 */
void vala_cfg_builder_build_cfg (ValaCFGBuilder* self, ValaCodeContext* context) {
	ValaCodeContext* _tmp1;
	ValaCodeContext* _tmp0;
	GeeList* source_files;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->context = (_tmp1 = (_tmp0 = context, (_tmp0 == NULL ? NULL : vala_code_context_ref (_tmp0))), (self->priv->context == NULL ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL))), _tmp1);
	/* we're only interested in non-pkg source files */
	source_files = vala_code_context_get_source_files (context);
	{
		GeeList* file_collection;
		int file_it;
		file_collection = source_files;
		for (file_it = 0; file_it < gee_collection_get_size (GEE_COLLECTION (file_collection)); file_it = file_it + 1) {
			ValaSourceFile* file;
			file = ((ValaSourceFile*) (gee_list_get (GEE_LIST (file_collection), file_it)));
			{
				if (!vala_source_file_get_external_package (file)) {
					vala_source_file_accept (file, ((ValaCodeVisitor*) (self)));
				}
				(file == NULL ? NULL : (file = (vala_source_file_unref (file), NULL)));
			}
		}
	}
	(source_files == NULL ? NULL : (source_files = (gee_collection_object_unref (source_files), NULL)));
}


static void vala_cfg_builder_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (source_file != NULL);
	vala_source_file_accept_children (source_file, ((ValaCodeVisitor*) (self)));
}


static void vala_cfg_builder_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (cl != NULL);
	vala_code_node_accept_children (((ValaCodeNode*) (cl)), ((ValaCodeVisitor*) (self)));
}


static void vala_cfg_builder_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (st != NULL);
	vala_code_node_accept_children (((ValaCodeNode*) (st)), ((ValaCodeVisitor*) (self)));
}


static void vala_cfg_builder_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (iface != NULL);
	vala_code_node_accept_children (((ValaCodeNode*) (iface)), ((ValaCodeVisitor*) (self)));
}


static void vala_cfg_builder_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (en != NULL);
	vala_code_node_accept_children (((ValaCodeNode*) (en)), ((ValaCodeVisitor*) (self)));
}


static void vala_cfg_builder_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (ed != NULL);
	vala_code_node_accept_children (((ValaCodeNode*) (ed)), ((ValaCodeVisitor*) (self)));
}


static void vala_cfg_builder_real_visit_method (ValaCodeVisitor* base, ValaMethod* m) {
	ValaCFGBuilder * self;
	ValaBasicBlock* _tmp0;
	ValaBasicBlock* _tmp1;
	ValaBasicBlock* _tmp2;
	ValaCFGBuilderJumpTarget* _tmp3;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (m != NULL);
	if (vala_method_get_body (m) == NULL) {
		return;
	}
	_tmp0 = NULL;
	vala_method_set_entry_block (m, (_tmp0 = vala_basic_block_new_entry ()));
	(_tmp0 == NULL ? NULL : (_tmp0 = (vala_basic_block_unref (_tmp0), NULL)));
	_tmp1 = NULL;
	vala_method_set_exit_block (m, (_tmp1 = vala_basic_block_new_exit ()));
	(_tmp1 == NULL ? NULL : (_tmp1 = (vala_basic_block_unref (_tmp1), NULL)));
	_tmp2 = NULL;
	self->priv->current_block = (_tmp2 = vala_basic_block_new (), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp2);
	vala_basic_block_connect (vala_method_get_entry_block (m), self->priv->current_block);
	_tmp3 = NULL;
	gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp3 = vala_cfg_builder_jump_target_new_return_target (vala_method_get_exit_block (m))));
	(_tmp3 == NULL ? NULL : (_tmp3 = (vala_cfg_builder_jump_target_unref (_tmp3), NULL)));
	vala_code_node_accept_children (((ValaCodeNode*) (m)), ((ValaCodeVisitor*) (self)));
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1);
	if (self->priv->current_block != NULL) {
		/* end of method body reachable*/
		if (!(VALA_IS_VOID_TYPE (vala_method_get_return_type (m)))) {
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (m))), "missing return statement at end of method body");
			vala_code_node_set_error (((ValaCodeNode*) (m)), TRUE);
		}
		vala_basic_block_connect (self->priv->current_block, vala_method_get_exit_block (m));
	}
}


static void vala_cfg_builder_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (prop != NULL);
	vala_code_node_accept_children (((ValaCodeNode*) (prop)), ((ValaCodeVisitor*) (self)));
}


static void vala_cfg_builder_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc) {
	ValaCFGBuilder * self;
	ValaBasicBlock* _tmp0;
	ValaBasicBlock* _tmp1;
	ValaBasicBlock* _tmp2;
	ValaCFGBuilderJumpTarget* _tmp3;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (acc != NULL);
	if (vala_property_accessor_get_body (acc) == NULL) {
		return;
	}
	_tmp0 = NULL;
	vala_property_accessor_set_entry_block (acc, (_tmp0 = vala_basic_block_new_entry ()));
	(_tmp0 == NULL ? NULL : (_tmp0 = (vala_basic_block_unref (_tmp0), NULL)));
	_tmp1 = NULL;
	vala_property_accessor_set_exit_block (acc, (_tmp1 = vala_basic_block_new_exit ()));
	(_tmp1 == NULL ? NULL : (_tmp1 = (vala_basic_block_unref (_tmp1), NULL)));
	_tmp2 = NULL;
	self->priv->current_block = (_tmp2 = vala_basic_block_new (), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp2);
	vala_basic_block_connect (vala_property_accessor_get_entry_block (acc), self->priv->current_block);
	_tmp3 = NULL;
	gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp3 = vala_cfg_builder_jump_target_new_return_target (vala_property_accessor_get_exit_block (acc))));
	(_tmp3 == NULL ? NULL : (_tmp3 = (vala_cfg_builder_jump_target_unref (_tmp3), NULL)));
	vala_code_node_accept_children (((ValaCodeNode*) (acc)), ((ValaCodeVisitor*) (self)));
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1);
	if (self->priv->current_block != NULL) {
		/* end of property accessor body reachable*/
		if (vala_property_accessor_get_readable (acc)) {
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (acc))), "missing return statement at end of property getter body");
			vala_code_node_set_error (((ValaCodeNode*) (acc)), TRUE);
		}
		vala_basic_block_connect (self->priv->current_block, vala_property_accessor_get_exit_block (acc));
	}
}


static void vala_cfg_builder_real_visit_block (ValaCodeVisitor* base, ValaBlock* b) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (b != NULL);
	vala_code_node_accept_children (((ValaCodeNode*) (b)), ((ValaCodeVisitor*) (self)));
}


static void vala_cfg_builder_real_visit_declaration_statement (ValaCodeVisitor* base, ValaDeclarationStatement* stmt) {
	ValaCFGBuilder * self;
	ValaLocalVariable* _tmp1;
	ValaSymbol* _tmp0;
	ValaLocalVariable* local;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	vala_basic_block_add_node (self->priv->current_block, ((ValaCodeNode*) (stmt)));
	_tmp1 = NULL;
	_tmp0 = NULL;
	local = (_tmp1 = (_tmp0 = vala_declaration_statement_get_declaration (stmt), (VALA_IS_LOCAL_VARIABLE (_tmp0) ? ((ValaLocalVariable*) (_tmp0)) : NULL)), (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1)));
	if (local != NULL && vala_local_variable_get_initializer (local) != NULL) {
		vala_cfg_builder_handle_errors (self, ((ValaCodeNode*) (vala_local_variable_get_initializer (local))));
	}
	(local == NULL ? NULL : (local = (vala_code_node_unref (local), NULL)));
}


static void vala_cfg_builder_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	vala_basic_block_add_node (self->priv->current_block, ((ValaCodeNode*) (stmt)));
	vala_cfg_builder_handle_errors (self, ((ValaCodeNode*) (stmt)));
	if (VALA_IS_INVOCATION_EXPRESSION (vala_expression_statement_get_expression (stmt))) {
		ValaInvocationExpression* _tmp0;
		ValaInvocationExpression* expr;
		ValaMemberAccess* _tmp2;
		ValaExpression* _tmp1;
		ValaMemberAccess* ma;
		ValaAttribute* _tmp3;
		gboolean _tmp4;
		_tmp0 = NULL;
		expr = (_tmp0 = VALA_INVOCATION_EXPRESSION (vala_expression_statement_get_expression (stmt)), (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0)));
		_tmp2 = NULL;
		_tmp1 = NULL;
		ma = (_tmp2 = (_tmp1 = vala_invocation_expression_get_call (expr), (VALA_IS_MEMBER_ACCESS (_tmp1) ? ((ValaMemberAccess*) (_tmp1)) : NULL)), (_tmp2 == NULL ? NULL : vala_code_node_ref (_tmp2)));
		_tmp3 = NULL;
		if ((_tmp4 = ma != NULL && vala_expression_get_symbol_reference (((ValaExpression*) (ma))) != NULL && (_tmp3 = vala_code_node_get_attribute (((ValaCodeNode*) (vala_expression_get_symbol_reference (((ValaExpression*) (ma))))), "NoReturn")) != NULL, (_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL))), _tmp4)) {
			ValaBasicBlock* _tmp5;
			_tmp5 = NULL;
			self->priv->current_block = (_tmp5 = NULL, (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp5);
			self->priv->unreachable_reported = FALSE;
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			(ma == NULL ? NULL : (ma = (vala_code_node_unref (ma), NULL)));
			return;
		}
		(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
		(ma == NULL ? NULL : (ma = (vala_code_node_unref (ma), NULL)));
	}
}


static void vala_cfg_builder_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt) {
	ValaCFGBuilder * self;
	ValaBasicBlock* _tmp0;
	ValaBasicBlock* last_block;
	ValaBasicBlock* _tmp1;
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* last_true_block;
	ValaBasicBlock* _tmp3;
	ValaBasicBlock* _tmp4;
	ValaBasicBlock* last_false_block;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	/* condition*/
	vala_basic_block_add_node (self->priv->current_block, ((ValaCodeNode*) (vala_if_statement_get_condition (stmt))));
	vala_cfg_builder_handle_errors (self, ((ValaCodeNode*) (vala_if_statement_get_condition (stmt))));
	/* true block*/
	_tmp0 = NULL;
	last_block = (_tmp0 = self->priv->current_block, (_tmp0 == NULL ? NULL : vala_basic_block_ref (_tmp0)));
	_tmp1 = NULL;
	self->priv->current_block = (_tmp1 = vala_basic_block_new (), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp1);
	vala_basic_block_connect (last_block, self->priv->current_block);
	vala_code_node_accept (((ValaCodeNode*) (vala_if_statement_get_true_statement (stmt))), ((ValaCodeVisitor*) (self)));
	/* false block*/
	_tmp2 = NULL;
	last_true_block = (_tmp2 = self->priv->current_block, (_tmp2 == NULL ? NULL : vala_basic_block_ref (_tmp2)));
	_tmp3 = NULL;
	self->priv->current_block = (_tmp3 = vala_basic_block_new (), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp3);
	vala_basic_block_connect (last_block, self->priv->current_block);
	if (vala_if_statement_get_false_statement (stmt) != NULL) {
		vala_code_node_accept (((ValaCodeNode*) (vala_if_statement_get_false_statement (stmt))), ((ValaCodeVisitor*) (self)));
	}
	/* after if/else*/
	_tmp4 = NULL;
	last_false_block = (_tmp4 = self->priv->current_block, (_tmp4 == NULL ? NULL : vala_basic_block_ref (_tmp4)));
	/* reachable?*/
	if (last_true_block != NULL || last_false_block != NULL) {
		ValaBasicBlock* _tmp5;
		_tmp5 = NULL;
		self->priv->current_block = (_tmp5 = vala_basic_block_new (), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp5);
		if (last_true_block != NULL) {
			vala_basic_block_connect (last_true_block, self->priv->current_block);
		}
		if (last_false_block != NULL) {
			vala_basic_block_connect (last_false_block, self->priv->current_block);
		}
	}
	(last_block == NULL ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL)));
	(last_true_block == NULL ? NULL : (last_true_block = (vala_basic_block_unref (last_true_block), NULL)));
	(last_false_block == NULL ? NULL : (last_false_block = (vala_basic_block_unref (last_false_block), NULL)));
}


static void vala_cfg_builder_real_visit_switch_statement (ValaCodeVisitor* base, ValaSwitchStatement* stmt) {
	ValaCFGBuilder * self;
	ValaBasicBlock* after_switch_block;
	ValaCFGBuilderJumpTarget* _tmp0;
	ValaBasicBlock* _tmp1;
	ValaBasicBlock* condition_block;
	gboolean has_default_label;
	GeeList* _tmp3;
	gboolean _tmp4;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	after_switch_block = vala_basic_block_new ();
	_tmp0 = NULL;
	gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp0 = vala_cfg_builder_jump_target_new_break_target (after_switch_block)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (vala_cfg_builder_jump_target_unref (_tmp0), NULL)));
	/* condition*/
	vala_basic_block_add_node (self->priv->current_block, ((ValaCodeNode*) (vala_switch_statement_get_expression (stmt))));
	_tmp1 = NULL;
	condition_block = (_tmp1 = self->priv->current_block, (_tmp1 == NULL ? NULL : vala_basic_block_ref (_tmp1)));
	vala_cfg_builder_handle_errors (self, ((ValaCodeNode*) (vala_switch_statement_get_expression (stmt))));
	has_default_label = FALSE;
	{
		GeeList* section_collection;
		int section_it;
		section_collection = vala_switch_statement_get_sections (stmt);
		for (section_it = 0; section_it < gee_collection_get_size (GEE_COLLECTION (section_collection)); section_it = section_it + 1) {
			ValaSwitchSection* section;
			section = ((ValaSwitchSection*) (gee_list_get (GEE_LIST (section_collection), section_it)));
			{
				ValaBasicBlock* _tmp2;
				_tmp2 = NULL;
				self->priv->current_block = (_tmp2 = vala_basic_block_new (), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp2);
				vala_basic_block_connect (condition_block, self->priv->current_block);
				{
					GeeList* stmt_collection;
					int stmt_it;
					stmt_collection = vala_block_get_statements (((ValaBlock*) (section)));
					for (stmt_it = 0; stmt_it < gee_collection_get_size (GEE_COLLECTION (stmt_collection)); stmt_it = stmt_it + 1) {
						ValaStatement* stmt;
						stmt = ((ValaStatement*) (gee_list_get (GEE_LIST (stmt_collection), stmt_it)));
						{
							vala_code_node_accept (((ValaCodeNode*) (stmt)), ((ValaCodeVisitor*) (self)));
							(stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)));
						}
					}
					(stmt_collection == NULL ? NULL : (stmt_collection = (gee_collection_object_unref (stmt_collection), NULL)));
				}
				if (vala_switch_section_has_default_label (section)) {
					has_default_label = TRUE;
				}
				if (self->priv->current_block != NULL) {
					/* end of switch section reachable
					 we don't allow fall-through*/
					vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (section))), "missing break statement at end of switch section");
					vala_code_node_set_error (((ValaCodeNode*) (section)), TRUE);
					vala_basic_block_connect (self->priv->current_block, after_switch_block);
				}
				(section == NULL ? NULL : (section = (vala_code_node_unref (section), NULL)));
			}
		}
		(section_collection == NULL ? NULL : (section_collection = (gee_collection_object_unref (section_collection), NULL)));
	}
	/* after switch
	 reachable?*/
	_tmp3 = NULL;
	if ((_tmp4 = !has_default_label || gee_collection_get_size (((GeeCollection*) ((_tmp3 = vala_basic_block_get_predecessors (after_switch_block))))) > 0, (_tmp3 == NULL ? NULL : (_tmp3 = (gee_collection_object_unref (_tmp3), NULL))), _tmp4)) {
		ValaBasicBlock* _tmp6;
		ValaBasicBlock* _tmp5;
		_tmp6 = NULL;
		_tmp5 = NULL;
		self->priv->current_block = (_tmp6 = (_tmp5 = after_switch_block, (_tmp5 == NULL ? NULL : vala_basic_block_ref (_tmp5))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp6);
	} else {
		ValaBasicBlock* _tmp7;
		_tmp7 = NULL;
		self->priv->current_block = (_tmp7 = NULL, (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp7);
		self->priv->unreachable_reported = FALSE;
	}
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1);
	(after_switch_block == NULL ? NULL : (after_switch_block = (vala_basic_block_unref (after_switch_block), NULL)));
	(condition_block == NULL ? NULL : (condition_block = (vala_basic_block_unref (condition_block), NULL)));
}


static void vala_cfg_builder_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt) {
	ValaCFGBuilder * self;
	ValaBasicBlock* condition_block;
	ValaCFGBuilderJumpTarget* _tmp0;
	ValaBasicBlock* after_loop_block;
	ValaCFGBuilderJumpTarget* _tmp1;
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* last_block;
	ValaBasicBlock* _tmp4;
	ValaBasicBlock* _tmp3;
	ValaBasicBlock* _tmp5;
	ValaBasicBlock* _tmp7;
	ValaBasicBlock* _tmp6;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	condition_block = vala_basic_block_new ();
	_tmp0 = NULL;
	gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp0 = vala_cfg_builder_jump_target_new_continue_target (condition_block)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (vala_cfg_builder_jump_target_unref (_tmp0), NULL)));
	after_loop_block = vala_basic_block_new ();
	_tmp1 = NULL;
	gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp1 = vala_cfg_builder_jump_target_new_break_target (after_loop_block)));
	(_tmp1 == NULL ? NULL : (_tmp1 = (vala_cfg_builder_jump_target_unref (_tmp1), NULL)));
	/* condition*/
	_tmp2 = NULL;
	last_block = (_tmp2 = self->priv->current_block, (_tmp2 == NULL ? NULL : vala_basic_block_ref (_tmp2)));
	vala_basic_block_connect (last_block, condition_block);
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_block = (_tmp4 = (_tmp3 = condition_block, (_tmp3 == NULL ? NULL : vala_basic_block_ref (_tmp3))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp4);
	vala_basic_block_add_node (self->priv->current_block, ((ValaCodeNode*) (vala_while_statement_get_condition (stmt))));
	vala_cfg_builder_handle_errors (self, ((ValaCodeNode*) (vala_while_statement_get_condition (stmt))));
	/* loop block*/
	_tmp5 = NULL;
	self->priv->current_block = (_tmp5 = vala_basic_block_new (), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp5);
	vala_basic_block_connect (condition_block, self->priv->current_block);
	vala_code_node_accept (((ValaCodeNode*) (vala_while_statement_get_body (stmt))), ((ValaCodeVisitor*) (self)));
	/* end of loop block reachable?*/
	if (self->priv->current_block != NULL) {
		vala_basic_block_connect (self->priv->current_block, condition_block);
	}
	/* after loop*/
	vala_basic_block_connect (condition_block, after_loop_block);
	_tmp7 = NULL;
	_tmp6 = NULL;
	self->priv->current_block = (_tmp7 = (_tmp6 = after_loop_block, (_tmp6 == NULL ? NULL : vala_basic_block_ref (_tmp6))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp7);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1);
	(condition_block == NULL ? NULL : (condition_block = (vala_basic_block_unref (condition_block), NULL)));
	(after_loop_block == NULL ? NULL : (after_loop_block = (vala_basic_block_unref (after_loop_block), NULL)));
	(last_block == NULL ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL)));
}


static void vala_cfg_builder_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt) {
	ValaCFGBuilder * self;
	ValaBasicBlock* condition_block;
	ValaCFGBuilderJumpTarget* _tmp0;
	ValaBasicBlock* after_loop_block;
	ValaCFGBuilderJumpTarget* _tmp1;
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* last_block;
	ValaBasicBlock* loop_block;
	ValaBasicBlock* _tmp4;
	ValaBasicBlock* _tmp3;
	GeeList* _tmp5;
	gboolean _tmp6;
	GeeList* _tmp11;
	gboolean _tmp12;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	condition_block = vala_basic_block_new ();
	_tmp0 = NULL;
	gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp0 = vala_cfg_builder_jump_target_new_continue_target (condition_block)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (vala_cfg_builder_jump_target_unref (_tmp0), NULL)));
	after_loop_block = vala_basic_block_new ();
	_tmp1 = NULL;
	gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp1 = vala_cfg_builder_jump_target_new_break_target (after_loop_block)));
	(_tmp1 == NULL ? NULL : (_tmp1 = (vala_cfg_builder_jump_target_unref (_tmp1), NULL)));
	/* loop block*/
	_tmp2 = NULL;
	last_block = (_tmp2 = self->priv->current_block, (_tmp2 == NULL ? NULL : vala_basic_block_ref (_tmp2)));
	loop_block = vala_basic_block_new ();
	vala_basic_block_connect (last_block, loop_block);
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_block = (_tmp4 = (_tmp3 = loop_block, (_tmp3 == NULL ? NULL : vala_basic_block_ref (_tmp3))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp4);
	vala_code_node_accept (((ValaCodeNode*) (vala_do_statement_get_body (stmt))), ((ValaCodeVisitor*) (self)));
	/* condition
	 reachable?*/
	_tmp5 = NULL;
	if ((_tmp6 = self->priv->current_block != NULL || gee_collection_get_size (((GeeCollection*) ((_tmp5 = vala_basic_block_get_predecessors (condition_block))))) > 0, (_tmp5 == NULL ? NULL : (_tmp5 = (gee_collection_object_unref (_tmp5), NULL))), _tmp6)) {
		ValaBasicBlock* _tmp10;
		ValaBasicBlock* _tmp9;
		if (self->priv->current_block != NULL) {
			ValaBasicBlock* _tmp8;
			ValaBasicBlock* _tmp7;
			_tmp8 = NULL;
			_tmp7 = NULL;
			last_block = (_tmp8 = (_tmp7 = self->priv->current_block, (_tmp7 == NULL ? NULL : vala_basic_block_ref (_tmp7))), (last_block == NULL ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL))), _tmp8);
			vala_basic_block_connect (last_block, condition_block);
		}
		vala_basic_block_add_node (condition_block, ((ValaCodeNode*) (vala_do_statement_get_condition (stmt))));
		vala_basic_block_connect (condition_block, loop_block);
		_tmp10 = NULL;
		_tmp9 = NULL;
		self->priv->current_block = (_tmp10 = (_tmp9 = condition_block, (_tmp9 == NULL ? NULL : vala_basic_block_ref (_tmp9))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp10);
		vala_cfg_builder_handle_errors (self, ((ValaCodeNode*) (vala_do_statement_get_condition (stmt))));
	}
	/* after loop
	 reachable?*/
	_tmp11 = NULL;
	if ((_tmp12 = self->priv->current_block != NULL || gee_collection_get_size (((GeeCollection*) ((_tmp11 = vala_basic_block_get_predecessors (after_loop_block))))) > 0, (_tmp11 == NULL ? NULL : (_tmp11 = (gee_collection_object_unref (_tmp11), NULL))), _tmp12)) {
		ValaBasicBlock* _tmp16;
		ValaBasicBlock* _tmp15;
		if (self->priv->current_block != NULL) {
			ValaBasicBlock* _tmp14;
			ValaBasicBlock* _tmp13;
			_tmp14 = NULL;
			_tmp13 = NULL;
			last_block = (_tmp14 = (_tmp13 = self->priv->current_block, (_tmp13 == NULL ? NULL : vala_basic_block_ref (_tmp13))), (last_block == NULL ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL))), _tmp14);
			vala_basic_block_connect (last_block, after_loop_block);
		}
		_tmp16 = NULL;
		_tmp15 = NULL;
		self->priv->current_block = (_tmp16 = (_tmp15 = after_loop_block, (_tmp15 == NULL ? NULL : vala_basic_block_ref (_tmp15))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp16);
	}
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1);
	(condition_block == NULL ? NULL : (condition_block = (vala_basic_block_unref (condition_block), NULL)));
	(after_loop_block == NULL ? NULL : (after_loop_block = (vala_basic_block_unref (after_loop_block), NULL)));
	(last_block == NULL ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL)));
	(loop_block == NULL ? NULL : (loop_block = (vala_basic_block_unref (loop_block), NULL)));
}


static void vala_cfg_builder_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt) {
	ValaCFGBuilder * self;
	ValaBasicBlock* iterator_block;
	ValaCFGBuilderJumpTarget* _tmp0;
	ValaBasicBlock* after_loop_block;
	ValaCFGBuilderJumpTarget* _tmp1;
	ValaBasicBlock* condition_block;
	ValaBasicBlock* _tmp3;
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp4;
	GeeList* _tmp5;
	gboolean _tmp6;
	ValaBasicBlock* _tmp10;
	ValaBasicBlock* _tmp9;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	/* initializer*/
	{
		GeeList* init_expr_collection;
		int init_expr_it;
		init_expr_collection = vala_for_statement_get_initializer (stmt);
		for (init_expr_it = 0; init_expr_it < gee_collection_get_size (GEE_COLLECTION (init_expr_collection)); init_expr_it = init_expr_it + 1) {
			ValaExpression* init_expr;
			init_expr = ((ValaExpression*) (gee_list_get (GEE_LIST (init_expr_collection), init_expr_it)));
			{
				vala_basic_block_add_node (self->priv->current_block, ((ValaCodeNode*) (init_expr)));
				vala_cfg_builder_handle_errors (self, ((ValaCodeNode*) (init_expr)));
				(init_expr == NULL ? NULL : (init_expr = (vala_code_node_unref (init_expr), NULL)));
			}
		}
		(init_expr_collection == NULL ? NULL : (init_expr_collection = (gee_collection_object_unref (init_expr_collection), NULL)));
	}
	iterator_block = vala_basic_block_new ();
	_tmp0 = NULL;
	gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp0 = vala_cfg_builder_jump_target_new_continue_target (iterator_block)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (vala_cfg_builder_jump_target_unref (_tmp0), NULL)));
	after_loop_block = vala_basic_block_new ();
	_tmp1 = NULL;
	gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp1 = vala_cfg_builder_jump_target_new_break_target (after_loop_block)));
	(_tmp1 == NULL ? NULL : (_tmp1 = (vala_cfg_builder_jump_target_unref (_tmp1), NULL)));
	/* condition*/
	condition_block = vala_basic_block_new ();
	vala_basic_block_connect (self->priv->current_block, condition_block);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_block = (_tmp3 = (_tmp2 = condition_block, (_tmp2 == NULL ? NULL : vala_basic_block_ref (_tmp2))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp3);
	if (vala_for_statement_get_condition (stmt) != NULL) {
		vala_basic_block_add_node (self->priv->current_block, ((ValaCodeNode*) (vala_for_statement_get_condition (stmt))));
	}
	if (vala_for_statement_get_condition (stmt) != NULL) {
		vala_cfg_builder_handle_errors (self, ((ValaCodeNode*) (vala_for_statement_get_condition (stmt))));
	}
	/* loop block*/
	_tmp4 = NULL;
	self->priv->current_block = (_tmp4 = vala_basic_block_new (), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp4);
	vala_basic_block_connect (condition_block, self->priv->current_block);
	vala_code_node_accept (((ValaCodeNode*) (vala_for_statement_get_body (stmt))), ((ValaCodeVisitor*) (self)));
	/* iterator
	 reachable?*/
	_tmp5 = NULL;
	if ((_tmp6 = self->priv->current_block != NULL || gee_collection_get_size (((GeeCollection*) ((_tmp5 = vala_basic_block_get_predecessors (iterator_block))))) > 0, (_tmp5 == NULL ? NULL : (_tmp5 = (gee_collection_object_unref (_tmp5), NULL))), _tmp6)) {
		ValaBasicBlock* _tmp8;
		ValaBasicBlock* _tmp7;
		if (self->priv->current_block != NULL) {
			vala_basic_block_connect (self->priv->current_block, iterator_block);
		}
		_tmp8 = NULL;
		_tmp7 = NULL;
		self->priv->current_block = (_tmp8 = (_tmp7 = iterator_block, (_tmp7 == NULL ? NULL : vala_basic_block_ref (_tmp7))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp8);
		{
			GeeList* it_expr_collection;
			int it_expr_it;
			it_expr_collection = vala_for_statement_get_iterator (stmt);
			for (it_expr_it = 0; it_expr_it < gee_collection_get_size (GEE_COLLECTION (it_expr_collection)); it_expr_it = it_expr_it + 1) {
				ValaExpression* it_expr;
				it_expr = ((ValaExpression*) (gee_list_get (GEE_LIST (it_expr_collection), it_expr_it)));
				{
					vala_basic_block_add_node (self->priv->current_block, ((ValaCodeNode*) (it_expr)));
					vala_cfg_builder_handle_errors (self, ((ValaCodeNode*) (it_expr)));
					(it_expr == NULL ? NULL : (it_expr = (vala_code_node_unref (it_expr), NULL)));
				}
			}
			(it_expr_collection == NULL ? NULL : (it_expr_collection = (gee_collection_object_unref (it_expr_collection), NULL)));
		}
		vala_basic_block_connect (self->priv->current_block, condition_block);
	}
	/* after loop*/
	vala_basic_block_connect (condition_block, after_loop_block);
	_tmp10 = NULL;
	_tmp9 = NULL;
	self->priv->current_block = (_tmp10 = (_tmp9 = after_loop_block, (_tmp9 == NULL ? NULL : vala_basic_block_ref (_tmp9))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp10);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1);
	(iterator_block == NULL ? NULL : (iterator_block = (vala_basic_block_unref (iterator_block), NULL)));
	(after_loop_block == NULL ? NULL : (after_loop_block = (vala_basic_block_unref (after_loop_block), NULL)));
	(condition_block == NULL ? NULL : (condition_block = (vala_basic_block_unref (condition_block), NULL)));
}


static void vala_cfg_builder_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt) {
	ValaCFGBuilder * self;
	ValaBasicBlock* loop_block;
	ValaCFGBuilderJumpTarget* _tmp0;
	ValaBasicBlock* after_loop_block;
	ValaCFGBuilderJumpTarget* _tmp1;
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* last_block;
	ValaBasicBlock* _tmp4;
	ValaBasicBlock* _tmp3;
	ValaBasicBlock* _tmp6;
	ValaBasicBlock* _tmp5;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	/* collection*/
	vala_basic_block_add_node (self->priv->current_block, ((ValaCodeNode*) (vala_foreach_statement_get_collection (stmt))));
	vala_cfg_builder_handle_errors (self, ((ValaCodeNode*) (vala_foreach_statement_get_collection (stmt))));
	loop_block = vala_basic_block_new ();
	_tmp0 = NULL;
	gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp0 = vala_cfg_builder_jump_target_new_continue_target (loop_block)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (vala_cfg_builder_jump_target_unref (_tmp0), NULL)));
	after_loop_block = vala_basic_block_new ();
	_tmp1 = NULL;
	gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp1 = vala_cfg_builder_jump_target_new_break_target (after_loop_block)));
	(_tmp1 == NULL ? NULL : (_tmp1 = (vala_cfg_builder_jump_target_unref (_tmp1), NULL)));
	/* loop block*/
	_tmp2 = NULL;
	last_block = (_tmp2 = self->priv->current_block, (_tmp2 == NULL ? NULL : vala_basic_block_ref (_tmp2)));
	vala_basic_block_connect (last_block, loop_block);
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_block = (_tmp4 = (_tmp3 = loop_block, (_tmp3 == NULL ? NULL : vala_basic_block_ref (_tmp3))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp4);
	vala_code_node_accept (((ValaCodeNode*) (vala_foreach_statement_get_body (stmt))), ((ValaCodeVisitor*) (self)));
	if (self->priv->current_block != NULL) {
		vala_basic_block_connect (self->priv->current_block, loop_block);
	}
	/* after loop*/
	vala_basic_block_connect (last_block, after_loop_block);
	if (self->priv->current_block != NULL) {
		vala_basic_block_connect (self->priv->current_block, after_loop_block);
	}
	_tmp6 = NULL;
	_tmp5 = NULL;
	self->priv->current_block = (_tmp6 = (_tmp5 = after_loop_block, (_tmp5 == NULL ? NULL : vala_basic_block_ref (_tmp5))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp6);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1);
	gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1);
	(loop_block == NULL ? NULL : (loop_block = (vala_basic_block_unref (loop_block), NULL)));
	(after_loop_block == NULL ? NULL : (after_loop_block = (vala_basic_block_unref (after_loop_block), NULL)));
	(last_block == NULL ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL)));
}


static void vala_cfg_builder_real_visit_break_statement (ValaCodeVisitor* base, ValaBreakStatement* stmt) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	vala_basic_block_add_node (self->priv->current_block, ((ValaCodeNode*) (stmt)));
	{
		gint i;
		i = gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1;
		for (; i >= 0; i--) {
			ValaCFGBuilderJumpTarget* jump_target;
			jump_target = ((ValaCFGBuilderJumpTarget*) (gee_list_get (((GeeList*) (self->priv->jump_stack)), i)));
			if (vala_cfg_builder_jump_target_get_is_break_target (jump_target)) {
				ValaBasicBlock* _tmp0;
				vala_basic_block_connect (self->priv->current_block, vala_cfg_builder_jump_target_get_basic_block (jump_target));
				_tmp0 = NULL;
				self->priv->current_block = (_tmp0 = NULL, (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp0);
				self->priv->unreachable_reported = FALSE;
				(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
				return;
			} else {
				if (vala_cfg_builder_jump_target_get_is_finally_clause (jump_target)) {
					ValaBasicBlock* _tmp2;
					ValaBasicBlock* _tmp1;
					vala_basic_block_connect (self->priv->current_block, vala_cfg_builder_jump_target_get_basic_block (jump_target));
					_tmp2 = NULL;
					_tmp1 = NULL;
					self->priv->current_block = (_tmp2 = (_tmp1 = vala_cfg_builder_jump_target_get_last_block (jump_target), (_tmp1 == NULL ? NULL : vala_basic_block_ref (_tmp1))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp2);
				}
			}
			(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
		}
	}
	vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (stmt))), "no enclosing loop or switch statement found");
	vala_code_node_set_error (((ValaCodeNode*) (stmt)), TRUE);
}


static void vala_cfg_builder_real_visit_continue_statement (ValaCodeVisitor* base, ValaContinueStatement* stmt) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	vala_basic_block_add_node (self->priv->current_block, ((ValaCodeNode*) (stmt)));
	{
		gint i;
		i = gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1;
		for (; i >= 0; i--) {
			ValaCFGBuilderJumpTarget* jump_target;
			jump_target = ((ValaCFGBuilderJumpTarget*) (gee_list_get (((GeeList*) (self->priv->jump_stack)), i)));
			if (vala_cfg_builder_jump_target_get_is_continue_target (jump_target)) {
				ValaBasicBlock* _tmp0;
				vala_basic_block_connect (self->priv->current_block, vala_cfg_builder_jump_target_get_basic_block (jump_target));
				_tmp0 = NULL;
				self->priv->current_block = (_tmp0 = NULL, (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp0);
				self->priv->unreachable_reported = FALSE;
				(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
				return;
			} else {
				if (vala_cfg_builder_jump_target_get_is_finally_clause (jump_target)) {
					ValaBasicBlock* _tmp2;
					ValaBasicBlock* _tmp1;
					vala_basic_block_connect (self->priv->current_block, vala_cfg_builder_jump_target_get_basic_block (jump_target));
					_tmp2 = NULL;
					_tmp1 = NULL;
					self->priv->current_block = (_tmp2 = (_tmp1 = vala_cfg_builder_jump_target_get_last_block (jump_target), (_tmp1 == NULL ? NULL : vala_basic_block_ref (_tmp1))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp2);
				}
			}
			(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
		}
	}
	vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (stmt))), "no enclosing loop found");
	vala_code_node_set_error (((ValaCodeNode*) (stmt)), TRUE);
}


static void vala_cfg_builder_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	vala_basic_block_add_node (self->priv->current_block, ((ValaCodeNode*) (stmt)));
	{
		gint i;
		i = gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1;
		for (; i >= 0; i--) {
			ValaCFGBuilderJumpTarget* jump_target;
			jump_target = ((ValaCFGBuilderJumpTarget*) (gee_list_get (((GeeList*) (self->priv->jump_stack)), i)));
			if (vala_cfg_builder_jump_target_get_is_return_target (jump_target)) {
				ValaBasicBlock* _tmp0;
				vala_basic_block_connect (self->priv->current_block, vala_cfg_builder_jump_target_get_basic_block (jump_target));
				_tmp0 = NULL;
				self->priv->current_block = (_tmp0 = NULL, (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp0);
				self->priv->unreachable_reported = FALSE;
				(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
				return;
			} else {
				if (vala_cfg_builder_jump_target_get_is_finally_clause (jump_target)) {
					ValaBasicBlock* _tmp2;
					ValaBasicBlock* _tmp1;
					vala_basic_block_connect (self->priv->current_block, vala_cfg_builder_jump_target_get_basic_block (jump_target));
					_tmp2 = NULL;
					_tmp1 = NULL;
					self->priv->current_block = (_tmp2 = (_tmp1 = vala_cfg_builder_jump_target_get_last_block (jump_target), (_tmp1 == NULL ? NULL : vala_basic_block_ref (_tmp1))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp2);
				}
			}
			(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
		}
	}
	vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (stmt))), "no enclosing loop found");
	vala_code_node_set_error (((ValaCodeNode*) (stmt)), TRUE);
}


static void vala_cfg_builder_handle_errors (ValaCFGBuilder* self, ValaCodeNode* node) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	if (vala_code_node_get_tree_can_fail (node)) {
		ValaBasicBlock* _tmp0;
		ValaBasicBlock* last_block;
		ValaBasicBlock* _tmp5;
		_tmp0 = NULL;
		last_block = (_tmp0 = self->priv->current_block, (_tmp0 == NULL ? NULL : vala_basic_block_ref (_tmp0)));
		{
			gint i;
			/* exceptional control flow*/
			i = gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1;
			for (; i >= 0; i--) {
				ValaCFGBuilderJumpTarget* jump_target;
				jump_target = ((ValaCFGBuilderJumpTarget*) (gee_list_get (((GeeList*) (self->priv->jump_stack)), i)));
				if (vala_cfg_builder_jump_target_get_is_return_target (jump_target)) {
					ValaBasicBlock* _tmp1;
					vala_basic_block_connect (self->priv->current_block, vala_cfg_builder_jump_target_get_basic_block (jump_target));
					_tmp1 = NULL;
					self->priv->current_block = (_tmp1 = NULL, (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp1);
					self->priv->unreachable_reported = FALSE;
					(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
					break;
				} else {
					if (vala_cfg_builder_jump_target_get_is_error_target (jump_target)) {
						/* TODO check whether jump target catches node.error_type*/
						vala_basic_block_connect (self->priv->current_block, vala_cfg_builder_jump_target_get_basic_block (jump_target));
						if (vala_cfg_builder_jump_target_get_error_domain (jump_target) == NULL) {
							ValaBasicBlock* _tmp2;
							/* catch all clause*/
							_tmp2 = NULL;
							self->priv->current_block = (_tmp2 = NULL, (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp2);
							self->priv->unreachable_reported = FALSE;
							(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
							break;
						}
					} else {
						if (vala_cfg_builder_jump_target_get_is_finally_clause (jump_target)) {
							ValaBasicBlock* _tmp4;
							ValaBasicBlock* _tmp3;
							vala_basic_block_connect (self->priv->current_block, vala_cfg_builder_jump_target_get_basic_block (jump_target));
							_tmp4 = NULL;
							_tmp3 = NULL;
							self->priv->current_block = (_tmp4 = (_tmp3 = vala_cfg_builder_jump_target_get_last_block (jump_target), (_tmp3 == NULL ? NULL : vala_basic_block_ref (_tmp3))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp4);
						}
					}
				}
				(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
			}
		}
		/* normal control flow*/
		_tmp5 = NULL;
		self->priv->current_block = (_tmp5 = vala_basic_block_new (), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp5);
		vala_basic_block_connect (last_block, self->priv->current_block);
		(last_block == NULL ? NULL : (last_block = (vala_basic_block_unref (last_block), NULL)));
	}
}


static void vala_cfg_builder_real_visit_yield_statement (ValaCodeVisitor* base, ValaYieldStatement* stmt) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	vala_code_node_accept_children (((ValaCodeNode*) (stmt)), ((ValaCodeVisitor*) (self)));
}


static void vala_cfg_builder_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	vala_basic_block_add_node (self->priv->current_block, ((ValaCodeNode*) (stmt)));
	{
		gint i;
		i = gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1;
		for (; i >= 0; i--) {
			ValaCFGBuilderJumpTarget* jump_target;
			jump_target = ((ValaCFGBuilderJumpTarget*) (gee_list_get (((GeeList*) (self->priv->jump_stack)), i)));
			if (vala_cfg_builder_jump_target_get_is_return_target (jump_target)) {
				ValaBasicBlock* _tmp0;
				vala_basic_block_connect (self->priv->current_block, vala_cfg_builder_jump_target_get_basic_block (jump_target));
				_tmp0 = NULL;
				self->priv->current_block = (_tmp0 = NULL, (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp0);
				self->priv->unreachable_reported = FALSE;
				(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
				return;
			} else {
				if (vala_cfg_builder_jump_target_get_is_error_target (jump_target)) {
					/* TODO check whether jump target catches stmt.error_type*/
					vala_basic_block_connect (self->priv->current_block, vala_cfg_builder_jump_target_get_basic_block (jump_target));
					if (vala_cfg_builder_jump_target_get_error_domain (jump_target) == NULL) {
						ValaBasicBlock* _tmp1;
						_tmp1 = NULL;
						self->priv->current_block = (_tmp1 = NULL, (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp1);
						self->priv->unreachable_reported = FALSE;
						(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
						return;
					}
				} else {
					if (vala_cfg_builder_jump_target_get_is_finally_clause (jump_target)) {
						ValaBasicBlock* _tmp3;
						ValaBasicBlock* _tmp2;
						vala_basic_block_connect (self->priv->current_block, vala_cfg_builder_jump_target_get_basic_block (jump_target));
						_tmp3 = NULL;
						_tmp2 = NULL;
						self->priv->current_block = (_tmp3 = (_tmp2 = vala_cfg_builder_jump_target_get_last_block (jump_target), (_tmp2 == NULL ? NULL : vala_basic_block_ref (_tmp2))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp3);
					}
				}
			}
			(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
		}
	}
	g_assert_not_reached ();
}


static void vala_cfg_builder_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt) {
	ValaCFGBuilder * self;
	ValaBasicBlock* after_try_block;
	ValaBasicBlock* finally_block;
	gint finally_jump_stack_size;
	GeeList* catch_clauses;
	GeeList* catch_stack;
	GeeList* _tmp17;
	gboolean _tmp18;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	after_try_block = vala_basic_block_new ();
	finally_block = NULL;
	if (vala_try_statement_get_finally_body (stmt) != NULL) {
		ValaBasicBlock* _tmp0;
		ValaBasicBlock* _tmp2;
		ValaBasicBlock* _tmp1;
		ValaCFGBuilderJumpTarget* _tmp3;
		_tmp0 = NULL;
		finally_block = (_tmp0 = vala_basic_block_new (), (finally_block == NULL ? NULL : (finally_block = (vala_basic_block_unref (finally_block), NULL))), _tmp0);
		_tmp2 = NULL;
		_tmp1 = NULL;
		self->priv->current_block = (_tmp2 = (_tmp1 = finally_block, (_tmp1 == NULL ? NULL : vala_basic_block_ref (_tmp1))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp2);
		vala_code_node_accept (((ValaCodeNode*) (vala_try_statement_get_finally_body (stmt))), ((ValaCodeVisitor*) (self)));
		if (self->priv->current_block == NULL) {
			/* don't allow finally blocks with e.g. return statements*/
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (stmt))), "end of finally block not reachable");
			vala_code_node_set_error (((ValaCodeNode*) (stmt)), TRUE);
			(after_try_block == NULL ? NULL : (after_try_block = (vala_basic_block_unref (after_try_block), NULL)));
			(finally_block == NULL ? NULL : (finally_block = (vala_basic_block_unref (finally_block), NULL)));
			return;
		}
		_tmp3 = NULL;
		gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp3 = vala_cfg_builder_jump_target_new_finally_clause (finally_block, self->priv->current_block)));
		(_tmp3 == NULL ? NULL : (_tmp3 = (vala_cfg_builder_jump_target_unref (_tmp3), NULL)));
	}
	finally_jump_stack_size = gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack)));
	catch_clauses = vala_try_statement_get_catch_clauses (stmt);
	{
		gint i;
		i = gee_collection_get_size (((GeeCollection*) (catch_clauses))) - 1;
		for (; i >= 0; i--) {
			ValaCatchClause* catch_clause;
			catch_clause = ((ValaCatchClause*) (gee_list_get (((GeeList*) (catch_clauses)), i)));
			if (vala_catch_clause_get_error_type (catch_clause) != NULL) {
				ValaCFGBuilderJumpTarget* _tmp6;
				ValaBasicBlock* _tmp4;
				ValaTypeSymbol* _tmp5;
				_tmp6 = NULL;
				_tmp4 = NULL;
				_tmp5 = NULL;
				gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp6 = vala_cfg_builder_jump_target_new_error_target ((_tmp4 = vala_basic_block_new ()), catch_clause, (_tmp5 = vala_data_type_get_data_type (vala_catch_clause_get_error_type (catch_clause)), (VALA_IS_ERROR_DOMAIN (_tmp5) ? ((ValaErrorDomain*) (_tmp5)) : NULL)), NULL)));
				(_tmp6 == NULL ? NULL : (_tmp6 = (vala_cfg_builder_jump_target_unref (_tmp6), NULL)));
				(_tmp4 == NULL ? NULL : (_tmp4 = (vala_basic_block_unref (_tmp4), NULL)));
			} else {
				ValaCFGBuilderJumpTarget* _tmp8;
				ValaBasicBlock* _tmp7;
				_tmp8 = NULL;
				_tmp7 = NULL;
				gee_collection_add (((GeeCollection*) (self->priv->jump_stack)), (_tmp8 = vala_cfg_builder_jump_target_new_error_target ((_tmp7 = vala_basic_block_new ()), catch_clause, NULL, NULL)));
				(_tmp8 == NULL ? NULL : (_tmp8 = (vala_cfg_builder_jump_target_unref (_tmp8), NULL)));
				(_tmp7 == NULL ? NULL : (_tmp7 = (vala_basic_block_unref (_tmp7), NULL)));
			}
			(catch_clause == NULL ? NULL : (catch_clause = (vala_code_node_unref (catch_clause), NULL)));
		}
	}
	vala_code_node_accept (((ValaCodeNode*) (vala_try_statement_get_body (stmt))), ((ValaCodeVisitor*) (self)));
	if (self->priv->current_block != NULL) {
		if (finally_block != NULL) {
			ValaBasicBlock* _tmp10;
			ValaBasicBlock* _tmp9;
			vala_basic_block_connect (self->priv->current_block, finally_block);
			_tmp10 = NULL;
			_tmp9 = NULL;
			self->priv->current_block = (_tmp10 = (_tmp9 = finally_block, (_tmp9 == NULL ? NULL : vala_basic_block_ref (_tmp9))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp10);
		}
		vala_basic_block_connect (self->priv->current_block, after_try_block);
	}
	/* remove catch clauses from jump stack*/
	catch_stack = ((GeeList*) (gee_array_list_new (VALA_CFG_BUILDER_TYPE_JUMP_TARGET, ((GBoxedCopyFunc) (vala_cfg_builder_jump_target_ref)), vala_cfg_builder_jump_target_unref, g_direct_equal)));
	{
		gint i;
		i = gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1;
		for (; i >= finally_jump_stack_size; i--) {
			ValaCFGBuilderJumpTarget* jump_target;
			jump_target = ((ValaCFGBuilderJumpTarget*) (gee_list_get (((GeeList*) (self->priv->jump_stack)), i)));
			gee_collection_add (((GeeCollection*) (catch_stack)), jump_target);
			gee_list_remove_at (self->priv->jump_stack, i);
			(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
		}
	}
	{
		GeeList* jump_target_collection;
		int jump_target_it;
		jump_target_collection = catch_stack;
		for (jump_target_it = 0; jump_target_it < gee_collection_get_size (GEE_COLLECTION (jump_target_collection)); jump_target_it = jump_target_it + 1) {
			ValaCFGBuilderJumpTarget* jump_target;
			jump_target = ((ValaCFGBuilderJumpTarget*) (gee_list_get (GEE_LIST (jump_target_collection), jump_target_it)));
			{
				GeeList* _tmp11;
				gboolean _tmp12;
				_tmp11 = NULL;
				if ((_tmp12 = gee_collection_get_size (((GeeCollection*) ((_tmp11 = vala_basic_block_get_predecessors (vala_cfg_builder_jump_target_get_basic_block (jump_target)))))) == 0, (_tmp11 == NULL ? NULL : (_tmp11 = (gee_collection_object_unref (_tmp11), NULL))), _tmp12)) {
					/* unreachable*/
					vala_report_warning (vala_code_node_get_source_reference (((ValaCodeNode*) (vala_cfg_builder_jump_target_get_catch_clause (jump_target)))), "unreachable catch clause detected");
				} else {
					ValaBasicBlock* _tmp14;
					ValaBasicBlock* _tmp13;
					_tmp14 = NULL;
					_tmp13 = NULL;
					self->priv->current_block = (_tmp14 = (_tmp13 = vala_cfg_builder_jump_target_get_basic_block (jump_target), (_tmp13 == NULL ? NULL : vala_basic_block_ref (_tmp13))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp14);
					vala_code_node_accept (((ValaCodeNode*) (vala_catch_clause_get_body (vala_cfg_builder_jump_target_get_catch_clause (jump_target)))), ((ValaCodeVisitor*) (self)));
					if (self->priv->current_block != NULL) {
						if (finally_block != NULL) {
							ValaBasicBlock* _tmp16;
							ValaBasicBlock* _tmp15;
							vala_basic_block_connect (self->priv->current_block, finally_block);
							_tmp16 = NULL;
							_tmp15 = NULL;
							self->priv->current_block = (_tmp16 = (_tmp15 = finally_block, (_tmp15 == NULL ? NULL : vala_basic_block_ref (_tmp15))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp16);
						}
						vala_basic_block_connect (self->priv->current_block, after_try_block);
					}
				}
				(jump_target == NULL ? NULL : (jump_target = (vala_cfg_builder_jump_target_unref (jump_target), NULL)));
			}
		}
	}
	if (finally_block != NULL) {
		gee_list_remove_at (self->priv->jump_stack, gee_collection_get_size (((GeeCollection*) (self->priv->jump_stack))) - 1);
	}
	/* after try statement
	 reachable?*/
	_tmp17 = NULL;
	if ((_tmp18 = gee_collection_get_size (((GeeCollection*) ((_tmp17 = vala_basic_block_get_predecessors (after_try_block))))) > 0, (_tmp17 == NULL ? NULL : (_tmp17 = (gee_collection_object_unref (_tmp17), NULL))), _tmp18)) {
		ValaBasicBlock* _tmp20;
		ValaBasicBlock* _tmp19;
		_tmp20 = NULL;
		_tmp19 = NULL;
		self->priv->current_block = (_tmp20 = (_tmp19 = after_try_block, (_tmp19 == NULL ? NULL : vala_basic_block_ref (_tmp19))), (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp20);
	} else {
		ValaBasicBlock* _tmp21;
		_tmp21 = NULL;
		self->priv->current_block = (_tmp21 = NULL, (self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL))), _tmp21);
		self->priv->unreachable_reported = FALSE;
	}
	(after_try_block == NULL ? NULL : (after_try_block = (vala_basic_block_unref (after_try_block), NULL)));
	(finally_block == NULL ? NULL : (finally_block = (vala_basic_block_unref (finally_block), NULL)));
	(catch_clauses == NULL ? NULL : (catch_clauses = (gee_collection_object_unref (catch_clauses), NULL)));
	(catch_stack == NULL ? NULL : (catch_stack = (gee_collection_object_unref (catch_stack), NULL)));
}


static void vala_cfg_builder_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt) {
	ValaCFGBuilder * self;
	self = ((ValaCFGBuilder*) (base));
	g_return_if_fail (stmt != NULL);
	if (vala_cfg_builder_unreachable (self, ((ValaCodeNode*) (stmt)))) {
		return;
	}
	vala_code_node_accept (((ValaCodeNode*) (vala_lock_statement_get_body (stmt))), ((ValaCodeVisitor*) (self)));
}


static gboolean vala_cfg_builder_unreachable (ValaCFGBuilder* self, ValaCodeNode* node) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (node != NULL, FALSE);
	if (self->priv->current_block == NULL) {
		if (!self->priv->unreachable_reported) {
			vala_report_warning (vala_code_node_get_source_reference (node), "unreachable code detected");
			self->priv->unreachable_reported = TRUE;
		}
		return TRUE;
	}
	return FALSE;
}


static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_construct_break_target (GType object_type, ValaBasicBlock* basic_block) {
	ValaCFGBuilderJumpTarget* self;
	g_return_val_if_fail (basic_block != NULL, NULL);
	self = ((ValaCFGBuilderJumpTarget*) (g_type_create_instance (object_type)));
	vala_cfg_builder_jump_target_set_basic_block (self, basic_block);
	vala_cfg_builder_jump_target_set_is_break_target (self, TRUE);
	return self;
}


static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_new_break_target (ValaBasicBlock* basic_block) {
	return vala_cfg_builder_jump_target_construct_break_target (VALA_CFG_BUILDER_TYPE_JUMP_TARGET, basic_block);
}


static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_construct_continue_target (GType object_type, ValaBasicBlock* basic_block) {
	ValaCFGBuilderJumpTarget* self;
	g_return_val_if_fail (basic_block != NULL, NULL);
	self = ((ValaCFGBuilderJumpTarget*) (g_type_create_instance (object_type)));
	vala_cfg_builder_jump_target_set_basic_block (self, basic_block);
	vala_cfg_builder_jump_target_set_is_continue_target (self, TRUE);
	return self;
}


static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_new_continue_target (ValaBasicBlock* basic_block) {
	return vala_cfg_builder_jump_target_construct_continue_target (VALA_CFG_BUILDER_TYPE_JUMP_TARGET, basic_block);
}


static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_construct_return_target (GType object_type, ValaBasicBlock* basic_block) {
	ValaCFGBuilderJumpTarget* self;
	g_return_val_if_fail (basic_block != NULL, NULL);
	self = ((ValaCFGBuilderJumpTarget*) (g_type_create_instance (object_type)));
	vala_cfg_builder_jump_target_set_basic_block (self, basic_block);
	vala_cfg_builder_jump_target_set_is_return_target (self, TRUE);
	return self;
}


static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_new_return_target (ValaBasicBlock* basic_block) {
	return vala_cfg_builder_jump_target_construct_return_target (VALA_CFG_BUILDER_TYPE_JUMP_TARGET, basic_block);
}


static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_construct_error_target (GType object_type, ValaBasicBlock* basic_block, ValaCatchClause* catch_clause, ValaErrorDomain* error_domain, ValaErrorCode* error_code) {
	ValaCFGBuilderJumpTarget* self;
	g_return_val_if_fail (basic_block != NULL, NULL);
	g_return_val_if_fail (catch_clause != NULL, NULL);
	self = ((ValaCFGBuilderJumpTarget*) (g_type_create_instance (object_type)));
	vala_cfg_builder_jump_target_set_basic_block (self, basic_block);
	vala_cfg_builder_jump_target_set_catch_clause (self, catch_clause);
	vala_cfg_builder_jump_target_set_error_domain (self, error_domain);
	vala_cfg_builder_jump_target_set_error_code (self, error_code);
	vala_cfg_builder_jump_target_set_is_error_target (self, TRUE);
	return self;
}


static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_new_error_target (ValaBasicBlock* basic_block, ValaCatchClause* catch_clause, ValaErrorDomain* error_domain, ValaErrorCode* error_code) {
	return vala_cfg_builder_jump_target_construct_error_target (VALA_CFG_BUILDER_TYPE_JUMP_TARGET, basic_block, catch_clause, error_domain, error_code);
}


static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_construct_finally_clause (GType object_type, ValaBasicBlock* basic_block, ValaBasicBlock* last_block) {
	ValaCFGBuilderJumpTarget* self;
	g_return_val_if_fail (basic_block != NULL, NULL);
	g_return_val_if_fail (last_block != NULL, NULL);
	self = ((ValaCFGBuilderJumpTarget*) (g_type_create_instance (object_type)));
	vala_cfg_builder_jump_target_set_basic_block (self, basic_block);
	vala_cfg_builder_jump_target_set_last_block (self, last_block);
	vala_cfg_builder_jump_target_set_is_finally_clause (self, TRUE);
	return self;
}


static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_new_finally_clause (ValaBasicBlock* basic_block, ValaBasicBlock* last_block) {
	return vala_cfg_builder_jump_target_construct_finally_clause (VALA_CFG_BUILDER_TYPE_JUMP_TARGET, basic_block, last_block);
}


static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_construct (GType object_type) {
	ValaCFGBuilderJumpTarget* self;
	self = ((ValaCFGBuilderJumpTarget*) (g_type_create_instance (object_type)));
	return self;
}


static ValaCFGBuilderJumpTarget* vala_cfg_builder_jump_target_new (void) {
	return vala_cfg_builder_jump_target_construct (VALA_CFG_BUILDER_TYPE_JUMP_TARGET);
}


static gboolean vala_cfg_builder_jump_target_get_is_break_target (ValaCFGBuilderJumpTarget* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_break_target;
}


static void vala_cfg_builder_jump_target_set_is_break_target (ValaCFGBuilderJumpTarget* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_break_target = value;
}


static gboolean vala_cfg_builder_jump_target_get_is_continue_target (ValaCFGBuilderJumpTarget* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_continue_target;
}


static void vala_cfg_builder_jump_target_set_is_continue_target (ValaCFGBuilderJumpTarget* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_continue_target = value;
}


static gboolean vala_cfg_builder_jump_target_get_is_return_target (ValaCFGBuilderJumpTarget* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_return_target;
}


static void vala_cfg_builder_jump_target_set_is_return_target (ValaCFGBuilderJumpTarget* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_return_target = value;
}


static gboolean vala_cfg_builder_jump_target_get_is_error_target (ValaCFGBuilderJumpTarget* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_error_target;
}


static void vala_cfg_builder_jump_target_set_is_error_target (ValaCFGBuilderJumpTarget* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_error_target = value;
}


static ValaErrorDomain* vala_cfg_builder_jump_target_get_error_domain (ValaCFGBuilderJumpTarget* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_error_domain;
}


static void vala_cfg_builder_jump_target_set_error_domain (ValaCFGBuilderJumpTarget* self, ValaErrorDomain* value) {
	ValaErrorDomain* _tmp2;
	ValaErrorDomain* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_error_domain = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_error_domain == NULL ? NULL : (self->priv->_error_domain = (vala_code_node_unref (self->priv->_error_domain), NULL))), _tmp2);
}


static ValaErrorCode* vala_cfg_builder_jump_target_get_error_code (ValaCFGBuilderJumpTarget* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_error_code;
}


static void vala_cfg_builder_jump_target_set_error_code (ValaCFGBuilderJumpTarget* self, ValaErrorCode* value) {
	ValaErrorCode* _tmp2;
	ValaErrorCode* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_error_code = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_error_code == NULL ? NULL : (self->priv->_error_code = (vala_code_node_unref (self->priv->_error_code), NULL))), _tmp2);
}


static gboolean vala_cfg_builder_jump_target_get_is_finally_clause (ValaCFGBuilderJumpTarget* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_finally_clause;
}


static void vala_cfg_builder_jump_target_set_is_finally_clause (ValaCFGBuilderJumpTarget* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_finally_clause = value;
}


static ValaBasicBlock* vala_cfg_builder_jump_target_get_basic_block (ValaCFGBuilderJumpTarget* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_basic_block;
}


static void vala_cfg_builder_jump_target_set_basic_block (ValaCFGBuilderJumpTarget* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_basic_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_basic_block_ref (_tmp1))), (self->priv->_basic_block == NULL ? NULL : (self->priv->_basic_block = (vala_basic_block_unref (self->priv->_basic_block), NULL))), _tmp2);
}


static ValaBasicBlock* vala_cfg_builder_jump_target_get_last_block (ValaCFGBuilderJumpTarget* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_last_block;
}


static void vala_cfg_builder_jump_target_set_last_block (ValaCFGBuilderJumpTarget* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_last_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_basic_block_ref (_tmp1))), (self->priv->_last_block == NULL ? NULL : (self->priv->_last_block = (vala_basic_block_unref (self->priv->_last_block), NULL))), _tmp2);
}


static ValaCatchClause* vala_cfg_builder_jump_target_get_catch_clause (ValaCFGBuilderJumpTarget* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_catch_clause;
}


static void vala_cfg_builder_jump_target_set_catch_clause (ValaCFGBuilderJumpTarget* self, ValaCatchClause* value) {
	ValaCatchClause* _tmp2;
	ValaCatchClause* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_catch_clause = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_catch_clause == NULL ? NULL : (self->priv->_catch_clause = (vala_code_node_unref (self->priv->_catch_clause), NULL))), _tmp2);
}


static void vala_cfg_builder_value_jump_target_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_cfg_builder_value_jump_target_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_cfg_builder_jump_target_unref (value->data[0].v_pointer);
	}
}


static void vala_cfg_builder_value_jump_target_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_cfg_builder_jump_target_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_cfg_builder_value_jump_target_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_cfg_builder_value_jump_target_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaCFGBuilderJumpTarget* object;
		object = value->data[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_OBJECT_TYPE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", G_OBJECT_TYPE (object), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_cfg_builder_value_jump_target_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaCFGBuilderJumpTarget** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_cfg_builder_jump_target_ref (value->data[0].v_pointer);
	}
	return NULL;
}


static GParamSpec* vala_cfg_builder_param_spec_jump_target (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaCFGBuilderParamSpecJumpTarget* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_CFG_BUILDER_TYPE_JUMP_TARGET), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


static gpointer vala_cfg_builder_value_get_jump_target (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_CFG_BUILDER_TYPE_JUMP_TARGET), NULL);
	return value->data[0].v_pointer;
}


static void vala_cfg_builder_value_set_jump_target (GValue* value, gpointer v_object) {
	ValaCFGBuilderJumpTarget* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_CFG_BUILDER_TYPE_JUMP_TARGET));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_CFG_BUILDER_TYPE_JUMP_TARGET));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_cfg_builder_jump_target_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_cfg_builder_jump_target_unref (old);
	}
}


static void vala_cfg_builder_jump_target_class_init (ValaCFGBuilderJumpTargetClass * klass) {
	vala_cfg_builder_jump_target_parent_class = g_type_class_peek_parent (klass);
	VALA_CFG_BUILDER_JUMP_TARGET_CLASS (klass)->finalize = vala_cfg_builder_jump_target_finalize;
	g_type_class_add_private (klass, sizeof (ValaCFGBuilderJumpTargetPrivate));
}


static void vala_cfg_builder_jump_target_instance_init (ValaCFGBuilderJumpTarget * self) {
	self->priv = VALA_CFG_BUILDER_JUMP_TARGET_GET_PRIVATE (self);
	self->ref_count = 1;
}


static void vala_cfg_builder_jump_target_finalize (ValaCFGBuilderJumpTarget* obj) {
	ValaCFGBuilderJumpTarget * self;
	self = VALA_CFG_BUILDER_JUMP_TARGET (obj);
	(self->priv->_error_domain == NULL ? NULL : (self->priv->_error_domain = (vala_code_node_unref (self->priv->_error_domain), NULL)));
	(self->priv->_error_code == NULL ? NULL : (self->priv->_error_code = (vala_code_node_unref (self->priv->_error_code), NULL)));
	(self->priv->_basic_block == NULL ? NULL : (self->priv->_basic_block = (vala_basic_block_unref (self->priv->_basic_block), NULL)));
	(self->priv->_last_block == NULL ? NULL : (self->priv->_last_block = (vala_basic_block_unref (self->priv->_last_block), NULL)));
	(self->priv->_catch_clause == NULL ? NULL : (self->priv->_catch_clause = (vala_code_node_unref (self->priv->_catch_clause), NULL)));
}


static GType vala_cfg_builder_jump_target_get_type (void) {
	static GType vala_cfg_builder_jump_target_type_id = 0;
	if (vala_cfg_builder_jump_target_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_cfg_builder_value_jump_target_init, vala_cfg_builder_value_jump_target_free_value, vala_cfg_builder_value_jump_target_copy_value, vala_cfg_builder_value_jump_target_peek_pointer, "p", vala_cfg_builder_value_jump_target_collect_value, "p", vala_cfg_builder_value_jump_target_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaCFGBuilderJumpTargetClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_cfg_builder_jump_target_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCFGBuilderJumpTarget), 0, (GInstanceInitFunc) vala_cfg_builder_jump_target_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_cfg_builder_jump_target_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaCFGBuilderJumpTarget", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_cfg_builder_jump_target_type_id;
}


static gpointer vala_cfg_builder_jump_target_ref (gpointer instance) {
	ValaCFGBuilderJumpTarget* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


static void vala_cfg_builder_jump_target_unref (gpointer instance) {
	ValaCFGBuilderJumpTarget* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_CFG_BUILDER_JUMP_TARGET_GET_CLASS (self)->finalize (self);
		g_type_free_instance (((GTypeInstance *) (self)));
	}
}


static void vala_cfg_builder_class_init (ValaCFGBuilderClass * klass) {
	vala_cfg_builder_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_cfg_builder_finalize;
	g_type_class_add_private (klass, sizeof (ValaCFGBuilderPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_cfg_builder_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_cfg_builder_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_cfg_builder_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_cfg_builder_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_cfg_builder_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_domain = vala_cfg_builder_real_visit_error_domain;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_cfg_builder_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_cfg_builder_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_cfg_builder_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_block = vala_cfg_builder_real_visit_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_declaration_statement = vala_cfg_builder_real_visit_declaration_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_cfg_builder_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_cfg_builder_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_statement = vala_cfg_builder_real_visit_switch_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_cfg_builder_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_do_statement = vala_cfg_builder_real_visit_do_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_cfg_builder_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_foreach_statement = vala_cfg_builder_real_visit_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_break_statement = vala_cfg_builder_real_visit_break_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_continue_statement = vala_cfg_builder_real_visit_continue_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_return_statement = vala_cfg_builder_real_visit_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_yield_statement = vala_cfg_builder_real_visit_yield_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_cfg_builder_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_cfg_builder_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lock_statement = vala_cfg_builder_real_visit_lock_statement;
}


static void vala_cfg_builder_instance_init (ValaCFGBuilder * self) {
	self->priv = VALA_CFG_BUILDER_GET_PRIVATE (self);
	self->priv->jump_stack = ((GeeList*) (gee_array_list_new (VALA_CFG_BUILDER_TYPE_JUMP_TARGET, ((GBoxedCopyFunc) (vala_cfg_builder_jump_target_ref)), vala_cfg_builder_jump_target_unref, g_direct_equal)));
}


static void vala_cfg_builder_finalize (ValaCodeVisitor* obj) {
	ValaCFGBuilder * self;
	self = VALA_CFG_BUILDER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)));
	(self->priv->current_block == NULL ? NULL : (self->priv->current_block = (vala_basic_block_unref (self->priv->current_block), NULL)));
	(self->priv->jump_stack == NULL ? NULL : (self->priv->jump_stack = (gee_collection_object_unref (self->priv->jump_stack), NULL)));
	VALA_CODE_VISITOR_CLASS (vala_cfg_builder_parent_class)->finalize (obj);
}


GType vala_cfg_builder_get_type (void) {
	static GType vala_cfg_builder_type_id = 0;
	if (vala_cfg_builder_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCFGBuilderClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_cfg_builder_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCFGBuilder), 0, (GInstanceInitFunc) vala_cfg_builder_instance_init, NULL };
		vala_cfg_builder_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaCFGBuilder", &g_define_type_info, 0);
	}
	return vala_cfg_builder_type_id;
}




