/* valaswitchstatement.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaswitchstatement.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valaexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valaswitchsection.h>
#include <vala/valacodevisitor.h>




struct _ValaSwitchStatementPrivate {
	ValaExpression* _expression;
	GeeList* sections;
};

#define VALA_SWITCH_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SWITCH_STATEMENT, ValaSwitchStatementPrivate))
enum  {
	VALA_SWITCH_STATEMENT_DUMMY_PROPERTY
};
static void vala_switch_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_switch_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gpointer vala_switch_statement_parent_class = NULL;
static ValaStatementIface* vala_switch_statement_vala_statement_parent_iface = NULL;
static void vala_switch_statement_finalize (ValaCodeNode* obj);



/**
 * Creates a new switch statement.
 *
 * @param expression       switch expression
 * @param source_reference reference to source code
 * @return                 newly created switch statement
 */
ValaSwitchStatement* vala_switch_statement_construct (GType object_type, ValaExpression* expression, ValaSourceReference* source_reference) {
	ValaSwitchStatement* self;
	g_return_val_if_fail (expression != NULL, NULL);
	g_return_val_if_fail (source_reference != NULL, NULL);
	self = ((ValaSwitchStatement*) (g_type_create_instance (object_type)));
	vala_code_node_set_source_reference (((ValaCodeNode*) (self)), source_reference);
	vala_switch_statement_set_expression (self, expression);
	return self;
}


ValaSwitchStatement* vala_switch_statement_new (ValaExpression* expression, ValaSourceReference* source_reference) {
	return vala_switch_statement_construct (VALA_TYPE_SWITCH_STATEMENT, expression, source_reference);
}


/**
 * Appends the specified section to the list of switch sections.
 *
 * @param section a switch section
 */
void vala_switch_statement_add_section (ValaSwitchStatement* self, ValaSwitchSection* section) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (section != NULL);
	vala_code_node_set_parent_node (((ValaCodeNode*) (section)), ((ValaCodeNode*) (self)));
	gee_collection_add (((GeeCollection*) (self->priv->sections)), section);
}


/**
 * Returns a copy of the list of switch sections.
 *
 * @return section list
 */
GeeList* vala_switch_statement_get_sections (ValaSwitchStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return ((GeeList*) (gee_read_only_list_new (VALA_TYPE_SWITCH_SECTION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->sections)));
}


static void vala_switch_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaSwitchStatement * self;
	self = ((ValaSwitchStatement*) (base));
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept (((ValaCodeNode*) (vala_switch_statement_get_expression (self))), visitor);
	vala_code_visitor_visit_end_full_expression (visitor, vala_switch_statement_get_expression (self));
	{
		GeeList* section_collection;
		int section_it;
		section_collection = self->priv->sections;
		for (section_it = 0; section_it < gee_collection_get_size (GEE_COLLECTION (section_collection)); section_it = section_it + 1) {
			ValaSwitchSection* section;
			section = ((ValaSwitchSection*) (gee_list_get (GEE_LIST (section_collection), section_it)));
			{
				vala_code_node_accept (((ValaCodeNode*) (section)), visitor);
				(section == NULL ? NULL : (section = (vala_code_node_unref (section), NULL)));
			}
		}
	}
	vala_code_visitor_visit_switch_statement (visitor, self);
}


static void vala_switch_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaSwitchStatement * self;
	self = ((ValaSwitchStatement*) (base));
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_switch_statement_get_expression (self) == old_node) {
		vala_switch_statement_set_expression (self, new_node);
	}
}


ValaExpression* vala_switch_statement_get_expression (ValaSwitchStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_expression;
}


void vala_switch_statement_set_expression (ValaSwitchStatement* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_expression == NULL ? NULL : (self->priv->_expression = (vala_code_node_unref (self->priv->_expression), NULL))), _tmp2);
	vala_code_node_set_parent_node (((ValaCodeNode*) (self->priv->_expression)), ((ValaCodeNode*) (self)));
}


static void vala_switch_statement_class_init (ValaSwitchStatementClass * klass) {
	vala_switch_statement_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_switch_statement_finalize;
	g_type_class_add_private (klass, sizeof (ValaSwitchStatementPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_switch_statement_real_accept;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_switch_statement_real_replace_expression;
}


static void vala_switch_statement_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_switch_statement_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_switch_statement_instance_init (ValaSwitchStatement * self) {
	self->priv = VALA_SWITCH_STATEMENT_GET_PRIVATE (self);
	self->priv->sections = ((GeeList*) (gee_array_list_new (VALA_TYPE_SWITCH_SECTION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)));
}


static void vala_switch_statement_finalize (ValaCodeNode* obj) {
	ValaSwitchStatement * self;
	self = VALA_SWITCH_STATEMENT (obj);
	(self->priv->_expression == NULL ? NULL : (self->priv->_expression = (vala_code_node_unref (self->priv->_expression), NULL)));
	(self->priv->sections == NULL ? NULL : (self->priv->sections = (gee_collection_object_unref (self->priv->sections), NULL)));
	VALA_CODE_NODE_CLASS (vala_switch_statement_parent_class)->finalize (obj);
}


GType vala_switch_statement_get_type (void) {
	static GType vala_switch_statement_type_id = 0;
	if (vala_switch_statement_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSwitchStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_switch_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSwitchStatement), 0, (GInstanceInitFunc) vala_switch_statement_instance_init, NULL };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_switch_statement_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_switch_statement_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaSwitchStatement", &g_define_type_info, 0);
		g_type_add_interface_static (vala_switch_statement_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_switch_statement_type_id;
}




