/* valaassignment.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaassignment.h>
#include <stdlib.h>
#include <string.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/list.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valamemberaccess.h>
#include <vala/valasignal.h>
#include <vala/valadynamicproperty.h>
#include <vala/valareport.h>
#include <vala/valasymbol.h>
#include <vala/valadynamicsignal.h>
#include <vala/valadelegatetype.h>
#include <vala/valadatatype.h>
#include <vala/valadelegate.h>
#include <vala/valaelementaccess.h>
#include <vala/valapointerindirection.h>
#include <vala/valabinaryexpression.h>
#include <vala/valaparenthesizedexpression.h>
#include <vala/valamethod.h>
#include <vala/valaformalparameter.h>
#include <vala/valaobjecttype.h>
#include <vala/valaobjecttypesymbol.h>
#include <vala/valaproperty.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valacreationmethod.h>
#include <vala/valalocalvariable.h>
#include <vala/valafield.h>
#include <vala/valapointertype.h>
#include <vala/valaarraytype.h>




struct _ValaAssignmentPrivate {
	ValaAssignmentOperator _operator;
	ValaExpression* _left;
	ValaExpression* _right;
};

#define VALA_ASSIGNMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ASSIGNMENT, ValaAssignmentPrivate))
enum  {
	VALA_ASSIGNMENT_DUMMY_PROPERTY
};
static void vala_assignment_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_assignment_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_assignment_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_assignment_real_is_pure (ValaExpression* base);
static gboolean vala_assignment_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_assignment_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
static void vala_assignment_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
static gpointer vala_assignment_parent_class = NULL;
static void vala_assignment_finalize (ValaCodeNode* obj);



/**
 * Creates a new assignment.
 *
 * @param left             left hand side
 * @param operator         assignment operator
 * @param right            right hand side
 * @param source_reference reference to source code
 * @return                 newly created assignment
 */
ValaAssignment* vala_assignment_construct (GType object_type, ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference) {
	ValaAssignment* self;
	g_return_val_if_fail (left != NULL, NULL);
	g_return_val_if_fail (right != NULL, NULL);
	self = (ValaAssignment*) g_type_create_instance (object_type);
	vala_assignment_set_right (self, right);
	vala_assignment_set_operator (self, operator);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_assignment_set_left (self, left);
	return self;
}


ValaAssignment* vala_assignment_new (ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference) {
	return vala_assignment_construct (VALA_TYPE_ASSIGNMENT, left, right, operator, source_reference);
}


static void vala_assignment_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_assignment (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_assignment_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_assignment_get_left (self), visitor);
	vala_code_node_accept ((ValaCodeNode*) vala_assignment_get_right (self), visitor);
}


static void vala_assignment_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_assignment_get_left (self) == old_node) {
		vala_assignment_set_left (self, new_node);
	}
	if (vala_assignment_get_right (self) == old_node) {
		vala_assignment_set_right (self, new_node);
	}
}


static gboolean vala_assignment_real_is_pure (ValaExpression* base) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	return FALSE;
}


static gboolean vala_assignment_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaAssignment * self;
	gboolean _tmp22;
	GeeList* _tmp106;
	GeeList* _tmp107;
	self = (ValaAssignment*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_expression_set_lvalue (vala_assignment_get_left (self), TRUE);
	if (!vala_code_node_check ((ValaCodeNode*) vala_assignment_get_left (self), analyzer)) {
		/* skip on error in inner expression*/
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self))) {
		ValaMemberAccess* _tmp2;
		ValaMemberAccess* ma;
		gboolean _tmp3;
		gboolean _tmp4;
		gboolean _tmp9;
		_tmp2 = NULL;
		ma = (_tmp2 = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
		_tmp3 = FALSE;
		_tmp4 = FALSE;
		if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
			_tmp4 = TRUE;
		} else {
			_tmp4 = VALA_IS_DYNAMIC_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma));
		}
		if (!(_tmp4)) {
			_tmp3 = vala_expression_get_value_type ((ValaExpression*) ma) == NULL;
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			gboolean _tmp5;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "unsupported lvalue in assignment");
			return (_tmp5 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp5);
		}
		if (vala_member_access_get_prototype_access (ma)) {
			char* _tmp7;
			char* _tmp6;
			gboolean _tmp8;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp7 = NULL;
			_tmp6 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp7 = g_strdup_printf ("Access to instance member `%s' denied", _tmp6 = vala_symbol_get_full_name (vala_expression_get_symbol_reference ((ValaExpression*) ma))));
			_tmp7 = (g_free (_tmp7), NULL);
			_tmp6 = (g_free (_tmp6), NULL);
			return (_tmp8 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp8);
		}
		_tmp9 = FALSE;
		if (vala_code_node_get_error ((ValaCodeNode*) ma)) {
			_tmp9 = TRUE;
		} else {
			_tmp9 = vala_expression_get_symbol_reference ((ValaExpression*) ma) == NULL;
		}
		if (_tmp9) {
			gboolean _tmp10;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			/* if no symbol found, skip this check */
			return (_tmp10 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp10);
		}
		if (VALA_IS_DYNAMIC_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
		} else {
			if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
				ValaSignal* _tmp11;
				ValaSignal* sig;
				ValaDelegateType* _tmp13;
				ValaDelegate* _tmp12;
				_tmp11 = NULL;
				sig = (_tmp11 = VALA_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp11 == NULL) ? NULL : vala_code_node_ref (_tmp11));
				_tmp13 = NULL;
				_tmp12 = NULL;
				vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp13 = vala_delegate_type_new (_tmp12 = vala_signal_get_delegate (sig, vala_expression_get_value_type (vala_member_access_get_inner (ma)), (ValaCodeNode*) self))));
				(_tmp13 == NULL) ? NULL : (_tmp13 = (vala_code_node_unref (_tmp13), NULL));
				(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_code_node_unref (_tmp12), NULL));
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			} else {
				vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type ((ValaExpression*) ma));
			}
		}
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	} else {
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (self))) {
			ValaElementAccess* _tmp14;
			ValaElementAccess* ea;
			gboolean _tmp15;
			_tmp14 = NULL;
			ea = (_tmp14 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (self)), (_tmp14 == NULL) ? NULL : vala_code_node_ref (_tmp14));
			_tmp15 = FALSE;
			if (VALA_IS_MEMBER_ACCESS (vala_element_access_get_container (ea))) {
				_tmp15 = VALA_IS_SIGNAL (vala_expression_get_symbol_reference (vala_element_access_get_container (ea)));
			} else {
				_tmp15 = FALSE;
			}
			if (_tmp15) {
				ValaMemberAccess* _tmp16;
				ValaMemberAccess* ma;
				ValaSignal* _tmp17;
				ValaSignal* sig;
				ValaDelegateType* _tmp19;
				ValaDelegate* _tmp18;
				_tmp16 = NULL;
				ma = (_tmp16 = VALA_MEMBER_ACCESS (vala_element_access_get_container (ea)), (_tmp16 == NULL) ? NULL : vala_code_node_ref (_tmp16));
				_tmp17 = NULL;
				sig = (_tmp17 = VALA_SIGNAL (vala_expression_get_symbol_reference (vala_element_access_get_container (ea))), (_tmp17 == NULL) ? NULL : vala_code_node_ref (_tmp17));
				_tmp19 = NULL;
				_tmp18 = NULL;
				vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp19 = vala_delegate_type_new (_tmp18 = vala_signal_get_delegate (sig, vala_expression_get_value_type (vala_member_access_get_inner (ma)), (ValaCodeNode*) self))));
				(_tmp19 == NULL) ? NULL : (_tmp19 = (vala_code_node_unref (_tmp19), NULL));
				(_tmp18 == NULL) ? NULL : (_tmp18 = (vala_code_node_unref (_tmp18), NULL));
				(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			} else {
				vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type (vala_assignment_get_left (self)));
			}
			(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
		} else {
			if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (self))) {
				vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type (vala_assignment_get_left (self)));
			} else {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "unsupported lvalue in assignment");
				return FALSE;
			}
		}
	}
	if (!vala_code_node_check ((ValaCodeNode*) vala_assignment_get_right (self), analyzer)) {
		/* skip on error in inner expression*/
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	_tmp22 = FALSE;
	if (self->priv->_operator != VALA_ASSIGNMENT_OPERATOR_SIMPLE) {
		_tmp22 = VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self));
	} else {
		_tmp22 = FALSE;
	}
	if (_tmp22) {
		ValaMemberAccess* _tmp23;
		ValaMemberAccess* ma;
		/* transform into simple assignment
		 FIXME: only do this if the backend doesn't support
		 the assignment natively*/
		_tmp23 = NULL;
		ma = (_tmp23 = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp23 == NULL) ? NULL : vala_code_node_ref (_tmp23));
		if (!(VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma)))) {
			ValaMemberAccess* old_value;
			ValaParenthesizedExpression* _tmp24;
			ValaBinaryExpression* _tmp25;
			ValaBinaryExpression* bin;
			ValaDataType* _tmp26;
			old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
			_tmp24 = NULL;
			_tmp25 = NULL;
			bin = (_tmp25 = vala_binary_expression_new (VALA_BINARY_OPERATOR_PLUS, (ValaExpression*) old_value, (ValaExpression*) (_tmp24 = vala_parenthesized_expression_new (vala_assignment_get_right (self), vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)))), NULL), (_tmp24 == NULL) ? NULL : (_tmp24 = (vala_code_node_unref (_tmp24), NULL)), _tmp25);
			vala_expression_set_target_type ((ValaExpression*) bin, vala_expression_get_target_type (vala_assignment_get_right (self)));
			_tmp26 = NULL;
			vala_expression_set_target_type (vala_assignment_get_right (self), _tmp26 = vala_data_type_copy (vala_expression_get_target_type (vala_assignment_get_right (self))));
			(_tmp26 == NULL) ? NULL : (_tmp26 = (vala_code_node_unref (_tmp26), NULL));
			vala_data_type_set_value_owned (vala_expression_get_target_type (vala_assignment_get_right (self)), FALSE);
			if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_OR) {
				vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_OR);
			} else {
				if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_AND) {
					vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_AND);
				} else {
					if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR) {
						vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_XOR);
					} else {
						if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_ADD) {
							vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_PLUS);
						} else {
							if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SUB) {
								vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MINUS);
							} else {
								if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_MUL) {
									vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MUL);
								} else {
									if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_DIV) {
										vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_DIV);
									} else {
										if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_PERCENT) {
											vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MOD);
										} else {
											if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT) {
												vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_LEFT);
											} else {
												if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT) {
													vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_RIGHT);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			vala_assignment_set_right (self, (ValaExpression*) bin);
			vala_code_node_check ((ValaCodeNode*) vala_assignment_get_right (self), analyzer);
			vala_assignment_set_operator (self, VALA_ASSIGNMENT_OPERATOR_SIMPLE);
			(old_value == NULL) ? NULL : (old_value = (vala_code_node_unref (old_value), NULL));
			(bin == NULL) ? NULL : (bin = (vala_code_node_unref (bin), NULL));
		}
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	}
	if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference (vala_assignment_get_left (self)))) {
		ValaSignal* _tmp27;
		ValaSignal* sig;
		ValaMethod* _tmp29;
		ValaSymbol* _tmp28;
		ValaMethod* m;
		ValaDynamicSignal* _tmp32;
		ValaSignal* _tmp31;
		ValaDynamicSignal* dynamic_sig;
		ValaMemberAccess* _tmp34;
		ValaExpression* _tmp33;
		ValaMemberAccess* right_ma;
		_tmp27 = NULL;
		sig = (_tmp27 = VALA_SIGNAL (vala_expression_get_symbol_reference (vala_assignment_get_left (self))), (_tmp27 == NULL) ? NULL : vala_code_node_ref (_tmp27));
		_tmp29 = NULL;
		_tmp28 = NULL;
		m = (_tmp29 = (_tmp28 = vala_expression_get_symbol_reference (vala_assignment_get_right (self)), VALA_IS_METHOD (_tmp28) ? ((ValaMethod*) _tmp28) : NULL), (_tmp29 == NULL) ? NULL : vala_code_node_ref (_tmp29));
		if (m == NULL) {
			gboolean _tmp30;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), "unsupported expression for signal handler");
			return (_tmp30 = FALSE, (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), _tmp30);
		}
		_tmp32 = NULL;
		_tmp31 = NULL;
		dynamic_sig = (_tmp32 = (_tmp31 = sig, VALA_IS_DYNAMIC_SIGNAL (_tmp31) ? ((ValaDynamicSignal*) _tmp31) : NULL), (_tmp32 == NULL) ? NULL : vala_code_node_ref (_tmp32));
		_tmp34 = NULL;
		_tmp33 = NULL;
		right_ma = (_tmp34 = (_tmp33 = vala_assignment_get_right (self), VALA_IS_MEMBER_ACCESS (_tmp33) ? ((ValaMemberAccess*) _tmp33) : NULL), (_tmp34 == NULL) ? NULL : vala_code_node_ref (_tmp34));
		if (dynamic_sig != NULL) {
			gboolean first;
			ValaDelegateType* _tmp40;
			ValaDelegate* _tmp39;
			ValaObjectType* _tmp38;
			first = TRUE;
			{
				GeeList* _tmp35;
				GeeIterator* _tmp36;
				GeeIterator* _param_it;
				_tmp35 = NULL;
				_tmp36 = NULL;
				_param_it = (_tmp36 = gee_iterable_iterator ((GeeIterable*) (_tmp35 = vala_data_type_get_parameters (vala_expression_get_value_type (vala_dynamic_signal_get_handler (dynamic_sig))))), (_tmp35 == NULL) ? NULL : (_tmp35 = (gee_collection_object_unref (_tmp35), NULL)), _tmp36);
				while (gee_iterator_next (_param_it)) {
					ValaFormalParameter* param;
					param = (ValaFormalParameter*) gee_iterator_get (_param_it);
					if (first) {
						/* skip sender parameter*/
						first = FALSE;
					} else {
						ValaFormalParameter* _tmp37;
						_tmp37 = NULL;
						vala_signal_add_parameter ((ValaSignal*) dynamic_sig, _tmp37 = vala_formal_parameter_copy (param));
						(_tmp37 == NULL) ? NULL : (_tmp37 = (vala_code_node_unref (_tmp37), NULL));
					}
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				}
				(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
			}
			_tmp40 = NULL;
			_tmp39 = NULL;
			_tmp38 = NULL;
			vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp40 = vala_delegate_type_new (_tmp39 = vala_signal_get_delegate (sig, (ValaDataType*) (_tmp38 = vala_object_type_new (VALA_OBJECT_TYPE_SYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) sig)))), (ValaCodeNode*) self))));
			(_tmp40 == NULL) ? NULL : (_tmp40 = (vala_code_node_unref (_tmp40), NULL));
			(_tmp39 == NULL) ? NULL : (_tmp39 = (vala_code_node_unref (_tmp39), NULL));
			(_tmp38 == NULL) ? NULL : (_tmp38 = (vala_code_node_unref (_tmp38), NULL));
		} else {
			if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_target_type (vala_assignment_get_right (self)))) {
				ValaDelegateType* _tmp41;
				ValaDelegateType* delegate_type;
				char* _tmp45;
				char* _tmp44;
				char* _tmp43;
				char* _tmp42;
				gboolean _tmp46;
				_tmp41 = NULL;
				delegate_type = (_tmp41 = VALA_DELEGATE_TYPE (vala_expression_get_target_type (vala_assignment_get_right (self))), (_tmp41 == NULL) ? NULL : vala_code_node_ref (_tmp41));
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp45 = NULL;
				_tmp44 = NULL;
				_tmp43 = NULL;
				_tmp42 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), _tmp45 = g_strdup_printf ("method `%s' is incompatible with signal `%s', expected `%s'", _tmp42 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp43 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_target_type (vala_assignment_get_right (self))), _tmp44 = vala_delegate_get_prototype_string (vala_delegate_type_get_delegate_symbol (delegate_type), vala_symbol_get_name ((ValaSymbol*) m))));
				_tmp45 = (g_free (_tmp45), NULL);
				_tmp44 = (g_free (_tmp44), NULL);
				_tmp43 = (g_free (_tmp43), NULL);
				_tmp42 = (g_free (_tmp42), NULL);
				return (_tmp46 = FALSE, (delegate_type == NULL) ? NULL : (delegate_type = (vala_code_node_unref (delegate_type), NULL)), (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL)), (right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL)), _tmp46);
			} else {
				gboolean _tmp47;
				_tmp47 = FALSE;
				if (right_ma != NULL) {
					_tmp47 = vala_member_access_get_prototype_access (right_ma);
				} else {
					_tmp47 = FALSE;
				}
				if (_tmp47) {
					char* _tmp49;
					char* _tmp48;
					gboolean _tmp50;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp49 = NULL;
					_tmp48 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), _tmp49 = g_strdup_printf ("Access to instance member `%s' denied", _tmp48 = vala_symbol_get_full_name ((ValaSymbol*) m)));
					_tmp49 = (g_free (_tmp49), NULL);
					_tmp48 = (g_free (_tmp48), NULL);
					return (_tmp50 = FALSE, (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL)), (right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL)), _tmp50);
				}
			}
		}
		(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
		(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		(dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL));
		(right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL));
	} else {
		if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self))) {
			ValaMemberAccess* _tmp51;
			ValaMemberAccess* ma;
			gboolean _tmp87;
			_tmp51 = NULL;
			ma = (_tmp51 = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp51 == NULL) ? NULL : vala_code_node_ref (_tmp51));
			if (VALA_IS_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
				ValaProperty* _tmp52;
				ValaProperty* prop;
				ValaDynamicProperty* _tmp54;
				ValaProperty* _tmp53;
				ValaDynamicProperty* dynamic_prop;
				gboolean _tmp57;
				_tmp52 = NULL;
				prop = (_tmp52 = VALA_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp52 == NULL) ? NULL : vala_code_node_ref (_tmp52));
				_tmp54 = NULL;
				_tmp53 = NULL;
				dynamic_prop = (_tmp54 = (_tmp53 = prop, VALA_IS_DYNAMIC_PROPERTY (_tmp53) ? ((ValaDynamicProperty*) _tmp53) : NULL), (_tmp54 == NULL) ? NULL : vala_code_node_ref (_tmp54));
				if (dynamic_prop != NULL) {
					ValaDataType* _tmp55;
					ValaDataType* _tmp56;
					_tmp55 = NULL;
					vala_property_set_property_type ((ValaProperty*) dynamic_prop, _tmp55 = vala_data_type_copy (vala_expression_get_value_type (vala_assignment_get_right (self))));
					(_tmp55 == NULL) ? NULL : (_tmp55 = (vala_code_node_unref (_tmp55), NULL));
					_tmp56 = NULL;
					vala_expression_set_value_type (vala_assignment_get_left (self), _tmp56 = vala_data_type_copy (vala_property_get_property_type ((ValaProperty*) dynamic_prop)));
					(_tmp56 == NULL) ? NULL : (_tmp56 = (vala_code_node_unref (_tmp56), NULL));
				}
				_tmp57 = FALSE;
				if (vala_property_get_set_accessor (prop) == NULL) {
					_tmp57 = TRUE;
				} else {
					gboolean _tmp58;
					_tmp58 = FALSE;
					if (!vala_property_accessor_get_writable (vala_property_get_set_accessor (prop))) {
						gboolean _tmp59;
						ValaMethod* _tmp60;
						gboolean _tmp61;
						_tmp59 = FALSE;
						_tmp60 = NULL;
						if ((_tmp61 = VALA_IS_CREATION_METHOD (_tmp60 = vala_semantic_analyzer_find_current_method (analyzer)), (_tmp60 == NULL) ? NULL : (_tmp60 = (vala_code_node_unref (_tmp60), NULL)), _tmp61)) {
							_tmp59 = TRUE;
						} else {
							_tmp59 = vala_semantic_analyzer_is_in_constructor (analyzer);
						}
						_tmp58 = !(_tmp59);
					} else {
						_tmp58 = FALSE;
					}
					_tmp57 = (_tmp58);
				}
				if (_tmp57) {
					char* _tmp63;
					char* _tmp62;
					gboolean _tmp64;
					vala_code_node_set_error ((ValaCodeNode*) ma, TRUE);
					_tmp63 = NULL;
					_tmp62 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) ma), _tmp63 = g_strdup_printf ("Property `%s' is read-only", _tmp62 = vala_symbol_get_full_name ((ValaSymbol*) prop)));
					_tmp63 = (g_free (_tmp63), NULL);
					_tmp62 = (g_free (_tmp62), NULL);
					return (_tmp64 = FALSE, (prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL)), (dynamic_prop == NULL) ? NULL : (dynamic_prop = (vala_code_node_unref (dynamic_prop), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp64);
				}
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				(dynamic_prop == NULL) ? NULL : (dynamic_prop = (vala_code_node_unref (dynamic_prop), NULL));
			} else {
				gboolean _tmp65;
				_tmp65 = FALSE;
				if (VALA_IS_LOCAL_VARIABLE (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
					_tmp65 = vala_expression_get_value_type (vala_assignment_get_right (self)) == NULL;
				} else {
					_tmp65 = FALSE;
				}
				if (_tmp65) {
					ValaLocalVariable* _tmp66;
					ValaLocalVariable* local;
					gboolean _tmp67;
					_tmp66 = NULL;
					local = (_tmp66 = VALA_LOCAL_VARIABLE (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp66 == NULL) ? NULL : vala_code_node_ref (_tmp66));
					_tmp67 = FALSE;
					if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self)))) {
						_tmp67 = VALA_IS_DELEGATE_TYPE (vala_local_variable_get_variable_type (local));
					} else {
						_tmp67 = FALSE;
					}
					if (_tmp67) {
						ValaMethod* _tmp68;
						ValaMethod* m;
						ValaDelegateType* _tmp69;
						ValaDelegateType* dt;
						ValaDelegate* _tmp70;
						ValaDelegate* cb;
						_tmp68 = NULL;
						m = (_tmp68 = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self))), (_tmp68 == NULL) ? NULL : vala_code_node_ref (_tmp68));
						_tmp69 = NULL;
						dt = (_tmp69 = VALA_DELEGATE_TYPE (vala_local_variable_get_variable_type (local)), (_tmp69 == NULL) ? NULL : vala_code_node_ref (_tmp69));
						_tmp70 = NULL;
						cb = (_tmp70 = vala_delegate_type_get_delegate_symbol (dt), (_tmp70 == NULL) ? NULL : vala_code_node_ref (_tmp70));
						/* check whether method matches callback type */
						if (!vala_delegate_matches_method (cb, m)) {
							char* _tmp73;
							char* _tmp72;
							char* _tmp71;
							gboolean _tmp74;
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							_tmp73 = NULL;
							_tmp72 = NULL;
							_tmp71 = NULL;
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp73 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", _tmp71 = vala_symbol_get_full_name ((ValaSymbol*) m), _tmp72 = vala_symbol_get_full_name ((ValaSymbol*) cb)));
							_tmp73 = (g_free (_tmp73), NULL);
							_tmp72 = (g_free (_tmp72), NULL);
							_tmp71 = (g_free (_tmp71), NULL);
							return (_tmp74 = FALSE, (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), (cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL)), (local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp74);
						}
						vala_expression_set_value_type (vala_assignment_get_right (self), vala_local_variable_get_variable_type (local));
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
						(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
						(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
					} else {
						gboolean _tmp75;
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Assignment: Invalid callback assignment attempt");
						return (_tmp75 = FALSE, (local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp75);
					}
					(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
				} else {
					gboolean _tmp76;
					_tmp76 = FALSE;
					if (VALA_IS_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
						_tmp76 = vala_expression_get_value_type (vala_assignment_get_right (self)) == NULL;
					} else {
						_tmp76 = FALSE;
					}
					if (_tmp76) {
						ValaField* _tmp77;
						ValaField* f;
						gboolean _tmp78;
						_tmp77 = NULL;
						f = (_tmp77 = VALA_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp77 == NULL) ? NULL : vala_code_node_ref (_tmp77));
						_tmp78 = FALSE;
						if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self)))) {
							_tmp78 = VALA_IS_DELEGATE_TYPE (vala_field_get_field_type (f));
						} else {
							_tmp78 = FALSE;
						}
						if (_tmp78) {
							ValaMethod* _tmp79;
							ValaMethod* m;
							ValaDelegateType* _tmp80;
							ValaDelegateType* dt;
							ValaDelegate* _tmp81;
							ValaDelegate* cb;
							_tmp79 = NULL;
							m = (_tmp79 = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self))), (_tmp79 == NULL) ? NULL : vala_code_node_ref (_tmp79));
							_tmp80 = NULL;
							dt = (_tmp80 = VALA_DELEGATE_TYPE (vala_field_get_field_type (f)), (_tmp80 == NULL) ? NULL : vala_code_node_ref (_tmp80));
							_tmp81 = NULL;
							cb = (_tmp81 = vala_delegate_type_get_delegate_symbol (dt), (_tmp81 == NULL) ? NULL : vala_code_node_ref (_tmp81));
							/* check whether method matches callback type */
							if (!vala_delegate_matches_method (cb, m)) {
								char* _tmp84;
								char* _tmp83;
								char* _tmp82;
								gboolean _tmp85;
								vala_code_node_set_error ((ValaCodeNode*) f, TRUE);
								_tmp84 = NULL;
								_tmp83 = NULL;
								_tmp82 = NULL;
								vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp84 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", _tmp82 = vala_symbol_get_full_name ((ValaSymbol*) m), _tmp83 = vala_symbol_get_full_name ((ValaSymbol*) cb)));
								_tmp84 = (g_free (_tmp84), NULL);
								_tmp83 = (g_free (_tmp83), NULL);
								_tmp82 = (g_free (_tmp82), NULL);
								return (_tmp85 = FALSE, (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), (cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL)), (f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp85);
							}
							vala_expression_set_value_type (vala_assignment_get_right (self), vala_field_get_field_type (f));
							(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
							(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
							(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
						} else {
							gboolean _tmp86;
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Assignment: Invalid callback assignment attempt");
							return (_tmp86 = FALSE, (f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp86);
						}
						(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
					}
				}
			}
			_tmp87 = FALSE;
			if (vala_expression_get_value_type (vala_assignment_get_left (self)) != NULL) {
				_tmp87 = vala_expression_get_value_type (vala_assignment_get_right (self)) != NULL;
			} else {
				_tmp87 = FALSE;
			}
			if (_tmp87) {
				/* if there was an error on either side,
				 * i.e. {left|right}.value_type == null, skip type check */
				if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					char* _tmp90;
					char* _tmp89;
					char* _tmp88;
					gboolean _tmp91;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp90 = NULL;
					_tmp89 = NULL;
					_tmp88 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp90 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", _tmp88 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp89 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_left (self)))));
					_tmp90 = (g_free (_tmp90), NULL);
					_tmp89 = (g_free (_tmp89), NULL);
					_tmp88 = (g_free (_tmp88), NULL);
					return (_tmp91 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp91);
				}
				if (!(VALA_IS_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma)))) {
					if (vala_data_type_is_disposable (vala_expression_get_value_type (vala_assignment_get_right (self)))) {
						gboolean _tmp92;
						_tmp92 = FALSE;
						if (!(VALA_IS_POINTER_TYPE (vala_expression_get_value_type (vala_assignment_get_left (self))))) {
							_tmp92 = !vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)));
						} else {
							_tmp92 = FALSE;
						}
						/* rhs transfers ownership of the expression */
						if (_tmp92) {
							/* lhs doesn't own the value */
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Invalid assignment from owned expression to unowned variable");
						}
					} else {
						if (vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)))) {
						}
					}
				}
			}
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
		} else {
			if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (self))) {
				ValaElementAccess* _tmp93;
				ValaElementAccess* ea;
				_tmp93 = NULL;
				ea = (_tmp93 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (self)), (_tmp93 == NULL) ? NULL : vala_code_node_ref (_tmp93));
				if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					char* _tmp96;
					char* _tmp95;
					char* _tmp94;
					gboolean _tmp97;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp96 = NULL;
					_tmp95 = NULL;
					_tmp94 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp96 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", _tmp94 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp95 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_left (self)))));
					_tmp96 = (g_free (_tmp96), NULL);
					_tmp95 = (g_free (_tmp95), NULL);
					_tmp94 = (g_free (_tmp94), NULL);
					return (_tmp97 = FALSE, (ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL)), _tmp97);
				}
				if (vala_data_type_is_disposable (vala_expression_get_value_type (vala_assignment_get_right (self)))) {
					ValaDataType* element_type;
					gboolean _tmp102;
					/* rhs transfers ownership of the expression */
					element_type = NULL;
					if (VALA_IS_ARRAY_TYPE (vala_expression_get_value_type (vala_element_access_get_container (ea)))) {
						ValaArrayType* _tmp98;
						ValaArrayType* array_type;
						ValaDataType* _tmp100;
						ValaDataType* _tmp99;
						_tmp98 = NULL;
						array_type = (_tmp98 = VALA_ARRAY_TYPE (vala_expression_get_value_type (vala_element_access_get_container (ea))), (_tmp98 == NULL) ? NULL : vala_code_node_ref (_tmp98));
						_tmp100 = NULL;
						_tmp99 = NULL;
						element_type = (_tmp100 = (_tmp99 = vala_array_type_get_element_type (array_type), (_tmp99 == NULL) ? NULL : vala_code_node_ref (_tmp99)), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp100);
						(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
					} else {
						GeeList* args;
						ValaDataType* _tmp101;
						args = vala_data_type_get_type_arguments (vala_expression_get_value_type (vala_element_access_get_container (ea)));
						g_assert (gee_collection_get_size ((GeeCollection*) args) == 1);
						_tmp101 = NULL;
						element_type = (_tmp101 = (ValaDataType*) gee_list_get (args, 0), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp101);
						(args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL));
					}
					_tmp102 = FALSE;
					if (!(VALA_IS_POINTER_TYPE (element_type))) {
						_tmp102 = !vala_data_type_get_value_owned (element_type);
					} else {
						_tmp102 = FALSE;
					}
					if (_tmp102) {
						gboolean _tmp103;
						/* lhs doesn't own the value */
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Invalid assignment from owned expression to unowned variable");
						return (_tmp103 = FALSE, (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), (ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL)), _tmp103);
					}
					(element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL));
				} else {
					if (vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					}
				}
				(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
			} else {
				/* lhs wants to own the value
				 * rhs doesn't transfer the ownership
				 * code generator needs to add reference
				 * increment calls */
				return TRUE;
			}
		}
	}
	if (vala_expression_get_value_type (vala_assignment_get_left (self)) != NULL) {
		ValaDataType* _tmp105;
		_tmp105 = NULL;
		vala_expression_set_value_type ((ValaExpression*) self, _tmp105 = vala_data_type_copy (vala_expression_get_value_type (vala_assignment_get_left (self))));
		(_tmp105 == NULL) ? NULL : (_tmp105 = (vala_code_node_unref (_tmp105), NULL));
		vala_data_type_set_value_owned (vala_expression_get_value_type ((ValaExpression*) self), FALSE);
	} else {
		vala_expression_set_value_type ((ValaExpression*) self, NULL);
	}
	_tmp106 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp106 = vala_code_node_get_error_types ((ValaCodeNode*) vala_assignment_get_left (self)));
	(_tmp106 == NULL) ? NULL : (_tmp106 = (gee_collection_object_unref (_tmp106), NULL));
	_tmp107 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp107 = vala_code_node_get_error_types ((ValaCodeNode*) vala_assignment_get_right (self)));
	(_tmp107 == NULL) ? NULL : (_tmp107 = (gee_collection_object_unref (_tmp107), NULL));
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


static void vala_assignment_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaAssignment * self;
	ValaLocalVariable* _tmp1;
	ValaSymbol* _tmp0;
	ValaLocalVariable* local;
	self = (ValaAssignment*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_assignment_get_right (self), collection);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_assignment_get_left (self), collection);
	_tmp1 = NULL;
	_tmp0 = NULL;
	local = (_tmp1 = (_tmp0 = vala_expression_get_symbol_reference (vala_assignment_get_left (self)), VALA_IS_LOCAL_VARIABLE (_tmp0) ? ((ValaLocalVariable*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	if (local != NULL) {
		gee_collection_add (collection, local);
	}
	(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
}


static void vala_assignment_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaAssignment * self;
	ValaMemberAccess* _tmp1;
	ValaExpression* _tmp0;
	ValaMemberAccess* ma;
	ValaElementAccess* _tmp3;
	ValaExpression* _tmp2;
	ValaElementAccess* ea;
	gboolean _tmp4;
	self = (ValaAssignment*) base;
	g_return_if_fail (collection != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	ma = (_tmp1 = (_tmp0 = vala_assignment_get_left (self), VALA_IS_MEMBER_ACCESS (_tmp0) ? ((ValaMemberAccess*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	_tmp3 = NULL;
	_tmp2 = NULL;
	ea = (_tmp3 = (_tmp2 = vala_assignment_get_left (self), VALA_IS_ELEMENT_ACCESS (_tmp2) ? ((ValaElementAccess*) _tmp2) : NULL), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
	_tmp4 = FALSE;
	if (ma != NULL) {
		_tmp4 = vala_member_access_get_inner (ma) != NULL;
	} else {
		_tmp4 = FALSE;
	}
	if (_tmp4) {
		vala_code_node_get_used_variables ((ValaCodeNode*) vala_member_access_get_inner (ma), collection);
	} else {
		if (ea != NULL) {
			vala_code_node_get_used_variables ((ValaCodeNode*) ea, collection);
		}
	}
	vala_code_node_get_used_variables ((ValaCodeNode*) vala_assignment_get_right (self), collection);
	(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
}


ValaExpression* vala_assignment_get_left (ValaAssignment* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_left;
}


void vala_assignment_set_left (ValaAssignment* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_left = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_code_node_unref (self->priv->_left), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_left, (ValaCodeNode*) self);
}


ValaAssignmentOperator vala_assignment_get_operator (ValaAssignment* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_operator;
}


void vala_assignment_set_operator (ValaAssignment* self, ValaAssignmentOperator value) {
	g_return_if_fail (self != NULL);
	self->priv->_operator = value;
}


ValaExpression* vala_assignment_get_right (ValaAssignment* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_right;
}


void vala_assignment_set_right (ValaAssignment* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_right = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_code_node_unref (self->priv->_right), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_right, (ValaCodeNode*) self);
}


static void vala_assignment_class_init (ValaAssignmentClass * klass) {
	vala_assignment_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_assignment_finalize;
	g_type_class_add_private (klass, sizeof (ValaAssignmentPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_assignment_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_assignment_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_assignment_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_assignment_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_assignment_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_assignment_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_assignment_real_get_used_variables;
}


static void vala_assignment_instance_init (ValaAssignment * self) {
	self->priv = VALA_ASSIGNMENT_GET_PRIVATE (self);
}


static void vala_assignment_finalize (ValaCodeNode* obj) {
	ValaAssignment * self;
	self = VALA_ASSIGNMENT (obj);
	(self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_code_node_unref (self->priv->_left), NULL));
	(self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_code_node_unref (self->priv->_right), NULL));
	VALA_CODE_NODE_CLASS (vala_assignment_parent_class)->finalize (obj);
}


GType vala_assignment_get_type (void) {
	static GType vala_assignment_type_id = 0;
	if (vala_assignment_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaAssignmentClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_assignment_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaAssignment), 0, (GInstanceInitFunc) vala_assignment_instance_init, NULL };
		vala_assignment_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaAssignment", &g_define_type_info, 0);
	}
	return vala_assignment_type_id;
}



GType vala_assignment_operator_get_type (void) {
	static GType vala_assignment_operator_type_id = 0;
	if (G_UNLIKELY (vala_assignment_operator_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_ASSIGNMENT_OPERATOR_NONE, "VALA_ASSIGNMENT_OPERATOR_NONE", "none"}, {VALA_ASSIGNMENT_OPERATOR_SIMPLE, "VALA_ASSIGNMENT_OPERATOR_SIMPLE", "simple"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_OR, "VALA_ASSIGNMENT_OPERATOR_BITWISE_OR", "bitwise-or"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_AND, "VALA_ASSIGNMENT_OPERATOR_BITWISE_AND", "bitwise-and"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR, "VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR", "bitwise-xor"}, {VALA_ASSIGNMENT_OPERATOR_ADD, "VALA_ASSIGNMENT_OPERATOR_ADD", "add"}, {VALA_ASSIGNMENT_OPERATOR_SUB, "VALA_ASSIGNMENT_OPERATOR_SUB", "sub"}, {VALA_ASSIGNMENT_OPERATOR_MUL, "VALA_ASSIGNMENT_OPERATOR_MUL", "mul"}, {VALA_ASSIGNMENT_OPERATOR_DIV, "VALA_ASSIGNMENT_OPERATOR_DIV", "div"}, {VALA_ASSIGNMENT_OPERATOR_PERCENT, "VALA_ASSIGNMENT_OPERATOR_PERCENT", "percent"}, {VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT, "VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT", "shift-left"}, {VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT, "VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT", "shift-right"}, {0, NULL, NULL}};
		vala_assignment_operator_type_id = g_enum_register_static ("ValaAssignmentOperator", values);
	}
	return vala_assignment_operator_type_id;
}




