/* valanullchecker.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valanullchecker.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valadatatype.h>
#include <vala/valaexpression.h>
#include <vala/valanulltype.h>
#include <vala/valareport.h>
#include <vala/valacodenode.h>
#include <vala/valasourcereference.h>
#include <vala/valamethodtype.h>
#include <vala/valamemberaccess.h>
#include <vala/valamember.h>




struct _ValaNullCheckerPrivate {
	ValaCodeContext* context;
	ValaDataType* current_return_type;
};

#define VALA_NULL_CHECKER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_NULL_CHECKER, ValaNullCheckerPrivate))
enum  {
	VALA_NULL_CHECKER_DUMMY_PROPERTY
};
static void vala_null_checker_check_compatible (ValaNullChecker* self, ValaExpression* expr, ValaDataType* target_type);
static void vala_null_checker_check_non_null (ValaNullChecker* self, ValaExpression* expr);
static void vala_null_checker_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file);
static void vala_null_checker_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_null_checker_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_null_checker_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_null_checker_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_null_checker_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed);
static void vala_null_checker_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_null_checker_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_null_checker_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_null_checker_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p);
static void vala_null_checker_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_null_checker_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_null_checker_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c);
static void vala_null_checker_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d);
static void vala_null_checker_real_visit_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_null_checker_real_visit_local_variable (ValaCodeVisitor* base, ValaLocalVariable* local);
static void vala_null_checker_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static void vala_null_checker_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt);
static void vala_null_checker_real_visit_switch_statement (ValaCodeVisitor* base, ValaSwitchStatement* stmt);
static void vala_null_checker_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section);
static void vala_null_checker_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt);
static void vala_null_checker_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt);
static void vala_null_checker_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt);
static void vala_null_checker_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_null_checker_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_null_checker_real_visit_yield_statement (ValaCodeVisitor* base, ValaYieldStatement* stmt);
static void vala_null_checker_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt);
static void vala_null_checker_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt);
static void vala_null_checker_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause);
static void vala_null_checker_real_visit_delete_statement (ValaCodeVisitor* base, ValaDeleteStatement* stmt);
static void vala_null_checker_real_visit_method_call (ValaCodeVisitor* base, ValaMethodCall* expr);
static void vala_null_checker_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr);
static void vala_null_checker_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr);
static void vala_null_checker_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr);
static void vala_null_checker_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr);
static void vala_null_checker_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l);
static void vala_null_checker_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static gpointer vala_null_checker_parent_class = NULL;
static void vala_null_checker_finalize (ValaCodeVisitor* obj);



ValaNullChecker* vala_null_checker_construct (GType object_type) {
	ValaNullChecker* self;
	self = (ValaNullChecker*) g_type_create_instance (object_type);
	return self;
}


ValaNullChecker* vala_null_checker_new (void) {
	return vala_null_checker_construct (VALA_TYPE_NULL_CHECKER);
}


void vala_null_checker_check (ValaNullChecker* self, ValaCodeContext* context) {
	ValaCodeContext* _tmp1;
	ValaCodeContext* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->context = (_tmp1 = (_tmp0 = context, (_tmp0 == NULL) ? NULL : vala_code_context_ref (_tmp0)), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp1);
	vala_code_context_accept (context, (ValaCodeVisitor*) self);
}


static void vala_null_checker_check_compatible (ValaNullChecker* self, ValaExpression* expr, ValaDataType* target_type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
	g_return_if_fail (target_type != NULL);
	if (!vala_data_type_get_nullable (target_type)) {
		if (VALA_IS_NULL_TYPE (vala_expression_get_value_type (expr))) {
			char* _tmp1;
			char* _tmp0;
			_tmp1 = NULL;
			_tmp0 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) expr), _tmp1 = g_strdup_printf ("`null' incompatible with `%s'", _tmp0 = vala_code_node_to_string ((ValaCodeNode*) target_type)));
			_tmp1 = (g_free (_tmp1), NULL);
			_tmp0 = (g_free (_tmp0), NULL);
		} else {
			if (vala_data_type_get_nullable (vala_expression_get_value_type (expr))) {
				char* _tmp4;
				char* _tmp3;
				char* _tmp2;
				_tmp4 = NULL;
				_tmp3 = NULL;
				_tmp2 = NULL;
				vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) expr), _tmp4 = g_strdup_printf ("`%s' incompatible with `%s'", _tmp2 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (expr)), _tmp3 = vala_code_node_to_string ((ValaCodeNode*) target_type)));
				_tmp4 = (g_free (_tmp4), NULL);
				_tmp3 = (g_free (_tmp3), NULL);
				_tmp2 = (g_free (_tmp2), NULL);
			}
		}
	}
}


static void vala_null_checker_check_non_null (ValaNullChecker* self, ValaExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
	if (VALA_IS_NULL_TYPE (vala_expression_get_value_type (expr))) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) expr), "null dereference");
	} else {
		if (vala_data_type_get_nullable (vala_expression_get_value_type (expr))) {
			vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) expr), "possible null dereference");
		}
	}
}


static void vala_null_checker_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (file != NULL);
	vala_source_file_accept_children (file, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (cl != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) cl, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (st != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) st, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (iface != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) iface, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (en != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) en, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (ed != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) ed, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_field (ValaCodeVisitor* base, ValaField* f) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (f != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) f, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_method (ValaCodeVisitor* base, ValaMethod* m) {
	ValaNullChecker * self;
	ValaDataType* _tmp0;
	ValaDataType* old_return_type;
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	ValaDataType* _tmp4;
	ValaDataType* _tmp3;
	self = (ValaNullChecker*) base;
	g_return_if_fail (m != NULL);
	_tmp0 = NULL;
	old_return_type = (_tmp0 = self->priv->current_return_type, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->current_return_type = (_tmp2 = (_tmp1 = vala_method_get_return_type (m), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->current_return_type == NULL) ? NULL : (self->priv->current_return_type = (vala_code_node_unref (self->priv->current_return_type), NULL)), _tmp2);
	vala_code_node_accept_children ((ValaCodeNode*) m, (ValaCodeVisitor*) self);
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_return_type = (_tmp4 = (_tmp3 = old_return_type, (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3)), (self->priv->current_return_type == NULL) ? NULL : (self->priv->current_return_type = (vala_code_node_unref (self->priv->current_return_type), NULL)), _tmp4);
	(old_return_type == NULL) ? NULL : (old_return_type = (vala_code_node_unref (old_return_type), NULL));
}


static void vala_null_checker_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (m != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) m, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (p != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) p, (ValaCodeVisitor*) self);
	if (vala_formal_parameter_get_default_expression (p) != NULL) {
		vala_null_checker_check_compatible (self, vala_formal_parameter_get_default_expression (p), vala_formal_parameter_get_parameter_type (p));
	}
}


static void vala_null_checker_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (prop != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) prop, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (acc != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) acc, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (c != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) c, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (d != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) d, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_block (ValaCodeVisitor* base, ValaBlock* b) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (b != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) b, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_local_variable (ValaCodeVisitor* base, ValaLocalVariable* local) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (local != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) local, (ValaCodeVisitor*) self);
	if (vala_local_variable_get_initializer (local) != NULL) {
		vala_null_checker_check_compatible (self, vala_local_variable_get_initializer (local), vala_local_variable_get_variable_type (local));
	}
}


static void vala_null_checker_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
	vala_null_checker_check_non_null (self, vala_if_statement_get_condition (stmt));
}


static void vala_null_checker_real_visit_switch_statement (ValaCodeVisitor* base, ValaSwitchStatement* stmt) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (section != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) section, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
	vala_null_checker_check_non_null (self, vala_while_statement_get_condition (stmt));
}


static void vala_null_checker_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
	vala_null_checker_check_non_null (self, vala_do_statement_get_condition (stmt));
}


static void vala_null_checker_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
	vala_null_checker_check_non_null (self, vala_for_statement_get_condition (stmt));
}


static void vala_null_checker_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
	vala_null_checker_check_non_null (self, vala_foreach_statement_get_collection (stmt));
}


static void vala_null_checker_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
	if (vala_return_statement_get_return_expression (stmt) != NULL) {
		vala_null_checker_check_compatible (self, vala_return_statement_get_return_expression (stmt), self->priv->current_return_type);
	}
}


static void vala_null_checker_real_visit_yield_statement (ValaCodeVisitor* base, ValaYieldStatement* stmt) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
	vala_null_checker_check_non_null (self, vala_throw_statement_get_error_expression (stmt));
}


static void vala_null_checker_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (clause != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) clause, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_delete_statement (ValaCodeVisitor* base, ValaDeleteStatement* stmt) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_method_call (ValaCodeVisitor* base, ValaMethodCall* expr) {
	ValaNullChecker * self;
	ValaMethodType* _tmp1;
	ValaDataType* _tmp0;
	ValaMethodType* mtype;
	ValaMemberAccess* _tmp3;
	ValaExpression* _tmp2;
	ValaMemberAccess* ma;
	gboolean _tmp4;
	gboolean _tmp5;
	self = (ValaNullChecker*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) self);
	_tmp1 = NULL;
	_tmp0 = NULL;
	mtype = (_tmp1 = (_tmp0 = vala_expression_get_value_type (vala_method_call_get_call (expr)), VALA_IS_METHOD_TYPE (_tmp0) ? ((ValaMethodType*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	_tmp3 = NULL;
	_tmp2 = NULL;
	ma = (_tmp3 = (_tmp2 = vala_method_call_get_call (expr), VALA_IS_MEMBER_ACCESS (_tmp2) ? ((ValaMemberAccess*) _tmp2) : NULL), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
	_tmp4 = FALSE;
	_tmp5 = FALSE;
	if (mtype != NULL) {
		_tmp5 = vala_method_get_binding (vala_method_type_get_method_symbol (mtype)) == MEMBER_BINDING_INSTANCE;
	} else {
		_tmp5 = FALSE;
	}
	if (_tmp5) {
		_tmp4 = ma != NULL;
	} else {
		_tmp4 = FALSE;
	}
	if (_tmp4) {
		vala_null_checker_check_non_null (self, vala_member_access_get_inner (ma));
	}
	(mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL));
	(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
}


static void vala_null_checker_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) self);
	vala_null_checker_check_non_null (self, vala_element_access_get_container (expr));
}


static void vala_null_checker_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (expr != NULL);
	vala_null_checker_check_non_null (self, vala_postfix_expression_get_inner (expr));
}


static void vala_null_checker_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (expr != NULL);
	switch (vala_unary_expression_get_operator (expr)) {
		case VALA_UNARY_OPERATOR_PLUS:
		case VALA_UNARY_OPERATOR_MINUS:
		case VALA_UNARY_OPERATOR_LOGICAL_NEGATION:
		case VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT:
		case VALA_UNARY_OPERATOR_INCREMENT:
		case VALA_UNARY_OPERATOR_DECREMENT:
		{
			vala_null_checker_check_non_null (self, vala_unary_expression_get_inner (expr));
			break;
		}
	}
}


static void vala_null_checker_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (l != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) l, (ValaCodeVisitor*) self);
}


static void vala_null_checker_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a) {
	ValaNullChecker * self;
	self = (ValaNullChecker*) base;
	g_return_if_fail (a != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) a, (ValaCodeVisitor*) self);
	vala_null_checker_check_compatible (self, vala_assignment_get_right (a), vala_expression_get_value_type (vala_assignment_get_left (a)));
}


static void vala_null_checker_class_init (ValaNullCheckerClass * klass) {
	vala_null_checker_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_null_checker_finalize;
	g_type_class_add_private (klass, sizeof (ValaNullCheckerPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_null_checker_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_null_checker_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_null_checker_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_null_checker_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_null_checker_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_domain = vala_null_checker_real_visit_error_domain;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_null_checker_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_null_checker_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_null_checker_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_formal_parameter = vala_null_checker_real_visit_formal_parameter;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_null_checker_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_null_checker_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constructor = vala_null_checker_real_visit_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_destructor = vala_null_checker_real_visit_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_block = vala_null_checker_real_visit_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_local_variable = vala_null_checker_real_visit_local_variable;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_null_checker_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_null_checker_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_statement = vala_null_checker_real_visit_switch_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_section = vala_null_checker_real_visit_switch_section;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_null_checker_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_do_statement = vala_null_checker_real_visit_do_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_null_checker_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_foreach_statement = vala_null_checker_real_visit_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_return_statement = vala_null_checker_real_visit_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_yield_statement = vala_null_checker_real_visit_yield_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_null_checker_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_null_checker_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_catch_clause = vala_null_checker_real_visit_catch_clause;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delete_statement = vala_null_checker_real_visit_delete_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method_call = vala_null_checker_real_visit_method_call;
	VALA_CODE_VISITOR_CLASS (klass)->visit_element_access = vala_null_checker_real_visit_element_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_postfix_expression = vala_null_checker_real_visit_postfix_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_unary_expression = vala_null_checker_real_visit_unary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_reference_transfer_expression = vala_null_checker_real_visit_reference_transfer_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lambda_expression = vala_null_checker_real_visit_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_assignment = vala_null_checker_real_visit_assignment;
}


static void vala_null_checker_instance_init (ValaNullChecker * self) {
	self->priv = VALA_NULL_CHECKER_GET_PRIVATE (self);
}


static void vala_null_checker_finalize (ValaCodeVisitor* obj) {
	ValaNullChecker * self;
	self = VALA_NULL_CHECKER (obj);
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
	(self->priv->current_return_type == NULL) ? NULL : (self->priv->current_return_type = (vala_code_node_unref (self->priv->current_return_type), NULL));
	VALA_CODE_VISITOR_CLASS (vala_null_checker_parent_class)->finalize (obj);
}


GType vala_null_checker_get_type (void) {
	static GType vala_null_checker_type_id = 0;
	if (vala_null_checker_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaNullCheckerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_null_checker_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaNullChecker), 0, (GInstanceInitFunc) vala_null_checker_instance_init, NULL };
		vala_null_checker_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaNullChecker", &g_define_type_info, 0);
	}
	return vala_null_checker_type_id;
}




