/* valavapigen.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vapigen/valavapigen.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valacodecontext.h>
#include <vala/valareport.h>
#include <stdio.h>
#include <vala/valasourcefile.h>
#include <glib/gstdio.h>
#include <vala/valasourcereference.h>
#include <vala/valaparser.h>
#include <vala/valasymbolresolver.h>
#include <vala/valasemanticanalyzer.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valacodewriter.h>
#include <config.h>
#include "vapigen/valagirparser.h"
#include "vapigen/valagidlparser.h"


#define VALA_TYPE_VAPI_GEN (vala_vapi_gen_get_type ())
#define VALA_VAPI_GEN(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_VAPI_GEN, ValaVAPIGen))
#define VALA_VAPI_GEN_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_VAPI_GEN, ValaVAPIGenClass))
#define VALA_IS_VAPI_GEN(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_VAPI_GEN))
#define VALA_IS_VAPI_GEN_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_VAPI_GEN))
#define VALA_VAPI_GEN_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_VAPI_GEN, ValaVAPIGenClass))

typedef struct _ValaVAPIGen ValaVAPIGen;
typedef struct _ValaVAPIGenClass ValaVAPIGenClass;
typedef struct _ValaVAPIGenPrivate ValaVAPIGenPrivate;

struct _ValaVAPIGen {
	GObject parent_instance;
	ValaVAPIGenPrivate * priv;
};

struct _ValaVAPIGenClass {
	GObjectClass parent_class;
};



static char* string_substring (const char* self, glong offset, glong len);
struct _ValaVAPIGenPrivate {
	ValaCodeContext* context;
};

#define VALA_VAPI_GEN_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_VAPI_GEN, ValaVAPIGenPrivate))
enum  {
	VALA_VAPI_GEN_DUMMY_PROPERTY
};
static char* vala_vapi_gen_directory = NULL;
static gboolean vala_vapi_gen_version = FALSE;
static gboolean vala_vapi_gen_quiet_mode = FALSE;
static char** vala_vapi_gen_sources = NULL;
static char** vala_vapi_gen_vapi_directories = NULL;
static char* vala_vapi_gen_library = NULL;
static char** vala_vapi_gen_packages = NULL;
static char* vala_vapi_gen_metadata_filename = NULL;
static gint vala_vapi_gen_quit (ValaVAPIGen* self);
static gboolean vala_vapi_gen_add_package (ValaVAPIGen* self, const char* pkg);
static char** vala_vapi_gen_get_packages_from_depsfile (const char* depsfile, int* result_length1);
static gint vala_vapi_gen_run (ValaVAPIGen* self);
static gint vala_vapi_gen_main (char** args, int args_length1);
static ValaVAPIGen* vala_vapi_gen_construct (GType object_type);
static ValaVAPIGen* vala_vapi_gen_new (void);
static gpointer vala_vapi_gen_parent_class = NULL;
static void vala_vapi_gen_finalize (GObject* obj);
static GType vala_vapi_gen_get_type (void);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);

static const GOptionEntry VALA_VAPI_GEN_options[] = {{"vapidir", (gchar) 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_vapi_gen_vapi_directories, "Look for package bindings in DIRECTORY", "DIRECTORY..."}, {"pkg", (gchar) 0, 0, G_OPTION_ARG_STRING_ARRAY, &vala_vapi_gen_packages, "Include binding for PACKAGE", "PACKAGE..."}, {"library", (gchar) 0, 0, G_OPTION_ARG_STRING, &vala_vapi_gen_library, "Library name", "NAME"}, {"metadata", (gchar) 0, 0, G_OPTION_ARG_FILENAME, &vala_vapi_gen_metadata_filename, "Metadata filename", "FILE"}, {"directory", 'd', 0, G_OPTION_ARG_FILENAME, &vala_vapi_gen_directory, "Output directory", "DIRECTORY"}, {"version", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_vapi_gen_version, "Display version number", NULL}, {"quiet", 'q', 0, G_OPTION_ARG_NONE, &vala_vapi_gen_quiet_mode, "Do not print messages to the console", NULL}, {"", (gchar) 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_vapi_gen_sources, NULL, "FILE..."}, {NULL}};


static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


static gint vala_vapi_gen_quit (ValaVAPIGen* self) {
	g_return_val_if_fail (self != NULL, 0);
	if (vala_report_get_errors () == 0) {
		if (!vala_vapi_gen_quiet_mode) {
			fprintf (stdout, "Generation succeeded - %d warning(s)\n", vala_report_get_warnings ());
		}
		return 0;
	} else {
		if (!vala_vapi_gen_quiet_mode) {
			fprintf (stdout, "Generation failed: %d error(s), %d warning(s)\n", vala_report_get_errors (), vala_report_get_warnings ());
		}
		return 1;
	}
}


static gboolean vala_vapi_gen_add_package (ValaVAPIGen* self, const char* pkg) {
	char* package_path;
	ValaSourceFile* _tmp2;
	gboolean _tmp3;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (pkg != NULL, FALSE);
	if (vala_code_context_has_package (self->priv->context, pkg)) {
		/* ignore multiple occurences of the same package*/
		return TRUE;
	}
	package_path = vala_code_context_get_package_path (self->priv->context, pkg, vala_vapi_gen_vapi_directories, _vala_array_length (vala_vapi_gen_vapi_directories));
	if (package_path == NULL) {
		gboolean _tmp1;
		return (_tmp1 = FALSE, package_path = (g_free (package_path), NULL), _tmp1);
	}
	vala_code_context_add_package (self->priv->context, pkg);
	_tmp2 = NULL;
	vala_code_context_add_source_file (self->priv->context, _tmp2 = vala_source_file_new (self->priv->context, package_path, TRUE, NULL));
	(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_file_unref (_tmp2), NULL));
	return (_tmp3 = TRUE, package_path = (g_free (package_path), NULL), _tmp3);
}


static char** vala_vapi_gen_get_packages_from_depsfile (const char* depsfile, int* result_length1) {
	GError * inner_error;
	g_return_val_if_fail (depsfile != NULL, NULL);
	inner_error = NULL;
	{
		char* contents;
		char* _tmp2;
		gboolean _tmp1;
		char* _tmp0;
		char** _tmp3;
		char** _tmp4;
		char** _tmp5;
		contents = NULL;
		_tmp2 = NULL;
		_tmp0 = NULL;
		_tmp1 = g_file_get_contents (depsfile, &_tmp0, NULL, &inner_error);
		contents = (_tmp2 = _tmp0, contents = (g_free (contents), NULL), _tmp2);
		_tmp1;
		if (inner_error != NULL) {
			contents = (g_free (contents), NULL);
			if (inner_error->domain == G_FILE_ERROR) {
				goto __catch4_g_file_error;
			}
			goto __finally4;
		}
		_tmp3 = NULL;
		_tmp4 = NULL;
		_tmp5 = NULL;
		return (_tmp5 = (_tmp4 = _tmp3 = g_strsplit (g_strstrip (contents), "\n", 0), *result_length1 = _vala_array_length (_tmp3), _tmp4), contents = (g_free (contents), NULL), _tmp5);
	}
	goto __finally4;
	__catch4_g_file_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			gpointer _tmp6;
			char** _tmp7;
			/* deps files are optional*/
			_tmp7 = NULL;
			return (_tmp7 = (_tmp6 = NULL, *result_length1 = 0, _tmp6), (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), _tmp7);
		}
	}
	__finally4:
	if (inner_error != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return NULL;
	}
}


static gint vala_vapi_gen_run (ValaVAPIGen* self) {
	ValaCodeContext* _tmp0;
	char** _tmp15;
	ValaParser* parser;
	ValaGirParser* girparser;
	ValaGIdlParser* gidlparser;
	ValaSymbolResolver* resolver;
	ValaSemanticAnalyzer* analyzer;
	gint _tmp26;
	g_return_val_if_fail (self != NULL, 0);
	_tmp0 = NULL;
	self->priv->context = (_tmp0 = vala_code_context_new (), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp0);
	/* default package */
	if (!vala_vapi_gen_add_package (self, "glib-2.0")) {
		vala_report_error (NULL, "glib-2.0 not found in specified Vala API directories");
	}
	if (!vala_vapi_gen_add_package (self, "gobject-2.0")) {
		vala_report_error (NULL, "gobject-2.0 not found in specified Vala API directories");
	}
	/* load packages from .deps file */
	{
		char** source_collection;
		int source_collection_length1;
		int source_it;
		source_collection = vala_vapi_gen_sources;
		source_collection_length1 = _vala_array_length (vala_vapi_gen_sources);
		for (source_it = 0; source_it < _vala_array_length (vala_vapi_gen_sources); source_it = source_it + 1) {
			const char* _tmp7;
			char* source;
			_tmp7 = NULL;
			source = (_tmp7 = source_collection[source_it], (_tmp7 == NULL) ? NULL : g_strdup (_tmp7));
			{
				char* _tmp1;
				char* _tmp2;
				char* depsfile;
				char** _tmp4;
				gint deps_size;
				gint deps_length1;
				gint _tmp3;
				char** deps;
				if (!g_str_has_suffix (source, ".gi")) {
					source = (g_free (source), NULL);
					continue;
				}
				_tmp1 = NULL;
				_tmp2 = NULL;
				depsfile = (_tmp2 = g_strconcat (_tmp1 = string_substring (source, (glong) 0, g_utf8_strlen (source, -1) - g_utf8_strlen ("gi", -1)), "deps", NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp2);
				if (!g_file_test (depsfile, G_FILE_TEST_EXISTS)) {
					source = (g_free (source), NULL);
					depsfile = (g_free (depsfile), NULL);
					continue;
				}
				_tmp4 = NULL;
				deps = (_tmp4 = vala_vapi_gen_get_packages_from_depsfile (depsfile, &_tmp3), deps_length1 = _tmp3, deps_size = deps_length1, _tmp4);
				{
					char** dep_collection;
					int dep_collection_length1;
					int dep_it;
					dep_collection = deps;
					dep_collection_length1 = deps_length1;
					for (dep_it = 0; dep_it < deps_length1; dep_it = dep_it + 1) {
						const char* _tmp6;
						char* dep;
						_tmp6 = NULL;
						dep = (_tmp6 = dep_collection[dep_it], (_tmp6 == NULL) ? NULL : g_strdup (_tmp6));
						{
							if (!vala_vapi_gen_add_package (self, dep)) {
								char* _tmp5;
								_tmp5 = NULL;
								vala_report_error (NULL, _tmp5 = g_strdup_printf ("%s not found in specified Vala API directories", dep));
								_tmp5 = (g_free (_tmp5), NULL);
							}
							dep = (g_free (dep), NULL);
						}
					}
				}
				source = (g_free (source), NULL);
				depsfile = (g_free (depsfile), NULL);
				deps = (_vala_array_free (deps, deps_length1, (GDestroyNotify) g_free), NULL);
			}
		}
	}
	if (vala_vapi_gen_packages != NULL) {
		char** _tmp10;
		{
			char** package_collection;
			int package_collection_length1;
			int package_it;
			package_collection = vala_vapi_gen_packages;
			package_collection_length1 = _vala_array_length (vala_vapi_gen_packages);
			for (package_it = 0; package_it < _vala_array_length (vala_vapi_gen_packages); package_it = package_it + 1) {
				const char* _tmp9;
				char* package;
				_tmp9 = NULL;
				package = (_tmp9 = package_collection[package_it], (_tmp9 == NULL) ? NULL : g_strdup (_tmp9));
				{
					if (!vala_vapi_gen_add_package (self, package)) {
						char* _tmp8;
						_tmp8 = NULL;
						vala_report_error (NULL, _tmp8 = g_strdup_printf ("%s not found in specified Vala API directories", package));
						_tmp8 = (g_free (_tmp8), NULL);
					}
					package = (g_free (package), NULL);
				}
			}
		}
		_tmp10 = NULL;
		vala_vapi_gen_packages = (_tmp10 = NULL, vala_vapi_gen_packages = (_vala_array_free (vala_vapi_gen_packages, _vala_array_length (vala_vapi_gen_packages), (GDestroyNotify) g_free), NULL), _tmp10);
	}
	if (vala_report_get_errors () > 0) {
		return vala_vapi_gen_quit (self);
	}
	{
		char** source_collection;
		int source_collection_length1;
		int source_it;
		source_collection = vala_vapi_gen_sources;
		source_collection_length1 = _vala_array_length (vala_vapi_gen_sources);
		for (source_it = 0; source_it < _vala_array_length (vala_vapi_gen_sources); source_it = source_it + 1) {
			const char* _tmp14;
			char* source;
			_tmp14 = NULL;
			source = (_tmp14 = source_collection[source_it], (_tmp14 == NULL) ? NULL : g_strdup (_tmp14));
			{
				if (g_file_test (source, G_FILE_TEST_EXISTS)) {
					ValaSourceFile* _tmp12;
					_tmp12 = NULL;
					vala_code_context_add_source_file (self->priv->context, _tmp12 = vala_source_file_new (self->priv->context, source, TRUE, NULL));
					(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_source_file_unref (_tmp12), NULL));
				} else {
					char* _tmp13;
					_tmp13 = NULL;
					vala_report_error (NULL, _tmp13 = g_strdup_printf ("%s not found", source));
					_tmp13 = (g_free (_tmp13), NULL);
				}
				source = (g_free (source), NULL);
			}
		}
	}
	_tmp15 = NULL;
	vala_vapi_gen_sources = (_tmp15 = NULL, vala_vapi_gen_sources = (_vala_array_free (vala_vapi_gen_sources, _vala_array_length (vala_vapi_gen_sources), (GDestroyNotify) g_free), NULL), _tmp15);
	if (vala_report_get_errors () > 0) {
		return vala_vapi_gen_quit (self);
	}
	parser = vala_parser_new ();
	vala_parser_parse (parser, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp17;
		return (_tmp17 = vala_vapi_gen_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), _tmp17);
	}
	girparser = vala_gir_parser_new ();
	if (vala_vapi_gen_metadata_filename != NULL) {
		vala_gir_parser_parse_metadata (girparser, vala_vapi_gen_metadata_filename);
	}
	vala_gir_parser_parse (girparser, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp18;
		return (_tmp18 = vala_vapi_gen_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (girparser == NULL) ? NULL : (girparser = (vala_code_visitor_unref (girparser), NULL)), _tmp18);
	}
	gidlparser = vala_gidl_parser_new ();
	vala_gidl_parser_parse (gidlparser, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp19;
		return (_tmp19 = vala_vapi_gen_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (girparser == NULL) ? NULL : (girparser = (vala_code_visitor_unref (girparser), NULL)), (gidlparser == NULL) ? NULL : (gidlparser = (vala_code_visitor_unref (gidlparser), NULL)), _tmp19);
	}
	resolver = vala_symbol_resolver_new ();
	vala_symbol_resolver_resolve (resolver, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp20;
		return (_tmp20 = vala_vapi_gen_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (girparser == NULL) ? NULL : (girparser = (vala_code_visitor_unref (girparser), NULL)), (gidlparser == NULL) ? NULL : (gidlparser = (vala_code_visitor_unref (gidlparser), NULL)), (resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL)), _tmp20);
	}
	analyzer = vala_semantic_analyzer_new ();
	vala_semantic_analyzer_analyze (analyzer, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp21;
		return (_tmp21 = vala_vapi_gen_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (girparser == NULL) ? NULL : (girparser = (vala_code_visitor_unref (girparser), NULL)), (gidlparser == NULL) ? NULL : (gidlparser = (vala_code_visitor_unref (gidlparser), NULL)), (resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL)), (analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL)), _tmp21);
	}
	if (vala_vapi_gen_library != NULL) {
		ValaCodeWriter* interface_writer;
		char* _tmp24;
		char* _tmp25;
		/* interface writer ignores external packages*/
		{
			GeeList* _tmp22;
			GeeIterator* _tmp23;
			GeeIterator* _file_it;
			/* interface writer ignores external packages*/
			_tmp22 = NULL;
			_tmp23 = NULL;
			_file_it = (_tmp23 = gee_iterable_iterator ((GeeIterable*) (_tmp22 = vala_code_context_get_source_files (self->priv->context))), (_tmp22 == NULL) ? NULL : (_tmp22 = (gee_collection_object_unref (_tmp22), NULL)), _tmp23);
			/* interface writer ignores external packages*/
			while (gee_iterator_next (_file_it)) {
				ValaSourceFile* file;
				/* interface writer ignores external packages*/
				file = (ValaSourceFile*) gee_iterator_get (_file_it);
				if (!g_str_has_suffix (vala_source_file_get_filename (file), ".vapi")) {
					vala_source_file_set_external_package (file, FALSE);
				}
				(file == NULL) ? NULL : (file = (vala_source_file_unref (file), NULL));
			}
			(_file_it == NULL) ? NULL : (_file_it = (gee_collection_object_unref (_file_it), NULL));
		}
		interface_writer = vala_code_writer_new (FALSE);
		_tmp24 = NULL;
		vala_code_writer_write_file (interface_writer, self->priv->context, _tmp24 = g_strdup_printf ("%s.vapi", vala_vapi_gen_library));
		_tmp24 = (g_free (_tmp24), NULL);
		_tmp25 = NULL;
		vala_vapi_gen_library = (_tmp25 = NULL, vala_vapi_gen_library = (g_free (vala_vapi_gen_library), NULL), _tmp25);
		(interface_writer == NULL) ? NULL : (interface_writer = (vala_code_visitor_unref (interface_writer), NULL));
	}
	return (_tmp26 = vala_vapi_gen_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (girparser == NULL) ? NULL : (girparser = (vala_code_visitor_unref (girparser), NULL)), (gidlparser == NULL) ? NULL : (gidlparser = (vala_code_visitor_unref (gidlparser), NULL)), (resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL)), (analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL)), _tmp26);
}


static gint vala_vapi_gen_main (char** args, int args_length1) {
	GError * inner_error;
	ValaVAPIGen* vapigen;
	gint _tmp3;
	inner_error = NULL;
	{
		GOptionContext* opt_context;
		opt_context = g_option_context_new ("- Vala API Generator");
		g_option_context_set_help_enabled (opt_context, TRUE);
		g_option_context_add_main_entries (opt_context, VALA_VAPI_GEN_options, NULL);
		g_option_context_parse (opt_context, &args_length1, &args, &inner_error);
		if (inner_error != NULL) {
			(opt_context == NULL) ? NULL : (opt_context = (g_option_context_free (opt_context), NULL));
			if (inner_error->domain == G_OPTION_ERROR) {
				goto __catch5_g_option_error;
			}
			goto __finally5;
		}
		(opt_context == NULL) ? NULL : (opt_context = (g_option_context_free (opt_context), NULL));
	}
	goto __finally5;
	__catch5_g_option_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			gint _tmp0;
			fprintf (stdout, "%s\n", e->message);
			fprintf (stdout, "Run '%s --help' to see a full list of available command line options.\n", args[0]);
			return (_tmp0 = 1, (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), _tmp0);
		}
	}
	__finally5:
	if (inner_error != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return 0;
	}
	if (vala_vapi_gen_version) {
		fprintf (stdout, "Vala API Generator %s\n", PACKAGE_VERSION);
		return 0;
	}
	if (vala_vapi_gen_sources == NULL) {
		fprintf (stderr, "No source file specified.\n");
		return 1;
	}
	vapigen = vala_vapi_gen_new ();
	return (_tmp3 = vala_vapi_gen_run (vapigen), (vapigen == NULL) ? NULL : (vapigen = (g_object_unref (vapigen), NULL)), _tmp3);
}


int main (int argc, char ** argv) {
	g_type_init ();
	return vala_vapi_gen_main (argv, argc);
}


static ValaVAPIGen* vala_vapi_gen_construct (GType object_type) {
	ValaVAPIGen * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


static ValaVAPIGen* vala_vapi_gen_new (void) {
	return vala_vapi_gen_construct (VALA_TYPE_VAPI_GEN);
}


static void vala_vapi_gen_class_init (ValaVAPIGenClass * klass) {
	vala_vapi_gen_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaVAPIGenPrivate));
	G_OBJECT_CLASS (klass)->finalize = vala_vapi_gen_finalize;
}


static void vala_vapi_gen_instance_init (ValaVAPIGen * self) {
	self->priv = VALA_VAPI_GEN_GET_PRIVATE (self);
}


static void vala_vapi_gen_finalize (GObject* obj) {
	ValaVAPIGen * self;
	self = VALA_VAPI_GEN (obj);
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
	G_OBJECT_CLASS (vala_vapi_gen_parent_class)->finalize (obj);
}


static GType vala_vapi_gen_get_type (void) {
	static GType vala_vapi_gen_type_id = 0;
	if (vala_vapi_gen_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaVAPIGenClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_vapi_gen_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaVAPIGen), 0, (GInstanceInitFunc) vala_vapi_gen_instance_init, NULL };
		vala_vapi_gen_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaVAPIGen", &g_define_type_info, 0);
	}
	return vala_vapi_gen_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}




