/* valabaseaccess.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valabaseaccess.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valareport.h>
#include <vala/valastruct.h>
#include <vala/valaclass.h>
#include <vala/valaobjecttype.h>
#include <vala/valaobjecttypesymbol.h>
#include <vala/valadatatype.h>




enum  {
	VALA_BASE_ACCESS_DUMMY_PROPERTY
};
static void vala_base_access_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_base_access_real_to_string (ValaCodeNode* base);
static gboolean vala_base_access_real_is_pure (ValaExpression* base);
static gboolean vala_base_access_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_base_access_parent_class = NULL;



/**
 * Creates a new base access expression.
 *
 * @param source reference to source code
 * @return       newly created base access expression
 */
ValaBaseAccess* vala_base_access_construct (GType object_type, ValaSourceReference* source) {
	ValaBaseAccess* self;
	self = (ValaBaseAccess*) g_type_create_instance (object_type);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source);
	return self;
}


ValaBaseAccess* vala_base_access_new (ValaSourceReference* source) {
	return vala_base_access_construct (VALA_TYPE_BASE_ACCESS, source);
}


static void vala_base_access_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaBaseAccess * self;
	self = (ValaBaseAccess*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_base_access (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static char* vala_base_access_real_to_string (ValaCodeNode* base) {
	ValaBaseAccess * self;
	self = (ValaBaseAccess*) base;
	return g_strdup ("base");
}


static gboolean vala_base_access_real_is_pure (ValaExpression* base) {
	ValaBaseAccess * self;
	self = (ValaBaseAccess*) base;
	return TRUE;
}


static gboolean vala_base_access_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaBaseAccess * self;
	self = (ValaBaseAccess*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (!vala_semantic_analyzer_is_in_instance_method (analyzer)) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Base access invalid outside of instance methods");
		return FALSE;
	}
	if (analyzer->current_class == NULL) {
		if (analyzer->current_struct == NULL) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Base access invalid outside of class and struct");
			return FALSE;
		} else {
			if (vala_struct_get_base_type (analyzer->current_struct) == NULL) {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Base access invalid without base type");
				return FALSE;
			}
		}
		vala_expression_set_value_type ((ValaExpression*) self, vala_struct_get_base_type (analyzer->current_struct));
	} else {
		if (vala_class_get_base_class (analyzer->current_class) == NULL) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Base access invalid without base class");
			return FALSE;
		} else {
			ValaObjectType* _tmp5;
			_tmp5 = NULL;
			vala_expression_set_value_type ((ValaExpression*) self, (ValaDataType*) (_tmp5 = vala_object_type_new ((ValaObjectTypeSymbol*) vala_class_get_base_class (analyzer->current_class))));
			(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
		}
	}
	vala_expression_set_symbol_reference ((ValaExpression*) self, (ValaSymbol*) vala_data_type_get_data_type (vala_expression_get_value_type ((ValaExpression*) self)));
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


static void vala_base_access_class_init (ValaBaseAccessClass * klass) {
	vala_base_access_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->accept = vala_base_access_real_accept;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_base_access_real_to_string;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_base_access_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_base_access_real_check;
}


static void vala_base_access_instance_init (ValaBaseAccess * self) {
}


GType vala_base_access_get_type (void) {
	static GType vala_base_access_type_id = 0;
	if (vala_base_access_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaBaseAccessClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_base_access_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaBaseAccess), 0, (GInstanceInitFunc) vala_base_access_instance_init, NULL };
		vala_base_access_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaBaseAccess", &g_define_type_info, 0);
	}
	return vala_base_access_type_id;
}




