/* valaparser.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaparser.h>
#include <stdlib.h>
#include <string.h>
#include <gee/list.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valascanner.h>
#include <vala/valatokentype.h>
#include <vala/valasourcelocation.h>
#include <vala/valaunresolvedsymbol.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valausingdirective.h>
#include <vala/valasymbol.h>
#include <vala/valanamespace.h>
#include <vala/valaattribute.h>
#include <vala/valaexpression.h>
#include <vala/valaunaryexpression.h>
#include <vala/valareferencetransferexpression.h>
#include <vala/valadatatype.h>
#include <vala/valavoidtype.h>
#include <vala/valapointertype.h>
#include <vala/valaunresolvedtype.h>
#include <vala/valaarraytype.h>
#include <vala/valacastexpression.h>
#include <vala/valapointerindirection.h>
#include <vala/valaaddressofexpression.h>
#include <vala/valabooleanliteral.h>
#include <vala/valaintegerliteral.h>
#include <vala/valarealliteral.h>
#include <vala/valacharacterliteral.h>
#include <vala/valacodenode.h>
#include <vala/valastringliteral.h>
#include <vala/valanullliteral.h>
#include <vala/valainitializerlist.h>
#include <vala/valatuple.h>
#include <vala/valamemberaccess.h>
#include <vala/valabaseaccess.h>
#include <vala/valamemberinitializer.h>
#include <vala/valaobjectcreationexpression.h>
#include <vala/valaarraycreationexpression.h>
#include <vala/valasizeofexpression.h>
#include <vala/valatypeofexpression.h>
#include <vala/valamethodcall.h>
#include <vala/valaelementaccess.h>
#include <vala/valapostfixexpression.h>
#include <vala/valabinaryexpression.h>
#include <vala/valatypecheck.h>
#include <vala/valaconditionalexpression.h>
#include <vala/valalambdaexpression.h>
#include <vala/valablock.h>
#include <vala/valastatement.h>
#include <vala/valaemptystatement.h>
#include <vala/valaswitchstatement.h>
#include <vala/valaswitchsection.h>
#include <vala/valaswitchlabel.h>
#include <vala/valawhilestatement.h>
#include <vala/valadostatement.h>
#include <vala/valalocalvariable.h>
#include <vala/valadeclarationstatement.h>
#include <vala/valaforstatement.h>
#include <vala/valaforeachstatement.h>
#include <vala/valabreakstatement.h>
#include <vala/valacontinuestatement.h>
#include <vala/valareturnstatement.h>
#include <vala/valayieldstatement.h>
#include <vala/valathrowstatement.h>
#include <vala/valacatchclause.h>
#include <vala/valatrystatement.h>
#include <vala/valalockstatement.h>
#include <vala/valadeletestatement.h>
#include <vala/valaexpressionstatement.h>
#include <vala/valaifstatement.h>
#include <vala/valaassignment.h>
#include <vala/valaconstructor.h>
#include <vala/valamember.h>
#include <vala/valadestructor.h>
#include <vala/valatypeparameter.h>
#include <vala/valaclass.h>
#include <vala/valaobjecttypesymbol.h>
#include <vala/valacreationmethod.h>
#include <vala/valamethod.h>
#include <vala/valaenum.h>
#include <vala/valaenumvalue.h>
#include <vala/valaerrordomain.h>
#include <vala/valaerrorcode.h>
#include <vala/valainterface.h>
#include <vala/valastruct.h>
#include <vala/valaformalparameter.h>
#include <vala/valadelegate.h>
#include <vala/valasignal.h>
#include <vala/valaconstant.h>
#include <vala/valafield.h>
#include <vala/valaproperty.h>
#include <vala/valapropertyaccessor.h>


#define VALA_PARSER_TYPE_MODIFIER_FLAGS (vala_parser_modifier_flags_get_type ())

#define VALA_PARSER_TYPE_RECOVERY_STATE (vala_parser_recovery_state_get_type ())
typedef struct _ValaParserTokenInfo ValaParserTokenInfo;

#define VALA_PARSER_TYPE_TOKEN_INFO (vala_parser_token_info_get_type ())

typedef enum  {
	VALA_PARSER_MODIFIER_FLAGS_NONE,
	VALA_PARSER_MODIFIER_FLAGS_ABSTRACT = 1 << 0,
	VALA_PARSER_MODIFIER_FLAGS_CLASS = 1 << 1,
	VALA_PARSER_MODIFIER_FLAGS_EXTERN = 1 << 2,
	VALA_PARSER_MODIFIER_FLAGS_INLINE = 1 << 3,
	VALA_PARSER_MODIFIER_FLAGS_NEW = 1 << 4,
	VALA_PARSER_MODIFIER_FLAGS_OVERRIDE = 1 << 5,
	VALA_PARSER_MODIFIER_FLAGS_STATIC = 1 << 6,
	VALA_PARSER_MODIFIER_FLAGS_VIRTUAL = 1 << 7
} ValaParserModifierFlags;

typedef enum  {
	VALA_PARSER_RECOVERY_STATE_EOF,
	VALA_PARSER_RECOVERY_STATE_DECLARATION_BEGIN,
	VALA_PARSER_RECOVERY_STATE_STATEMENT_BEGIN
} ValaParserRecoveryState;

struct _ValaParserTokenInfo {
	ValaTokenType type;
	ValaSourceLocation begin;
	ValaSourceLocation end;
};



static gboolean string_contains (const char* self, const char* needle);
static glong string_get_length (const char* self);
static char* string_substring (const char* self, glong offset, glong len);
struct _ValaParserPrivate {
	ValaScanner* scanner;
	ValaCodeContext* context;
	ValaParserTokenInfo* tokens;
	gint tokens_length1;
	gint tokens_size;
	gint index;
	gint size;
	char* comment;
};

#define VALA_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_PARSER, ValaParserPrivate))
enum  {
	VALA_PARSER_DUMMY_PROPERTY
};
GType vala_parser_modifier_flags_get_type (void);
GType vala_parser_recovery_state_get_type (void);
#define VALA_PARSER_BUFFER_SIZE 32
static void vala_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static inline gboolean vala_parser_next (ValaParser* self);
static inline void vala_parser_prev (ValaParser* self);
static inline ValaTokenType vala_parser_current (ValaParser* self);
static inline gboolean vala_parser_accept (ValaParser* self, ValaTokenType type);
static char* vala_parser_get_error (ValaParser* self, const char* msg);
static inline gboolean vala_parser_expect (ValaParser* self, ValaTokenType type, GError** error);
static inline ValaSourceLocation vala_parser_get_location (ValaParser* self);
static char* vala_parser_get_current_string (ValaParser* self);
static char* vala_parser_get_last_string (ValaParser* self);
static ValaSourceReference* vala_parser_get_src (ValaParser* self, const ValaSourceLocation* begin);
static ValaSourceReference* vala_parser_get_src_com (ValaParser* self, const ValaSourceLocation* begin);
static ValaSourceReference* vala_parser_get_current_src (ValaParser* self);
static ValaSourceReference* vala_parser_get_last_src (ValaParser* self);
static void vala_parser_rollback (ValaParser* self, const ValaSourceLocation* location);
static void vala_parser_skip_identifier (ValaParser* self, GError** error);
static char* vala_parser_parse_identifier (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_literal (ValaParser* self, GError** error);
static void vala_parser_skip_symbol_name (ValaParser* self, GError** error);
static ValaUnresolvedSymbol* vala_parser_parse_symbol_name (ValaParser* self, GError** error);
static void vala_parser_skip_type (ValaParser* self, GError** error);
static ValaDataType* vala_parser_parse_type (ValaParser* self, gboolean owned_by_default, GError** error);
static GeeList* vala_parser_parse_argument_list (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_argument (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_primary_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_simple_name (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_tuple (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_member_access (ValaParser* self, const ValaSourceLocation* begin, ValaExpression* inner, GError** error);
static ValaExpression* vala_parser_parse_pointer_member_access (ValaParser* self, const ValaSourceLocation* begin, ValaExpression* inner, GError** error);
static ValaExpression* vala_parser_parse_method_call (ValaParser* self, const ValaSourceLocation* begin, ValaExpression* inner, GError** error);
static ValaExpression* vala_parser_parse_element_access (ValaParser* self, const ValaSourceLocation* begin, ValaExpression* inner, GError** error);
static GeeList* vala_parser_parse_expression_list (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_this_access (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_base_access (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_post_increment_expression (ValaParser* self, const ValaSourceLocation* begin, ValaExpression* inner, GError** error);
static ValaExpression* vala_parser_parse_post_decrement_expression (ValaParser* self, const ValaSourceLocation* begin, ValaExpression* inner, GError** error);
static ValaExpression* vala_parser_parse_object_or_array_creation_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_object_creation_expression (ValaParser* self, const ValaSourceLocation* begin, ValaMemberAccess* member, GError** error);
static ValaExpression* vala_parser_parse_array_creation_expression (ValaParser* self, const ValaSourceLocation* begin, ValaMemberAccess* member, GError** error);
static GeeList* vala_parser_parse_object_initializer (ValaParser* self, GError** error);
static ValaMemberInitializer* vala_parser_parse_member_initializer (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_sizeof_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_typeof_expression (ValaParser* self, GError** error);
static ValaUnaryOperator vala_parser_get_unary_operator (ValaParser* self, ValaTokenType token_type);
static ValaExpression* vala_parser_parse_unary_expression (ValaParser* self, GError** error);
static ValaBinaryOperator vala_parser_get_binary_operator (ValaParser* self, ValaTokenType token_type);
static ValaExpression* vala_parser_parse_multiplicative_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_additive_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_shift_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_relational_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_equality_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_and_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_exclusive_or_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_inclusive_or_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_in_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_conditional_and_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_conditional_or_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_conditional_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_lambda_expression (ValaParser* self, GError** error);
static ValaAssignmentOperator vala_parser_get_assignment_operator (ValaParser* self, ValaTokenType token_type);
static ValaExpression* vala_parser_parse_expression (ValaParser* self, GError** error);
static void vala_parser_parse_statements (ValaParser* self, ValaBlock* block, GError** error);
static gboolean vala_parser_is_expression (ValaParser* self, GError** error);
static ValaBlock* vala_parser_parse_embedded_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_embedded_statement_without_block (ValaParser* self, GError** error);
static ValaBlock* vala_parser_parse_block (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_empty_statement (ValaParser* self, GError** error);
static void vala_parser_parse_local_variable_declarations (ValaParser* self, ValaBlock* block, GError** error);
static ValaLocalVariable* vala_parser_parse_local_variable (ValaParser* self, ValaDataType* variable_type, GError** error);
static ValaStatement* vala_parser_parse_expression_statement (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_statement_expression (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_if_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_switch_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_while_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_do_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_for_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_foreach_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_break_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_continue_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_return_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_yield_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_throw_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_try_statement (ValaParser* self, GError** error);
static void vala_parser_parse_catch_clauses (ValaParser* self, GeeList* catch_clauses, GError** error);
static ValaBlock* vala_parser_parse_finally_clause (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_lock_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_delete_statement (ValaParser* self, GError** error);
static GeeList* vala_parser_parse_attributes (ValaParser* self, GError** error);
static void vala_parser_set_attributes (ValaParser* self, ValaCodeNode* node, GeeList* attributes);
static ValaSymbol* vala_parser_parse_declaration (ValaParser* self, GError** error);
static void vala_parser_parse_declarations (ValaParser* self, ValaSymbol* parent, gboolean root, GError** error);
static ValaParserRecoveryState vala_parser_recover (ValaParser* self);
static ValaNamespace* vala_parser_parse_namespace_declaration (ValaParser* self, GeeList* attrs, GError** error);
static void vala_parser_parse_namespace_member (ValaParser* self, ValaNamespace* ns, GError** error);
static void vala_parser_parse_using_directives (ValaParser* self, GError** error);
static ValaSymbol* vala_parser_parse_class_declaration (ValaParser* self, GeeList* attrs, GError** error);
static void vala_parser_parse_class_member (ValaParser* self, ValaClass* cl, GError** error);
static ValaConstant* vala_parser_parse_constant_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaField* vala_parser_parse_field_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaInitializerList* vala_parser_parse_initializer (ValaParser* self, GError** error);
static ValaMethod* vala_parser_parse_method_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaProperty* vala_parser_parse_property_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaSignal* vala_parser_parse_signal_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaConstructor* vala_parser_parse_constructor_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaDestructor* vala_parser_parse_destructor_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaSymbol* vala_parser_parse_struct_declaration (ValaParser* self, GeeList* attrs, GError** error);
static void vala_parser_parse_struct_member (ValaParser* self, ValaStruct* st, GError** error);
static ValaSymbol* vala_parser_parse_interface_declaration (ValaParser* self, GeeList* attrs, GError** error);
static void vala_parser_parse_interface_member (ValaParser* self, ValaInterface* iface, GError** error);
static ValaSymbol* vala_parser_parse_enum_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaSymbol* vala_parser_parse_errordomain_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaSymbolAccessibility vala_parser_parse_access_modifier (ValaParser* self, ValaSymbolAccessibility default_access);
static ValaParserModifierFlags vala_parser_parse_type_declaration_modifiers (ValaParser* self);
static ValaParserModifierFlags vala_parser_parse_member_declaration_modifiers (ValaParser* self);
static ValaFormalParameter* vala_parser_parse_parameter (ValaParser* self, GError** error);
static ValaCreationMethod* vala_parser_parse_creation_method_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaSymbol* vala_parser_parse_delegate_declaration (ValaParser* self, GeeList* attrs, GError** error);
static GeeList* vala_parser_parse_type_parameter_list (ValaParser* self, GError** error);
static void vala_parser_skip_type_argument_list (ValaParser* self, GError** error);
static GeeList* vala_parser_parse_type_argument_list (ValaParser* self, gboolean maybe_expression, GError** error);
static ValaMemberAccess* vala_parser_parse_member_name (ValaParser* self, GError** error);
static gboolean vala_parser_is_declaration_keyword (ValaParser* self, ValaTokenType type);
static ValaParserTokenInfo* vala_parser_token_info_dup (const ValaParserTokenInfo* self);
static void vala_parser_token_info_free (ValaParserTokenInfo* self);
static GType vala_parser_token_info_get_type (void);
static gpointer vala_parser_parent_class = NULL;
static void vala_parser_finalize (ValaCodeVisitor* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



static gboolean string_contains (const char* self, const char* needle) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (needle != NULL, FALSE);
	return strstr (self, needle) != NULL;
}


static glong string_get_length (const char* self) {
	g_return_val_if_fail (self != NULL, 0L);
	return g_utf8_strlen (self, -1);
}


static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}



GType vala_parser_modifier_flags_get_type (void) {
	static GType vala_parser_modifier_flags_type_id = 0;
	if (G_UNLIKELY (vala_parser_modifier_flags_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_PARSER_MODIFIER_FLAGS_NONE, "VALA_PARSER_MODIFIER_FLAGS_NONE", "none"}, {VALA_PARSER_MODIFIER_FLAGS_ABSTRACT, "VALA_PARSER_MODIFIER_FLAGS_ABSTRACT", "abstract"}, {VALA_PARSER_MODIFIER_FLAGS_CLASS, "VALA_PARSER_MODIFIER_FLAGS_CLASS", "class"}, {VALA_PARSER_MODIFIER_FLAGS_EXTERN, "VALA_PARSER_MODIFIER_FLAGS_EXTERN", "extern"}, {VALA_PARSER_MODIFIER_FLAGS_INLINE, "VALA_PARSER_MODIFIER_FLAGS_INLINE", "inline"}, {VALA_PARSER_MODIFIER_FLAGS_NEW, "VALA_PARSER_MODIFIER_FLAGS_NEW", "new"}, {VALA_PARSER_MODIFIER_FLAGS_OVERRIDE, "VALA_PARSER_MODIFIER_FLAGS_OVERRIDE", "override"}, {VALA_PARSER_MODIFIER_FLAGS_STATIC, "VALA_PARSER_MODIFIER_FLAGS_STATIC", "static"}, {VALA_PARSER_MODIFIER_FLAGS_VIRTUAL, "VALA_PARSER_MODIFIER_FLAGS_VIRTUAL", "virtual"}, {0, NULL, NULL}};
		vala_parser_modifier_flags_type_id = g_enum_register_static ("ValaParserModifierFlags", values);
	}
	return vala_parser_modifier_flags_type_id;
}



GType vala_parser_recovery_state_get_type (void) {
	static GType vala_parser_recovery_state_type_id = 0;
	if (G_UNLIKELY (vala_parser_recovery_state_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_PARSER_RECOVERY_STATE_EOF, "VALA_PARSER_RECOVERY_STATE_EOF", "eof"}, {VALA_PARSER_RECOVERY_STATE_DECLARATION_BEGIN, "VALA_PARSER_RECOVERY_STATE_DECLARATION_BEGIN", "declaration-begin"}, {VALA_PARSER_RECOVERY_STATE_STATEMENT_BEGIN, "VALA_PARSER_RECOVERY_STATE_STATEMENT_BEGIN", "statement-begin"}, {0, NULL, NULL}};
		vala_parser_recovery_state_type_id = g_enum_register_static ("ValaParserRecoveryState", values);
	}
	return vala_parser_recovery_state_type_id;
}


ValaParser* vala_parser_construct (GType object_type) {
	ValaParser* self;
	ValaParserTokenInfo* _tmp0;
	self = (ValaParser*) g_type_create_instance (object_type);
	_tmp0 = NULL;
	self->priv->tokens = (_tmp0 = g_new0 (ValaParserTokenInfo, VALA_PARSER_BUFFER_SIZE), self->priv->tokens = (g_free (self->priv->tokens), NULL), self->priv->tokens_length1 = VALA_PARSER_BUFFER_SIZE, self->priv->tokens_size = self->priv->tokens_length1, _tmp0);
	return self;
}


ValaParser* vala_parser_new (void) {
	return vala_parser_construct (VALA_TYPE_PARSER);
}


/**
 * Parses all .vala and .vapi source files in the specified code
 * context and builds a code tree.
 *
 * @param context a code context
 */
void vala_parser_parse (ValaParser* self, ValaCodeContext* context) {
	ValaCodeContext* _tmp1;
	ValaCodeContext* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->context = (_tmp1 = (_tmp0 = context, (_tmp0 == NULL) ? NULL : vala_code_context_ref (_tmp0)), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp1);
	vala_code_context_accept (context, (ValaCodeVisitor*) self);
}


static void vala_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file) {
	ValaParser * self;
	gboolean _tmp0;
	self = (ValaParser*) base;
	g_return_if_fail (source_file != NULL);
	_tmp0 = FALSE;
	if (g_str_has_suffix (vala_source_file_get_filename (source_file), ".vala")) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = g_str_has_suffix (vala_source_file_get_filename (source_file), ".vapi");
	}
	if (_tmp0) {
		vala_parser_parse_file (self, source_file);
	}
}


static inline gboolean vala_parser_next (ValaParser* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	self->priv->index = (self->priv->index + 1) % VALA_PARSER_BUFFER_SIZE;
	self->priv->size--;
	if (self->priv->size <= 0) {
		ValaSourceLocation begin = {0};
		ValaSourceLocation end = {0};
		ValaTokenType type;
		type = vala_scanner_read_token (self->priv->scanner, &begin, &end);
		self->priv->tokens[self->priv->index].type = type;
		self->priv->tokens[self->priv->index].begin = begin;
		self->priv->tokens[self->priv->index].end = end;
		self->priv->size = 1;
	}
	return self->priv->tokens[self->priv->index].type != VALA_TOKEN_TYPE_EOF;
}


static inline void vala_parser_prev (ValaParser* self) {
	g_return_if_fail (self != NULL);
	self->priv->index = ((self->priv->index - 1) + VALA_PARSER_BUFFER_SIZE) % VALA_PARSER_BUFFER_SIZE;
	self->priv->size++;
	g_assert (self->priv->size <= VALA_PARSER_BUFFER_SIZE);
}


static inline ValaTokenType vala_parser_current (ValaParser* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->tokens[self->priv->index].type;
}


static inline gboolean vala_parser_accept (ValaParser* self, ValaTokenType type) {
	g_return_val_if_fail (self != NULL, FALSE);
	if (vala_parser_current (self) == type) {
		vala_parser_next (self);
		return TRUE;
	}
	return FALSE;
}


static char* vala_parser_get_error (ValaParser* self, const char* msg) {
	ValaSourceLocation begin;
	char* _tmp1;
	ValaSourceReference* _tmp0;
	const char* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (msg != NULL, NULL);
	begin = vala_parser_get_location (self);
	vala_parser_next (self);
	_tmp1 = NULL;
	_tmp0 = NULL;
	vala_report_error (_tmp0 = vala_parser_get_src (self, &begin), _tmp1 = g_strconcat ("syntax error, ", msg, NULL));
	_tmp1 = (g_free (_tmp1), NULL);
	(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
	_tmp2 = NULL;
	return (_tmp2 = msg, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
}


static inline gboolean vala_parser_expect (ValaParser* self, ValaTokenType type, GError** error) {
	GError * inner_error;
	char* _tmp2;
	char* _tmp1;
	GError* _tmp3;
	g_return_val_if_fail (self != NULL, FALSE);
	inner_error = NULL;
	if (vala_parser_accept (self, type)) {
		return TRUE;
	}
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp3 = NULL;
	inner_error = (_tmp3 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp2 = vala_parser_get_error (self, _tmp1 = g_strdup_printf ("expected %s", vala_token_type_to_string (type)))), _tmp2 = (g_free (_tmp2), NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp3);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return FALSE;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return FALSE;
		}
	}
}


static inline ValaSourceLocation vala_parser_get_location (ValaParser* self) {
	return self->priv->tokens[self->priv->index].begin;
}


static char* vala_parser_get_current_string (ValaParser* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return g_strndup ((const char*) self->priv->tokens[self->priv->index].begin.pos, self->priv->tokens[self->priv->index].end.pos - self->priv->tokens[self->priv->index].begin.pos);
}


static char* vala_parser_get_last_string (ValaParser* self) {
	gint last_index;
	g_return_val_if_fail (self != NULL, NULL);
	last_index = ((self->priv->index + VALA_PARSER_BUFFER_SIZE) - 1) % VALA_PARSER_BUFFER_SIZE;
	return g_strndup ((const char*) self->priv->tokens[last_index].begin.pos, self->priv->tokens[last_index].end.pos - self->priv->tokens[last_index].begin.pos);
}


static ValaSourceReference* vala_parser_get_src (ValaParser* self, const ValaSourceLocation* begin) {
	gint last_index;
	g_return_val_if_fail (self != NULL, NULL);
	last_index = ((self->priv->index + VALA_PARSER_BUFFER_SIZE) - 1) % VALA_PARSER_BUFFER_SIZE;
	return vala_source_reference_new (vala_scanner_get_source_file (self->priv->scanner), (*begin).line, (*begin).column, self->priv->tokens[last_index].end.line, self->priv->tokens[last_index].end.column);
}


static ValaSourceReference* vala_parser_get_src_com (ValaParser* self, const ValaSourceLocation* begin) {
	gint last_index;
	ValaSourceReference* src;
	char* _tmp0;
	g_return_val_if_fail (self != NULL, NULL);
	last_index = ((self->priv->index + VALA_PARSER_BUFFER_SIZE) - 1) % VALA_PARSER_BUFFER_SIZE;
	src = vala_source_reference_new_with_comment (vala_scanner_get_source_file (self->priv->scanner), (*begin).line, (*begin).column, self->priv->tokens[last_index].end.line, self->priv->tokens[last_index].end.column, self->priv->comment);
	_tmp0 = NULL;
	self->priv->comment = (_tmp0 = NULL, self->priv->comment = (g_free (self->priv->comment), NULL), _tmp0);
	return src;
}


static ValaSourceReference* vala_parser_get_current_src (ValaParser* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return vala_source_reference_new (vala_scanner_get_source_file (self->priv->scanner), self->priv->tokens[self->priv->index].begin.line, self->priv->tokens[self->priv->index].begin.column, self->priv->tokens[self->priv->index].end.line, self->priv->tokens[self->priv->index].end.column);
}


static ValaSourceReference* vala_parser_get_last_src (ValaParser* self) {
	gint last_index;
	g_return_val_if_fail (self != NULL, NULL);
	last_index = ((self->priv->index + VALA_PARSER_BUFFER_SIZE) - 1) % VALA_PARSER_BUFFER_SIZE;
	return vala_source_reference_new (vala_scanner_get_source_file (self->priv->scanner), self->priv->tokens[last_index].begin.line, self->priv->tokens[last_index].begin.column, self->priv->tokens[last_index].end.line, self->priv->tokens[last_index].end.column);
}


static void vala_parser_rollback (ValaParser* self, const ValaSourceLocation* location) {
	g_return_if_fail (self != NULL);
	while (self->priv->tokens[self->priv->index].begin.pos != (*location).pos) {
		vala_parser_prev (self);
	}
}


static void vala_parser_skip_identifier (ValaParser* self, GError** error) {
	GError * inner_error;
	char* _tmp1;
	GError* _tmp2;
	g_return_if_fail (self != NULL);
	inner_error = NULL;
	/* also accept keywords as identifiers where there is no conflict*/
	switch (vala_parser_current (self)) {
		case VALA_TOKEN_TYPE_ABSTRACT:
		case VALA_TOKEN_TYPE_AS:
		case VALA_TOKEN_TYPE_BASE:
		case VALA_TOKEN_TYPE_BREAK:
		case VALA_TOKEN_TYPE_CASE:
		case VALA_TOKEN_TYPE_CATCH:
		case VALA_TOKEN_TYPE_CLASS:
		case VALA_TOKEN_TYPE_CONST:
		case VALA_TOKEN_TYPE_CONSTRUCT:
		case VALA_TOKEN_TYPE_CONTINUE:
		case VALA_TOKEN_TYPE_DEFAULT:
		case VALA_TOKEN_TYPE_DELEGATE:
		case VALA_TOKEN_TYPE_DELETE:
		case VALA_TOKEN_TYPE_DO:
		case VALA_TOKEN_TYPE_DYNAMIC:
		case VALA_TOKEN_TYPE_ELSE:
		case VALA_TOKEN_TYPE_ENUM:
		case VALA_TOKEN_TYPE_ENSURES:
		case VALA_TOKEN_TYPE_ERRORDOMAIN:
		case VALA_TOKEN_TYPE_EXTERN:
		case VALA_TOKEN_TYPE_FALSE:
		case VALA_TOKEN_TYPE_FINALLY:
		case VALA_TOKEN_TYPE_FOR:
		case VALA_TOKEN_TYPE_FOREACH:
		case VALA_TOKEN_TYPE_GET:
		case VALA_TOKEN_TYPE_IDENTIFIER:
		case VALA_TOKEN_TYPE_IF:
		case VALA_TOKEN_TYPE_IN:
		case VALA_TOKEN_TYPE_INLINE:
		case VALA_TOKEN_TYPE_INTERFACE:
		case VALA_TOKEN_TYPE_INTERNAL:
		case VALA_TOKEN_TYPE_IS:
		case VALA_TOKEN_TYPE_LOCK:
		case VALA_TOKEN_TYPE_NAMESPACE:
		case VALA_TOKEN_TYPE_NEW:
		case VALA_TOKEN_TYPE_NULL:
		case VALA_TOKEN_TYPE_OUT:
		case VALA_TOKEN_TYPE_OVERRIDE:
		case VALA_TOKEN_TYPE_OWNED:
		case VALA_TOKEN_TYPE_PARAMS:
		case VALA_TOKEN_TYPE_PRIVATE:
		case VALA_TOKEN_TYPE_PROTECTED:
		case VALA_TOKEN_TYPE_PUBLIC:
		case VALA_TOKEN_TYPE_REF:
		case VALA_TOKEN_TYPE_REQUIRES:
		case VALA_TOKEN_TYPE_RETURN:
		case VALA_TOKEN_TYPE_SET:
		case VALA_TOKEN_TYPE_SIGNAL:
		case VALA_TOKEN_TYPE_SIZEOF:
		case VALA_TOKEN_TYPE_STATIC:
		case VALA_TOKEN_TYPE_STRUCT:
		case VALA_TOKEN_TYPE_SWITCH:
		case VALA_TOKEN_TYPE_THIS:
		case VALA_TOKEN_TYPE_THROW:
		case VALA_TOKEN_TYPE_THROWS:
		case VALA_TOKEN_TYPE_TRUE:
		case VALA_TOKEN_TYPE_TRY:
		case VALA_TOKEN_TYPE_TYPEOF:
		case VALA_TOKEN_TYPE_UNOWNED:
		case VALA_TOKEN_TYPE_USING:
		case VALA_TOKEN_TYPE_VAR:
		case VALA_TOKEN_TYPE_VIRTUAL:
		case VALA_TOKEN_TYPE_VOID:
		case VALA_TOKEN_TYPE_VOLATILE:
		case VALA_TOKEN_TYPE_WEAK:
		case VALA_TOKEN_TYPE_WHILE:
		case VALA_TOKEN_TYPE_YIELD:
		case VALA_TOKEN_TYPE_YIELDS:
		{
			vala_parser_next (self);
			return;
		}
		case VALA_TOKEN_TYPE_INTEGER_LITERAL:
		case VALA_TOKEN_TYPE_REAL_LITERAL:
		{
			char* id;
			gboolean _tmp0;
			id = vala_parser_get_current_string (self);
			_tmp0 = FALSE;
			if (g_unichar_isalpha (g_utf8_get_char (g_utf8_offset_to_pointer (id, string_get_length (id) - 1)))) {
				_tmp0 = !string_contains (id, ".");
			} else {
				_tmp0 = FALSE;
			}
			if (_tmp0) {
				vala_parser_next (self);
				id = (g_free (id), NULL);
				return;
			}
			id = (g_free (id), NULL);
			break;
		}
	}
	_tmp1 = NULL;
	_tmp2 = NULL;
	inner_error = (_tmp2 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp1 = vala_parser_get_error (self, "expected identifier")), _tmp1 = (g_free (_tmp1), NULL), _tmp2);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return;
		}
	}
}


static char* vala_parser_parse_identifier (ValaParser* self, GError** error) {
	GError * inner_error;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	vala_parser_skip_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	return vala_parser_get_last_string (self);
}


static ValaExpression* vala_parser_parse_literal (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	switch (vala_parser_current (self)) {
		case VALA_TOKEN_TYPE_TRUE:
		{
			ValaSourceReference* _tmp0;
			ValaExpression* _tmp1;
			vala_parser_next (self);
			_tmp0 = NULL;
			_tmp1 = NULL;
			return (_tmp1 = (ValaExpression*) vala_boolean_literal_new (TRUE, _tmp0 = vala_parser_get_src (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
		}
		case VALA_TOKEN_TYPE_FALSE:
		{
			ValaSourceReference* _tmp3;
			ValaExpression* _tmp4;
			vala_parser_next (self);
			_tmp3 = NULL;
			_tmp4 = NULL;
			return (_tmp4 = (ValaExpression*) vala_boolean_literal_new (FALSE, _tmp3 = vala_parser_get_src (self, &begin)), (_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)), _tmp4);
		}
		case VALA_TOKEN_TYPE_INTEGER_LITERAL:
		{
			ValaSourceReference* _tmp7;
			char* _tmp6;
			ValaExpression* _tmp8;
			vala_parser_next (self);
			_tmp7 = NULL;
			_tmp6 = NULL;
			_tmp8 = NULL;
			return (_tmp8 = (ValaExpression*) vala_integer_literal_new (_tmp6 = vala_parser_get_last_string (self), _tmp7 = vala_parser_get_src (self, &begin)), (_tmp7 == NULL) ? NULL : (_tmp7 = (vala_source_reference_unref (_tmp7), NULL)), _tmp6 = (g_free (_tmp6), NULL), _tmp8);
		}
		case VALA_TOKEN_TYPE_REAL_LITERAL:
		{
			ValaSourceReference* _tmp11;
			char* _tmp10;
			ValaExpression* _tmp12;
			vala_parser_next (self);
			_tmp11 = NULL;
			_tmp10 = NULL;
			_tmp12 = NULL;
			return (_tmp12 = (ValaExpression*) vala_real_literal_new (_tmp10 = vala_parser_get_last_string (self), _tmp11 = vala_parser_get_src (self, &begin)), (_tmp11 == NULL) ? NULL : (_tmp11 = (vala_source_reference_unref (_tmp11), NULL)), _tmp10 = (g_free (_tmp10), NULL), _tmp12);
		}
		case VALA_TOKEN_TYPE_CHARACTER_LITERAL:
		{
			ValaSourceReference* _tmp15;
			char* _tmp14;
			ValaCharacterLiteral* _tmp16;
			ValaCharacterLiteral* lit;
			vala_parser_next (self);
			_tmp15 = NULL;
			_tmp14 = NULL;
			_tmp16 = NULL;
			lit = (_tmp16 = vala_character_literal_new (_tmp14 = vala_parser_get_last_string (self), _tmp15 = vala_parser_get_src (self, &begin)), (_tmp15 == NULL) ? NULL : (_tmp15 = (vala_source_reference_unref (_tmp15), NULL)), _tmp14 = (g_free (_tmp14), NULL), _tmp16);
			if (vala_code_node_get_error ((ValaCodeNode*) lit)) {
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) lit), "invalid character literal");
			}
			return (ValaExpression*) lit;
		}
		case VALA_TOKEN_TYPE_STRING_LITERAL:
		{
			ValaSourceReference* _tmp19;
			char* _tmp18;
			ValaExpression* _tmp20;
			vala_parser_next (self);
			_tmp19 = NULL;
			_tmp18 = NULL;
			_tmp20 = NULL;
			return (_tmp20 = (ValaExpression*) vala_string_literal_new (_tmp18 = vala_parser_get_last_string (self), _tmp19 = vala_parser_get_src (self, &begin)), (_tmp19 == NULL) ? NULL : (_tmp19 = (vala_source_reference_unref (_tmp19), NULL)), _tmp18 = (g_free (_tmp18), NULL), _tmp20);
		}
		case VALA_TOKEN_TYPE_VERBATIM_STRING_LITERAL:
		{
			char* raw_string;
			char* _tmp22;
			char* _tmp23;
			char* escaped_string;
			ValaSourceReference* _tmp25;
			char* _tmp24;
			ValaExpression* _tmp26;
			ValaExpression* _tmp27;
			vala_parser_next (self);
			raw_string = vala_parser_get_last_string (self);
			_tmp22 = NULL;
			_tmp23 = NULL;
			escaped_string = (_tmp23 = g_strescape (_tmp22 = string_substring (raw_string, (glong) 3, g_utf8_strlen (raw_string, -1) - 6), ""), _tmp22 = (g_free (_tmp22), NULL), _tmp23);
			_tmp25 = NULL;
			_tmp24 = NULL;
			_tmp26 = NULL;
			_tmp27 = NULL;
			return (_tmp27 = (_tmp26 = (ValaExpression*) vala_string_literal_new (_tmp24 = g_strdup_printf ("\"%s\"", escaped_string), _tmp25 = vala_parser_get_src (self, &begin)), (_tmp25 == NULL) ? NULL : (_tmp25 = (vala_source_reference_unref (_tmp25), NULL)), _tmp24 = (g_free (_tmp24), NULL), _tmp26), raw_string = (g_free (raw_string), NULL), escaped_string = (g_free (escaped_string), NULL), _tmp27);
		}
		case VALA_TOKEN_TYPE_NULL:
		{
			ValaSourceReference* _tmp28;
			ValaExpression* _tmp29;
			vala_parser_next (self);
			_tmp28 = NULL;
			_tmp29 = NULL;
			return (_tmp29 = (ValaExpression*) vala_null_literal_new (_tmp28 = vala_parser_get_src (self, &begin)), (_tmp28 == NULL) ? NULL : (_tmp28 = (vala_source_reference_unref (_tmp28), NULL)), _tmp29);
		}
		default:
		{
			char* _tmp31;
			GError* _tmp32;
			_tmp31 = NULL;
			_tmp32 = NULL;
			inner_error = (_tmp32 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp31 = vala_parser_get_error (self, "expected literal")), _tmp31 = (g_free (_tmp31), NULL), _tmp32);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
		}
	}
}


void vala_parser_parse_file (ValaParser* self, ValaSourceFile* source_file) {
	GError * inner_error;
	ValaScanner* _tmp0;
	ValaScanner* _tmp1;
	g_return_if_fail (self != NULL);
	g_return_if_fail (source_file != NULL);
	inner_error = NULL;
	_tmp0 = NULL;
	self->priv->scanner = (_tmp0 = vala_scanner_new (source_file), (self->priv->scanner == NULL) ? NULL : (self->priv->scanner = (vala_scanner_unref (self->priv->scanner), NULL)), _tmp0);
	self->priv->index = -1;
	self->priv->size = 0;
	vala_parser_next (self);
	{
		vala_parser_parse_using_directives (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				goto __catch3_vala_parse_error;
			}
			goto __finally3;
		}
		vala_parser_parse_declarations (self, (ValaSymbol*) vala_code_context_get_root (self->priv->context), TRUE, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				goto __catch3_vala_parse_error;
			}
			goto __finally3;
		}
	}
	goto __finally3;
	__catch3_vala_parse_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally3:
	if (inner_error != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	/* already reported*/
	_tmp1 = NULL;
	self->priv->scanner = (_tmp1 = NULL, (self->priv->scanner == NULL) ? NULL : (self->priv->scanner = (vala_scanner_unref (self->priv->scanner), NULL)), _tmp1);
}


static void vala_parser_skip_symbol_name (ValaParser* self, GError** error) {
	GError * inner_error;
	gboolean _tmp0;
	g_return_if_fail (self != NULL);
	inner_error = NULL;
	_tmp0 = FALSE;
	do {
		if (_tmp0) {
			gboolean _tmp1;
			_tmp1 = FALSE;
			if (vala_parser_accept (self, VALA_TOKEN_TYPE_DOT)) {
				_tmp1 = TRUE;
			} else {
				_tmp1 = vala_parser_accept (self, VALA_TOKEN_TYPE_DOUBLE_COLON);
			}
			if (!_tmp1) {
				break;
			}
		}
		_tmp0 = TRUE;
		vala_parser_skip_identifier (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				return;
			} else {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return;
			}
		}
	} while (TRUE);
}


static ValaUnresolvedSymbol* vala_parser_parse_symbol_name (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaUnresolvedSymbol* sym;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	sym = NULL;
	do {
		char* name;
		gboolean _tmp0;
		ValaUnresolvedSymbol* _tmp6;
		ValaSourceReference* _tmp5;
		name = vala_parser_parse_identifier (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				return NULL;
			} else {
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp0 = FALSE;
		if (_vala_strcmp0 (name, "global") == 0) {
			_tmp0 = vala_parser_accept (self, VALA_TOKEN_TYPE_DOUBLE_COLON);
		} else {
			_tmp0 = FALSE;
		}
		if (_tmp0) {
			char* _tmp1;
			char* _tmp2;
			ValaUnresolvedSymbol* _tmp4;
			ValaSourceReference* _tmp3;
			_tmp1 = vala_parser_parse_identifier (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					name = (g_free (name), NULL);
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					return NULL;
				} else {
					name = (g_free (name), NULL);
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			/* global::Name
			 qualified access to global symbol*/
			_tmp2 = NULL;
			name = (_tmp2 = _tmp1, name = (g_free (name), NULL), _tmp2);
			_tmp4 = NULL;
			_tmp3 = NULL;
			sym = (_tmp4 = vala_unresolved_symbol_new (sym, name, _tmp3 = vala_parser_get_src (self, &begin)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp4);
			(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL));
			vala_unresolved_symbol_set_qualified (sym, TRUE);
			name = (g_free (name), NULL);
			continue;
		}
		_tmp6 = NULL;
		_tmp5 = NULL;
		sym = (_tmp6 = vala_unresolved_symbol_new (sym, name, _tmp5 = vala_parser_get_src (self, &begin)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp6);
		(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL));
		name = (g_free (name), NULL);
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_DOT));
	return sym;
}


static void vala_parser_skip_type (ValaParser* self, GError** error) {
	GError * inner_error;
	g_return_if_fail (self != NULL);
	inner_error = NULL;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_VOID)) {
		while (vala_parser_accept (self, VALA_TOKEN_TYPE_STAR)) {
		}
		return;
	}
	vala_parser_accept (self, VALA_TOKEN_TYPE_DYNAMIC);
	vala_parser_accept (self, VALA_TOKEN_TYPE_OWNED);
	vala_parser_accept (self, VALA_TOKEN_TYPE_UNOWNED);
	vala_parser_accept (self, VALA_TOKEN_TYPE_WEAK);
	vala_parser_skip_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return;
		}
	}
	vala_parser_skip_type_argument_list (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return;
		}
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_STAR)) {
	}
	vala_parser_accept (self, VALA_TOKEN_TYPE_INTERR);
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_BRACKET)) {
		do {
			gboolean _tmp0;
			_tmp0 = FALSE;
			if (vala_parser_current (self) != VALA_TOKEN_TYPE_COMMA) {
				_tmp0 = vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACKET;
			} else {
				_tmp0 = FALSE;
			}
			if (_tmp0) {
				ValaExpression* _tmp1;
				_tmp1 = NULL;
				_tmp1 = vala_parser_parse_expression (self, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						return;
					} else {
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return;
					}
				}
				(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
			}
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACKET, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				return;
			} else {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return;
			}
		}
		vala_parser_accept (self, VALA_TOKEN_TYPE_INTERR);
	}
	vala_parser_accept (self, VALA_TOKEN_TYPE_OP_NEG);
	vala_parser_accept (self, VALA_TOKEN_TYPE_HASH);
}


static ValaDataType* vala_parser_parse_type (ValaParser* self, gboolean owned_by_default, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	gboolean is_dynamic;
	gboolean value_owned;
	ValaUnresolvedSymbol* sym;
	GeeList* type_arg_list;
	ValaSourceReference* _tmp5;
	ValaDataType* _tmp6;
	ValaDataType* type;
	ValaDataType* _tmp14;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_VOID)) {
		ValaSourceReference* _tmp0;
		ValaDataType* _tmp1;
		ValaDataType* type;
		_tmp0 = NULL;
		_tmp1 = NULL;
		type = (_tmp1 = (ValaDataType*) vala_void_type_new (_tmp0 = vala_parser_get_src (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
		while (vala_parser_accept (self, VALA_TOKEN_TYPE_STAR)) {
			ValaDataType* _tmp2;
			_tmp2 = NULL;
			type = (_tmp2 = (ValaDataType*) vala_pointer_type_new (type, NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp2);
		}
		return type;
	}
	is_dynamic = vala_parser_accept (self, VALA_TOKEN_TYPE_DYNAMIC);
	value_owned = owned_by_default;
	if (owned_by_default) {
		gboolean _tmp4;
		_tmp4 = FALSE;
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_UNOWNED)) {
			_tmp4 = TRUE;
		} else {
			_tmp4 = vala_parser_accept (self, VALA_TOKEN_TYPE_WEAK);
		}
		if (_tmp4) {
			value_owned = FALSE;
		}
	} else {
		value_owned = vala_parser_accept (self, VALA_TOKEN_TYPE_OWNED);
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	type_arg_list = vala_parser_parse_type_argument_list (self, FALSE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp5 = NULL;
	_tmp6 = NULL;
	type = (_tmp6 = (ValaDataType*) vala_unresolved_type_new_from_symbol (sym, _tmp5 = vala_parser_get_src (self, &begin)), (_tmp5 == NULL) ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL)), _tmp6);
	if (type_arg_list != NULL) {
		{
			GeeIterator* _type_arg_it;
			_type_arg_it = gee_iterable_iterator ((GeeIterable*) type_arg_list);
			while (gee_iterator_next (_type_arg_it)) {
				ValaDataType* type_arg;
				type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
				vala_data_type_add_type_argument (type, type_arg);
				(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
			}
			(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
		}
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_STAR)) {
		ValaDataType* _tmp8;
		ValaSourceReference* _tmp7;
		_tmp8 = NULL;
		_tmp7 = NULL;
		type = (_tmp8 = (ValaDataType*) vala_pointer_type_new (type, _tmp7 = vala_parser_get_src (self, &begin)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp8);
		(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_source_reference_unref (_tmp7), NULL));
	}
	if (!VALA_IS_POINTER_TYPE (type)) {
		vala_data_type_set_nullable (type, vala_parser_accept (self, VALA_TOKEN_TYPE_INTERR));
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_BRACKET)) {
		gint array_rank;
		ValaDataType* _tmp12;
		ValaSourceReference* _tmp11;
		array_rank = 0;
		do {
			gboolean _tmp9;
			array_rank++;
			_tmp9 = FALSE;
			if (vala_parser_current (self) != VALA_TOKEN_TYPE_COMMA) {
				_tmp9 = vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACKET;
			} else {
				_tmp9 = FALSE;
			}
			/* support for stack-allocated arrays
			 also required for decision between expression and declaration statement*/
			if (_tmp9) {
				ValaExpression* _tmp10;
				_tmp10 = NULL;
				_tmp10 = vala_parser_parse_expression (self, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
						(type_arg_list == NULL) ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL));
						(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
						return NULL;
					} else {
						(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
						(type_arg_list == NULL) ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL));
						(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				(_tmp10 == NULL) ? NULL : (_tmp10 = (vala_code_node_unref (_tmp10), NULL));
			}
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACKET, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(type_arg_list == NULL) ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL));
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				return NULL;
			} else {
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(type_arg_list == NULL) ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL));
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		/* arrays contain strong references by default*/
		vala_data_type_set_value_owned (type, TRUE);
		_tmp12 = NULL;
		_tmp11 = NULL;
		type = (_tmp12 = (ValaDataType*) vala_array_type_new (type, array_rank, _tmp11 = vala_parser_get_src (self, &begin)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp12);
		(_tmp11 == NULL) ? NULL : (_tmp11 = (vala_source_reference_unref (_tmp11), NULL));
		vala_data_type_set_nullable (type, vala_parser_accept (self, VALA_TOKEN_TYPE_INTERR));
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OP_NEG)) {
		ValaSourceReference* _tmp13;
		_tmp13 = NULL;
		vala_report_warning (_tmp13 = vala_parser_get_last_src (self), "obsolete syntax, types are non-null by default");
		(_tmp13 == NULL) ? NULL : (_tmp13 = (vala_source_reference_unref (_tmp13), NULL));
	}
	if (!owned_by_default) {
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_HASH)) {
			/* TODO enable warning after releasing Vala 0.5.5
			 Report.warning (get_last_src (), "deprecated syntax, use `owned` modifier");*/
			value_owned = TRUE;
		}
	}
	vala_data_type_set_is_dynamic (type, is_dynamic);
	vala_data_type_set_value_owned (type, value_owned);
	_tmp14 = NULL;
	return (_tmp14 = type, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (type_arg_list == NULL) ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL)), _tmp14);
}


static GeeList* vala_parser_parse_argument_list (ValaParser* self, GError** error) {
	GError * inner_error;
	GeeArrayList* list;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	list = gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaExpression* _tmp0;
			ValaExpression* _tmp1;
			_tmp0 = vala_parser_parse_argument (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
					return NULL;
				} else {
					(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp1 = NULL;
			gee_collection_add ((GeeCollection*) list, _tmp1 = _tmp0);
			(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	return (GeeList*) list;
}


static ValaExpression* vala_parser_parse_argument (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_REF)) {
		ValaExpression* inner;
		ValaSourceReference* _tmp0;
		ValaExpression* _tmp1;
		ValaExpression* _tmp2;
		inner = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				return NULL;
			} else {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp0 = NULL;
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = (_tmp1 = (ValaExpression*) vala_unary_expression_new (VALA_UNARY_OPERATOR_REF, inner, _tmp0 = vala_parser_get_src (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1), (inner == NULL) ? NULL : (inner = (vala_code_node_unref (inner), NULL)), _tmp2);
	} else {
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_OUT)) {
			ValaExpression* inner;
			ValaSourceReference* _tmp3;
			ValaExpression* _tmp4;
			ValaExpression* _tmp5;
			inner = vala_parser_parse_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp3 = NULL;
			_tmp4 = NULL;
			_tmp5 = NULL;
			return (_tmp5 = (_tmp4 = (ValaExpression*) vala_unary_expression_new (VALA_UNARY_OPERATOR_OUT, inner, _tmp3 = vala_parser_get_src (self, &begin)), (_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)), _tmp4), (inner == NULL) ? NULL : (inner = (vala_code_node_unref (inner), NULL)), _tmp5);
		} else {
			ValaExpression* _tmp6;
			_tmp6 = vala_parser_parse_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp6;
		}
	}
}


static ValaExpression* vala_parser_parse_primary_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	gboolean found;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	expr = NULL;
	switch (vala_parser_current (self)) {
		case VALA_TOKEN_TYPE_TRUE:
		case VALA_TOKEN_TYPE_FALSE:
		case VALA_TOKEN_TYPE_INTEGER_LITERAL:
		case VALA_TOKEN_TYPE_REAL_LITERAL:
		case VALA_TOKEN_TYPE_CHARACTER_LITERAL:
		case VALA_TOKEN_TYPE_STRING_LITERAL:
		case VALA_TOKEN_TYPE_VERBATIM_STRING_LITERAL:
		case VALA_TOKEN_TYPE_NULL:
		{
			ValaExpression* _tmp0;
			ValaExpression* _tmp1;
			_tmp0 = vala_parser_parse_literal (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					return NULL;
				} else {
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp1 = NULL;
			expr = (_tmp1 = _tmp0, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp1);
			break;
		}
		case VALA_TOKEN_TYPE_OPEN_BRACE:
		{
			ValaInitializerList* _tmp2;
			ValaExpression* _tmp3;
			_tmp2 = vala_parser_parse_initializer (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					return NULL;
				} else {
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp3 = NULL;
			expr = (_tmp3 = (ValaExpression*) _tmp2, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp3);
			break;
		}
		case VALA_TOKEN_TYPE_OPEN_PARENS:
		{
			ValaExpression* _tmp4;
			ValaExpression* _tmp5;
			_tmp4 = vala_parser_parse_tuple (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					return NULL;
				} else {
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp5 = NULL;
			expr = (_tmp5 = _tmp4, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp5);
			break;
		}
		case VALA_TOKEN_TYPE_THIS:
		{
			ValaExpression* _tmp6;
			ValaExpression* _tmp7;
			_tmp6 = vala_parser_parse_this_access (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					return NULL;
				} else {
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp7 = NULL;
			expr = (_tmp7 = _tmp6, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp7);
			break;
		}
		case VALA_TOKEN_TYPE_BASE:
		{
			ValaExpression* _tmp8;
			ValaExpression* _tmp9;
			_tmp8 = vala_parser_parse_base_access (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					return NULL;
				} else {
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp9 = NULL;
			expr = (_tmp9 = _tmp8, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp9);
			break;
		}
		case VALA_TOKEN_TYPE_NEW:
		{
			ValaExpression* _tmp10;
			ValaExpression* _tmp11;
			_tmp10 = vala_parser_parse_object_or_array_creation_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					return NULL;
				} else {
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp11 = NULL;
			expr = (_tmp11 = _tmp10, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp11);
			break;
		}
		case VALA_TOKEN_TYPE_SIZEOF:
		{
			ValaExpression* _tmp12;
			ValaExpression* _tmp13;
			_tmp12 = vala_parser_parse_sizeof_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					return NULL;
				} else {
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp13 = NULL;
			expr = (_tmp13 = _tmp12, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp13);
			break;
		}
		case VALA_TOKEN_TYPE_TYPEOF:
		{
			ValaExpression* _tmp14;
			ValaExpression* _tmp15;
			_tmp14 = vala_parser_parse_typeof_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					return NULL;
				} else {
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp15 = NULL;
			expr = (_tmp15 = _tmp14, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp15);
			break;
		}
		default:
		{
			ValaExpression* _tmp16;
			ValaExpression* _tmp17;
			_tmp16 = vala_parser_parse_simple_name (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					return NULL;
				} else {
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp17 = NULL;
			expr = (_tmp17 = _tmp16, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp17);
			break;
		}
	}
	/* process primary expressions that start with an inner primary expression*/
	found = TRUE;
	while (found) {
		switch (vala_parser_current (self)) {
			case VALA_TOKEN_TYPE_DOT:
			{
				ValaExpression* _tmp18;
				ValaExpression* _tmp19;
				_tmp18 = vala_parser_parse_member_access (self, &begin, expr, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
						return NULL;
					} else {
						(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp19 = NULL;
				expr = (_tmp19 = _tmp18, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp19);
				break;
			}
			case VALA_TOKEN_TYPE_OP_PTR:
			{
				ValaExpression* _tmp20;
				ValaExpression* _tmp21;
				_tmp20 = vala_parser_parse_pointer_member_access (self, &begin, expr, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
						return NULL;
					} else {
						(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp21 = NULL;
				expr = (_tmp21 = _tmp20, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp21);
				break;
			}
			case VALA_TOKEN_TYPE_OPEN_PARENS:
			{
				ValaExpression* _tmp22;
				ValaExpression* _tmp23;
				_tmp22 = vala_parser_parse_method_call (self, &begin, expr, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
						return NULL;
					} else {
						(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp23 = NULL;
				expr = (_tmp23 = _tmp22, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp23);
				break;
			}
			case VALA_TOKEN_TYPE_OPEN_BRACKET:
			{
				ValaExpression* _tmp24;
				ValaExpression* _tmp25;
				_tmp24 = vala_parser_parse_element_access (self, &begin, expr, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
						return NULL;
					} else {
						(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp25 = NULL;
				expr = (_tmp25 = _tmp24, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp25);
				break;
			}
			case VALA_TOKEN_TYPE_OP_INC:
			{
				ValaExpression* _tmp26;
				ValaExpression* _tmp27;
				_tmp26 = vala_parser_parse_post_increment_expression (self, &begin, expr, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
						return NULL;
					} else {
						(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp27 = NULL;
				expr = (_tmp27 = _tmp26, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp27);
				break;
			}
			case VALA_TOKEN_TYPE_OP_DEC:
			{
				ValaExpression* _tmp28;
				ValaExpression* _tmp29;
				_tmp28 = vala_parser_parse_post_decrement_expression (self, &begin, expr, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
						return NULL;
					} else {
						(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp29 = NULL;
				expr = (_tmp29 = _tmp28, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp29);
				break;
			}
			default:
			{
				found = FALSE;
				break;
			}
		}
	}
	return expr;
}


static ValaExpression* vala_parser_parse_simple_name (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	char* id;
	gboolean qualified;
	gboolean _tmp0;
	GeeList* type_arg_list;
	ValaSourceReference* _tmp3;
	ValaMemberAccess* _tmp4;
	ValaMemberAccess* expr;
	ValaExpression* _tmp5;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	qualified = FALSE;
	_tmp0 = FALSE;
	if (_vala_strcmp0 (id, "global") == 0) {
		_tmp0 = vala_parser_accept (self, VALA_TOKEN_TYPE_DOUBLE_COLON);
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		char* _tmp1;
		char* _tmp2;
		_tmp1 = vala_parser_parse_identifier (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				id = (g_free (id), NULL);
				return NULL;
			} else {
				id = (g_free (id), NULL);
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp2 = NULL;
		id = (_tmp2 = _tmp1, id = (g_free (id), NULL), _tmp2);
		qualified = TRUE;
	}
	type_arg_list = vala_parser_parse_type_argument_list (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			id = (g_free (id), NULL);
			return NULL;
		} else {
			id = (g_free (id), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp3 = NULL;
	_tmp4 = NULL;
	expr = (_tmp4 = vala_member_access_new (NULL, id, _tmp3 = vala_parser_get_src (self, &begin)), (_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)), _tmp4);
	vala_member_access_set_qualified (expr, qualified);
	if (type_arg_list != NULL) {
		{
			GeeIterator* _type_arg_it;
			_type_arg_it = gee_iterable_iterator ((GeeIterable*) type_arg_list);
			while (gee_iterator_next (_type_arg_it)) {
				ValaDataType* type_arg;
				type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
				vala_member_access_add_type_argument (expr, type_arg);
				(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
			}
			(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
		}
	}
	_tmp5 = NULL;
	return (_tmp5 = (ValaExpression*) expr, id = (g_free (id), NULL), (type_arg_list == NULL) ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL)), _tmp5);
}


static ValaExpression* vala_parser_parse_tuple (ValaParser* self, GError** error) {
	GError * inner_error;
	GeeArrayList* expr_list;
	ValaExpression* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	expr_list = gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaExpression* _tmp0;
			ValaExpression* _tmp1;
			_tmp0 = vala_parser_parse_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(expr_list == NULL) ? NULL : (expr_list = (gee_collection_object_unref (expr_list), NULL));
					return NULL;
				} else {
					(expr_list == NULL) ? NULL : (expr_list = (gee_collection_object_unref (expr_list), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp1 = NULL;
			gee_collection_add ((GeeCollection*) expr_list, _tmp1 = _tmp0);
			(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(expr_list == NULL) ? NULL : (expr_list = (gee_collection_object_unref (expr_list), NULL));
			return NULL;
		} else {
			(expr_list == NULL) ? NULL : (expr_list = (gee_collection_object_unref (expr_list), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if (gee_collection_get_size ((GeeCollection*) expr_list) != 1) {
		ValaTuple* tuple;
		ValaExpression* _tmp2;
		tuple = vala_tuple_new ();
		{
			GeeIterator* _expr_it;
			_expr_it = gee_iterable_iterator ((GeeIterable*) expr_list);
			while (gee_iterator_next (_expr_it)) {
				ValaExpression* expr;
				expr = (ValaExpression*) gee_iterator_get (_expr_it);
				vala_tuple_add_expression (tuple, expr);
				(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			}
			(_expr_it == NULL) ? NULL : (_expr_it = (gee_collection_object_unref (_expr_it), NULL));
		}
		_tmp2 = NULL;
		return (_tmp2 = (ValaExpression*) tuple, (expr_list == NULL) ? NULL : (expr_list = (gee_collection_object_unref (expr_list), NULL)), _tmp2);
	}
	_tmp3 = NULL;
	return (_tmp3 = (ValaExpression*) gee_list_get ((GeeList*) expr_list, 0), (expr_list == NULL) ? NULL : (expr_list = (gee_collection_object_unref (expr_list), NULL)), _tmp3);
}


static ValaExpression* vala_parser_parse_member_access (ValaParser* self, const ValaSourceLocation* begin, ValaExpression* inner, GError** error) {
	GError * inner_error;
	char* id;
	GeeList* type_arg_list;
	ValaSourceReference* _tmp0;
	ValaMemberAccess* _tmp1;
	ValaMemberAccess* expr;
	ValaExpression* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (inner != NULL, NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_DOT, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	type_arg_list = vala_parser_parse_type_argument_list (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			id = (g_free (id), NULL);
			return NULL;
		} else {
			id = (g_free (id), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	expr = (_tmp1 = vala_member_access_new (inner, id, _tmp0 = vala_parser_get_src (self, &(*begin))), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	if (type_arg_list != NULL) {
		{
			GeeIterator* _type_arg_it;
			_type_arg_it = gee_iterable_iterator ((GeeIterable*) type_arg_list);
			while (gee_iterator_next (_type_arg_it)) {
				ValaDataType* type_arg;
				type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
				vala_member_access_add_type_argument (expr, type_arg);
				(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
			}
			(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = (ValaExpression*) expr, id = (g_free (id), NULL), (type_arg_list == NULL) ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL)), _tmp2);
}


static ValaExpression* vala_parser_parse_pointer_member_access (ValaParser* self, const ValaSourceLocation* begin, ValaExpression* inner, GError** error) {
	GError * inner_error;
	char* id;
	GeeList* type_arg_list;
	ValaSourceReference* _tmp0;
	ValaMemberAccess* _tmp1;
	ValaMemberAccess* expr;
	ValaExpression* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (inner != NULL, NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_OP_PTR, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	type_arg_list = vala_parser_parse_type_argument_list (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			id = (g_free (id), NULL);
			return NULL;
		} else {
			id = (g_free (id), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	expr = (_tmp1 = vala_member_access_new_pointer (inner, id, _tmp0 = vala_parser_get_src (self, &(*begin))), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	if (type_arg_list != NULL) {
		{
			GeeIterator* _type_arg_it;
			_type_arg_it = gee_iterable_iterator ((GeeIterable*) type_arg_list);
			while (gee_iterator_next (_type_arg_it)) {
				ValaDataType* type_arg;
				type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
				vala_member_access_add_type_argument (expr, type_arg);
				(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
			}
			(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = (ValaExpression*) expr, id = (g_free (id), NULL), (type_arg_list == NULL) ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL)), _tmp2);
}


static ValaExpression* vala_parser_parse_method_call (ValaParser* self, const ValaSourceLocation* begin, ValaExpression* inner, GError** error) {
	GError * inner_error;
	GeeList* arg_list;
	GeeList* init_list;
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (inner != NULL, NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	arg_list = vala_parser_parse_argument_list (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(arg_list == NULL) ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL));
			return NULL;
		} else {
			(arg_list == NULL) ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	init_list = vala_parser_parse_object_initializer (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(arg_list == NULL) ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL));
			return NULL;
		} else {
			(arg_list == NULL) ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = FALSE;
	if (gee_collection_get_size ((GeeCollection*) init_list) > 0) {
		_tmp0 = VALA_IS_MEMBER_ACCESS (inner);
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		ValaMemberAccess* _tmp1;
		ValaMemberAccess* member;
		ValaSourceReference* _tmp2;
		ValaObjectCreationExpression* _tmp3;
		ValaObjectCreationExpression* expr;
		ValaExpression* _tmp4;
		/* struct creation expression*/
		_tmp1 = NULL;
		member = (_tmp1 = VALA_MEMBER_ACCESS (inner), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
		vala_member_access_set_creation_member (member, TRUE);
		_tmp2 = NULL;
		_tmp3 = NULL;
		expr = (_tmp3 = vala_object_creation_expression_new (member, _tmp2 = vala_parser_get_src (self, &(*begin))), (_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)), _tmp3);
		vala_object_creation_expression_set_struct_creation (expr, TRUE);
		{
			GeeIterator* _arg_it;
			_arg_it = gee_iterable_iterator ((GeeIterable*) arg_list);
			while (gee_iterator_next (_arg_it)) {
				ValaExpression* arg;
				arg = (ValaExpression*) gee_iterator_get (_arg_it);
				vala_object_creation_expression_add_argument (expr, arg);
				(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
			}
			(_arg_it == NULL) ? NULL : (_arg_it = (gee_collection_object_unref (_arg_it), NULL));
		}
		{
			GeeIterator* _initializer_it;
			_initializer_it = gee_iterable_iterator ((GeeIterable*) init_list);
			while (gee_iterator_next (_initializer_it)) {
				ValaMemberInitializer* initializer;
				initializer = (ValaMemberInitializer*) gee_iterator_get (_initializer_it);
				vala_object_creation_expression_add_member_initializer (expr, initializer);
				(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
			}
			(_initializer_it == NULL) ? NULL : (_initializer_it = (gee_collection_object_unref (_initializer_it), NULL));
		}
		_tmp4 = NULL;
		return (_tmp4 = (ValaExpression*) expr, (member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL)), (arg_list == NULL) ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL)), (init_list == NULL) ? NULL : (init_list = (gee_collection_object_unref (init_list), NULL)), _tmp4);
	} else {
		ValaSourceReference* _tmp5;
		ValaMethodCall* _tmp6;
		ValaMethodCall* expr;
		ValaExpression* _tmp7;
		_tmp5 = NULL;
		_tmp6 = NULL;
		expr = (_tmp6 = vala_method_call_new (inner, _tmp5 = vala_parser_get_src (self, &(*begin))), (_tmp5 == NULL) ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL)), _tmp6);
		{
			GeeIterator* _arg_it;
			_arg_it = gee_iterable_iterator ((GeeIterable*) arg_list);
			while (gee_iterator_next (_arg_it)) {
				ValaExpression* arg;
				arg = (ValaExpression*) gee_iterator_get (_arg_it);
				vala_method_call_add_argument (expr, arg);
				(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
			}
			(_arg_it == NULL) ? NULL : (_arg_it = (gee_collection_object_unref (_arg_it), NULL));
		}
		_tmp7 = NULL;
		return (_tmp7 = (ValaExpression*) expr, (arg_list == NULL) ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL)), (init_list == NULL) ? NULL : (init_list = (gee_collection_object_unref (init_list), NULL)), _tmp7);
	}
	(arg_list == NULL) ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL));
	(init_list == NULL) ? NULL : (init_list = (gee_collection_object_unref (init_list), NULL));
}


static ValaExpression* vala_parser_parse_element_access (ValaParser* self, const ValaSourceLocation* begin, ValaExpression* inner, GError** error) {
	GError * inner_error;
	GeeList* index_list;
	ValaSourceReference* _tmp0;
	ValaElementAccess* _tmp1;
	ValaElementAccess* expr;
	ValaExpression* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (inner != NULL, NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACKET, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	index_list = vala_parser_parse_expression_list (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACKET, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(index_list == NULL) ? NULL : (index_list = (gee_collection_object_unref (index_list), NULL));
			return NULL;
		} else {
			(index_list == NULL) ? NULL : (index_list = (gee_collection_object_unref (index_list), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	expr = (_tmp1 = vala_element_access_new (inner, _tmp0 = vala_parser_get_src (self, &(*begin))), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	{
		GeeIterator* _index_it;
		_index_it = gee_iterable_iterator ((GeeIterable*) index_list);
		while (gee_iterator_next (_index_it)) {
			ValaExpression* index;
			index = (ValaExpression*) gee_iterator_get (_index_it);
			vala_element_access_append_index (expr, index);
			(index == NULL) ? NULL : (index = (vala_code_node_unref (index), NULL));
		}
		(_index_it == NULL) ? NULL : (_index_it = (gee_collection_object_unref (_index_it), NULL));
	}
	_tmp2 = NULL;
	return (_tmp2 = (ValaExpression*) expr, (index_list == NULL) ? NULL : (index_list = (gee_collection_object_unref (index_list), NULL)), _tmp2);
}


static GeeList* vala_parser_parse_expression_list (ValaParser* self, GError** error) {
	GError * inner_error;
	GeeArrayList* list;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	list = gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	do {
		ValaExpression* _tmp0;
		ValaExpression* _tmp1;
		_tmp0 = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
				return NULL;
			} else {
				(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		gee_collection_add ((GeeCollection*) list, _tmp1 = _tmp0);
		(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	return (GeeList*) list;
}


static ValaExpression* vala_parser_parse_this_access (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSourceReference* _tmp0;
	ValaExpression* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_THIS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = (ValaExpression*) vala_member_access_new (NULL, "this", _tmp0 = vala_parser_get_src (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
}


static ValaExpression* vala_parser_parse_base_access (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSourceReference* _tmp0;
	ValaExpression* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_BASE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = (ValaExpression*) vala_base_access_new (_tmp0 = vala_parser_get_src (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
}


static ValaExpression* vala_parser_parse_post_increment_expression (ValaParser* self, const ValaSourceLocation* begin, ValaExpression* inner, GError** error) {
	GError * inner_error;
	ValaSourceReference* _tmp0;
	ValaExpression* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (inner != NULL, NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_OP_INC, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = (ValaExpression*) vala_postfix_expression_new (inner, TRUE, _tmp0 = vala_parser_get_src (self, &(*begin))), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
}


static ValaExpression* vala_parser_parse_post_decrement_expression (ValaParser* self, const ValaSourceLocation* begin, ValaExpression* inner, GError** error) {
	GError * inner_error;
	ValaSourceReference* _tmp0;
	ValaExpression* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (inner != NULL, NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_OP_DEC, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = (ValaExpression*) vala_postfix_expression_new (inner, FALSE, _tmp0 = vala_parser_get_src (self, &(*begin))), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
}


static ValaExpression* vala_parser_parse_object_or_array_creation_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaMemberAccess* member;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_NEW, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	member = vala_parser_parse_member_name (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_PARENS)) {
		ValaExpression* expr;
		ValaExpression* _tmp0;
		expr = vala_parser_parse_object_creation_expression (self, &begin, member, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL));
				return NULL;
			} else {
				(member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp0 = NULL;
		return (_tmp0 = expr, (member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL)), _tmp0);
	} else {
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_BRACKET)) {
			ValaExpression* expr;
			ValaExpression* _tmp1;
			expr = vala_parser_parse_array_creation_expression (self, &begin, member, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL));
					return NULL;
				} else {
					(member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp1 = NULL;
			return (_tmp1 = expr, (member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL)), _tmp1);
		} else {
			char* _tmp2;
			GError* _tmp3;
			_tmp2 = NULL;
			_tmp3 = NULL;
			inner_error = (_tmp3 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp2 = vala_parser_get_error (self, "expected ( or [")), _tmp2 = (g_free (_tmp2), NULL), _tmp3);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL));
					return NULL;
				} else {
					(member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
		}
	}
	(member == NULL) ? NULL : (member = (vala_code_node_unref (member), NULL));
}


static ValaExpression* vala_parser_parse_object_creation_expression (ValaParser* self, const ValaSourceLocation* begin, ValaMemberAccess* member, GError** error) {
	GError * inner_error;
	GeeList* arg_list;
	GeeList* init_list;
	ValaSourceReference* _tmp0;
	ValaObjectCreationExpression* _tmp1;
	ValaObjectCreationExpression* expr;
	ValaExpression* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (member != NULL, NULL);
	inner_error = NULL;
	vala_member_access_set_creation_member (member, TRUE);
	arg_list = vala_parser_parse_argument_list (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(arg_list == NULL) ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL));
			return NULL;
		} else {
			(arg_list == NULL) ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	init_list = vala_parser_parse_object_initializer (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(arg_list == NULL) ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL));
			return NULL;
		} else {
			(arg_list == NULL) ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	expr = (_tmp1 = vala_object_creation_expression_new (member, _tmp0 = vala_parser_get_src (self, &(*begin))), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	{
		GeeIterator* _arg_it;
		_arg_it = gee_iterable_iterator ((GeeIterable*) arg_list);
		while (gee_iterator_next (_arg_it)) {
			ValaExpression* arg;
			arg = (ValaExpression*) gee_iterator_get (_arg_it);
			vala_object_creation_expression_add_argument (expr, arg);
			(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
		}
		(_arg_it == NULL) ? NULL : (_arg_it = (gee_collection_object_unref (_arg_it), NULL));
	}
	{
		GeeIterator* _initializer_it;
		_initializer_it = gee_iterable_iterator ((GeeIterable*) init_list);
		while (gee_iterator_next (_initializer_it)) {
			ValaMemberInitializer* initializer;
			initializer = (ValaMemberInitializer*) gee_iterator_get (_initializer_it);
			vala_object_creation_expression_add_member_initializer (expr, initializer);
			(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
		}
		(_initializer_it == NULL) ? NULL : (_initializer_it = (gee_collection_object_unref (_initializer_it), NULL));
	}
	_tmp2 = NULL;
	return (_tmp2 = (ValaExpression*) expr, (arg_list == NULL) ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL)), (init_list == NULL) ? NULL : (init_list = (gee_collection_object_unref (init_list), NULL)), _tmp2);
}


static ValaExpression* vala_parser_parse_array_creation_expression (ValaParser* self, const ValaSourceLocation* begin, ValaMemberAccess* member, GError** error) {
	GError * inner_error;
	gboolean size_specified;
	GeeList* size_specifier_list;
	gboolean first;
	ValaDataType* element_type;
	ValaInitializerList* initializer;
	ValaSourceReference* _tmp7;
	ValaArrayCreationExpression* _tmp8;
	ValaArrayCreationExpression* expr;
	ValaExpression* _tmp9;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (member != NULL, NULL);
	inner_error = NULL;
	size_specified = FALSE;
	size_specifier_list = NULL;
	first = TRUE;
	element_type = (ValaDataType*) vala_unresolved_type_new_from_expression ((ValaExpression*) member);
	do {
		GeeList* _tmp1;
		if (!first) {
			ValaDataType* _tmp0;
			/* array of arrays: new T[][42]*/
			_tmp0 = NULL;
			element_type = (_tmp0 = (ValaDataType*) vala_array_type_new (element_type, gee_collection_get_size ((GeeCollection*) size_specifier_list), vala_code_node_get_source_reference ((ValaCodeNode*) element_type)), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp0);
		} else {
			first = FALSE;
		}
		_tmp1 = NULL;
		size_specifier_list = (_tmp1 = (GeeList*) gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal), (size_specifier_list == NULL) ? NULL : (size_specifier_list = (gee_collection_object_unref (size_specifier_list), NULL)), _tmp1);
		do {
			ValaExpression* size;
			gboolean _tmp2;
			size = NULL;
			_tmp2 = FALSE;
			if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACKET) {
				_tmp2 = vala_parser_current (self) != VALA_TOKEN_TYPE_COMMA;
			} else {
				_tmp2 = FALSE;
			}
			if (_tmp2) {
				ValaExpression* _tmp3;
				ValaExpression* _tmp4;
				_tmp3 = vala_parser_parse_expression (self, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(size == NULL) ? NULL : (size = (vala_code_node_unref (size), NULL));
						(size_specifier_list == NULL) ? NULL : (size_specifier_list = (gee_collection_object_unref (size_specifier_list), NULL));
						(element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL));
						return NULL;
					} else {
						(size == NULL) ? NULL : (size = (vala_code_node_unref (size), NULL));
						(size_specifier_list == NULL) ? NULL : (size_specifier_list = (gee_collection_object_unref (size_specifier_list), NULL));
						(element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp4 = NULL;
				size = (_tmp4 = _tmp3, (size == NULL) ? NULL : (size = (vala_code_node_unref (size), NULL)), _tmp4);
				size_specified = TRUE;
			}
			gee_collection_add ((GeeCollection*) size_specifier_list, size);
			(size == NULL) ? NULL : (size = (vala_code_node_unref (size), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACKET, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(size_specifier_list == NULL) ? NULL : (size_specifier_list = (gee_collection_object_unref (size_specifier_list), NULL));
				(element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL));
				return NULL;
			} else {
				(size_specifier_list == NULL) ? NULL : (size_specifier_list = (gee_collection_object_unref (size_specifier_list), NULL));
				(element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_BRACKET));
	initializer = NULL;
	if (vala_parser_current (self) == VALA_TOKEN_TYPE_OPEN_BRACE) {
		ValaInitializerList* _tmp5;
		ValaInitializerList* _tmp6;
		_tmp5 = vala_parser_parse_initializer (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(size_specifier_list == NULL) ? NULL : (size_specifier_list = (gee_collection_object_unref (size_specifier_list), NULL));
				(element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL));
				(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
				return NULL;
			} else {
				(size_specifier_list == NULL) ? NULL : (size_specifier_list = (gee_collection_object_unref (size_specifier_list), NULL));
				(element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL));
				(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp6 = NULL;
		initializer = (_tmp6 = _tmp5, (initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)), _tmp6);
	}
	_tmp7 = NULL;
	_tmp8 = NULL;
	expr = (_tmp8 = vala_array_creation_expression_new (element_type, gee_collection_get_size ((GeeCollection*) size_specifier_list), initializer, _tmp7 = vala_parser_get_src (self, &(*begin))), (_tmp7 == NULL) ? NULL : (_tmp7 = (vala_source_reference_unref (_tmp7), NULL)), _tmp8);
	if (size_specified) {
		{
			GeeIterator* _size_it;
			_size_it = gee_iterable_iterator ((GeeIterable*) size_specifier_list);
			while (gee_iterator_next (_size_it)) {
				ValaExpression* size;
				size = (ValaExpression*) gee_iterator_get (_size_it);
				vala_array_creation_expression_append_size (expr, size);
				(size == NULL) ? NULL : (size = (vala_code_node_unref (size), NULL));
			}
			(_size_it == NULL) ? NULL : (_size_it = (gee_collection_object_unref (_size_it), NULL));
		}
	}
	_tmp9 = NULL;
	return (_tmp9 = (ValaExpression*) expr, (size_specifier_list == NULL) ? NULL : (size_specifier_list = (gee_collection_object_unref (size_specifier_list), NULL)), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), (initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)), _tmp9);
}


static GeeList* vala_parser_parse_object_initializer (ValaParser* self, GError** error) {
	GError * inner_error;
	GeeArrayList* list;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	list = gee_array_list_new (VALA_TYPE_MEMBER_INITIALIZER, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_BRACE)) {
		do {
			ValaMemberInitializer* _tmp0;
			ValaMemberInitializer* _tmp1;
			_tmp0 = vala_parser_parse_member_initializer (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
					return NULL;
				} else {
					(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp1 = NULL;
			gee_collection_add ((GeeCollection*) list, _tmp1 = _tmp0);
			(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACE, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
				return NULL;
			} else {
				(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
	}
	return (GeeList*) list;
}


static ValaMemberInitializer* vala_parser_parse_member_initializer (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	char* id;
	ValaExpression* expr;
	ValaSourceReference* _tmp0;
	ValaMemberInitializer* _tmp1;
	ValaMemberInitializer* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_ASSIGN, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			id = (g_free (id), NULL);
			return NULL;
		} else {
			id = (g_free (id), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	expr = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			id = (g_free (id), NULL);
			return NULL;
		} else {
			id = (g_free (id), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = vala_member_initializer_new (id, expr, _tmp0 = vala_parser_get_src (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1), id = (g_free (id), NULL), (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp2);
}


static ValaExpression* vala_parser_parse_sizeof_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaDataType* type;
	ValaSourceReference* _tmp0;
	ValaExpression* _tmp1;
	ValaExpression* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_SIZEOF, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	type = vala_parser_parse_type (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = (ValaExpression*) vala_sizeof_expression_new (type, _tmp0 = vala_parser_get_src (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp2);
}


static ValaExpression* vala_parser_parse_typeof_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaDataType* type;
	ValaSourceReference* _tmp0;
	ValaExpression* _tmp1;
	ValaExpression* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_TYPEOF, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	type = vala_parser_parse_type (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = (ValaExpression*) vala_typeof_expression_new (type, _tmp0 = vala_parser_get_src (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp2);
}


static ValaUnaryOperator vala_parser_get_unary_operator (ValaParser* self, ValaTokenType token_type) {
	g_return_val_if_fail (self != NULL, 0);
	switch (token_type) {
		case VALA_TOKEN_TYPE_PLUS:
		{
			return VALA_UNARY_OPERATOR_PLUS;
		}
		case VALA_TOKEN_TYPE_MINUS:
		{
			return VALA_UNARY_OPERATOR_MINUS;
		}
		case VALA_TOKEN_TYPE_OP_NEG:
		{
			return VALA_UNARY_OPERATOR_LOGICAL_NEGATION;
		}
		case VALA_TOKEN_TYPE_TILDE:
		{
			return VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT;
		}
		case VALA_TOKEN_TYPE_OP_INC:
		{
			return VALA_UNARY_OPERATOR_INCREMENT;
		}
		case VALA_TOKEN_TYPE_OP_DEC:
		{
			return VALA_UNARY_OPERATOR_DECREMENT;
		}
		default:
		{
			return VALA_UNARY_OPERATOR_NONE;
		}
	}
}


static ValaExpression* vala_parser_parse_unary_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaUnaryOperator operator;
	ValaExpression* expr;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	operator = vala_parser_get_unary_operator (self, vala_parser_current (self));
	if (operator != VALA_UNARY_OPERATOR_NONE) {
		ValaExpression* op;
		ValaSourceReference* _tmp0;
		ValaExpression* _tmp1;
		ValaExpression* _tmp2;
		vala_parser_next (self);
		op = vala_parser_parse_unary_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				return NULL;
			} else {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp0 = NULL;
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = (_tmp1 = (ValaExpression*) vala_unary_expression_new (operator, op, _tmp0 = vala_parser_get_src (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1), (op == NULL) ? NULL : (op = (vala_code_node_unref (op), NULL)), _tmp2);
	}
	switch (vala_parser_current (self)) {
		case VALA_TOKEN_TYPE_HASH:
		{
			ValaExpression* op;
			ValaSourceReference* _tmp3;
			ValaExpression* _tmp4;
			ValaExpression* _tmp5;
			vala_parser_next (self);
			op = vala_parser_parse_unary_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp3 = NULL;
			_tmp4 = NULL;
			_tmp5 = NULL;
			return (_tmp5 = (_tmp4 = (ValaExpression*) vala_reference_transfer_expression_new (op, _tmp3 = vala_parser_get_src (self, &begin)), (_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)), _tmp4), (op == NULL) ? NULL : (op = (vala_code_node_unref (op), NULL)), _tmp5);
		}
		case VALA_TOKEN_TYPE_OPEN_PARENS:
		{
			vala_parser_next (self);
			switch (vala_parser_current (self)) {
				case VALA_TOKEN_TYPE_OWNED:
				{
					vala_parser_next (self);
					if (vala_parser_accept (self, VALA_TOKEN_TYPE_CLOSE_PARENS)) {
						ValaExpression* op;
						ValaSourceReference* _tmp6;
						ValaExpression* _tmp7;
						ValaExpression* _tmp8;
						op = vala_parser_parse_unary_expression (self, &inner_error);
						if (inner_error != NULL) {
							if (inner_error->domain == VALA_PARSE_ERROR) {
								g_propagate_error (error, inner_error);
								return NULL;
							} else {
								g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
								g_clear_error (&inner_error);
								return NULL;
							}
						}
						_tmp6 = NULL;
						_tmp7 = NULL;
						_tmp8 = NULL;
						return (_tmp8 = (_tmp7 = (ValaExpression*) vala_reference_transfer_expression_new (op, _tmp6 = vala_parser_get_src (self, &begin)), (_tmp6 == NULL) ? NULL : (_tmp6 = (vala_source_reference_unref (_tmp6), NULL)), _tmp7), (op == NULL) ? NULL : (op = (vala_code_node_unref (op), NULL)), _tmp8);
					}
					break;
				}
				case VALA_TOKEN_TYPE_VOID:
				case VALA_TOKEN_TYPE_DYNAMIC:
				case VALA_TOKEN_TYPE_IDENTIFIER:
				{
					ValaDataType* type;
					type = vala_parser_parse_type (self, TRUE, &inner_error);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							g_propagate_error (error, inner_error);
							return NULL;
						} else {
							g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
							g_clear_error (&inner_error);
							return NULL;
						}
					}
					if (vala_parser_accept (self, VALA_TOKEN_TYPE_CLOSE_PARENS)) {
						/* check follower to decide whether to create cast expression*/
						switch (vala_parser_current (self)) {
							case VALA_TOKEN_TYPE_OP_NEG:
							case VALA_TOKEN_TYPE_TILDE:
							case VALA_TOKEN_TYPE_OPEN_PARENS:
							case VALA_TOKEN_TYPE_TRUE:
							case VALA_TOKEN_TYPE_FALSE:
							case VALA_TOKEN_TYPE_INTEGER_LITERAL:
							case VALA_TOKEN_TYPE_REAL_LITERAL:
							case VALA_TOKEN_TYPE_CHARACTER_LITERAL:
							case VALA_TOKEN_TYPE_STRING_LITERAL:
							case VALA_TOKEN_TYPE_VERBATIM_STRING_LITERAL:
							case VALA_TOKEN_TYPE_NULL:
							case VALA_TOKEN_TYPE_THIS:
							case VALA_TOKEN_TYPE_BASE:
							case VALA_TOKEN_TYPE_NEW:
							case VALA_TOKEN_TYPE_SIZEOF:
							case VALA_TOKEN_TYPE_TYPEOF:
							case VALA_TOKEN_TYPE_IDENTIFIER:
							case VALA_TOKEN_TYPE_PARAMS:
							{
								ValaExpression* inner;
								ValaSourceReference* _tmp9;
								ValaExpression* _tmp10;
								ValaExpression* _tmp11;
								inner = vala_parser_parse_unary_expression (self, &inner_error);
								if (inner_error != NULL) {
									if (inner_error->domain == VALA_PARSE_ERROR) {
										g_propagate_error (error, inner_error);
										(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
										return NULL;
									} else {
										(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
										g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
										g_clear_error (&inner_error);
										return NULL;
									}
								}
								_tmp9 = NULL;
								_tmp10 = NULL;
								_tmp11 = NULL;
								return (_tmp11 = (_tmp10 = (ValaExpression*) vala_cast_expression_new (inner, type, _tmp9 = vala_parser_get_src (self, &begin), FALSE), (_tmp9 == NULL) ? NULL : (_tmp9 = (vala_source_reference_unref (_tmp9), NULL)), _tmp10), (inner == NULL) ? NULL : (inner = (vala_code_node_unref (inner), NULL)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp11);
							}
							default:
							{
								break;
							}
						}
					}
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					break;
				}
				default:
				{
					break;
				}
			}
			vala_parser_rollback (self, &begin);
			break;
		}
		case VALA_TOKEN_TYPE_STAR:
		{
			ValaExpression* op;
			ValaSourceReference* _tmp12;
			ValaExpression* _tmp13;
			ValaExpression* _tmp14;
			vala_parser_next (self);
			op = vala_parser_parse_unary_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp12 = NULL;
			_tmp13 = NULL;
			_tmp14 = NULL;
			return (_tmp14 = (_tmp13 = (ValaExpression*) vala_pointer_indirection_new (op, _tmp12 = vala_parser_get_src (self, &begin)), (_tmp12 == NULL) ? NULL : (_tmp12 = (vala_source_reference_unref (_tmp12), NULL)), _tmp13), (op == NULL) ? NULL : (op = (vala_code_node_unref (op), NULL)), _tmp14);
		}
		case VALA_TOKEN_TYPE_BITWISE_AND:
		{
			ValaExpression* op;
			ValaSourceReference* _tmp15;
			ValaExpression* _tmp16;
			ValaExpression* _tmp17;
			vala_parser_next (self);
			op = vala_parser_parse_unary_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp15 = NULL;
			_tmp16 = NULL;
			_tmp17 = NULL;
			return (_tmp17 = (_tmp16 = (ValaExpression*) vala_addressof_expression_new (op, _tmp15 = vala_parser_get_src (self, &begin)), (_tmp15 == NULL) ? NULL : (_tmp15 = (vala_source_reference_unref (_tmp15), NULL)), _tmp16), (op == NULL) ? NULL : (op = (vala_code_node_unref (op), NULL)), _tmp17);
		}
		default:
		{
			break;
		}
	}
	expr = vala_parser_parse_primary_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	return expr;
}


static ValaBinaryOperator vala_parser_get_binary_operator (ValaParser* self, ValaTokenType token_type) {
	g_return_val_if_fail (self != NULL, 0);
	switch (token_type) {
		case VALA_TOKEN_TYPE_STAR:
		{
			return VALA_BINARY_OPERATOR_MUL;
		}
		case VALA_TOKEN_TYPE_DIV:
		{
			return VALA_BINARY_OPERATOR_DIV;
		}
		case VALA_TOKEN_TYPE_PERCENT:
		{
			return VALA_BINARY_OPERATOR_MOD;
		}
		case VALA_TOKEN_TYPE_PLUS:
		{
			return VALA_BINARY_OPERATOR_PLUS;
		}
		case VALA_TOKEN_TYPE_MINUS:
		{
			return VALA_BINARY_OPERATOR_MINUS;
		}
		case VALA_TOKEN_TYPE_OP_LT:
		{
			return VALA_BINARY_OPERATOR_LESS_THAN;
		}
		case VALA_TOKEN_TYPE_OP_GT:
		{
			return VALA_BINARY_OPERATOR_GREATER_THAN;
		}
		case VALA_TOKEN_TYPE_OP_LE:
		{
			return VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL;
		}
		case VALA_TOKEN_TYPE_OP_GE:
		{
			return VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL;
		}
		case VALA_TOKEN_TYPE_OP_EQ:
		{
			return VALA_BINARY_OPERATOR_EQUALITY;
		}
		case VALA_TOKEN_TYPE_OP_NE:
		{
			return VALA_BINARY_OPERATOR_INEQUALITY;
		}
		default:
		{
			return VALA_BINARY_OPERATOR_NONE;
		}
	}
}


static ValaExpression* vala_parser_parse_multiplicative_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	gboolean found;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_unary_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	found = TRUE;
	while (found) {
		ValaBinaryOperator operator;
		operator = vala_parser_get_binary_operator (self, vala_parser_current (self));
		switch (operator) {
			case VALA_BINARY_OPERATOR_MUL:
			case VALA_BINARY_OPERATOR_DIV:
			case VALA_BINARY_OPERATOR_MOD:
			{
				ValaExpression* right;
				ValaExpression* _tmp1;
				ValaSourceReference* _tmp0;
				vala_parser_next (self);
				right = vala_parser_parse_unary_expression (self, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
						return NULL;
					} else {
						(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp1 = NULL;
				_tmp0 = NULL;
				left = (_tmp1 = (ValaExpression*) vala_binary_expression_new (operator, left, right, _tmp0 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp1);
				(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
				(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
				break;
			}
			default:
			{
				found = FALSE;
				break;
			}
		}
	}
	return left;
}


static ValaExpression* vala_parser_parse_additive_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	gboolean found;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_multiplicative_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	found = TRUE;
	while (found) {
		ValaBinaryOperator operator;
		operator = vala_parser_get_binary_operator (self, vala_parser_current (self));
		switch (operator) {
			case VALA_BINARY_OPERATOR_PLUS:
			case VALA_BINARY_OPERATOR_MINUS:
			{
				ValaExpression* right;
				ValaExpression* _tmp1;
				ValaSourceReference* _tmp0;
				vala_parser_next (self);
				right = vala_parser_parse_multiplicative_expression (self, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
						return NULL;
					} else {
						(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp1 = NULL;
				_tmp0 = NULL;
				left = (_tmp1 = (ValaExpression*) vala_binary_expression_new (operator, left, right, _tmp0 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp1);
				(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
				(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
				break;
			}
			default:
			{
				found = FALSE;
				break;
			}
		}
	}
	return left;
}


static ValaExpression* vala_parser_parse_shift_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	gboolean found;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_additive_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	found = TRUE;
	while (found) {
		switch (vala_parser_current (self)) {
			case VALA_TOKEN_TYPE_OP_SHIFT_LEFT:
			{
				ValaExpression* right;
				ValaExpression* _tmp1;
				ValaSourceReference* _tmp0;
				vala_parser_next (self);
				right = vala_parser_parse_additive_expression (self, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
						return NULL;
					} else {
						(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp1 = NULL;
				_tmp0 = NULL;
				left = (_tmp1 = (ValaExpression*) vala_binary_expression_new (VALA_BINARY_OPERATOR_SHIFT_LEFT, left, right, _tmp0 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp1);
				(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
				(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
				break;
			}
			case VALA_TOKEN_TYPE_OP_GT:
			{
				gchar* first_gt_pos;
				gboolean _tmp2;
				first_gt_pos = self->priv->tokens[self->priv->index].begin.pos;
				vala_parser_next (self);
				_tmp2 = FALSE;
				if (vala_parser_current (self) == VALA_TOKEN_TYPE_OP_GT) {
					_tmp2 = self->priv->tokens[self->priv->index].begin.pos == (first_gt_pos + 1);
				} else {
					_tmp2 = FALSE;
				}
				if (_tmp2) {
					ValaExpression* right;
					ValaExpression* _tmp4;
					ValaSourceReference* _tmp3;
					vala_parser_next (self);
					right = vala_parser_parse_additive_expression (self, &inner_error);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							g_propagate_error (error, inner_error);
							(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
							return NULL;
						} else {
							(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
							g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
							g_clear_error (&inner_error);
							return NULL;
						}
					}
					_tmp4 = NULL;
					_tmp3 = NULL;
					left = (_tmp4 = (ValaExpression*) vala_binary_expression_new (VALA_BINARY_OPERATOR_SHIFT_RIGHT, left, right, _tmp3 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp4);
					(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL));
					(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
				} else {
					vala_parser_prev (self);
					found = FALSE;
				}
				break;
			}
			default:
			{
				found = FALSE;
				break;
			}
		}
	}
	return left;
}


static ValaExpression* vala_parser_parse_relational_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	gboolean found;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_shift_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	found = TRUE;
	while (found) {
		ValaBinaryOperator operator;
		operator = vala_parser_get_binary_operator (self, vala_parser_current (self));
		switch (operator) {
			case VALA_BINARY_OPERATOR_LESS_THAN:
			case VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL:
			case VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL:
			{
				ValaExpression* right;
				ValaExpression* _tmp1;
				ValaSourceReference* _tmp0;
				vala_parser_next (self);
				right = vala_parser_parse_shift_expression (self, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
						return NULL;
					} else {
						(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp1 = NULL;
				_tmp0 = NULL;
				left = (_tmp1 = (ValaExpression*) vala_binary_expression_new (operator, left, right, _tmp0 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp1);
				(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
				(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
				break;
			}
			case VALA_BINARY_OPERATOR_GREATER_THAN:
			{
				gboolean _tmp2;
				vala_parser_next (self);
				_tmp2 = FALSE;
				if (vala_parser_current (self) != VALA_TOKEN_TYPE_OP_GT) {
					_tmp2 = vala_parser_current (self) != VALA_TOKEN_TYPE_OP_GE;
				} else {
					_tmp2 = FALSE;
				}
				if (_tmp2) {
					ValaExpression* right;
					ValaExpression* _tmp4;
					ValaSourceReference* _tmp3;
					right = vala_parser_parse_shift_expression (self, &inner_error);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							g_propagate_error (error, inner_error);
							(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
							return NULL;
						} else {
							(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
							g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
							g_clear_error (&inner_error);
							return NULL;
						}
					}
					_tmp4 = NULL;
					_tmp3 = NULL;
					left = (_tmp4 = (ValaExpression*) vala_binary_expression_new (operator, left, right, _tmp3 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp4);
					(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL));
					(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
				} else {
					vala_parser_prev (self);
					found = FALSE;
				}
				break;
			}
			default:
			{
				switch (vala_parser_current (self)) {
					case VALA_TOKEN_TYPE_IS:
					{
						ValaDataType* type;
						ValaExpression* _tmp6;
						ValaSourceReference* _tmp5;
						vala_parser_next (self);
						type = vala_parser_parse_type (self, TRUE, &inner_error);
						if (inner_error != NULL) {
							if (inner_error->domain == VALA_PARSE_ERROR) {
								g_propagate_error (error, inner_error);
								(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
								return NULL;
							} else {
								(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
								g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
								g_clear_error (&inner_error);
								return NULL;
							}
						}
						_tmp6 = NULL;
						_tmp5 = NULL;
						left = (_tmp6 = (ValaExpression*) vala_typecheck_new (left, type, _tmp5 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp6);
						(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL));
						(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
						break;
					}
					case VALA_TOKEN_TYPE_AS:
					{
						ValaDataType* type;
						ValaExpression* _tmp8;
						ValaSourceReference* _tmp7;
						vala_parser_next (self);
						type = vala_parser_parse_type (self, TRUE, &inner_error);
						if (inner_error != NULL) {
							if (inner_error->domain == VALA_PARSE_ERROR) {
								g_propagate_error (error, inner_error);
								(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
								return NULL;
							} else {
								(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
								g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
								g_clear_error (&inner_error);
								return NULL;
							}
						}
						_tmp8 = NULL;
						_tmp7 = NULL;
						left = (_tmp8 = (ValaExpression*) vala_cast_expression_new (left, type, _tmp7 = vala_parser_get_src (self, &begin), TRUE), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp8);
						(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_source_reference_unref (_tmp7), NULL));
						(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
						break;
					}
					default:
					{
						found = FALSE;
						break;
					}
				}
				break;
			}
		}
	}
	return left;
}


static ValaExpression* vala_parser_parse_equality_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	gboolean found;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_relational_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	found = TRUE;
	while (found) {
		ValaBinaryOperator operator;
		operator = vala_parser_get_binary_operator (self, vala_parser_current (self));
		switch (operator) {
			case VALA_BINARY_OPERATOR_EQUALITY:
			case VALA_BINARY_OPERATOR_INEQUALITY:
			{
				ValaExpression* right;
				ValaExpression* _tmp1;
				ValaSourceReference* _tmp0;
				vala_parser_next (self);
				right = vala_parser_parse_relational_expression (self, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
						return NULL;
					} else {
						(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp1 = NULL;
				_tmp0 = NULL;
				left = (_tmp1 = (ValaExpression*) vala_binary_expression_new (operator, left, right, _tmp0 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp1);
				(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
				(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
				break;
			}
			default:
			{
				found = FALSE;
				break;
			}
		}
	}
	return left;
}


static ValaExpression* vala_parser_parse_and_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_equality_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_BITWISE_AND)) {
		ValaExpression* right;
		ValaExpression* _tmp1;
		ValaSourceReference* _tmp0;
		right = vala_parser_parse_equality_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
				return NULL;
			} else {
				(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		left = (_tmp1 = (ValaExpression*) vala_binary_expression_new (VALA_BINARY_OPERATOR_BITWISE_AND, left, right, _tmp0 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp1);
		(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
		(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
	}
	return left;
}


static ValaExpression* vala_parser_parse_exclusive_or_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_and_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_CARRET)) {
		ValaExpression* right;
		ValaExpression* _tmp1;
		ValaSourceReference* _tmp0;
		right = vala_parser_parse_and_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
				return NULL;
			} else {
				(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		left = (_tmp1 = (ValaExpression*) vala_binary_expression_new (VALA_BINARY_OPERATOR_BITWISE_XOR, left, right, _tmp0 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp1);
		(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
		(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
	}
	return left;
}


static ValaExpression* vala_parser_parse_inclusive_or_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_exclusive_or_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_BITWISE_OR)) {
		ValaExpression* right;
		ValaExpression* _tmp1;
		ValaSourceReference* _tmp0;
		right = vala_parser_parse_exclusive_or_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
				return NULL;
			} else {
				(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		left = (_tmp1 = (ValaExpression*) vala_binary_expression_new (VALA_BINARY_OPERATOR_BITWISE_OR, left, right, _tmp0 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp1);
		(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
		(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
	}
	return left;
}


static ValaExpression* vala_parser_parse_in_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_inclusive_or_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_IN)) {
		ValaExpression* right;
		ValaExpression* _tmp1;
		ValaSourceReference* _tmp0;
		right = vala_parser_parse_inclusive_or_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
				return NULL;
			} else {
				(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		left = (_tmp1 = (ValaExpression*) vala_binary_expression_new (VALA_BINARY_OPERATOR_IN, left, right, _tmp0 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp1);
		(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
		(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
	}
	return left;
}


static ValaExpression* vala_parser_parse_conditional_and_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_in_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_OP_AND)) {
		ValaExpression* right;
		ValaExpression* _tmp1;
		ValaSourceReference* _tmp0;
		right = vala_parser_parse_in_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
				return NULL;
			} else {
				(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		left = (_tmp1 = (ValaExpression*) vala_binary_expression_new (VALA_BINARY_OPERATOR_AND, left, right, _tmp0 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp1);
		(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
		(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
	}
	return left;
}


static ValaExpression* vala_parser_parse_conditional_or_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_conditional_and_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_OP_OR)) {
		ValaExpression* right;
		ValaExpression* _tmp1;
		ValaSourceReference* _tmp0;
		right = vala_parser_parse_conditional_and_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
				return NULL;
			} else {
				(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		left = (_tmp1 = (ValaExpression*) vala_binary_expression_new (VALA_BINARY_OPERATOR_OR, left, right, _tmp0 = vala_parser_get_src (self, &begin)), (left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL)), _tmp1);
		(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
		(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
	}
	return left;
}


static ValaExpression* vala_parser_parse_conditional_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* condition;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	condition = vala_parser_parse_conditional_or_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_INTERR)) {
		ValaExpression* true_expr;
		ValaExpression* false_expr;
		ValaSourceReference* _tmp0;
		ValaExpression* _tmp1;
		ValaExpression* _tmp2;
		true_expr = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
				return NULL;
			} else {
				(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		vala_parser_expect (self, VALA_TOKEN_TYPE_COLON, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(true_expr == NULL) ? NULL : (true_expr = (vala_code_node_unref (true_expr), NULL));
				(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
				return NULL;
			} else {
				(true_expr == NULL) ? NULL : (true_expr = (vala_code_node_unref (true_expr), NULL));
				(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		false_expr = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(true_expr == NULL) ? NULL : (true_expr = (vala_code_node_unref (true_expr), NULL));
				(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
				return NULL;
			} else {
				(true_expr == NULL) ? NULL : (true_expr = (vala_code_node_unref (true_expr), NULL));
				(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp0 = NULL;
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = (_tmp1 = (ValaExpression*) vala_conditional_expression_new (condition, true_expr, false_expr, _tmp0 = vala_parser_get_src (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1), (true_expr == NULL) ? NULL : (true_expr = (vala_code_node_unref (true_expr), NULL)), (false_expr == NULL) ? NULL : (false_expr = (vala_code_node_unref (false_expr), NULL)), (condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL)), _tmp2);
	} else {
		return condition;
	}
	(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
}


static ValaExpression* vala_parser_parse_lambda_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	GeeList* params;
	ValaLambdaExpression* lambda;
	ValaExpression* _tmp8;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	params = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_PARENS)) {
		if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
			do {
				char* _tmp0;
				char* _tmp1;
				_tmp0 = vala_parser_parse_identifier (self, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
						return NULL;
					} else {
						(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp1 = NULL;
				gee_collection_add ((GeeCollection*) params, _tmp1 = _tmp0);
				_tmp1 = (g_free (_tmp1), NULL);
			} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		}
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
				return NULL;
			} else {
				(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
	} else {
		char* _tmp2;
		char* _tmp3;
		_tmp2 = vala_parser_parse_identifier (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
				return NULL;
			} else {
				(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp3 = NULL;
		gee_collection_add ((GeeCollection*) params, _tmp3 = _tmp2);
		_tmp3 = (g_free (_tmp3), NULL);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_LAMBDA, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
			return NULL;
		} else {
			(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	lambda = NULL;
	if (vala_parser_current (self) == VALA_TOKEN_TYPE_OPEN_BRACE) {
		ValaBlock* block;
		ValaLambdaExpression* _tmp5;
		ValaSourceReference* _tmp4;
		block = vala_parser_parse_block (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
				(lambda == NULL) ? NULL : (lambda = (vala_code_node_unref (lambda), NULL));
				return NULL;
			} else {
				(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
				(lambda == NULL) ? NULL : (lambda = (vala_code_node_unref (lambda), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp5 = NULL;
		_tmp4 = NULL;
		lambda = (_tmp5 = vala_lambda_expression_new_with_statement_body (block, _tmp4 = vala_parser_get_src (self, &begin)), (lambda == NULL) ? NULL : (lambda = (vala_code_node_unref (lambda), NULL)), _tmp5);
		(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL));
		(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
	} else {
		ValaExpression* expr;
		ValaLambdaExpression* _tmp7;
		ValaSourceReference* _tmp6;
		expr = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
				(lambda == NULL) ? NULL : (lambda = (vala_code_node_unref (lambda), NULL));
				return NULL;
			} else {
				(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
				(lambda == NULL) ? NULL : (lambda = (vala_code_node_unref (lambda), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp7 = NULL;
		_tmp6 = NULL;
		lambda = (_tmp7 = vala_lambda_expression_new (expr, _tmp6 = vala_parser_get_src (self, &begin)), (lambda == NULL) ? NULL : (lambda = (vala_code_node_unref (lambda), NULL)), _tmp7);
		(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_source_reference_unref (_tmp6), NULL));
		(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
	}
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) params);
		while (gee_iterator_next (_param_it)) {
			char* param;
			param = (char*) gee_iterator_get (_param_it);
			vala_lambda_expression_add_parameter (lambda, param);
			param = (g_free (param), NULL);
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	_tmp8 = NULL;
	return (_tmp8 = (ValaExpression*) lambda, (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), _tmp8);
}


static ValaAssignmentOperator vala_parser_get_assignment_operator (ValaParser* self, ValaTokenType token_type) {
	g_return_val_if_fail (self != NULL, 0);
	switch (token_type) {
		case VALA_TOKEN_TYPE_ASSIGN:
		{
			return VALA_ASSIGNMENT_OPERATOR_SIMPLE;
		}
		case VALA_TOKEN_TYPE_ASSIGN_ADD:
		{
			return VALA_ASSIGNMENT_OPERATOR_ADD;
		}
		case VALA_TOKEN_TYPE_ASSIGN_SUB:
		{
			return VALA_ASSIGNMENT_OPERATOR_SUB;
		}
		case VALA_TOKEN_TYPE_ASSIGN_BITWISE_OR:
		{
			return VALA_ASSIGNMENT_OPERATOR_BITWISE_OR;
		}
		case VALA_TOKEN_TYPE_ASSIGN_BITWISE_AND:
		{
			return VALA_ASSIGNMENT_OPERATOR_BITWISE_AND;
		}
		case VALA_TOKEN_TYPE_ASSIGN_BITWISE_XOR:
		{
			return VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR;
		}
		case VALA_TOKEN_TYPE_ASSIGN_DIV:
		{
			return VALA_ASSIGNMENT_OPERATOR_DIV;
		}
		case VALA_TOKEN_TYPE_ASSIGN_MUL:
		{
			return VALA_ASSIGNMENT_OPERATOR_MUL;
		}
		case VALA_TOKEN_TYPE_ASSIGN_PERCENT:
		{
			return VALA_ASSIGNMENT_OPERATOR_PERCENT;
		}
		case VALA_TOKEN_TYPE_ASSIGN_SHIFT_LEFT:
		{
			return VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT;
		}
		default:
		{
			return VALA_ASSIGNMENT_OPERATOR_NONE;
		}
	}
}


static ValaExpression* vala_parser_parse_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	expr = vala_parser_parse_conditional_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if (vala_parser_current (self) == VALA_TOKEN_TYPE_LAMBDA) {
		ValaExpression* lambda;
		ValaExpression* _tmp0;
		vala_parser_rollback (self, &begin);
		lambda = vala_parser_parse_lambda_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
				return NULL;
			} else {
				(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp0 = NULL;
		return (_tmp0 = lambda, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp0);
	}
	while (TRUE) {
		ValaAssignmentOperator operator;
		operator = vala_parser_get_assignment_operator (self, vala_parser_current (self));
		if (operator != VALA_ASSIGNMENT_OPERATOR_NONE) {
			ValaExpression* rhs;
			ValaExpression* _tmp2;
			ValaSourceReference* _tmp1;
			vala_parser_next (self);
			rhs = vala_parser_parse_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					return NULL;
				} else {
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp2 = NULL;
			_tmp1 = NULL;
			expr = (_tmp2 = (ValaExpression*) vala_assignment_new (expr, rhs, operator, _tmp1 = vala_parser_get_src (self, &begin)), (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp2);
			(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL));
			(rhs == NULL) ? NULL : (rhs = (vala_code_node_unref (rhs), NULL));
		} else {
			if (vala_parser_current (self) == VALA_TOKEN_TYPE_OP_GT) {
				gchar* first_gt_pos;
				gboolean _tmp3;
				/* >>=*/
				first_gt_pos = self->priv->tokens[self->priv->index].begin.pos;
				vala_parser_next (self);
				_tmp3 = FALSE;
				if (vala_parser_current (self) == VALA_TOKEN_TYPE_OP_GE) {
					_tmp3 = self->priv->tokens[self->priv->index].begin.pos == (first_gt_pos + 1);
				} else {
					_tmp3 = FALSE;
				}
				/* only accept >>= when there is no space between the two > signs*/
				if (_tmp3) {
					ValaExpression* rhs;
					ValaExpression* _tmp5;
					ValaSourceReference* _tmp4;
					vala_parser_next (self);
					rhs = vala_parser_parse_expression (self, &inner_error);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							g_propagate_error (error, inner_error);
							(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
							return NULL;
						} else {
							(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
							g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
							g_clear_error (&inner_error);
							return NULL;
						}
					}
					_tmp5 = NULL;
					_tmp4 = NULL;
					expr = (_tmp5 = (ValaExpression*) vala_assignment_new (expr, rhs, VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT, _tmp4 = vala_parser_get_src (self, &begin)), (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp5);
					(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL));
					(rhs == NULL) ? NULL : (rhs = (vala_code_node_unref (rhs), NULL));
				} else {
					vala_parser_prev (self);
					break;
				}
			} else {
				break;
			}
		}
	}
	return expr;
}


static void vala_parser_parse_statements (ValaParser* self, ValaBlock* block, GError** error) {
	GError * inner_error;
	g_return_if_fail (self != NULL);
	g_return_if_fail (block != NULL);
	inner_error = NULL;
	while (TRUE) {
		gboolean _tmp0;
		gboolean _tmp1;
		_tmp0 = FALSE;
		_tmp1 = FALSE;
		if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE) {
			_tmp1 = vala_parser_current (self) != VALA_TOKEN_TYPE_CASE;
		} else {
			_tmp1 = FALSE;
		}
		if (_tmp1) {
			_tmp0 = vala_parser_current (self) != VALA_TOKEN_TYPE_DEFAULT;
		} else {
			_tmp0 = FALSE;
		}
		if (!_tmp0) {
			break;
		}
		{
			ValaStatement* stmt;
			gboolean is_decl;
			char* _tmp2;
			stmt = NULL;
			is_decl = FALSE;
			_tmp2 = NULL;
			self->priv->comment = (_tmp2 = vala_scanner_pop_comment (self->priv->scanner), self->priv->comment = (g_free (self->priv->comment), NULL), _tmp2);
			switch (vala_parser_current (self)) {
				case VALA_TOKEN_TYPE_OPEN_BRACE:
				{
					ValaBlock* _tmp3;
					ValaStatement* _tmp4;
					_tmp3 = vala_parser_parse_block (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp4 = NULL;
					stmt = (_tmp4 = (ValaStatement*) _tmp3, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp4);
					break;
				}
				case VALA_TOKEN_TYPE_SEMICOLON:
				{
					ValaStatement* _tmp5;
					ValaStatement* _tmp6;
					_tmp5 = vala_parser_parse_empty_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp6 = NULL;
					stmt = (_tmp6 = _tmp5, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp6);
					break;
				}
				case VALA_TOKEN_TYPE_IF:
				{
					ValaStatement* _tmp7;
					ValaStatement* _tmp8;
					_tmp7 = vala_parser_parse_if_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp8 = NULL;
					stmt = (_tmp8 = _tmp7, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp8);
					break;
				}
				case VALA_TOKEN_TYPE_SWITCH:
				{
					ValaStatement* _tmp9;
					ValaStatement* _tmp10;
					_tmp9 = vala_parser_parse_switch_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp10 = NULL;
					stmt = (_tmp10 = _tmp9, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp10);
					break;
				}
				case VALA_TOKEN_TYPE_WHILE:
				{
					ValaStatement* _tmp11;
					ValaStatement* _tmp12;
					_tmp11 = vala_parser_parse_while_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp12 = NULL;
					stmt = (_tmp12 = _tmp11, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp12);
					break;
				}
				case VALA_TOKEN_TYPE_DO:
				{
					ValaStatement* _tmp13;
					ValaStatement* _tmp14;
					_tmp13 = vala_parser_parse_do_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp14 = NULL;
					stmt = (_tmp14 = _tmp13, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp14);
					break;
				}
				case VALA_TOKEN_TYPE_FOR:
				{
					ValaStatement* _tmp15;
					ValaStatement* _tmp16;
					_tmp15 = vala_parser_parse_for_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp16 = NULL;
					stmt = (_tmp16 = _tmp15, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp16);
					break;
				}
				case VALA_TOKEN_TYPE_FOREACH:
				{
					ValaStatement* _tmp17;
					ValaStatement* _tmp18;
					_tmp17 = vala_parser_parse_foreach_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp18 = NULL;
					stmt = (_tmp18 = _tmp17, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp18);
					break;
				}
				case VALA_TOKEN_TYPE_BREAK:
				{
					ValaStatement* _tmp19;
					ValaStatement* _tmp20;
					_tmp19 = vala_parser_parse_break_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp20 = NULL;
					stmt = (_tmp20 = _tmp19, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp20);
					break;
				}
				case VALA_TOKEN_TYPE_CONTINUE:
				{
					ValaStatement* _tmp21;
					ValaStatement* _tmp22;
					_tmp21 = vala_parser_parse_continue_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp22 = NULL;
					stmt = (_tmp22 = _tmp21, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp22);
					break;
				}
				case VALA_TOKEN_TYPE_RETURN:
				{
					ValaStatement* _tmp23;
					ValaStatement* _tmp24;
					_tmp23 = vala_parser_parse_return_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp24 = NULL;
					stmt = (_tmp24 = _tmp23, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp24);
					break;
				}
				case VALA_TOKEN_TYPE_YIELD:
				{
					ValaStatement* _tmp25;
					ValaStatement* _tmp26;
					_tmp25 = vala_parser_parse_yield_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp26 = NULL;
					stmt = (_tmp26 = _tmp25, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp26);
					break;
				}
				case VALA_TOKEN_TYPE_THROW:
				{
					ValaStatement* _tmp27;
					ValaStatement* _tmp28;
					_tmp27 = vala_parser_parse_throw_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp28 = NULL;
					stmt = (_tmp28 = _tmp27, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp28);
					break;
				}
				case VALA_TOKEN_TYPE_TRY:
				{
					ValaStatement* _tmp29;
					ValaStatement* _tmp30;
					_tmp29 = vala_parser_parse_try_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp30 = NULL;
					stmt = (_tmp30 = _tmp29, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp30);
					break;
				}
				case VALA_TOKEN_TYPE_LOCK:
				{
					ValaStatement* _tmp31;
					ValaStatement* _tmp32;
					_tmp31 = vala_parser_parse_lock_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp32 = NULL;
					stmt = (_tmp32 = _tmp31, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp32);
					break;
				}
				case VALA_TOKEN_TYPE_DELETE:
				{
					ValaStatement* _tmp33;
					ValaStatement* _tmp34;
					_tmp33 = vala_parser_parse_delete_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp34 = NULL;
					stmt = (_tmp34 = _tmp33, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp34);
					break;
				}
				case VALA_TOKEN_TYPE_VAR:
				{
					is_decl = TRUE;
					vala_parser_parse_local_variable_declarations (self, block, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					break;
				}
				case VALA_TOKEN_TYPE_OP_INC:
				case VALA_TOKEN_TYPE_OP_DEC:
				case VALA_TOKEN_TYPE_BASE:
				case VALA_TOKEN_TYPE_THIS:
				case VALA_TOKEN_TYPE_OPEN_PARENS:
				case VALA_TOKEN_TYPE_STAR:
				case VALA_TOKEN_TYPE_NEW:
				{
					ValaStatement* _tmp35;
					ValaStatement* _tmp36;
					_tmp35 = vala_parser_parse_expression_statement (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					_tmp36 = NULL;
					stmt = (_tmp36 = _tmp35, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp36);
					break;
				}
				default:
				{
					gboolean is_expr;
					is_expr = vala_parser_is_expression (self, &inner_error);
					if (inner_error != NULL) {
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						goto __finally4;
					}
					if (is_expr) {
						ValaStatement* _tmp37;
						ValaStatement* _tmp38;
						_tmp37 = vala_parser_parse_expression_statement (self, &inner_error);
						if (inner_error != NULL) {
							(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
							if (inner_error->domain == VALA_PARSE_ERROR) {
								goto __catch4_vala_parse_error;
							}
							goto __finally4;
						}
						_tmp38 = NULL;
						stmt = (_tmp38 = _tmp37, (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp38);
					} else {
						is_decl = TRUE;
						vala_parser_parse_local_variable_declarations (self, block, &inner_error);
						if (inner_error != NULL) {
							(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
							if (inner_error->domain == VALA_PARSE_ERROR) {
								goto __catch4_vala_parse_error;
							}
							goto __finally4;
						}
					}
					break;
				}
			}
			if (!is_decl) {
				vala_block_add_statement (block, stmt);
			}
			(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
		}
		goto __finally4;
		__catch4_vala_parse_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				if (vala_parser_recover (self) != VALA_PARSER_RECOVERY_STATE_STATEMENT_BEGIN) {
					/* beginning of next declaration or end of file reached
					 return what we have so far*/
					(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
					break;
				}
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally4:
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				return;
			} else {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return;
			}
		}
	}
}


static gboolean vala_parser_is_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	g_return_val_if_fail (self != NULL, FALSE);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	/* decide between declaration and expression statement*/
	vala_parser_skip_type (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return FALSE;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return FALSE;
		}
	}
	switch (vala_parser_current (self)) {
		case VALA_TOKEN_TYPE_OPEN_PARENS:
		case VALA_TOKEN_TYPE_OP_INC:
		case VALA_TOKEN_TYPE_OP_DEC:
		case VALA_TOKEN_TYPE_ASSIGN:
		case VALA_TOKEN_TYPE_ASSIGN_ADD:
		case VALA_TOKEN_TYPE_ASSIGN_BITWISE_AND:
		case VALA_TOKEN_TYPE_ASSIGN_BITWISE_OR:
		case VALA_TOKEN_TYPE_ASSIGN_BITWISE_XOR:
		case VALA_TOKEN_TYPE_ASSIGN_DIV:
		case VALA_TOKEN_TYPE_ASSIGN_MUL:
		case VALA_TOKEN_TYPE_ASSIGN_PERCENT:
		case VALA_TOKEN_TYPE_ASSIGN_SHIFT_LEFT:
		case VALA_TOKEN_TYPE_ASSIGN_SUB:
		case VALA_TOKEN_TYPE_OP_GT:
		case VALA_TOKEN_TYPE_DOT:
		case VALA_TOKEN_TYPE_OP_PTR:
		{
			vala_parser_rollback (self, &begin);
			return TRUE;
		}
		default:
		{
			vala_parser_rollback (self, &begin);
			return FALSE;
		}
	}
}


static ValaBlock* vala_parser_parse_embedded_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	char* _tmp1;
	ValaSourceReference* _tmp3;
	ValaSourceLocation _tmp2 = {0};
	ValaBlock* _tmp4;
	ValaBlock* block;
	ValaStatement* _tmp5;
	ValaStatement* _tmp6;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	if (vala_parser_current (self) == VALA_TOKEN_TYPE_OPEN_BRACE) {
		ValaBlock* block;
		block = vala_parser_parse_block (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				return NULL;
			} else {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		return block;
	}
	_tmp1 = NULL;
	self->priv->comment = (_tmp1 = vala_scanner_pop_comment (self->priv->scanner), self->priv->comment = (g_free (self->priv->comment), NULL), _tmp1);
	_tmp3 = NULL;
	_tmp4 = NULL;
	block = (_tmp4 = vala_block_new (_tmp3 = vala_parser_get_src_com (self, (_tmp2 = vala_parser_get_location (self), &_tmp2))), (_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)), _tmp4);
	_tmp5 = vala_parser_parse_embedded_statement_without_block (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			return NULL;
		} else {
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp6 = NULL;
	vala_block_add_statement (block, _tmp6 = _tmp5);
	(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
	return block;
}


static ValaStatement* vala_parser_parse_embedded_statement_without_block (ValaParser* self, GError** error) {
	GError * inner_error;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	switch (vala_parser_current (self)) {
		case VALA_TOKEN_TYPE_SEMICOLON:
		{
			ValaStatement* _tmp0;
			_tmp0 = vala_parser_parse_empty_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp0;
		}
		case VALA_TOKEN_TYPE_IF:
		{
			ValaStatement* _tmp2;
			_tmp2 = vala_parser_parse_if_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp2;
		}
		case VALA_TOKEN_TYPE_SWITCH:
		{
			ValaStatement* _tmp4;
			_tmp4 = vala_parser_parse_switch_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp4;
		}
		case VALA_TOKEN_TYPE_WHILE:
		{
			ValaStatement* _tmp6;
			_tmp6 = vala_parser_parse_while_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp6;
		}
		case VALA_TOKEN_TYPE_DO:
		{
			ValaStatement* _tmp8;
			_tmp8 = vala_parser_parse_do_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp8;
		}
		case VALA_TOKEN_TYPE_FOR:
		{
			ValaStatement* _tmp10;
			_tmp10 = vala_parser_parse_for_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp10;
		}
		case VALA_TOKEN_TYPE_FOREACH:
		{
			ValaStatement* _tmp12;
			_tmp12 = vala_parser_parse_foreach_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp12;
		}
		case VALA_TOKEN_TYPE_BREAK:
		{
			ValaStatement* _tmp14;
			_tmp14 = vala_parser_parse_break_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp14;
		}
		case VALA_TOKEN_TYPE_CONTINUE:
		{
			ValaStatement* _tmp16;
			_tmp16 = vala_parser_parse_continue_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp16;
		}
		case VALA_TOKEN_TYPE_RETURN:
		{
			ValaStatement* _tmp18;
			_tmp18 = vala_parser_parse_return_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp18;
		}
		case VALA_TOKEN_TYPE_YIELD:
		{
			ValaStatement* _tmp20;
			_tmp20 = vala_parser_parse_yield_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp20;
		}
		case VALA_TOKEN_TYPE_THROW:
		{
			ValaStatement* _tmp22;
			_tmp22 = vala_parser_parse_throw_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp22;
		}
		case VALA_TOKEN_TYPE_TRY:
		{
			ValaStatement* _tmp24;
			_tmp24 = vala_parser_parse_try_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp24;
		}
		case VALA_TOKEN_TYPE_LOCK:
		{
			ValaStatement* _tmp26;
			_tmp26 = vala_parser_parse_lock_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp26;
		}
		case VALA_TOKEN_TYPE_DELETE:
		{
			ValaStatement* _tmp28;
			_tmp28 = vala_parser_parse_delete_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp28;
		}
		default:
		{
			ValaStatement* _tmp30;
			_tmp30 = vala_parser_parse_expression_statement (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return NULL;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			return _tmp30;
		}
	}
}


static ValaBlock* vala_parser_parse_block (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSourceReference* _tmp0;
	ValaBlock* _tmp1;
	ValaBlock* block;
	ValaSourceReference* _tmp3;
	ValaSourceReference* _tmp4;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	block = (_tmp1 = vala_block_new (_tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	vala_parser_parse_statements (self, block, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			return NULL;
		} else {
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if (!vala_parser_accept (self, VALA_TOKEN_TYPE_CLOSE_BRACE)) {
		/* only report error if it's not a secondary error*/
		if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) == 0) {
			ValaSourceReference* _tmp2;
			_tmp2 = NULL;
			vala_report_error (_tmp2 = vala_parser_get_current_src (self), "expected `}'");
			(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL));
		}
	}
	_tmp3 = NULL;
	vala_source_reference_set_last_line (vala_code_node_get_source_reference ((ValaCodeNode*) block), vala_source_reference_get_last_line (_tmp3 = vala_parser_get_current_src (self)));
	(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL));
	_tmp4 = NULL;
	vala_source_reference_set_last_column (vala_code_node_get_source_reference ((ValaCodeNode*) block), vala_source_reference_get_last_column (_tmp4 = vala_parser_get_current_src (self)));
	(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL));
	return block;
}


static ValaStatement* vala_parser_parse_empty_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = (ValaStatement*) vala_empty_statement_new (_tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
}


static void vala_parser_parse_local_variable_declarations (ValaParser* self, ValaBlock* block, GError** error) {
	GError * inner_error;
	ValaDataType* variable_type;
	g_return_if_fail (self != NULL);
	g_return_if_fail (block != NULL);
	inner_error = NULL;
	variable_type = NULL;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_VAR)) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		variable_type = (_tmp0 = NULL, (variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL)), _tmp0);
	} else {
		ValaDataType* _tmp1;
		ValaDataType* _tmp2;
		_tmp1 = vala_parser_parse_type (self, TRUE, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL));
				return;
			} else {
				(variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return;
			}
		}
		_tmp2 = NULL;
		variable_type = (_tmp2 = _tmp1, (variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL)), _tmp2);
	}
	do {
		ValaDataType* type_copy;
		ValaLocalVariable* local;
		ValaDeclarationStatement* _tmp4;
		type_copy = NULL;
		if (variable_type != NULL) {
			ValaDataType* _tmp3;
			_tmp3 = NULL;
			type_copy = (_tmp3 = vala_data_type_copy (variable_type), (type_copy == NULL) ? NULL : (type_copy = (vala_code_node_unref (type_copy), NULL)), _tmp3);
		}
		local = vala_parser_parse_local_variable (self, type_copy, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(type_copy == NULL) ? NULL : (type_copy = (vala_code_node_unref (type_copy), NULL));
				(variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL));
				return;
			} else {
				(type_copy == NULL) ? NULL : (type_copy = (vala_code_node_unref (type_copy), NULL));
				(variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return;
			}
		}
		_tmp4 = NULL;
		vala_block_add_statement (block, (ValaStatement*) (_tmp4 = vala_declaration_statement_new ((ValaSymbol*) local, vala_code_node_get_source_reference ((ValaCodeNode*) local))));
		(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
		(type_copy == NULL) ? NULL : (type_copy = (vala_code_node_unref (type_copy), NULL));
		(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL));
			return;
		} else {
			(variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return;
		}
	}
	(variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL));
}


static ValaLocalVariable* vala_parser_parse_local_variable (ValaParser* self, ValaDataType* variable_type, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	char* id;
	ValaExpression* initializer;
	ValaSourceReference* _tmp2;
	ValaLocalVariable* _tmp3;
	ValaLocalVariable* _tmp4;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	initializer = NULL;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_ASSIGN)) {
		ValaExpression* _tmp0;
		ValaExpression* _tmp1;
		_tmp0 = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				id = (g_free (id), NULL);
				(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
				return NULL;
			} else {
				id = (g_free (id), NULL);
				(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		initializer = (_tmp1 = _tmp0, (initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)), _tmp1);
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	_tmp4 = NULL;
	return (_tmp4 = (_tmp3 = vala_local_variable_new (variable_type, id, initializer, _tmp2 = vala_parser_get_src_com (self, &begin)), (_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)), _tmp3), id = (g_free (id), NULL), (initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)), _tmp4);
}


static ValaStatement* vala_parser_parse_expression_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	ValaStatement* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	expr = vala_parser_parse_statement_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			return NULL;
		} else {
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = (ValaStatement*) vala_expression_statement_new (expr, _tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1), (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp2);
}


static ValaExpression* vala_parser_parse_statement_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaExpression* expr;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	/* invocation expression, assignment,
	 or pre/post increment/decrement expression*/
	expr = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	return expr;
}


static ValaStatement* vala_parser_parse_if_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* condition;
	ValaSourceReference* src;
	ValaBlock* true_stmt;
	ValaBlock* false_stmt;
	ValaStatement* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_IF, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	condition = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			return NULL;
		} else {
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	src = vala_parser_get_src_com (self, &begin);
	true_stmt = vala_parser_parse_embedded_statement (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			(src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL));
			return NULL;
		} else {
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			(src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	false_stmt = NULL;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_ELSE)) {
		ValaBlock* _tmp0;
		ValaBlock* _tmp1;
		_tmp0 = vala_parser_parse_embedded_statement (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
				(src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL));
				(true_stmt == NULL) ? NULL : (true_stmt = (vala_code_node_unref (true_stmt), NULL));
				(false_stmt == NULL) ? NULL : (false_stmt = (vala_code_node_unref (false_stmt), NULL));
				return NULL;
			} else {
				(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
				(src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL));
				(true_stmt == NULL) ? NULL : (true_stmt = (vala_code_node_unref (true_stmt), NULL));
				(false_stmt == NULL) ? NULL : (false_stmt = (vala_code_node_unref (false_stmt), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		false_stmt = (_tmp1 = _tmp0, (false_stmt == NULL) ? NULL : (false_stmt = (vala_code_node_unref (false_stmt), NULL)), _tmp1);
	}
	_tmp2 = NULL;
	return (_tmp2 = (ValaStatement*) vala_if_statement_new (condition, true_stmt, false_stmt, src), (condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL)), (src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL)), (true_stmt == NULL) ? NULL : (true_stmt = (vala_code_node_unref (true_stmt), NULL)), (false_stmt == NULL) ? NULL : (false_stmt = (vala_code_node_unref (false_stmt), NULL)), _tmp2);
}


static ValaStatement* vala_parser_parse_switch_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* condition;
	ValaSourceReference* _tmp0;
	ValaSwitchStatement* _tmp1;
	ValaSwitchStatement* stmt;
	ValaStatement* _tmp12;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_SWITCH, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	condition = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			return NULL;
		} else {
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	stmt = (_tmp1 = vala_switch_statement_new (condition, _tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
			return NULL;
		} else {
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	while (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE) {
		ValaSourceReference* _tmp2;
		ValaSwitchSection* _tmp3;
		ValaSwitchSection* section;
		gboolean _tmp4;
		_tmp2 = NULL;
		_tmp3 = NULL;
		section = (_tmp3 = vala_switch_section_new (_tmp2 = vala_parser_get_src_com (self, &begin)), (_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)), _tmp3);
		_tmp4 = FALSE;
		do {
			if (_tmp4) {
				gboolean _tmp5;
				_tmp5 = FALSE;
				if (vala_parser_current (self) == VALA_TOKEN_TYPE_CASE) {
					_tmp5 = TRUE;
				} else {
					_tmp5 = vala_parser_current (self) == VALA_TOKEN_TYPE_DEFAULT;
				}
				if (!_tmp5) {
					break;
				}
			}
			_tmp4 = TRUE;
			if (vala_parser_accept (self, VALA_TOKEN_TYPE_CASE)) {
				ValaExpression* _tmp6;
				ValaSwitchLabel* _tmp9;
				ValaSourceReference* _tmp8;
				ValaExpression* _tmp7;
				_tmp6 = vala_parser_parse_expression (self, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(section == NULL) ? NULL : (section = (vala_code_node_unref (section), NULL));
						(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						return NULL;
					} else {
						(section == NULL) ? NULL : (section = (vala_code_node_unref (section), NULL));
						(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp9 = NULL;
				_tmp8 = NULL;
				_tmp7 = NULL;
				vala_switch_section_add_label (section, _tmp9 = vala_switch_label_new (_tmp7 = _tmp6, _tmp8 = vala_parser_get_src_com (self, &begin)));
				(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL));
				(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_source_reference_unref (_tmp8), NULL));
				(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_code_node_unref (_tmp7), NULL));
			} else {
				ValaSwitchLabel* _tmp11;
				ValaSourceReference* _tmp10;
				vala_parser_expect (self, VALA_TOKEN_TYPE_DEFAULT, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(section == NULL) ? NULL : (section = (vala_code_node_unref (section), NULL));
						(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						return NULL;
					} else {
						(section == NULL) ? NULL : (section = (vala_code_node_unref (section), NULL));
						(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp11 = NULL;
				_tmp10 = NULL;
				vala_switch_section_add_label (section, _tmp11 = vala_switch_label_new_with_default (_tmp10 = vala_parser_get_src_com (self, &begin)));
				(_tmp11 == NULL) ? NULL : (_tmp11 = (vala_code_node_unref (_tmp11), NULL));
				(_tmp10 == NULL) ? NULL : (_tmp10 = (vala_source_reference_unref (_tmp10), NULL));
			}
			vala_parser_expect (self, VALA_TOKEN_TYPE_COLON, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(section == NULL) ? NULL : (section = (vala_code_node_unref (section), NULL));
					(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
					(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
					return NULL;
				} else {
					(section == NULL) ? NULL : (section = (vala_code_node_unref (section), NULL));
					(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
					(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
		} while (TRUE);
		vala_parser_parse_statements (self, (ValaBlock*) section, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(section == NULL) ? NULL : (section = (vala_code_node_unref (section), NULL));
				(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
				(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
				return NULL;
			} else {
				(section == NULL) ? NULL : (section = (vala_code_node_unref (section), NULL));
				(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
				(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		vala_switch_statement_add_section (stmt, section);
		(section == NULL) ? NULL : (section = (vala_code_node_unref (section), NULL));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
			return NULL;
		} else {
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp12 = NULL;
	return (_tmp12 = (ValaStatement*) stmt, (condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL)), _tmp12);
}


static ValaStatement* vala_parser_parse_while_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* condition;
	ValaBlock* body;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	ValaStatement* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_WHILE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	condition = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			return NULL;
		} else {
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	body = vala_parser_parse_embedded_statement (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			return NULL;
		} else {
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = (ValaStatement*) vala_while_statement_new (condition, body, _tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1), (condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL)), (body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL)), _tmp2);
}


static ValaStatement* vala_parser_parse_do_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaBlock* body;
	ValaExpression* condition;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	ValaStatement* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_DO, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	body = vala_parser_parse_embedded_statement (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_WHILE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL));
			return NULL;
		} else {
			(body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL));
			return NULL;
		} else {
			(body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	condition = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL));
			return NULL;
		} else {
			(body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL));
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			return NULL;
		} else {
			(body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL));
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL));
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			return NULL;
		} else {
			(body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL));
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = (ValaStatement*) vala_do_statement_new (body, condition, _tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1), (body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL)), (condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL)), _tmp2);
}


static ValaStatement* vala_parser_parse_for_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaBlock* block;
	GeeArrayList* initializer_list;
	ValaExpression* condition;
	GeeArrayList* iterator_list;
	ValaSourceReference* src;
	ValaBlock* body;
	ValaForStatement* stmt;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	block = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_FOR, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			return NULL;
		} else {
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			return NULL;
		} else {
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	initializer_list = gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	if (!vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
		gboolean is_expr;
		is_expr = FALSE;
		switch (vala_parser_current (self)) {
			case VALA_TOKEN_TYPE_VAR:
			{
				is_expr = FALSE;
				break;
			}
			case VALA_TOKEN_TYPE_OP_INC:
			case VALA_TOKEN_TYPE_OP_DEC:
			{
				is_expr = TRUE;
				break;
			}
			default:
			{
				gboolean _tmp0;
				_tmp0 = vala_parser_is_expression (self, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
						(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
						return NULL;
					} else {
						(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
						(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				is_expr = _tmp0;
				break;
			}
		}
		if (is_expr) {
			do {
				ValaExpression* _tmp1;
				ValaExpression* _tmp2;
				_tmp1 = vala_parser_parse_statement_expression (self, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
						(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
						return NULL;
					} else {
						(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
						(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp2 = NULL;
				gee_collection_add ((GeeCollection*) initializer_list, _tmp2 = _tmp1);
				(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL));
			} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		} else {
			ValaBlock* _tmp4;
			ValaSourceReference* _tmp3;
			ValaDataType* variable_type;
			ValaLocalVariable* local;
			ValaDeclarationStatement* _tmp8;
			_tmp4 = NULL;
			_tmp3 = NULL;
			block = (_tmp4 = vala_block_new (_tmp3 = vala_parser_get_src (self, &begin)), (block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL)), _tmp4);
			(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL));
			variable_type = NULL;
			if (vala_parser_accept (self, VALA_TOKEN_TYPE_VAR)) {
				ValaDataType* _tmp5;
				_tmp5 = NULL;
				variable_type = (_tmp5 = NULL, (variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL)), _tmp5);
			} else {
				ValaDataType* _tmp6;
				ValaDataType* _tmp7;
				_tmp6 = vala_parser_parse_type (self, TRUE, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL));
						(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
						(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
						return NULL;
					} else {
						(variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL));
						(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
						(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
				_tmp7 = NULL;
				variable_type = (_tmp7 = _tmp6, (variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL)), _tmp7);
			}
			local = vala_parser_parse_local_variable (self, variable_type, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL));
					(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
					(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
					return NULL;
				} else {
					(variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL));
					(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
					(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp8 = NULL;
			vala_block_add_statement (block, (ValaStatement*) (_tmp8 = vala_declaration_statement_new ((ValaSymbol*) local, vala_code_node_get_source_reference ((ValaCodeNode*) local))));
			(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL));
			(variable_type == NULL) ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL));
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
		}
		vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
				(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
				return NULL;
			} else {
				(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
				(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
	}
	condition = NULL;
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_SEMICOLON) {
		ValaExpression* _tmp9;
		ValaExpression* _tmp10;
		_tmp9 = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
				(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
				(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
				return NULL;
			} else {
				(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
				(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
				(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp10 = NULL;
		condition = (_tmp10 = _tmp9, (condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL)), _tmp10);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			return NULL;
		} else {
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	iterator_list = gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaExpression* _tmp11;
			ValaExpression* _tmp12;
			_tmp11 = vala_parser_parse_statement_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
					(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
					(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
					(iterator_list == NULL) ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL));
					return NULL;
				} else {
					(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
					(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
					(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
					(iterator_list == NULL) ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp12 = NULL;
			gee_collection_add ((GeeCollection*) iterator_list, _tmp12 = _tmp11);
			(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_code_node_unref (_tmp12), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			(iterator_list == NULL) ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL));
			return NULL;
		} else {
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			(iterator_list == NULL) ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	src = vala_parser_get_src_com (self, &begin);
	body = vala_parser_parse_embedded_statement (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			(iterator_list == NULL) ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL));
			(src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL));
			return NULL;
		} else {
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
			(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
			(iterator_list == NULL) ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL));
			(src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	stmt = vala_for_statement_new (condition, body, src);
	{
		GeeIterator* _init_it;
		_init_it = gee_iterable_iterator ((GeeIterable*) initializer_list);
		while (gee_iterator_next (_init_it)) {
			ValaExpression* init;
			init = (ValaExpression*) gee_iterator_get (_init_it);
			vala_for_statement_add_initializer (stmt, init);
			(init == NULL) ? NULL : (init = (vala_code_node_unref (init), NULL));
		}
		(_init_it == NULL) ? NULL : (_init_it = (gee_collection_object_unref (_init_it), NULL));
	}
	{
		GeeIterator* _iter_it;
		_iter_it = gee_iterable_iterator ((GeeIterable*) iterator_list);
		while (gee_iterator_next (_iter_it)) {
			ValaExpression* iter;
			iter = (ValaExpression*) gee_iterator_get (_iter_it);
			vala_for_statement_add_iterator (stmt, iter);
			(iter == NULL) ? NULL : (iter = (vala_code_node_unref (iter), NULL));
		}
		(_iter_it == NULL) ? NULL : (_iter_it = (gee_collection_object_unref (_iter_it), NULL));
	}
	if (block != NULL) {
		ValaStatement* _tmp13;
		vala_block_add_statement (block, (ValaStatement*) stmt);
		_tmp13 = NULL;
		return (_tmp13 = (ValaStatement*) block, (initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL)), (condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL)), (iterator_list == NULL) ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL)), (src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL)), (body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL)), (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp13);
	} else {
		ValaStatement* _tmp14;
		_tmp14 = NULL;
		return (_tmp14 = (ValaStatement*) stmt, (block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL)), (initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL)), (condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL)), (iterator_list == NULL) ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL)), (src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL)), (body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL)), _tmp14);
	}
	(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
	(initializer_list == NULL) ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL));
	(condition == NULL) ? NULL : (condition = (vala_code_node_unref (condition), NULL));
	(iterator_list == NULL) ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL));
	(src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL));
	(body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL));
	(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
}


static ValaStatement* vala_parser_parse_foreach_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaDataType* type;
	char* id;
	ValaExpression* collection;
	ValaSourceReference* src;
	ValaBlock* body;
	ValaStatement* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_FOREACH, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	type = NULL;
	if (!vala_parser_accept (self, VALA_TOKEN_TYPE_VAR)) {
		ValaDataType* _tmp0;
		ValaDataType* _tmp1;
		_tmp0 = vala_parser_parse_type (self, TRUE, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				return NULL;
			} else {
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		type = (_tmp1 = _tmp0, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp1);
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_IN, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	collection = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(collection == NULL) ? NULL : (collection = (vala_code_node_unref (collection), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(collection == NULL) ? NULL : (collection = (vala_code_node_unref (collection), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	src = vala_parser_get_src_com (self, &begin);
	body = vala_parser_parse_embedded_statement (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(collection == NULL) ? NULL : (collection = (vala_code_node_unref (collection), NULL));
			(src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(collection == NULL) ? NULL : (collection = (vala_code_node_unref (collection), NULL));
			(src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = (ValaStatement*) vala_foreach_statement_new (type, id, collection, body, src), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), id = (g_free (id), NULL), (collection == NULL) ? NULL : (collection = (vala_code_node_unref (collection), NULL)), (src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL)), (body == NULL) ? NULL : (body = (vala_code_node_unref (body), NULL)), _tmp2);
}


static ValaStatement* vala_parser_parse_break_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_BREAK, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = (ValaStatement*) vala_break_statement_new (_tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
}


static ValaStatement* vala_parser_parse_continue_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_CONTINUE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = (ValaStatement*) vala_continue_statement_new (_tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
}


static ValaStatement* vala_parser_parse_return_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	ValaSourceReference* _tmp2;
	ValaStatement* _tmp3;
	ValaStatement* _tmp4;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_RETURN, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	expr = NULL;
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_SEMICOLON) {
		ValaExpression* _tmp0;
		ValaExpression* _tmp1;
		_tmp0 = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
				return NULL;
			} else {
				(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		expr = (_tmp1 = _tmp0, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp1);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			return NULL;
		} else {
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	_tmp4 = NULL;
	return (_tmp4 = (_tmp3 = (ValaStatement*) vala_return_statement_new (expr, _tmp2 = vala_parser_get_src_com (self, &begin)), (_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)), _tmp3), (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp4);
}


static ValaStatement* vala_parser_parse_yield_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	ValaSourceReference* _tmp2;
	ValaStatement* _tmp3;
	ValaStatement* _tmp4;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_YIELD, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	expr = NULL;
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_SEMICOLON) {
		ValaExpression* _tmp0;
		ValaExpression* _tmp1;
		_tmp0 = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
				return NULL;
			} else {
				(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		expr = (_tmp1 = _tmp0, (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp1);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			return NULL;
		} else {
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	_tmp4 = NULL;
	return (_tmp4 = (_tmp3 = (ValaStatement*) vala_yield_statement_new (expr, _tmp2 = vala_parser_get_src (self, &begin)), (_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)), _tmp3), (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp4);
}


static ValaStatement* vala_parser_parse_throw_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	ValaStatement* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_THROW, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	expr = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			return NULL;
		} else {
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = (ValaStatement*) vala_throw_statement_new (expr, _tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1), (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp2);
}


static ValaStatement* vala_parser_parse_try_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaBlock* try_block;
	ValaBlock* finally_clause;
	GeeArrayList* catch_clauses;
	ValaSourceReference* _tmp4;
	ValaTryStatement* _tmp5;
	ValaTryStatement* stmt;
	ValaStatement* _tmp6;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_TRY, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	try_block = vala_parser_parse_block (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	finally_clause = NULL;
	catch_clauses = gee_array_list_new (VALA_TYPE_CATCH_CLAUSE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	if (vala_parser_current (self) == VALA_TOKEN_TYPE_CATCH) {
		vala_parser_parse_catch_clauses (self, (GeeList*) catch_clauses, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(try_block == NULL) ? NULL : (try_block = (vala_code_node_unref (try_block), NULL));
				(finally_clause == NULL) ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL));
				(catch_clauses == NULL) ? NULL : (catch_clauses = (gee_collection_object_unref (catch_clauses), NULL));
				return NULL;
			} else {
				(try_block == NULL) ? NULL : (try_block = (vala_code_node_unref (try_block), NULL));
				(finally_clause == NULL) ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL));
				(catch_clauses == NULL) ? NULL : (catch_clauses = (gee_collection_object_unref (catch_clauses), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		if (vala_parser_current (self) == VALA_TOKEN_TYPE_FINALLY) {
			ValaBlock* _tmp0;
			ValaBlock* _tmp1;
			_tmp0 = vala_parser_parse_finally_clause (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(try_block == NULL) ? NULL : (try_block = (vala_code_node_unref (try_block), NULL));
					(finally_clause == NULL) ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL));
					(catch_clauses == NULL) ? NULL : (catch_clauses = (gee_collection_object_unref (catch_clauses), NULL));
					return NULL;
				} else {
					(try_block == NULL) ? NULL : (try_block = (vala_code_node_unref (try_block), NULL));
					(finally_clause == NULL) ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL));
					(catch_clauses == NULL) ? NULL : (catch_clauses = (gee_collection_object_unref (catch_clauses), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp1 = NULL;
			finally_clause = (_tmp1 = _tmp0, (finally_clause == NULL) ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL)), _tmp1);
		}
	} else {
		ValaBlock* _tmp2;
		ValaBlock* _tmp3;
		_tmp2 = vala_parser_parse_finally_clause (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(try_block == NULL) ? NULL : (try_block = (vala_code_node_unref (try_block), NULL));
				(finally_clause == NULL) ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL));
				(catch_clauses == NULL) ? NULL : (catch_clauses = (gee_collection_object_unref (catch_clauses), NULL));
				return NULL;
			} else {
				(try_block == NULL) ? NULL : (try_block = (vala_code_node_unref (try_block), NULL));
				(finally_clause == NULL) ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL));
				(catch_clauses == NULL) ? NULL : (catch_clauses = (gee_collection_object_unref (catch_clauses), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp3 = NULL;
		finally_clause = (_tmp3 = _tmp2, (finally_clause == NULL) ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL)), _tmp3);
	}
	_tmp4 = NULL;
	_tmp5 = NULL;
	stmt = (_tmp5 = vala_try_statement_new (try_block, finally_clause, _tmp4 = vala_parser_get_src_com (self, &begin)), (_tmp4 == NULL) ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL)), _tmp5);
	{
		GeeIterator* _clause_it;
		_clause_it = gee_iterable_iterator ((GeeIterable*) catch_clauses);
		while (gee_iterator_next (_clause_it)) {
			ValaCatchClause* clause;
			clause = (ValaCatchClause*) gee_iterator_get (_clause_it);
			vala_try_statement_add_catch_clause (stmt, clause);
			(clause == NULL) ? NULL : (clause = (vala_code_node_unref (clause), NULL));
		}
		(_clause_it == NULL) ? NULL : (_clause_it = (gee_collection_object_unref (_clause_it), NULL));
	}
	_tmp6 = NULL;
	return (_tmp6 = (ValaStatement*) stmt, (try_block == NULL) ? NULL : (try_block = (vala_code_node_unref (try_block), NULL)), (finally_clause == NULL) ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL)), (catch_clauses == NULL) ? NULL : (catch_clauses = (gee_collection_object_unref (catch_clauses), NULL)), _tmp6);
}


static void vala_parser_parse_catch_clauses (ValaParser* self, GeeList* catch_clauses, GError** error) {
	GError * inner_error;
	g_return_if_fail (self != NULL);
	g_return_if_fail (catch_clauses != NULL);
	inner_error = NULL;
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_CATCH)) {
		ValaSourceLocation begin;
		ValaDataType* type;
		char* id;
		ValaBlock* block;
		ValaCatchClause* _tmp5;
		ValaSourceReference* _tmp4;
		begin = vala_parser_get_location (self);
		type = NULL;
		id = NULL;
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_PARENS)) {
			ValaDataType* _tmp0;
			ValaDataType* _tmp1;
			char* _tmp2;
			char* _tmp3;
			_tmp0 = vala_parser_parse_type (self, TRUE, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					return;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return;
				}
			}
			_tmp1 = NULL;
			type = (_tmp1 = _tmp0, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp1);
			_tmp2 = vala_parser_parse_identifier (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					return;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return;
				}
			}
			_tmp3 = NULL;
			id = (_tmp3 = _tmp2, id = (g_free (id), NULL), _tmp3);
			vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					return;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return;
				}
			}
		}
		block = vala_parser_parse_block (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				return;
			} else {
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return;
			}
		}
		_tmp5 = NULL;
		_tmp4 = NULL;
		gee_collection_add ((GeeCollection*) catch_clauses, _tmp5 = vala_catch_clause_new (type, id, block, _tmp4 = vala_parser_get_src (self, &begin)));
		(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
		(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL));
		(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		id = (g_free (id), NULL);
		(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
	}
}


static ValaBlock* vala_parser_parse_finally_clause (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaBlock* block;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_FINALLY, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	block = vala_parser_parse_block (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	return block;
}


static ValaStatement* vala_parser_parse_lock_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	ValaBlock* stmt;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	ValaStatement* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_LOCK, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	expr = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			return NULL;
		} else {
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	stmt = vala_parser_parse_embedded_statement (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			return NULL;
		} else {
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = (ValaStatement*) vala_lock_statement_new (expr, stmt, _tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1), (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), _tmp2);
}


static ValaStatement* vala_parser_parse_delete_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	ValaStatement* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_DELETE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	expr = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			return NULL;
		} else {
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = (ValaStatement*) vala_delete_statement_new (expr, _tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1), (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp2);
}


static GeeList* vala_parser_parse_attributes (ValaParser* self, GError** error) {
	GError * inner_error;
	GeeArrayList* attrs;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_OPEN_BRACKET) {
		return NULL;
	}
	attrs = gee_array_list_new (VALA_TYPE_ATTRIBUTE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_BRACKET)) {
		do {
			ValaSourceLocation begin;
			char* id;
			ValaSourceReference* _tmp1;
			ValaAttribute* _tmp2;
			ValaAttribute* attr;
			begin = vala_parser_get_location (self);
			id = vala_parser_parse_identifier (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
					return NULL;
				} else {
					(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp1 = NULL;
			_tmp2 = NULL;
			attr = (_tmp2 = vala_attribute_new (id, _tmp1 = vala_parser_get_src (self, &begin)), (_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL)), _tmp2);
			if (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_PARENS)) {
				if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
					do {
						char* _tmp3;
						char* _tmp4;
						ValaExpression* expr;
						_tmp3 = vala_parser_parse_identifier (self, &inner_error);
						if (inner_error != NULL) {
							if (inner_error->domain == VALA_PARSE_ERROR) {
								g_propagate_error (error, inner_error);
								id = (g_free (id), NULL);
								(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
								(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
								return NULL;
							} else {
								id = (g_free (id), NULL);
								(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
								(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
								g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
								g_clear_error (&inner_error);
								return NULL;
							}
						}
						_tmp4 = NULL;
						id = (_tmp4 = _tmp3, id = (g_free (id), NULL), _tmp4);
						vala_parser_expect (self, VALA_TOKEN_TYPE_ASSIGN, &inner_error);
						if (inner_error != NULL) {
							if (inner_error->domain == VALA_PARSE_ERROR) {
								g_propagate_error (error, inner_error);
								id = (g_free (id), NULL);
								(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
								(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
								return NULL;
							} else {
								id = (g_free (id), NULL);
								(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
								(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
								g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
								g_clear_error (&inner_error);
								return NULL;
							}
						}
						expr = vala_parser_parse_expression (self, &inner_error);
						if (inner_error != NULL) {
							if (inner_error->domain == VALA_PARSE_ERROR) {
								g_propagate_error (error, inner_error);
								id = (g_free (id), NULL);
								(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
								(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
								return NULL;
							} else {
								id = (g_free (id), NULL);
								(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
								(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
								g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
								g_clear_error (&inner_error);
								return NULL;
							}
						}
						vala_attribute_add_argument (attr, id, expr);
						(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
				}
				vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						id = (g_free (id), NULL);
						(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
						(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
						return NULL;
					} else {
						id = (g_free (id), NULL);
						(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
						(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
			}
			gee_collection_add ((GeeCollection*) attrs, attr);
			id = (g_free (id), NULL);
			(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACKET, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
				return NULL;
			} else {
				(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
	}
	return (GeeList*) attrs;
}


static void vala_parser_set_attributes (ValaParser* self, ValaCodeNode* node, GeeList* attributes) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	if (attributes != NULL) {
		{
			GeeIterator* _attr_it;
			_attr_it = gee_iterable_iterator ((GeeIterable*) GEE_LIST (attributes));
			while (gee_iterator_next (_attr_it)) {
				ValaAttribute* attr;
				ValaAttribute* _tmp0;
				attr = (ValaAttribute*) gee_iterator_get (_attr_it);
				_tmp0 = NULL;
				node->attributes = g_list_append (node->attributes, (_tmp0 = attr, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0)));
				(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
			}
			(_attr_it == NULL) ? NULL : (_attr_it = (gee_collection_object_unref (_attr_it), NULL));
		}
	}
}


static ValaSymbol* vala_parser_parse_declaration (ValaParser* self, GError** error) {
	GError * inner_error;
	char* _tmp0;
	GeeList* attrs;
	ValaSourceLocation begin;
	ValaTokenType last_keyword;
	char* _tmp31;
	GError* _tmp32;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	_tmp0 = NULL;
	self->priv->comment = (_tmp0 = vala_scanner_pop_comment (self->priv->scanner), self->priv->comment = (g_free (self->priv->comment), NULL), _tmp0);
	attrs = vala_parser_parse_attributes (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	begin = vala_parser_get_location (self);
	last_keyword = vala_parser_current (self);
	while (vala_parser_is_declaration_keyword (self, vala_parser_current (self))) {
		last_keyword = vala_parser_current (self);
		vala_parser_next (self);
	}
	switch (vala_parser_current (self)) {
		case VALA_TOKEN_TYPE_CONSTRUCT:
		{
			ValaConstructor* _tmp1;
			ValaSymbol* _tmp2;
			vala_parser_rollback (self, &begin);
			_tmp1 = vala_parser_parse_constructor_declaration (self, attrs, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
					return NULL;
				} else {
					(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp2 = NULL;
			return (_tmp2 = (ValaSymbol*) _tmp1, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp2);
		}
		case VALA_TOKEN_TYPE_TILDE:
		{
			ValaDestructor* _tmp3;
			ValaSymbol* _tmp4;
			vala_parser_rollback (self, &begin);
			_tmp3 = vala_parser_parse_destructor_declaration (self, attrs, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
					return NULL;
				} else {
					(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp4 = NULL;
			return (_tmp4 = (ValaSymbol*) _tmp3, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp4);
		}
		default:
		{
			vala_parser_skip_type (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
					return NULL;
				} else {
					(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			switch (vala_parser_current (self)) {
				case VALA_TOKEN_TYPE_OPEN_BRACE:
				case VALA_TOKEN_TYPE_SEMICOLON:
				case VALA_TOKEN_TYPE_COLON:
				{
					vala_parser_rollback (self, &begin);
					switch (last_keyword) {
						case VALA_TOKEN_TYPE_CLASS:
						{
							ValaSymbol* _tmp5;
							ValaSymbol* _tmp6;
							_tmp5 = vala_parser_parse_class_declaration (self, attrs, &inner_error);
							if (inner_error != NULL) {
								if (inner_error->domain == VALA_PARSE_ERROR) {
									g_propagate_error (error, inner_error);
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									return NULL;
								} else {
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
									g_clear_error (&inner_error);
									return NULL;
								}
							}
							_tmp6 = NULL;
							return (_tmp6 = _tmp5, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp6);
						}
						case VALA_TOKEN_TYPE_ENUM:
						{
							ValaSymbol* _tmp7;
							ValaSymbol* _tmp8;
							_tmp7 = vala_parser_parse_enum_declaration (self, attrs, &inner_error);
							if (inner_error != NULL) {
								if (inner_error->domain == VALA_PARSE_ERROR) {
									g_propagate_error (error, inner_error);
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									return NULL;
								} else {
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
									g_clear_error (&inner_error);
									return NULL;
								}
							}
							_tmp8 = NULL;
							return (_tmp8 = _tmp7, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp8);
						}
						case VALA_TOKEN_TYPE_ERRORDOMAIN:
						{
							ValaSymbol* _tmp9;
							ValaSymbol* _tmp10;
							_tmp9 = vala_parser_parse_errordomain_declaration (self, attrs, &inner_error);
							if (inner_error != NULL) {
								if (inner_error->domain == VALA_PARSE_ERROR) {
									g_propagate_error (error, inner_error);
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									return NULL;
								} else {
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
									g_clear_error (&inner_error);
									return NULL;
								}
							}
							_tmp10 = NULL;
							return (_tmp10 = _tmp9, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp10);
						}
						case VALA_TOKEN_TYPE_INTERFACE:
						{
							ValaSymbol* _tmp11;
							ValaSymbol* _tmp12;
							_tmp11 = vala_parser_parse_interface_declaration (self, attrs, &inner_error);
							if (inner_error != NULL) {
								if (inner_error->domain == VALA_PARSE_ERROR) {
									g_propagate_error (error, inner_error);
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									return NULL;
								} else {
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
									g_clear_error (&inner_error);
									return NULL;
								}
							}
							_tmp12 = NULL;
							return (_tmp12 = _tmp11, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp12);
						}
						case VALA_TOKEN_TYPE_NAMESPACE:
						{
							ValaNamespace* _tmp13;
							ValaSymbol* _tmp14;
							_tmp13 = vala_parser_parse_namespace_declaration (self, attrs, &inner_error);
							if (inner_error != NULL) {
								if (inner_error->domain == VALA_PARSE_ERROR) {
									g_propagate_error (error, inner_error);
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									return NULL;
								} else {
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
									g_clear_error (&inner_error);
									return NULL;
								}
							}
							_tmp14 = NULL;
							return (_tmp14 = (ValaSymbol*) _tmp13, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp14);
						}
						case VALA_TOKEN_TYPE_STRUCT:
						{
							ValaSymbol* _tmp15;
							ValaSymbol* _tmp16;
							_tmp15 = vala_parser_parse_struct_declaration (self, attrs, &inner_error);
							if (inner_error != NULL) {
								if (inner_error->domain == VALA_PARSE_ERROR) {
									g_propagate_error (error, inner_error);
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									return NULL;
								} else {
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
									g_clear_error (&inner_error);
									return NULL;
								}
							}
							_tmp16 = NULL;
							return (_tmp16 = _tmp15, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp16);
						}
						default:
						{
							break;
						}
					}
					break;
				}
				case VALA_TOKEN_TYPE_OPEN_PARENS:
				{
					ValaCreationMethod* _tmp17;
					ValaSymbol* _tmp18;
					vala_parser_rollback (self, &begin);
					_tmp17 = vala_parser_parse_creation_method_declaration (self, attrs, &inner_error);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							g_propagate_error (error, inner_error);
							(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
							return NULL;
						} else {
							(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
							g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
							g_clear_error (&inner_error);
							return NULL;
						}
					}
					_tmp18 = NULL;
					return (_tmp18 = (ValaSymbol*) _tmp17, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp18);
				}
				default:
				{
					vala_parser_skip_type (self, &inner_error);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							g_propagate_error (error, inner_error);
							(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
							return NULL;
						} else {
							(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
							g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
							g_clear_error (&inner_error);
							return NULL;
						}
					}
					switch (vala_parser_current (self)) {
						case VALA_TOKEN_TYPE_OPEN_PARENS:
						{
							vala_parser_rollback (self, &begin);
							switch (last_keyword) {
								case VALA_TOKEN_TYPE_DELEGATE:
								{
									ValaSymbol* _tmp19;
									ValaSymbol* _tmp20;
									_tmp19 = vala_parser_parse_delegate_declaration (self, attrs, &inner_error);
									if (inner_error != NULL) {
										if (inner_error->domain == VALA_PARSE_ERROR) {
											g_propagate_error (error, inner_error);
											(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
											return NULL;
										} else {
											(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
											g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
											g_clear_error (&inner_error);
											return NULL;
										}
									}
									_tmp20 = NULL;
									return (_tmp20 = _tmp19, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp20);
								}
								case VALA_TOKEN_TYPE_SIGNAL:
								{
									ValaSignal* _tmp21;
									ValaSymbol* _tmp22;
									_tmp21 = vala_parser_parse_signal_declaration (self, attrs, &inner_error);
									if (inner_error != NULL) {
										if (inner_error->domain == VALA_PARSE_ERROR) {
											g_propagate_error (error, inner_error);
											(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
											return NULL;
										} else {
											(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
											g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
											g_clear_error (&inner_error);
											return NULL;
										}
									}
									_tmp22 = NULL;
									return (_tmp22 = (ValaSymbol*) _tmp21, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp22);
								}
								default:
								{
									ValaMethod* _tmp23;
									ValaSymbol* _tmp24;
									_tmp23 = vala_parser_parse_method_declaration (self, attrs, &inner_error);
									if (inner_error != NULL) {
										if (inner_error->domain == VALA_PARSE_ERROR) {
											g_propagate_error (error, inner_error);
											(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
											return NULL;
										} else {
											(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
											g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
											g_clear_error (&inner_error);
											return NULL;
										}
									}
									_tmp24 = NULL;
									return (_tmp24 = (ValaSymbol*) _tmp23, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp24);
								}
							}
						}
						case VALA_TOKEN_TYPE_ASSIGN:
						case VALA_TOKEN_TYPE_SEMICOLON:
						{
							vala_parser_rollback (self, &begin);
							switch (last_keyword) {
								case VALA_TOKEN_TYPE_CONST:
								{
									ValaConstant* _tmp25;
									ValaSymbol* _tmp26;
									_tmp25 = vala_parser_parse_constant_declaration (self, attrs, &inner_error);
									if (inner_error != NULL) {
										if (inner_error->domain == VALA_PARSE_ERROR) {
											g_propagate_error (error, inner_error);
											(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
											return NULL;
										} else {
											(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
											g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
											g_clear_error (&inner_error);
											return NULL;
										}
									}
									_tmp26 = NULL;
									return (_tmp26 = (ValaSymbol*) _tmp25, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp26);
								}
								default:
								{
									ValaField* _tmp27;
									ValaSymbol* _tmp28;
									_tmp27 = vala_parser_parse_field_declaration (self, attrs, &inner_error);
									if (inner_error != NULL) {
										if (inner_error->domain == VALA_PARSE_ERROR) {
											g_propagate_error (error, inner_error);
											(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
											return NULL;
										} else {
											(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
											g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
											g_clear_error (&inner_error);
											return NULL;
										}
									}
									_tmp28 = NULL;
									return (_tmp28 = (ValaSymbol*) _tmp27, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp28);
								}
							}
						}
						case VALA_TOKEN_TYPE_OPEN_BRACE:
						{
							ValaProperty* _tmp29;
							ValaSymbol* _tmp30;
							vala_parser_rollback (self, &begin);
							_tmp29 = vala_parser_parse_property_declaration (self, attrs, &inner_error);
							if (inner_error != NULL) {
								if (inner_error->domain == VALA_PARSE_ERROR) {
									g_propagate_error (error, inner_error);
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									return NULL;
								} else {
									(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
									g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
									g_clear_error (&inner_error);
									return NULL;
								}
							}
							_tmp30 = NULL;
							return (_tmp30 = (ValaSymbol*) _tmp29, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp30);
						}
						default:
						{
							break;
						}
					}
					break;
				}
			}
			break;
		}
	}
	vala_parser_rollback (self, &begin);
	_tmp31 = NULL;
	_tmp32 = NULL;
	inner_error = (_tmp32 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp31 = vala_parser_get_error (self, "expected declaration")), _tmp31 = (g_free (_tmp31), NULL), _tmp32);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
			return NULL;
		} else {
			(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
}


static void vala_parser_parse_declarations (ValaParser* self, ValaSymbol* parent, gboolean root, GError** error) {
	GError * inner_error;
	g_return_if_fail (self != NULL);
	g_return_if_fail (parent != NULL);
	inner_error = NULL;
	if (!root) {
		vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				return;
			} else {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return;
			}
		}
	}
	while (TRUE) {
		gboolean _tmp0;
		_tmp0 = FALSE;
		if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE) {
			_tmp0 = vala_parser_current (self) != VALA_TOKEN_TYPE_EOF;
		} else {
			_tmp0 = FALSE;
		}
		if (!_tmp0) {
			break;
		}
		{
			if (VALA_IS_NAMESPACE (parent)) {
				vala_parser_parse_namespace_member (self, VALA_NAMESPACE (parent), &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch5_vala_parse_error;
					}
					goto __finally5;
				}
			} else {
				if (VALA_IS_CLASS (parent)) {
					vala_parser_parse_class_member (self, VALA_CLASS (parent), &inner_error);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch5_vala_parse_error;
						}
						goto __finally5;
					}
				} else {
					if (VALA_IS_STRUCT (parent)) {
						vala_parser_parse_struct_member (self, VALA_STRUCT (parent), &inner_error);
						if (inner_error != NULL) {
							if (inner_error->domain == VALA_PARSE_ERROR) {
								goto __catch5_vala_parse_error;
							}
							goto __finally5;
						}
					} else {
						if (VALA_IS_INTERFACE (parent)) {
							vala_parser_parse_interface_member (self, VALA_INTERFACE (parent), &inner_error);
							if (inner_error != NULL) {
								if (inner_error->domain == VALA_PARSE_ERROR) {
									goto __catch5_vala_parse_error;
								}
								goto __finally5;
							}
						}
					}
				}
			}
		}
		goto __finally5;
		__catch5_vala_parse_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				gint r;
				r = 0;
				do {
					r = (gint) vala_parser_recover (self);
					if (r == VALA_PARSER_RECOVERY_STATE_STATEMENT_BEGIN) {
						vala_parser_next (self);
					} else {
						break;
					}
				} while (TRUE);
				if (r == VALA_PARSER_RECOVERY_STATE_EOF) {
					(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
					return;
				}
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally5:
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				return;
			} else {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return;
			}
		}
	}
	if (!root) {
		if (!vala_parser_accept (self, VALA_TOKEN_TYPE_CLOSE_BRACE)) {
			/* only report error if it's not a secondary error*/
			if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) == 0) {
				ValaSourceReference* _tmp1;
				_tmp1 = NULL;
				vala_report_error (_tmp1 = vala_parser_get_current_src (self), "expected `}'");
				(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL));
			}
		}
	}
}


static ValaParserRecoveryState vala_parser_recover (ValaParser* self) {
	g_return_val_if_fail (self != NULL, 0);
	while (vala_parser_current (self) != VALA_TOKEN_TYPE_EOF) {
		switch (vala_parser_current (self)) {
			case VALA_TOKEN_TYPE_ABSTRACT:
			case VALA_TOKEN_TYPE_CLASS:
			case VALA_TOKEN_TYPE_CONST:
			case VALA_TOKEN_TYPE_CONSTRUCT:
			case VALA_TOKEN_TYPE_DELEGATE:
			case VALA_TOKEN_TYPE_ENUM:
			case VALA_TOKEN_TYPE_ERRORDOMAIN:
			case VALA_TOKEN_TYPE_EXTERN:
			case VALA_TOKEN_TYPE_INLINE:
			case VALA_TOKEN_TYPE_INTERFACE:
			case VALA_TOKEN_TYPE_INTERNAL:
			case VALA_TOKEN_TYPE_NAMESPACE:
			case VALA_TOKEN_TYPE_NEW:
			case VALA_TOKEN_TYPE_OVERRIDE:
			case VALA_TOKEN_TYPE_PRIVATE:
			case VALA_TOKEN_TYPE_PROTECTED:
			case VALA_TOKEN_TYPE_PUBLIC:
			case VALA_TOKEN_TYPE_SIGNAL:
			case VALA_TOKEN_TYPE_STATIC:
			case VALA_TOKEN_TYPE_STRUCT:
			case VALA_TOKEN_TYPE_VIRTUAL:
			case VALA_TOKEN_TYPE_VOLATILE:
			{
				return VALA_PARSER_RECOVERY_STATE_DECLARATION_BEGIN;
			}
			case VALA_TOKEN_TYPE_BREAK:
			case VALA_TOKEN_TYPE_CONTINUE:
			case VALA_TOKEN_TYPE_DELETE:
			case VALA_TOKEN_TYPE_DO:
			case VALA_TOKEN_TYPE_FOR:
			case VALA_TOKEN_TYPE_FOREACH:
			case VALA_TOKEN_TYPE_IF:
			case VALA_TOKEN_TYPE_LOCK:
			case VALA_TOKEN_TYPE_RETURN:
			case VALA_TOKEN_TYPE_SWITCH:
			case VALA_TOKEN_TYPE_THROW:
			case VALA_TOKEN_TYPE_TRY:
			case VALA_TOKEN_TYPE_VAR:
			case VALA_TOKEN_TYPE_WHILE:
			case VALA_TOKEN_TYPE_YIELD:
			{
				return VALA_PARSER_RECOVERY_STATE_STATEMENT_BEGIN;
			}
			default:
			{
				vala_parser_next (self);
				break;
			}
		}
	}
	return VALA_PARSER_RECOVERY_STATE_EOF;
}


static ValaNamespace* vala_parser_parse_namespace_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaUnresolvedSymbol* sym;
	ValaSourceReference* _tmp0;
	ValaNamespace* _tmp1;
	ValaNamespace* ns;
	ValaNamespace* _tmp2;
	ValaNamespace* result;
	ValaNamespace* _tmp8;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_NAMESPACE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	ns = (_tmp1 = vala_namespace_new (vala_symbol_get_name ((ValaSymbol*) sym), _tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	vala_parser_set_attributes (self, (ValaCodeNode*) ns, attrs);
	vala_parser_parse_declarations (self, (ValaSymbol*) ns, FALSE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp2 = NULL;
	result = (_tmp2 = ns, (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp4;
		ValaUnresolvedSymbol* _tmp3;
		ValaNamespace* _tmp5;
		ValaNamespace* _tmp7;
		ValaNamespace* _tmp6;
		_tmp4 = NULL;
		_tmp3 = NULL;
		sym = (_tmp4 = (_tmp3 = vala_unresolved_symbol_get_inner (sym), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp4);
		_tmp5 = NULL;
		ns = (_tmp5 = vala_namespace_new (vala_symbol_get_name ((ValaSymbol*) sym), vala_code_node_get_source_reference ((ValaCodeNode*) result)), (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), _tmp5);
		vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		_tmp7 = NULL;
		_tmp6 = NULL;
		result = (_tmp7 = (_tmp6 = ns, (_tmp6 == NULL) ? NULL : vala_code_node_ref (_tmp6)), (result == NULL) ? NULL : (result = (vala_code_node_unref (result), NULL)), _tmp7);
	}
	_tmp8 = NULL;
	return (_tmp8 = result, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), _tmp8);
}


static void vala_parser_parse_namespace_member (ValaParser* self, ValaNamespace* ns, GError** error) {
	GError * inner_error;
	ValaSymbol* sym;
	g_return_if_fail (self != NULL);
	g_return_if_fail (ns != NULL);
	inner_error = NULL;
	sym = vala_parser_parse_declaration (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return;
		}
	}
	if (VALA_IS_NAMESPACE (sym)) {
		vala_namespace_add_namespace (ns, VALA_NAMESPACE (sym));
	} else {
		if (VALA_IS_CLASS (sym)) {
			vala_namespace_add_class (ns, VALA_CLASS (sym));
		} else {
			if (VALA_IS_INTERFACE (sym)) {
				vala_namespace_add_interface (ns, VALA_INTERFACE (sym));
			} else {
				if (VALA_IS_STRUCT (sym)) {
					vala_namespace_add_struct (ns, VALA_STRUCT (sym));
				} else {
					if (VALA_IS_ENUM (sym)) {
						vala_namespace_add_enum (ns, VALA_ENUM (sym));
					} else {
						if (VALA_IS_ERROR_DOMAIN (sym)) {
							vala_namespace_add_error_domain (ns, VALA_ERROR_DOMAIN (sym));
						} else {
							if (VALA_IS_DELEGATE (sym)) {
								vala_namespace_add_delegate (ns, VALA_DELEGATE (sym));
							} else {
								if (VALA_IS_METHOD (sym)) {
									ValaMethod* _tmp0;
									ValaMethod* method;
									_tmp0 = NULL;
									method = (_tmp0 = VALA_METHOD (sym), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
									if (vala_method_get_binding (method) == MEMBER_BINDING_INSTANCE) {
										/* default to static member binding*/
										vala_method_set_binding (method, MEMBER_BINDING_STATIC);
									}
									vala_namespace_add_method (ns, method);
									(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
								} else {
									if (VALA_IS_FIELD (sym)) {
										ValaField* _tmp1;
										ValaField* field;
										_tmp1 = NULL;
										field = (_tmp1 = VALA_FIELD (sym), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
										if (vala_field_get_binding (field) == MEMBER_BINDING_INSTANCE) {
											/* default to static member binding*/
											vala_field_set_binding (field, MEMBER_BINDING_STATIC);
										}
										vala_namespace_add_field (ns, field);
										(field == NULL) ? NULL : (field = (vala_code_node_unref (field), NULL));
									} else {
										if (VALA_IS_CONSTANT (sym)) {
											vala_namespace_add_constant (ns, VALA_CONSTANT (sym));
										} else {
											vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) sym), "unexpected declaration in namespace");
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), (ValaCodeNode*) sym);
	(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
}


static void vala_parser_parse_using_directives (ValaParser* self, GError** error) {
	GError * inner_error;
	g_return_if_fail (self != NULL);
	inner_error = NULL;
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_USING)) {
		do {
			ValaSourceLocation begin;
			ValaUnresolvedSymbol* sym;
			ValaSourceReference* _tmp0;
			ValaUsingDirective* _tmp1;
			ValaUsingDirective* ns_ref;
			begin = vala_parser_get_location (self);
			sym = vala_parser_parse_symbol_name (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return;
				}
			}
			_tmp0 = NULL;
			_tmp1 = NULL;
			ns_ref = (_tmp1 = vala_using_directive_new ((ValaSymbol*) sym, _tmp0 = vala_parser_get_src (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
			vala_source_file_add_using_directive (vala_scanner_get_source_file (self->priv->scanner), ns_ref);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(ns_ref == NULL) ? NULL : (ns_ref = (vala_code_node_unref (ns_ref), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				return;
			} else {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return;
			}
		}
	}
}


static ValaSymbol* vala_parser_parse_class_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaUnresolvedSymbol* sym;
	GeeList* type_param_list;
	GeeArrayList* base_types;
	ValaSourceReference* _tmp2;
	ValaClass* _tmp3;
	ValaClass* cl;
	gboolean _tmp4;
	gboolean _tmp5;
	gboolean _tmp6;
	ValaSymbol* _tmp8;
	ValaSymbol* result;
	ValaSymbol* _tmp13;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_type_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLASS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	type_param_list = vala_parser_parse_type_parameter_list (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	base_types = gee_array_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_COLON)) {
		do {
			ValaDataType* _tmp0;
			ValaDataType* _tmp1;
			_tmp0 = vala_parser_parse_type (self, TRUE, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
					(base_types == NULL) ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL));
					return NULL;
				} else {
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
					(base_types == NULL) ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp1 = NULL;
			gee_collection_add ((GeeCollection*) base_types, _tmp1 = _tmp0);
			(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	cl = (_tmp3 = vala_class_new (vala_symbol_get_name ((ValaSymbol*) sym), _tmp2 = vala_parser_get_src_com (self, &begin)), (_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)), _tmp3);
	vala_symbol_set_access ((ValaSymbol*) cl, access);
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_ABSTRACT) == VALA_PARSER_MODIFIER_FLAGS_ABSTRACT) {
		vala_class_set_is_abstract (cl, TRUE);
	}
	_tmp4 = FALSE;
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_EXTERN) == VALA_PARSER_MODIFIER_FLAGS_EXTERN) {
		_tmp4 = TRUE;
	} else {
		_tmp4 = vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner));
	}
	if (_tmp4) {
		vala_symbol_set_external ((ValaSymbol*) cl, TRUE);
	}
	vala_parser_set_attributes (self, (ValaCodeNode*) cl, attrs);
	{
		GeeIterator* _type_param_it;
		_type_param_it = gee_iterable_iterator ((GeeIterable*) type_param_list);
		while (gee_iterator_next (_type_param_it)) {
			ValaTypeParameter* type_param;
			type_param = (ValaTypeParameter*) gee_iterator_get (_type_param_it);
			vala_object_type_symbol_add_type_parameter ((ValaObjectTypeSymbol*) cl, type_param);
			(type_param == NULL) ? NULL : (type_param = (vala_code_node_unref (type_param), NULL));
		}
		(_type_param_it == NULL) ? NULL : (_type_param_it = (gee_collection_object_unref (_type_param_it), NULL));
	}
	{
		GeeIterator* _base_type_it;
		_base_type_it = gee_iterable_iterator ((GeeIterable*) base_types);
		while (gee_iterator_next (_base_type_it)) {
			ValaDataType* base_type;
			base_type = (ValaDataType*) gee_iterator_get (_base_type_it);
			vala_class_add_base_type (cl, base_type);
			(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
		}
		(_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL));
	}
	vala_parser_parse_declarations (self, (ValaSymbol*) cl, FALSE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
			(base_types == NULL) ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL));
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
			(base_types == NULL) ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL));
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp5 = FALSE;
	_tmp6 = FALSE;
	if (!vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
		_tmp6 = !vala_class_get_is_abstract (cl);
	} else {
		_tmp6 = FALSE;
	}
	if (_tmp6) {
		_tmp5 = vala_class_get_default_construction_method (cl) == NULL;
	} else {
		_tmp5 = FALSE;
	}
	/* ensure there is always a default construction method*/
	if (_tmp5) {
		ValaCreationMethod* m;
		ValaBlock* _tmp7;
		m = vala_creation_method_new (vala_symbol_get_name ((ValaSymbol*) cl), NULL, vala_code_node_get_source_reference ((ValaCodeNode*) cl));
		vala_symbol_set_access ((ValaSymbol*) m, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
		_tmp7 = NULL;
		vala_method_set_body ((ValaMethod*) m, _tmp7 = vala_block_new (vala_code_node_get_source_reference ((ValaCodeNode*) cl)));
		(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_code_node_unref (_tmp7), NULL));
		vala_class_add_method (cl, (ValaMethod*) m);
		(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
	}
	_tmp8 = NULL;
	result = (_tmp8 = (ValaSymbol*) cl, (_tmp8 == NULL) ? NULL : vala_code_node_ref (_tmp8));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp10;
		ValaUnresolvedSymbol* _tmp9;
		ValaNamespace* ns;
		ValaSymbol* _tmp12;
		ValaSymbol* _tmp11;
		_tmp10 = NULL;
		_tmp9 = NULL;
		sym = (_tmp10 = (_tmp9 = vala_unresolved_symbol_get_inner (sym), (_tmp9 == NULL) ? NULL : vala_code_node_ref (_tmp9)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp10);
		ns = vala_namespace_new (vala_symbol_get_name ((ValaSymbol*) sym), vala_code_node_get_source_reference ((ValaCodeNode*) cl));
		if (VALA_IS_NAMESPACE (result)) {
			vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		} else {
			vala_namespace_add_class (ns, VALA_CLASS (result));
			vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), (ValaCodeNode*) result);
		}
		_tmp12 = NULL;
		_tmp11 = NULL;
		result = (_tmp12 = (_tmp11 = (ValaSymbol*) ns, (_tmp11 == NULL) ? NULL : vala_code_node_ref (_tmp11)), (result == NULL) ? NULL : (result = (vala_code_node_unref (result), NULL)), _tmp12);
		(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
	}
	_tmp13 = NULL;
	return (_tmp13 = result, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL)), (base_types == NULL) ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL)), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp13);
}


static void vala_parser_parse_class_member (ValaParser* self, ValaClass* cl, GError** error) {
	GError * inner_error;
	ValaSymbol* sym;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	inner_error = NULL;
	sym = vala_parser_parse_declaration (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return;
		}
	}
	if (VALA_IS_CLASS (sym)) {
		vala_class_add_class (cl, VALA_CLASS (sym));
	} else {
		if (VALA_IS_STRUCT (sym)) {
			vala_class_add_struct (cl, VALA_STRUCT (sym));
		} else {
			if (VALA_IS_ENUM (sym)) {
				vala_class_add_enum (cl, VALA_ENUM (sym));
			} else {
				if (VALA_IS_DELEGATE (sym)) {
					vala_class_add_delegate (cl, VALA_DELEGATE (sym));
				} else {
					if (VALA_IS_METHOD (sym)) {
						vala_class_add_method (cl, VALA_METHOD (sym));
					} else {
						if (VALA_IS_SIGNAL (sym)) {
							vala_class_add_signal (cl, VALA_SIGNAL (sym));
						} else {
							if (VALA_IS_FIELD (sym)) {
								vala_class_add_field (cl, VALA_FIELD (sym));
							} else {
								if (VALA_IS_CONSTANT (sym)) {
									vala_class_add_constant (cl, VALA_CONSTANT (sym));
								} else {
									if (VALA_IS_PROPERTY (sym)) {
										vala_class_add_property (cl, VALA_PROPERTY (sym));
									} else {
										if (VALA_IS_CONSTRUCTOR (sym)) {
											ValaConstructor* _tmp0;
											ValaConstructor* c;
											_tmp0 = NULL;
											c = (_tmp0 = VALA_CONSTRUCTOR (sym), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
											if (vala_constructor_get_binding (c) == MEMBER_BINDING_INSTANCE) {
												vala_class_set_constructor (cl, c);
											} else {
												if (vala_constructor_get_binding (c) == MEMBER_BINDING_CLASS) {
													vala_class_set_class_constructor (cl, c);
												} else {
													vala_class_set_static_constructor (cl, c);
												}
											}
											(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
										} else {
											if (VALA_IS_DESTRUCTOR (sym)) {
												ValaDestructor* _tmp1;
												ValaDestructor* d;
												_tmp1 = NULL;
												d = (_tmp1 = VALA_DESTRUCTOR (sym), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
												if (vala_destructor_get_binding (d) == MEMBER_BINDING_STATIC) {
													vala_class_set_static_destructor (cl, VALA_DESTRUCTOR (d));
												} else {
													if (vala_destructor_get_binding (d) == MEMBER_BINDING_CLASS) {
														vala_class_set_class_destructor (cl, VALA_DESTRUCTOR (d));
													} else {
														vala_class_set_destructor (cl, VALA_DESTRUCTOR (d));
													}
												}
												(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
											} else {
												vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) sym), "unexpected declaration in class");
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
}


static ValaConstant* vala_parser_parse_constant_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaDataType* type;
	char* id;
	ValaExpression* initializer;
	ValaArrayType* _tmp3;
	ValaDataType* _tmp2;
	ValaArrayType* array_type;
	ValaSourceReference* _tmp4;
	ValaConstant* _tmp5;
	ValaConstant* c;
	gboolean _tmp6;
	ValaConstant* _tmp7;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_CONST, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	type = vala_parser_parse_type (self, FALSE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	initializer = NULL;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_ASSIGN)) {
		ValaExpression* _tmp0;
		ValaExpression* _tmp1;
		_tmp0 = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
				return NULL;
			} else {
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		initializer = (_tmp1 = _tmp0, (initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)), _tmp1);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	/* constant arrays don't own their element*/
	_tmp3 = NULL;
	_tmp2 = NULL;
	array_type = (_tmp3 = (_tmp2 = type, VALA_IS_ARRAY_TYPE (_tmp2) ? ((ValaArrayType*) _tmp2) : NULL), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
	if (array_type != NULL) {
		vala_data_type_set_value_owned (vala_array_type_get_element_type (array_type), FALSE);
	}
	_tmp4 = NULL;
	_tmp5 = NULL;
	c = (_tmp5 = vala_constant_new (id, type, initializer, _tmp4 = vala_parser_get_src_com (self, &begin)), (_tmp4 == NULL) ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL)), _tmp5);
	vala_symbol_set_access ((ValaSymbol*) c, access);
	_tmp6 = FALSE;
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_EXTERN) == VALA_PARSER_MODIFIER_FLAGS_EXTERN) {
		_tmp6 = TRUE;
	} else {
		_tmp6 = vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner));
	}
	if (_tmp6) {
		vala_symbol_set_external ((ValaSymbol*) c, TRUE);
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_NEW) == VALA_PARSER_MODIFIER_FLAGS_NEW) {
		vala_member_set_hides ((ValaMember*) c, TRUE);
	}
	vala_parser_set_attributes (self, (ValaCodeNode*) c, attrs);
	_tmp7 = NULL;
	return (_tmp7 = c, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), id = (g_free (id), NULL), (initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)), (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), _tmp7);
}


static ValaField* vala_parser_parse_field_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaDataType* type;
	char* id;
	ValaSourceReference* _tmp0;
	ValaField* _tmp1;
	ValaField* f;
	gboolean _tmp2;
	gboolean _tmp3;
	gboolean _tmp4;
	ValaField* _tmp7;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	type = vala_parser_parse_type (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	f = (_tmp1 = vala_field_new (id, type, NULL, _tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access ((ValaSymbol*) f, access);
	vala_parser_set_attributes (self, (ValaCodeNode*) f, attrs);
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_STATIC) == VALA_PARSER_MODIFIER_FLAGS_STATIC) {
		vala_field_set_binding (f, MEMBER_BINDING_STATIC);
	} else {
		if ((flags & VALA_PARSER_MODIFIER_FLAGS_CLASS) == VALA_PARSER_MODIFIER_FLAGS_CLASS) {
			vala_field_set_binding (f, MEMBER_BINDING_CLASS);
		}
	}
	_tmp2 = FALSE;
	_tmp3 = FALSE;
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_ABSTRACT) == VALA_PARSER_MODIFIER_FLAGS_ABSTRACT) {
		_tmp3 = TRUE;
	} else {
		_tmp3 = (flags & VALA_PARSER_MODIFIER_FLAGS_VIRTUAL) == VALA_PARSER_MODIFIER_FLAGS_VIRTUAL;
	}
	if (_tmp3) {
		_tmp2 = TRUE;
	} else {
		_tmp2 = (flags & VALA_PARSER_MODIFIER_FLAGS_OVERRIDE) == VALA_PARSER_MODIFIER_FLAGS_OVERRIDE;
	}
	if (_tmp2) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) f), "abstract, virtual, and override modifiers are not applicable to fields");
	}
	_tmp4 = FALSE;
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_EXTERN) == VALA_PARSER_MODIFIER_FLAGS_EXTERN) {
		_tmp4 = TRUE;
	} else {
		_tmp4 = vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner));
	}
	if (_tmp4) {
		vala_symbol_set_external ((ValaSymbol*) f, TRUE);
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_NEW) == VALA_PARSER_MODIFIER_FLAGS_NEW) {
		vala_member_set_hides ((ValaMember*) f, TRUE);
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_ASSIGN)) {
		ValaExpression* _tmp5;
		ValaExpression* _tmp6;
		_tmp5 = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
				return NULL;
			} else {
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp6 = NULL;
		vala_field_set_initializer (f, _tmp6 = _tmp5);
		(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp7 = NULL;
	return (_tmp7 = f, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), id = (g_free (id), NULL), _tmp7);
}


static ValaInitializerList* vala_parser_parse_initializer (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSourceReference* _tmp0;
	ValaInitializerList* _tmp1;
	ValaInitializerList* initializer;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	initializer = (_tmp1 = vala_initializer_list_new (_tmp0 = vala_parser_get_src (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE) {
		do {
			ValaExpression* init;
			init = vala_parser_parse_argument (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
					return NULL;
				} else {
					(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			vala_initializer_list_append (initializer, init);
			(init == NULL) ? NULL : (init = (vala_code_node_unref (init), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
			return NULL;
		} else {
			(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	return initializer;
}


static ValaMethod* vala_parser_parse_method_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaDataType* type;
	char* id;
	GeeList* _tmp0;
	ValaSourceReference* _tmp1;
	ValaMethod* _tmp2;
	ValaMethod* method;
	ValaMethod* _tmp22;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	type = vala_parser_parse_type (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp0 = vala_parser_parse_type_parameter_list (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	(_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL));
	_tmp1 = NULL;
	_tmp2 = NULL;
	method = (_tmp2 = vala_method_new (id, type, _tmp1 = vala_parser_get_src_com (self, &begin)), (_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL)), _tmp2);
	vala_symbol_set_access ((ValaSymbol*) method, access);
	vala_parser_set_attributes (self, (ValaCodeNode*) method, attrs);
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_STATIC) == VALA_PARSER_MODIFIER_FLAGS_STATIC) {
		vala_method_set_binding (method, MEMBER_BINDING_STATIC);
	} else {
		if ((flags & VALA_PARSER_MODIFIER_FLAGS_CLASS) == VALA_PARSER_MODIFIER_FLAGS_CLASS) {
			vala_method_set_binding (method, MEMBER_BINDING_CLASS);
		}
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_NEW) == VALA_PARSER_MODIFIER_FLAGS_NEW) {
		vala_member_set_hides ((ValaMember*) method, TRUE);
	}
	if (vala_method_get_binding (method) == MEMBER_BINDING_INSTANCE) {
		gboolean _tmp3;
		gboolean _tmp4;
		gboolean _tmp5;
		if ((flags & VALA_PARSER_MODIFIER_FLAGS_ABSTRACT) == VALA_PARSER_MODIFIER_FLAGS_ABSTRACT) {
			vala_method_set_is_abstract (method, TRUE);
		}
		if ((flags & VALA_PARSER_MODIFIER_FLAGS_VIRTUAL) == VALA_PARSER_MODIFIER_FLAGS_VIRTUAL) {
			vala_method_set_is_virtual (method, TRUE);
		}
		if ((flags & VALA_PARSER_MODIFIER_FLAGS_OVERRIDE) == VALA_PARSER_MODIFIER_FLAGS_OVERRIDE) {
			vala_method_set_overrides (method, TRUE);
		}
		_tmp3 = FALSE;
		_tmp4 = FALSE;
		_tmp5 = FALSE;
		if (vala_method_get_is_abstract (method)) {
			_tmp5 = vala_method_get_is_virtual (method);
		} else {
			_tmp5 = FALSE;
		}
		if (_tmp5) {
			_tmp4 = TRUE;
		} else {
			gboolean _tmp6;
			_tmp6 = FALSE;
			if (vala_method_get_is_abstract (method)) {
				_tmp6 = vala_method_get_overrides (method);
			} else {
				_tmp6 = FALSE;
			}
			_tmp4 = _tmp6;
		}
		if (_tmp4) {
			_tmp3 = TRUE;
		} else {
			gboolean _tmp7;
			_tmp7 = FALSE;
			if (vala_method_get_is_virtual (method)) {
				_tmp7 = vala_method_get_overrides (method);
			} else {
				_tmp7 = FALSE;
			}
			_tmp3 = _tmp7;
		}
		if (_tmp3) {
			char* _tmp8;
			GError* _tmp9;
			_tmp8 = NULL;
			_tmp9 = NULL;
			inner_error = (_tmp9 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp8 = vala_parser_get_error (self, "only one of `abstract', `virtual', or `override' may be specified")), _tmp8 = (g_free (_tmp8), NULL), _tmp9);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
					return NULL;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
		}
	} else {
		gboolean _tmp10;
		gboolean _tmp11;
		_tmp10 = FALSE;
		_tmp11 = FALSE;
		if ((flags & VALA_PARSER_MODIFIER_FLAGS_ABSTRACT) == VALA_PARSER_MODIFIER_FLAGS_ABSTRACT) {
			_tmp11 = TRUE;
		} else {
			_tmp11 = (flags & VALA_PARSER_MODIFIER_FLAGS_VIRTUAL) == VALA_PARSER_MODIFIER_FLAGS_VIRTUAL;
		}
		if (_tmp11) {
			_tmp10 = TRUE;
		} else {
			_tmp10 = (flags & VALA_PARSER_MODIFIER_FLAGS_OVERRIDE) == VALA_PARSER_MODIFIER_FLAGS_OVERRIDE;
		}
		if (_tmp10) {
			char* _tmp12;
			GError* _tmp13;
			_tmp12 = NULL;
			_tmp13 = NULL;
			inner_error = (_tmp13 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp12 = vala_parser_get_error (self, "the modifiers `abstract', `virtual', and `override' are not valid for static methods")), _tmp12 = (g_free (_tmp12), NULL), _tmp13);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
					return NULL;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
		}
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_INLINE) == VALA_PARSER_MODIFIER_FLAGS_INLINE) {
		vala_method_set_is_inline (method, TRUE);
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_EXTERN) == VALA_PARSER_MODIFIER_FLAGS_EXTERN) {
		vala_symbol_set_external ((ValaSymbol*) method, TRUE);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaFormalParameter* param;
			param = vala_parser_parse_parameter (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
					return NULL;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			vala_method_add_parameter (method, param);
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_YIELDS)) {
		vala_method_set_coroutine (method, TRUE);
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_THROWS)) {
		do {
			ValaDataType* _tmp14;
			ValaDataType* _tmp15;
			_tmp14 = vala_parser_parse_type (self, TRUE, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
					return NULL;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp15 = NULL;
			vala_code_node_add_error_type ((ValaCodeNode*) method, _tmp15 = _tmp14);
			(_tmp15 == NULL) ? NULL : (_tmp15 = (vala_code_node_unref (_tmp15), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_REQUIRES)) {
		ValaExpression* _tmp16;
		ValaExpression* _tmp17;
		vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				return NULL;
			} else {
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp16 = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				return NULL;
			} else {
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp17 = NULL;
		vala_method_add_precondition (method, _tmp17 = _tmp16);
		(_tmp17 == NULL) ? NULL : (_tmp17 = (vala_code_node_unref (_tmp17), NULL));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				return NULL;
			} else {
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_ENSURES)) {
		ValaExpression* _tmp18;
		ValaExpression* _tmp19;
		vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				return NULL;
			} else {
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp18 = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				return NULL;
			} else {
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp19 = NULL;
		vala_method_add_postcondition (method, _tmp19 = _tmp18);
		(_tmp19 == NULL) ? NULL : (_tmp19 = (vala_code_node_unref (_tmp19), NULL));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				return NULL;
			} else {
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
	}
	if (!vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
		ValaBlock* _tmp20;
		ValaBlock* _tmp21;
		_tmp20 = vala_parser_parse_block (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				return NULL;
			} else {
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp21 = NULL;
		vala_method_set_body (method, _tmp21 = _tmp20);
		(_tmp21 == NULL) ? NULL : (_tmp21 = (vala_code_node_unref (_tmp21), NULL));
	} else {
		if (vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
			vala_symbol_set_external ((ValaSymbol*) method, TRUE);
		}
	}
	_tmp22 = NULL;
	return (_tmp22 = method, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), id = (g_free (id), NULL), _tmp22);
}


static ValaProperty* vala_parser_parse_property_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaDataType* type;
	gboolean getter_owned;
	char* id;
	ValaSourceReference* _tmp1;
	ValaProperty* _tmp2;
	ValaProperty* prop;
	gboolean _tmp3;
	gboolean _tmp22;
	ValaProperty* _tmp28;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	type = vala_parser_parse_type (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	getter_owned = FALSE;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_HASH)) {
		if (!vala_code_context_get_deprecated (self->priv->context)) {
			ValaSourceReference* _tmp0;
			_tmp0 = NULL;
			vala_report_warning (_tmp0 = vala_parser_get_last_src (self), "deprecated syntax, use `owned` modifier before `get'");
			(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
		}
		getter_owned = TRUE;
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp1 = NULL;
	_tmp2 = NULL;
	prop = (_tmp2 = vala_property_new (id, type, NULL, NULL, _tmp1 = vala_parser_get_src_com (self, &begin)), (_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL)), _tmp2);
	vala_symbol_set_access ((ValaSymbol*) prop, access);
	vala_parser_set_attributes (self, (ValaCodeNode*) prop, attrs);
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_STATIC) == VALA_PARSER_MODIFIER_FLAGS_STATIC) {
		vala_property_set_binding (prop, MEMBER_BINDING_STATIC);
	} else {
		if ((flags & VALA_PARSER_MODIFIER_FLAGS_CLASS) == VALA_PARSER_MODIFIER_FLAGS_CLASS) {
			vala_property_set_binding (prop, MEMBER_BINDING_CLASS);
		}
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_ABSTRACT) == VALA_PARSER_MODIFIER_FLAGS_ABSTRACT) {
		vala_property_set_is_abstract (prop, TRUE);
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_VIRTUAL) == VALA_PARSER_MODIFIER_FLAGS_VIRTUAL) {
		vala_property_set_is_virtual (prop, TRUE);
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_OVERRIDE) == VALA_PARSER_MODIFIER_FLAGS_OVERRIDE) {
		vala_property_set_overrides (prop, TRUE);
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_NEW) == VALA_PARSER_MODIFIER_FLAGS_NEW) {
		vala_member_set_hides ((ValaMember*) prop, TRUE);
	}
	_tmp3 = FALSE;
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_EXTERN) == VALA_PARSER_MODIFIER_FLAGS_EXTERN) {
		_tmp3 = TRUE;
	} else {
		_tmp3 = vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner));
	}
	if (_tmp3) {
		vala_symbol_set_external ((ValaSymbol*) prop, TRUE);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	while (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE) {
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_DEFAULT)) {
			ValaExpression* _tmp6;
			ValaExpression* _tmp7;
			if (vala_property_get_default_expression (prop) != NULL) {
				char* _tmp4;
				GError* _tmp5;
				_tmp4 = NULL;
				_tmp5 = NULL;
				inner_error = (_tmp5 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp4 = vala_parser_get_error (self, "property default value already defined")), _tmp4 = (g_free (_tmp4), NULL), _tmp5);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						g_propagate_error (error, inner_error);
						(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
						id = (g_free (id), NULL);
						(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
						return NULL;
					} else {
						(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
						id = (g_free (id), NULL);
						(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
						return NULL;
					}
				}
			}
			vala_parser_expect (self, VALA_TOKEN_TYPE_ASSIGN, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
					return NULL;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp6 = vala_parser_parse_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
					return NULL;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp7 = NULL;
			vala_property_set_default_expression (prop, _tmp7 = _tmp6);
			(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_code_node_unref (_tmp7), NULL));
			vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
					return NULL;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
		} else {
			ValaSourceLocation accessor_begin;
			GeeList* accessor_attrs;
			ValaSymbolAccessibility accessor_access;
			ValaDataType* value_type;
			accessor_begin = vala_parser_get_location (self);
			accessor_attrs = vala_parser_parse_attributes (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
					return NULL;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			accessor_access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			value_type = vala_data_type_copy (type);
			vala_data_type_set_value_owned (value_type, vala_parser_accept (self, VALA_TOKEN_TYPE_OWNED));
			if (vala_parser_accept (self, VALA_TOKEN_TYPE_GET)) {
				ValaBlock* block;
				ValaPropertyAccessor* _tmp13;
				ValaSourceReference* _tmp12;
				if (vala_property_get_get_accessor (prop) != NULL) {
					char* _tmp8;
					GError* _tmp9;
					_tmp8 = NULL;
					_tmp9 = NULL;
					inner_error = (_tmp9 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp8 = vala_parser_get_error (self, "property get accessor already defined")), _tmp8 = (g_free (_tmp8), NULL), _tmp9);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							g_propagate_error (error, inner_error);
							(accessor_attrs == NULL) ? NULL : (accessor_attrs = (gee_collection_object_unref (accessor_attrs), NULL));
							(value_type == NULL) ? NULL : (value_type = (vala_code_node_unref (value_type), NULL));
							(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
							id = (g_free (id), NULL);
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
							return NULL;
						} else {
							(accessor_attrs == NULL) ? NULL : (accessor_attrs = (gee_collection_object_unref (accessor_attrs), NULL));
							(value_type == NULL) ? NULL : (value_type = (vala_code_node_unref (value_type), NULL));
							(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
							id = (g_free (id), NULL);
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
							g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
							g_clear_error (&inner_error);
							return NULL;
						}
					}
				}
				if (getter_owned) {
					vala_data_type_set_value_owned (value_type, TRUE);
				}
				block = NULL;
				if (!vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
					ValaBlock* _tmp10;
					ValaBlock* _tmp11;
					_tmp10 = vala_parser_parse_block (self, &inner_error);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							g_propagate_error (error, inner_error);
							(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
							(accessor_attrs == NULL) ? NULL : (accessor_attrs = (gee_collection_object_unref (accessor_attrs), NULL));
							(value_type == NULL) ? NULL : (value_type = (vala_code_node_unref (value_type), NULL));
							(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
							id = (g_free (id), NULL);
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
							return NULL;
						} else {
							(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
							(accessor_attrs == NULL) ? NULL : (accessor_attrs = (gee_collection_object_unref (accessor_attrs), NULL));
							(value_type == NULL) ? NULL : (value_type = (vala_code_node_unref (value_type), NULL));
							(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
							id = (g_free (id), NULL);
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
							g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
							g_clear_error (&inner_error);
							return NULL;
						}
					}
					_tmp11 = NULL;
					block = (_tmp11 = _tmp10, (block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL)), _tmp11);
					vala_symbol_set_external ((ValaSymbol*) prop, FALSE);
				}
				_tmp13 = NULL;
				_tmp12 = NULL;
				vala_property_set_get_accessor (prop, _tmp13 = vala_property_accessor_new (TRUE, FALSE, FALSE, value_type, block, _tmp12 = vala_parser_get_src (self, &accessor_begin)));
				(_tmp13 == NULL) ? NULL : (_tmp13 = (vala_code_node_unref (_tmp13), NULL));
				(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_source_reference_unref (_tmp12), NULL));
				vala_parser_set_attributes (self, (ValaCodeNode*) vala_property_get_get_accessor (prop), accessor_attrs);
				vala_property_accessor_set_access (vala_property_get_get_accessor (prop), accessor_access);
				(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			} else {
				gboolean writable;
				gboolean _construct;
				ValaBlock* block;
				ValaPropertyAccessor* _tmp21;
				ValaSourceReference* _tmp20;
				writable = FALSE;
				_construct = FALSE;
				if (vala_parser_accept (self, VALA_TOKEN_TYPE_SET)) {
					writable = TRUE;
					_construct = vala_parser_accept (self, VALA_TOKEN_TYPE_CONSTRUCT);
				} else {
					if (vala_parser_accept (self, VALA_TOKEN_TYPE_CONSTRUCT)) {
						_construct = TRUE;
						writable = vala_parser_accept (self, VALA_TOKEN_TYPE_SET);
					} else {
						char* _tmp14;
						GError* _tmp15;
						_tmp14 = NULL;
						_tmp15 = NULL;
						inner_error = (_tmp15 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp14 = vala_parser_get_error (self, "expected get, set, or construct")), _tmp14 = (g_free (_tmp14), NULL), _tmp15);
						if (inner_error != NULL) {
							if (inner_error->domain == VALA_PARSE_ERROR) {
								g_propagate_error (error, inner_error);
								(accessor_attrs == NULL) ? NULL : (accessor_attrs = (gee_collection_object_unref (accessor_attrs), NULL));
								(value_type == NULL) ? NULL : (value_type = (vala_code_node_unref (value_type), NULL));
								(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
								id = (g_free (id), NULL);
								(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
								return NULL;
							} else {
								(accessor_attrs == NULL) ? NULL : (accessor_attrs = (gee_collection_object_unref (accessor_attrs), NULL));
								(value_type == NULL) ? NULL : (value_type = (vala_code_node_unref (value_type), NULL));
								(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
								id = (g_free (id), NULL);
								(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
								g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
								g_clear_error (&inner_error);
								return NULL;
							}
						}
					}
				}
				if (vala_property_get_set_accessor (prop) != NULL) {
					char* _tmp16;
					GError* _tmp17;
					_tmp16 = NULL;
					_tmp17 = NULL;
					inner_error = (_tmp17 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp16 = vala_parser_get_error (self, "property set accessor already defined")), _tmp16 = (g_free (_tmp16), NULL), _tmp17);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							g_propagate_error (error, inner_error);
							(accessor_attrs == NULL) ? NULL : (accessor_attrs = (gee_collection_object_unref (accessor_attrs), NULL));
							(value_type == NULL) ? NULL : (value_type = (vala_code_node_unref (value_type), NULL));
							(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
							id = (g_free (id), NULL);
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
							return NULL;
						} else {
							(accessor_attrs == NULL) ? NULL : (accessor_attrs = (gee_collection_object_unref (accessor_attrs), NULL));
							(value_type == NULL) ? NULL : (value_type = (vala_code_node_unref (value_type), NULL));
							(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
							id = (g_free (id), NULL);
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
							g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
							g_clear_error (&inner_error);
							return NULL;
						}
					}
				}
				block = NULL;
				if (!vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
					ValaBlock* _tmp18;
					ValaBlock* _tmp19;
					_tmp18 = vala_parser_parse_block (self, &inner_error);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							g_propagate_error (error, inner_error);
							(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
							(accessor_attrs == NULL) ? NULL : (accessor_attrs = (gee_collection_object_unref (accessor_attrs), NULL));
							(value_type == NULL) ? NULL : (value_type = (vala_code_node_unref (value_type), NULL));
							(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
							id = (g_free (id), NULL);
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
							return NULL;
						} else {
							(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
							(accessor_attrs == NULL) ? NULL : (accessor_attrs = (gee_collection_object_unref (accessor_attrs), NULL));
							(value_type == NULL) ? NULL : (value_type = (vala_code_node_unref (value_type), NULL));
							(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
							id = (g_free (id), NULL);
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
							g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
							g_clear_error (&inner_error);
							return NULL;
						}
					}
					_tmp19 = NULL;
					block = (_tmp19 = _tmp18, (block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL)), _tmp19);
					vala_symbol_set_external ((ValaSymbol*) prop, FALSE);
				}
				_tmp21 = NULL;
				_tmp20 = NULL;
				vala_property_set_set_accessor (prop, _tmp21 = vala_property_accessor_new (FALSE, writable, _construct, value_type, block, _tmp20 = vala_parser_get_src (self, &accessor_begin)));
				(_tmp21 == NULL) ? NULL : (_tmp21 = (vala_code_node_unref (_tmp21), NULL));
				(_tmp20 == NULL) ? NULL : (_tmp20 = (vala_source_reference_unref (_tmp20), NULL));
				vala_parser_set_attributes (self, (ValaCodeNode*) vala_property_get_set_accessor (prop), accessor_attrs);
				vala_property_accessor_set_access (vala_property_get_set_accessor (prop), accessor_access);
				(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
			}
			(accessor_attrs == NULL) ? NULL : (accessor_attrs = (gee_collection_object_unref (accessor_attrs), NULL));
			(value_type == NULL) ? NULL : (value_type = (vala_code_node_unref (value_type), NULL));
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp22 = FALSE;
	if (!vala_property_get_is_abstract (prop)) {
		_tmp22 = !vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner));
	} else {
		_tmp22 = FALSE;
	}
	if (_tmp22) {
		gboolean _tmp23;
		gboolean empty_get;
		gboolean _tmp24;
		gboolean empty_set;
		gboolean _tmp25;
		_tmp23 = FALSE;
		if (vala_property_get_get_accessor (prop) != NULL) {
			_tmp23 = vala_property_accessor_get_body (vala_property_get_get_accessor (prop)) == NULL;
		} else {
			_tmp23 = FALSE;
		}
		empty_get = _tmp23;
		_tmp24 = FALSE;
		if (vala_property_get_set_accessor (prop) != NULL) {
			_tmp24 = vala_property_accessor_get_body (vala_property_get_set_accessor (prop)) == NULL;
		} else {
			_tmp24 = FALSE;
		}
		empty_set = _tmp24;
		if (empty_get != empty_set) {
			if (empty_get) {
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) prop), "property getter must have a body");
			} else {
				if (empty_set) {
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) prop), "property setter must have a body");
				}
			}
			vala_code_node_set_error ((ValaCodeNode*) prop, TRUE);
		}
		_tmp25 = FALSE;
		if (empty_get) {
			_tmp25 = empty_set;
		} else {
			_tmp25 = FALSE;
		}
		if (_tmp25) {
			ValaDataType* field_type;
			ValaField* _tmp27;
			char* _tmp26;
			/* automatic property accessor body generation */
			field_type = vala_data_type_copy (vala_property_get_property_type (prop));
			_tmp27 = NULL;
			_tmp26 = NULL;
			vala_property_set_field (prop, _tmp27 = vala_field_new (_tmp26 = g_strdup_printf ("_%s", vala_symbol_get_name ((ValaSymbol*) prop)), field_type, vala_property_get_default_expression (prop), vala_code_node_get_source_reference ((ValaCodeNode*) prop)));
			(_tmp27 == NULL) ? NULL : (_tmp27 = (vala_code_node_unref (_tmp27), NULL));
			_tmp26 = (g_free (_tmp26), NULL);
			vala_symbol_set_access ((ValaSymbol*) vala_property_get_field (prop), VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
			(field_type == NULL) ? NULL : (field_type = (vala_code_node_unref (field_type), NULL));
		}
	}
	_tmp28 = NULL;
	return (_tmp28 = prop, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), id = (g_free (id), NULL), _tmp28);
}


static ValaSignal* vala_parser_parse_signal_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaDataType* type;
	char* id;
	ValaSourceReference* _tmp0;
	ValaSignal* _tmp1;
	ValaSignal* sig;
	ValaSignal* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_SIGNAL, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	type = vala_parser_parse_type (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	sig = (_tmp1 = vala_signal_new (id, type, _tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access ((ValaSymbol*) sig, access);
	vala_parser_set_attributes (self, (ValaCodeNode*) sig, attrs);
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_VIRTUAL) == VALA_PARSER_MODIFIER_FLAGS_VIRTUAL) {
		vala_signal_set_is_virtual (sig, TRUE);
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_NEW) == VALA_PARSER_MODIFIER_FLAGS_NEW) {
		vala_member_set_hides ((ValaMember*) sig, TRUE);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaFormalParameter* param;
			param = vala_parser_parse_parameter (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
					return NULL;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					id = (g_free (id), NULL);
					(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			vala_signal_add_parameter (sig, param);
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			id = (g_free (id), NULL);
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = sig, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), id = (g_free (id), NULL), _tmp2);
}


static ValaConstructor* vala_parser_parse_constructor_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaParserModifierFlags flags;
	ValaSourceReference* _tmp2;
	ValaConstructor* _tmp3;
	ValaConstructor* c;
	ValaBlock* _tmp4;
	ValaBlock* _tmp5;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_CONSTRUCT, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_NEW) == VALA_PARSER_MODIFIER_FLAGS_NEW) {
		char* _tmp0;
		GError* _tmp1;
		_tmp0 = NULL;
		_tmp1 = NULL;
		inner_error = (_tmp1 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp0 = vala_parser_get_error (self, "`new' modifier not allowed on constructor")), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				return NULL;
			} else {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	c = (_tmp3 = vala_constructor_new (_tmp2 = vala_parser_get_src_com (self, &begin)), (_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)), _tmp3);
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_STATIC) == VALA_PARSER_MODIFIER_FLAGS_STATIC) {
		vala_constructor_set_binding (c, MEMBER_BINDING_STATIC);
	} else {
		if ((flags & VALA_PARSER_MODIFIER_FLAGS_CLASS) == VALA_PARSER_MODIFIER_FLAGS_CLASS) {
			vala_constructor_set_binding (c, MEMBER_BINDING_CLASS);
		}
	}
	_tmp4 = vala_parser_parse_block (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
			return NULL;
		} else {
			(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp5 = NULL;
	vala_constructor_set_body (c, _tmp5 = _tmp4);
	(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
	return c;
}


static ValaDestructor* vala_parser_parse_destructor_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaParserModifierFlags flags;
	char* _tmp0;
	ValaSourceReference* _tmp3;
	ValaDestructor* _tmp4;
	ValaDestructor* d;
	ValaBlock* _tmp5;
	ValaBlock* _tmp6;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_TILDE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp0 = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = (g_free (_tmp0), NULL);
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_NEW) == VALA_PARSER_MODIFIER_FLAGS_NEW) {
		char* _tmp1;
		GError* _tmp2;
		_tmp1 = NULL;
		_tmp2 = NULL;
		inner_error = (_tmp2 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp1 = vala_parser_get_error (self, "`new' modifier not allowed on destructor")), _tmp1 = (g_free (_tmp1), NULL), _tmp2);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				return NULL;
			} else {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
	}
	_tmp3 = NULL;
	_tmp4 = NULL;
	d = (_tmp4 = vala_destructor_new (_tmp3 = vala_parser_get_src_com (self, &begin)), (_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)), _tmp4);
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_STATIC) == VALA_PARSER_MODIFIER_FLAGS_STATIC) {
		vala_destructor_set_binding (d, MEMBER_BINDING_STATIC);
	} else {
		if ((flags & VALA_PARSER_MODIFIER_FLAGS_CLASS) == VALA_PARSER_MODIFIER_FLAGS_CLASS) {
			vala_destructor_set_binding (d, MEMBER_BINDING_CLASS);
		}
	}
	_tmp5 = vala_parser_parse_block (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
			return NULL;
		} else {
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp6 = NULL;
	vala_destructor_set_body (d, _tmp6 = _tmp5);
	(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
	return d;
}


static ValaSymbol* vala_parser_parse_struct_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaUnresolvedSymbol* sym;
	GeeList* type_param_list;
	ValaDataType* base_type;
	ValaSourceReference* _tmp2;
	ValaStruct* _tmp3;
	ValaStruct* st;
	gboolean _tmp4;
	ValaSymbol* _tmp5;
	ValaSymbol* result;
	ValaSymbol* _tmp10;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_type_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_STRUCT, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	type_param_list = vala_parser_parse_type_parameter_list (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	base_type = NULL;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_COLON)) {
		ValaDataType* _tmp0;
		ValaDataType* _tmp1;
		_tmp0 = vala_parser_parse_type (self, TRUE, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
				return NULL;
			} else {
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp1 = NULL;
		base_type = (_tmp1 = _tmp0, (base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL)), _tmp1);
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	st = (_tmp3 = vala_struct_new (vala_symbol_get_name ((ValaSymbol*) sym), _tmp2 = vala_parser_get_src_com (self, &begin)), (_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)), _tmp3);
	vala_symbol_set_access ((ValaSymbol*) st, access);
	_tmp4 = FALSE;
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_EXTERN) == VALA_PARSER_MODIFIER_FLAGS_EXTERN) {
		_tmp4 = TRUE;
	} else {
		_tmp4 = vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner));
	}
	if (_tmp4) {
		vala_symbol_set_external ((ValaSymbol*) st, TRUE);
	}
	vala_parser_set_attributes (self, (ValaCodeNode*) st, attrs);
	{
		GeeIterator* _type_param_it;
		_type_param_it = gee_iterable_iterator ((GeeIterable*) type_param_list);
		while (gee_iterator_next (_type_param_it)) {
			ValaTypeParameter* type_param;
			type_param = (ValaTypeParameter*) gee_iterator_get (_type_param_it);
			vala_struct_add_type_parameter (st, type_param);
			(type_param == NULL) ? NULL : (type_param = (vala_code_node_unref (type_param), NULL));
		}
		(_type_param_it == NULL) ? NULL : (_type_param_it = (gee_collection_object_unref (_type_param_it), NULL));
	}
	if (base_type != NULL) {
		vala_struct_set_base_type (st, base_type);
	}
	vala_parser_parse_declarations (self, (ValaSymbol*) st, FALSE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
			(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
			(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp5 = NULL;
	result = (_tmp5 = (ValaSymbol*) st, (_tmp5 == NULL) ? NULL : vala_code_node_ref (_tmp5));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp7;
		ValaUnresolvedSymbol* _tmp6;
		ValaNamespace* ns;
		ValaSymbol* _tmp9;
		ValaSymbol* _tmp8;
		_tmp7 = NULL;
		_tmp6 = NULL;
		sym = (_tmp7 = (_tmp6 = vala_unresolved_symbol_get_inner (sym), (_tmp6 == NULL) ? NULL : vala_code_node_ref (_tmp6)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp7);
		ns = vala_namespace_new (vala_symbol_get_name ((ValaSymbol*) sym), vala_code_node_get_source_reference ((ValaCodeNode*) st));
		if (VALA_IS_NAMESPACE (result)) {
			vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		} else {
			vala_namespace_add_struct (ns, VALA_STRUCT (result));
			vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), (ValaCodeNode*) result);
		}
		_tmp9 = NULL;
		_tmp8 = NULL;
		result = (_tmp9 = (_tmp8 = (ValaSymbol*) ns, (_tmp8 == NULL) ? NULL : vala_code_node_ref (_tmp8)), (result == NULL) ? NULL : (result = (vala_code_node_unref (result), NULL)), _tmp9);
		(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
	}
	_tmp10 = NULL;
	return (_tmp10 = result, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL)), (base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL)), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp10);
}


static void vala_parser_parse_struct_member (ValaParser* self, ValaStruct* st, GError** error) {
	GError * inner_error;
	ValaSymbol* sym;
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	inner_error = NULL;
	sym = vala_parser_parse_declaration (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return;
		}
	}
	if (VALA_IS_METHOD (sym)) {
		vala_struct_add_method (st, VALA_METHOD (sym));
	} else {
		if (VALA_IS_FIELD (sym)) {
			vala_struct_add_field (st, VALA_FIELD (sym));
		} else {
			if (VALA_IS_CONSTANT (sym)) {
				vala_struct_add_constant (st, VALA_CONSTANT (sym));
			} else {
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) sym), "unexpected declaration in struct");
			}
		}
	}
	(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
}


static ValaSymbol* vala_parser_parse_interface_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaUnresolvedSymbol* sym;
	GeeList* type_param_list;
	GeeArrayList* base_types;
	ValaSourceReference* _tmp0;
	ValaInterface* _tmp1;
	ValaInterface* iface;
	gboolean _tmp2;
	ValaSymbol* _tmp3;
	ValaSymbol* result;
	ValaSymbol* _tmp8;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_type_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_INTERFACE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	type_param_list = vala_parser_parse_type_parameter_list (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	base_types = gee_array_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_COLON)) {
		do {
			ValaDataType* type;
			type = vala_parser_parse_type (self, TRUE, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
					(base_types == NULL) ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL));
					return NULL;
				} else {
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
					(base_types == NULL) ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			gee_collection_add ((GeeCollection*) base_types, type);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	iface = (_tmp1 = vala_interface_new (vala_symbol_get_name ((ValaSymbol*) sym), _tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access ((ValaSymbol*) iface, access);
	_tmp2 = FALSE;
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_EXTERN) == VALA_PARSER_MODIFIER_FLAGS_EXTERN) {
		_tmp2 = TRUE;
	} else {
		_tmp2 = vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner));
	}
	if (_tmp2) {
		vala_symbol_set_external ((ValaSymbol*) iface, TRUE);
	}
	vala_parser_set_attributes (self, (ValaCodeNode*) iface, attrs);
	{
		GeeIterator* _type_param_it;
		_type_param_it = gee_iterable_iterator ((GeeIterable*) type_param_list);
		while (gee_iterator_next (_type_param_it)) {
			ValaTypeParameter* type_param;
			type_param = (ValaTypeParameter*) gee_iterator_get (_type_param_it);
			vala_object_type_symbol_add_type_parameter ((ValaObjectTypeSymbol*) iface, type_param);
			(type_param == NULL) ? NULL : (type_param = (vala_code_node_unref (type_param), NULL));
		}
		(_type_param_it == NULL) ? NULL : (_type_param_it = (gee_collection_object_unref (_type_param_it), NULL));
	}
	{
		GeeIterator* _base_type_it;
		_base_type_it = gee_iterable_iterator ((GeeIterable*) base_types);
		while (gee_iterator_next (_base_type_it)) {
			ValaDataType* base_type;
			base_type = (ValaDataType*) gee_iterator_get (_base_type_it);
			vala_interface_add_prerequisite (iface, base_type);
			(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
		}
		(_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL));
	}
	vala_parser_parse_declarations (self, (ValaSymbol*) iface, FALSE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
			(base_types == NULL) ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL));
			(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
			(base_types == NULL) ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL));
			(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp3 = NULL;
	result = (_tmp3 = (ValaSymbol*) iface, (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp5;
		ValaUnresolvedSymbol* _tmp4;
		ValaNamespace* ns;
		ValaSymbol* _tmp7;
		ValaSymbol* _tmp6;
		_tmp5 = NULL;
		_tmp4 = NULL;
		sym = (_tmp5 = (_tmp4 = vala_unresolved_symbol_get_inner (sym), (_tmp4 == NULL) ? NULL : vala_code_node_ref (_tmp4)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp5);
		ns = vala_namespace_new (vala_symbol_get_name ((ValaSymbol*) sym), vala_code_node_get_source_reference ((ValaCodeNode*) iface));
		if (VALA_IS_NAMESPACE (result)) {
			vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		} else {
			vala_namespace_add_interface (ns, VALA_INTERFACE (result));
			vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), (ValaCodeNode*) result);
		}
		_tmp7 = NULL;
		_tmp6 = NULL;
		result = (_tmp7 = (_tmp6 = (ValaSymbol*) ns, (_tmp6 == NULL) ? NULL : vala_code_node_ref (_tmp6)), (result == NULL) ? NULL : (result = (vala_code_node_unref (result), NULL)), _tmp7);
		(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
	}
	_tmp8 = NULL;
	return (_tmp8 = result, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL)), (base_types == NULL) ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL)), (iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL)), _tmp8);
}


static void vala_parser_parse_interface_member (ValaParser* self, ValaInterface* iface, GError** error) {
	GError * inner_error;
	ValaSymbol* sym;
	g_return_if_fail (self != NULL);
	g_return_if_fail (iface != NULL);
	inner_error = NULL;
	sym = vala_parser_parse_declaration (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return;
		}
	}
	if (VALA_IS_CLASS (sym)) {
		vala_interface_add_class (iface, VALA_CLASS (sym));
	} else {
		if (VALA_IS_STRUCT (sym)) {
			vala_interface_add_struct (iface, VALA_STRUCT (sym));
		} else {
			if (VALA_IS_ENUM (sym)) {
				vala_interface_add_enum (iface, VALA_ENUM (sym));
			} else {
				if (VALA_IS_DELEGATE (sym)) {
					vala_interface_add_delegate (iface, VALA_DELEGATE (sym));
				} else {
					if (VALA_IS_METHOD (sym)) {
						vala_interface_add_method (iface, VALA_METHOD (sym));
					} else {
						if (VALA_IS_SIGNAL (sym)) {
							vala_interface_add_signal (iface, VALA_SIGNAL (sym));
						} else {
							if (VALA_IS_FIELD (sym)) {
								vala_interface_add_field (iface, VALA_FIELD (sym));
							} else {
								if (VALA_IS_PROPERTY (sym)) {
									vala_interface_add_property (iface, VALA_PROPERTY (sym));
								} else {
									vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) sym), "unexpected declaration in interface");
								}
							}
						}
					}
				}
			}
		}
	}
	(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
}


static ValaSymbol* vala_parser_parse_enum_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaUnresolvedSymbol* sym;
	ValaSourceReference* _tmp0;
	ValaEnum* _tmp1;
	ValaEnum* en;
	gboolean _tmp2;
	ValaSymbol* _tmp9;
	ValaSymbol* result;
	ValaSymbol* _tmp14;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_type_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_ENUM, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	en = (_tmp1 = vala_enum_new (vala_symbol_get_name ((ValaSymbol*) sym), _tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access ((ValaSymbol*) en, access);
	_tmp2 = FALSE;
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_EXTERN) == VALA_PARSER_MODIFIER_FLAGS_EXTERN) {
		_tmp2 = TRUE;
	} else {
		_tmp2 = vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner));
	}
	if (_tmp2) {
		vala_symbol_set_external ((ValaSymbol*) en, TRUE);
	}
	vala_parser_set_attributes (self, (ValaCodeNode*) en, attrs);
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	do {
		gboolean _tmp3;
		GeeList* value_attrs;
		ValaSourceLocation value_begin;
		char* id;
		ValaSourceReference* _tmp5;
		ValaEnumValue* _tmp6;
		ValaEnumValue* ev;
		_tmp3 = FALSE;
		if (vala_parser_current (self) == VALA_TOKEN_TYPE_CLOSE_BRACE) {
			GeeList* _tmp4;
			_tmp4 = NULL;
			_tmp3 = gee_collection_get_size ((GeeCollection*) (_tmp4 = vala_enum_get_values (en))) > 0;
			(_tmp4 == NULL) ? NULL : (_tmp4 = (gee_collection_object_unref (_tmp4), NULL));
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			/* allow trailing comma*/
			break;
		}
		value_attrs = vala_parser_parse_attributes (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
				return NULL;
			} else {
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		value_begin = vala_parser_get_location (self);
		id = vala_parser_parse_identifier (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(value_attrs == NULL) ? NULL : (value_attrs = (gee_collection_object_unref (value_attrs), NULL));
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
				return NULL;
			} else {
				(value_attrs == NULL) ? NULL : (value_attrs = (gee_collection_object_unref (value_attrs), NULL));
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp5 = NULL;
		_tmp6 = NULL;
		ev = (_tmp6 = vala_enum_value_new (id, _tmp5 = vala_parser_get_src (self, &value_begin)), (_tmp5 == NULL) ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL)), _tmp6);
		vala_parser_set_attributes (self, (ValaCodeNode*) ev, value_attrs);
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_ASSIGN)) {
			ValaExpression* _tmp7;
			ValaExpression* _tmp8;
			_tmp7 = vala_parser_parse_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(value_attrs == NULL) ? NULL : (value_attrs = (gee_collection_object_unref (value_attrs), NULL));
					id = (g_free (id), NULL);
					(ev == NULL) ? NULL : (ev = (vala_code_node_unref (ev), NULL));
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
					return NULL;
				} else {
					(value_attrs == NULL) ? NULL : (value_attrs = (gee_collection_object_unref (value_attrs), NULL));
					id = (g_free (id), NULL);
					(ev == NULL) ? NULL : (ev = (vala_code_node_unref (ev), NULL));
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp8 = NULL;
			vala_enum_value_set_value (ev, _tmp8 = _tmp7);
			(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL));
		}
		vala_enum_add_value (en, ev);
		(value_attrs == NULL) ? NULL : (value_attrs = (gee_collection_object_unref (value_attrs), NULL));
		id = (g_free (id), NULL);
		(ev == NULL) ? NULL : (ev = (vala_code_node_unref (ev), NULL));
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
		while (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE) {
			ValaSymbol* member_sym;
			member_sym = vala_parser_parse_declaration (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
					return NULL;
				} else {
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			if (VALA_IS_METHOD (member_sym)) {
				vala_enum_add_method (en, VALA_METHOD (member_sym));
			} else {
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) member_sym), "unexpected declaration in enum");
			}
			(member_sym == NULL) ? NULL : (member_sym = (vala_code_node_unref (member_sym), NULL));
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp9 = NULL;
	result = (_tmp9 = (ValaSymbol*) en, (_tmp9 == NULL) ? NULL : vala_code_node_ref (_tmp9));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp11;
		ValaUnresolvedSymbol* _tmp10;
		ValaNamespace* ns;
		ValaSymbol* _tmp13;
		ValaSymbol* _tmp12;
		_tmp11 = NULL;
		_tmp10 = NULL;
		sym = (_tmp11 = (_tmp10 = vala_unresolved_symbol_get_inner (sym), (_tmp10 == NULL) ? NULL : vala_code_node_ref (_tmp10)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp11);
		ns = vala_namespace_new (vala_symbol_get_name ((ValaSymbol*) sym), vala_code_node_get_source_reference ((ValaCodeNode*) en));
		if (VALA_IS_NAMESPACE (result)) {
			vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		} else {
			vala_namespace_add_enum (ns, VALA_ENUM (result));
			vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), (ValaCodeNode*) result);
		}
		_tmp13 = NULL;
		_tmp12 = NULL;
		result = (_tmp13 = (_tmp12 = (ValaSymbol*) ns, (_tmp12 == NULL) ? NULL : vala_code_node_ref (_tmp12)), (result == NULL) ? NULL : (result = (vala_code_node_unref (result), NULL)), _tmp13);
		(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
	}
	_tmp14 = NULL;
	return (_tmp14 = result, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL)), _tmp14);
}


static ValaSymbol* vala_parser_parse_errordomain_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaUnresolvedSymbol* sym;
	ValaSourceReference* _tmp0;
	ValaErrorDomain* _tmp1;
	ValaErrorDomain* ed;
	gboolean _tmp2;
	ValaSymbol* _tmp9;
	ValaSymbol* result;
	ValaSymbol* _tmp14;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_type_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_ERRORDOMAIN, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	ed = (_tmp1 = vala_error_domain_new (vala_symbol_get_name ((ValaSymbol*) sym), _tmp0 = vala_parser_get_src_com (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access ((ValaSymbol*) ed, access);
	_tmp2 = FALSE;
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_EXTERN) == VALA_PARSER_MODIFIER_FLAGS_EXTERN) {
		_tmp2 = TRUE;
	} else {
		_tmp2 = vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner));
	}
	if (_tmp2) {
		vala_symbol_set_external ((ValaSymbol*) ed, TRUE);
	}
	vala_parser_set_attributes (self, (ValaCodeNode*) ed, attrs);
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	do {
		gboolean _tmp3;
		GeeList* code_attrs;
		ValaSourceLocation code_begin;
		char* id;
		ValaSourceReference* _tmp5;
		ValaErrorCode* _tmp6;
		ValaErrorCode* ec;
		_tmp3 = FALSE;
		if (vala_parser_current (self) == VALA_TOKEN_TYPE_CLOSE_BRACE) {
			GeeList* _tmp4;
			_tmp4 = NULL;
			_tmp3 = gee_collection_get_size ((GeeCollection*) (_tmp4 = vala_error_domain_get_codes (ed))) > 0;
			(_tmp4 == NULL) ? NULL : (_tmp4 = (gee_collection_object_unref (_tmp4), NULL));
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			/* allow trailing comma*/
			break;
		}
		code_attrs = vala_parser_parse_attributes (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
				return NULL;
			} else {
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		code_begin = vala_parser_get_location (self);
		id = vala_parser_parse_identifier (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(code_attrs == NULL) ? NULL : (code_attrs = (gee_collection_object_unref (code_attrs), NULL));
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
				return NULL;
			} else {
				(code_attrs == NULL) ? NULL : (code_attrs = (gee_collection_object_unref (code_attrs), NULL));
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp5 = NULL;
		_tmp6 = NULL;
		ec = (_tmp6 = vala_error_code_new (id, _tmp5 = vala_parser_get_src (self, &code_begin)), (_tmp5 == NULL) ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL)), _tmp6);
		vala_parser_set_attributes (self, (ValaCodeNode*) ec, code_attrs);
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_ASSIGN)) {
			ValaExpression* _tmp7;
			ValaExpression* _tmp8;
			_tmp7 = vala_parser_parse_expression (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(code_attrs == NULL) ? NULL : (code_attrs = (gee_collection_object_unref (code_attrs), NULL));
					id = (g_free (id), NULL);
					(ec == NULL) ? NULL : (ec = (vala_code_node_unref (ec), NULL));
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
					return NULL;
				} else {
					(code_attrs == NULL) ? NULL : (code_attrs = (gee_collection_object_unref (code_attrs), NULL));
					id = (g_free (id), NULL);
					(ec == NULL) ? NULL : (ec = (vala_code_node_unref (ec), NULL));
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp8 = NULL;
			vala_error_code_set_value (ec, _tmp8 = _tmp7);
			(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL));
		}
		vala_error_domain_add_code (ed, ec);
		(code_attrs == NULL) ? NULL : (code_attrs = (gee_collection_object_unref (code_attrs), NULL));
		id = (g_free (id), NULL);
		(ec == NULL) ? NULL : (ec = (vala_code_node_unref (ec), NULL));
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
		while (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE) {
			ValaSymbol* member_sym;
			member_sym = vala_parser_parse_declaration (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
					return NULL;
				} else {
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			if (VALA_IS_METHOD (member_sym)) {
				vala_error_domain_add_method (ed, VALA_METHOD (member_sym));
			} else {
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) member_sym), "unexpected declaration in errordomain");
			}
			(member_sym == NULL) ? NULL : (member_sym = (vala_code_node_unref (member_sym), NULL));
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp9 = NULL;
	result = (_tmp9 = (ValaSymbol*) ed, (_tmp9 == NULL) ? NULL : vala_code_node_ref (_tmp9));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp11;
		ValaUnresolvedSymbol* _tmp10;
		ValaNamespace* ns;
		ValaSymbol* _tmp13;
		ValaSymbol* _tmp12;
		_tmp11 = NULL;
		_tmp10 = NULL;
		sym = (_tmp11 = (_tmp10 = vala_unresolved_symbol_get_inner (sym), (_tmp10 == NULL) ? NULL : vala_code_node_ref (_tmp10)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp11);
		ns = vala_namespace_new (vala_symbol_get_name ((ValaSymbol*) sym), vala_code_node_get_source_reference ((ValaCodeNode*) ed));
		if (VALA_IS_NAMESPACE (result)) {
			vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		} else {
			vala_namespace_add_error_domain (ns, VALA_ERROR_DOMAIN (result));
			vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), (ValaCodeNode*) result);
		}
		_tmp13 = NULL;
		_tmp12 = NULL;
		result = (_tmp13 = (_tmp12 = (ValaSymbol*) ns, (_tmp12 == NULL) ? NULL : vala_code_node_ref (_tmp12)), (result == NULL) ? NULL : (result = (vala_code_node_unref (result), NULL)), _tmp13);
		(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
	}
	_tmp14 = NULL;
	return (_tmp14 = result, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL)), _tmp14);
}


static ValaSymbolAccessibility vala_parser_parse_access_modifier (ValaParser* self, ValaSymbolAccessibility default_access) {
	g_return_val_if_fail (self != NULL, 0);
	switch (vala_parser_current (self)) {
		case VALA_TOKEN_TYPE_PRIVATE:
		{
			vala_parser_next (self);
			return VALA_SYMBOL_ACCESSIBILITY_PRIVATE;
		}
		case VALA_TOKEN_TYPE_PROTECTED:
		{
			vala_parser_next (self);
			return VALA_SYMBOL_ACCESSIBILITY_PROTECTED;
		}
		case VALA_TOKEN_TYPE_INTERNAL:
		{
			vala_parser_next (self);
			return VALA_SYMBOL_ACCESSIBILITY_INTERNAL;
		}
		case VALA_TOKEN_TYPE_PUBLIC:
		{
			vala_parser_next (self);
			return VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
		}
		default:
		{
			return default_access;
		}
	}
}


static ValaParserModifierFlags vala_parser_parse_type_declaration_modifiers (ValaParser* self) {
	ValaParserModifierFlags flags;
	g_return_val_if_fail (self != NULL, 0);
	flags = 0;
	while (TRUE) {
		switch (vala_parser_current (self)) {
			case VALA_TOKEN_TYPE_ABSTRACT:
			{
				vala_parser_next (self);
				flags = flags | VALA_PARSER_MODIFIER_FLAGS_ABSTRACT;
				break;
			}
			case VALA_TOKEN_TYPE_EXTERN:
			{
				vala_parser_next (self);
				flags = flags | VALA_PARSER_MODIFIER_FLAGS_EXTERN;
				break;
			}
			case VALA_TOKEN_TYPE_STATIC:
			{
				vala_parser_next (self);
				flags = flags | VALA_PARSER_MODIFIER_FLAGS_STATIC;
				break;
			}
			default:
			{
				return flags;
			}
		}
	}
}


static ValaParserModifierFlags vala_parser_parse_member_declaration_modifiers (ValaParser* self) {
	ValaParserModifierFlags flags;
	g_return_val_if_fail (self != NULL, 0);
	flags = 0;
	while (TRUE) {
		switch (vala_parser_current (self)) {
			case VALA_TOKEN_TYPE_ABSTRACT:
			{
				vala_parser_next (self);
				flags = flags | VALA_PARSER_MODIFIER_FLAGS_ABSTRACT;
				break;
			}
			case VALA_TOKEN_TYPE_CLASS:
			{
				vala_parser_next (self);
				flags = flags | VALA_PARSER_MODIFIER_FLAGS_CLASS;
				break;
			}
			case VALA_TOKEN_TYPE_EXTERN:
			{
				vala_parser_next (self);
				flags = flags | VALA_PARSER_MODIFIER_FLAGS_EXTERN;
				break;
			}
			case VALA_TOKEN_TYPE_INLINE:
			{
				vala_parser_next (self);
				flags = flags | VALA_PARSER_MODIFIER_FLAGS_INLINE;
				break;
			}
			case VALA_TOKEN_TYPE_NEW:
			{
				vala_parser_next (self);
				flags = flags | VALA_PARSER_MODIFIER_FLAGS_NEW;
				break;
			}
			case VALA_TOKEN_TYPE_OVERRIDE:
			{
				vala_parser_next (self);
				flags = flags | VALA_PARSER_MODIFIER_FLAGS_OVERRIDE;
				break;
			}
			case VALA_TOKEN_TYPE_STATIC:
			{
				vala_parser_next (self);
				flags = flags | VALA_PARSER_MODIFIER_FLAGS_STATIC;
				break;
			}
			case VALA_TOKEN_TYPE_VIRTUAL:
			{
				vala_parser_next (self);
				flags = flags | VALA_PARSER_MODIFIER_FLAGS_VIRTUAL;
				break;
			}
			default:
			{
				return flags;
			}
		}
	}
}


static ValaFormalParameter* vala_parser_parse_parameter (ValaParser* self, GError** error) {
	GError * inner_error;
	GeeList* attrs;
	ValaSourceLocation begin;
	gboolean params_array;
	ValaParameterDirection direction;
	ValaDataType* type;
	char* id;
	ValaSourceReference* _tmp7;
	ValaFormalParameter* _tmp8;
	ValaFormalParameter* param;
	ValaFormalParameter* _tmp11;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	attrs = vala_parser_parse_attributes (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	begin = vala_parser_get_location (self);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_ELLIPSIS)) {
		ValaSourceReference* _tmp0;
		ValaFormalParameter* _tmp1;
		ValaFormalParameter* _tmp2;
		/* varargs*/
		_tmp0 = NULL;
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = (_tmp1 = vala_formal_parameter_new_with_ellipsis (_tmp0 = vala_parser_get_src (self, &begin)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1), (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), _tmp2);
	}
	params_array = vala_parser_accept (self, VALA_TOKEN_TYPE_PARAMS);
	direction = VALA_PARAMETER_DIRECTION_IN;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OUT)) {
		direction = VALA_PARAMETER_DIRECTION_OUT;
	} else {
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_REF)) {
			direction = VALA_PARAMETER_DIRECTION_REF;
		}
	}
	type = NULL;
	if (direction == VALA_PARAMETER_DIRECTION_IN) {
		ValaDataType* _tmp3;
		ValaDataType* _tmp4;
		_tmp3 = vala_parser_parse_type (self, FALSE, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				return NULL;
			} else {
				(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		/* in parameters are weak by default*/
		_tmp4 = NULL;
		type = (_tmp4 = _tmp3, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp4);
	} else {
		ValaDataType* _tmp5;
		ValaDataType* _tmp6;
		_tmp5 = vala_parser_parse_type (self, TRUE, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				return NULL;
			} else {
				(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		/* out parameters own the value by default*/
		_tmp6 = NULL;
		type = (_tmp6 = _tmp5, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp6);
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			return NULL;
		} else {
			(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp7 = NULL;
	_tmp8 = NULL;
	param = (_tmp8 = vala_formal_parameter_new (id, type, _tmp7 = vala_parser_get_src (self, &begin)), (_tmp7 == NULL) ? NULL : (_tmp7 = (vala_source_reference_unref (_tmp7), NULL)), _tmp8);
	vala_parser_set_attributes (self, (ValaCodeNode*) param, attrs);
	vala_formal_parameter_set_direction (param, direction);
	vala_formal_parameter_set_params_array (param, params_array);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_ASSIGN)) {
		ValaExpression* _tmp9;
		ValaExpression* _tmp10;
		_tmp9 = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				return NULL;
			} else {
				(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				id = (g_free (id), NULL);
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp10 = NULL;
		vala_formal_parameter_set_default_expression (param, _tmp10 = _tmp9);
		(_tmp10 == NULL) ? NULL : (_tmp10 = (vala_code_node_unref (_tmp10), NULL));
	}
	_tmp11 = NULL;
	return (_tmp11 = param, (attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), id = (g_free (id), NULL), _tmp11);
}


static ValaCreationMethod* vala_parser_parse_creation_method_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaUnresolvedSymbol* sym;
	ValaCreationMethod* method;
	ValaCreationMethod* _tmp10;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_NEW) == VALA_PARSER_MODIFIER_FLAGS_NEW) {
		char* _tmp0;
		GError* _tmp1;
		_tmp0 = NULL;
		_tmp1 = NULL;
		inner_error = (_tmp1 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp0 = vala_parser_get_error (self, "`new' modifier not allowed on creation method")), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				return NULL;
			} else {
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
	}
	method = NULL;
	if (vala_unresolved_symbol_get_inner (sym) == NULL) {
		ValaCreationMethod* _tmp3;
		ValaSourceReference* _tmp2;
		_tmp3 = NULL;
		_tmp2 = NULL;
		method = (_tmp3 = vala_creation_method_new (vala_symbol_get_name ((ValaSymbol*) sym), NULL, _tmp2 = vala_parser_get_src_com (self, &begin)), (method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL)), _tmp3);
		(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL));
	} else {
		ValaCreationMethod* _tmp5;
		ValaSourceReference* _tmp4;
		_tmp5 = NULL;
		_tmp4 = NULL;
		method = (_tmp5 = vala_creation_method_new (vala_symbol_get_name ((ValaSymbol*) vala_unresolved_symbol_get_inner (sym)), vala_symbol_get_name ((ValaSymbol*) sym), _tmp4 = vala_parser_get_src_com (self, &begin)), (method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL)), _tmp5);
		(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL));
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_EXTERN) == VALA_PARSER_MODIFIER_FLAGS_EXTERN) {
		vala_symbol_set_external ((ValaSymbol*) method, TRUE);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaFormalParameter* param;
			param = vala_parser_parse_parameter (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
					return NULL;
				} else {
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			vala_method_add_parameter ((ValaMethod*) method, param);
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
			return NULL;
		} else {
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_YIELDS)) {
		vala_method_set_coroutine ((ValaMethod*) method, TRUE);
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_THROWS)) {
		do {
			ValaDataType* _tmp6;
			ValaDataType* _tmp7;
			_tmp6 = vala_parser_parse_type (self, TRUE, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
					return NULL;
				} else {
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp7 = NULL;
			vala_code_node_add_error_type ((ValaCodeNode*) method, _tmp7 = _tmp6);
			(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_code_node_unref (_tmp7), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_symbol_set_access ((ValaSymbol*) method, access);
	vala_parser_set_attributes (self, (ValaCodeNode*) method, attrs);
	if (!vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
		ValaBlock* _tmp8;
		ValaBlock* _tmp9;
		_tmp8 = vala_parser_parse_block (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				return NULL;
			} else {
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				(method == NULL) ? NULL : (method = (vala_code_node_unref (method), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp9 = NULL;
		vala_method_set_body ((ValaMethod*) method, _tmp9 = _tmp8);
		(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL));
	} else {
		if (vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
			vala_symbol_set_external ((ValaSymbol*) method, TRUE);
		}
	}
	_tmp10 = NULL;
	return (_tmp10 = method, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp10);
}


static ValaSymbol* vala_parser_parse_delegate_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaDataType* type;
	ValaUnresolvedSymbol* sym;
	GeeList* type_param_list;
	ValaSourceReference* _tmp2;
	ValaDelegate* _tmp3;
	ValaDelegate* d;
	gboolean _tmp4;
	ValaSymbol* _tmp7;
	ValaSymbol* result;
	ValaSymbol* _tmp12;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_DELEGATE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_NEW) == VALA_PARSER_MODIFIER_FLAGS_NEW) {
		char* _tmp0;
		GError* _tmp1;
		_tmp0 = NULL;
		_tmp1 = NULL;
		inner_error = (_tmp1 = g_error_new_literal (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, _tmp0 = vala_parser_get_error (self, "`new' modifier not allowed on delegates")), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				return NULL;
			} else {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
	}
	type = vala_parser_parse_type (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			return NULL;
		} else {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	type_param_list = vala_parser_parse_type_parameter_list (self, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	d = (_tmp3 = vala_delegate_new (vala_symbol_get_name ((ValaSymbol*) sym), type, _tmp2 = vala_parser_get_src_com (self, &begin)), (_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)), _tmp3);
	vala_symbol_set_access ((ValaSymbol*) d, access);
	vala_parser_set_attributes (self, (ValaCodeNode*) d, attrs);
	if (!((flags & VALA_PARSER_MODIFIER_FLAGS_STATIC) == VALA_PARSER_MODIFIER_FLAGS_STATIC)) {
		vala_delegate_set_has_target (d, TRUE);
	}
	_tmp4 = FALSE;
	if ((flags & VALA_PARSER_MODIFIER_FLAGS_EXTERN) == VALA_PARSER_MODIFIER_FLAGS_EXTERN) {
		_tmp4 = TRUE;
	} else {
		_tmp4 = vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner));
	}
	if (_tmp4) {
		vala_symbol_set_external ((ValaSymbol*) d, TRUE);
	}
	{
		GeeIterator* _type_param_it;
		_type_param_it = gee_iterable_iterator ((GeeIterable*) type_param_list);
		while (gee_iterator_next (_type_param_it)) {
			ValaTypeParameter* type_param;
			type_param = (ValaTypeParameter*) gee_iterator_get (_type_param_it);
			vala_delegate_add_type_parameter (d, type_param);
			(type_param == NULL) ? NULL : (type_param = (vala_code_node_unref (type_param), NULL));
		}
		(_type_param_it == NULL) ? NULL : (_type_param_it = (gee_collection_object_unref (_type_param_it), NULL));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaFormalParameter* param;
			param = vala_parser_parse_parameter (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
					(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
					return NULL;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
					(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			vala_delegate_add_parameter (d, param);
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_THROWS)) {
		do {
			ValaDataType* _tmp5;
			ValaDataType* _tmp6;
			_tmp5 = vala_parser_parse_type (self, TRUE, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
					(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
					return NULL;
				} else {
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
					(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp6 = NULL;
			vala_code_node_add_error_type ((ValaCodeNode*) d, _tmp6 = _tmp5);
			(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		if (inner_error->domain == VALA_PARSE_ERROR) {
			g_propagate_error (error, inner_error);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
			return NULL;
		} else {
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			(type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL));
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	_tmp7 = NULL;
	result = (_tmp7 = (ValaSymbol*) d, (_tmp7 == NULL) ? NULL : vala_code_node_ref (_tmp7));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp9;
		ValaUnresolvedSymbol* _tmp8;
		ValaNamespace* ns;
		ValaSymbol* _tmp11;
		ValaSymbol* _tmp10;
		_tmp9 = NULL;
		_tmp8 = NULL;
		sym = (_tmp9 = (_tmp8 = vala_unresolved_symbol_get_inner (sym), (_tmp8 == NULL) ? NULL : vala_code_node_ref (_tmp8)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp9);
		ns = vala_namespace_new (vala_symbol_get_name ((ValaSymbol*) sym), vala_code_node_get_source_reference ((ValaCodeNode*) d));
		if (VALA_IS_NAMESPACE (result)) {
			vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		} else {
			vala_namespace_add_delegate (ns, VALA_DELEGATE (result));
			vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), (ValaCodeNode*) result);
		}
		_tmp11 = NULL;
		_tmp10 = NULL;
		result = (_tmp11 = (_tmp10 = (ValaSymbol*) ns, (_tmp10 == NULL) ? NULL : vala_code_node_ref (_tmp10)), (result == NULL) ? NULL : (result = (vala_code_node_unref (result), NULL)), _tmp11);
		(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
	}
	_tmp12 = NULL;
	return (_tmp12 = result, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (type_param_list == NULL) ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL)), (d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL)), _tmp12);
}


static GeeList* vala_parser_parse_type_parameter_list (ValaParser* self, GError** error) {
	GError * inner_error;
	GeeArrayList* list;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	list = gee_array_list_new (VALA_TYPE_TYPEPARAMETER, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OP_LT)) {
		do {
			ValaSourceLocation begin;
			char* id;
			ValaTypeParameter* _tmp1;
			ValaSourceReference* _tmp0;
			begin = vala_parser_get_location (self);
			id = vala_parser_parse_identifier (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
					return NULL;
				} else {
					(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp1 = NULL;
			_tmp0 = NULL;
			gee_collection_add ((GeeCollection*) list, _tmp1 = vala_typeparameter_new (id, _tmp0 = vala_parser_get_src (self, &begin)));
			(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
			(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
			id = (g_free (id), NULL);
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_OP_GT, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
				return NULL;
			} else {
				(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
	}
	return (GeeList*) list;
}


static void vala_parser_skip_type_argument_list (ValaParser* self, GError** error) {
	GError * inner_error;
	g_return_if_fail (self != NULL);
	inner_error = NULL;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OP_LT)) {
		do {
			vala_parser_skip_type (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					return;
				} else {
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return;
				}
			}
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_OP_GT, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				return;
			} else {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return;
			}
		}
	}
}


/* try to parse type argument list*/
static GeeList* vala_parser_parse_type_argument_list (ValaParser* self, gboolean maybe_expression, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OP_LT)) {
		GeeArrayList* list;
		list = gee_array_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
		do {
			switch (vala_parser_current (self)) {
				case VALA_TOKEN_TYPE_VOID:
				case VALA_TOKEN_TYPE_DYNAMIC:
				case VALA_TOKEN_TYPE_UNOWNED:
				case VALA_TOKEN_TYPE_WEAK:
				case VALA_TOKEN_TYPE_IDENTIFIER:
				{
					ValaDataType* type;
					type = vala_parser_parse_type (self, TRUE, &inner_error);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							g_propagate_error (error, inner_error);
							(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
							return NULL;
						} else {
							(list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL));
							g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
							g_clear_error (&inner_error);
							return NULL;
						}
					}
					gee_collection_add ((GeeCollection*) list, type);
					(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
					break;
				}
				default:
				{
					GeeList* _tmp0;
					vala_parser_rollback (self, &begin);
					_tmp0 = NULL;
					return (_tmp0 = NULL, (list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL)), _tmp0);
				}
			}
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		if (!vala_parser_accept (self, VALA_TOKEN_TYPE_OP_GT)) {
			GeeList* _tmp1;
			vala_parser_rollback (self, &begin);
			_tmp1 = NULL;
			return (_tmp1 = NULL, (list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL)), _tmp1);
		}
		if (maybe_expression) {
			/* check follower to decide whether to keep type argument list*/
			switch (vala_parser_current (self)) {
				case VALA_TOKEN_TYPE_OPEN_PARENS:
				case VALA_TOKEN_TYPE_CLOSE_PARENS:
				case VALA_TOKEN_TYPE_CLOSE_BRACKET:
				case VALA_TOKEN_TYPE_COLON:
				case VALA_TOKEN_TYPE_SEMICOLON:
				case VALA_TOKEN_TYPE_COMMA:
				case VALA_TOKEN_TYPE_DOT:
				case VALA_TOKEN_TYPE_INTERR:
				case VALA_TOKEN_TYPE_OP_EQ:
				case VALA_TOKEN_TYPE_OP_NE:
				{
					break;
				}
				default:
				{
					GeeList* _tmp2;
					vala_parser_rollback (self, &begin);
					_tmp2 = NULL;
					return (_tmp2 = NULL, (list == NULL) ? NULL : (list = (gee_collection_object_unref (list), NULL)), _tmp2);
				}
			}
		}
		return (GeeList*) list;
	}
	return NULL;
}


static ValaMemberAccess* vala_parser_parse_member_name (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaMemberAccess* expr;
	gboolean first;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	expr = NULL;
	first = TRUE;
	do {
		char* id;
		gboolean qualified;
		gboolean _tmp0;
		gboolean _tmp1;
		GeeList* type_arg_list;
		ValaMemberAccess* _tmp5;
		ValaSourceReference* _tmp4;
		id = vala_parser_parse_identifier (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
				return NULL;
			} else {
				(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		/* The first member access can be global:: qualified*/
		qualified = FALSE;
		_tmp0 = FALSE;
		_tmp1 = FALSE;
		if (first) {
			_tmp1 = _vala_strcmp0 (id, "global") == 0;
		} else {
			_tmp1 = FALSE;
		}
		if (_tmp1) {
			_tmp0 = vala_parser_accept (self, VALA_TOKEN_TYPE_DOUBLE_COLON);
		} else {
			_tmp0 = FALSE;
		}
		if (_tmp0) {
			char* _tmp2;
			char* _tmp3;
			_tmp2 = vala_parser_parse_identifier (self, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == VALA_PARSE_ERROR) {
					g_propagate_error (error, inner_error);
					id = (g_free (id), NULL);
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					return NULL;
				} else {
					id = (g_free (id), NULL);
					(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
					return NULL;
				}
			}
			_tmp3 = NULL;
			id = (_tmp3 = _tmp2, id = (g_free (id), NULL), _tmp3);
			qualified = TRUE;
		}
		type_arg_list = vala_parser_parse_type_argument_list (self, FALSE, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				g_propagate_error (error, inner_error);
				id = (g_free (id), NULL);
				(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
				return NULL;
			} else {
				id = (g_free (id), NULL);
				(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return NULL;
			}
		}
		_tmp5 = NULL;
		_tmp4 = NULL;
		expr = (_tmp5 = vala_member_access_new ((ValaExpression*) expr, id, _tmp4 = vala_parser_get_src (self, &begin)), (expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL)), _tmp5);
		(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL));
		vala_member_access_set_qualified (expr, qualified);
		if (type_arg_list != NULL) {
			{
				GeeIterator* _type_arg_it;
				_type_arg_it = gee_iterable_iterator ((GeeIterable*) type_arg_list);
				while (gee_iterator_next (_type_arg_it)) {
					ValaDataType* type_arg;
					type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
					vala_member_access_add_type_argument (expr, type_arg);
					(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
				}
				(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
			}
		}
		first = FALSE;
		id = (g_free (id), NULL);
		(type_arg_list == NULL) ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL));
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_DOT));
	return expr;
}


static gboolean vala_parser_is_declaration_keyword (ValaParser* self, ValaTokenType type) {
	g_return_val_if_fail (self != NULL, FALSE);
	switch (type) {
		case VALA_TOKEN_TYPE_ABSTRACT:
		case VALA_TOKEN_TYPE_CLASS:
		case VALA_TOKEN_TYPE_CONST:
		case VALA_TOKEN_TYPE_DELEGATE:
		case VALA_TOKEN_TYPE_ENUM:
		case VALA_TOKEN_TYPE_ERRORDOMAIN:
		case VALA_TOKEN_TYPE_EXTERN:
		case VALA_TOKEN_TYPE_INLINE:
		case VALA_TOKEN_TYPE_INTERFACE:
		case VALA_TOKEN_TYPE_INTERNAL:
		case VALA_TOKEN_TYPE_NAMESPACE:
		case VALA_TOKEN_TYPE_NEW:
		case VALA_TOKEN_TYPE_OVERRIDE:
		case VALA_TOKEN_TYPE_PRIVATE:
		case VALA_TOKEN_TYPE_PROTECTED:
		case VALA_TOKEN_TYPE_PUBLIC:
		case VALA_TOKEN_TYPE_SIGNAL:
		case VALA_TOKEN_TYPE_STATIC:
		case VALA_TOKEN_TYPE_STRUCT:
		case VALA_TOKEN_TYPE_VIRTUAL:
		case VALA_TOKEN_TYPE_VOLATILE:
		{
			return TRUE;
		}
		default:
		{
			return FALSE;
		}
	}
}


static ValaParserTokenInfo* vala_parser_token_info_dup (const ValaParserTokenInfo* self) {
	ValaParserTokenInfo* dup;
	dup = g_new0 (ValaParserTokenInfo, 1);
	memcpy (dup, self, sizeof (ValaParserTokenInfo));
	return dup;
}


static void vala_parser_token_info_free (ValaParserTokenInfo* self) {
	g_free (self);
}


static GType vala_parser_token_info_get_type (void) {
	static GType vala_parser_token_info_type_id = 0;
	if (vala_parser_token_info_type_id == 0) {
		vala_parser_token_info_type_id = g_boxed_type_register_static ("ValaParserTokenInfo", (GBoxedCopyFunc) vala_parser_token_info_dup, (GBoxedFreeFunc) vala_parser_token_info_free);
	}
	return vala_parser_token_info_type_id;
}


static void vala_parser_class_init (ValaParserClass * klass) {
	vala_parser_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_parser_finalize;
	g_type_class_add_private (klass, sizeof (ValaParserPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_parser_real_visit_source_file;
}


static void vala_parser_instance_init (ValaParser * self) {
	self->priv = VALA_PARSER_GET_PRIVATE (self);
}


static void vala_parser_finalize (ValaCodeVisitor* obj) {
	ValaParser * self;
	self = VALA_PARSER (obj);
	(self->priv->scanner == NULL) ? NULL : (self->priv->scanner = (vala_scanner_unref (self->priv->scanner), NULL));
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
	self->priv->tokens = (g_free (self->priv->tokens), NULL);
	self->priv->comment = (g_free (self->priv->comment), NULL);
	VALA_CODE_VISITOR_CLASS (vala_parser_parent_class)->finalize (obj);
}


GType vala_parser_get_type (void) {
	static GType vala_parser_type_id = 0;
	if (vala_parser_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaParser), 0, (GInstanceInitFunc) vala_parser_instance_init, NULL };
		vala_parser_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaParser", &g_define_type_info, 0);
	}
	return vala_parser_type_id;
}


GQuark vala_parse_error_quark (void) {
	return g_quark_from_static_string ("vala_parse_error-quark");
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




