/* valamethodtype.vala
 *
 * Copyright (C) 2007-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaDataTypePrivate ValaDataTypePrivate;

#define VALA_TYPE_SCOPE (vala_scope_get_type ())
#define VALA_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SCOPE, ValaScope))
#define VALA_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SCOPE, ValaScopeClass))
#define VALA_IS_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SCOPE))
#define VALA_IS_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SCOPE))
#define VALA_SCOPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SCOPE, ValaScopeClass))

typedef struct _ValaScope ValaScope;
typedef struct _ValaScopeClass ValaScopeClass;

#define VALA_TYPE_FORMAL_PARAMETER (vala_formal_parameter_get_type ())
#define VALA_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameter))
#define VALA_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))
#define VALA_IS_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_IS_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_FORMAL_PARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))

typedef struct _ValaFormalParameter ValaFormalParameter;
typedef struct _ValaFormalParameterClass ValaFormalParameterClass;

#define VALA_TYPE_METHOD_TYPE (vala_method_type_get_type ())
#define VALA_METHOD_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_METHOD_TYPE, ValaMethodType))
#define VALA_METHOD_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_METHOD_TYPE, ValaMethodTypeClass))
#define VALA_IS_METHOD_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_METHOD_TYPE))
#define VALA_IS_METHOD_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_METHOD_TYPE))
#define VALA_METHOD_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_METHOD_TYPE, ValaMethodTypeClass))

typedef struct _ValaMethodType ValaMethodType;
typedef struct _ValaMethodTypeClass ValaMethodTypeClass;
typedef struct _ValaMethodTypePrivate ValaMethodTypePrivate;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;

#define VALA_TYPE_METHOD (vala_method_get_type ())
#define VALA_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_METHOD, ValaMethod))
#define VALA_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_METHOD, ValaMethodClass))
#define VALA_IS_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_METHOD))
#define VALA_IS_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_METHOD))
#define VALA_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_METHOD, ValaMethodClass))

typedef struct _ValaMethod ValaMethod;
typedef struct _ValaMethodClass ValaMethodClass;

#define VALA_TYPE_DELEGATE_TYPE (vala_delegate_type_get_type ())
#define VALA_DELEGATE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DELEGATE_TYPE, ValaDelegateType))
#define VALA_DELEGATE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DELEGATE_TYPE, ValaDelegateTypeClass))
#define VALA_IS_DELEGATE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DELEGATE_TYPE))
#define VALA_IS_DELEGATE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DELEGATE_TYPE))
#define VALA_DELEGATE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DELEGATE_TYPE, ValaDelegateTypeClass))

typedef struct _ValaDelegateType ValaDelegateType;
typedef struct _ValaDelegateTypeClass ValaDelegateTypeClass;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;

#define VALA_TYPE_DELEGATE (vala_delegate_get_type ())
#define VALA_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DELEGATE, ValaDelegate))
#define VALA_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DELEGATE, ValaDelegateClass))
#define VALA_IS_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DELEGATE))
#define VALA_IS_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DELEGATE))
#define VALA_DELEGATE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DELEGATE, ValaDelegateClass))

typedef struct _ValaDelegate ValaDelegate;
typedef struct _ValaDelegateClass ValaDelegateClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * A reference to a data type. This is used to specify static types of
 * expressions.
 */
struct _ValaDataType {
	ValaCodeNode parent_instance;
	ValaDataTypePrivate * priv;
};

struct _ValaDataTypeClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cname) (ValaDataType* self);
	char* (*get_lower_case_cname) (ValaDataType* self, const char* infix);
	char* (*to_qualified_string) (ValaDataType* self, ValaScope* scope);
	ValaDataType* (*copy) (ValaDataType* self);
	gboolean (*equals) (ValaDataType* self, ValaDataType* type2);
	gboolean (*stricter) (ValaDataType* self, ValaDataType* type2);
	gboolean (*compatible) (ValaDataType* self, ValaDataType* target_type);
	gboolean (*is_invokable) (ValaDataType* self);
	ValaDataType* (*get_return_type) (ValaDataType* self);
	GeeList* (*get_parameters) (ValaDataType* self);
	gboolean (*is_reference_type_or_type_parameter) (ValaDataType* self);
	gboolean (*is_array) (ValaDataType* self);
	GeeList* (*get_symbols) (ValaDataType* self);
	ValaSymbol* (*get_member) (ValaDataType* self, const char* member_name);
	ValaSymbol* (*get_pointer_member) (ValaDataType* self, const char* member_name);
	gboolean (*is_real_struct_type) (ValaDataType* self);
	char* (*get_type_id) (ValaDataType* self);
	char* (*get_type_signature) (ValaDataType* self);
	gboolean (*is_disposable) (ValaDataType* self);
};

/**
 * The type of a method referencea.
 */
struct _ValaMethodType {
	ValaDataType parent_instance;
	ValaMethodTypePrivate * priv;
};

struct _ValaMethodTypeClass {
	ValaDataTypeClass parent_class;
};

struct _ValaMethodTypePrivate {
	ValaMethod* _method_symbol;
};



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
gpointer vala_scope_ref (gpointer instance);
void vala_scope_unref (gpointer instance);
GParamSpec* vala_param_spec_scope (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_scope (GValue* value, gpointer v_object);
gpointer vala_value_get_scope (const GValue* value);
GType vala_scope_get_type (void);
GType vala_formal_parameter_get_type (void);
GType vala_method_type_get_type (void);
GType vala_member_get_type (void);
GType vala_method_get_type (void);
#define VALA_METHOD_TYPE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_METHOD_TYPE, ValaMethodTypePrivate))
enum  {
	VALA_METHOD_TYPE_DUMMY_PROPERTY
};
void vala_method_type_set_method_symbol (ValaMethodType* self, ValaMethod* value);
ValaMethodType* vala_method_type_new (ValaMethod* method_symbol);
ValaMethodType* vala_method_type_construct (GType object_type, ValaMethod* method_symbol);
ValaMethodType* vala_method_type_new (ValaMethod* method_symbol);
static gboolean vala_method_type_real_is_invokable (ValaDataType* base);
ValaMethod* vala_method_type_get_method_symbol (ValaMethodType* self);
ValaDataType* vala_method_get_return_type (ValaMethod* self);
static ValaDataType* vala_method_type_real_get_return_type (ValaDataType* base);
GeeList* vala_method_get_parameters (ValaMethod* self);
static GeeList* vala_method_type_real_get_parameters (ValaDataType* base);
static ValaDataType* vala_method_type_real_copy (ValaDataType* base);
GType vala_delegate_type_get_type (void);
GType vala_typesymbol_get_type (void);
GType vala_delegate_get_type (void);
ValaDelegate* vala_delegate_type_get_delegate_symbol (ValaDelegateType* self);
gboolean vala_delegate_matches_method (ValaDelegate* self, ValaMethod* m);
static gboolean vala_method_type_real_compatible (ValaDataType* base, ValaDataType* target_type);
char* vala_symbol_get_full_name (ValaSymbol* self);
static char* vala_method_type_real_to_qualified_string (ValaDataType* base, ValaScope* scope);
static char* vala_method_type_real_get_cname (ValaDataType* base);
gboolean vala_method_get_coroutine (ValaMethod* self);
static ValaSymbol* vala_method_type_real_get_member (ValaDataType* base, const char* member_name);
static gpointer vala_method_type_parent_class = NULL;
static void vala_method_type_finalize (ValaCodeNode* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



ValaMethodType* vala_method_type_construct (GType object_type, ValaMethod* method_symbol) {
	ValaMethodType* self;
	g_return_val_if_fail (method_symbol != NULL, NULL);
	self = (ValaMethodType*) g_type_create_instance (object_type);
	vala_method_type_set_method_symbol (self, method_symbol);
	return self;
}


ValaMethodType* vala_method_type_new (ValaMethod* method_symbol) {
	return vala_method_type_construct (VALA_TYPE_METHOD_TYPE, method_symbol);
}


static gboolean vala_method_type_real_is_invokable (ValaDataType* base) {
	ValaMethodType * self;
	self = (ValaMethodType*) base;
	return TRUE;
}


static ValaDataType* vala_method_type_real_get_return_type (ValaDataType* base) {
	ValaMethodType * self;
	ValaDataType* _tmp0;
	self = (ValaMethodType*) base;
	_tmp0 = NULL;
	return (_tmp0 = vala_method_get_return_type (self->priv->_method_symbol), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
}


static GeeList* vala_method_type_real_get_parameters (ValaDataType* base) {
	ValaMethodType * self;
	self = (ValaMethodType*) base;
	return vala_method_get_parameters (self->priv->_method_symbol);
}


static ValaDataType* vala_method_type_real_copy (ValaDataType* base) {
	ValaMethodType * self;
	self = (ValaMethodType*) base;
	return (ValaDataType*) vala_method_type_new (self->priv->_method_symbol);
}


static gboolean vala_method_type_real_compatible (ValaDataType* base, ValaDataType* target_type) {
	ValaMethodType * self;
	ValaDelegateType* _tmp1;
	ValaDataType* _tmp0;
	ValaDelegateType* dt;
	gboolean _tmp3;
	self = (ValaMethodType*) base;
	g_return_val_if_fail (target_type != NULL, FALSE);
	_tmp1 = NULL;
	_tmp0 = NULL;
	dt = (_tmp1 = (_tmp0 = target_type, VALA_IS_DELEGATE_TYPE (_tmp0) ? ((ValaDelegateType*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	if (dt == NULL) {
		gboolean _tmp2;
		/* method types incompatible to anything but delegates*/
		return (_tmp2 = FALSE, (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), _tmp2);
	}
	return (_tmp3 = vala_delegate_matches_method (vala_delegate_type_get_delegate_symbol (dt), self->priv->_method_symbol), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), _tmp3);
}


static char* vala_method_type_real_to_qualified_string (ValaDataType* base, ValaScope* scope) {
	ValaMethodType * self;
	self = (ValaMethodType*) base;
	return vala_symbol_get_full_name ((ValaSymbol*) self->priv->_method_symbol);
}


static char* vala_method_type_real_get_cname (ValaDataType* base) {
	ValaMethodType * self;
	self = (ValaMethodType*) base;
	return g_strdup ("gpointer");
}


static ValaSymbol* vala_method_type_real_get_member (ValaDataType* base, const char* member_name) {
	ValaMethodType * self;
	gboolean _tmp0;
	self = (ValaMethodType*) base;
	g_return_val_if_fail (member_name != NULL, NULL);
	_tmp0 = FALSE;
	if (vala_method_get_coroutine (self->priv->_method_symbol)) {
		_tmp0 = _vala_strcmp0 (member_name, "begin") == 0;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		ValaSymbol* _tmp1;
		_tmp1 = NULL;
		return (_tmp1 = (ValaSymbol*) self->priv->_method_symbol, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	}
	return NULL;
}


ValaMethod* vala_method_type_get_method_symbol (ValaMethodType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_method_symbol;
}


void vala_method_type_set_method_symbol (ValaMethodType* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_method_symbol = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_method_symbol == NULL) ? NULL : (self->priv->_method_symbol = (vala_code_node_unref (self->priv->_method_symbol), NULL)), _tmp2);
}


static void vala_method_type_class_init (ValaMethodTypeClass * klass) {
	vala_method_type_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_method_type_finalize;
	g_type_class_add_private (klass, sizeof (ValaMethodTypePrivate));
	VALA_DATA_TYPE_CLASS (klass)->is_invokable = vala_method_type_real_is_invokable;
	VALA_DATA_TYPE_CLASS (klass)->get_return_type = vala_method_type_real_get_return_type;
	VALA_DATA_TYPE_CLASS (klass)->get_parameters = vala_method_type_real_get_parameters;
	VALA_DATA_TYPE_CLASS (klass)->copy = vala_method_type_real_copy;
	VALA_DATA_TYPE_CLASS (klass)->compatible = vala_method_type_real_compatible;
	VALA_DATA_TYPE_CLASS (klass)->to_qualified_string = vala_method_type_real_to_qualified_string;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_method_type_real_get_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_member = vala_method_type_real_get_member;
}


static void vala_method_type_instance_init (ValaMethodType * self) {
	self->priv = VALA_METHOD_TYPE_GET_PRIVATE (self);
}


static void vala_method_type_finalize (ValaCodeNode* obj) {
	ValaMethodType * self;
	self = VALA_METHOD_TYPE (obj);
	(self->priv->_method_symbol == NULL) ? NULL : (self->priv->_method_symbol = (vala_code_node_unref (self->priv->_method_symbol), NULL));
	VALA_CODE_NODE_CLASS (vala_method_type_parent_class)->finalize (obj);
}


GType vala_method_type_get_type (void) {
	static GType vala_method_type_type_id = 0;
	if (vala_method_type_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMethodTypeClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_method_type_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMethodType), 0, (GInstanceInitFunc) vala_method_type_instance_init, NULL };
		vala_method_type_type_id = g_type_register_static (VALA_TYPE_DATA_TYPE, "ValaMethodType", &g_define_type_info, 0);
	}
	return vala_method_type_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




