/* valaproperty.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>
#include <valaccode.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;
typedef struct _ValaMemberPrivate ValaMemberPrivate;

#define VALA_TYPE_LOCKABLE (vala_lockable_get_type ())
#define VALA_LOCKABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCKABLE, ValaLockable))
#define VALA_IS_LOCKABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCKABLE))
#define VALA_LOCKABLE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), VALA_TYPE_LOCKABLE, ValaLockableIface))

typedef struct _ValaLockable ValaLockable;
typedef struct _ValaLockableIface ValaLockableIface;

#define VALA_TYPE_PROPERTY (vala_property_get_type ())
#define VALA_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_PROPERTY, ValaProperty))
#define VALA_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_PROPERTY, ValaPropertyClass))
#define VALA_IS_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_PROPERTY))
#define VALA_IS_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_PROPERTY))
#define VALA_PROPERTY_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_PROPERTY, ValaPropertyClass))

typedef struct _ValaProperty ValaProperty;
typedef struct _ValaPropertyClass ValaPropertyClass;
typedef struct _ValaPropertyPrivate ValaPropertyPrivate;

#define VALA_TYPE_FORMAL_PARAMETER (vala_formal_parameter_get_type ())
#define VALA_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameter))
#define VALA_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))
#define VALA_IS_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_IS_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_FORMAL_PARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))

typedef struct _ValaFormalParameter ValaFormalParameter;
typedef struct _ValaFormalParameterClass ValaFormalParameterClass;

#define VALA_TYPE_FIELD (vala_field_get_type ())
#define VALA_FIELD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FIELD, ValaField))
#define VALA_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FIELD, ValaFieldClass))
#define VALA_IS_FIELD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FIELD))
#define VALA_IS_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FIELD))
#define VALA_FIELD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FIELD, ValaFieldClass))

typedef struct _ValaField ValaField;
typedef struct _ValaFieldClass ValaFieldClass;

#define TYPE_MEMBER_BINDING (member_binding_get_type ())

#define VALA_TYPE_PROPERTY_ACCESSOR (vala_property_accessor_get_type ())
#define VALA_PROPERTY_ACCESSOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessor))
#define VALA_PROPERTY_ACCESSOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessorClass))
#define VALA_IS_PROPERTY_ACCESSOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_PROPERTY_ACCESSOR))
#define VALA_IS_PROPERTY_ACCESSOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_PROPERTY_ACCESSOR))
#define VALA_PROPERTY_ACCESSOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessorClass))

typedef struct _ValaPropertyAccessor ValaPropertyAccessor;
typedef struct _ValaPropertyAccessorClass ValaPropertyAccessorClass;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;

#define VALA_TYPE_OBJECT_TYPE_SYMBOL (vala_object_type_symbol_get_type ())
#define VALA_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbol))
#define VALA_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))
#define VALA_IS_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_IS_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_OBJECT_TYPE_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))

typedef struct _ValaObjectTypeSymbol ValaObjectTypeSymbol;
typedef struct _ValaObjectTypeSymbolClass ValaObjectTypeSymbolClass;

#define VALA_TYPE_CLASS (vala_class_get_type ())
#define VALA_CLASS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CLASS, ValaClass))
#define VALA_CLASS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CLASS, ValaClassClass))
#define VALA_IS_CLASS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CLASS))
#define VALA_IS_CLASS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CLASS))
#define VALA_CLASS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CLASS, ValaClassClass))

typedef struct _ValaClass ValaClass;
typedef struct _ValaClassClass ValaClassClass;

#define VALA_TYPE_INTERFACE (vala_interface_get_type ())
#define VALA_INTERFACE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_INTERFACE, ValaInterface))
#define VALA_INTERFACE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_INTERFACE, ValaInterfaceClass))
#define VALA_IS_INTERFACE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_INTERFACE))
#define VALA_IS_INTERFACE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_INTERFACE))
#define VALA_INTERFACE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_INTERFACE, ValaInterfaceClass))

typedef struct _ValaInterface ValaInterface;
typedef struct _ValaInterfaceClass ValaInterfaceClass;

#define VALA_TYPE_SCOPE (vala_scope_get_type ())
#define VALA_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SCOPE, ValaScope))
#define VALA_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SCOPE, ValaScopeClass))
#define VALA_IS_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SCOPE))
#define VALA_IS_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SCOPE))
#define VALA_SCOPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SCOPE, ValaScopeClass))

typedef struct _ValaScope ValaScope;
typedef struct _ValaScopeClass ValaScopeClass;

#define VALA_TYPE_SOURCE_FILE (vala_source_file_get_type ())
#define VALA_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFile))
#define VALA_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))
#define VALA_IS_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_FILE))
#define VALA_IS_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_FILE))
#define VALA_SOURCE_FILE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))

typedef struct _ValaSourceFile ValaSourceFile;
typedef struct _ValaSourceFileClass ValaSourceFileClass;

#define VALA_TYPE_SYMBOL_ACCESSIBILITY (vala_symbol_accessibility_get_type ())

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Represents a node in the symbol tree.
 */
struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
};

/**
 * Represents a general class member.
 */
struct _ValaMember {
	ValaSymbol parent_instance;
	ValaMemberPrivate * priv;
};

struct _ValaMemberClass {
	ValaSymbolClass parent_class;
};

/**
 * Represents a lockable object.
 */
struct _ValaLockableIface {
	GTypeInterface parent_iface;
	gboolean (*get_lock_used) (ValaLockable* self);
	void (*set_lock_used) (ValaLockable* self, gboolean used);
};

/**
 * Represents a property declaration in the source code.
 */
struct _ValaProperty {
	ValaMember parent_instance;
	ValaPropertyPrivate * priv;
};

struct _ValaPropertyClass {
	ValaMemberClass parent_class;
};

typedef enum  {
	MEMBER_BINDING_INSTANCE,
	MEMBER_BINDING_CLASS,
	MEMBER_BINDING_STATIC
} MemberBinding;

struct _ValaPropertyPrivate {
	ValaFormalParameter* _this_parameter;
	gboolean _notify;
	gboolean _no_accessor_method;
	gboolean _interface_only;
	gboolean _is_abstract;
	gboolean _is_virtual;
	gboolean _overrides;
	ValaField* _field;
	MemberBinding _binding;
	ValaExpression* _default_expression;
	gboolean lock_used;
	ValaDataType* _data_type;
	char* _nick;
	char* _blurb;
	ValaProperty* _base_property;
	ValaProperty* _base_interface_property;
	gboolean base_properties_valid;
	ValaPropertyAccessor* _get_accessor;
	ValaPropertyAccessor* _set_accessor;
};

typedef enum  {
	VALA_SYMBOL_ACCESSIBILITY_PRIVATE,
	VALA_SYMBOL_ACCESSIBILITY_INTERNAL,
	VALA_SYMBOL_ACCESSIBILITY_PROTECTED,
	VALA_SYMBOL_ACCESSIBILITY_PUBLIC
} ValaSymbolAccessibility;



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_member_get_type (void);
GType vala_lockable_get_type (void);
GType vala_property_get_type (void);
GType vala_formal_parameter_get_type (void);
GType vala_field_get_type (void);
GType member_binding_get_type (void);
GType vala_property_accessor_get_type (void);
#define VALA_PROPERTY_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_PROPERTY, ValaPropertyPrivate))
enum  {
	VALA_PROPERTY_DUMMY_PROPERTY
};
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
ValaMember* vala_member_new (const char* name, ValaSourceReference* source_reference);
ValaMember* vala_member_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
void vala_property_set_property_type (ValaProperty* self, ValaDataType* value);
void vala_property_set_get_accessor (ValaProperty* self, ValaPropertyAccessor* value);
void vala_property_set_set_accessor (ValaProperty* self, ValaPropertyAccessor* value);
ValaProperty* vala_property_new (const char* name, ValaDataType* property_type, ValaPropertyAccessor* get_accessor, ValaPropertyAccessor* set_accessor, ValaSourceReference* source_reference);
ValaProperty* vala_property_construct (GType object_type, const char* name, ValaDataType* property_type, ValaPropertyAccessor* get_accessor, ValaPropertyAccessor* set_accessor, ValaSourceReference* source_reference);
ValaProperty* vala_property_new (const char* name, ValaDataType* property_type, ValaPropertyAccessor* get_accessor, ValaPropertyAccessor* set_accessor, ValaSourceReference* source_reference);
void vala_code_visitor_visit_member (ValaCodeVisitor* self, ValaMember* m);
void vala_code_visitor_visit_property (ValaCodeVisitor* self, ValaProperty* prop);
static void vala_property_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaDataType* vala_property_get_property_type (ValaProperty* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
ValaPropertyAccessor* vala_property_get_get_accessor (ValaProperty* self);
ValaPropertyAccessor* vala_property_get_set_accessor (ValaProperty* self);
ValaExpression* vala_property_get_default_expression (ValaProperty* self);
static void vala_property_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
char* vala_symbol_get_lower_case_cname (ValaSymbol* self, const char* infix);
char* vala_symbol_camel_case_to_lower_case (const char* camel_case);
const char* vala_symbol_get_name (ValaSymbol* self);
char* vala_property_get_upper_case_cname (ValaProperty* self);
static char* vala_property_get_canonical_name (ValaProperty* self);
ValaCCodeConstant* vala_property_get_canonical_cconstant (ValaProperty* self);
gboolean vala_attribute_has_argument (ValaAttribute* self, const char* name);
gboolean vala_attribute_get_bool (ValaAttribute* self, const char* name);
void vala_property_set_notify (ValaProperty* self, gboolean value);
static void vala_property_process_ccode_attribute (ValaProperty* self, ValaAttribute* a);
const char* vala_attribute_get_name (ValaAttribute* self);
void vala_property_set_no_accessor_method (ValaProperty* self, gboolean value);
char* vala_attribute_get_string (ValaAttribute* self, const char* name);
void vala_property_set_nick (ValaProperty* self, const char* value);
void vala_property_set_blurb (ValaProperty* self, const char* value);
void vala_property_process_attributes (ValaProperty* self);
static gboolean vala_property_real_get_lock_used (ValaLockable* base);
static void vala_property_real_set_lock_used (ValaLockable* base, gboolean used);
gboolean vala_data_type_equals (ValaDataType* self, ValaDataType* type2);
gboolean vala_property_accessor_get_writable (ValaPropertyAccessor* self);
gboolean vala_property_accessor_get_construction (ValaPropertyAccessor* self);
gboolean vala_property_equals (ValaProperty* self, ValaProperty* prop2);
static void vala_property_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
GType vala_typesymbol_get_type (void);
GType vala_object_type_symbol_get_type (void);
GType vala_class_get_type (void);
static void vala_property_find_base_interface_property (ValaProperty* self, ValaClass* cl);
gboolean vala_property_get_is_virtual (ValaProperty* self);
gboolean vala_property_get_overrides (ValaProperty* self);
static void vala_property_find_base_class_property (ValaProperty* self, ValaClass* cl);
GType vala_interface_get_type (void);
gboolean vala_property_get_is_abstract (ValaProperty* self);
static void vala_property_find_base_properties (ValaProperty* self);
gpointer vala_scope_ref (gpointer instance);
void vala_scope_unref (gpointer instance);
GParamSpec* vala_param_spec_scope (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_scope (GValue* value, gpointer v_object);
gpointer vala_value_get_scope (const GValue* value);
GType vala_scope_get_type (void);
ValaScope* vala_symbol_get_scope (ValaSymbol* self);
ValaSymbol* vala_scope_lookup (ValaScope* self, const char* name);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
void vala_report_error (ValaSourceReference* source, const char* message);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
char* vala_symbol_get_full_name (ValaSymbol* self);
ValaClass* vala_class_get_base_class (ValaClass* self);
GeeList* vala_class_get_base_types (ValaClass* self);
ValaTypeSymbol* vala_data_type_get_data_type (ValaDataType* self);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gpointer vala_source_file_ref (gpointer instance);
void vala_source_file_unref (gpointer instance);
GParamSpec* vala_param_spec_source_file (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_file (GValue* value, gpointer v_object);
gpointer vala_value_get_source_file (const GValue* value);
GType vala_source_file_get_type (void);
ValaSourceFile* vala_semantic_analyzer_get_current_source_file (ValaSemanticAnalyzer* self);
ValaSymbol* vala_semantic_analyzer_get_current_symbol (ValaSemanticAnalyzer* self);
ValaSourceFile* vala_source_reference_get_file (ValaSourceReference* self);
void vala_semantic_analyzer_set_current_source_file (ValaSemanticAnalyzer* self, ValaSourceFile* value);
void vala_semantic_analyzer_set_current_symbol (ValaSemanticAnalyzer* self, ValaSymbol* value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
gboolean vala_semantic_analyzer_is_type_accessible (ValaSemanticAnalyzer* self, ValaSymbol* sym, ValaDataType* type);
char* vala_code_node_to_string (ValaCodeNode* self);
ValaProperty* vala_property_get_base_property (ValaProperty* self);
gboolean vala_symbol_get_external_package (ValaSymbol* self);
gboolean vala_member_get_hides (ValaMember* self);
ValaSymbol* vala_member_get_hidden_member (ValaMember* self);
void vala_report_warning (ValaSourceReference* source, const char* message);
GType vala_symbol_accessibility_get_type (void);
ValaSymbolAccessibility vala_symbol_get_access (ValaSymbol* self);
ValaDataType* vala_expression_get_value_type (ValaExpression* self);
gboolean vala_data_type_compatible (ValaDataType* self, ValaDataType* target_type);
static gboolean vala_property_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
void vala_property_accessor_set_prop (ValaPropertyAccessor* self, ValaProperty* value);
ValaFormalParameter* vala_property_get_this_parameter (ValaProperty* self);
void vala_property_set_this_parameter (ValaProperty* self, ValaFormalParameter* value);
gboolean vala_property_get_notify (ValaProperty* self);
gboolean vala_property_get_no_accessor_method (ValaProperty* self);
gboolean vala_property_get_interface_only (ValaProperty* self);
void vala_property_set_interface_only (ValaProperty* self, gboolean value);
void vala_property_set_is_abstract (ValaProperty* self, gboolean value);
void vala_property_set_is_virtual (ValaProperty* self, gboolean value);
void vala_property_set_overrides (ValaProperty* self, gboolean value);
ValaField* vala_property_get_field (ValaProperty* self);
void vala_property_set_field (ValaProperty* self, ValaField* value);
MemberBinding vala_property_get_binding (ValaProperty* self);
void vala_property_set_binding (ValaProperty* self, MemberBinding value);
ValaProperty* vala_property_get_base_interface_property (ValaProperty* self);
void vala_property_set_default_expression (ValaProperty* self, ValaExpression* value);
const char* vala_property_get_nick (ValaProperty* self);
const char* vala_property_get_blurb (ValaProperty* self);
static gpointer vala_property_parent_class = NULL;
static ValaLockableIface* vala_property_vala_lockable_parent_iface = NULL;
static void vala_property_finalize (ValaCodeNode* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new property.
 *
 * @param name         property name
 * @param type         property type
 * @param get_accessor get accessor
 * @param set_accessor set/construct accessor
 * @param source       reference to source code
 * @return             newly created property
 */
ValaProperty* vala_property_construct (GType object_type, const char* name, ValaDataType* property_type, ValaPropertyAccessor* get_accessor, ValaPropertyAccessor* set_accessor, ValaSourceReference* source_reference) {
	ValaProperty* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = (ValaProperty*) vala_member_construct (object_type, name, source_reference);
	vala_property_set_property_type (self, property_type);
	vala_property_set_get_accessor (self, get_accessor);
	vala_property_set_set_accessor (self, set_accessor);
	return self;
}


ValaProperty* vala_property_new (const char* name, ValaDataType* property_type, ValaPropertyAccessor* get_accessor, ValaPropertyAccessor* set_accessor, ValaSourceReference* source_reference) {
	return vala_property_construct (VALA_TYPE_PROPERTY, name, property_type, get_accessor, set_accessor, source_reference);
}


static void vala_property_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaProperty * self;
	self = (ValaProperty*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_member (visitor, (ValaMember*) self);
	vala_code_visitor_visit_property (visitor, self);
}


static void vala_property_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaProperty * self;
	self = (ValaProperty*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_property_get_property_type (self), visitor);
	if (vala_property_get_get_accessor (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_property_get_get_accessor (self), visitor);
	}
	if (vala_property_get_set_accessor (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_property_get_set_accessor (self), visitor);
	}
	if (self->priv->_default_expression != NULL) {
		vala_code_node_accept ((ValaCodeNode*) self->priv->_default_expression, visitor);
	}
}


/**
 * Returns the C name of this property in upper case. Words are
 * separated by underscores. The upper case C name of the class is
 * prefix of the result.
 *
 * @return the upper case name to be used in C code
 */
char* vala_property_get_upper_case_cname (ValaProperty* self) {
	char* _tmp2;
	char* _tmp1;
	char* _tmp0;
	char* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = g_utf8_strup (_tmp2 = g_strdup_printf ("%s_%s", _tmp0 = vala_symbol_get_lower_case_cname (vala_symbol_get_parent_symbol ((ValaSymbol*) self), NULL), _tmp1 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name ((ValaSymbol*) self))), -1), _tmp2 = (g_free (_tmp2), NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp3);
}


/**
 * Returns the string literal of this property to be used in C code.
 *
 * @return string literal to be used in C code
 */
ValaCCodeConstant* vala_property_get_canonical_cconstant (ValaProperty* self) {
	char* _tmp1;
	char* _tmp0;
	ValaCCodeConstant* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = vala_ccode_constant_new (_tmp1 = g_strdup_printf ("\"%s\"", _tmp0 = vala_property_get_canonical_name (self))), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
}


static char* vala_property_get_canonical_name (ValaProperty* self) {
	GString* str;
	const char* _tmp0;
	char* i;
	const char* _tmp3;
	char* _tmp4;
	g_return_val_if_fail (self != NULL, NULL);
	str = g_string_new ("");
	_tmp0 = NULL;
	i = (_tmp0 = vala_symbol_get_name ((ValaSymbol*) self), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* _tmp2;
		const char* _tmp1;
		c = g_utf8_get_char (i);
		if (c == '_') {
			g_string_append_c (str, '-');
		} else {
			g_string_append_unichar (str, c);
		}
		_tmp2 = NULL;
		_tmp1 = NULL;
		i = (_tmp2 = (_tmp1 = g_utf8_next_char (i), (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), i = (g_free (i), NULL), _tmp2);
	}
	_tmp3 = NULL;
	_tmp4 = NULL;
	return (_tmp4 = (_tmp3 = str->str, (_tmp3 == NULL) ? NULL : g_strdup (_tmp3)), (str == NULL) ? NULL : (str = (g_string_free (str, TRUE), NULL)), i = (g_free (i), NULL), _tmp4);
}


static void vala_property_process_ccode_attribute (ValaProperty* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "notify")) {
		vala_property_set_notify (self, vala_attribute_get_bool (a, "notify"));
	}
}


/**
 * Process all associated attributes.
 */
void vala_property_process_attributes (ValaProperty* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp2;
			ValaAttribute* a;
			_tmp2 = NULL;
			a = (_tmp2 = (ValaAttribute*) a_it->data, (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_property_process_ccode_attribute (self, a);
				} else {
					if (_vala_strcmp0 (vala_attribute_get_name (a), "NoAccessorMethod") == 0) {
						vala_property_set_no_accessor_method (self, TRUE);
					} else {
						if (_vala_strcmp0 (vala_attribute_get_name (a), "Description") == 0) {
							if (vala_attribute_has_argument (a, "nick")) {
								char* _tmp0;
								_tmp0 = NULL;
								vala_property_set_nick (self, _tmp0 = vala_attribute_get_string (a, "nick"));
								_tmp0 = (g_free (_tmp0), NULL);
							}
							if (vala_attribute_has_argument (a, "blurb")) {
								char* _tmp1;
								_tmp1 = NULL;
								vala_property_set_blurb (self, _tmp1 = vala_attribute_get_string (a, "blurb"));
								_tmp1 = (g_free (_tmp1), NULL);
							}
						}
					}
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


static gboolean vala_property_real_get_lock_used (ValaLockable* base) {
	ValaProperty * self;
	self = (ValaProperty*) base;
	return self->priv->lock_used;
}


static void vala_property_real_set_lock_used (ValaLockable* base, gboolean used) {
	ValaProperty * self;
	self = (ValaProperty*) base;
	self->priv->lock_used = used;
}


/**
 * Checks whether the accessors and type of the specified property
 * matches this property.
 *
 * @param prop a property
 * @return     true if the specified property is compatible to this
 *             property
 */
gboolean vala_property_equals (ValaProperty* self, ValaProperty* prop2) {
	gboolean _tmp1;
	gboolean _tmp2;
	gboolean _tmp5;
	gboolean _tmp6;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (prop2 != NULL, FALSE);
	if (!vala_data_type_equals (vala_property_get_property_type (prop2), vala_property_get_property_type (self))) {
		return FALSE;
	}
	_tmp1 = FALSE;
	_tmp2 = FALSE;
	if (vala_property_get_get_accessor (self) == NULL) {
		_tmp2 = vala_property_get_get_accessor (prop2) != NULL;
	} else {
		_tmp2 = FALSE;
	}
	if (_tmp2) {
		_tmp1 = TRUE;
	} else {
		gboolean _tmp3;
		_tmp3 = FALSE;
		if (vala_property_get_get_accessor (self) != NULL) {
			_tmp3 = vala_property_get_get_accessor (prop2) == NULL;
		} else {
			_tmp3 = FALSE;
		}
		_tmp1 = _tmp3;
	}
	if (_tmp1) {
		return FALSE;
	}
	_tmp5 = FALSE;
	_tmp6 = FALSE;
	if (vala_property_get_set_accessor (self) == NULL) {
		_tmp6 = vala_property_get_set_accessor (prop2) != NULL;
	} else {
		_tmp6 = FALSE;
	}
	if (_tmp6) {
		_tmp5 = TRUE;
	} else {
		gboolean _tmp7;
		_tmp7 = FALSE;
		if (vala_property_get_set_accessor (self) != NULL) {
			_tmp7 = vala_property_get_set_accessor (prop2) == NULL;
		} else {
			_tmp7 = FALSE;
		}
		_tmp5 = _tmp7;
	}
	if (_tmp5) {
		return FALSE;
	}
	if (vala_property_get_set_accessor (self) != NULL) {
		if (vala_property_accessor_get_writable (vala_property_get_set_accessor (self)) != vala_property_accessor_get_writable (vala_property_get_set_accessor (prop2))) {
			return FALSE;
		}
		if (vala_property_accessor_get_construction (vala_property_get_set_accessor (self)) != vala_property_accessor_get_construction (vala_property_get_set_accessor (prop2))) {
			return FALSE;
		}
	}
	return TRUE;
}


static void vala_property_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaProperty * self;
	self = (ValaProperty*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_property_get_property_type (self) == old_type) {
		vala_property_set_property_type (self, new_type);
	}
}


static void vala_property_find_base_properties (ValaProperty* self) {
	g_return_if_fail (self != NULL);
	if (self->priv->base_properties_valid) {
		return;
	}
	if (VALA_IS_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self))) {
		gboolean _tmp0;
		vala_property_find_base_interface_property (self, VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self)));
		_tmp0 = FALSE;
		if (self->priv->_is_virtual) {
			_tmp0 = TRUE;
		} else {
			_tmp0 = self->priv->_overrides;
		}
		if (_tmp0) {
			vala_property_find_base_class_property (self, VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self)));
		}
	} else {
		if (VALA_IS_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) self))) {
			gboolean _tmp1;
			_tmp1 = FALSE;
			if (self->priv->_is_virtual) {
				_tmp1 = TRUE;
			} else {
				_tmp1 = self->priv->_is_abstract;
			}
			if (_tmp1) {
				ValaProperty* _tmp3;
				ValaProperty* _tmp2;
				_tmp3 = NULL;
				_tmp2 = NULL;
				self->priv->_base_interface_property = (_tmp3 = (_tmp2 = self, (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2)), (self->priv->_base_interface_property == NULL) ? NULL : (self->priv->_base_interface_property = (vala_code_node_unref (self->priv->_base_interface_property), NULL)), _tmp3);
			}
		}
	}
	self->priv->base_properties_valid = TRUE;
}


static void vala_property_find_base_class_property (ValaProperty* self, ValaClass* cl) {
	ValaSymbol* sym;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	sym = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) cl), vala_symbol_get_name ((ValaSymbol*) self));
	if (VALA_IS_PROPERTY (sym)) {
		ValaProperty* _tmp0;
		ValaProperty* base_property;
		gboolean _tmp1;
		_tmp0 = NULL;
		base_property = (_tmp0 = VALA_PROPERTY (sym), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
		_tmp1 = FALSE;
		if (base_property->priv->_is_abstract) {
			_tmp1 = TRUE;
		} else {
			_tmp1 = base_property->priv->_is_virtual;
		}
		if (_tmp1) {
			if (!vala_property_equals (self, base_property)) {
				char* _tmp4;
				char* _tmp3;
				char* _tmp2;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp4 = NULL;
				_tmp3 = NULL;
				_tmp2 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp4 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", _tmp2 = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp3 = vala_symbol_get_full_name ((ValaSymbol*) base_property)));
				_tmp4 = (g_free (_tmp4), NULL);
				_tmp3 = (g_free (_tmp3), NULL);
				_tmp2 = (g_free (_tmp2), NULL);
				(base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL));
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				return;
			}
			self->priv->_base_property = base_property;
			(base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL));
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			return;
		}
		(base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_property_find_base_class_property (self, vala_class_get_base_class (cl));
	}
	(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
}


static void vala_property_find_base_interface_property (ValaProperty* self, ValaClass* cl) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	/* FIXME report error if multiple possible base properties are found*/
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _type_it;
		/* FIXME report error if multiple possible base properties are found*/
		_tmp0 = NULL;
		_tmp1 = NULL;
		_type_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_class_get_base_types (cl))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		/* FIXME report error if multiple possible base properties are found*/
		while (gee_iterator_next (_type_it)) {
			ValaDataType* type;
			/* FIXME report error if multiple possible base properties are found*/
			type = (ValaDataType*) gee_iterator_get (_type_it);
			if (VALA_IS_INTERFACE (vala_data_type_get_data_type (type))) {
				ValaSymbol* sym;
				sym = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) vala_data_type_get_data_type (type)), vala_symbol_get_name ((ValaSymbol*) self));
				if (VALA_IS_PROPERTY (sym)) {
					ValaProperty* _tmp2;
					ValaProperty* base_property;
					_tmp2 = NULL;
					base_property = (_tmp2 = VALA_PROPERTY (sym), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
					if (base_property->priv->_is_abstract) {
						ValaProperty* _tmp7;
						ValaProperty* _tmp6;
						if (!vala_property_equals (self, base_property)) {
							char* _tmp5;
							char* _tmp4;
							char* _tmp3;
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							_tmp5 = NULL;
							_tmp4 = NULL;
							_tmp3 = NULL;
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp5 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", _tmp3 = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp4 = vala_symbol_get_full_name ((ValaSymbol*) base_property)));
							_tmp5 = (g_free (_tmp5), NULL);
							_tmp4 = (g_free (_tmp4), NULL);
							_tmp3 = (g_free (_tmp3), NULL);
							(base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL));
							(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
							(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
							(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
							return;
						}
						_tmp7 = NULL;
						_tmp6 = NULL;
						self->priv->_base_interface_property = (_tmp7 = (_tmp6 = base_property, (_tmp6 == NULL) ? NULL : vala_code_node_ref (_tmp6)), (self->priv->_base_interface_property == NULL) ? NULL : (self->priv->_base_interface_property = (vala_code_node_unref (self->priv->_base_interface_property), NULL)), _tmp7);
						(base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL));
						(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
						(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
						(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
						return;
					}
					(base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL));
				}
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			}
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		}
		(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
	}
}


static gboolean vala_property_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaProperty * self;
	ValaSourceFile* _tmp1;
	ValaSourceFile* old_source_file;
	ValaSymbol* _tmp2;
	ValaSymbol* old_symbol;
	gboolean _tmp6;
	gboolean _tmp9;
	gboolean _tmp10;
	gboolean _tmp11;
	gboolean _tmp17;
	gboolean _tmp20;
	gboolean _tmp24;
	self = (ValaProperty*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_property_process_attributes (self);
	_tmp1 = NULL;
	old_source_file = (_tmp1 = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp1 == NULL) ? NULL : vala_source_file_ref (_tmp1));
	_tmp2 = NULL;
	old_symbol = (_tmp2 = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, (ValaSymbol*) self);
	vala_code_node_check ((ValaCodeNode*) vala_property_get_property_type (self), analyzer);
	if (vala_property_get_get_accessor (self) != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_property_get_get_accessor (self), analyzer);
	}
	if (vala_property_get_set_accessor (self) != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_property_get_set_accessor (self), analyzer);
	}
	if (self->priv->_default_expression != NULL) {
		vala_code_node_check ((ValaCodeNode*) self->priv->_default_expression, analyzer);
	}
	/* check whether property type is at least as accessible as the property*/
	if (!vala_semantic_analyzer_is_type_accessible (analyzer, (ValaSymbol*) self, vala_property_get_property_type (self))) {
		char* _tmp5;
		char* _tmp4;
		char* _tmp3;
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		_tmp5 = NULL;
		_tmp4 = NULL;
		_tmp3 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp5 = g_strdup_printf ("property type `%s` is less accessible than property `%s`", _tmp3 = vala_code_node_to_string ((ValaCodeNode*) vala_property_get_property_type (self)), _tmp4 = vala_symbol_get_full_name ((ValaSymbol*) self)));
		_tmp5 = (g_free (_tmp5), NULL);
		_tmp4 = (g_free (_tmp4), NULL);
		_tmp3 = (g_free (_tmp3), NULL);
	}
	_tmp6 = FALSE;
	if (self->priv->_overrides) {
		_tmp6 = vala_property_get_base_property (self) == NULL;
	} else {
		_tmp6 = FALSE;
	}
	if (_tmp6) {
		char* _tmp8;
		char* _tmp7;
		_tmp8 = NULL;
		_tmp7 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp8 = g_strdup_printf ("%s: no suitable property found to override", _tmp7 = vala_symbol_get_full_name ((ValaSymbol*) self)));
		_tmp8 = (g_free (_tmp8), NULL);
		_tmp7 = (g_free (_tmp7), NULL);
	}
	_tmp9 = FALSE;
	_tmp10 = FALSE;
	_tmp11 = FALSE;
	if (!vala_symbol_get_external_package ((ValaSymbol*) self)) {
		_tmp11 = !self->priv->_overrides;
	} else {
		_tmp11 = FALSE;
	}
	if (_tmp11) {
		_tmp10 = !vala_member_get_hides ((ValaMember*) self);
	} else {
		_tmp10 = FALSE;
	}
	if (_tmp10) {
		ValaSymbol* _tmp12;
		_tmp12 = NULL;
		_tmp9 = (_tmp12 = vala_member_get_hidden_member ((ValaMember*) self)) != NULL;
		(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_code_node_unref (_tmp12), NULL));
	} else {
		_tmp9 = FALSE;
	}
	if (_tmp9) {
		char* _tmp16;
		char* _tmp15;
		ValaSymbol* _tmp14;
		char* _tmp13;
		_tmp16 = NULL;
		_tmp15 = NULL;
		_tmp14 = NULL;
		_tmp13 = NULL;
		vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp16 = g_strdup_printf ("%s hides inherited property `%s'. Use the `new' keyword if hiding was intentional", _tmp13 = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp15 = vala_symbol_get_full_name (_tmp14 = vala_member_get_hidden_member ((ValaMember*) self))));
		_tmp16 = (g_free (_tmp16), NULL);
		_tmp15 = (g_free (_tmp15), NULL);
		(_tmp14 == NULL) ? NULL : (_tmp14 = (vala_code_node_unref (_tmp14), NULL));
		_tmp13 = (g_free (_tmp13), NULL);
	}
	_tmp17 = FALSE;
	if (vala_property_get_set_accessor (self) != NULL) {
		_tmp17 = vala_property_accessor_get_construction (vala_property_get_set_accessor (self));
	} else {
		_tmp17 = FALSE;
	}
	/* construct properties must be public */
	if (_tmp17) {
		if (vala_symbol_get_access ((ValaSymbol*) self) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
			char* _tmp19;
			char* _tmp18;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp19 = NULL;
			_tmp18 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp19 = g_strdup_printf ("%s: construct properties must be public", _tmp18 = vala_symbol_get_full_name ((ValaSymbol*) self)));
			_tmp19 = (g_free (_tmp19), NULL);
			_tmp18 = (g_free (_tmp18), NULL);
		}
	}
	_tmp20 = FALSE;
	if (self->priv->_default_expression != NULL) {
		_tmp20 = !vala_data_type_compatible (vala_expression_get_value_type (self->priv->_default_expression), vala_property_get_property_type (self));
	} else {
		_tmp20 = FALSE;
	}
	if (_tmp20) {
		char* _tmp23;
		char* _tmp22;
		char* _tmp21;
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		_tmp23 = NULL;
		_tmp22 = NULL;
		_tmp21 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self->priv->_default_expression), _tmp23 = g_strdup_printf ("Expected initializer of type `%s' but got `%s'", _tmp21 = vala_code_node_to_string ((ValaCodeNode*) vala_property_get_property_type (self)), _tmp22 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (self->priv->_default_expression))));
		_tmp23 = (g_free (_tmp23), NULL);
		_tmp22 = (g_free (_tmp22), NULL);
		_tmp21 = (g_free (_tmp21), NULL);
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	vala_semantic_analyzer_set_current_symbol (analyzer, old_symbol);
	return (_tmp24 = !vala_code_node_get_error ((ValaCodeNode*) self), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), _tmp24);
}


ValaDataType* vala_property_get_property_type (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_data_type;
}


void vala_property_set_property_type (ValaProperty* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL)), _tmp2);
	if (value != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_data_type, (ValaCodeNode*) self);
	}
}


ValaPropertyAccessor* vala_property_get_get_accessor (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_get_accessor;
}


void vala_property_set_get_accessor (ValaProperty* self, ValaPropertyAccessor* value) {
	ValaPropertyAccessor* _tmp2;
	ValaPropertyAccessor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_get_accessor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_get_accessor == NULL) ? NULL : (self->priv->_get_accessor = (vala_code_node_unref (self->priv->_get_accessor), NULL)), _tmp2);
	if (value != NULL) {
		vala_property_accessor_set_prop (value, self);
	}
}


ValaPropertyAccessor* vala_property_get_set_accessor (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_set_accessor;
}


void vala_property_set_set_accessor (ValaProperty* self, ValaPropertyAccessor* value) {
	ValaPropertyAccessor* _tmp2;
	ValaPropertyAccessor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_set_accessor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_set_accessor == NULL) ? NULL : (self->priv->_set_accessor = (vala_code_node_unref (self->priv->_set_accessor), NULL)), _tmp2);
	if (value != NULL) {
		vala_property_accessor_set_prop (value, self);
	}
}


ValaFormalParameter* vala_property_get_this_parameter (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_this_parameter;
}


void vala_property_set_this_parameter (ValaProperty* self, ValaFormalParameter* value) {
	ValaFormalParameter* _tmp2;
	ValaFormalParameter* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_this_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_this_parameter == NULL) ? NULL : (self->priv->_this_parameter = (vala_code_node_unref (self->priv->_this_parameter), NULL)), _tmp2);
}


gboolean vala_property_get_notify (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_notify;
}


void vala_property_set_notify (ValaProperty* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_notify = value;
}


gboolean vala_property_get_no_accessor_method (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_no_accessor_method;
}


void vala_property_set_no_accessor_method (ValaProperty* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_no_accessor_method = value;
}


gboolean vala_property_get_interface_only (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_interface_only;
}


void vala_property_set_interface_only (ValaProperty* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_interface_only = value;
}


gboolean vala_property_get_is_abstract (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_abstract;
}


void vala_property_set_is_abstract (ValaProperty* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_abstract = value;
}


gboolean vala_property_get_is_virtual (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_virtual;
}


void vala_property_set_is_virtual (ValaProperty* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_virtual = value;
}


gboolean vala_property_get_overrides (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_overrides;
}


void vala_property_set_overrides (ValaProperty* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_overrides = value;
}


ValaField* vala_property_get_field (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_field;
}


void vala_property_set_field (ValaProperty* self, ValaField* value) {
	ValaField* _tmp2;
	ValaField* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_field = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_field == NULL) ? NULL : (self->priv->_field = (vala_code_node_unref (self->priv->_field), NULL)), _tmp2);
}


MemberBinding vala_property_get_binding (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_binding;
}


void vala_property_set_binding (ValaProperty* self, MemberBinding value) {
	g_return_if_fail (self != NULL);
	self->priv->_binding = value;
}


ValaProperty* vala_property_get_base_property (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	vala_property_find_base_properties (self);
	return self->priv->_base_property;
}


ValaProperty* vala_property_get_base_interface_property (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	vala_property_find_base_properties (self);
	return self->priv->_base_interface_property;
}


ValaExpression* vala_property_get_default_expression (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_default_expression;
}


void vala_property_set_default_expression (ValaProperty* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_default_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_default_expression == NULL) ? NULL : (self->priv->_default_expression = (vala_code_node_unref (self->priv->_default_expression), NULL)), _tmp2);
}


const char* vala_property_get_nick (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_nick == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->_nick = (_tmp0 = vala_property_get_canonical_name (self), self->priv->_nick = (g_free (self->priv->_nick), NULL), _tmp0);
	}
	return self->priv->_nick;
}


void vala_property_set_nick (ValaProperty* self, const char* value) {
	char* _tmp3;
	const char* _tmp2;
	g_return_if_fail (self != NULL);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->_nick = (_tmp3 = (_tmp2 = value, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), self->priv->_nick = (g_free (self->priv->_nick), NULL), _tmp3);
}


const char* vala_property_get_blurb (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_blurb == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->_blurb = (_tmp0 = vala_property_get_canonical_name (self), self->priv->_blurb = (g_free (self->priv->_blurb), NULL), _tmp0);
	}
	return self->priv->_blurb;
}


void vala_property_set_blurb (ValaProperty* self, const char* value) {
	char* _tmp3;
	const char* _tmp2;
	g_return_if_fail (self != NULL);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->_blurb = (_tmp3 = (_tmp2 = value, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), self->priv->_blurb = (g_free (self->priv->_blurb), NULL), _tmp3);
}


static void vala_property_class_init (ValaPropertyClass * klass) {
	vala_property_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_property_finalize;
	g_type_class_add_private (klass, sizeof (ValaPropertyPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_property_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_property_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_property_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_property_real_check;
}


static void vala_property_vala_lockable_interface_init (ValaLockableIface * iface) {
	vala_property_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_property_real_get_lock_used;
	iface->set_lock_used = vala_property_real_set_lock_used;
}


static void vala_property_instance_init (ValaProperty * self) {
	self->priv = VALA_PROPERTY_GET_PRIVATE (self);
	self->priv->_notify = TRUE;
	self->priv->_binding = MEMBER_BINDING_INSTANCE;
	self->priv->lock_used = FALSE;
}


static void vala_property_finalize (ValaCodeNode* obj) {
	ValaProperty * self;
	self = VALA_PROPERTY (obj);
	(self->priv->_this_parameter == NULL) ? NULL : (self->priv->_this_parameter = (vala_code_node_unref (self->priv->_this_parameter), NULL));
	(self->priv->_field == NULL) ? NULL : (self->priv->_field = (vala_code_node_unref (self->priv->_field), NULL));
	(self->priv->_default_expression == NULL) ? NULL : (self->priv->_default_expression = (vala_code_node_unref (self->priv->_default_expression), NULL));
	(self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL));
	self->priv->_nick = (g_free (self->priv->_nick), NULL);
	self->priv->_blurb = (g_free (self->priv->_blurb), NULL);
	(self->priv->_base_interface_property == NULL) ? NULL : (self->priv->_base_interface_property = (vala_code_node_unref (self->priv->_base_interface_property), NULL));
	(self->priv->_get_accessor == NULL) ? NULL : (self->priv->_get_accessor = (vala_code_node_unref (self->priv->_get_accessor), NULL));
	(self->priv->_set_accessor == NULL) ? NULL : (self->priv->_set_accessor = (vala_code_node_unref (self->priv->_set_accessor), NULL));
	VALA_CODE_NODE_CLASS (vala_property_parent_class)->finalize (obj);
}


GType vala_property_get_type (void) {
	static GType vala_property_type_id = 0;
	if (vala_property_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaPropertyClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_property_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaProperty), 0, (GInstanceInitFunc) vala_property_instance_init, NULL };
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_property_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_property_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaProperty", &g_define_type_info, 0);
		g_type_add_interface_static (vala_property_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_property_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




