/* valasymbol.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;

#define VALA_TYPE_SYMBOL_ACCESSIBILITY (vala_symbol_accessibility_get_type ())

#define VALA_TYPE_SCOPE (vala_scope_get_type ())
#define VALA_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SCOPE, ValaScope))
#define VALA_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SCOPE, ValaScopeClass))
#define VALA_IS_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SCOPE))
#define VALA_IS_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SCOPE))
#define VALA_SCOPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SCOPE, ValaScopeClass))

typedef struct _ValaScope ValaScope;
typedef struct _ValaScopeClass ValaScopeClass;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_SOURCE_FILE (vala_source_file_get_type ())
#define VALA_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFile))
#define VALA_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))
#define VALA_IS_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_FILE))
#define VALA_IS_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_FILE))
#define VALA_SOURCE_FILE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))

typedef struct _ValaSourceFile ValaSourceFile;
typedef struct _ValaSourceFileClass ValaSourceFileClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Represents a node in the symbol tree.
 */
struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
};

typedef enum  {
	VALA_SYMBOL_ACCESSIBILITY_PRIVATE,
	VALA_SYMBOL_ACCESSIBILITY_INTERNAL,
	VALA_SYMBOL_ACCESSIBILITY_PROTECTED,
	VALA_SYMBOL_ACCESSIBILITY_PUBLIC
} ValaSymbolAccessibility;

struct _ValaSymbolPrivate {
	char* _name;
	gboolean _active;
	gboolean _used;
	ValaSymbolAccessibility _access;
	gboolean _external;
	ValaScope* _owner;
	ValaScope* _scope;
};



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_symbol_accessibility_get_type (void);
gpointer vala_scope_ref (gpointer instance);
void vala_scope_unref (gpointer instance);
GParamSpec* vala_param_spec_scope (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_scope (GValue* value, gpointer v_object);
gpointer vala_value_get_scope (const GValue* value);
GType vala_scope_get_type (void);
#define VALA_SYMBOL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SYMBOL, ValaSymbolPrivate))
enum  {
	VALA_SYMBOL_DUMMY_PROPERTY
};
gboolean vala_symbol_get_external (ValaSymbol* self);
gboolean vala_symbol_get_external_package (ValaSymbol* self);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
ValaSymbolAccessibility vala_symbol_get_access (ValaSymbol* self);
gboolean vala_symbol_is_internal_symbol (ValaSymbol* self);
gboolean vala_symbol_is_private_symbol (ValaSymbol* self);
void vala_symbol_set_name (ValaSymbol* self, const char* value);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
void vala_code_node_set_source_reference (ValaCodeNode* self, ValaSourceReference* value);
ValaScope* vala_scope_new (ValaSymbol* owner);
ValaScope* vala_scope_construct (GType object_type, ValaSymbol* owner);
ValaSymbol* vala_symbol_new (const char* name, ValaSourceReference* source_reference);
ValaSymbol* vala_symbol_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
ValaSymbol* vala_symbol_new (const char* name, ValaSourceReference* source_reference);
const char* vala_symbol_get_name (ValaSymbol* self);
char* vala_symbol_get_full_name (ValaSymbol* self);
char* vala_symbol_get_cprefix (ValaSymbol* self);
static char* vala_symbol_real_get_cprefix (ValaSymbol* self);
char* vala_symbol_get_lower_case_cname (ValaSymbol* self, const char* infix);
static char* vala_symbol_real_get_lower_case_cname (ValaSymbol* self, const char* infix);
char* vala_symbol_get_lower_case_cprefix (ValaSymbol* self);
static char* vala_symbol_real_get_lower_case_cprefix (ValaSymbol* self);
GeeList* vala_symbol_get_cheader_filenames (ValaSymbol* self);
static GeeList* vala_symbol_real_get_cheader_filenames (ValaSymbol* self);
char* vala_symbol_camel_case_to_lower_case (const char* camel_case);
char* vala_symbol_lower_case_to_camel_case (const char* lower_case);
ValaScope* vala_symbol_get_owner (ValaSymbol* self);
ValaScope* vala_symbol_get_scope (ValaSymbol* self);
ValaScope* vala_symbol_get_top_accessible_scope (ValaSymbol* self, gboolean is_internal);
ValaSymbol* vala_scope_get_owner (ValaScope* self);
void vala_scope_set_parent_scope (ValaScope* self, ValaScope* value);
void vala_symbol_set_owner (ValaSymbol* self, ValaScope* value);
gboolean vala_symbol_get_active (ValaSymbol* self);
void vala_symbol_set_active (ValaSymbol* self, gboolean value);
gboolean vala_symbol_get_used (ValaSymbol* self);
void vala_symbol_set_used (ValaSymbol* self, gboolean value);
void vala_symbol_set_access (ValaSymbol* self, ValaSymbolAccessibility value);
void vala_symbol_set_external (ValaSymbol* self, gboolean value);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
gpointer vala_source_file_ref (gpointer instance);
void vala_source_file_unref (gpointer instance);
GParamSpec* vala_param_spec_source_file (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_file (GValue* value, gpointer v_object);
gpointer vala_value_get_source_file (const GValue* value);
GType vala_source_file_get_type (void);
ValaSourceFile* vala_source_reference_get_file (ValaSourceReference* self);
gboolean vala_source_file_get_external_package (ValaSourceFile* self);
static gpointer vala_symbol_parent_class = NULL;
static void vala_symbol_finalize (ValaCodeNode* obj);



/**
 * Check if this symbol is just internal API (and therefore doesn't need 
 * to be listed in header files for instance) by traversing parent symbols
 * and checking their accessibility.
 */
gboolean vala_symbol_is_internal_symbol (ValaSymbol* self) {
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0 = FALSE;
	if (!self->priv->_external) {
		_tmp0 = vala_symbol_get_external_package (self);
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		/* non-external symbols in VAPI files are internal symbols*/
		return TRUE;
	}
	{
		ValaSymbol* _tmp2;
		ValaSymbol* sym;
		ValaSymbol* _tmp4;
		ValaSymbol* _tmp3;
		_tmp2 = NULL;
		sym = (_tmp2 = self, (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
		_tmp4 = NULL;
		_tmp3 = NULL;
		for (; NULL != sym; sym = (_tmp4 = (_tmp3 = vala_symbol_get_parent_symbol (sym), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp4)) {
			gboolean _tmp5;
			_tmp5 = FALSE;
			if (sym->priv->_access == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
				_tmp5 = TRUE;
			} else {
				_tmp5 = sym->priv->_access == VALA_SYMBOL_ACCESSIBILITY_INTERNAL;
			}
			if (_tmp5) {
				gboolean _tmp6;
				return (_tmp6 = TRUE, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp6);
			}
		}
		(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
	}
	return FALSE;
}


gboolean vala_symbol_is_private_symbol (ValaSymbol* self) {
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0 = FALSE;
	if (!self->priv->_external) {
		_tmp0 = vala_symbol_get_external_package (self);
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		/* non-external symbols in VAPI files are private symbols*/
		return TRUE;
	}
	{
		ValaSymbol* _tmp2;
		ValaSymbol* sym;
		ValaSymbol* _tmp4;
		ValaSymbol* _tmp3;
		_tmp2 = NULL;
		sym = (_tmp2 = self, (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
		_tmp4 = NULL;
		_tmp3 = NULL;
		for (; NULL != sym; sym = (_tmp4 = (_tmp3 = vala_symbol_get_parent_symbol (sym), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp4)) {
			if (sym->priv->_access == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
				gboolean _tmp5;
				return (_tmp5 = TRUE, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp5);
			}
		}
		(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
	}
	return FALSE;
}


ValaSymbol* vala_symbol_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaSymbol* self;
	ValaScope* _tmp0;
	self = (ValaSymbol*) g_type_create_instance (object_type);
	vala_symbol_set_name (self, name);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	_tmp0 = NULL;
	self->priv->_scope = (_tmp0 = vala_scope_new (self), (self->priv->_scope == NULL) ? NULL : (self->priv->_scope = (vala_scope_unref (self->priv->_scope), NULL)), _tmp0);
	return self;
}


ValaSymbol* vala_symbol_new (const char* name, ValaSourceReference* source_reference) {
	return vala_symbol_construct (VALA_TYPE_SYMBOL, name, source_reference);
}


/**
 * Returns the fully expanded name of this symbol for use in
 * human-readable messages.
 *
 * @return full name
 */
char* vala_symbol_get_full_name (ValaSymbol* self) {
	char* _tmp3;
	gboolean _tmp4;
	char* _tmp7;
	char* _tmp8;
	g_return_val_if_fail (self != NULL, NULL);
	if (vala_symbol_get_parent_symbol (self) == NULL) {
		const char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = self->priv->_name, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
	}
	if (self->priv->_name == NULL) {
		return vala_symbol_get_full_name (vala_symbol_get_parent_symbol (self));
	}
	_tmp3 = NULL;
	if ((_tmp4 = (_tmp3 = vala_symbol_get_full_name (vala_symbol_get_parent_symbol (self))) == NULL, _tmp3 = (g_free (_tmp3), NULL), _tmp4)) {
		const char* _tmp5;
		_tmp5 = NULL;
		return (_tmp5 = self->priv->_name, (_tmp5 == NULL) ? NULL : g_strdup (_tmp5));
	}
	_tmp7 = NULL;
	_tmp8 = NULL;
	return (_tmp8 = g_strdup_printf ("%s.%s", _tmp7 = vala_symbol_get_full_name (vala_symbol_get_parent_symbol (self)), self->priv->_name), _tmp7 = (g_free (_tmp7), NULL), _tmp8);
}


/**
 * Returns the camel case string to be prepended to the name of members
 * of this symbol when used in C code.
 *
 * @return the camel case prefix to be used in C code
 */
static char* vala_symbol_real_get_cprefix (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_name == NULL) {
		return g_strdup ("");
	} else {
		const char* _tmp1;
		_tmp1 = NULL;
		return (_tmp1 = self->priv->_name, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
	}
}


char* vala_symbol_get_cprefix (ValaSymbol* self) {
	return VALA_SYMBOL_GET_CLASS (self)->get_cprefix (self);
}


/**
 * Returns the C name of this symbol in lower case. Words are
 * separated by underscores. The lower case C name of the parent symbol
 * is prefix of the result, if there is one.
 *
 * @param infix a string to be placed between namespace and data type
 *              name or null
 * @return      the lower case name to be used in C code
 */
static char* vala_symbol_real_get_lower_case_cname (ValaSymbol* self, const char* infix) {
	g_return_val_if_fail (self != NULL, NULL);
	return NULL;
}


char* vala_symbol_get_lower_case_cname (ValaSymbol* self, const char* infix) {
	return VALA_SYMBOL_GET_CLASS (self)->get_lower_case_cname (self, infix);
}


/**
 * Returns the string to be prefixed to members of this symbol in
 * lower case when used in C code.
 *
 * @return      the lower case prefix to be used in C code
 */
static char* vala_symbol_real_get_lower_case_cprefix (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return g_strdup ("");
}


char* vala_symbol_get_lower_case_cprefix (ValaSymbol* self) {
	return VALA_SYMBOL_GET_CLASS (self)->get_lower_case_cprefix (self);
}


/**
 * Returns a list of C header filenames users of this symbol must
 * include.
 *
 * @return list of C header filenames for this symbol
 */
static GeeList* vala_symbol_real_get_cheader_filenames (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
}


GeeList* vala_symbol_get_cheader_filenames (ValaSymbol* self) {
	return VALA_SYMBOL_GET_CLASS (self)->get_cheader_filenames (self);
}


/**
 * Converts a string from CamelCase to lower_case.
 *
 * @param camel_case a string in camel case
 * @return           the specified string converted to lower case
 */
char* vala_symbol_camel_case_to_lower_case (const char* camel_case) {
	GString* result_builder;
	const char* i;
	gboolean first;
	const char* _tmp4;
	char* _tmp5;
	g_return_val_if_fail (camel_case != NULL, NULL);
	result_builder = g_string_new ("");
	i = camel_case;
	first = TRUE;
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		gboolean _tmp0;
		c = g_utf8_get_char (i);
		_tmp0 = FALSE;
		if (g_unichar_isupper (c)) {
			_tmp0 = !first;
		} else {
			_tmp0 = FALSE;
		}
		if (_tmp0) {
			const char* t;
			gboolean prev_upper;
			gboolean next_upper;
			gboolean _tmp1;
			/* current character is upper case and
			 * we're not at the beginning */
			t = g_utf8_prev_char (i);
			prev_upper = g_unichar_isupper (g_utf8_get_char (t));
			t = g_utf8_next_char (i);
			next_upper = g_unichar_isupper (g_utf8_get_char (t));
			_tmp1 = FALSE;
			if (!prev_upper) {
				_tmp1 = TRUE;
			} else {
				gboolean _tmp2;
				_tmp2 = FALSE;
				if (g_utf8_strlen (i, -1) >= 2) {
					_tmp2 = !next_upper;
				} else {
					_tmp2 = FALSE;
				}
				_tmp1 = _tmp2;
			}
			if (_tmp1) {
				glong len;
				gboolean _tmp3;
				/* previous character wasn't upper case or
				 * next character isn't upper case*/
				len = g_utf8_strlen (result_builder->str, -1);
				_tmp3 = FALSE;
				if (len != 1) {
					_tmp3 = g_utf8_get_char (g_utf8_offset_to_pointer (result_builder->str, len - 2)) != '_';
				} else {
					_tmp3 = FALSE;
				}
				if (_tmp3) {
					/* we're not creating 1 character words */
					g_string_append_c (result_builder, '_');
				}
			}
		}
		g_string_append_unichar (result_builder, g_unichar_tolower (c));
		first = FALSE;
		i = g_utf8_next_char (i);
	}
	_tmp4 = NULL;
	_tmp5 = NULL;
	return (_tmp5 = (_tmp4 = result_builder->str, (_tmp4 == NULL) ? NULL : g_strdup (_tmp4)), (result_builder == NULL) ? NULL : (result_builder = (g_string_free (result_builder, TRUE), NULL)), _tmp5);
}


/**
 * Converts a string from lower_case to CamelCase.
 *
 * @param lower_case a string in lower case
 * @return           the specified string converted to camel case
 */
char* vala_symbol_lower_case_to_camel_case (const char* lower_case) {
	GString* result_builder;
	const char* i;
	gboolean last_underscore;
	const char* _tmp2;
	char* _tmp3;
	g_return_val_if_fail (lower_case != NULL, NULL);
	result_builder = g_string_new ("");
	i = lower_case;
	last_underscore = TRUE;
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		c = g_utf8_get_char (i);
		if (c == '_') {
			last_underscore = TRUE;
		} else {
			if (g_unichar_isupper (c)) {
				const char* _tmp0;
				char* _tmp1;
				/* original string is not lower_case, don't apply transformation*/
				_tmp0 = NULL;
				_tmp1 = NULL;
				return (_tmp1 = (_tmp0 = lower_case, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), (result_builder == NULL) ? NULL : (result_builder = (g_string_free (result_builder, TRUE), NULL)), _tmp1);
			} else {
				if (last_underscore) {
					g_string_append_unichar (result_builder, g_unichar_toupper (c));
					last_underscore = FALSE;
				} else {
					g_string_append_unichar (result_builder, c);
				}
			}
		}
		i = g_utf8_next_char (i);
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = (_tmp2 = result_builder->str, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), (result_builder == NULL) ? NULL : (result_builder = (g_string_free (result_builder, TRUE), NULL)), _tmp3);
}


/* get the top scope from where this symbol is still accessible*/
ValaScope* vala_symbol_get_top_accessible_scope (ValaSymbol* self, gboolean is_internal) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_access == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		ValaScope* _tmp0;
		/* private symbols are accessible within the scope where the symbol has been declared*/
		_tmp0 = NULL;
		return (_tmp0 = vala_symbol_get_owner (self), (_tmp0 == NULL) ? NULL : vala_scope_ref (_tmp0));
	}
	if (self->priv->_access == VALA_SYMBOL_ACCESSIBILITY_INTERNAL) {
		is_internal = TRUE;
	}
	if (vala_symbol_get_parent_symbol (self) == NULL) {
		/* this is the root symbol*/
		if (is_internal) {
			ValaScope* _tmp2;
			/* only accessible within the same library
			 return root scope*/
			_tmp2 = NULL;
			return (_tmp2 = vala_symbol_get_scope (self), (_tmp2 == NULL) ? NULL : vala_scope_ref (_tmp2));
		} else {
			/* unlimited access*/
			return NULL;
		}
	}
	/* if this is a public symbol, it's equally accessible as the parent symbol*/
	return vala_symbol_get_top_accessible_scope (vala_symbol_get_parent_symbol (self), is_internal);
}


ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (vala_symbol_get_owner (self) == NULL) {
		return NULL;
	} else {
		return vala_scope_get_owner (vala_symbol_get_owner (self));
	}
}


ValaScope* vala_symbol_get_owner (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_owner;
}


void vala_symbol_set_owner (ValaSymbol* self, ValaScope* value) {
	g_return_if_fail (self != NULL);
	self->priv->_owner = value;
	vala_scope_set_parent_scope (self->priv->_scope, value);
}


const char* vala_symbol_get_name (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_name;
}


void vala_symbol_set_name (ValaSymbol* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_name = (g_free (self->priv->_name), NULL), _tmp2);
}


gboolean vala_symbol_get_active (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_active;
}


void vala_symbol_set_active (ValaSymbol* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_active = value;
}


gboolean vala_symbol_get_used (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_used;
}


void vala_symbol_set_used (ValaSymbol* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_used = value;
}


ValaSymbolAccessibility vala_symbol_get_access (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_access;
}


void vala_symbol_set_access (ValaSymbol* self, ValaSymbolAccessibility value) {
	g_return_if_fail (self != NULL);
	self->priv->_access = value;
}


ValaScope* vala_symbol_get_scope (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_scope;
}


gboolean vala_symbol_get_external (ValaSymbol* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_external;
}


void vala_symbol_set_external (ValaSymbol* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_external = value;
}


gboolean vala_symbol_get_external_package (ValaSymbol* self) {
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0 = FALSE;
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		_tmp0 = vala_source_file_get_external_package (vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	} else {
		_tmp0 = FALSE;
	}
	return _tmp0;
}


static void vala_symbol_class_init (ValaSymbolClass * klass) {
	vala_symbol_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_symbol_finalize;
	g_type_class_add_private (klass, sizeof (ValaSymbolPrivate));
	VALA_SYMBOL_CLASS (klass)->get_cprefix = vala_symbol_real_get_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_symbol_real_get_lower_case_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_symbol_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_cheader_filenames = vala_symbol_real_get_cheader_filenames;
}


static void vala_symbol_instance_init (ValaSymbol * self) {
	self->priv = VALA_SYMBOL_GET_PRIVATE (self);
	self->priv->_active = TRUE;
}


static void vala_symbol_finalize (ValaCodeNode* obj) {
	ValaSymbol * self;
	self = VALA_SYMBOL (obj);
	self->priv->_name = (g_free (self->priv->_name), NULL);
	(self->priv->_scope == NULL) ? NULL : (self->priv->_scope = (vala_scope_unref (self->priv->_scope), NULL));
	VALA_CODE_NODE_CLASS (vala_symbol_parent_class)->finalize (obj);
}


GType vala_symbol_get_type (void) {
	static GType vala_symbol_type_id = 0;
	if (vala_symbol_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSymbolClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_symbol_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSymbol), 0, (GInstanceInitFunc) vala_symbol_instance_init, NULL };
		vala_symbol_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaSymbol", &g_define_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return vala_symbol_type_id;
}



GType vala_symbol_accessibility_get_type (void) {
	static GType vala_symbol_accessibility_type_id = 0;
	if (G_UNLIKELY (vala_symbol_accessibility_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_SYMBOL_ACCESSIBILITY_PRIVATE, "VALA_SYMBOL_ACCESSIBILITY_PRIVATE", "private"}, {VALA_SYMBOL_ACCESSIBILITY_INTERNAL, "VALA_SYMBOL_ACCESSIBILITY_INTERNAL", "internal"}, {VALA_SYMBOL_ACCESSIBILITY_PROTECTED, "VALA_SYMBOL_ACCESSIBILITY_PROTECTED", "protected"}, {VALA_SYMBOL_ACCESSIBILITY_PUBLIC, "VALA_SYMBOL_ACCESSIBILITY_PUBLIC", "public"}, {0, NULL, NULL}};
		vala_symbol_accessibility_type_id = g_enum_register_static ("ValaSymbolAccessibility", values);
	}
	return vala_symbol_accessibility_type_id;
}




