/* valaccodebasemodule.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 * Copyright (C) 2006-2008  Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 * 	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <vala.h>
#include <stdlib.h>
#include <string.h>
#include <valaccode.h>
#include <gee.h>
#include <float.h>
#include <math.h>


#define VALA_TYPE_CCODE_MODULE (vala_ccode_module_get_type ())
#define VALA_CCODE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_MODULE, ValaCCodeModule))
#define VALA_CCODE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_MODULE, ValaCCodeModuleClass))
#define VALA_IS_CCODE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_MODULE))
#define VALA_IS_CCODE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_MODULE))
#define VALA_CCODE_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_MODULE, ValaCCodeModuleClass))

typedef struct _ValaCCodeModule ValaCCodeModule;
typedef struct _ValaCCodeModuleClass ValaCCodeModuleClass;
typedef struct _ValaCCodeModulePrivate ValaCCodeModulePrivate;

#define VALA_TYPE_CCODE_BASE_MODULE (vala_ccode_base_module_get_type ())
#define VALA_CCODE_BASE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_BASE_MODULE, ValaCCodeBaseModule))
#define VALA_CCODE_BASE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_BASE_MODULE, ValaCCodeBaseModuleClass))
#define VALA_IS_CCODE_BASE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_BASE_MODULE))
#define VALA_IS_CCODE_BASE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_BASE_MODULE))
#define VALA_CCODE_BASE_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_BASE_MODULE, ValaCCodeBaseModuleClass))

typedef struct _ValaCCodeBaseModule ValaCCodeBaseModule;
typedef struct _ValaCCodeBaseModuleClass ValaCCodeBaseModuleClass;
typedef struct _ValaCCodeBaseModulePrivate ValaCCodeBaseModulePrivate;

#define VALA_TYPE_CCODE_DECLARATION_SPACE (vala_ccode_declaration_space_get_type ())
#define VALA_CCODE_DECLARATION_SPACE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_DECLARATION_SPACE, ValaCCodeDeclarationSpace))
#define VALA_CCODE_DECLARATION_SPACE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_DECLARATION_SPACE, ValaCCodeDeclarationSpaceClass))
#define VALA_IS_CCODE_DECLARATION_SPACE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_DECLARATION_SPACE))
#define VALA_IS_CCODE_DECLARATION_SPACE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_DECLARATION_SPACE))
#define VALA_CCODE_DECLARATION_SPACE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_DECLARATION_SPACE, ValaCCodeDeclarationSpaceClass))

typedef struct _ValaCCodeDeclarationSpace ValaCCodeDeclarationSpace;
typedef struct _ValaCCodeDeclarationSpaceClass ValaCCodeDeclarationSpaceClass;

#define VALA_TYPE_CCODE_GENERATOR (vala_ccode_generator_get_type ())
#define VALA_CCODE_GENERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_GENERATOR, ValaCCodeGenerator))
#define VALA_CCODE_GENERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_GENERATOR, ValaCCodeGeneratorClass))
#define VALA_IS_CCODE_GENERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_GENERATOR))
#define VALA_IS_CCODE_GENERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_GENERATOR))
#define VALA_CCODE_GENERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_GENERATOR, ValaCCodeGeneratorClass))

typedef struct _ValaCCodeGenerator ValaCCodeGenerator;
typedef struct _ValaCCodeGeneratorClass ValaCCodeGeneratorClass;
typedef struct _ValaCCodeDeclarationSpacePrivate ValaCCodeDeclarationSpacePrivate;

/**
 * Code visitor generating C Code.
 */
struct _ValaCCodeModule {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeModulePrivate * priv;
};

struct _ValaCCodeModuleClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeModule *self);
	void (*emit) (ValaCCodeModule* self, ValaCodeContext* context);
	void (*visit_source_file) (ValaCCodeModule* self, ValaSourceFile* source_file);
	void (*visit_class) (ValaCCodeModule* self, ValaClass* cl);
	void (*visit_interface) (ValaCCodeModule* self, ValaInterface* iface);
	void (*visit_struct) (ValaCCodeModule* self, ValaStruct* st);
	void (*visit_enum) (ValaCCodeModule* self, ValaEnum* en);
	void (*visit_error_domain) (ValaCCodeModule* self, ValaErrorDomain* edomain);
	void (*visit_delegate) (ValaCCodeModule* self, ValaDelegate* d);
	void (*visit_member) (ValaCCodeModule* self, ValaMember* m);
	void (*visit_constant) (ValaCCodeModule* self, ValaConstant* c);
	void (*visit_field) (ValaCCodeModule* self, ValaField* f);
	void (*visit_method) (ValaCCodeModule* self, ValaMethod* m);
	void (*visit_creation_method) (ValaCCodeModule* self, ValaCreationMethod* m);
	void (*visit_formal_parameter) (ValaCCodeModule* self, ValaFormalParameter* p);
	void (*visit_property) (ValaCCodeModule* self, ValaProperty* prop);
	void (*visit_property_accessor) (ValaCCodeModule* self, ValaPropertyAccessor* acc);
	void (*visit_signal) (ValaCCodeModule* self, ValaSignal* sig);
	void (*visit_constructor) (ValaCCodeModule* self, ValaConstructor* c);
	void (*visit_destructor) (ValaCCodeModule* self, ValaDestructor* d);
	void (*visit_block) (ValaCCodeModule* self, ValaBlock* b);
	void (*visit_empty_statement) (ValaCCodeModule* self, ValaEmptyStatement* stmt);
	void (*visit_declaration_statement) (ValaCCodeModule* self, ValaDeclarationStatement* stmt);
	void (*visit_local_variable) (ValaCCodeModule* self, ValaLocalVariable* local);
	void (*visit_initializer_list) (ValaCCodeModule* self, ValaInitializerList* list);
	void (*visit_end_full_expression) (ValaCCodeModule* self, ValaExpression* expr);
	void (*visit_expression_statement) (ValaCCodeModule* self, ValaExpressionStatement* stmt);
	void (*visit_if_statement) (ValaCCodeModule* self, ValaIfStatement* stmt);
	void (*visit_switch_statement) (ValaCCodeModule* self, ValaSwitchStatement* stmt);
	void (*visit_switch_section) (ValaCCodeModule* self, ValaSwitchSection* section);
	void (*visit_switch_label) (ValaCCodeModule* self, ValaSwitchLabel* label);
	void (*visit_while_statement) (ValaCCodeModule* self, ValaWhileStatement* stmt);
	void (*visit_do_statement) (ValaCCodeModule* self, ValaDoStatement* stmt);
	void (*visit_for_statement) (ValaCCodeModule* self, ValaForStatement* stmt);
	void (*visit_foreach_statement) (ValaCCodeModule* self, ValaForeachStatement* stmt);
	void (*visit_break_statement) (ValaCCodeModule* self, ValaBreakStatement* stmt);
	void (*visit_continue_statement) (ValaCCodeModule* self, ValaContinueStatement* stmt);
	void (*visit_return_statement) (ValaCCodeModule* self, ValaReturnStatement* stmt);
	void (*visit_yield_statement) (ValaCCodeModule* self, ValaYieldStatement* stmt);
	void (*visit_throw_statement) (ValaCCodeModule* self, ValaThrowStatement* stmt);
	void (*visit_try_statement) (ValaCCodeModule* self, ValaTryStatement* stmt);
	void (*visit_catch_clause) (ValaCCodeModule* self, ValaCatchClause* clause);
	void (*visit_lock_statement) (ValaCCodeModule* self, ValaLockStatement* stmt);
	void (*visit_delete_statement) (ValaCCodeModule* self, ValaDeleteStatement* stmt);
	void (*visit_expression) (ValaCCodeModule* self, ValaExpression* expr);
	void (*visit_array_creation_expression) (ValaCCodeModule* self, ValaArrayCreationExpression* expr);
	void (*visit_boolean_literal) (ValaCCodeModule* self, ValaBooleanLiteral* expr);
	void (*visit_character_literal) (ValaCCodeModule* self, ValaCharacterLiteral* expr);
	void (*visit_integer_literal) (ValaCCodeModule* self, ValaIntegerLiteral* expr);
	void (*visit_real_literal) (ValaCCodeModule* self, ValaRealLiteral* expr);
	void (*visit_string_literal) (ValaCCodeModule* self, ValaStringLiteral* expr);
	void (*visit_null_literal) (ValaCCodeModule* self, ValaNullLiteral* expr);
	void (*visit_member_access) (ValaCCodeModule* self, ValaMemberAccess* expr);
	void (*visit_method_call) (ValaCCodeModule* self, ValaMethodCall* expr);
	void (*visit_element_access) (ValaCCodeModule* self, ValaElementAccess* expr);
	void (*visit_base_access) (ValaCCodeModule* self, ValaBaseAccess* expr);
	void (*visit_postfix_expression) (ValaCCodeModule* self, ValaPostfixExpression* expr);
	void (*visit_object_creation_expression) (ValaCCodeModule* self, ValaObjectCreationExpression* expr);
	void (*visit_sizeof_expression) (ValaCCodeModule* self, ValaSizeofExpression* expr);
	void (*visit_typeof_expression) (ValaCCodeModule* self, ValaTypeofExpression* expr);
	void (*visit_unary_expression) (ValaCCodeModule* self, ValaUnaryExpression* expr);
	void (*visit_cast_expression) (ValaCCodeModule* self, ValaCastExpression* expr);
	void (*visit_pointer_indirection) (ValaCCodeModule* self, ValaPointerIndirection* expr);
	void (*visit_addressof_expression) (ValaCCodeModule* self, ValaAddressofExpression* expr);
	void (*visit_reference_transfer_expression) (ValaCCodeModule* self, ValaReferenceTransferExpression* expr);
	void (*visit_binary_expression) (ValaCCodeModule* self, ValaBinaryExpression* expr);
	void (*visit_type_check) (ValaCCodeModule* self, ValaTypeCheck* expr);
	void (*visit_lambda_expression) (ValaCCodeModule* self, ValaLambdaExpression* l);
	void (*visit_assignment) (ValaCCodeModule* self, ValaAssignment* a);
	char* (*get_custom_creturn_type) (ValaCCodeModule* self, ValaMethod* m);
	void (*generate_dynamic_method_wrapper) (ValaCCodeModule* self, ValaDynamicMethod* method);
	gboolean (*method_has_wrapper) (ValaCCodeModule* self, ValaMethod* method);
	ValaCCodeIdentifier* (*get_value_setter_function) (ValaCCodeModule* self, ValaDataType* type_reference);
	ValaCCodeExpression* (*get_construct_property_assignment) (ValaCCodeModule* self, ValaCCodeConstant* canonical_cconstant, ValaDataType* property_type, ValaCCodeExpression* value);
	ValaCCodeFunctionCall* (*get_param_spec) (ValaCCodeModule* self, ValaProperty* prop);
	ValaCCodeFunctionCall* (*get_signal_creation) (ValaCCodeModule* self, ValaSignal* sig, ValaTypeSymbol* type);
	ValaCCodeFragment* (*register_dbus_info) (ValaCCodeModule* self, ValaObjectTypeSymbol* bindable);
	char* (*get_dynamic_property_getter_cname) (ValaCCodeModule* self, ValaDynamicProperty* node);
	char* (*get_dynamic_property_setter_cname) (ValaCCodeModule* self, ValaDynamicProperty* node);
	char* (*get_dynamic_signal_cname) (ValaCCodeModule* self, ValaDynamicSignal* node);
	char* (*get_dynamic_signal_connect_wrapper_name) (ValaCCodeModule* self, ValaDynamicSignal* node);
	char* (*get_dynamic_signal_disconnect_wrapper_name) (ValaCCodeModule* self, ValaDynamicSignal* node);
	void (*generate_marshaller) (ValaCCodeModule* self, GeeList* params, ValaDataType* return_type, gboolean dbus);
	char* (*get_marshaller_function) (ValaCCodeModule* self, GeeList* params, ValaDataType* return_type, const char* prefix, gboolean dbus);
	char* (*get_array_length_cname) (ValaCCodeModule* self, const char* array_cname, gint dim);
	ValaCCodeExpression* (*get_array_length_cexpression) (ValaCCodeModule* self, ValaExpression* array_expr, gint dim);
	char* (*get_array_size_cname) (ValaCCodeModule* self, const char* array_cname);
	ValaCCodeExpression* (*get_array_size_cexpression) (ValaCCodeModule* self, ValaExpression* array_expr);
	void (*add_simple_check) (ValaCCodeModule* self, ValaCodeNode* node, ValaCCodeFragment* cfrag);
};

/**
 * Code visitor generating C Code.
 */
struct _ValaCCodeBaseModule {
	ValaCCodeModule parent_instance;
	ValaCCodeBaseModulePrivate * priv;
	ValaSymbol* root_symbol;
	ValaSymbol* current_symbol;
	ValaTypeSymbol* current_type_symbol;
	ValaClass* current_class;
	ValaMethod* current_method;
	ValaDataType* current_return_type;
	ValaTryStatement* current_try;
	ValaPropertyAccessor* current_property_accessor;
	ValaCCodeDeclarationSpace* header_declarations;
	ValaCCodeDeclarationSpace* internal_header_declarations;
	ValaCCodeDeclarationSpace* source_declarations;
	ValaCCodeFragment* source_signal_marshaller_declaration;
	ValaCCodeFragment* source_type_member_definition;
	ValaCCodeFragment* class_init_fragment;
	ValaCCodeFragment* base_init_fragment;
	ValaCCodeFragment* class_finalize_fragment;
	ValaCCodeFragment* base_finalize_fragment;
	ValaCCodeFragment* instance_init_fragment;
	ValaCCodeFragment* instance_finalize_fragment;
	ValaCCodeFragment* source_signal_marshaller_definition;
	ValaCCodeFragment* module_init_fragment;
	ValaCCodeStruct* param_spec_struct;
	ValaCCodeStruct* closure_struct;
	ValaCCodeEnum* prop_enum;
	ValaCCodeFunction* function;
	ValaCCodeFragment* pre_statement_fragment;
	GeeArrayList* temp_vars;
	GeeArrayList* temp_ref_vars;
	GeeSet* user_marshal_set;
	GeeSet* predefined_marshal_set;
	gint next_temp_var_id;
	gboolean in_creation_method;
	gboolean in_constructor;
	gboolean in_static_or_class_ctor;
	gboolean current_method_inner_error;
	gint next_coroutine_state;
	ValaDataType* bool_type;
	ValaDataType* char_type;
	ValaDataType* uchar_type;
	ValaDataType* unichar_type;
	ValaDataType* short_type;
	ValaDataType* ushort_type;
	ValaDataType* int_type;
	ValaDataType* uint_type;
	ValaDataType* long_type;
	ValaDataType* ulong_type;
	ValaDataType* int8_type;
	ValaDataType* uint8_type;
	ValaDataType* int16_type;
	ValaDataType* uint16_type;
	ValaDataType* int32_type;
	ValaDataType* uint32_type;
	ValaDataType* int64_type;
	ValaDataType* uint64_type;
	ValaDataType* string_type;
	ValaDataType* float_type;
	ValaDataType* double_type;
	ValaTypeSymbol* gtype_type;
	ValaTypeSymbol* gobject_type;
	ValaErrorType* gerror_type;
	ValaClass* glist_type;
	ValaClass* gslist_type;
	ValaTypeSymbol* gstringbuilder_type;
	ValaTypeSymbol* garray_type;
	ValaTypeSymbol* gbytearray_type;
	ValaTypeSymbol* gptrarray_type;
	ValaDataType* gquark_type;
	ValaStruct* gvalue_type;
	ValaStruct* mutex_type;
	ValaTypeSymbol* type_module_type;
	ValaTypeSymbol* dbus_object_type;
	gboolean in_plugin;
	char* module_init_param_name;
	gboolean gvaluecollector_h_needed;
	gboolean gio_h_needed;
	gboolean requires_array_free;
	gboolean requires_array_move;
	gboolean requires_array_length;
	gboolean requires_strcmp0;
	gboolean dbus_glib_h_needed;
	gboolean dbus_glib_h_needed_in_header;
	GeeSet* wrappers;
	GeeMap* variable_name_map;
};

struct _ValaCCodeBaseModuleClass {
	ValaCCodeModuleClass parent_class;
	void (*append_vala_array_free) (ValaCCodeBaseModule* self);
	void (*append_vala_array_move) (ValaCCodeBaseModule* self);
	void (*append_vala_array_length) (ValaCCodeBaseModule* self);
	void (*generate_class_struct_declaration) (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_struct_declaration) (ValaCCodeBaseModule* self, ValaStruct* st, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_delegate_declaration) (ValaCCodeBaseModule* self, ValaDelegate* d, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_cparameters) (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, ValaCCodeFunction* func, ValaCCodeFunctionDeclarator* vdeclarator, GeeMap* carg_map, ValaCCodeFunctionCall* vcall, gint direction);
	ValaCCodeExpression* (*get_dup_func_expression) (ValaCCodeBaseModule* self, ValaDataType* type, ValaSourceReference* source_reference);
	ValaCCodeExpression* (*get_unref_expression) (ValaCCodeBaseModule* self, ValaCCodeExpression* cvar, ValaDataType* type, ValaExpression* expr);
	char* (*get_delegate_target_cname) (ValaCCodeBaseModule* self, const char* delegate_cname);
	ValaCCodeExpression* (*get_delegate_target_cexpression) (ValaCCodeBaseModule* self, ValaExpression* delegate_expr);
	char* (*get_delegate_target_destroy_notify_cname) (ValaCCodeBaseModule* self, const char* delegate_cname);
	ValaCCodeExpression* (*get_ref_cexpression) (ValaCCodeBaseModule* self, ValaDataType* expression_type, ValaCCodeExpression* cexpr, ValaExpression* expr, ValaCodeNode* node);
	void (*generate_class_declaration) (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_interface_declaration) (ValaCCodeBaseModule* self, ValaInterface* iface, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_method_declaration) (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_error_domain_declaration) (ValaCCodeBaseModule* self, ValaErrorDomain* edomain, ValaCCodeDeclarationSpace* decl_space);
	ValaCCodeExpression* (*get_implicit_cast_expression) (ValaCCodeBaseModule* self, ValaCCodeExpression* source_cexpr, ValaDataType* expression_type, ValaDataType* target_type, ValaExpression* expr);
};

struct _ValaCCodeBaseModulePrivate {
	ValaCodeContext* _context;
	GeeSet* reserved_identifiers;
	GeeSet* generated_external_symbols;
};

struct _ValaCCodeDeclarationSpace {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeDeclarationSpacePrivate * priv;
	ValaCCodeFragment* include_directives;
	ValaCCodeFragment* type_declaration;
	ValaCCodeFragment* type_definition;
	ValaCCodeFragment* type_member_declaration;
	ValaCCodeFragment* constant_declaration;
};

struct _ValaCCodeDeclarationSpaceClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeDeclarationSpace *self);
};



gpointer vala_ccode_module_ref (gpointer instance);
void vala_ccode_module_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_module (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_module (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_module (const GValue* value);
GType vala_ccode_module_get_type (void);
GType vala_ccode_base_module_get_type (void);
gpointer vala_ccode_declaration_space_ref (gpointer instance);
void vala_ccode_declaration_space_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_declaration_space (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_declaration_space (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_declaration_space (const GValue* value);
GType vala_ccode_declaration_space_get_type (void);
#define VALA_CCODE_BASE_MODULE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_BASE_MODULE, ValaCCodeBaseModulePrivate))
enum  {
	VALA_CCODE_BASE_MODULE_DUMMY_PROPERTY
};
GType vala_ccode_generator_get_type (void);
ValaCCodeModule* vala_ccode_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaCCodeModule* vala_ccode_module_construct (GType object_type, ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaCCodeBaseModule* vala_ccode_base_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaCCodeBaseModule* vala_ccode_base_module_construct (GType object_type, ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaCCodeBaseModule* vala_ccode_base_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next);
void vala_ccode_base_module_set_context (ValaCCodeBaseModule* self, ValaCodeContext* value);
ValaCCodeDeclarationSpace* vala_ccode_declaration_space_new (void);
ValaCCodeDeclarationSpace* vala_ccode_declaration_space_construct (GType object_type);
ValaCCodeGenerator* vala_ccode_module_get_codegen (ValaCCodeModule* self);
static char* vala_ccode_base_module_get_define_for_filename (const char* filename);
static void vala_ccode_base_module_real_emit (ValaCCodeModule* base, ValaCodeContext* context);
static ValaCCodeIdentifier* vala_ccode_base_module_real_get_value_setter_function (ValaCCodeModule* base, ValaDataType* type_reference);
void vala_ccode_base_module_append_vala_array_free (ValaCCodeBaseModule* self);
static void vala_ccode_base_module_real_append_vala_array_free (ValaCCodeBaseModule* self);
void vala_ccode_base_module_append_vala_array_move (ValaCCodeBaseModule* self);
static void vala_ccode_base_module_real_append_vala_array_move (ValaCCodeBaseModule* self);
void vala_ccode_base_module_append_vala_array_length (ValaCCodeBaseModule* self);
static void vala_ccode_base_module_real_append_vala_array_length (ValaCCodeBaseModule* self);
void vala_ccode_declaration_space_add_type_member_declaration (ValaCCodeDeclarationSpace* self, ValaCCodeNode* node);
static void vala_ccode_base_module_append_vala_strcmp0 (ValaCCodeBaseModule* self);
ValaCodeContext* vala_ccode_base_module_get_context (ValaCCodeBaseModule* self);
void vala_ccode_declaration_space_add_include (ValaCCodeDeclarationSpace* self, const char* filename, gboolean local);
void vala_ccode_declaration_space_add_type_definition (ValaCCodeDeclarationSpace* self, ValaCCodeNode* node);
void vala_ccode_declaration_space_add_type_declaration (ValaCCodeDeclarationSpace* self, ValaCCodeNode* node);
static void vala_ccode_base_module_real_visit_source_file (ValaCCodeModule* base, ValaSourceFile* source_file);
gboolean vala_ccode_declaration_space_add_symbol_declaration (ValaCCodeDeclarationSpace* self, ValaSymbol* sym, const char* name);
void vala_ccode_base_module_generate_enum_declaration (ValaCCodeBaseModule* self, ValaEnum* en, ValaCCodeDeclarationSpace* decl_space);
static void vala_ccode_base_module_real_visit_enum (ValaCCodeModule* base, ValaEnum* en);
char* vala_ccode_base_module_get_symbol_lock_name (ValaCCodeBaseModule* self, ValaSymbol* sym);
static void vala_ccode_base_module_real_visit_member (ValaCCodeModule* base, ValaMember* m);
void vala_ccode_declaration_space_add_constant_declaration (ValaCCodeDeclarationSpace* self, ValaCCodeNode* node);
void vala_ccode_base_module_generate_constant_declaration (ValaCCodeBaseModule* self, ValaConstant* c, ValaCCodeDeclarationSpace* decl_space);
static void vala_ccode_base_module_real_visit_constant (ValaCCodeModule* base, ValaConstant* c);
ValaCCodeModule* vala_ccode_module_get_head (ValaCCodeModule* self);
char* vala_ccode_module_get_array_length_cname (ValaCCodeModule* self, const char* array_cname, gint dim);
char* vala_ccode_base_module_get_delegate_target_cname (ValaCCodeBaseModule* self, const char* delegate_cname);
void vala_ccode_base_module_generate_field_declaration (ValaCCodeBaseModule* self, ValaField* f, ValaCCodeDeclarationSpace* decl_space);
void vala_ccode_base_module_check_type (ValaCCodeBaseModule* self, ValaDataType* type);
ValaDataType* vala_ccode_base_module_get_data_type_for_symbol (ValaTypeSymbol* sym);
ValaCCodeExpression* vala_ccode_module_get_array_length_cexpression (ValaCCodeModule* self, ValaExpression* array_expr, gint dim);
void vala_ccode_base_module_append_temp_decl (ValaCCodeBaseModule* self, ValaCCodeFragment* cfrag, GeeList* temp_vars);
gboolean vala_ccode_base_module_requires_destroy (ValaCCodeBaseModule* self, ValaDataType* type);
ValaCCodeExpression* vala_ccode_base_module_get_unref_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* cvar, ValaDataType* type, ValaExpression* expr);
ValaCCodeExpression* vala_ccode_base_module_default_value_for_type (ValaCCodeBaseModule* self, ValaDataType* type, gboolean initializer_expression);
gboolean vala_ccode_base_module_is_constant_ccode_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* cexpr);
char* vala_ccode_module_get_array_size_cname (ValaCCodeModule* self, const char* array_cname);
static void vala_ccode_base_module_real_visit_field (ValaCCodeModule* base, ValaField* f);
gboolean vala_ccode_base_module_is_pure_ccode_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* cexpr);
static void vala_ccode_base_module_real_visit_formal_parameter (ValaCCodeModule* base, ValaFormalParameter* p);
static void vala_ccode_base_module_real_visit_property (ValaCCodeModule* base, ValaProperty* prop);
void vala_ccode_base_module_generate_class_declaration (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space);
void vala_ccode_base_module_generate_interface_declaration (ValaCCodeBaseModule* self, ValaInterface* iface, ValaCCodeDeclarationSpace* decl_space);
void vala_ccode_base_module_generate_delegate_declaration (ValaCCodeBaseModule* self, ValaDelegate* d, ValaCCodeDeclarationSpace* decl_space);
void vala_ccode_base_module_generate_struct_declaration (ValaCCodeBaseModule* self, ValaStruct* st, ValaCCodeDeclarationSpace* decl_space);
void vala_ccode_base_module_generate_type_declaration (ValaCCodeBaseModule* self, ValaDataType* type, ValaCCodeDeclarationSpace* decl_space);
void vala_ccode_base_module_generate_error_domain_declaration (ValaCCodeBaseModule* self, ValaErrorDomain* edomain, ValaCCodeDeclarationSpace* decl_space);
void vala_ccode_base_module_generate_class_struct_declaration (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space);
static void vala_ccode_base_module_real_generate_class_struct_declaration (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space);
static void vala_ccode_base_module_real_generate_struct_declaration (ValaCCodeBaseModule* self, ValaStruct* st, ValaCCodeDeclarationSpace* decl_space);
static void vala_ccode_base_module_real_generate_delegate_declaration (ValaCCodeBaseModule* self, ValaDelegate* d, ValaCCodeDeclarationSpace* decl_space);
void vala_ccode_base_module_generate_cparameters (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, ValaCCodeFunction* func, ValaCCodeFunctionDeclarator* vdeclarator, GeeMap* carg_map, ValaCCodeFunctionCall* vcall, gint direction);
static void vala_ccode_base_module_real_generate_cparameters (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, ValaCCodeFunction* func, ValaCCodeFunctionDeclarator* vdeclarator, GeeMap* carg_map, ValaCCodeFunctionCall* vcall, gint direction);
void vala_ccode_base_module_generate_property_accessor_declaration (ValaCCodeBaseModule* self, ValaPropertyAccessor* acc, ValaCCodeDeclarationSpace* decl_space);
ValaCCodeExpression* vala_ccode_base_module_transform_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* source_cexpr, ValaDataType* expression_type, ValaDataType* target_type, ValaExpression* expr);
static ValaCCodeStatement* vala_ccode_base_module_create_property_type_check_statement (ValaCCodeBaseModule* self, ValaProperty* prop, gboolean check_return_type, ValaTypeSymbol* t, gboolean non_null, const char* var_name);
static void vala_ccode_base_module_real_visit_property_accessor (ValaCCodeModule* base, ValaPropertyAccessor* acc);
static void vala_ccode_base_module_real_visit_destructor (ValaCCodeModule* base, ValaDestructor* d);
ValaCCodeExpression* vala_ccode_base_module_get_variable_cexpression (ValaCCodeBaseModule* self, const char* name);
static void vala_ccode_base_module_real_visit_block (ValaCCodeModule* base, ValaBlock* b);
static void vala_ccode_base_module_real_visit_empty_statement (ValaCCodeModule* base, ValaEmptyStatement* stmt);
void vala_ccode_base_module_create_temp_decl (ValaCCodeBaseModule* self, ValaStatement* stmt, GeeList* temp_vars);
static void vala_ccode_base_module_real_visit_declaration_statement (ValaCCodeModule* base, ValaDeclarationStatement* stmt);
char* vala_ccode_base_module_get_variable_cname (ValaCCodeBaseModule* self, const char* name);
ValaLocalVariable* vala_ccode_base_module_get_temp_variable (ValaCCodeBaseModule* self, ValaDataType* type, gboolean value_owned, ValaCodeNode* node_reference);
ValaCCodeExpression* vala_ccode_base_module_get_delegate_target_cexpression (ValaCCodeBaseModule* self, ValaExpression* delegate_expr);
void vala_ccode_module_add_simple_check (ValaCCodeModule* self, ValaCodeNode* node, ValaCCodeFragment* cfrag);
static void vala_ccode_base_module_real_visit_local_variable (ValaCCodeModule* base, ValaLocalVariable* local);
static void vala_ccode_base_module_real_visit_initializer_list (ValaCCodeModule* base, ValaInitializerList* list);
static ValaCCodeExpression* vala_ccode_base_module_get_type_id_expression (ValaCCodeBaseModule* self, ValaDataType* type);
static char* vala_ccode_base_module_generate_struct_dup_wrapper (ValaCCodeBaseModule* self, ValaValueType* value_type);
ValaCCodeExpression* vala_ccode_base_module_get_dup_func_expression (ValaCCodeBaseModule* self, ValaDataType* type, ValaSourceReference* source_reference);
static ValaCCodeExpression* vala_ccode_base_module_real_get_dup_func_expression (ValaCCodeBaseModule* self, ValaDataType* type, ValaSourceReference* source_reference);
gboolean vala_ccode_base_module_add_wrapper (ValaCCodeBaseModule* self, const char* wrapper_name);
ValaCCodeExpression* vala_ccode_base_module_get_destroy_func_expression (ValaCCodeBaseModule* self, ValaDataType* type);
static char* vala_ccode_base_module_generate_glist_free_wrapper (ValaCCodeBaseModule* self, ValaDataType* list_type, ValaCCodeIdentifier* element_destroy_func_expression);
static ValaCCodeExpression* vala_ccode_base_module_real_get_unref_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* cvar, ValaDataType* type, ValaExpression* expr);
static void vala_ccode_base_module_real_visit_end_full_expression (ValaCCodeModule* base, ValaExpression* expr);
static void vala_ccode_base_module_real_visit_expression_statement (ValaCCodeModule* base, ValaExpressionStatement* stmt);
void vala_ccode_base_module_append_local_free (ValaCCodeBaseModule* self, ValaSymbol* sym, ValaCCodeFragment* cfrag, gboolean stop_at_loop);
static void vala_ccode_base_module_append_param_free (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeFragment* cfrag);
void vala_ccode_base_module_append_error_free (ValaCCodeBaseModule* self, ValaSymbol* sym, ValaCCodeFragment* cfrag, ValaTryStatement* current_try);
void vala_ccode_base_module_create_local_free (ValaCCodeBaseModule* self, ValaCodeNode* stmt, gboolean stop_at_loop);
static gboolean vala_ccode_base_module_append_local_free_expr (ValaCCodeBaseModule* self, ValaSymbol* sym, ValaCCodeCommaExpression* ccomma, gboolean stop_at_loop);
static gboolean vala_ccode_base_module_append_param_free_expr (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeCommaExpression* ccomma);
static void vala_ccode_base_module_create_local_free_expr (ValaCCodeBaseModule* self, ValaExpression* expr);
static void vala_ccode_base_module_real_visit_return_statement (ValaCCodeModule* base, ValaReturnStatement* stmt);
static void vala_ccode_base_module_real_visit_lock_statement (ValaCCodeModule* base, ValaLockStatement* stmt);
static void vala_ccode_base_module_real_visit_delete_statement (ValaCCodeModule* base, ValaDeleteStatement* stmt);
ValaCCodeExpression* vala_ccode_base_module_convert_from_generic_pointer (ValaCCodeBaseModule* self, ValaCCodeExpression* cexpr, ValaDataType* actual_type);
ValaCCodeExpression* vala_ccode_base_module_convert_to_generic_pointer (ValaCCodeBaseModule* self, ValaCCodeExpression* cexpr, ValaDataType* actual_type);
static void vala_ccode_base_module_real_visit_expression (ValaCCodeModule* base, ValaExpression* expr);
static void vala_ccode_base_module_real_visit_boolean_literal (ValaCCodeModule* base, ValaBooleanLiteral* expr);
static void vala_ccode_base_module_real_visit_character_literal (ValaCCodeModule* base, ValaCharacterLiteral* expr);
static void vala_ccode_base_module_real_visit_integer_literal (ValaCCodeModule* base, ValaIntegerLiteral* expr);
static void vala_ccode_base_module_real_visit_real_literal (ValaCCodeModule* base, ValaRealLiteral* expr);
static void vala_ccode_base_module_real_visit_string_literal (ValaCCodeModule* base, ValaStringLiteral* expr);
static void vala_ccode_base_module_real_visit_null_literal (ValaCCodeModule* base, ValaNullLiteral* expr);
static char* vala_ccode_base_module_real_get_delegate_target_cname (ValaCCodeBaseModule* self, const char* delegate_cname);
static ValaCCodeExpression* vala_ccode_base_module_real_get_delegate_target_cexpression (ValaCCodeBaseModule* self, ValaExpression* delegate_expr);
char* vala_ccode_base_module_get_delegate_target_destroy_notify_cname (ValaCCodeBaseModule* self, const char* delegate_cname);
static char* vala_ccode_base_module_real_get_delegate_target_destroy_notify_cname (ValaCCodeBaseModule* self, const char* delegate_cname);
static void vala_ccode_base_module_real_visit_base_access (ValaCCodeModule* base, ValaBaseAccess* expr);
static ValaMemberAccess* vala_ccode_base_module_find_property_access (ValaCCodeBaseModule* self, ValaExpression* expr);
ValaCCodeFunctionCall* vala_ccode_base_module_get_property_set_call (ValaCCodeBaseModule* self, ValaProperty* prop, ValaMemberAccess* ma, ValaCCodeExpression* cexpr);
static void vala_ccode_base_module_real_visit_postfix_expression (ValaCCodeModule* base, ValaPostfixExpression* expr);
gboolean vala_ccode_base_module_requires_copy (ValaCCodeBaseModule* self, ValaDataType* type);
static gboolean vala_ccode_base_module_is_ref_function_void (ValaCCodeBaseModule* self, ValaDataType* type);
ValaCCodeExpression* vala_ccode_base_module_get_ref_cexpression (ValaCCodeBaseModule* self, ValaDataType* expression_type, ValaCCodeExpression* cexpr, ValaExpression* expr, ValaCodeNode* node);
static ValaCCodeExpression* vala_ccode_base_module_real_get_ref_cexpression (ValaCCodeBaseModule* self, ValaDataType* expression_type, ValaCCodeExpression* cexpr, ValaExpression* expr, ValaCodeNode* node);
static gboolean vala_ccode_base_module_is_reference_type_argument (ValaCCodeBaseModule* self, ValaDataType* type_arg);
static gboolean vala_ccode_base_module_is_nullable_value_type_argument (ValaCCodeBaseModule* self, ValaDataType* type_arg);
static gboolean vala_ccode_base_module_is_signed_integer_type_argument (ValaCCodeBaseModule* self, ValaDataType* type_arg);
static gboolean vala_ccode_base_module_is_unsigned_integer_type_argument (ValaCCodeBaseModule* self, ValaDataType* type_arg);
static void vala_ccode_base_module_check_type_argument (ValaCCodeBaseModule* self, ValaDataType* type_arg);
static void vala_ccode_base_module_real_generate_class_declaration (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space);
static void vala_ccode_base_module_real_generate_interface_declaration (ValaCCodeBaseModule* self, ValaInterface* iface, ValaCCodeDeclarationSpace* decl_space);
void vala_ccode_base_module_generate_method_declaration (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space);
static void vala_ccode_base_module_real_generate_method_declaration (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space);
static void vala_ccode_base_module_real_generate_error_domain_declaration (ValaCCodeBaseModule* self, ValaErrorDomain* edomain, ValaCCodeDeclarationSpace* decl_space);
gint vala_ccode_base_module_get_param_pos (ValaCCodeBaseModule* self, double param_pos, gboolean ellipsis);
ValaCCodeExpression* vala_ccode_base_module_handle_struct_argument (ValaCCodeBaseModule* self, ValaFormalParameter* param, ValaExpression* arg, ValaCCodeExpression* cexpr);
char* vala_ccode_module_get_custom_creturn_type (ValaCCodeModule* self, ValaMethod* m);
static void vala_ccode_base_module_real_visit_object_creation_expression (ValaCCodeModule* base, ValaObjectCreationExpression* expr);
static void vala_ccode_base_module_real_visit_sizeof_expression (ValaCCodeModule* base, ValaSizeofExpression* expr);
static void vala_ccode_base_module_real_visit_typeof_expression (ValaCCodeModule* base, ValaTypeofExpression* expr);
static void vala_ccode_base_module_real_visit_unary_expression (ValaCCodeModule* base, ValaUnaryExpression* expr);
static ValaCCodeExpression* vala_ccode_base_module_create_type_check (ValaCCodeBaseModule* self, ValaCCodeNode* ccodenode, ValaDataType* type);
static void vala_ccode_base_module_real_visit_cast_expression (ValaCCodeModule* base, ValaCastExpression* expr);
static void vala_ccode_base_module_real_visit_pointer_indirection (ValaCCodeModule* base, ValaPointerIndirection* expr);
static void vala_ccode_base_module_real_visit_addressof_expression (ValaCCodeModule* base, ValaAddressofExpression* expr);
static void vala_ccode_base_module_real_visit_reference_transfer_expression (ValaCCodeModule* base, ValaReferenceTransferExpression* expr);
static void vala_ccode_base_module_real_visit_binary_expression (ValaCCodeModule* base, ValaBinaryExpression* expr);
char* vala_ccode_base_module_get_type_check_function (ValaCCodeBaseModule* self, ValaTypeSymbol* type);
static void vala_ccode_base_module_real_visit_type_check (ValaCCodeModule* base, ValaTypeCheck* expr);
static void vala_ccode_base_module_real_visit_lambda_expression (ValaCCodeModule* base, ValaLambdaExpression* l);
ValaCCodeExpression* vala_ccode_base_module_get_implicit_cast_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* source_cexpr, ValaDataType* expression_type, ValaDataType* target_type, ValaExpression* expr);
static ValaCCodeExpression* vala_ccode_base_module_real_get_implicit_cast_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* source_cexpr, ValaDataType* expression_type, ValaDataType* target_type, ValaExpression* expr);
ValaCCodeNode* vala_ccode_base_module_get_ccodenode (ValaCCodeBaseModule* self, ValaCodeNode* node);
char* vala_ccode_module_get_dynamic_property_setter_cname (ValaCCodeModule* self, ValaDynamicProperty* node);
gboolean vala_ccode_base_module_is_address_of_possible (ValaCCodeBaseModule* self, ValaExpression* e);
ValaCCodeExpression* vala_ccode_base_module_get_address_of_expression (ValaCCodeBaseModule* self, ValaExpression* e, ValaCCodeExpression* ce);
gboolean vala_ccode_base_module_add_generated_external_symbol (ValaCCodeBaseModule* self, ValaSymbol* external_symbol);
ValaCCodeStatement* vala_ccode_base_module_create_type_check_statement (ValaCCodeBaseModule* self, ValaCodeNode* method_node, ValaDataType* ret_type, ValaTypeSymbol* t, gboolean non_null, const char* var_name);
static void vala_ccode_base_module_real_visit_class (ValaCCodeModule* base, ValaClass* cl);
static gpointer vala_ccode_base_module_parent_class = NULL;
static void vala_ccode_base_module_finalize (ValaCCodeModule* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



ValaCCodeBaseModule* vala_ccode_base_module_construct (GType object_type, ValaCCodeGenerator* codegen, ValaCCodeModule* next) {
	ValaCCodeBaseModule* self;
	GeeSet* _tmp0_;
	GeeSet* _tmp1_;
	g_return_val_if_fail (codegen != NULL, NULL);
	self = (ValaCCodeBaseModule*) vala_ccode_module_construct (object_type, codegen, next);
	_tmp0_ = NULL;
	self->predefined_marshal_set = (_tmp0_ = (GeeSet*) gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal), (self->predefined_marshal_set == NULL) ? NULL : (self->predefined_marshal_set = (gee_collection_object_unref (self->predefined_marshal_set), NULL)), _tmp0_);
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:VOID");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:BOOLEAN");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:CHAR");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:UCHAR");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:INT");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:UINT");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:LONG");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:ULONG");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:ENUM");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:FLAGS");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:FLOAT");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:DOUBLE");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:STRING");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:POINTER");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:OBJECT");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "STRING:OBJECT,POINTER");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "VOID:UINT,POINTER");
	gee_collection_add ((GeeCollection*) self->predefined_marshal_set, "BOOLEAN:FLAGS");
	_tmp1_ = NULL;
	self->priv->reserved_identifiers = (_tmp1_ = (GeeSet*) gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal), (self->priv->reserved_identifiers == NULL) ? NULL : (self->priv->reserved_identifiers = (gee_collection_object_unref (self->priv->reserved_identifiers), NULL)), _tmp1_);
	/* C99 keywords*/
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "_Bool");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "_Complex");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "_Imaginary");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "auto");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "break");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "case");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "char");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "const");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "continue");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "default");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "do");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "double");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "else");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "enum");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "extern");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "float");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "for");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "goto");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "if");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "inline");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "int");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "long");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "register");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "restrict");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "return");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "short");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "signed");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "sizeof");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "static");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "struct");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "switch");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "typedef");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "union");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "unsigned");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "void");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "volatile");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "while");
	/* MSVC keywords*/
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "cdecl");
	/* reserved for Vala/GObject naming conventions*/
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "error");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "result");
	gee_collection_add ((GeeCollection*) self->priv->reserved_identifiers, "self");
	return self;
}


ValaCCodeBaseModule* vala_ccode_base_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next) {
	return vala_ccode_base_module_construct (VALA_TYPE_CCODE_BASE_MODULE, codegen, next);
}


static void vala_ccode_base_module_real_emit (ValaCCodeModule* base, ValaCodeContext* context) {
	ValaCCodeBaseModule * self;
	ValaSymbol* _tmp1_;
	ValaSymbol* _tmp0_;
	ValaDataType* _tmp3_;
	ValaStruct* _tmp2_;
	ValaDataType* _tmp5_;
	ValaStruct* _tmp4_;
	ValaDataType* _tmp7_;
	ValaStruct* _tmp6_;
	ValaDataType* _tmp9_;
	ValaStruct* _tmp8_;
	ValaDataType* _tmp11_;
	ValaStruct* _tmp10_;
	ValaDataType* _tmp13_;
	ValaStruct* _tmp12_;
	ValaDataType* _tmp15_;
	ValaStruct* _tmp14_;
	ValaDataType* _tmp17_;
	ValaStruct* _tmp16_;
	ValaDataType* _tmp19_;
	ValaStruct* _tmp18_;
	ValaDataType* _tmp21_;
	ValaStruct* _tmp20_;
	ValaDataType* _tmp23_;
	ValaStruct* _tmp22_;
	ValaDataType* _tmp25_;
	ValaStruct* _tmp24_;
	ValaDataType* _tmp27_;
	ValaStruct* _tmp26_;
	ValaDataType* _tmp29_;
	ValaStruct* _tmp28_;
	ValaDataType* _tmp31_;
	ValaStruct* _tmp30_;
	ValaDataType* _tmp33_;
	ValaStruct* _tmp32_;
	ValaDataType* _tmp35_;
	ValaStruct* _tmp34_;
	ValaDataType* _tmp37_;
	ValaStruct* _tmp36_;
	ValaDataType* _tmp39_;
	ValaStruct* _tmp38_;
	ValaDataType* _tmp41_;
	ValaClass* _tmp40_;
	ValaStruct* unichar_struct;
	ValaCCodeDeclarationSpace* _tmp63_;
	ValaCCodeDeclarationSpace* _tmp64_;
	GeeList* source_files;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (context != NULL);
	vala_ccode_base_module_set_context (self, context);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->root_symbol = (_tmp1_ = (_tmp0_ = (ValaSymbol*) vala_code_context_get_root (context), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_)), (self->root_symbol == NULL) ? NULL : (self->root_symbol = (vala_code_node_unref (self->root_symbol), NULL)), _tmp1_);
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	self->bool_type = (_tmp3_ = (ValaDataType*) vala_boolean_type_new (_tmp2_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "bool"))), (self->bool_type == NULL) ? NULL : (self->bool_type = (vala_code_node_unref (self->bool_type), NULL)), _tmp3_);
	(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_code_node_unref (_tmp2_), NULL));
	_tmp5_ = NULL;
	_tmp4_ = NULL;
	self->char_type = (_tmp5_ = (ValaDataType*) vala_integer_type_new (_tmp4_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "char")), NULL, NULL), (self->char_type == NULL) ? NULL : (self->char_type = (vala_code_node_unref (self->char_type), NULL)), _tmp5_);
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_code_node_unref (_tmp4_), NULL));
	_tmp7_ = NULL;
	_tmp6_ = NULL;
	self->uchar_type = (_tmp7_ = (ValaDataType*) vala_integer_type_new (_tmp6_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "uchar")), NULL, NULL), (self->uchar_type == NULL) ? NULL : (self->uchar_type = (vala_code_node_unref (self->uchar_type), NULL)), _tmp7_);
	(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_code_node_unref (_tmp6_), NULL));
	_tmp9_ = NULL;
	_tmp8_ = NULL;
	self->short_type = (_tmp9_ = (ValaDataType*) vala_integer_type_new (_tmp8_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "short")), NULL, NULL), (self->short_type == NULL) ? NULL : (self->short_type = (vala_code_node_unref (self->short_type), NULL)), _tmp9_);
	(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_code_node_unref (_tmp8_), NULL));
	_tmp11_ = NULL;
	_tmp10_ = NULL;
	self->ushort_type = (_tmp11_ = (ValaDataType*) vala_integer_type_new (_tmp10_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "ushort")), NULL, NULL), (self->ushort_type == NULL) ? NULL : (self->ushort_type = (vala_code_node_unref (self->ushort_type), NULL)), _tmp11_);
	(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_code_node_unref (_tmp10_), NULL));
	_tmp13_ = NULL;
	_tmp12_ = NULL;
	self->int_type = (_tmp13_ = (ValaDataType*) vala_integer_type_new (_tmp12_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "int")), NULL, NULL), (self->int_type == NULL) ? NULL : (self->int_type = (vala_code_node_unref (self->int_type), NULL)), _tmp13_);
	(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_code_node_unref (_tmp12_), NULL));
	_tmp15_ = NULL;
	_tmp14_ = NULL;
	self->uint_type = (_tmp15_ = (ValaDataType*) vala_integer_type_new (_tmp14_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "uint")), NULL, NULL), (self->uint_type == NULL) ? NULL : (self->uint_type = (vala_code_node_unref (self->uint_type), NULL)), _tmp15_);
	(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_code_node_unref (_tmp14_), NULL));
	_tmp17_ = NULL;
	_tmp16_ = NULL;
	self->long_type = (_tmp17_ = (ValaDataType*) vala_integer_type_new (_tmp16_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "long")), NULL, NULL), (self->long_type == NULL) ? NULL : (self->long_type = (vala_code_node_unref (self->long_type), NULL)), _tmp17_);
	(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_code_node_unref (_tmp16_), NULL));
	_tmp19_ = NULL;
	_tmp18_ = NULL;
	self->ulong_type = (_tmp19_ = (ValaDataType*) vala_integer_type_new (_tmp18_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "ulong")), NULL, NULL), (self->ulong_type == NULL) ? NULL : (self->ulong_type = (vala_code_node_unref (self->ulong_type), NULL)), _tmp19_);
	(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_code_node_unref (_tmp18_), NULL));
	_tmp21_ = NULL;
	_tmp20_ = NULL;
	self->int8_type = (_tmp21_ = (ValaDataType*) vala_integer_type_new (_tmp20_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "int8")), NULL, NULL), (self->int8_type == NULL) ? NULL : (self->int8_type = (vala_code_node_unref (self->int8_type), NULL)), _tmp21_);
	(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_code_node_unref (_tmp20_), NULL));
	_tmp23_ = NULL;
	_tmp22_ = NULL;
	self->uint8_type = (_tmp23_ = (ValaDataType*) vala_integer_type_new (_tmp22_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "uint8")), NULL, NULL), (self->uint8_type == NULL) ? NULL : (self->uint8_type = (vala_code_node_unref (self->uint8_type), NULL)), _tmp23_);
	(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_code_node_unref (_tmp22_), NULL));
	_tmp25_ = NULL;
	_tmp24_ = NULL;
	self->int16_type = (_tmp25_ = (ValaDataType*) vala_integer_type_new (_tmp24_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "int16")), NULL, NULL), (self->int16_type == NULL) ? NULL : (self->int16_type = (vala_code_node_unref (self->int16_type), NULL)), _tmp25_);
	(_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_code_node_unref (_tmp24_), NULL));
	_tmp27_ = NULL;
	_tmp26_ = NULL;
	self->uint16_type = (_tmp27_ = (ValaDataType*) vala_integer_type_new (_tmp26_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "uint16")), NULL, NULL), (self->uint16_type == NULL) ? NULL : (self->uint16_type = (vala_code_node_unref (self->uint16_type), NULL)), _tmp27_);
	(_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_code_node_unref (_tmp26_), NULL));
	_tmp29_ = NULL;
	_tmp28_ = NULL;
	self->int32_type = (_tmp29_ = (ValaDataType*) vala_integer_type_new (_tmp28_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "int32")), NULL, NULL), (self->int32_type == NULL) ? NULL : (self->int32_type = (vala_code_node_unref (self->int32_type), NULL)), _tmp29_);
	(_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_code_node_unref (_tmp28_), NULL));
	_tmp31_ = NULL;
	_tmp30_ = NULL;
	self->uint32_type = (_tmp31_ = (ValaDataType*) vala_integer_type_new (_tmp30_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "uint32")), NULL, NULL), (self->uint32_type == NULL) ? NULL : (self->uint32_type = (vala_code_node_unref (self->uint32_type), NULL)), _tmp31_);
	(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_code_node_unref (_tmp30_), NULL));
	_tmp33_ = NULL;
	_tmp32_ = NULL;
	self->int64_type = (_tmp33_ = (ValaDataType*) vala_integer_type_new (_tmp32_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "int64")), NULL, NULL), (self->int64_type == NULL) ? NULL : (self->int64_type = (vala_code_node_unref (self->int64_type), NULL)), _tmp33_);
	(_tmp32_ == NULL) ? NULL : (_tmp32_ = (vala_code_node_unref (_tmp32_), NULL));
	_tmp35_ = NULL;
	_tmp34_ = NULL;
	self->uint64_type = (_tmp35_ = (ValaDataType*) vala_integer_type_new (_tmp34_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "uint64")), NULL, NULL), (self->uint64_type == NULL) ? NULL : (self->uint64_type = (vala_code_node_unref (self->uint64_type), NULL)), _tmp35_);
	(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_code_node_unref (_tmp34_), NULL));
	_tmp37_ = NULL;
	_tmp36_ = NULL;
	self->float_type = (_tmp37_ = (ValaDataType*) vala_floating_type_new (_tmp36_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "float"))), (self->float_type == NULL) ? NULL : (self->float_type = (vala_code_node_unref (self->float_type), NULL)), _tmp37_);
	(_tmp36_ == NULL) ? NULL : (_tmp36_ = (vala_code_node_unref (_tmp36_), NULL));
	_tmp39_ = NULL;
	_tmp38_ = NULL;
	self->double_type = (_tmp39_ = (ValaDataType*) vala_floating_type_new (_tmp38_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "double"))), (self->double_type == NULL) ? NULL : (self->double_type = (vala_code_node_unref (self->double_type), NULL)), _tmp39_);
	(_tmp38_ == NULL) ? NULL : (_tmp38_ = (vala_code_node_unref (_tmp38_), NULL));
	_tmp41_ = NULL;
	_tmp40_ = NULL;
	self->string_type = (_tmp41_ = (ValaDataType*) vala_object_type_new ((ValaObjectTypeSymbol*) (_tmp40_ = VALA_CLASS (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "string")))), (self->string_type == NULL) ? NULL : (self->string_type = (vala_code_node_unref (self->string_type), NULL)), _tmp41_);
	(_tmp40_ == NULL) ? NULL : (_tmp40_ = (vala_code_node_unref (_tmp40_), NULL));
	unichar_struct = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "unichar"));
	if (unichar_struct != NULL) {
		ValaDataType* _tmp42_;
		_tmp42_ = NULL;
		self->unichar_type = (_tmp42_ = (ValaDataType*) vala_integer_type_new (unichar_struct, NULL, NULL), (self->unichar_type == NULL) ? NULL : (self->unichar_type = (vala_code_node_unref (self->unichar_type), NULL)), _tmp42_);
	}
	if (vala_code_context_get_profile (context) == VALA_PROFILE_GOBJECT) {
		ValaSymbol* glib_ns;
		ValaTypeSymbol* _tmp43_;
		ValaTypeSymbol* _tmp44_;
		ValaErrorType* _tmp45_;
		ValaClass* _tmp46_;
		ValaClass* _tmp47_;
		ValaTypeSymbol* _tmp48_;
		ValaTypeSymbol* _tmp49_;
		ValaTypeSymbol* _tmp50_;
		ValaTypeSymbol* _tmp51_;
		ValaDataType* _tmp53_;
		ValaStruct* _tmp52_;
		ValaStruct* _tmp54_;
		ValaStruct* _tmp55_;
		ValaTypeSymbol* _tmp56_;
		ValaSymbol* dbus_ns;
		glib_ns = vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "GLib");
		_tmp43_ = NULL;
		self->gtype_type = (_tmp43_ = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Type")), (self->gtype_type == NULL) ? NULL : (self->gtype_type = (vala_code_node_unref (self->gtype_type), NULL)), _tmp43_);
		_tmp44_ = NULL;
		self->gobject_type = (_tmp44_ = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Object")), (self->gobject_type == NULL) ? NULL : (self->gobject_type = (vala_code_node_unref (self->gobject_type), NULL)), _tmp44_);
		_tmp45_ = NULL;
		self->gerror_type = (_tmp45_ = vala_error_type_new (NULL, NULL, NULL), (self->gerror_type == NULL) ? NULL : (self->gerror_type = (vala_code_node_unref (self->gerror_type), NULL)), _tmp45_);
		_tmp46_ = NULL;
		self->glist_type = (_tmp46_ = VALA_CLASS (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "List")), (self->glist_type == NULL) ? NULL : (self->glist_type = (vala_code_node_unref (self->glist_type), NULL)), _tmp46_);
		_tmp47_ = NULL;
		self->gslist_type = (_tmp47_ = VALA_CLASS (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "SList")), (self->gslist_type == NULL) ? NULL : (self->gslist_type = (vala_code_node_unref (self->gslist_type), NULL)), _tmp47_);
		_tmp48_ = NULL;
		self->gstringbuilder_type = (_tmp48_ = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "StringBuilder")), (self->gstringbuilder_type == NULL) ? NULL : (self->gstringbuilder_type = (vala_code_node_unref (self->gstringbuilder_type), NULL)), _tmp48_);
		_tmp49_ = NULL;
		self->garray_type = (_tmp49_ = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Array")), (self->garray_type == NULL) ? NULL : (self->garray_type = (vala_code_node_unref (self->garray_type), NULL)), _tmp49_);
		_tmp50_ = NULL;
		self->gbytearray_type = (_tmp50_ = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "ByteArray")), (self->gbytearray_type == NULL) ? NULL : (self->gbytearray_type = (vala_code_node_unref (self->gbytearray_type), NULL)), _tmp50_);
		_tmp51_ = NULL;
		self->gptrarray_type = (_tmp51_ = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "PtrArray")), (self->gptrarray_type == NULL) ? NULL : (self->gptrarray_type = (vala_code_node_unref (self->gptrarray_type), NULL)), _tmp51_);
		_tmp53_ = NULL;
		_tmp52_ = NULL;
		self->gquark_type = (_tmp53_ = (ValaDataType*) vala_integer_type_new (_tmp52_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Quark")), NULL, NULL), (self->gquark_type == NULL) ? NULL : (self->gquark_type = (vala_code_node_unref (self->gquark_type), NULL)), _tmp53_);
		(_tmp52_ == NULL) ? NULL : (_tmp52_ = (vala_code_node_unref (_tmp52_), NULL));
		_tmp54_ = NULL;
		self->gvalue_type = (_tmp54_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Value")), (self->gvalue_type == NULL) ? NULL : (self->gvalue_type = (vala_code_node_unref (self->gvalue_type), NULL)), _tmp54_);
		_tmp55_ = NULL;
		self->mutex_type = (_tmp55_ = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "StaticRecMutex")), (self->mutex_type == NULL) ? NULL : (self->mutex_type = (vala_code_node_unref (self->mutex_type), NULL)), _tmp55_);
		_tmp56_ = NULL;
		self->type_module_type = (_tmp56_ = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "TypeModule")), (self->type_module_type == NULL) ? NULL : (self->type_module_type = (vala_code_node_unref (self->type_module_type), NULL)), _tmp56_);
		if (vala_code_context_get_module_init_method (context) != NULL) {
			ValaCCodeFragment* _tmp57_;
			_tmp57_ = NULL;
			self->module_init_fragment = (_tmp57_ = vala_ccode_fragment_new (), (self->module_init_fragment == NULL) ? NULL : (self->module_init_fragment = (vala_ccode_node_unref (self->module_init_fragment), NULL)), _tmp57_);
			{
				GeeList* _tmp58_;
				GeeIterator* _tmp59_;
				GeeIterator* _parameter_it;
				_tmp58_ = NULL;
				_tmp59_ = NULL;
				_parameter_it = (_tmp59_ = gee_iterable_iterator ((GeeIterable*) (_tmp58_ = vala_method_get_parameters (vala_code_context_get_module_init_method (context)))), (_tmp58_ == NULL) ? NULL : (_tmp58_ = (gee_collection_object_unref (_tmp58_), NULL)), _tmp59_);
				while (gee_iterator_next (_parameter_it)) {
					ValaFormalParameter* parameter;
					parameter = (ValaFormalParameter*) gee_iterator_get (_parameter_it);
					if (vala_data_type_get_data_type (vala_formal_parameter_get_parameter_type (parameter)) == self->type_module_type) {
						char* _tmp61_;
						const char* _tmp60_;
						self->in_plugin = TRUE;
						_tmp61_ = NULL;
						_tmp60_ = NULL;
						self->module_init_param_name = (_tmp61_ = (_tmp60_ = vala_symbol_get_name ((ValaSymbol*) parameter), (_tmp60_ == NULL) ? NULL : g_strdup (_tmp60_)), self->module_init_param_name = (g_free (self->module_init_param_name), NULL), _tmp61_);
						(parameter == NULL) ? NULL : (parameter = (vala_code_node_unref (parameter), NULL));
						break;
					}
					(parameter == NULL) ? NULL : (parameter = (vala_code_node_unref (parameter), NULL));
				}
				(_parameter_it == NULL) ? NULL : (_parameter_it = (gee_collection_object_unref (_parameter_it), NULL));
			}
		}
		dbus_ns = vala_scope_lookup (vala_symbol_get_scope (self->root_symbol), "DBus");
		if (dbus_ns != NULL) {
			ValaTypeSymbol* _tmp62_;
			_tmp62_ = NULL;
			self->dbus_object_type = (_tmp62_ = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (dbus_ns), "Object")), (self->dbus_object_type == NULL) ? NULL : (self->dbus_object_type = (vala_code_node_unref (self->dbus_object_type), NULL)), _tmp62_);
		}
		(glib_ns == NULL) ? NULL : (glib_ns = (vala_code_node_unref (glib_ns), NULL));
		(dbus_ns == NULL) ? NULL : (dbus_ns = (vala_code_node_unref (dbus_ns), NULL));
	}
	_tmp63_ = NULL;
	self->header_declarations = (_tmp63_ = vala_ccode_declaration_space_new (), (self->header_declarations == NULL) ? NULL : (self->header_declarations = (vala_ccode_declaration_space_unref (self->header_declarations), NULL)), _tmp63_);
	_tmp64_ = NULL;
	self->internal_header_declarations = (_tmp64_ = vala_ccode_declaration_space_new (), (self->internal_header_declarations == NULL) ? NULL : (self->internal_header_declarations = (vala_ccode_declaration_space_unref (self->internal_header_declarations), NULL)), _tmp64_);
	/* we're only interested in non-pkg source files */
	source_files = vala_code_context_get_source_files (context);
	{
		GeeIterator* _file_it;
		_file_it = gee_iterable_iterator ((GeeIterable*) source_files);
		while (gee_iterator_next (_file_it)) {
			ValaSourceFile* file;
			file = (ValaSourceFile*) gee_iterator_get (_file_it);
			if (!vala_source_file_get_external_package (file)) {
				vala_source_file_accept (file, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
			}
			(file == NULL) ? NULL : (file = (vala_source_file_unref (file), NULL));
		}
		(_file_it == NULL) ? NULL : (_file_it = (gee_collection_object_unref (_file_it), NULL));
	}
	/* generate C header file for public API*/
	if (vala_code_context_get_header_filename (context) != NULL) {
		ValaCCodeWriter* writer;
		char* _tmp66_;
		ValaCCodeOnceSection* _tmp67_;
		ValaCCodeOnceSection* once;
		ValaCCodeNewline* _tmp68_;
		ValaCCodeNewline* _tmp69_;
		ValaCCodeNewline* _tmp72_;
		ValaCCodeNewline* _tmp73_;
		ValaCCodeNewline* _tmp74_;
		ValaCCodeNewline* _tmp75_;
		ValaCCodeNewline* _tmp76_;
		ValaCCodeNewline* _tmp79_;
		writer = vala_ccode_writer_new (vala_code_context_get_header_filename (context));
		if (!vala_ccode_writer_open (writer)) {
			char* _tmp65_;
			_tmp65_ = NULL;
			vala_report_error (NULL, _tmp65_ = g_strdup_printf ("unable to open `%s' for writing", vala_ccode_writer_get_filename (writer)));
			_tmp65_ = (g_free (_tmp65_), NULL);
			(writer == NULL) ? NULL : (writer = (vala_ccode_writer_unref (writer), NULL));
			(unichar_struct == NULL) ? NULL : (unichar_struct = (vala_code_node_unref (unichar_struct), NULL));
			(source_files == NULL) ? NULL : (source_files = (gee_collection_object_unref (source_files), NULL));
			return;
		}
		vala_ccode_writer_write_newline (writer);
		_tmp66_ = NULL;
		_tmp67_ = NULL;
		once = (_tmp67_ = vala_ccode_once_section_new (_tmp66_ = vala_ccode_base_module_get_define_for_filename (vala_ccode_writer_get_filename (writer))), _tmp66_ = (g_free (_tmp66_), NULL), _tmp67_);
		_tmp68_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp68_ = vala_ccode_newline_new ()));
		(_tmp68_ == NULL) ? NULL : (_tmp68_ = (vala_ccode_node_unref (_tmp68_), NULL));
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) self->header_declarations->include_directives);
		_tmp69_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp69_ = vala_ccode_newline_new ()));
		(_tmp69_ == NULL) ? NULL : (_tmp69_ = (vala_ccode_node_unref (_tmp69_), NULL));
		if (vala_code_context_get_profile (context) == VALA_PROFILE_GOBJECT) {
			ValaCCodeIdentifier* _tmp70_;
			ValaCCodeNewline* _tmp71_;
			_tmp70_ = NULL;
			vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp70_ = vala_ccode_identifier_new ("G_BEGIN_DECLS")));
			(_tmp70_ == NULL) ? NULL : (_tmp70_ = (vala_ccode_node_unref (_tmp70_), NULL));
			_tmp71_ = NULL;
			vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp71_ = vala_ccode_newline_new ()));
			(_tmp71_ == NULL) ? NULL : (_tmp71_ = (vala_ccode_node_unref (_tmp71_), NULL));
		}
		_tmp72_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp72_ = vala_ccode_newline_new ()));
		(_tmp72_ == NULL) ? NULL : (_tmp72_ = (vala_ccode_node_unref (_tmp72_), NULL));
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) self->header_declarations->type_declaration);
		_tmp73_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp73_ = vala_ccode_newline_new ()));
		(_tmp73_ == NULL) ? NULL : (_tmp73_ = (vala_ccode_node_unref (_tmp73_), NULL));
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) self->header_declarations->type_definition);
		_tmp74_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp74_ = vala_ccode_newline_new ()));
		(_tmp74_ == NULL) ? NULL : (_tmp74_ = (vala_ccode_node_unref (_tmp74_), NULL));
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) self->header_declarations->type_member_declaration);
		_tmp75_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp75_ = vala_ccode_newline_new ()));
		(_tmp75_ == NULL) ? NULL : (_tmp75_ = (vala_ccode_node_unref (_tmp75_), NULL));
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) self->header_declarations->constant_declaration);
		_tmp76_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp76_ = vala_ccode_newline_new ()));
		(_tmp76_ == NULL) ? NULL : (_tmp76_ = (vala_ccode_node_unref (_tmp76_), NULL));
		if (vala_code_context_get_profile (context) == VALA_PROFILE_GOBJECT) {
			ValaCCodeIdentifier* _tmp77_;
			ValaCCodeNewline* _tmp78_;
			_tmp77_ = NULL;
			vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp77_ = vala_ccode_identifier_new ("G_END_DECLS")));
			(_tmp77_ == NULL) ? NULL : (_tmp77_ = (vala_ccode_node_unref (_tmp77_), NULL));
			_tmp78_ = NULL;
			vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp78_ = vala_ccode_newline_new ()));
			(_tmp78_ == NULL) ? NULL : (_tmp78_ = (vala_ccode_node_unref (_tmp78_), NULL));
		}
		_tmp79_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp79_ = vala_ccode_newline_new ()));
		(_tmp79_ == NULL) ? NULL : (_tmp79_ = (vala_ccode_node_unref (_tmp79_), NULL));
		vala_ccode_node_write ((ValaCCodeNode*) once, writer);
		vala_ccode_writer_close (writer);
		(writer == NULL) ? NULL : (writer = (vala_ccode_writer_unref (writer), NULL));
		(once == NULL) ? NULL : (once = (vala_ccode_node_unref (once), NULL));
	}
	/* generate C header file for internal API*/
	if (vala_code_context_get_internal_header_filename (context) != NULL) {
		ValaCCodeWriter* writer;
		char* _tmp81_;
		ValaCCodeOnceSection* _tmp82_;
		ValaCCodeOnceSection* once;
		ValaCCodeNewline* _tmp83_;
		ValaCCodeNewline* _tmp84_;
		ValaCCodeNewline* _tmp87_;
		ValaCCodeNewline* _tmp88_;
		ValaCCodeNewline* _tmp89_;
		ValaCCodeNewline* _tmp90_;
		ValaCCodeNewline* _tmp91_;
		ValaCCodeNewline* _tmp94_;
		writer = vala_ccode_writer_new (vala_code_context_get_internal_header_filename (context));
		if (!vala_ccode_writer_open (writer)) {
			char* _tmp80_;
			_tmp80_ = NULL;
			vala_report_error (NULL, _tmp80_ = g_strdup_printf ("unable to open `%s' for writing", vala_ccode_writer_get_filename (writer)));
			_tmp80_ = (g_free (_tmp80_), NULL);
			(writer == NULL) ? NULL : (writer = (vala_ccode_writer_unref (writer), NULL));
			(unichar_struct == NULL) ? NULL : (unichar_struct = (vala_code_node_unref (unichar_struct), NULL));
			(source_files == NULL) ? NULL : (source_files = (gee_collection_object_unref (source_files), NULL));
			return;
		}
		vala_ccode_writer_write_newline (writer);
		_tmp81_ = NULL;
		_tmp82_ = NULL;
		once = (_tmp82_ = vala_ccode_once_section_new (_tmp81_ = vala_ccode_base_module_get_define_for_filename (vala_ccode_writer_get_filename (writer))), _tmp81_ = (g_free (_tmp81_), NULL), _tmp82_);
		_tmp83_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp83_ = vala_ccode_newline_new ()));
		(_tmp83_ == NULL) ? NULL : (_tmp83_ = (vala_ccode_node_unref (_tmp83_), NULL));
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) self->internal_header_declarations->include_directives);
		_tmp84_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp84_ = vala_ccode_newline_new ()));
		(_tmp84_ == NULL) ? NULL : (_tmp84_ = (vala_ccode_node_unref (_tmp84_), NULL));
		if (vala_code_context_get_profile (context) == VALA_PROFILE_GOBJECT) {
			ValaCCodeIdentifier* _tmp85_;
			ValaCCodeNewline* _tmp86_;
			_tmp85_ = NULL;
			vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp85_ = vala_ccode_identifier_new ("G_BEGIN_DECLS")));
			(_tmp85_ == NULL) ? NULL : (_tmp85_ = (vala_ccode_node_unref (_tmp85_), NULL));
			_tmp86_ = NULL;
			vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp86_ = vala_ccode_newline_new ()));
			(_tmp86_ == NULL) ? NULL : (_tmp86_ = (vala_ccode_node_unref (_tmp86_), NULL));
		}
		_tmp87_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp87_ = vala_ccode_newline_new ()));
		(_tmp87_ == NULL) ? NULL : (_tmp87_ = (vala_ccode_node_unref (_tmp87_), NULL));
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) self->internal_header_declarations->type_declaration);
		_tmp88_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp88_ = vala_ccode_newline_new ()));
		(_tmp88_ == NULL) ? NULL : (_tmp88_ = (vala_ccode_node_unref (_tmp88_), NULL));
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) self->internal_header_declarations->type_definition);
		_tmp89_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp89_ = vala_ccode_newline_new ()));
		(_tmp89_ == NULL) ? NULL : (_tmp89_ = (vala_ccode_node_unref (_tmp89_), NULL));
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) self->internal_header_declarations->type_member_declaration);
		_tmp90_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp90_ = vala_ccode_newline_new ()));
		(_tmp90_ == NULL) ? NULL : (_tmp90_ = (vala_ccode_node_unref (_tmp90_), NULL));
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) self->internal_header_declarations->constant_declaration);
		_tmp91_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp91_ = vala_ccode_newline_new ()));
		(_tmp91_ == NULL) ? NULL : (_tmp91_ = (vala_ccode_node_unref (_tmp91_), NULL));
		if (vala_code_context_get_profile (context) == VALA_PROFILE_GOBJECT) {
			ValaCCodeIdentifier* _tmp92_;
			ValaCCodeNewline* _tmp93_;
			_tmp92_ = NULL;
			vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp92_ = vala_ccode_identifier_new ("G_END_DECLS")));
			(_tmp92_ == NULL) ? NULL : (_tmp92_ = (vala_ccode_node_unref (_tmp92_), NULL));
			_tmp93_ = NULL;
			vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp93_ = vala_ccode_newline_new ()));
			(_tmp93_ == NULL) ? NULL : (_tmp93_ = (vala_ccode_node_unref (_tmp93_), NULL));
		}
		_tmp94_ = NULL;
		vala_ccode_fragment_append ((ValaCCodeFragment*) once, (ValaCCodeNode*) (_tmp94_ = vala_ccode_newline_new ()));
		(_tmp94_ == NULL) ? NULL : (_tmp94_ = (vala_ccode_node_unref (_tmp94_), NULL));
		vala_ccode_node_write ((ValaCCodeNode*) once, writer);
		vala_ccode_writer_close (writer);
		(writer == NULL) ? NULL : (writer = (vala_ccode_writer_unref (writer), NULL));
		(once == NULL) ? NULL : (once = (vala_ccode_node_unref (once), NULL));
	}
	(unichar_struct == NULL) ? NULL : (unichar_struct = (vala_code_node_unref (unichar_struct), NULL));
	(source_files == NULL) ? NULL : (source_files = (gee_collection_object_unref (source_files), NULL));
}


static ValaCCodeIdentifier* vala_ccode_base_module_real_get_value_setter_function (ValaCCodeModule* base, ValaDataType* type_reference) {
	ValaCCodeBaseModule * self;
	self = (ValaCCodeBaseModule*) base;
	g_return_val_if_fail (type_reference != NULL, NULL);
	if (vala_data_type_get_data_type (type_reference) != NULL) {
		char* _tmp0_;
		ValaCCodeIdentifier* _tmp1_;
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		return (_tmp1_ = vala_ccode_identifier_new (_tmp0_ = vala_typesymbol_get_set_value_function (vala_data_type_get_data_type (type_reference))), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	} else {
		return vala_ccode_identifier_new ("g_value_set_pointer");
	}
}


static void vala_ccode_base_module_real_append_vala_array_free (ValaCCodeBaseModule* self) {
	g_return_if_fail (self != NULL);
}


void vala_ccode_base_module_append_vala_array_free (ValaCCodeBaseModule* self) {
	VALA_CCODE_BASE_MODULE_GET_CLASS (self)->append_vala_array_free (self);
}


static void vala_ccode_base_module_real_append_vala_array_move (ValaCCodeBaseModule* self) {
	g_return_if_fail (self != NULL);
}


void vala_ccode_base_module_append_vala_array_move (ValaCCodeBaseModule* self) {
	VALA_CCODE_BASE_MODULE_GET_CLASS (self)->append_vala_array_move (self);
}


static void vala_ccode_base_module_real_append_vala_array_length (ValaCCodeBaseModule* self) {
	g_return_if_fail (self != NULL);
}


void vala_ccode_base_module_append_vala_array_length (ValaCCodeBaseModule* self) {
	VALA_CCODE_BASE_MODULE_GET_CLASS (self)->append_vala_array_length (self);
}


static void vala_ccode_base_module_append_vala_strcmp0 (ValaCCodeBaseModule* self) {
	ValaCCodeFunction* fun;
	ValaCCodeFormalParameter* _tmp0_;
	ValaCCodeFormalParameter* _tmp1_;
	ValaCCodeFunction* _tmp2_;
	ValaCCodeIdentifier* _tmp4_;
	ValaCCodeIdentifier* _tmp3_;
	ValaCCodeBinaryExpression* _tmp5_;
	ValaCCodeBinaryExpression* cineq;
	ValaCCodeBlock* _tmp6_;
	ValaCCodeBlock* cblock;
	ValaCCodeBinaryExpression* _tmp9_;
	ValaCCodeConstant* _tmp8_;
	ValaCCodeIdentifier* _tmp7_;
	ValaCCodeIfStatement* _tmp10_;
	ValaCCodeIfStatement* cif;
	ValaCCodeReturnStatement* _tmp12_;
	ValaCCodeUnaryExpression* _tmp11_;
	ValaCCodeBlock* _tmp13_;
	ValaCCodeIfStatement* _tmp17_;
	ValaCCodeBinaryExpression* _tmp16_;
	ValaCCodeConstant* _tmp15_;
	ValaCCodeIdentifier* _tmp14_;
	ValaCCodeReturnStatement* _tmp18_;
	ValaCCodeIdentifier* _tmp19_;
	ValaCCodeFunctionCall* _tmp20_;
	ValaCCodeFunctionCall* ccall;
	ValaCCodeIdentifier* _tmp21_;
	ValaCCodeIdentifier* _tmp22_;
	ValaCCodeReturnStatement* _tmp23_;
	g_return_if_fail (self != NULL);
	fun = vala_ccode_function_new ("_vala_strcmp0", "int");
	vala_ccode_function_set_modifiers (fun, VALA_CCODE_MODIFIERS_STATIC);
	_tmp0_ = NULL;
	vala_ccode_function_add_parameter (fun, _tmp0_ = vala_ccode_formal_parameter_new ("str1", "const char *"));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
	_tmp1_ = NULL;
	vala_ccode_function_add_parameter (fun, _tmp1_ = vala_ccode_formal_parameter_new ("str2", "const char *"));
	(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_ccode_node_unref (_tmp1_), NULL));
	_tmp2_ = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (self->source_declarations, (ValaCCodeNode*) (_tmp2_ = vala_ccode_function_copy (fun)));
	(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_ccode_node_unref (_tmp2_), NULL));
	/* (str1 != str2)*/
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	_tmp5_ = NULL;
	cineq = (_tmp5_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_INEQUALITY, (ValaCCodeExpression*) (_tmp3_ = vala_ccode_identifier_new ("str1")), (ValaCCodeExpression*) (_tmp4_ = vala_ccode_identifier_new ("str2"))), (_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL)), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL)), _tmp5_);
	_tmp6_ = NULL;
	vala_ccode_function_set_block (fun, _tmp6_ = vala_ccode_block_new ());
	(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
	cblock = vala_ccode_block_new ();
	/* if (str1 == NULL)*/
	_tmp9_ = NULL;
	_tmp8_ = NULL;
	_tmp7_ = NULL;
	_tmp10_ = NULL;
	cif = (_tmp10_ = vala_ccode_if_statement_new ((ValaCCodeExpression*) (_tmp9_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, (ValaCCodeExpression*) (_tmp7_ = vala_ccode_identifier_new ("str1")), (ValaCCodeExpression*) (_tmp8_ = vala_ccode_constant_new ("NULL")))), (ValaCCodeStatement*) cblock, NULL), (_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL)), (_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL)), (_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL)), _tmp10_);
	/* return -(str1 != str2);*/
	_tmp12_ = NULL;
	_tmp11_ = NULL;
	vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp12_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp11_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_MINUS, (ValaCCodeExpression*) cineq)))));
	(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
	(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
	vala_ccode_block_add_statement (vala_ccode_function_get_block (fun), (ValaCCodeNode*) cif);
	_tmp13_ = NULL;
	cblock = (_tmp13_ = vala_ccode_block_new (), (cblock == NULL) ? NULL : (cblock = (vala_ccode_node_unref (cblock), NULL)), _tmp13_);
	/* if (str2 == NULL)*/
	_tmp17_ = NULL;
	_tmp16_ = NULL;
	_tmp15_ = NULL;
	_tmp14_ = NULL;
	cif = (_tmp17_ = vala_ccode_if_statement_new ((ValaCCodeExpression*) (_tmp16_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, (ValaCCodeExpression*) (_tmp14_ = vala_ccode_identifier_new ("str2")), (ValaCCodeExpression*) (_tmp15_ = vala_ccode_constant_new ("NULL")))), (ValaCCodeStatement*) cblock, NULL), (cif == NULL) ? NULL : (cif = (vala_ccode_node_unref (cif), NULL)), _tmp17_);
	(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL));
	(_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL));
	(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
	/* return (str1 != str2);*/
	_tmp18_ = NULL;
	vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp18_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) cineq)));
	(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
	vala_ccode_block_add_statement (vala_ccode_function_get_block (fun), (ValaCCodeNode*) cif);
	/* strcmp (str1, str2)*/
	_tmp19_ = NULL;
	_tmp20_ = NULL;
	ccall = (_tmp20_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp19_ = vala_ccode_identifier_new ("strcmp"))), (_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL)), _tmp20_);
	_tmp21_ = NULL;
	vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp21_ = vala_ccode_identifier_new ("str1")));
	(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL));
	_tmp22_ = NULL;
	vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp22_ = vala_ccode_identifier_new ("str2")));
	(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL));
	/* return strcmp (str1, str2);*/
	_tmp23_ = NULL;
	vala_ccode_block_add_statement (vala_ccode_function_get_block (fun), (ValaCCodeNode*) (_tmp23_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) ccall)));
	(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_ccode_node_unref (_tmp23_), NULL));
	vala_ccode_fragment_append (self->source_type_member_definition, (ValaCCodeNode*) fun);
	(fun == NULL) ? NULL : (fun = (vala_ccode_node_unref (fun), NULL));
	(cineq == NULL) ? NULL : (cineq = (vala_ccode_node_unref (cineq), NULL));
	(cblock == NULL) ? NULL : (cblock = (vala_ccode_node_unref (cblock), NULL));
	(cif == NULL) ? NULL : (cif = (vala_ccode_node_unref (cif), NULL));
	(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
}


static void vala_ccode_base_module_real_visit_source_file (ValaCCodeModule* base, ValaSourceFile* source_file) {
	ValaCCodeBaseModule * self;
	ValaCCodeDeclarationSpace* _tmp0_;
	ValaCCodeFragment* _tmp1_;
	ValaCCodeFragment* _tmp2_;
	ValaCCodeFragment* _tmp3_;
	GeeSet* _tmp4_;
	GeeSet* _tmp5_;
	GeeSet* _tmp6_;
	gboolean _tmp7_;
	ValaCCodeComment* comment;
	char* _tmp62_;
	ValaCCodeWriter* _tmp63_;
	ValaCCodeWriter* writer;
	ValaCCodeDeclarationSpace* _tmp65_;
	ValaCCodeFragment* _tmp66_;
	ValaCCodeFragment* _tmp67_;
	ValaCCodeFragment* _tmp68_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (source_file != NULL);
	_tmp0_ = NULL;
	self->source_declarations = (_tmp0_ = vala_ccode_declaration_space_new (), (self->source_declarations == NULL) ? NULL : (self->source_declarations = (vala_ccode_declaration_space_unref (self->source_declarations), NULL)), _tmp0_);
	_tmp1_ = NULL;
	self->source_type_member_definition = (_tmp1_ = vala_ccode_fragment_new (), (self->source_type_member_definition == NULL) ? NULL : (self->source_type_member_definition = (vala_ccode_node_unref (self->source_type_member_definition), NULL)), _tmp1_);
	_tmp2_ = NULL;
	self->source_signal_marshaller_definition = (_tmp2_ = vala_ccode_fragment_new (), (self->source_signal_marshaller_definition == NULL) ? NULL : (self->source_signal_marshaller_definition = (vala_ccode_node_unref (self->source_signal_marshaller_definition), NULL)), _tmp2_);
	_tmp3_ = NULL;
	self->source_signal_marshaller_declaration = (_tmp3_ = vala_ccode_fragment_new (), (self->source_signal_marshaller_declaration == NULL) ? NULL : (self->source_signal_marshaller_declaration = (vala_ccode_node_unref (self->source_signal_marshaller_declaration), NULL)), _tmp3_);
	_tmp4_ = NULL;
	self->user_marshal_set = (_tmp4_ = (GeeSet*) gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal), (self->user_marshal_set == NULL) ? NULL : (self->user_marshal_set = (gee_collection_object_unref (self->user_marshal_set), NULL)), _tmp4_);
	self->next_temp_var_id = 0;
	gee_map_clear (self->variable_name_map);
	self->gvaluecollector_h_needed = FALSE;
	self->gio_h_needed = FALSE;
	self->dbus_glib_h_needed = FALSE;
	self->dbus_glib_h_needed_in_header = FALSE;
	self->requires_array_free = FALSE;
	self->requires_array_move = FALSE;
	self->requires_array_length = FALSE;
	self->requires_strcmp0 = FALSE;
	_tmp5_ = NULL;
	self->wrappers = (_tmp5_ = (GeeSet*) gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal), (self->wrappers == NULL) ? NULL : (self->wrappers = (gee_collection_object_unref (self->wrappers), NULL)), _tmp5_);
	_tmp6_ = NULL;
	self->priv->generated_external_symbols = (_tmp6_ = (GeeSet*) gee_hash_set_new (VALA_TYPE_SYMBOL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_hash, g_direct_equal), (self->priv->generated_external_symbols == NULL) ? NULL : (self->priv->generated_external_symbols = (gee_collection_object_unref (self->priv->generated_external_symbols), NULL)), _tmp6_);
	if (vala_code_context_get_profile (self->priv->_context) == VALA_PROFILE_GOBJECT) {
		vala_ccode_declaration_space_add_include (self->header_declarations, "glib.h", FALSE);
		vala_ccode_declaration_space_add_include (self->internal_header_declarations, "glib.h", FALSE);
		vala_ccode_declaration_space_add_include (self->source_declarations, "glib.h", FALSE);
		vala_ccode_declaration_space_add_include (self->source_declarations, "glib-object.h", FALSE);
	}
	vala_source_file_accept_children (source_file, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->_context)) > 0) {
		return;
	}
	if (self->requires_array_free) {
		vala_ccode_base_module_append_vala_array_free (self);
	}
	if (self->requires_array_move) {
		vala_ccode_base_module_append_vala_array_move (self);
	}
	if (self->requires_array_length) {
		vala_ccode_base_module_append_vala_array_length (self);
	}
	if (self->requires_strcmp0) {
		vala_ccode_base_module_append_vala_strcmp0 (self);
	}
	if (self->gvaluecollector_h_needed) {
		vala_ccode_declaration_space_add_include (self->source_declarations, "gobject/gvaluecollector.h", FALSE);
	}
	if (self->dbus_glib_h_needed) {
		vala_ccode_declaration_space_add_include (self->source_declarations, "dbus/dbus.h", FALSE);
		vala_ccode_declaration_space_add_include (self->source_declarations, "dbus/dbus-glib.h", FALSE);
		vala_ccode_declaration_space_add_include (self->source_declarations, "dbus/dbus-glib-lowlevel.h", FALSE);
	}
	_tmp7_ = FALSE;
	if (self->dbus_glib_h_needed_in_header) {
		_tmp7_ = TRUE;
	} else {
		_tmp7_ = self->dbus_glib_h_needed;
	}
	if (_tmp7_) {
		ValaCCodeStruct* dbusvtable;
		ValaCCodeTypeDefinition* _tmp9_;
		ValaCCodeVariableDeclarator* _tmp8_;
		ValaCCodeFunction* cfunc;
		ValaCCodeFormalParameter* _tmp10_;
		ValaCCodeFormalParameter* _tmp11_;
		ValaCCodeFormalParameter* _tmp12_;
		ValaCCodeFunction* _tmp13_;
		ValaCCodeBlock* block;
		ValaCCodeDeclaration* _cdecl_;
		ValaCCodeVariableDeclarator* _tmp14_;
		ValaCCodeIdentifier* _tmp15_;
		ValaCCodeFunctionCall* _tmp16_;
		ValaCCodeFunctionCall* quark;
		ValaCCodeConstant* _tmp17_;
		ValaCCodeIdentifier* _tmp18_;
		ValaCCodeFunctionCall* _tmp19_;
		ValaCCodeFunctionCall* get_qdata;
		ValaCCodeIdentifier* _tmp20_;
		ValaCCodeExpressionStatement* _tmp23_;
		ValaCCodeAssignment* _tmp22_;
		ValaCCodeIdentifier* _tmp21_;
		ValaCCodeMemberAccess* _tmp25_;
		ValaCCodeIdentifier* _tmp24_;
		ValaCCodeFunctionCall* _tmp26_;
		ValaCCodeFunctionCall* cregister;
		ValaCCodeIdentifier* _tmp27_;
		ValaCCodeIdentifier* _tmp28_;
		ValaCCodeIdentifier* _tmp29_;
		ValaCCodeBlock* ifblock;
		ValaCCodeExpressionStatement* _tmp30_;
		ValaCCodeBlock* elseblock;
		ValaCCodeIdentifier* _tmp31_;
		ValaCCodeFunctionCall* _tmp32_;
		ValaCCodeFunctionCall* warn;
		ValaCCodeConstant* _tmp33_;
		ValaCCodeExpressionStatement* _tmp34_;
		ValaCCodeIfStatement* _tmp36_;
		ValaCCodeIdentifier* _tmp35_;
		ValaCCodeFunction* _tmp37_;
		ValaCCodeFormalParameter* _tmp38_;
		ValaCCodeFormalParameter* _tmp39_;
		ValaCCodeFunction* _tmp40_;
		ValaCCodeBlock* _tmp41_;
		ValaCCodeDeclaration* _tmp42_;
		ValaCCodeVariableDeclarator* _tmp43_;
		ValaCCodeIdentifier* _tmp44_;
		ValaCCodeFunctionCall* _tmp45_;
		ValaCCodeFunctionCall* path;
		ValaCCodeCastExpression* _tmp47_;
		ValaCCodeIdentifier* _tmp46_;
		ValaCCodeConstant* _tmp48_;
		ValaCCodeExpressionStatement* _tmp51_;
		ValaCCodeAssignment* _tmp50_;
		ValaCCodeIdentifier* _tmp49_;
		ValaCCodeIdentifier* _tmp52_;
		ValaCCodeFunctionCall* _tmp53_;
		ValaCCodeFunctionCall* unregister_call;
		ValaCCodeIdentifier* _tmp54_;
		ValaCCodeIdentifier* _tmp55_;
		ValaCCodeExpressionStatement* _tmp56_;
		ValaCCodeIdentifier* _tmp57_;
		ValaCCodeFunctionCall* _tmp58_;
		ValaCCodeFunctionCall* path_free;
		ValaCCodeIdentifier* _tmp59_;
		ValaCCodeExpressionStatement* _tmp60_;
		dbusvtable = vala_ccode_struct_new ("_DBusObjectVTable");
		vala_ccode_struct_add_field (dbusvtable, "void", "(*register_object) (DBusConnection*, const char*, void*)");
		vala_ccode_declaration_space_add_type_definition (self->source_declarations, (ValaCCodeNode*) dbusvtable);
		_tmp9_ = NULL;
		_tmp8_ = NULL;
		vala_ccode_declaration_space_add_type_declaration (self->source_declarations, (ValaCCodeNode*) (_tmp9_ = vala_ccode_type_definition_new ("struct _DBusObjectVTable", (ValaCCodeDeclarator*) (_tmp8_ = vala_ccode_variable_declarator_new ("_DBusObjectVTable", NULL, NULL)))));
		(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
		(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL));
		cfunc = vala_ccode_function_new ("_vala_dbus_register_object", "void");
		_tmp10_ = NULL;
		vala_ccode_function_add_parameter (cfunc, _tmp10_ = vala_ccode_formal_parameter_new ("connection", "DBusConnection*"));
		(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
		_tmp11_ = NULL;
		vala_ccode_function_add_parameter (cfunc, _tmp11_ = vala_ccode_formal_parameter_new ("path", "const char*"));
		(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
		_tmp12_ = NULL;
		vala_ccode_function_add_parameter (cfunc, _tmp12_ = vala_ccode_formal_parameter_new ("object", "void*"));
		(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
		vala_ccode_function_set_modifiers (cfunc, vala_ccode_function_get_modifiers (cfunc) | VALA_CCODE_MODIFIERS_STATIC);
		_tmp13_ = NULL;
		vala_ccode_declaration_space_add_type_member_declaration (self->source_declarations, (ValaCCodeNode*) (_tmp13_ = vala_ccode_function_copy (cfunc)));
		(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL));
		block = vala_ccode_block_new ();
		vala_ccode_function_set_block (cfunc, block);
		_cdecl_ = vala_ccode_declaration_new ("const _DBusObjectVTable *");
		_tmp14_ = NULL;
		vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp14_ = vala_ccode_variable_declarator_new ("vtable", NULL, NULL)));
		(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
		_tmp15_ = NULL;
		_tmp16_ = NULL;
		quark = (_tmp16_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp15_ = vala_ccode_identifier_new ("g_quark_from_static_string"))), (_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL)), _tmp16_);
		_tmp17_ = NULL;
		vala_ccode_function_call_add_argument (quark, (ValaCCodeExpression*) (_tmp17_ = vala_ccode_constant_new ("\"DBusObjectVTable\"")));
		(_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_ccode_node_unref (_tmp17_), NULL));
		_tmp18_ = NULL;
		_tmp19_ = NULL;
		get_qdata = (_tmp19_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp18_ = vala_ccode_identifier_new ("g_type_get_qdata"))), (_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL)), _tmp19_);
		_tmp20_ = NULL;
		vala_ccode_function_call_add_argument (get_qdata, (ValaCCodeExpression*) (_tmp20_ = vala_ccode_identifier_new ("G_TYPE_FROM_INSTANCE (object)")));
		(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL));
		vala_ccode_function_call_add_argument (get_qdata, (ValaCCodeExpression*) quark);
		_tmp23_ = NULL;
		_tmp22_ = NULL;
		_tmp21_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp23_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp22_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp21_ = vala_ccode_identifier_new ("vtable")), (ValaCCodeExpression*) get_qdata, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
		(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_ccode_node_unref (_tmp23_), NULL));
		(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL));
		(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL));
		_tmp25_ = NULL;
		_tmp24_ = NULL;
		_tmp26_ = NULL;
		cregister = (_tmp26_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp25_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp24_ = vala_ccode_identifier_new ("vtable")), "register_object"))), (_tmp25_ == NULL) ? NULL : (_tmp25_ = (vala_ccode_node_unref (_tmp25_), NULL)), (_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_ccode_node_unref (_tmp24_), NULL)), _tmp26_);
		_tmp27_ = NULL;
		vala_ccode_function_call_add_argument (cregister, (ValaCCodeExpression*) (_tmp27_ = vala_ccode_identifier_new ("connection")));
		(_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL));
		_tmp28_ = NULL;
		vala_ccode_function_call_add_argument (cregister, (ValaCCodeExpression*) (_tmp28_ = vala_ccode_identifier_new ("path")));
		(_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_ccode_node_unref (_tmp28_), NULL));
		_tmp29_ = NULL;
		vala_ccode_function_call_add_argument (cregister, (ValaCCodeExpression*) (_tmp29_ = vala_ccode_identifier_new ("object")));
		(_tmp29_ == NULL) ? NULL : (_tmp29_ = (vala_ccode_node_unref (_tmp29_), NULL));
		ifblock = vala_ccode_block_new ();
		_tmp30_ = NULL;
		vala_ccode_block_add_statement (ifblock, (ValaCCodeNode*) (_tmp30_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) cregister)));
		(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL));
		elseblock = vala_ccode_block_new ();
		_tmp31_ = NULL;
		_tmp32_ = NULL;
		warn = (_tmp32_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp31_ = vala_ccode_identifier_new ("g_warning"))), (_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_ccode_node_unref (_tmp31_), NULL)), _tmp32_);
		_tmp33_ = NULL;
		vala_ccode_function_call_add_argument (warn, (ValaCCodeExpression*) (_tmp33_ = vala_ccode_constant_new ("\"Object does not implement any D-Bus interface\"")));
		(_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_ccode_node_unref (_tmp33_), NULL));
		_tmp34_ = NULL;
		vala_ccode_block_add_statement (elseblock, (ValaCCodeNode*) (_tmp34_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) warn)));
		(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_ccode_node_unref (_tmp34_), NULL));
		_tmp36_ = NULL;
		_tmp35_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp36_ = vala_ccode_if_statement_new ((ValaCCodeExpression*) (_tmp35_ = vala_ccode_identifier_new ("vtable")), (ValaCCodeStatement*) ifblock, (ValaCCodeStatement*) elseblock)));
		(_tmp36_ == NULL) ? NULL : (_tmp36_ = (vala_ccode_node_unref (_tmp36_), NULL));
		(_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_ccode_node_unref (_tmp35_), NULL));
		vala_ccode_fragment_append (self->source_type_member_definition, (ValaCCodeNode*) cfunc);
		/* unregister function*/
		_tmp37_ = NULL;
		cfunc = (_tmp37_ = vala_ccode_function_new ("_vala_dbus_unregister_object", "void"), (cfunc == NULL) ? NULL : (cfunc = (vala_ccode_node_unref (cfunc), NULL)), _tmp37_);
		_tmp38_ = NULL;
		vala_ccode_function_add_parameter (cfunc, _tmp38_ = vala_ccode_formal_parameter_new ("connection", "gpointer"));
		(_tmp38_ == NULL) ? NULL : (_tmp38_ = (vala_ccode_node_unref (_tmp38_), NULL));
		_tmp39_ = NULL;
		vala_ccode_function_add_parameter (cfunc, _tmp39_ = vala_ccode_formal_parameter_new ("object", "GObject*"));
		(_tmp39_ == NULL) ? NULL : (_tmp39_ = (vala_ccode_node_unref (_tmp39_), NULL));
		vala_ccode_function_set_modifiers (cfunc, vala_ccode_function_get_modifiers (cfunc) | VALA_CCODE_MODIFIERS_STATIC);
		_tmp40_ = NULL;
		vala_ccode_declaration_space_add_type_member_declaration (self->source_declarations, (ValaCCodeNode*) (_tmp40_ = vala_ccode_function_copy (cfunc)));
		(_tmp40_ == NULL) ? NULL : (_tmp40_ = (vala_ccode_node_unref (_tmp40_), NULL));
		_tmp41_ = NULL;
		block = (_tmp41_ = vala_ccode_block_new (), (block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL)), _tmp41_);
		vala_ccode_function_set_block (cfunc, block);
		_tmp42_ = NULL;
		_cdecl_ = (_tmp42_ = vala_ccode_declaration_new ("char*"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp42_);
		_tmp43_ = NULL;
		vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp43_ = vala_ccode_variable_declarator_new ("path", NULL, NULL)));
		(_tmp43_ == NULL) ? NULL : (_tmp43_ = (vala_ccode_node_unref (_tmp43_), NULL));
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
		_tmp44_ = NULL;
		_tmp45_ = NULL;
		path = (_tmp45_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp44_ = vala_ccode_identifier_new ("g_object_steal_data"))), (_tmp44_ == NULL) ? NULL : (_tmp44_ = (vala_ccode_node_unref (_tmp44_), NULL)), _tmp45_);
		_tmp47_ = NULL;
		_tmp46_ = NULL;
		vala_ccode_function_call_add_argument (path, (ValaCCodeExpression*) (_tmp47_ = vala_ccode_cast_expression_new ((ValaCCodeExpression*) (_tmp46_ = vala_ccode_identifier_new ("object")), "GObject*")));
		(_tmp47_ == NULL) ? NULL : (_tmp47_ = (vala_ccode_node_unref (_tmp47_), NULL));
		(_tmp46_ == NULL) ? NULL : (_tmp46_ = (vala_ccode_node_unref (_tmp46_), NULL));
		_tmp48_ = NULL;
		vala_ccode_function_call_add_argument (path, (ValaCCodeExpression*) (_tmp48_ = vala_ccode_constant_new ("\"dbus_object_path\"")));
		(_tmp48_ == NULL) ? NULL : (_tmp48_ = (vala_ccode_node_unref (_tmp48_), NULL));
		_tmp51_ = NULL;
		_tmp50_ = NULL;
		_tmp49_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp51_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp50_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp49_ = vala_ccode_identifier_new ("path")), (ValaCCodeExpression*) path, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
		(_tmp51_ == NULL) ? NULL : (_tmp51_ = (vala_ccode_node_unref (_tmp51_), NULL));
		(_tmp50_ == NULL) ? NULL : (_tmp50_ = (vala_ccode_node_unref (_tmp50_), NULL));
		(_tmp49_ == NULL) ? NULL : (_tmp49_ = (vala_ccode_node_unref (_tmp49_), NULL));
		_tmp52_ = NULL;
		_tmp53_ = NULL;
		unregister_call = (_tmp53_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp52_ = vala_ccode_identifier_new ("dbus_connection_unregister_object_path"))), (_tmp52_ == NULL) ? NULL : (_tmp52_ = (vala_ccode_node_unref (_tmp52_), NULL)), _tmp53_);
		_tmp54_ = NULL;
		vala_ccode_function_call_add_argument (unregister_call, (ValaCCodeExpression*) (_tmp54_ = vala_ccode_identifier_new ("connection")));
		(_tmp54_ == NULL) ? NULL : (_tmp54_ = (vala_ccode_node_unref (_tmp54_), NULL));
		_tmp55_ = NULL;
		vala_ccode_function_call_add_argument (unregister_call, (ValaCCodeExpression*) (_tmp55_ = vala_ccode_identifier_new ("path")));
		(_tmp55_ == NULL) ? NULL : (_tmp55_ = (vala_ccode_node_unref (_tmp55_), NULL));
		_tmp56_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp56_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) unregister_call)));
		(_tmp56_ == NULL) ? NULL : (_tmp56_ = (vala_ccode_node_unref (_tmp56_), NULL));
		_tmp57_ = NULL;
		_tmp58_ = NULL;
		path_free = (_tmp58_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp57_ = vala_ccode_identifier_new ("g_free"))), (_tmp57_ == NULL) ? NULL : (_tmp57_ = (vala_ccode_node_unref (_tmp57_), NULL)), _tmp58_);
		_tmp59_ = NULL;
		vala_ccode_function_call_add_argument (path_free, (ValaCCodeExpression*) (_tmp59_ = vala_ccode_identifier_new ("path")));
		(_tmp59_ == NULL) ? NULL : (_tmp59_ = (vala_ccode_node_unref (_tmp59_), NULL));
		_tmp60_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp60_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) path_free)));
		(_tmp60_ == NULL) ? NULL : (_tmp60_ = (vala_ccode_node_unref (_tmp60_), NULL));
		vala_ccode_fragment_append (self->source_type_member_definition, (ValaCCodeNode*) cfunc);
		(dbusvtable == NULL) ? NULL : (dbusvtable = (vala_ccode_node_unref (dbusvtable), NULL));
		(cfunc == NULL) ? NULL : (cfunc = (vala_ccode_node_unref (cfunc), NULL));
		(block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL));
		(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
		(quark == NULL) ? NULL : (quark = (vala_ccode_node_unref (quark), NULL));
		(get_qdata == NULL) ? NULL : (get_qdata = (vala_ccode_node_unref (get_qdata), NULL));
		(cregister == NULL) ? NULL : (cregister = (vala_ccode_node_unref (cregister), NULL));
		(ifblock == NULL) ? NULL : (ifblock = (vala_ccode_node_unref (ifblock), NULL));
		(elseblock == NULL) ? NULL : (elseblock = (vala_ccode_node_unref (elseblock), NULL));
		(warn == NULL) ? NULL : (warn = (vala_ccode_node_unref (warn), NULL));
		(path == NULL) ? NULL : (path = (vala_ccode_node_unref (path), NULL));
		(unregister_call == NULL) ? NULL : (unregister_call = (vala_ccode_node_unref (unregister_call), NULL));
		(path_free == NULL) ? NULL : (path_free = (vala_ccode_node_unref (path_free), NULL));
	}
	comment = NULL;
	if (vala_source_file_get_comment (source_file) != NULL) {
		ValaCCodeComment* _tmp61_;
		_tmp61_ = NULL;
		comment = (_tmp61_ = vala_ccode_comment_new (vala_source_file_get_comment (source_file)), (comment == NULL) ? NULL : (comment = (vala_ccode_node_unref (comment), NULL)), _tmp61_);
	}
	_tmp62_ = NULL;
	_tmp63_ = NULL;
	writer = (_tmp63_ = vala_ccode_writer_new (_tmp62_ = vala_source_file_get_csource_filename (source_file)), _tmp62_ = (g_free (_tmp62_), NULL), _tmp63_);
	if (!vala_ccode_writer_open (writer)) {
		char* _tmp64_;
		_tmp64_ = NULL;
		vala_report_error (NULL, _tmp64_ = g_strdup_printf ("unable to open `%s' for writing", vala_ccode_writer_get_filename (writer)));
		_tmp64_ = (g_free (_tmp64_), NULL);
		(comment == NULL) ? NULL : (comment = (vala_ccode_node_unref (comment), NULL));
		(writer == NULL) ? NULL : (writer = (vala_ccode_writer_unref (writer), NULL));
		return;
	}
	vala_ccode_writer_set_line_directives (writer, vala_code_context_get_debug (self->priv->_context));
	if (comment != NULL) {
		vala_ccode_node_write ((ValaCCodeNode*) comment, writer);
	}
	vala_ccode_writer_write_newline (writer);
	vala_ccode_node_write ((ValaCCodeNode*) self->source_declarations->include_directives, writer);
	vala_ccode_writer_write_newline (writer);
	vala_ccode_node_write_combined ((ValaCCodeNode*) self->source_declarations->type_declaration, writer);
	vala_ccode_writer_write_newline (writer);
	vala_ccode_node_write_combined ((ValaCCodeNode*) self->source_declarations->type_definition, writer);
	vala_ccode_writer_write_newline (writer);
	vala_ccode_node_write_declaration ((ValaCCodeNode*) self->source_declarations->type_member_declaration, writer);
	vala_ccode_writer_write_newline (writer);
	vala_ccode_node_write ((ValaCCodeNode*) self->source_declarations->type_member_declaration, writer);
	vala_ccode_writer_write_newline (writer);
	vala_ccode_node_write ((ValaCCodeNode*) self->source_declarations->constant_declaration, writer);
	vala_ccode_writer_write_newline (writer);
	vala_ccode_node_write_declaration ((ValaCCodeNode*) self->source_signal_marshaller_declaration, writer);
	vala_ccode_node_write ((ValaCCodeNode*) self->source_signal_marshaller_declaration, writer);
	vala_ccode_writer_write_newline (writer);
	vala_ccode_node_write ((ValaCCodeNode*) self->source_type_member_definition, writer);
	vala_ccode_writer_write_newline (writer);
	vala_ccode_node_write ((ValaCCodeNode*) self->source_signal_marshaller_definition, writer);
	vala_ccode_writer_write_newline (writer);
	vala_ccode_writer_close (writer);
	_tmp65_ = NULL;
	self->source_declarations = (_tmp65_ = NULL, (self->source_declarations == NULL) ? NULL : (self->source_declarations = (vala_ccode_declaration_space_unref (self->source_declarations), NULL)), _tmp65_);
	_tmp66_ = NULL;
	self->source_type_member_definition = (_tmp66_ = NULL, (self->source_type_member_definition == NULL) ? NULL : (self->source_type_member_definition = (vala_ccode_node_unref (self->source_type_member_definition), NULL)), _tmp66_);
	_tmp67_ = NULL;
	self->source_signal_marshaller_definition = (_tmp67_ = NULL, (self->source_signal_marshaller_definition == NULL) ? NULL : (self->source_signal_marshaller_definition = (vala_ccode_node_unref (self->source_signal_marshaller_definition), NULL)), _tmp67_);
	_tmp68_ = NULL;
	self->source_signal_marshaller_declaration = (_tmp68_ = NULL, (self->source_signal_marshaller_declaration == NULL) ? NULL : (self->source_signal_marshaller_declaration = (vala_ccode_node_unref (self->source_signal_marshaller_declaration), NULL)), _tmp68_);
	(comment == NULL) ? NULL : (comment = (vala_ccode_node_unref (comment), NULL));
	(writer == NULL) ? NULL : (writer = (vala_ccode_writer_unref (writer), NULL));
}


static char* vala_ccode_base_module_get_define_for_filename (const char* filename) {
	GString* define;
	const char* _tmp0_;
	char* i;
	const char* _tmp4_;
	char* _tmp5_;
	g_return_val_if_fail (filename != NULL, NULL);
	define = g_string_new ("__");
	_tmp0_ = NULL;
	i = (_tmp0_ = filename, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		gboolean _tmp1_;
		char* _tmp3_;
		const char* _tmp2_;
		c = g_utf8_get_char (i);
		_tmp1_ = FALSE;
		if (g_unichar_isalnum (c)) {
			_tmp1_ = c < 0x80;
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			g_string_append_unichar (define, g_unichar_toupper (c));
		} else {
			g_string_append_c (define, '_');
		}
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		i = (_tmp3_ = (_tmp2_ = g_utf8_next_char (i), (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_)), i = (g_free (i), NULL), _tmp3_);
	}
	g_string_append (define, "__");
	_tmp4_ = NULL;
	_tmp5_ = NULL;
	return (_tmp5_ = (_tmp4_ = define->str, (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_)), (define == NULL) ? NULL : (define = (g_string_free (define, TRUE), NULL)), i = (g_free (i), NULL), _tmp5_);
}


void vala_ccode_base_module_generate_enum_declaration (ValaCCodeBaseModule* self, ValaEnum* en, ValaCCodeDeclarationSpace* decl_space) {
	char* _tmp0_;
	gboolean _tmp1_;
	char* _tmp2_;
	ValaCCodeEnum* _tmp3_;
	ValaCCodeEnum* cenum;
	ValaCCodeNewline* _tmp11_;
	ValaCCodeNewline* _tmp12_;
	char* _tmp13_;
	char* _tmp14_;
	char* macro;
	ValaCCodeMacroReplacement* _tmp16_;
	char* _tmp15_;
	char* _tmp17_;
	char* _tmp18_;
	char* fun_name;
	ValaCCodeFunction* regfun;
	g_return_if_fail (self != NULL);
	g_return_if_fail (en != NULL);
	g_return_if_fail (decl_space != NULL);
	_tmp0_ = NULL;
	if ((_tmp1_ = vala_ccode_declaration_space_add_symbol_declaration (decl_space, (ValaSymbol*) en, _tmp0_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) en, FALSE)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_)) {
		return;
	}
	_tmp2_ = NULL;
	_tmp3_ = NULL;
	cenum = (_tmp3_ = vala_ccode_enum_new (_tmp2_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) en, FALSE)), _tmp2_ = (g_free (_tmp2_), NULL), _tmp3_);
	{
		GeeList* _tmp4_;
		GeeIterator* _tmp5_;
		GeeIterator* _ev_it;
		_tmp4_ = NULL;
		_tmp5_ = NULL;
		_ev_it = (_tmp5_ = gee_iterable_iterator ((GeeIterable*) (_tmp4_ = vala_enum_get_values (en))), (_tmp4_ == NULL) ? NULL : (_tmp4_ = (gee_collection_object_unref (_tmp4_), NULL)), _tmp5_);
		while (gee_iterator_next (_ev_it)) {
			ValaEnumValue* ev;
			ev = (ValaEnumValue*) gee_iterator_get (_ev_it);
			if (vala_enum_value_get_value (ev) == NULL) {
				ValaCCodeEnumValue* _tmp7_;
				char* _tmp6_;
				_tmp7_ = NULL;
				_tmp6_ = NULL;
				vala_ccode_enum_add_value (cenum, _tmp7_ = vala_ccode_enum_value_new (_tmp6_ = vala_enum_value_get_cname (ev), NULL));
				(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
				_tmp6_ = (g_free (_tmp6_), NULL);
			} else {
				ValaCCodeEnumValue* _tmp9_;
				char* _tmp8_;
				vala_code_node_accept ((ValaCodeNode*) vala_enum_value_get_value (ev), (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
				_tmp9_ = NULL;
				_tmp8_ = NULL;
				vala_ccode_enum_add_value (cenum, _tmp9_ = vala_ccode_enum_value_new (_tmp8_ = vala_enum_value_get_cname (ev), VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_enum_value_get_value (ev)))));
				(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
				_tmp8_ = (g_free (_tmp8_), NULL);
			}
			(ev == NULL) ? NULL : (ev = (vala_code_node_unref (ev), NULL));
		}
		(_ev_it == NULL) ? NULL : (_ev_it = (gee_collection_object_unref (_ev_it), NULL));
	}
	if (vala_source_reference_get_comment (vala_code_node_get_source_reference ((ValaCodeNode*) en)) != NULL) {
		ValaCCodeComment* _tmp10_;
		_tmp10_ = NULL;
		vala_ccode_declaration_space_add_type_definition (decl_space, (ValaCCodeNode*) (_tmp10_ = vala_ccode_comment_new (vala_source_reference_get_comment (vala_code_node_get_source_reference ((ValaCodeNode*) en)))));
		(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
	}
	vala_ccode_declaration_space_add_type_definition (decl_space, (ValaCCodeNode*) cenum);
	_tmp11_ = NULL;
	vala_ccode_declaration_space_add_type_definition (decl_space, (ValaCCodeNode*) (_tmp11_ = vala_ccode_newline_new ()));
	(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
	if (!vala_enum_get_has_type_id (en)) {
		(cenum == NULL) ? NULL : (cenum = (vala_ccode_node_unref (cenum), NULL));
		return;
	}
	_tmp12_ = NULL;
	vala_ccode_declaration_space_add_type_declaration (decl_space, (ValaCCodeNode*) (_tmp12_ = vala_ccode_newline_new ()));
	(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
	_tmp13_ = NULL;
	_tmp14_ = NULL;
	macro = (_tmp14_ = g_strdup_printf ("(%s_get_type ())", _tmp13_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) en, NULL)), _tmp13_ = (g_free (_tmp13_), NULL), _tmp14_);
	_tmp16_ = NULL;
	_tmp15_ = NULL;
	vala_ccode_declaration_space_add_type_declaration (decl_space, (ValaCCodeNode*) (_tmp16_ = vala_ccode_macro_replacement_new (_tmp15_ = vala_typesymbol_get_type_id ((ValaTypeSymbol*) en), macro)));
	(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL));
	_tmp15_ = (g_free (_tmp15_), NULL);
	_tmp17_ = NULL;
	_tmp18_ = NULL;
	fun_name = (_tmp18_ = g_strdup_printf ("%s_get_type", _tmp17_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) en, NULL)), _tmp17_ = (g_free (_tmp17_), NULL), _tmp18_);
	regfun = vala_ccode_function_new (fun_name, "GType");
	if (vala_symbol_get_access ((ValaSymbol*) en) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_ccode_function_set_modifiers (regfun, VALA_CCODE_MODIFIERS_STATIC);
	}
	vala_ccode_declaration_space_add_type_member_declaration (decl_space, (ValaCCodeNode*) regfun);
	(cenum == NULL) ? NULL : (cenum = (vala_ccode_node_unref (cenum), NULL));
	macro = (g_free (macro), NULL);
	fun_name = (g_free (fun_name), NULL);
	(regfun == NULL) ? NULL : (regfun = (vala_ccode_node_unref (regfun), NULL));
}


static void vala_ccode_base_module_real_visit_enum (ValaCCodeModule* base, ValaEnum* en) {
	ValaCCodeBaseModule * self;
	ValaCCodeInitializerList* clist;
	ValaCCodeInitializerList* clist_ev;
	ValaCCodeInitializerList* _tmp9_;
	ValaCCodeConstant* _tmp10_;
	ValaCCodeConstant* _tmp11_;
	ValaCCodeConstant* _tmp12_;
	ValaCCodeVariableDeclarator* enum_decl;
	ValaCCodeDeclaration* _cdecl_;
	ValaCCodeBlock* type_init;
	char* _tmp15_;
	char* _tmp16_;
	char* fun_name;
	ValaCCodeFunction* regfun;
	ValaCCodeBlock* regblock;
	ValaCCodeDeclaration* _tmp17_;
	char* _tmp18_;
	char* _tmp19_;
	char* type_id_name;
	ValaCCodeVariableDeclarator* _tmp21_;
	ValaCCodeConstant* _tmp20_;
	ValaCCodeFunctionCall* reg_call;
	ValaCCodeConstant* _tmp28_;
	char* _tmp27_;
	char* _tmp26_;
	ValaCCodeIdentifier* _tmp29_;
	ValaCCodeExpressionStatement* _tmp32_;
	ValaCCodeAssignment* _tmp31_;
	ValaCCodeIdentifier* _tmp30_;
	ValaCCodeIdentifier* _tmp33_;
	ValaCCodeFunctionCall* _tmp34_;
	ValaCCodeFunctionCall* cond;
	ValaCCodeBinaryExpression* _tmp37_;
	ValaCCodeConstant* _tmp36_;
	ValaCCodeIdentifier* _tmp35_;
	ValaCCodeIfStatement* cif;
	ValaCCodeReturnStatement* _tmp39_;
	ValaCCodeConstant* _tmp38_;
	ValaCCodeNewline* _tmp40_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (en != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) en, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	vala_ccode_base_module_generate_enum_declaration (self, en, self->source_declarations);
	if (!vala_symbol_is_internal_symbol ((ValaSymbol*) en)) {
		vala_ccode_base_module_generate_enum_declaration (self, en, self->header_declarations);
	}
	vala_ccode_base_module_generate_enum_declaration (self, en, self->internal_header_declarations);
	if (!vala_enum_get_has_type_id (en)) {
		return;
	}
	clist = vala_ccode_initializer_list_new ();
	/* or during visit time? */
	clist_ev = NULL;
	{
		GeeList* _tmp0_;
		GeeIterator* _tmp1_;
		GeeIterator* _ev_it;
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		_ev_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = vala_enum_get_values (en))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (gee_collection_object_unref (_tmp0_), NULL)), _tmp1_);
		while (gee_iterator_next (_ev_it)) {
			ValaEnumValue* ev;
			ValaCCodeInitializerList* _tmp2_;
			ValaCCodeConstant* _tmp4_;
			char* _tmp3_;
			ValaCCodeIdentifier* _tmp7_;
			char* _tmp6_;
			char* _tmp5_;
			ValaCCodeConstant* _tmp8_;
			ev = (ValaEnumValue*) gee_iterator_get (_ev_it);
			_tmp2_ = NULL;
			clist_ev = (_tmp2_ = vala_ccode_initializer_list_new (), (clist_ev == NULL) ? NULL : (clist_ev = (vala_ccode_node_unref (clist_ev), NULL)), _tmp2_);
			_tmp4_ = NULL;
			_tmp3_ = NULL;
			vala_ccode_initializer_list_append (clist_ev, (ValaCCodeExpression*) (_tmp4_ = vala_ccode_constant_new (_tmp3_ = vala_enum_value_get_cname (ev))));
			(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
			_tmp3_ = (g_free (_tmp3_), NULL);
			_tmp7_ = NULL;
			_tmp6_ = NULL;
			_tmp5_ = NULL;
			vala_ccode_initializer_list_append (clist_ev, (ValaCCodeExpression*) (_tmp7_ = vala_ccode_identifier_new (_tmp6_ = g_strdup_printf ("\"%s\"", _tmp5_ = vala_enum_value_get_cname (ev)))));
			(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
			_tmp6_ = (g_free (_tmp6_), NULL);
			_tmp5_ = (g_free (_tmp5_), NULL);
			_tmp8_ = NULL;
			vala_ccode_initializer_list_append (clist_ev, (ValaCCodeExpression*) (_tmp8_ = vala_enum_value_get_canonical_cconstant (ev)));
			(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL));
			vala_ccode_initializer_list_append (clist, (ValaCCodeExpression*) clist_ev);
			(ev == NULL) ? NULL : (ev = (vala_code_node_unref (ev), NULL));
		}
		(_ev_it == NULL) ? NULL : (_ev_it = (gee_collection_object_unref (_ev_it), NULL));
	}
	_tmp9_ = NULL;
	clist_ev = (_tmp9_ = vala_ccode_initializer_list_new (), (clist_ev == NULL) ? NULL : (clist_ev = (vala_ccode_node_unref (clist_ev), NULL)), _tmp9_);
	_tmp10_ = NULL;
	vala_ccode_initializer_list_append (clist_ev, (ValaCCodeExpression*) (_tmp10_ = vala_ccode_constant_new ("0")));
	(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
	_tmp11_ = NULL;
	vala_ccode_initializer_list_append (clist_ev, (ValaCCodeExpression*) (_tmp11_ = vala_ccode_constant_new ("NULL")));
	(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
	_tmp12_ = NULL;
	vala_ccode_initializer_list_append (clist_ev, (ValaCCodeExpression*) (_tmp12_ = vala_ccode_constant_new ("NULL")));
	(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
	vala_ccode_initializer_list_append (clist, (ValaCCodeExpression*) clist_ev);
	enum_decl = vala_ccode_variable_declarator_new ("values[]", (ValaCCodeExpression*) clist, NULL);
	_cdecl_ = NULL;
	if (vala_enum_get_is_flags (en)) {
		ValaCCodeDeclaration* _tmp13_;
		_tmp13_ = NULL;
		_cdecl_ = (_tmp13_ = vala_ccode_declaration_new ("const GFlagsValue"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp13_);
	} else {
		ValaCCodeDeclaration* _tmp14_;
		_tmp14_ = NULL;
		_cdecl_ = (_tmp14_ = vala_ccode_declaration_new ("const GEnumValue"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp14_);
	}
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) enum_decl);
	vala_ccode_declaration_set_modifiers (_cdecl_, VALA_CCODE_MODIFIERS_STATIC);
	type_init = vala_ccode_block_new ();
	vala_ccode_block_add_statement (type_init, (ValaCCodeNode*) _cdecl_);
	_tmp15_ = NULL;
	_tmp16_ = NULL;
	fun_name = (_tmp16_ = g_strdup_printf ("%s_get_type", _tmp15_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) en, NULL)), _tmp15_ = (g_free (_tmp15_), NULL), _tmp16_);
	regfun = vala_ccode_function_new (fun_name, "GType");
	regblock = vala_ccode_block_new ();
	_tmp17_ = NULL;
	_cdecl_ = (_tmp17_ = vala_ccode_declaration_new ("GType"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp17_);
	_tmp18_ = NULL;
	_tmp19_ = NULL;
	type_id_name = (_tmp19_ = g_strdup_printf ("%s_type_id", _tmp18_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) en, NULL)), _tmp18_ = (g_free (_tmp18_), NULL), _tmp19_);
	_tmp21_ = NULL;
	_tmp20_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp21_ = vala_ccode_variable_declarator_new (type_id_name, (ValaCCodeExpression*) (_tmp20_ = vala_ccode_constant_new ("0")), NULL)));
	(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL));
	(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL));
	vala_ccode_declaration_set_modifiers (_cdecl_, VALA_CCODE_MODIFIERS_STATIC);
	vala_ccode_block_add_statement (regblock, (ValaCCodeNode*) _cdecl_);
	reg_call = NULL;
	if (vala_enum_get_is_flags (en)) {
		ValaCCodeFunctionCall* _tmp23_;
		ValaCCodeIdentifier* _tmp22_;
		_tmp23_ = NULL;
		_tmp22_ = NULL;
		reg_call = (_tmp23_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp22_ = vala_ccode_identifier_new ("g_flags_register_static"))), (reg_call == NULL) ? NULL : (reg_call = (vala_ccode_node_unref (reg_call), NULL)), _tmp23_);
		(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL));
	} else {
		ValaCCodeFunctionCall* _tmp25_;
		ValaCCodeIdentifier* _tmp24_;
		_tmp25_ = NULL;
		_tmp24_ = NULL;
		reg_call = (_tmp25_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp24_ = vala_ccode_identifier_new ("g_enum_register_static"))), (reg_call == NULL) ? NULL : (reg_call = (vala_ccode_node_unref (reg_call), NULL)), _tmp25_);
		(_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_ccode_node_unref (_tmp24_), NULL));
	}
	_tmp28_ = NULL;
	_tmp27_ = NULL;
	_tmp26_ = NULL;
	vala_ccode_function_call_add_argument (reg_call, (ValaCCodeExpression*) (_tmp28_ = vala_ccode_constant_new (_tmp27_ = g_strdup_printf ("\"%s\"", _tmp26_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) en, FALSE)))));
	(_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_ccode_node_unref (_tmp28_), NULL));
	_tmp27_ = (g_free (_tmp27_), NULL);
	_tmp26_ = (g_free (_tmp26_), NULL);
	_tmp29_ = NULL;
	vala_ccode_function_call_add_argument (reg_call, (ValaCCodeExpression*) (_tmp29_ = vala_ccode_identifier_new ("values")));
	(_tmp29_ == NULL) ? NULL : (_tmp29_ = (vala_ccode_node_unref (_tmp29_), NULL));
	_tmp32_ = NULL;
	_tmp31_ = NULL;
	_tmp30_ = NULL;
	vala_ccode_block_add_statement (type_init, (ValaCCodeNode*) (_tmp32_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp31_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp30_ = vala_ccode_identifier_new (type_id_name)), (ValaCCodeExpression*) reg_call, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
	(_tmp32_ == NULL) ? NULL : (_tmp32_ = (vala_ccode_node_unref (_tmp32_), NULL));
	(_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_ccode_node_unref (_tmp31_), NULL));
	(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL));
	_tmp33_ = NULL;
	_tmp34_ = NULL;
	cond = (_tmp34_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp33_ = vala_ccode_identifier_new ("G_UNLIKELY"))), (_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_ccode_node_unref (_tmp33_), NULL)), _tmp34_);
	_tmp37_ = NULL;
	_tmp36_ = NULL;
	_tmp35_ = NULL;
	vala_ccode_function_call_add_argument (cond, (ValaCCodeExpression*) (_tmp37_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, (ValaCCodeExpression*) (_tmp35_ = vala_ccode_identifier_new (type_id_name)), (ValaCCodeExpression*) (_tmp36_ = vala_ccode_constant_new ("0")))));
	(_tmp37_ == NULL) ? NULL : (_tmp37_ = (vala_ccode_node_unref (_tmp37_), NULL));
	(_tmp36_ == NULL) ? NULL : (_tmp36_ = (vala_ccode_node_unref (_tmp36_), NULL));
	(_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_ccode_node_unref (_tmp35_), NULL));
	cif = vala_ccode_if_statement_new ((ValaCCodeExpression*) cond, (ValaCCodeStatement*) type_init, NULL);
	vala_ccode_block_add_statement (regblock, (ValaCCodeNode*) cif);
	_tmp39_ = NULL;
	_tmp38_ = NULL;
	vala_ccode_block_add_statement (regblock, (ValaCCodeNode*) (_tmp39_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp38_ = vala_ccode_constant_new (type_id_name)))));
	(_tmp39_ == NULL) ? NULL : (_tmp39_ = (vala_ccode_node_unref (_tmp39_), NULL));
	(_tmp38_ == NULL) ? NULL : (_tmp38_ = (vala_ccode_node_unref (_tmp38_), NULL));
	if (vala_symbol_get_access ((ValaSymbol*) en) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_ccode_function_set_modifiers (regfun, VALA_CCODE_MODIFIERS_STATIC);
	}
	vala_ccode_function_set_block (regfun, regblock);
	_tmp40_ = NULL;
	vala_ccode_fragment_append (self->source_type_member_definition, (ValaCCodeNode*) (_tmp40_ = vala_ccode_newline_new ()));
	(_tmp40_ == NULL) ? NULL : (_tmp40_ = (vala_ccode_node_unref (_tmp40_), NULL));
	vala_ccode_fragment_append (self->source_type_member_definition, (ValaCCodeNode*) regfun);
	(clist == NULL) ? NULL : (clist = (vala_ccode_node_unref (clist), NULL));
	(clist_ev == NULL) ? NULL : (clist_ev = (vala_ccode_node_unref (clist_ev), NULL));
	(enum_decl == NULL) ? NULL : (enum_decl = (vala_ccode_node_unref (enum_decl), NULL));
	(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
	(type_init == NULL) ? NULL : (type_init = (vala_ccode_node_unref (type_init), NULL));
	fun_name = (g_free (fun_name), NULL);
	(regfun == NULL) ? NULL : (regfun = (vala_ccode_node_unref (regfun), NULL));
	(regblock == NULL) ? NULL : (regblock = (vala_ccode_node_unref (regblock), NULL));
	type_id_name = (g_free (type_id_name), NULL);
	(reg_call == NULL) ? NULL : (reg_call = (vala_ccode_node_unref (reg_call), NULL));
	(cond == NULL) ? NULL : (cond = (vala_ccode_node_unref (cond), NULL));
	(cif == NULL) ? NULL : (cif = (vala_ccode_node_unref (cif), NULL));
}


static void vala_ccode_base_module_real_visit_member (ValaCCodeModule* base, ValaMember* m) {
	ValaCCodeBaseModule * self;
	gboolean _tmp0_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (m != NULL);
	_tmp0_ = FALSE;
	if (VALA_IS_LOCKABLE (m)) {
		_tmp0_ = vala_lockable_get_lock_used (VALA_LOCKABLE (m));
	} else {
		_tmp0_ = FALSE;
	}
	/* stuff meant for all lockable members */
	if (_tmp0_) {
		ValaCCodeExpression* l;
		ValaCCodeExpression* _tmp3_;
		char* _tmp2_;
		ValaCCodeMemberAccess* _tmp1_;
		ValaCCodeIdentifier* _tmp5_;
		char* _tmp4_;
		ValaCCodeFunctionCall* _tmp6_;
		ValaCCodeFunctionCall* initf;
		ValaCCodeUnaryExpression* _tmp7_;
		ValaCCodeExpressionStatement* _tmp8_;
		l = (ValaCCodeExpression*) vala_ccode_identifier_new ("self");
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		l = (_tmp3_ = (ValaCCodeExpression*) vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp1_ = vala_ccode_member_access_new_pointer (l, "priv")), _tmp2_ = vala_ccode_base_module_get_symbol_lock_name (self, (ValaSymbol*) m)), (l == NULL) ? NULL : (l = (vala_ccode_node_unref (l), NULL)), _tmp3_);
		_tmp2_ = (g_free (_tmp2_), NULL);
		(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_ccode_node_unref (_tmp1_), NULL));
		_tmp5_ = NULL;
		_tmp4_ = NULL;
		_tmp6_ = NULL;
		initf = (_tmp6_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp5_ = vala_ccode_identifier_new (_tmp4_ = vala_method_get_cname (vala_struct_get_default_construction_method (self->mutex_type))))), (_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL)), _tmp4_ = (g_free (_tmp4_), NULL), _tmp6_);
		_tmp7_ = NULL;
		vala_ccode_function_call_add_argument (initf, (ValaCCodeExpression*) (_tmp7_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, l)));
		(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
		_tmp8_ = NULL;
		vala_ccode_fragment_append (self->instance_init_fragment, (ValaCCodeNode*) (_tmp8_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) initf)));
		(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL));
		if (self->instance_finalize_fragment != NULL) {
			ValaCCodeIdentifier* _tmp9_;
			ValaCCodeFunctionCall* _tmp10_;
			ValaCCodeFunctionCall* fc;
			ValaCCodeUnaryExpression* _tmp11_;
			ValaCCodeExpressionStatement* _tmp12_;
			_tmp9_ = NULL;
			_tmp10_ = NULL;
			fc = (_tmp10_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp9_ = vala_ccode_identifier_new ("g_static_rec_mutex_free"))), (_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL)), _tmp10_);
			_tmp11_ = NULL;
			vala_ccode_function_call_add_argument (fc, (ValaCCodeExpression*) (_tmp11_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, l)));
			(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
			_tmp12_ = NULL;
			vala_ccode_fragment_append (self->instance_finalize_fragment, (ValaCCodeNode*) (_tmp12_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) fc)));
			(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
			(fc == NULL) ? NULL : (fc = (vala_ccode_node_unref (fc), NULL));
		}
		(l == NULL) ? NULL : (l = (vala_ccode_node_unref (l), NULL));
		(initf == NULL) ? NULL : (initf = (vala_ccode_node_unref (initf), NULL));
	}
}


void vala_ccode_base_module_generate_constant_declaration (ValaCCodeBaseModule* self, ValaConstant* c, ValaCCodeDeclarationSpace* decl_space) {
	char* _tmp0_;
	gboolean _tmp1_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (c != NULL);
	g_return_if_fail (decl_space != NULL);
	_tmp0_ = NULL;
	if ((_tmp1_ = vala_ccode_declaration_space_add_symbol_declaration (decl_space, (ValaSymbol*) c, _tmp0_ = vala_constant_get_cname (c)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_)) {
		return;
	}
	vala_code_node_accept_children ((ValaCodeNode*) c, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	if (!vala_symbol_get_external ((ValaSymbol*) c)) {
		if (VALA_IS_INITIALIZER_LIST (vala_constant_get_initializer (c))) {
			char* _tmp2_;
			ValaCCodeDeclaration* _tmp3_;
			ValaCCodeDeclaration* _cdecl_;
			char* arr;
			ValaCCodeVariableDeclarator* _tmp7_;
			char* _tmp6_;
			char* _tmp5_;
			_tmp2_ = NULL;
			_tmp3_ = NULL;
			_cdecl_ = (_tmp3_ = vala_ccode_declaration_new (_tmp2_ = vala_data_type_get_const_cname (vala_constant_get_type_reference (c))), _tmp2_ = (g_free (_tmp2_), NULL), _tmp3_);
			arr = g_strdup ("");
			if (VALA_IS_ARRAY_TYPE (vala_constant_get_type_reference (c))) {
				char* _tmp4_;
				_tmp4_ = NULL;
				arr = (_tmp4_ = g_strdup ("[]"), arr = (g_free (arr), NULL), _tmp4_);
			}
			_tmp7_ = NULL;
			_tmp6_ = NULL;
			_tmp5_ = NULL;
			vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp7_ = vala_ccode_variable_declarator_new (_tmp6_ = g_strdup_printf ("%s%s", _tmp5_ = vala_constant_get_cname (c), arr), VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_constant_get_initializer (c))), NULL)));
			(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
			_tmp6_ = (g_free (_tmp6_), NULL);
			_tmp5_ = (g_free (_tmp5_), NULL);
			vala_ccode_declaration_set_modifiers (_cdecl_, VALA_CCODE_MODIFIERS_STATIC);
			vala_ccode_declaration_space_add_constant_declaration (decl_space, (ValaCCodeNode*) _cdecl_);
			(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
			arr = (g_free (arr), NULL);
		} else {
			char* _tmp8_;
			ValaCCodeMacroReplacement* _tmp9_;
			ValaCCodeMacroReplacement* cdefine;
			_tmp8_ = NULL;
			_tmp9_ = NULL;
			cdefine = (_tmp9_ = vala_ccode_macro_replacement_new_with_expression (_tmp8_ = vala_constant_get_cname (c), VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_constant_get_initializer (c)))), _tmp8_ = (g_free (_tmp8_), NULL), _tmp9_);
			vala_ccode_declaration_space_add_type_member_declaration (decl_space, (ValaCCodeNode*) cdefine);
			(cdefine == NULL) ? NULL : (cdefine = (vala_ccode_node_unref (cdefine), NULL));
		}
	}
}


static void vala_ccode_base_module_real_visit_constant (ValaCCodeModule* base, ValaConstant* c) {
	ValaCCodeBaseModule * self;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (c != NULL);
	vala_ccode_base_module_generate_constant_declaration (self, c, self->source_declarations);
	if (!vala_symbol_is_internal_symbol ((ValaSymbol*) c)) {
		vala_ccode_base_module_generate_constant_declaration (self, c, self->header_declarations);
	}
	vala_ccode_base_module_generate_constant_declaration (self, c, self->internal_header_declarations);
}


void vala_ccode_base_module_generate_field_declaration (ValaCCodeBaseModule* self, ValaField* f, ValaCCodeDeclarationSpace* decl_space) {
	char* _tmp0_;
	gboolean _tmp1_;
	char* field_ctype;
	ValaCCodeDeclaration* _cdecl_;
	ValaCCodeVariableDeclarator* _tmp4_;
	char* _tmp3_;
	gboolean _tmp5_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (f != NULL);
	g_return_if_fail (decl_space != NULL);
	_tmp0_ = NULL;
	if ((_tmp1_ = vala_ccode_declaration_space_add_symbol_declaration (decl_space, (ValaSymbol*) f, _tmp0_ = vala_field_get_cname (f)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_)) {
		return;
	}
	field_ctype = vala_data_type_get_cname (vala_field_get_field_type (f));
	if (vala_field_get_is_volatile (f)) {
		char* _tmp2_;
		_tmp2_ = NULL;
		field_ctype = (_tmp2_ = g_strconcat ("volatile ", field_ctype, NULL), field_ctype = (g_free (field_ctype), NULL), _tmp2_);
	}
	_cdecl_ = vala_ccode_declaration_new (field_ctype);
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp4_ = vala_ccode_variable_declarator_new (_tmp3_ = vala_field_get_cname (f), NULL, NULL)));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	_tmp3_ = (g_free (_tmp3_), NULL);
	if (vala_symbol_is_private_symbol ((ValaSymbol*) f)) {
		vala_ccode_declaration_set_modifiers (_cdecl_, VALA_CCODE_MODIFIERS_STATIC);
	} else {
		vala_ccode_declaration_set_modifiers (_cdecl_, VALA_CCODE_MODIFIERS_EXTERN);
	}
	vala_ccode_declaration_space_add_type_member_declaration (decl_space, (ValaCCodeNode*) _cdecl_);
	_tmp5_ = FALSE;
	if (VALA_IS_ARRAY_TYPE (vala_field_get_field_type (f))) {
		_tmp5_ = !vala_field_get_no_array_length (f);
	} else {
		_tmp5_ = FALSE;
	}
	if (_tmp5_) {
		ValaArrayType* _tmp6_;
		ValaArrayType* array_type;
		_tmp6_ = NULL;
		array_type = (_tmp6_ = VALA_ARRAY_TYPE (vala_field_get_field_type (f)), (_tmp6_ == NULL) ? NULL : vala_code_node_ref (_tmp6_));
		{
			gint dim;
			dim = 1;
			for (; dim <= vala_array_type_get_rank (array_type); dim++) {
				ValaDataType* len_type;
				ValaCCodeDeclaration* _tmp8_;
				char* _tmp7_;
				ValaCCodeVariableDeclarator* _tmp11_;
				char* _tmp10_;
				char* _tmp9_;
				len_type = vala_data_type_copy (self->int_type);
				_tmp8_ = NULL;
				_tmp7_ = NULL;
				_cdecl_ = (_tmp8_ = vala_ccode_declaration_new (_tmp7_ = vala_data_type_get_cname (len_type)), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp8_);
				_tmp7_ = (g_free (_tmp7_), NULL);
				_tmp11_ = NULL;
				_tmp10_ = NULL;
				_tmp9_ = NULL;
				vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp11_ = vala_ccode_variable_declarator_new (_tmp10_ = vala_ccode_module_get_array_length_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), _tmp9_ = vala_field_get_cname (f), dim), NULL, NULL)));
				(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
				_tmp10_ = (g_free (_tmp10_), NULL);
				_tmp9_ = (g_free (_tmp9_), NULL);
				if (vala_symbol_is_private_symbol ((ValaSymbol*) f)) {
					vala_ccode_declaration_set_modifiers (_cdecl_, VALA_CCODE_MODIFIERS_STATIC);
				} else {
					vala_ccode_declaration_set_modifiers (_cdecl_, VALA_CCODE_MODIFIERS_EXTERN);
				}
				vala_ccode_declaration_space_add_type_member_declaration (decl_space, (ValaCCodeNode*) _cdecl_);
				(len_type == NULL) ? NULL : (len_type = (vala_code_node_unref (len_type), NULL));
			}
		}
		(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
	} else {
		if (VALA_IS_DELEGATE_TYPE (vala_field_get_field_type (f))) {
			ValaDelegateType* _tmp12_;
			ValaDelegateType* delegate_type;
			_tmp12_ = NULL;
			delegate_type = (_tmp12_ = VALA_DELEGATE_TYPE (vala_field_get_field_type (f)), (_tmp12_ == NULL) ? NULL : vala_code_node_ref (_tmp12_));
			if (vala_delegate_get_has_target (vala_delegate_type_get_delegate_symbol (delegate_type))) {
				ValaCCodeDeclaration* _tmp13_;
				ValaCCodeVariableDeclarator* _tmp16_;
				char* _tmp15_;
				char* _tmp14_;
				/* create field to store delegate target*/
				_tmp13_ = NULL;
				_cdecl_ = (_tmp13_ = vala_ccode_declaration_new ("gpointer"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp13_);
				_tmp16_ = NULL;
				_tmp15_ = NULL;
				_tmp14_ = NULL;
				vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp16_ = vala_ccode_variable_declarator_new (_tmp15_ = vala_ccode_base_module_get_delegate_target_cname (self, _tmp14_ = vala_field_get_cname (f)), NULL, NULL)));
				(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL));
				_tmp15_ = (g_free (_tmp15_), NULL);
				_tmp14_ = (g_free (_tmp14_), NULL);
				if (vala_symbol_is_private_symbol ((ValaSymbol*) f)) {
					vala_ccode_declaration_set_modifiers (_cdecl_, VALA_CCODE_MODIFIERS_STATIC);
				} else {
					vala_ccode_declaration_set_modifiers (_cdecl_, VALA_CCODE_MODIFIERS_EXTERN);
				}
				vala_ccode_declaration_space_add_type_member_declaration (decl_space, (ValaCCodeNode*) _cdecl_);
			}
			(delegate_type == NULL) ? NULL : (delegate_type = (vala_code_node_unref (delegate_type), NULL));
		}
	}
	field_ctype = (g_free (field_ctype), NULL);
	(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
}


static void vala_ccode_base_module_real_visit_field (ValaCCodeModule* base, ValaField* f) {
	ValaCCodeBaseModule * self;
	ValaClass* _tmp1_;
	ValaSymbol* _tmp0_;
	ValaClass* cl;
	gboolean _tmp2_;
	gboolean is_gtypeinstance;
	ValaCCodeExpression* lhs;
	char* field_ctype;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (f != NULL);
	vala_ccode_base_module_check_type (self, vala_field_get_field_type (f));
	vala_code_node_accept_children ((ValaCodeNode*) f, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	cl = (_tmp1_ = (_tmp0_ = vala_symbol_get_parent_symbol ((ValaSymbol*) f), VALA_IS_CLASS (_tmp0_) ? ((ValaClass*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp2_ = FALSE;
	if (cl != NULL) {
		_tmp2_ = !vala_class_get_is_compact (cl);
	} else {
		_tmp2_ = FALSE;
	}
	is_gtypeinstance = _tmp2_;
	lhs = NULL;
	field_ctype = vala_data_type_get_cname (vala_field_get_field_type (f));
	if (vala_field_get_is_volatile (f)) {
		char* _tmp3_;
		_tmp3_ = NULL;
		field_ctype = (_tmp3_ = g_strconcat ("volatile ", field_ctype, NULL), field_ctype = (g_free (field_ctype), NULL), _tmp3_);
	}
	if (vala_field_get_binding (f) == MEMBER_BINDING_INSTANCE) {
		gboolean _tmp4_;
		gboolean _tmp22_;
		_tmp4_ = FALSE;
		if (is_gtypeinstance) {
			_tmp4_ = vala_symbol_get_access ((ValaSymbol*) f) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE;
		} else {
			_tmp4_ = FALSE;
		}
		if (_tmp4_) {
			ValaCCodeExpression* _tmp8_;
			char* _tmp7_;
			ValaCCodeMemberAccess* _tmp6_;
			ValaCCodeIdentifier* _tmp5_;
			_tmp8_ = NULL;
			_tmp7_ = NULL;
			_tmp6_ = NULL;
			_tmp5_ = NULL;
			lhs = (_tmp8_ = (ValaCCodeExpression*) vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp6_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp5_ = vala_ccode_identifier_new ("self")), "priv")), _tmp7_ = vala_field_get_cname (f)), (lhs == NULL) ? NULL : (lhs = (vala_ccode_node_unref (lhs), NULL)), _tmp8_);
			_tmp7_ = (g_free (_tmp7_), NULL);
			(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
			(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
		} else {
			ValaCCodeExpression* _tmp11_;
			char* _tmp10_;
			ValaCCodeIdentifier* _tmp9_;
			_tmp11_ = NULL;
			_tmp10_ = NULL;
			_tmp9_ = NULL;
			lhs = (_tmp11_ = (ValaCCodeExpression*) vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp9_ = vala_ccode_identifier_new ("self")), _tmp10_ = vala_field_get_cname (f)), (lhs == NULL) ? NULL : (lhs = (vala_ccode_node_unref (lhs), NULL)), _tmp11_);
			_tmp10_ = (g_free (_tmp10_), NULL);
			(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
		}
		if (vala_field_get_initializer (f) != NULL) {
			ValaCCodeExpression* _tmp12_;
			ValaCCodeExpression* rhs;
			ValaCCodeExpressionStatement* _tmp14_;
			ValaCCodeAssignment* _tmp13_;
			gboolean _tmp15_;
			gboolean _tmp16_;
			_tmp12_ = NULL;
			rhs = (_tmp12_ = VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_field_get_initializer (f))), (_tmp12_ == NULL) ? NULL : vala_ccode_node_ref (_tmp12_));
			_tmp14_ = NULL;
			_tmp13_ = NULL;
			vala_ccode_fragment_append (self->instance_init_fragment, (ValaCCodeNode*) (_tmp14_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp13_ = vala_ccode_assignment_new (lhs, rhs, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
			(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
			(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL));
			_tmp15_ = FALSE;
			_tmp16_ = FALSE;
			if (VALA_IS_ARRAY_TYPE (vala_field_get_field_type (f))) {
				_tmp16_ = !vala_field_get_no_array_length (f);
			} else {
				_tmp16_ = FALSE;
			}
			if (_tmp16_) {
				_tmp15_ = VALA_IS_ARRAY_CREATION_EXPRESSION (vala_field_get_initializer (f));
			} else {
				_tmp15_ = FALSE;
			}
			if (_tmp15_) {
				ValaArrayType* _tmp17_;
				ValaArrayType* array_type;
				ValaMemberAccess* this_access;
				ValaDataType* _tmp18_;
				ValaCCodeIdentifier* _tmp19_;
				ValaMemberAccess* ma;
				GeeList* sizes;
				_tmp17_ = NULL;
				array_type = (_tmp17_ = VALA_ARRAY_TYPE (vala_field_get_field_type (f)), (_tmp17_ == NULL) ? NULL : vala_code_node_ref (_tmp17_));
				this_access = vala_member_access_new_simple ("this", NULL);
				_tmp18_ = NULL;
				vala_expression_set_value_type ((ValaExpression*) this_access, _tmp18_ = vala_ccode_base_module_get_data_type_for_symbol (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) f))));
				(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_code_node_unref (_tmp18_), NULL));
				_tmp19_ = NULL;
				vala_code_node_set_ccodenode ((ValaCodeNode*) this_access, (ValaCCodeNode*) (_tmp19_ = vala_ccode_identifier_new ("self")));
				(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
				ma = vala_member_access_new ((ValaExpression*) this_access, vala_symbol_get_name ((ValaSymbol*) f), NULL);
				vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) f);
				sizes = vala_array_creation_expression_get_sizes (VALA_ARRAY_CREATION_EXPRESSION (vala_field_get_initializer (f)));
				{
					gint dim;
					dim = 1;
					for (; dim <= vala_array_type_get_rank (array_type); dim++) {
						ValaCCodeExpression* array_len_lhs;
						ValaExpression* size;
						ValaCCodeExpressionStatement* _tmp21_;
						ValaCCodeAssignment* _tmp20_;
						array_len_lhs = vala_ccode_module_get_array_length_cexpression (vala_ccode_module_get_head ((ValaCCodeModule*) self), (ValaExpression*) ma, dim);
						size = (ValaExpression*) gee_list_get (sizes, dim - 1);
						_tmp21_ = NULL;
						_tmp20_ = NULL;
						vala_ccode_fragment_append (self->instance_init_fragment, (ValaCCodeNode*) (_tmp21_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp20_ = vala_ccode_assignment_new (array_len_lhs, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) size)), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
						(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL));
						(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL));
						(array_len_lhs == NULL) ? NULL : (array_len_lhs = (vala_ccode_node_unref (array_len_lhs), NULL));
						(size == NULL) ? NULL : (size = (vala_code_node_unref (size), NULL));
					}
				}
				(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
				(this_access == NULL) ? NULL : (this_access = (vala_code_node_unref (this_access), NULL));
				(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
				(sizes == NULL) ? NULL : (sizes = (gee_collection_object_unref (sizes), NULL));
			}
			vala_ccode_base_module_append_temp_decl (self, self->instance_init_fragment, (GeeList*) self->temp_vars);
			gee_collection_clear ((GeeCollection*) self->temp_vars);
			(rhs == NULL) ? NULL : (rhs = (vala_ccode_node_unref (rhs), NULL));
		}
		_tmp22_ = FALSE;
		if (vala_ccode_base_module_requires_destroy (self, vala_field_get_field_type (f))) {
			_tmp22_ = self->instance_finalize_fragment != NULL;
		} else {
			_tmp22_ = FALSE;
		}
		if (_tmp22_) {
			ValaMemberAccess* this_access;
			ValaDataType* _tmp23_;
			ValaStruct* _tmp25_;
			ValaSymbol* _tmp24_;
			ValaStruct* field_st;
			gboolean _tmp26_;
			ValaMemberAccess* ma;
			ValaCCodeExpressionStatement* _tmp30_;
			ValaCCodeExpression* _tmp29_;
			this_access = vala_member_access_new_simple ("this", NULL);
			_tmp23_ = NULL;
			vala_expression_set_value_type ((ValaExpression*) this_access, _tmp23_ = vala_ccode_base_module_get_data_type_for_symbol (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) f))));
			(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_code_node_unref (_tmp23_), NULL));
			_tmp25_ = NULL;
			_tmp24_ = NULL;
			field_st = (_tmp25_ = (_tmp24_ = vala_symbol_get_parent_symbol ((ValaSymbol*) f), VALA_IS_STRUCT (_tmp24_) ? ((ValaStruct*) _tmp24_) : NULL), (_tmp25_ == NULL) ? NULL : vala_code_node_ref (_tmp25_));
			_tmp26_ = FALSE;
			if (field_st != NULL) {
				_tmp26_ = !vala_struct_is_simple_type (field_st);
			} else {
				_tmp26_ = FALSE;
			}
			if (_tmp26_) {
				ValaCCodeIdentifier* _tmp27_;
				_tmp27_ = NULL;
				vala_code_node_set_ccodenode ((ValaCodeNode*) this_access, (ValaCCodeNode*) (_tmp27_ = vala_ccode_identifier_new ("(*self)")));
				(_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL));
			} else {
				ValaCCodeIdentifier* _tmp28_;
				_tmp28_ = NULL;
				vala_code_node_set_ccodenode ((ValaCodeNode*) this_access, (ValaCCodeNode*) (_tmp28_ = vala_ccode_identifier_new ("self")));
				(_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_ccode_node_unref (_tmp28_), NULL));
			}
			ma = vala_member_access_new ((ValaExpression*) this_access, vala_symbol_get_name ((ValaSymbol*) f), NULL);
			vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) f);
			_tmp30_ = NULL;
			_tmp29_ = NULL;
			vala_ccode_fragment_append (self->instance_finalize_fragment, (ValaCCodeNode*) (_tmp30_ = vala_ccode_expression_statement_new (_tmp29_ = vala_ccode_base_module_get_unref_expression (self, lhs, vala_field_get_field_type (f), (ValaExpression*) ma))));
			(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL));
			(_tmp29_ == NULL) ? NULL : (_tmp29_ = (vala_ccode_node_unref (_tmp29_), NULL));
			(this_access == NULL) ? NULL : (this_access = (vala_code_node_unref (this_access), NULL));
			(field_st == NULL) ? NULL : (field_st = (vala_code_node_unref (field_st), NULL));
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
		}
	} else {
		if (vala_field_get_binding (f) == MEMBER_BINDING_CLASS) {
			if (!is_gtypeinstance) {
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) f), "class fields are not supported in compact classes");
				vala_code_node_set_error ((ValaCodeNode*) f, TRUE);
				(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
				(lhs == NULL) ? NULL : (lhs = (vala_ccode_node_unref (lhs), NULL));
				field_ctype = (g_free (field_ctype), NULL);
				return;
			}
		} else {
			ValaCCodeExpression* _tmp32_;
			char* _tmp31_;
			char* _tmp33_;
			ValaCCodeVariableDeclarator* _tmp34_;
			ValaCCodeVariableDeclarator* var_decl;
			ValaCCodeExpression* _tmp35_;
			ValaCCodeDeclaration* var_def;
			gboolean _tmp37_;
			vala_ccode_base_module_generate_field_declaration (self, f, self->source_declarations);
			if (!vala_symbol_is_internal_symbol ((ValaSymbol*) f)) {
				vala_ccode_base_module_generate_field_declaration (self, f, self->header_declarations);
			}
			vala_ccode_base_module_generate_field_declaration (self, f, self->internal_header_declarations);
			_tmp32_ = NULL;
			_tmp31_ = NULL;
			lhs = (_tmp32_ = (ValaCCodeExpression*) vala_ccode_identifier_new (_tmp31_ = vala_field_get_cname (f)), (lhs == NULL) ? NULL : (lhs = (vala_ccode_node_unref (lhs), NULL)), _tmp32_);
			_tmp31_ = (g_free (_tmp31_), NULL);
			_tmp33_ = NULL;
			_tmp34_ = NULL;
			var_decl = (_tmp34_ = vala_ccode_variable_declarator_new (_tmp33_ = vala_field_get_cname (f), NULL, NULL), _tmp33_ = (g_free (_tmp33_), NULL), _tmp34_);
			_tmp35_ = NULL;
			vala_ccode_variable_declarator_set_initializer (var_decl, _tmp35_ = vala_ccode_base_module_default_value_for_type (self, vala_field_get_field_type (f), TRUE));
			(_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_ccode_node_unref (_tmp35_), NULL));
			if (vala_field_get_initializer (f) != NULL) {
				ValaCCodeExpression* _tmp36_;
				ValaCCodeExpression* init;
				_tmp36_ = NULL;
				init = (_tmp36_ = VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_field_get_initializer (f))), (_tmp36_ == NULL) ? NULL : vala_ccode_node_ref (_tmp36_));
				if (vala_ccode_base_module_is_constant_ccode_expression (self, init)) {
					vala_ccode_variable_declarator_set_initializer (var_decl, init);
				}
				(init == NULL) ? NULL : (init = (vala_ccode_node_unref (init), NULL));
			}
			var_def = vala_ccode_declaration_new (field_ctype);
			vala_ccode_declaration_add_declarator (var_def, (ValaCCodeDeclarator*) var_decl);
			if (!vala_symbol_is_private_symbol ((ValaSymbol*) f)) {
				vala_ccode_declaration_set_modifiers (var_def, VALA_CCODE_MODIFIERS_EXTERN);
			} else {
				vala_ccode_declaration_set_modifiers (var_def, VALA_CCODE_MODIFIERS_STATIC);
			}
			vala_ccode_declaration_space_add_type_member_declaration (self->source_declarations, (ValaCCodeNode*) var_def);
			_tmp37_ = FALSE;
			if (VALA_IS_ARRAY_TYPE (vala_field_get_field_type (f))) {
				_tmp37_ = !vala_field_get_no_array_length (f);
			} else {
				_tmp37_ = FALSE;
			}
			/* add array length fields where necessary */
			if (_tmp37_) {
				ValaArrayType* _tmp38_;
				ValaArrayType* array_type;
				gboolean _tmp45_;
				_tmp38_ = NULL;
				array_type = (_tmp38_ = VALA_ARRAY_TYPE (vala_field_get_field_type (f)), (_tmp38_ == NULL) ? NULL : vala_code_node_ref (_tmp38_));
				{
					gint dim;
					dim = 1;
					for (; dim <= vala_array_type_get_rank (array_type); dim++) {
						ValaDataType* len_type;
						char* _tmp39_;
						ValaCCodeDeclaration* _tmp40_;
						ValaCCodeDeclaration* len_def;
						ValaCCodeVariableDeclarator* _tmp44_;
						ValaCCodeConstant* _tmp43_;
						char* _tmp42_;
						char* _tmp41_;
						len_type = vala_data_type_copy (self->int_type);
						_tmp39_ = NULL;
						_tmp40_ = NULL;
						len_def = (_tmp40_ = vala_ccode_declaration_new (_tmp39_ = vala_data_type_get_cname (len_type)), _tmp39_ = (g_free (_tmp39_), NULL), _tmp40_);
						_tmp44_ = NULL;
						_tmp43_ = NULL;
						_tmp42_ = NULL;
						_tmp41_ = NULL;
						vala_ccode_declaration_add_declarator (len_def, (ValaCCodeDeclarator*) (_tmp44_ = vala_ccode_variable_declarator_new (_tmp42_ = vala_ccode_module_get_array_length_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), _tmp41_ = vala_field_get_cname (f), dim), (ValaCCodeExpression*) (_tmp43_ = vala_ccode_constant_new ("0")), NULL)));
						(_tmp44_ == NULL) ? NULL : (_tmp44_ = (vala_ccode_node_unref (_tmp44_), NULL));
						(_tmp43_ == NULL) ? NULL : (_tmp43_ = (vala_ccode_node_unref (_tmp43_), NULL));
						_tmp42_ = (g_free (_tmp42_), NULL);
						_tmp41_ = (g_free (_tmp41_), NULL);
						if (!vala_symbol_is_private_symbol ((ValaSymbol*) f)) {
							vala_ccode_declaration_set_modifiers (len_def, VALA_CCODE_MODIFIERS_EXTERN);
						} else {
							vala_ccode_declaration_set_modifiers (len_def, VALA_CCODE_MODIFIERS_STATIC);
						}
						vala_ccode_declaration_space_add_type_member_declaration (self->source_declarations, (ValaCCodeNode*) len_def);
						(len_type == NULL) ? NULL : (len_type = (vala_code_node_unref (len_type), NULL));
						(len_def == NULL) ? NULL : (len_def = (vala_ccode_node_unref (len_def), NULL));
					}
				}
				_tmp45_ = FALSE;
				if (vala_array_type_get_rank (array_type) == 1) {
					_tmp45_ = vala_symbol_is_internal_symbol ((ValaSymbol*) f);
				} else {
					_tmp45_ = FALSE;
				}
				if (_tmp45_) {
					ValaDataType* len_type;
					char* _tmp46_;
					ValaCCodeDeclaration* _tmp47_;
					ValaCCodeDeclaration* _cdecl_;
					ValaCCodeVariableDeclarator* _tmp51_;
					ValaCCodeConstant* _tmp50_;
					char* _tmp49_;
					char* _tmp48_;
					len_type = vala_data_type_copy (self->int_type);
					_tmp46_ = NULL;
					_tmp47_ = NULL;
					_cdecl_ = (_tmp47_ = vala_ccode_declaration_new (_tmp46_ = vala_data_type_get_cname (len_type)), _tmp46_ = (g_free (_tmp46_), NULL), _tmp47_);
					_tmp51_ = NULL;
					_tmp50_ = NULL;
					_tmp49_ = NULL;
					_tmp48_ = NULL;
					vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp51_ = vala_ccode_variable_declarator_new (_tmp49_ = vala_ccode_module_get_array_size_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), _tmp48_ = vala_field_get_cname (f)), (ValaCCodeExpression*) (_tmp50_ = vala_ccode_constant_new ("0")), NULL)));
					(_tmp51_ == NULL) ? NULL : (_tmp51_ = (vala_ccode_node_unref (_tmp51_), NULL));
					(_tmp50_ == NULL) ? NULL : (_tmp50_ = (vala_ccode_node_unref (_tmp50_), NULL));
					_tmp49_ = (g_free (_tmp49_), NULL);
					_tmp48_ = (g_free (_tmp48_), NULL);
					vala_ccode_declaration_set_modifiers (_cdecl_, VALA_CCODE_MODIFIERS_STATIC);
					vala_ccode_declaration_space_add_type_member_declaration (self->source_declarations, (ValaCCodeNode*) _cdecl_);
					(len_type == NULL) ? NULL : (len_type = (vala_code_node_unref (len_type), NULL));
					(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
				}
				(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
			} else {
				if (VALA_IS_DELEGATE_TYPE (vala_field_get_field_type (f))) {
					ValaDelegateType* _tmp52_;
					ValaDelegateType* delegate_type;
					_tmp52_ = NULL;
					delegate_type = (_tmp52_ = VALA_DELEGATE_TYPE (vala_field_get_field_type (f)), (_tmp52_ == NULL) ? NULL : vala_code_node_ref (_tmp52_));
					if (vala_delegate_get_has_target (vala_delegate_type_get_delegate_symbol (delegate_type))) {
						ValaCCodeDeclaration* target_def;
						ValaCCodeVariableDeclarator* _tmp56_;
						ValaCCodeConstant* _tmp55_;
						char* _tmp54_;
						char* _tmp53_;
						/* create field to store delegate target*/
						target_def = vala_ccode_declaration_new ("gpointer");
						_tmp56_ = NULL;
						_tmp55_ = NULL;
						_tmp54_ = NULL;
						_tmp53_ = NULL;
						vala_ccode_declaration_add_declarator (target_def, (ValaCCodeDeclarator*) (_tmp56_ = vala_ccode_variable_declarator_new (_tmp54_ = vala_ccode_base_module_get_delegate_target_cname (self, _tmp53_ = vala_field_get_cname (f)), (ValaCCodeExpression*) (_tmp55_ = vala_ccode_constant_new ("NULL")), NULL)));
						(_tmp56_ == NULL) ? NULL : (_tmp56_ = (vala_ccode_node_unref (_tmp56_), NULL));
						(_tmp55_ == NULL) ? NULL : (_tmp55_ = (vala_ccode_node_unref (_tmp55_), NULL));
						_tmp54_ = (g_free (_tmp54_), NULL);
						_tmp53_ = (g_free (_tmp53_), NULL);
						if (!vala_symbol_is_private_symbol ((ValaSymbol*) f)) {
							vala_ccode_declaration_set_modifiers (target_def, VALA_CCODE_MODIFIERS_EXTERN);
						} else {
							vala_ccode_declaration_set_modifiers (target_def, VALA_CCODE_MODIFIERS_STATIC);
						}
						vala_ccode_declaration_space_add_type_member_declaration (self->source_declarations, (ValaCCodeNode*) target_def);
						(target_def == NULL) ? NULL : (target_def = (vala_ccode_node_unref (target_def), NULL));
					}
					(delegate_type == NULL) ? NULL : (delegate_type = (vala_code_node_unref (delegate_type), NULL));
				}
			}
			if (vala_field_get_initializer (f) != NULL) {
				ValaCCodeExpression* _tmp57_;
				ValaCCodeExpression* rhs;
				_tmp57_ = NULL;
				rhs = (_tmp57_ = VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_field_get_initializer (f))), (_tmp57_ == NULL) ? NULL : vala_ccode_node_ref (_tmp57_));
				if (!vala_ccode_base_module_is_constant_ccode_expression (self, rhs)) {
					if (VALA_IS_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) f))) {
						ValaCCodeExpressionStatement* _tmp59_;
						ValaCCodeAssignment* _tmp58_;
						gboolean _tmp60_;
						gboolean _tmp61_;
						_tmp59_ = NULL;
						_tmp58_ = NULL;
						vala_ccode_fragment_append (self->class_init_fragment, (ValaCCodeNode*) (_tmp59_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp58_ = vala_ccode_assignment_new (lhs, rhs, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
						(_tmp59_ == NULL) ? NULL : (_tmp59_ = (vala_ccode_node_unref (_tmp59_), NULL));
						(_tmp58_ == NULL) ? NULL : (_tmp58_ = (vala_ccode_node_unref (_tmp58_), NULL));
						_tmp60_ = FALSE;
						_tmp61_ = FALSE;
						if (VALA_IS_ARRAY_TYPE (vala_field_get_field_type (f))) {
							_tmp61_ = !vala_field_get_no_array_length (f);
						} else {
							_tmp61_ = FALSE;
						}
						if (_tmp61_) {
							_tmp60_ = VALA_IS_ARRAY_CREATION_EXPRESSION (vala_field_get_initializer (f));
						} else {
							_tmp60_ = FALSE;
						}
						if (_tmp60_) {
							ValaArrayType* _tmp62_;
							ValaArrayType* array_type;
							ValaMemberAccess* ma;
							GeeList* sizes;
							_tmp62_ = NULL;
							array_type = (_tmp62_ = VALA_ARRAY_TYPE (vala_field_get_field_type (f)), (_tmp62_ == NULL) ? NULL : vala_code_node_ref (_tmp62_));
							ma = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) f), NULL);
							vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) f);
							sizes = vala_array_creation_expression_get_sizes (VALA_ARRAY_CREATION_EXPRESSION (vala_field_get_initializer (f)));
							{
								gint dim;
								dim = 1;
								for (; dim <= vala_array_type_get_rank (array_type); dim++) {
									ValaCCodeExpression* array_len_lhs;
									ValaExpression* size;
									ValaCCodeExpressionStatement* _tmp64_;
									ValaCCodeAssignment* _tmp63_;
									array_len_lhs = vala_ccode_module_get_array_length_cexpression (vala_ccode_module_get_head ((ValaCCodeModule*) self), (ValaExpression*) ma, dim);
									size = (ValaExpression*) gee_list_get (sizes, dim - 1);
									_tmp64_ = NULL;
									_tmp63_ = NULL;
									vala_ccode_fragment_append (self->class_init_fragment, (ValaCCodeNode*) (_tmp64_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp63_ = vala_ccode_assignment_new (array_len_lhs, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) size)), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
									(_tmp64_ == NULL) ? NULL : (_tmp64_ = (vala_ccode_node_unref (_tmp64_), NULL));
									(_tmp63_ == NULL) ? NULL : (_tmp63_ = (vala_ccode_node_unref (_tmp63_), NULL));
									(array_len_lhs == NULL) ? NULL : (array_len_lhs = (vala_ccode_node_unref (array_len_lhs), NULL));
									(size == NULL) ? NULL : (size = (vala_code_node_unref (size), NULL));
								}
							}
							(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
							(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
							(sizes == NULL) ? NULL : (sizes = (gee_collection_object_unref (sizes), NULL));
						}
						vala_ccode_base_module_append_temp_decl (self, self->class_init_fragment, (GeeList*) self->temp_vars);
						gee_collection_clear ((GeeCollection*) self->temp_vars);
					} else {
						vala_code_node_set_error ((ValaCodeNode*) f, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) f), "Non-constant field initializers not supported in this context");
						(rhs == NULL) ? NULL : (rhs = (vala_ccode_node_unref (rhs), NULL));
						(var_decl == NULL) ? NULL : (var_decl = (vala_ccode_node_unref (var_decl), NULL));
						(var_def == NULL) ? NULL : (var_def = (vala_ccode_node_unref (var_def), NULL));
						(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
						(lhs == NULL) ? NULL : (lhs = (vala_ccode_node_unref (lhs), NULL));
						field_ctype = (g_free (field_ctype), NULL);
						return;
					}
				}
				(rhs == NULL) ? NULL : (rhs = (vala_ccode_node_unref (rhs), NULL));
			}
			(var_decl == NULL) ? NULL : (var_decl = (vala_ccode_node_unref (var_decl), NULL));
			(var_def == NULL) ? NULL : (var_def = (vala_ccode_node_unref (var_def), NULL));
		}
	}
	(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
	(lhs == NULL) ? NULL : (lhs = (vala_ccode_node_unref (lhs), NULL));
	field_ctype = (g_free (field_ctype), NULL);
}


gboolean vala_ccode_base_module_is_constant_ccode_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* cexpr) {
	ValaCCodeParenthesizedExpression* _tmp5_;
	ValaCCodeExpression* _tmp4_;
	ValaCCodeParenthesizedExpression* cparenthesized;
	gboolean _tmp6_;
	gboolean _tmp7_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (cexpr != NULL, FALSE);
	if (VALA_IS_CCODE_CONSTANT (cexpr)) {
		return TRUE;
	} else {
		if (VALA_IS_CCODE_BINARY_EXPRESSION (cexpr)) {
			ValaCCodeBinaryExpression* _tmp1_;
			ValaCCodeBinaryExpression* cbinary;
			gboolean _tmp2_;
			gboolean _tmp3_;
			_tmp1_ = NULL;
			cbinary = (_tmp1_ = VALA_CCODE_BINARY_EXPRESSION (cexpr), (_tmp1_ == NULL) ? NULL : vala_ccode_node_ref (_tmp1_));
			_tmp2_ = FALSE;
			if (vala_ccode_base_module_is_constant_ccode_expression (self, vala_ccode_binary_expression_get_left (cbinary))) {
				_tmp2_ = vala_ccode_base_module_is_constant_ccode_expression (self, vala_ccode_binary_expression_get_right (cbinary));
			} else {
				_tmp2_ = FALSE;
			}
			return (_tmp3_ = _tmp2_, (cbinary == NULL) ? NULL : (cbinary = (vala_ccode_node_unref (cbinary), NULL)), _tmp3_);
		}
	}
	_tmp5_ = NULL;
	_tmp4_ = NULL;
	cparenthesized = (_tmp5_ = (_tmp4_ = cexpr, VALA_IS_CCODE_PARENTHESIZED_EXPRESSION (_tmp4_) ? ((ValaCCodeParenthesizedExpression*) _tmp4_) : NULL), (_tmp5_ == NULL) ? NULL : vala_ccode_node_ref (_tmp5_));
	_tmp6_ = FALSE;
	if (NULL != cparenthesized) {
		_tmp6_ = vala_ccode_base_module_is_constant_ccode_expression (self, vala_ccode_parenthesized_expression_get_inner (cparenthesized));
	} else {
		_tmp6_ = FALSE;
	}
	return (_tmp7_ = _tmp6_, (cparenthesized == NULL) ? NULL : (cparenthesized = (vala_ccode_node_unref (cparenthesized), NULL)), _tmp7_);
}


/**
 * Returns whether the passed cexpr is a pure expression, i.e. an
 * expression without side-effects.
 */
gboolean vala_ccode_base_module_is_pure_ccode_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* cexpr) {
	gboolean _tmp0_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (cexpr != NULL, FALSE);
	_tmp0_ = FALSE;
	if (VALA_IS_CCODE_CONSTANT (cexpr)) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = VALA_IS_CCODE_IDENTIFIER (cexpr);
	}
	if (_tmp0_) {
		return TRUE;
	} else {
		if (VALA_IS_CCODE_BINARY_EXPRESSION (cexpr)) {
			ValaCCodeBinaryExpression* _tmp2_;
			ValaCCodeBinaryExpression* cbinary;
			gboolean _tmp3_;
			gboolean _tmp4_;
			_tmp2_ = NULL;
			cbinary = (_tmp2_ = VALA_CCODE_BINARY_EXPRESSION (cexpr), (_tmp2_ == NULL) ? NULL : vala_ccode_node_ref (_tmp2_));
			_tmp3_ = FALSE;
			if (vala_ccode_base_module_is_pure_ccode_expression (self, vala_ccode_binary_expression_get_left (cbinary))) {
				_tmp3_ = vala_ccode_base_module_is_constant_ccode_expression (self, vala_ccode_binary_expression_get_right (cbinary));
			} else {
				_tmp3_ = FALSE;
			}
			return (_tmp4_ = _tmp3_, (cbinary == NULL) ? NULL : (cbinary = (vala_ccode_node_unref (cbinary), NULL)), _tmp4_);
		} else {
			if (VALA_IS_CCODE_UNARY_EXPRESSION (cexpr)) {
				ValaCCodeUnaryExpression* _tmp5_;
				ValaCCodeUnaryExpression* cunary;
				_tmp5_ = NULL;
				cunary = (_tmp5_ = VALA_CCODE_UNARY_EXPRESSION (cexpr), (_tmp5_ == NULL) ? NULL : vala_ccode_node_ref (_tmp5_));
				switch (vala_ccode_unary_expression_get_operator (cunary)) {
					case VALA_CCODE_UNARY_OPERATOR_PREFIX_INCREMENT:
					case VALA_CCODE_UNARY_OPERATOR_PREFIX_DECREMENT:
					case VALA_CCODE_UNARY_OPERATOR_POSTFIX_INCREMENT:
					case VALA_CCODE_UNARY_OPERATOR_POSTFIX_DECREMENT:
					{
						gboolean _tmp6_;
						return (_tmp6_ = FALSE, (cunary == NULL) ? NULL : (cunary = (vala_ccode_node_unref (cunary), NULL)), _tmp6_);
					}
					default:
					{
						gboolean _tmp7_;
						return (_tmp7_ = vala_ccode_base_module_is_pure_ccode_expression (self, vala_ccode_unary_expression_get_inner (cunary)), (cunary == NULL) ? NULL : (cunary = (vala_ccode_node_unref (cunary), NULL)), _tmp7_);
					}
				}
				(cunary == NULL) ? NULL : (cunary = (vala_ccode_node_unref (cunary), NULL));
			} else {
				if (VALA_IS_CCODE_MEMBER_ACCESS (cexpr)) {
					ValaCCodeMemberAccess* _tmp8_;
					ValaCCodeMemberAccess* cma;
					gboolean _tmp9_;
					_tmp8_ = NULL;
					cma = (_tmp8_ = VALA_CCODE_MEMBER_ACCESS (cexpr), (_tmp8_ == NULL) ? NULL : vala_ccode_node_ref (_tmp8_));
					return (_tmp9_ = vala_ccode_base_module_is_pure_ccode_expression (self, vala_ccode_member_access_get_inner (cma)), (cma == NULL) ? NULL : (cma = (vala_ccode_node_unref (cma), NULL)), _tmp9_);
				} else {
					if (VALA_IS_CCODE_ELEMENT_ACCESS (cexpr)) {
						ValaCCodeElementAccess* _tmp10_;
						ValaCCodeElementAccess* cea;
						gboolean _tmp11_;
						gboolean _tmp12_;
						_tmp10_ = NULL;
						cea = (_tmp10_ = VALA_CCODE_ELEMENT_ACCESS (cexpr), (_tmp10_ == NULL) ? NULL : vala_ccode_node_ref (_tmp10_));
						_tmp11_ = FALSE;
						if (vala_ccode_base_module_is_pure_ccode_expression (self, vala_ccode_element_access_get_container (cea))) {
							_tmp11_ = vala_ccode_base_module_is_pure_ccode_expression (self, vala_ccode_element_access_get_index (cea));
						} else {
							_tmp11_ = FALSE;
						}
						return (_tmp12_ = _tmp11_, (cea == NULL) ? NULL : (cea = (vala_ccode_node_unref (cea), NULL)), _tmp12_);
					} else {
						if (VALA_IS_CCODE_CAST_EXPRESSION (cexpr)) {
							ValaCCodeCastExpression* _tmp13_;
							ValaCCodeCastExpression* ccast;
							gboolean _tmp14_;
							_tmp13_ = NULL;
							ccast = (_tmp13_ = VALA_CCODE_CAST_EXPRESSION (cexpr), (_tmp13_ == NULL) ? NULL : vala_ccode_node_ref (_tmp13_));
							return (_tmp14_ = vala_ccode_base_module_is_pure_ccode_expression (self, vala_ccode_cast_expression_get_inner (ccast)), (ccast == NULL) ? NULL : (ccast = (vala_ccode_node_unref (ccast), NULL)), _tmp14_);
						} else {
							if (VALA_IS_CCODE_PARENTHESIZED_EXPRESSION (cexpr)) {
								ValaCCodeParenthesizedExpression* _tmp15_;
								ValaCCodeParenthesizedExpression* cparenthesized;
								gboolean _tmp16_;
								_tmp15_ = NULL;
								cparenthesized = (_tmp15_ = VALA_CCODE_PARENTHESIZED_EXPRESSION (cexpr), (_tmp15_ == NULL) ? NULL : vala_ccode_node_ref (_tmp15_));
								return (_tmp16_ = vala_ccode_base_module_is_pure_ccode_expression (self, vala_ccode_parenthesized_expression_get_inner (cparenthesized)), (cparenthesized == NULL) ? NULL : (cparenthesized = (vala_ccode_node_unref (cparenthesized), NULL)), _tmp16_);
							}
						}
					}
				}
			}
		}
	}
	return FALSE;
}


static void vala_ccode_base_module_real_visit_formal_parameter (ValaCCodeModule* base, ValaFormalParameter* p) {
	ValaCCodeBaseModule * self;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (p != NULL);
	vala_ccode_base_module_check_type (self, vala_formal_parameter_get_parameter_type (p));
	vala_code_node_accept_children ((ValaCodeNode*) p, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
}


static void vala_ccode_base_module_real_visit_property (ValaCCodeModule* base, ValaProperty* prop) {
	ValaCCodeBaseModule * self;
	gint old_next_temp_var_id;
	GeeArrayList* _tmp0_;
	GeeArrayList* old_temp_vars;
	GeeArrayList* _tmp1_;
	GeeArrayList* old_temp_ref_vars;
	GeeMap* _tmp2_;
	GeeMap* old_variable_name_map;
	GeeArrayList* _tmp3_;
	GeeArrayList* _tmp4_;
	GeeMap* _tmp5_;
	GeeArrayList* _tmp7_;
	GeeArrayList* _tmp6_;
	GeeArrayList* _tmp9_;
	GeeArrayList* _tmp8_;
	GeeMap* _tmp11_;
	GeeMap* _tmp10_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (prop != NULL);
	vala_ccode_base_module_check_type (self, vala_property_get_property_type (prop));
	old_next_temp_var_id = self->next_temp_var_id;
	_tmp0_ = NULL;
	old_temp_vars = (_tmp0_ = self->temp_vars, (_tmp0_ == NULL) ? NULL : gee_collection_object_ref (_tmp0_));
	_tmp1_ = NULL;
	old_temp_ref_vars = (_tmp1_ = self->temp_ref_vars, (_tmp1_ == NULL) ? NULL : gee_collection_object_ref (_tmp1_));
	_tmp2_ = NULL;
	old_variable_name_map = (_tmp2_ = self->variable_name_map, (_tmp2_ == NULL) ? NULL : gee_collection_object_ref (_tmp2_));
	self->next_temp_var_id = 0;
	_tmp3_ = NULL;
	self->temp_vars = (_tmp3_ = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal), (self->temp_vars == NULL) ? NULL : (self->temp_vars = (gee_collection_object_unref (self->temp_vars), NULL)), _tmp3_);
	_tmp4_ = NULL;
	self->temp_ref_vars = (_tmp4_ = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal), (self->temp_ref_vars == NULL) ? NULL : (self->temp_ref_vars = (gee_collection_object_unref (self->temp_ref_vars), NULL)), _tmp4_);
	_tmp5_ = NULL;
	self->variable_name_map = (_tmp5_ = (GeeMap*) gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal), (self->variable_name_map == NULL) ? NULL : (self->variable_name_map = (gee_collection_object_unref (self->variable_name_map), NULL)), _tmp5_);
	vala_code_node_accept_children ((ValaCodeNode*) prop, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	self->next_temp_var_id = old_next_temp_var_id;
	_tmp7_ = NULL;
	_tmp6_ = NULL;
	self->temp_vars = (_tmp7_ = (_tmp6_ = old_temp_vars, (_tmp6_ == NULL) ? NULL : gee_collection_object_ref (_tmp6_)), (self->temp_vars == NULL) ? NULL : (self->temp_vars = (gee_collection_object_unref (self->temp_vars), NULL)), _tmp7_);
	_tmp9_ = NULL;
	_tmp8_ = NULL;
	self->temp_ref_vars = (_tmp9_ = (_tmp8_ = old_temp_ref_vars, (_tmp8_ == NULL) ? NULL : gee_collection_object_ref (_tmp8_)), (self->temp_ref_vars == NULL) ? NULL : (self->temp_ref_vars = (gee_collection_object_unref (self->temp_ref_vars), NULL)), _tmp9_);
	_tmp11_ = NULL;
	_tmp10_ = NULL;
	self->variable_name_map = (_tmp11_ = (_tmp10_ = old_variable_name_map, (_tmp10_ == NULL) ? NULL : gee_collection_object_ref (_tmp10_)), (self->variable_name_map == NULL) ? NULL : (self->variable_name_map = (gee_collection_object_unref (self->variable_name_map), NULL)), _tmp11_);
	(old_temp_vars == NULL) ? NULL : (old_temp_vars = (gee_collection_object_unref (old_temp_vars), NULL));
	(old_temp_ref_vars == NULL) ? NULL : (old_temp_ref_vars = (gee_collection_object_unref (old_temp_ref_vars), NULL));
	(old_variable_name_map == NULL) ? NULL : (old_variable_name_map = (gee_collection_object_unref (old_variable_name_map), NULL));
}


void vala_ccode_base_module_generate_type_declaration (ValaCCodeBaseModule* self, ValaDataType* type, ValaCCodeDeclarationSpace* decl_space) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	g_return_if_fail (decl_space != NULL);
	if (VALA_IS_OBJECT_TYPE (type)) {
		ValaObjectType* _tmp0_;
		ValaObjectType* object_type;
		_tmp0_ = NULL;
		object_type = (_tmp0_ = VALA_OBJECT_TYPE (type), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
		if (VALA_IS_CLASS (vala_object_type_get_type_symbol (object_type))) {
			vala_ccode_base_module_generate_class_declaration (self, VALA_CLASS (vala_object_type_get_type_symbol (object_type)), decl_space);
		} else {
			if (VALA_IS_INTERFACE (vala_object_type_get_type_symbol (object_type))) {
				vala_ccode_base_module_generate_interface_declaration (self, VALA_INTERFACE (vala_object_type_get_type_symbol (object_type)), decl_space);
			}
		}
		(object_type == NULL) ? NULL : (object_type = (vala_code_node_unref (object_type), NULL));
	} else {
		if (VALA_IS_DELEGATE_TYPE (type)) {
			ValaDelegateType* _tmp1_;
			ValaDelegateType* deleg_type;
			ValaDelegate* _tmp2_;
			ValaDelegate* d;
			_tmp1_ = NULL;
			deleg_type = (_tmp1_ = VALA_DELEGATE_TYPE (type), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
			_tmp2_ = NULL;
			d = (_tmp2_ = vala_delegate_type_get_delegate_symbol (deleg_type), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
			vala_ccode_base_module_generate_delegate_declaration (self, d, decl_space);
			(deleg_type == NULL) ? NULL : (deleg_type = (vala_code_node_unref (deleg_type), NULL));
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
		} else {
			if (VALA_IS_ENUM (vala_data_type_get_data_type (type))) {
				ValaEnum* _tmp3_;
				ValaEnum* en;
				_tmp3_ = NULL;
				en = (_tmp3_ = VALA_ENUM (vala_data_type_get_data_type (type)), (_tmp3_ == NULL) ? NULL : vala_code_node_ref (_tmp3_));
				vala_ccode_base_module_generate_enum_declaration (self, en, decl_space);
				(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
			} else {
				if (VALA_IS_VALUE_TYPE (type)) {
					ValaValueType* _tmp4_;
					ValaValueType* value_type;
					_tmp4_ = NULL;
					value_type = (_tmp4_ = VALA_VALUE_TYPE (type), (_tmp4_ == NULL) ? NULL : vala_code_node_ref (_tmp4_));
					vala_ccode_base_module_generate_struct_declaration (self, VALA_STRUCT (vala_value_type_get_type_symbol (value_type)), decl_space);
					(value_type == NULL) ? NULL : (value_type = (vala_code_node_unref (value_type), NULL));
				} else {
					if (VALA_IS_ARRAY_TYPE (type)) {
						ValaArrayType* _tmp5_;
						ValaArrayType* array_type;
						_tmp5_ = NULL;
						array_type = (_tmp5_ = VALA_ARRAY_TYPE (type), (_tmp5_ == NULL) ? NULL : vala_code_node_ref (_tmp5_));
						vala_ccode_base_module_generate_type_declaration (self, vala_array_type_get_element_type (array_type), decl_space);
						(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
					} else {
						if (VALA_IS_ERROR_TYPE (type)) {
							ValaErrorType* _tmp6_;
							ValaErrorType* error_type;
							_tmp6_ = NULL;
							error_type = (_tmp6_ = VALA_ERROR_TYPE (type), (_tmp6_ == NULL) ? NULL : vala_code_node_ref (_tmp6_));
							if (vala_error_type_get_error_domain (error_type) != NULL) {
								vala_ccode_base_module_generate_error_domain_declaration (self, vala_error_type_get_error_domain (error_type), decl_space);
							}
							(error_type == NULL) ? NULL : (error_type = (vala_code_node_unref (error_type), NULL));
						} else {
							if (VALA_IS_POINTER_TYPE (type)) {
								ValaPointerType* _tmp7_;
								ValaPointerType* pointer_type;
								_tmp7_ = NULL;
								pointer_type = (_tmp7_ = VALA_POINTER_TYPE (type), (_tmp7_ == NULL) ? NULL : vala_code_node_ref (_tmp7_));
								vala_ccode_base_module_generate_type_declaration (self, vala_pointer_type_get_base_type (pointer_type), decl_space);
								(pointer_type == NULL) ? NULL : (pointer_type = (vala_code_node_unref (pointer_type), NULL));
							}
						}
					}
				}
			}
		}
	}
	{
		GeeList* _tmp8_;
		GeeIterator* _tmp9_;
		GeeIterator* _type_arg_it;
		_tmp8_ = NULL;
		_tmp9_ = NULL;
		_type_arg_it = (_tmp9_ = gee_iterable_iterator ((GeeIterable*) (_tmp8_ = vala_data_type_get_type_arguments (type))), (_tmp8_ == NULL) ? NULL : (_tmp8_ = (gee_collection_object_unref (_tmp8_), NULL)), _tmp9_);
		while (gee_iterator_next (_type_arg_it)) {
			ValaDataType* type_arg;
			type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
			vala_ccode_base_module_generate_type_declaration (self, type_arg, decl_space);
			(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
		}
		(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
	}
}


static void vala_ccode_base_module_real_generate_class_struct_declaration (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	g_return_if_fail (decl_space != NULL);
}


void vala_ccode_base_module_generate_class_struct_declaration (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space) {
	VALA_CCODE_BASE_MODULE_GET_CLASS (self)->generate_class_struct_declaration (self, cl, decl_space);
}


static void vala_ccode_base_module_real_generate_struct_declaration (ValaCCodeBaseModule* self, ValaStruct* st, ValaCCodeDeclarationSpace* decl_space) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	g_return_if_fail (decl_space != NULL);
}


void vala_ccode_base_module_generate_struct_declaration (ValaCCodeBaseModule* self, ValaStruct* st, ValaCCodeDeclarationSpace* decl_space) {
	VALA_CCODE_BASE_MODULE_GET_CLASS (self)->generate_struct_declaration (self, st, decl_space);
}


static void vala_ccode_base_module_real_generate_delegate_declaration (ValaCCodeBaseModule* self, ValaDelegate* d, ValaCCodeDeclarationSpace* decl_space) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (d != NULL);
	g_return_if_fail (decl_space != NULL);
}


void vala_ccode_base_module_generate_delegate_declaration (ValaCCodeBaseModule* self, ValaDelegate* d, ValaCCodeDeclarationSpace* decl_space) {
	VALA_CCODE_BASE_MODULE_GET_CLASS (self)->generate_delegate_declaration (self, d, decl_space);
}


static void vala_ccode_base_module_real_generate_cparameters (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, ValaCCodeFunction* func, ValaCCodeFunctionDeclarator* vdeclarator, GeeMap* carg_map, ValaCCodeFunctionCall* vcall, gint direction) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	g_return_if_fail (decl_space != NULL);
	g_return_if_fail (cparam_map != NULL);
	g_return_if_fail (func != NULL);
}


void vala_ccode_base_module_generate_cparameters (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, ValaCCodeFunction* func, ValaCCodeFunctionDeclarator* vdeclarator, GeeMap* carg_map, ValaCCodeFunctionCall* vcall, gint direction) {
	VALA_CCODE_BASE_MODULE_GET_CLASS (self)->generate_cparameters (self, m, decl_space, cparam_map, func, vdeclarator, carg_map, vcall, direction);
}


void vala_ccode_base_module_generate_property_accessor_declaration (ValaCCodeBaseModule* self, ValaPropertyAccessor* acc, ValaCCodeDeclarationSpace* decl_space) {
	char* _tmp0_;
	gboolean _tmp1_;
	ValaProperty* _tmp2_;
	ValaProperty* prop;
	gboolean returns_real_struct;
	ValaObjectTypeSymbol* _tmp3_;
	ValaObjectTypeSymbol* t;
	ValaObjectType* this_type;
	char* _tmp4_;
	ValaCCodeFormalParameter* _tmp5_;
	ValaCCodeFormalParameter* cselfparam;
	ValaCCodeFormalParameter* cvalueparam;
	gboolean _tmp11_;
	gboolean _tmp17_;
	gboolean _tmp18_;
	gboolean _tmp19_;
	gboolean _tmp20_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (acc != NULL);
	g_return_if_fail (decl_space != NULL);
	_tmp0_ = NULL;
	if ((_tmp1_ = vala_ccode_declaration_space_add_symbol_declaration (decl_space, (ValaSymbol*) vala_property_accessor_get_prop (acc), _tmp0_ = vala_property_accessor_get_cname (acc)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_)) {
		return;
	}
	_tmp2_ = NULL;
	prop = (_tmp2_ = VALA_PROPERTY (vala_property_accessor_get_prop (acc)), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
	returns_real_struct = vala_data_type_is_real_struct_type (vala_property_get_property_type (prop));
	_tmp3_ = NULL;
	t = (_tmp3_ = VALA_OBJECT_TYPE_SYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) prop)), (_tmp3_ == NULL) ? NULL : vala_code_node_ref (_tmp3_));
	this_type = vala_object_type_new (t);
	vala_ccode_base_module_generate_type_declaration (self, (ValaDataType*) this_type, decl_space);
	_tmp4_ = NULL;
	_tmp5_ = NULL;
	cselfparam = (_tmp5_ = vala_ccode_formal_parameter_new ("self", _tmp4_ = vala_data_type_get_cname ((ValaDataType*) this_type)), _tmp4_ = (g_free (_tmp4_), NULL), _tmp5_);
	cvalueparam = NULL;
	if (returns_real_struct) {
		ValaCCodeFormalParameter* _tmp8_;
		char* _tmp7_;
		char* _tmp6_;
		_tmp8_ = NULL;
		_tmp7_ = NULL;
		_tmp6_ = NULL;
		cvalueparam = (_tmp8_ = vala_ccode_formal_parameter_new ("value", _tmp7_ = g_strconcat (_tmp6_ = vala_data_type_get_cname (vala_property_accessor_get_value_type (acc)), "*", NULL)), (cvalueparam == NULL) ? NULL : (cvalueparam = (vala_ccode_node_unref (cvalueparam), NULL)), _tmp8_);
		_tmp7_ = (g_free (_tmp7_), NULL);
		_tmp6_ = (g_free (_tmp6_), NULL);
	} else {
		ValaCCodeFormalParameter* _tmp10_;
		char* _tmp9_;
		_tmp10_ = NULL;
		_tmp9_ = NULL;
		cvalueparam = (_tmp10_ = vala_ccode_formal_parameter_new ("value", _tmp9_ = vala_data_type_get_cname (vala_property_accessor_get_value_type (acc))), (cvalueparam == NULL) ? NULL : (cvalueparam = (vala_ccode_node_unref (cvalueparam), NULL)), _tmp10_);
		_tmp9_ = (g_free (_tmp9_), NULL);
	}
	vala_ccode_base_module_generate_type_declaration (self, vala_property_accessor_get_value_type (acc), decl_space);
	_tmp11_ = FALSE;
	if (vala_property_accessor_get_readable (acc)) {
		_tmp11_ = !returns_real_struct;
	} else {
		_tmp11_ = FALSE;
	}
	if (_tmp11_) {
		ValaCCodeFunction* _tmp14_;
		char* _tmp13_;
		char* _tmp12_;
		_tmp14_ = NULL;
		_tmp13_ = NULL;
		_tmp12_ = NULL;
		self->function = (_tmp14_ = vala_ccode_function_new (_tmp12_ = vala_property_accessor_get_cname (acc), _tmp13_ = vala_data_type_get_cname (vala_property_accessor_get_value_type (acc))), (self->function == NULL) ? NULL : (self->function = (vala_ccode_node_unref (self->function), NULL)), _tmp14_);
		_tmp13_ = (g_free (_tmp13_), NULL);
		_tmp12_ = (g_free (_tmp12_), NULL);
	} else {
		ValaCCodeFunction* _tmp16_;
		char* _tmp15_;
		_tmp16_ = NULL;
		_tmp15_ = NULL;
		self->function = (_tmp16_ = vala_ccode_function_new (_tmp15_ = vala_property_accessor_get_cname (acc), "void"), (self->function == NULL) ? NULL : (self->function = (vala_ccode_node_unref (self->function), NULL)), _tmp16_);
		_tmp15_ = (g_free (_tmp15_), NULL);
	}
	vala_ccode_function_add_parameter (self->function, cselfparam);
	_tmp17_ = FALSE;
	_tmp18_ = FALSE;
	if (vala_property_accessor_get_writable (acc)) {
		_tmp18_ = TRUE;
	} else {
		_tmp18_ = vala_property_accessor_get_construction (acc);
	}
	if (_tmp18_) {
		_tmp17_ = TRUE;
	} else {
		_tmp17_ = returns_real_struct;
	}
	if (_tmp17_) {
		vala_ccode_function_add_parameter (self->function, cvalueparam);
	}
	_tmp19_ = FALSE;
	_tmp20_ = FALSE;
	if (vala_symbol_is_private_symbol ((ValaSymbol*) prop)) {
		_tmp20_ = TRUE;
	} else {
		gboolean _tmp21_;
		_tmp21_ = FALSE;
		if (!vala_property_accessor_get_readable (acc)) {
			_tmp21_ = !vala_property_accessor_get_writable (acc);
		} else {
			_tmp21_ = FALSE;
		}
		_tmp20_ = _tmp21_;
	}
	if (_tmp20_) {
		_tmp19_ = TRUE;
	} else {
		_tmp19_ = vala_property_accessor_get_access (acc) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE;
	}
	if (_tmp19_) {
		vala_ccode_function_set_modifiers (self->function, vala_ccode_function_get_modifiers (self->function) | VALA_CCODE_MODIFIERS_STATIC);
	}
	vala_ccode_declaration_space_add_type_member_declaration (decl_space, (ValaCCodeNode*) self->function);
	(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
	(t == NULL) ? NULL : (t = (vala_code_node_unref (t), NULL));
	(this_type == NULL) ? NULL : (this_type = (vala_code_node_unref (this_type), NULL));
	(cselfparam == NULL) ? NULL : (cselfparam = (vala_ccode_node_unref (cselfparam), NULL));
	(cvalueparam == NULL) ? NULL : (cvalueparam = (vala_ccode_node_unref (cvalueparam), NULL));
}


static void vala_ccode_base_module_real_visit_property_accessor (ValaCCodeModule* base, ValaPropertyAccessor* acc) {
	ValaCCodeBaseModule * self;
	ValaPropertyAccessor* _tmp0_;
	ValaPropertyAccessor* old_property_accessor;
	gboolean old_method_inner_error;
	ValaPropertyAccessor* _tmp2_;
	ValaPropertyAccessor* _tmp1_;
	ValaProperty* _tmp3_;
	ValaProperty* prop;
	gboolean returns_real_struct;
	ValaDataType* _tmp4_;
	ValaDataType* old_return_type;
	gboolean _tmp5_;
	ValaObjectTypeSymbol* _tmp9_;
	ValaObjectTypeSymbol* t;
	gboolean _tmp10_;
	gboolean _tmp11_;
	gboolean _tmp12_;
	ValaObjectType* this_type;
	char* _tmp17_;
	ValaCCodeFormalParameter* _tmp18_;
	ValaCCodeFormalParameter* cselfparam;
	ValaCCodeFormalParameter* cvalueparam;
	gboolean _tmp24_;
	ValaPropertyAccessor* _tmp106_;
	ValaPropertyAccessor* _tmp105_;
	ValaDataType* _tmp108_;
	ValaDataType* _tmp107_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (acc != NULL);
	_tmp0_ = NULL;
	old_property_accessor = (_tmp0_ = self->current_property_accessor, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	old_method_inner_error = self->current_method_inner_error;
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->current_property_accessor = (_tmp2_ = (_tmp1_ = acc, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->current_property_accessor == NULL) ? NULL : (self->current_property_accessor = (vala_code_node_unref (self->current_property_accessor), NULL)), _tmp2_);
	self->current_method_inner_error = FALSE;
	_tmp3_ = NULL;
	prop = (_tmp3_ = VALA_PROPERTY (vala_property_accessor_get_prop (acc)), (_tmp3_ == NULL) ? NULL : vala_code_node_ref (_tmp3_));
	returns_real_struct = vala_data_type_is_real_struct_type (vala_property_get_property_type (prop));
	_tmp4_ = NULL;
	old_return_type = (_tmp4_ = self->current_return_type, (_tmp4_ == NULL) ? NULL : vala_code_node_ref (_tmp4_));
	_tmp5_ = FALSE;
	if (vala_property_accessor_get_readable (acc)) {
		_tmp5_ = !returns_real_struct;
	} else {
		_tmp5_ = FALSE;
	}
	if (_tmp5_) {
		ValaDataType* _tmp7_;
		ValaDataType* _tmp6_;
		_tmp7_ = NULL;
		_tmp6_ = NULL;
		self->current_return_type = (_tmp7_ = (_tmp6_ = vala_property_accessor_get_value_type (acc), (_tmp6_ == NULL) ? NULL : vala_code_node_ref (_tmp6_)), (self->current_return_type == NULL) ? NULL : (self->current_return_type = (vala_code_node_unref (self->current_return_type), NULL)), _tmp7_);
	} else {
		ValaDataType* _tmp8_;
		_tmp8_ = NULL;
		self->current_return_type = (_tmp8_ = (ValaDataType*) vala_void_type_new (NULL), (self->current_return_type == NULL) ? NULL : (self->current_return_type = (vala_code_node_unref (self->current_return_type), NULL)), _tmp8_);
	}
	vala_code_node_accept_children ((ValaCodeNode*) acc, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	_tmp9_ = NULL;
	t = (_tmp9_ = VALA_OBJECT_TYPE_SYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) prop)), (_tmp9_ == NULL) ? NULL : vala_code_node_ref (_tmp9_));
	_tmp10_ = FALSE;
	if (vala_property_accessor_get_construction (acc)) {
		_tmp10_ = !vala_typesymbol_is_subtype_of ((ValaTypeSymbol*) t, self->gobject_type);
	} else {
		_tmp10_ = FALSE;
	}
	if (_tmp10_) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) acc), "construct properties require GLib.Object");
		vala_code_node_set_error ((ValaCodeNode*) acc, TRUE);
		(old_property_accessor == NULL) ? NULL : (old_property_accessor = (vala_code_node_unref (old_property_accessor), NULL));
		(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
		(old_return_type == NULL) ? NULL : (old_return_type = (vala_code_node_unref (old_return_type), NULL));
		(t == NULL) ? NULL : (t = (vala_code_node_unref (t), NULL));
		return;
	}
	_tmp11_ = FALSE;
	_tmp12_ = FALSE;
	if (vala_property_get_is_abstract (prop)) {
		_tmp12_ = TRUE;
	} else {
		_tmp12_ = vala_property_get_is_virtual (prop);
	}
	if (_tmp12_) {
		_tmp11_ = TRUE;
	} else {
		gboolean _tmp13_;
		_tmp13_ = FALSE;
		if (vala_property_get_base_property (prop) == NULL) {
			_tmp13_ = vala_property_get_base_interface_property (prop) == NULL;
		} else {
			_tmp13_ = FALSE;
		}
		_tmp11_ = _tmp13_;
	}
	/* do not declare overriding properties and interface implementations*/
	if (_tmp11_) {
		gboolean _tmp14_;
		vala_ccode_base_module_generate_property_accessor_declaration (self, acc, self->source_declarations);
		_tmp14_ = FALSE;
		if (vala_property_accessor_get_readable (acc)) {
			_tmp14_ = TRUE;
		} else {
			_tmp14_ = vala_property_accessor_get_writable (acc);
		}
		/* do not declare construct-only properties in header files*/
		if (_tmp14_) {
			gboolean _tmp15_;
			_tmp15_ = FALSE;
			if (!vala_symbol_is_internal_symbol ((ValaSymbol*) prop)) {
				gboolean _tmp16_;
				_tmp16_ = FALSE;
				if (vala_property_accessor_get_access (acc) == VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
					_tmp16_ = TRUE;
				} else {
					_tmp16_ = vala_property_accessor_get_access (acc) == VALA_SYMBOL_ACCESSIBILITY_PROTECTED;
				}
				_tmp15_ = _tmp16_;
			} else {
				_tmp15_ = FALSE;
			}
			if (_tmp15_) {
				vala_ccode_base_module_generate_property_accessor_declaration (self, acc, self->header_declarations);
			}
			vala_ccode_base_module_generate_property_accessor_declaration (self, acc, self->internal_header_declarations);
		}
	}
	this_type = vala_object_type_new (t);
	_tmp17_ = NULL;
	_tmp18_ = NULL;
	cselfparam = (_tmp18_ = vala_ccode_formal_parameter_new ("self", _tmp17_ = vala_data_type_get_cname ((ValaDataType*) this_type)), _tmp17_ = (g_free (_tmp17_), NULL), _tmp18_);
	cvalueparam = NULL;
	if (returns_real_struct) {
		ValaCCodeFormalParameter* _tmp21_;
		char* _tmp20_;
		char* _tmp19_;
		_tmp21_ = NULL;
		_tmp20_ = NULL;
		_tmp19_ = NULL;
		cvalueparam = (_tmp21_ = vala_ccode_formal_parameter_new ("value", _tmp20_ = g_strconcat (_tmp19_ = vala_data_type_get_cname (vala_property_accessor_get_value_type (acc)), "*", NULL)), (cvalueparam == NULL) ? NULL : (cvalueparam = (vala_ccode_node_unref (cvalueparam), NULL)), _tmp21_);
		_tmp20_ = (g_free (_tmp20_), NULL);
		_tmp19_ = (g_free (_tmp19_), NULL);
	} else {
		ValaCCodeFormalParameter* _tmp23_;
		char* _tmp22_;
		_tmp23_ = NULL;
		_tmp22_ = NULL;
		cvalueparam = (_tmp23_ = vala_ccode_formal_parameter_new ("value", _tmp22_ = vala_data_type_get_cname (vala_property_accessor_get_value_type (acc))), (cvalueparam == NULL) ? NULL : (cvalueparam = (vala_ccode_node_unref (cvalueparam), NULL)), _tmp23_);
		_tmp22_ = (g_free (_tmp22_), NULL);
	}
	_tmp24_ = FALSE;
	if (vala_property_get_is_abstract (prop)) {
		_tmp24_ = TRUE;
	} else {
		_tmp24_ = vala_property_get_is_virtual (prop);
	}
	if (_tmp24_) {
		gboolean _tmp30_;
		gboolean _tmp31_;
		gboolean _tmp32_;
		gboolean _tmp33_;
		ValaCCodeBlock* block;
		ValaCCodeFunctionCall* vcast;
		ValaCCodeIdentifier* _tmp45_;
		if (vala_property_accessor_get_readable (acc)) {
			ValaCCodeFunction* _tmp27_;
			char* _tmp26_;
			char* _tmp25_;
			_tmp27_ = NULL;
			_tmp26_ = NULL;
			_tmp25_ = NULL;
			self->function = (_tmp27_ = vala_ccode_function_new (_tmp25_ = vala_property_accessor_get_cname (acc), _tmp26_ = vala_data_type_get_cname (self->current_return_type)), (self->function == NULL) ? NULL : (self->function = (vala_ccode_node_unref (self->function), NULL)), _tmp27_);
			_tmp26_ = (g_free (_tmp26_), NULL);
			_tmp25_ = (g_free (_tmp25_), NULL);
		} else {
			ValaCCodeFunction* _tmp29_;
			char* _tmp28_;
			_tmp29_ = NULL;
			_tmp28_ = NULL;
			self->function = (_tmp29_ = vala_ccode_function_new (_tmp28_ = vala_property_accessor_get_cname (acc), "void"), (self->function == NULL) ? NULL : (self->function = (vala_ccode_node_unref (self->function), NULL)), _tmp29_);
			_tmp28_ = (g_free (_tmp28_), NULL);
		}
		vala_ccode_function_add_parameter (self->function, cselfparam);
		_tmp30_ = FALSE;
		_tmp31_ = FALSE;
		if (vala_property_accessor_get_writable (acc)) {
			_tmp31_ = TRUE;
		} else {
			_tmp31_ = vala_property_accessor_get_construction (acc);
		}
		if (_tmp31_) {
			_tmp30_ = TRUE;
		} else {
			_tmp30_ = returns_real_struct;
		}
		if (_tmp30_) {
			vala_ccode_function_add_parameter (self->function, cvalueparam);
		}
		_tmp32_ = FALSE;
		_tmp33_ = FALSE;
		if (vala_symbol_is_private_symbol ((ValaSymbol*) prop)) {
			_tmp33_ = TRUE;
		} else {
			gboolean _tmp34_;
			_tmp34_ = FALSE;
			if (vala_property_accessor_get_readable (acc)) {
				_tmp34_ = TRUE;
			} else {
				_tmp34_ = vala_property_accessor_get_writable (acc);
			}
			_tmp33_ = !_tmp34_;
		}
		if (_tmp33_) {
			_tmp32_ = TRUE;
		} else {
			_tmp32_ = vala_property_accessor_get_access (acc) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE;
		}
		if (_tmp32_) {
			/* accessor function should be private if the property is an internal symbol or it's a construct-only setter*/
			vala_ccode_function_set_modifiers (self->function, vala_ccode_function_get_modifiers (self->function) | VALA_CCODE_MODIFIERS_STATIC);
		}
		block = vala_ccode_block_new ();
		vala_ccode_function_set_block (self->function, block);
		vcast = NULL;
		if (VALA_IS_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) prop))) {
			ValaInterface* _tmp35_;
			ValaInterface* iface;
			ValaCCodeFunctionCall* _tmp39_;
			ValaCCodeIdentifier* _tmp38_;
			char* _tmp37_;
			char* _tmp36_;
			_tmp35_ = NULL;
			iface = (_tmp35_ = VALA_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) prop)), (_tmp35_ == NULL) ? NULL : vala_code_node_ref (_tmp35_));
			_tmp39_ = NULL;
			_tmp38_ = NULL;
			_tmp37_ = NULL;
			_tmp36_ = NULL;
			vcast = (_tmp39_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp38_ = vala_ccode_identifier_new (_tmp37_ = g_strdup_printf ("%s_GET_INTERFACE", _tmp36_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) iface, NULL))))), (vcast == NULL) ? NULL : (vcast = (vala_ccode_node_unref (vcast), NULL)), _tmp39_);
			(_tmp38_ == NULL) ? NULL : (_tmp38_ = (vala_ccode_node_unref (_tmp38_), NULL));
			_tmp37_ = (g_free (_tmp37_), NULL);
			_tmp36_ = (g_free (_tmp36_), NULL);
			(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
		} else {
			ValaClass* _tmp40_;
			ValaClass* cl;
			ValaCCodeFunctionCall* _tmp44_;
			ValaCCodeIdentifier* _tmp43_;
			char* _tmp42_;
			char* _tmp41_;
			_tmp40_ = NULL;
			cl = (_tmp40_ = VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) prop)), (_tmp40_ == NULL) ? NULL : vala_code_node_ref (_tmp40_));
			_tmp44_ = NULL;
			_tmp43_ = NULL;
			_tmp42_ = NULL;
			_tmp41_ = NULL;
			vcast = (_tmp44_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp43_ = vala_ccode_identifier_new (_tmp42_ = g_strdup_printf ("%s_GET_CLASS", _tmp41_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) cl, NULL))))), (vcast == NULL) ? NULL : (vcast = (vala_ccode_node_unref (vcast), NULL)), _tmp44_);
			(_tmp43_ == NULL) ? NULL : (_tmp43_ = (vala_ccode_node_unref (_tmp43_), NULL));
			_tmp42_ = (g_free (_tmp42_), NULL);
			_tmp41_ = (g_free (_tmp41_), NULL);
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		}
		_tmp45_ = NULL;
		vala_ccode_function_call_add_argument (vcast, (ValaCCodeExpression*) (_tmp45_ = vala_ccode_identifier_new ("self")));
		(_tmp45_ == NULL) ? NULL : (_tmp45_ = (vala_ccode_node_unref (_tmp45_), NULL));
		if (vala_property_accessor_get_readable (acc)) {
			ValaCCodeMemberAccess* _tmp47_;
			char* _tmp46_;
			ValaCCodeFunctionCall* _tmp48_;
			ValaCCodeFunctionCall* vcall;
			ValaCCodeIdentifier* _tmp49_;
			_tmp47_ = NULL;
			_tmp46_ = NULL;
			_tmp48_ = NULL;
			vcall = (_tmp48_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp47_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) vcast, _tmp46_ = g_strdup_printf ("get_%s", vala_symbol_get_name ((ValaSymbol*) prop))))), (_tmp47_ == NULL) ? NULL : (_tmp47_ = (vala_ccode_node_unref (_tmp47_), NULL)), _tmp46_ = (g_free (_tmp46_), NULL), _tmp48_);
			_tmp49_ = NULL;
			vala_ccode_function_call_add_argument (vcall, (ValaCCodeExpression*) (_tmp49_ = vala_ccode_identifier_new ("self")));
			(_tmp49_ == NULL) ? NULL : (_tmp49_ = (vala_ccode_node_unref (_tmp49_), NULL));
			if (returns_real_struct) {
				ValaCCodeIdentifier* _tmp50_;
				ValaCCodeExpressionStatement* _tmp51_;
				_tmp50_ = NULL;
				vala_ccode_function_call_add_argument (vcall, (ValaCCodeExpression*) (_tmp50_ = vala_ccode_identifier_new ("value")));
				(_tmp50_ == NULL) ? NULL : (_tmp50_ = (vala_ccode_node_unref (_tmp50_), NULL));
				_tmp51_ = NULL;
				vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp51_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) vcall)));
				(_tmp51_ == NULL) ? NULL : (_tmp51_ = (vala_ccode_node_unref (_tmp51_), NULL));
			} else {
				ValaCCodeReturnStatement* _tmp52_;
				_tmp52_ = NULL;
				vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp52_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) vcall)));
				(_tmp52_ == NULL) ? NULL : (_tmp52_ = (vala_ccode_node_unref (_tmp52_), NULL));
			}
			(vcall == NULL) ? NULL : (vcall = (vala_ccode_node_unref (vcall), NULL));
		} else {
			ValaCCodeMemberAccess* _tmp54_;
			char* _tmp53_;
			ValaCCodeFunctionCall* _tmp55_;
			ValaCCodeFunctionCall* vcall;
			ValaCCodeIdentifier* _tmp56_;
			ValaCCodeIdentifier* _tmp57_;
			ValaCCodeExpressionStatement* _tmp58_;
			_tmp54_ = NULL;
			_tmp53_ = NULL;
			_tmp55_ = NULL;
			vcall = (_tmp55_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp54_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) vcast, _tmp53_ = g_strdup_printf ("set_%s", vala_symbol_get_name ((ValaSymbol*) prop))))), (_tmp54_ == NULL) ? NULL : (_tmp54_ = (vala_ccode_node_unref (_tmp54_), NULL)), _tmp53_ = (g_free (_tmp53_), NULL), _tmp55_);
			_tmp56_ = NULL;
			vala_ccode_function_call_add_argument (vcall, (ValaCCodeExpression*) (_tmp56_ = vala_ccode_identifier_new ("self")));
			(_tmp56_ == NULL) ? NULL : (_tmp56_ = (vala_ccode_node_unref (_tmp56_), NULL));
			_tmp57_ = NULL;
			vala_ccode_function_call_add_argument (vcall, (ValaCCodeExpression*) (_tmp57_ = vala_ccode_identifier_new ("value")));
			(_tmp57_ == NULL) ? NULL : (_tmp57_ = (vala_ccode_node_unref (_tmp57_), NULL));
			_tmp58_ = NULL;
			vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp58_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) vcall)));
			(_tmp58_ == NULL) ? NULL : (_tmp58_ = (vala_ccode_node_unref (_tmp58_), NULL));
			(vcall == NULL) ? NULL : (vcall = (vala_ccode_node_unref (vcall), NULL));
		}
		vala_ccode_fragment_append (self->source_type_member_definition, (ValaCCodeNode*) self->function);
		(block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL));
		(vcast == NULL) ? NULL : (vcast = (vala_ccode_node_unref (vcast), NULL));
	}
	if (!vala_property_get_is_abstract (prop)) {
		gboolean _tmp59_;
		gboolean is_virtual;
		char* cname;
		gboolean _tmp65_;
		gboolean _tmp66_;
		ValaObjectType* base_type;
		gboolean _tmp74_;
		gboolean _tmp75_;
		gboolean _tmp86_;
		ValaTypeSymbol* _tmp89_;
		ValaTypeSymbol* typesymbol;
		ValaStruct* _tmp91_;
		ValaTypeSymbol* _tmp90_;
		ValaStruct* st;
		gboolean _tmp92_;
		gboolean _tmp93_;
		gboolean _tmp94_;
		gboolean _tmp95_;
		gboolean _tmp96_;
		_tmp59_ = FALSE;
		if (vala_property_get_base_property (prop) != NULL) {
			_tmp59_ = TRUE;
		} else {
			_tmp59_ = vala_property_get_base_interface_property (prop) != NULL;
		}
		is_virtual = _tmp59_;
		cname = NULL;
		if (is_virtual) {
			if (vala_property_accessor_get_readable (acc)) {
				char* _tmp61_;
				char* _tmp60_;
				_tmp61_ = NULL;
				_tmp60_ = NULL;
				cname = (_tmp61_ = g_strdup_printf ("%s_real_get_%s", _tmp60_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) t, NULL), vala_symbol_get_name ((ValaSymbol*) prop)), cname = (g_free (cname), NULL), _tmp61_);
				_tmp60_ = (g_free (_tmp60_), NULL);
			} else {
				char* _tmp63_;
				char* _tmp62_;
				_tmp63_ = NULL;
				_tmp62_ = NULL;
				cname = (_tmp63_ = g_strdup_printf ("%s_real_set_%s", _tmp62_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) t, NULL), vala_symbol_get_name ((ValaSymbol*) prop)), cname = (g_free (cname), NULL), _tmp63_);
				_tmp62_ = (g_free (_tmp62_), NULL);
			}
		} else {
			char* _tmp64_;
			_tmp64_ = NULL;
			cname = (_tmp64_ = vala_property_accessor_get_cname (acc), cname = (g_free (cname), NULL), _tmp64_);
		}
		_tmp65_ = FALSE;
		_tmp66_ = FALSE;
		if (vala_property_accessor_get_writable (acc)) {
			_tmp66_ = TRUE;
		} else {
			_tmp66_ = vala_property_accessor_get_construction (acc);
		}
		if (_tmp66_) {
			_tmp65_ = TRUE;
		} else {
			_tmp65_ = returns_real_struct;
		}
		if (_tmp65_) {
			ValaCCodeFunction* _tmp67_;
			_tmp67_ = NULL;
			self->function = (_tmp67_ = vala_ccode_function_new (cname, "void"), (self->function == NULL) ? NULL : (self->function = (vala_ccode_node_unref (self->function), NULL)), _tmp67_);
		} else {
			ValaCCodeFunction* _tmp69_;
			char* _tmp68_;
			_tmp69_ = NULL;
			_tmp68_ = NULL;
			self->function = (_tmp69_ = vala_ccode_function_new (cname, _tmp68_ = vala_data_type_get_cname (vala_property_accessor_get_value_type (acc))), (self->function == NULL) ? NULL : (self->function = (vala_ccode_node_unref (self->function), NULL)), _tmp69_);
			_tmp68_ = (g_free (_tmp68_), NULL);
		}
		base_type = NULL;
		if (vala_property_get_binding (prop) == MEMBER_BINDING_INSTANCE) {
			if (is_virtual) {
				ValaCCodeFormalParameter* _tmp73_;
				char* _tmp72_;
				if (vala_property_get_base_property (prop) != NULL) {
					ValaObjectType* _tmp70_;
					_tmp70_ = NULL;
					base_type = (_tmp70_ = vala_object_type_new (VALA_OBJECT_TYPE_SYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) vala_property_get_base_property (prop)))), (base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL)), _tmp70_);
				} else {
					if (vala_property_get_base_interface_property (prop) != NULL) {
						ValaObjectType* _tmp71_;
						_tmp71_ = NULL;
						base_type = (_tmp71_ = vala_object_type_new (VALA_OBJECT_TYPE_SYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) vala_property_get_base_interface_property (prop)))), (base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL)), _tmp71_);
					}
				}
				vala_ccode_function_set_modifiers (self->function, vala_ccode_function_get_modifiers (self->function) | VALA_CCODE_MODIFIERS_STATIC);
				_tmp73_ = NULL;
				_tmp72_ = NULL;
				vala_ccode_function_add_parameter (self->function, _tmp73_ = vala_ccode_formal_parameter_new ("base", _tmp72_ = vala_data_type_get_cname ((ValaDataType*) base_type)));
				(_tmp73_ == NULL) ? NULL : (_tmp73_ = (vala_ccode_node_unref (_tmp73_), NULL));
				_tmp72_ = (g_free (_tmp72_), NULL);
			} else {
				vala_ccode_function_add_parameter (self->function, cselfparam);
			}
		}
		_tmp74_ = FALSE;
		_tmp75_ = FALSE;
		if (vala_property_accessor_get_writable (acc)) {
			_tmp75_ = TRUE;
		} else {
			_tmp75_ = vala_property_accessor_get_construction (acc);
		}
		if (_tmp75_) {
			_tmp74_ = TRUE;
		} else {
			_tmp74_ = returns_real_struct;
		}
		if (_tmp74_) {
			vala_ccode_function_add_parameter (self->function, cvalueparam);
		}
		if (!is_virtual) {
			gboolean _tmp76_;
			gboolean _tmp77_;
			_tmp76_ = FALSE;
			_tmp77_ = FALSE;
			if (vala_symbol_is_private_symbol ((ValaSymbol*) prop)) {
				_tmp77_ = TRUE;
			} else {
				gboolean _tmp78_;
				_tmp78_ = FALSE;
				if (vala_property_accessor_get_readable (acc)) {
					_tmp78_ = TRUE;
				} else {
					_tmp78_ = vala_property_accessor_get_writable (acc);
				}
				_tmp77_ = !_tmp78_;
			}
			if (_tmp77_) {
				_tmp76_ = TRUE;
			} else {
				_tmp76_ = vala_property_accessor_get_access (acc) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE;
			}
			if (_tmp76_) {
				/* accessor function should be private if the property is an internal symbol or it's a construct-only setter*/
				vala_ccode_function_set_modifiers (self->function, vala_ccode_function_get_modifiers (self->function) | VALA_CCODE_MODIFIERS_STATIC);
			}
		}
		vala_ccode_function_set_block (self->function, VALA_CCODE_BLOCK (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_property_accessor_get_body (acc))));
		if (is_virtual) {
			char* _tmp79_;
			ValaCCodeDeclaration* _tmp80_;
			ValaCCodeDeclaration* _cdecl_;
			ValaCCodeVariableDeclarator* _tmp83_;
			ValaCCodeExpression* _tmp82_;
			ValaCCodeIdentifier* _tmp81_;
			_tmp79_ = NULL;
			_tmp80_ = NULL;
			_cdecl_ = (_tmp80_ = vala_ccode_declaration_new (_tmp79_ = vala_data_type_get_cname ((ValaDataType*) this_type)), _tmp79_ = (g_free (_tmp79_), NULL), _tmp80_);
			_tmp83_ = NULL;
			_tmp82_ = NULL;
			_tmp81_ = NULL;
			vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp83_ = vala_ccode_variable_declarator_new ("self", _tmp82_ = vala_ccode_base_module_transform_expression (self, (ValaCCodeExpression*) (_tmp81_ = vala_ccode_identifier_new ("base")), (ValaDataType*) base_type, (ValaDataType*) this_type, NULL), NULL)));
			(_tmp83_ == NULL) ? NULL : (_tmp83_ = (vala_ccode_node_unref (_tmp83_), NULL));
			(_tmp82_ == NULL) ? NULL : (_tmp82_ = (vala_ccode_node_unref (_tmp82_), NULL));
			(_tmp81_ == NULL) ? NULL : (_tmp81_ = (vala_ccode_node_unref (_tmp81_), NULL));
			vala_ccode_block_prepend_statement (vala_ccode_function_get_block (self->function), (ValaCCodeNode*) _cdecl_);
			(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
		}
		if (self->current_method_inner_error) {
			ValaCCodeDeclaration* _cdecl_;
			ValaCCodeVariableDeclarator* _tmp85_;
			ValaCCodeConstant* _tmp84_;
			_cdecl_ = vala_ccode_declaration_new ("GError *");
			_tmp85_ = NULL;
			_tmp84_ = NULL;
			vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp85_ = vala_ccode_variable_declarator_new ("_inner_error_", (ValaCCodeExpression*) (_tmp84_ = vala_ccode_constant_new ("NULL")), NULL)));
			(_tmp85_ == NULL) ? NULL : (_tmp85_ = (vala_ccode_node_unref (_tmp85_), NULL));
			(_tmp84_ == NULL) ? NULL : (_tmp84_ = (vala_ccode_node_unref (_tmp84_), NULL));
			vala_ccode_block_prepend_statement (vala_ccode_function_get_block (self->function), (ValaCCodeNode*) _cdecl_);
			(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
		}
		_tmp86_ = FALSE;
		if (vala_property_get_binding (prop) == MEMBER_BINDING_INSTANCE) {
			_tmp86_ = !is_virtual;
		} else {
			_tmp86_ = FALSE;
		}
		if (_tmp86_) {
			ValaCCodeStatement* check_stmt;
			check_stmt = NULL;
			if (returns_real_struct) {
				ValaCCodeStatement* _tmp87_;
				_tmp87_ = NULL;
				check_stmt = (_tmp87_ = vala_ccode_base_module_create_property_type_check_statement (self, prop, FALSE, (ValaTypeSymbol*) t, TRUE, "self"), (check_stmt == NULL) ? NULL : (check_stmt = (vala_ccode_node_unref (check_stmt), NULL)), _tmp87_);
			} else {
				ValaCCodeStatement* _tmp88_;
				_tmp88_ = NULL;
				check_stmt = (_tmp88_ = vala_ccode_base_module_create_property_type_check_statement (self, prop, vala_property_accessor_get_readable (acc), (ValaTypeSymbol*) t, TRUE, "self"), (check_stmt == NULL) ? NULL : (check_stmt = (vala_ccode_node_unref (check_stmt), NULL)), _tmp88_);
			}
			if (check_stmt != NULL) {
				vala_ccode_block_prepend_statement (vala_ccode_function_get_block (self->function), (ValaCCodeNode*) check_stmt);
			}
			(check_stmt == NULL) ? NULL : (check_stmt = (vala_ccode_node_unref (check_stmt), NULL));
		}
		/* notify on property changes*/
		_tmp89_ = NULL;
		typesymbol = (_tmp89_ = VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) prop)), (_tmp89_ == NULL) ? NULL : vala_code_node_ref (_tmp89_));
		_tmp91_ = NULL;
		_tmp90_ = NULL;
		st = (_tmp91_ = (_tmp90_ = vala_data_type_get_data_type (vala_property_get_property_type (prop)), VALA_IS_STRUCT (_tmp90_) ? ((ValaStruct*) _tmp90_) : NULL), (_tmp91_ == NULL) ? NULL : vala_code_node_ref (_tmp91_));
		_tmp92_ = FALSE;
		_tmp93_ = FALSE;
		_tmp94_ = FALSE;
		_tmp95_ = FALSE;
		_tmp96_ = FALSE;
		if (vala_typesymbol_is_subtype_of (typesymbol, self->gobject_type)) {
			gboolean _tmp97_;
			_tmp97_ = FALSE;
			if (st == NULL) {
				_tmp97_ = TRUE;
			} else {
				_tmp97_ = vala_struct_get_has_type_id (st);
			}
			_tmp96_ = _tmp97_;
		} else {
			_tmp96_ = FALSE;
		}
		if (_tmp96_) {
			_tmp95_ = vala_property_get_notify (prop);
		} else {
			_tmp95_ = FALSE;
		}
		if (_tmp95_) {
			_tmp94_ = vala_symbol_get_access ((ValaSymbol*) prop) != VALA_SYMBOL_ACCESSIBILITY_PRIVATE;
		} else {
			_tmp94_ = FALSE;
		}
		if (_tmp94_) {
			_tmp93_ = vala_property_get_binding (prop) == MEMBER_BINDING_INSTANCE;
		} else {
			_tmp93_ = FALSE;
		}
		if (_tmp93_) {
			gboolean _tmp98_;
			_tmp98_ = FALSE;
			if (vala_property_accessor_get_writable (acc)) {
				_tmp98_ = TRUE;
			} else {
				_tmp98_ = vala_property_accessor_get_construction (acc);
			}
			_tmp92_ = _tmp98_;
		} else {
			_tmp92_ = FALSE;
		}
		if (_tmp92_) {
			ValaCCodeIdentifier* _tmp99_;
			ValaCCodeFunctionCall* _tmp100_;
			ValaCCodeFunctionCall* notify_call;
			ValaCCodeCastExpression* _tmp102_;
			ValaCCodeIdentifier* _tmp101_;
			ValaCCodeConstant* _tmp103_;
			ValaCCodeExpressionStatement* _tmp104_;
			/* FIXME: use better means to detect gobject properties*/
			_tmp99_ = NULL;
			_tmp100_ = NULL;
			notify_call = (_tmp100_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp99_ = vala_ccode_identifier_new ("g_object_notify"))), (_tmp99_ == NULL) ? NULL : (_tmp99_ = (vala_ccode_node_unref (_tmp99_), NULL)), _tmp100_);
			_tmp102_ = NULL;
			_tmp101_ = NULL;
			vala_ccode_function_call_add_argument (notify_call, (ValaCCodeExpression*) (_tmp102_ = vala_ccode_cast_expression_new ((ValaCCodeExpression*) (_tmp101_ = vala_ccode_identifier_new ("self")), "GObject *")));
			(_tmp102_ == NULL) ? NULL : (_tmp102_ = (vala_ccode_node_unref (_tmp102_), NULL));
			(_tmp101_ == NULL) ? NULL : (_tmp101_ = (vala_ccode_node_unref (_tmp101_), NULL));
			_tmp103_ = NULL;
			vala_ccode_function_call_add_argument (notify_call, (ValaCCodeExpression*) (_tmp103_ = vala_property_get_canonical_cconstant (prop)));
			(_tmp103_ == NULL) ? NULL : (_tmp103_ = (vala_ccode_node_unref (_tmp103_), NULL));
			_tmp104_ = NULL;
			vala_ccode_block_add_statement (vala_ccode_function_get_block (self->function), (ValaCCodeNode*) (_tmp104_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) notify_call)));
			(_tmp104_ == NULL) ? NULL : (_tmp104_ = (vala_ccode_node_unref (_tmp104_), NULL));
			(notify_call == NULL) ? NULL : (notify_call = (vala_ccode_node_unref (notify_call), NULL));
		}
		vala_ccode_fragment_append (self->source_type_member_definition, (ValaCCodeNode*) self->function);
		cname = (g_free (cname), NULL);
		(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
		(typesymbol == NULL) ? NULL : (typesymbol = (vala_code_node_unref (typesymbol), NULL));
		(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
	}
	_tmp106_ = NULL;
	_tmp105_ = NULL;
	self->current_property_accessor = (_tmp106_ = (_tmp105_ = old_property_accessor, (_tmp105_ == NULL) ? NULL : vala_code_node_ref (_tmp105_)), (self->current_property_accessor == NULL) ? NULL : (self->current_property_accessor = (vala_code_node_unref (self->current_property_accessor), NULL)), _tmp106_);
	_tmp108_ = NULL;
	_tmp107_ = NULL;
	self->current_return_type = (_tmp108_ = (_tmp107_ = old_return_type, (_tmp107_ == NULL) ? NULL : vala_code_node_ref (_tmp107_)), (self->current_return_type == NULL) ? NULL : (self->current_return_type = (vala_code_node_unref (self->current_return_type), NULL)), _tmp108_);
	self->current_method_inner_error = old_method_inner_error;
	(old_property_accessor == NULL) ? NULL : (old_property_accessor = (vala_code_node_unref (old_property_accessor), NULL));
	(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
	(old_return_type == NULL) ? NULL : (old_return_type = (vala_code_node_unref (old_return_type), NULL));
	(t == NULL) ? NULL : (t = (vala_code_node_unref (t), NULL));
	(this_type == NULL) ? NULL : (this_type = (vala_code_node_unref (this_type), NULL));
	(cselfparam == NULL) ? NULL : (cselfparam = (vala_ccode_node_unref (cselfparam), NULL));
	(cvalueparam == NULL) ? NULL : (cvalueparam = (vala_ccode_node_unref (cvalueparam), NULL));
}


static void vala_ccode_base_module_real_visit_destructor (ValaCCodeModule* base, ValaDestructor* d) {
	ValaCCodeBaseModule * self;
	gboolean old_method_inner_error;
	gboolean _tmp0_;
	ValaCCodeFragment* cfrag;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (d != NULL);
	old_method_inner_error = self->current_method_inner_error;
	self->current_method_inner_error = FALSE;
	vala_code_node_accept_children ((ValaCodeNode*) d, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	_tmp0_ = FALSE;
	if (vala_destructor_get_binding (d) == MEMBER_BINDING_STATIC) {
		_tmp0_ = !self->in_plugin;
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) d), "static destructors are only supported for dynamic types");
		vala_code_node_set_error ((ValaCodeNode*) d, TRUE);
		return;
	}
	cfrag = vala_ccode_fragment_new ();
	if (self->current_method_inner_error) {
		ValaCCodeDeclaration* _cdecl_;
		ValaCCodeVariableDeclarator* _tmp2_;
		ValaCCodeConstant* _tmp1_;
		_cdecl_ = vala_ccode_declaration_new ("GError *");
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp2_ = vala_ccode_variable_declarator_new ("_inner_error_", (ValaCCodeExpression*) (_tmp1_ = vala_ccode_constant_new ("NULL")), NULL)));
		(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_ccode_node_unref (_tmp2_), NULL));
		(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_ccode_node_unref (_tmp1_), NULL));
		vala_ccode_fragment_append (cfrag, (ValaCCodeNode*) _cdecl_);
		(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
	}
	vala_ccode_fragment_append (cfrag, vala_code_node_get_ccodenode ((ValaCodeNode*) vala_destructor_get_body (d)));
	vala_code_node_set_ccodenode ((ValaCodeNode*) d, (ValaCCodeNode*) cfrag);
	self->current_method_inner_error = old_method_inner_error;
	(cfrag == NULL) ? NULL : (cfrag = (vala_ccode_node_unref (cfrag), NULL));
}


static void vala_ccode_base_module_real_visit_block (ValaCCodeModule* base, ValaBlock* b) {
	ValaCCodeBaseModule * self;
	ValaSymbol* _tmp0_;
	ValaSymbol* old_symbol;
	ValaSymbol* _tmp2_;
	ValaSymbol* _tmp1_;
	GeeList* local_vars;
	ValaCCodeBlock* cblock;
	ValaSymbol* _tmp22_;
	ValaSymbol* _tmp21_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (b != NULL);
	_tmp0_ = NULL;
	old_symbol = (_tmp0_ = self->current_symbol, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->current_symbol = (_tmp2_ = (_tmp1_ = (ValaSymbol*) b, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->current_symbol == NULL) ? NULL : (self->current_symbol = (vala_code_node_unref (self->current_symbol), NULL)), _tmp2_);
	vala_code_node_accept_children ((ValaCodeNode*) b, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	local_vars = vala_block_get_local_variables (b);
	{
		GeeIterator* _local_it;
		_local_it = gee_iterable_iterator ((GeeIterable*) local_vars);
		while (gee_iterator_next (_local_it)) {
			ValaLocalVariable* local;
			local = (ValaLocalVariable*) gee_iterator_get (_local_it);
			vala_symbol_set_active ((ValaSymbol*) local, FALSE);
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
		}
		(_local_it == NULL) ? NULL : (_local_it = (gee_collection_object_unref (_local_it), NULL));
	}
	cblock = vala_ccode_block_new ();
	{
		GeeList* _tmp3_;
		GeeIterator* _tmp4_;
		GeeIterator* _stmt_it;
		_tmp3_ = NULL;
		_tmp4_ = NULL;
		_stmt_it = (_tmp4_ = gee_iterable_iterator ((GeeIterable*) (_tmp3_ = vala_block_get_statements (b))), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (gee_collection_object_unref (_tmp3_), NULL)), _tmp4_);
		while (gee_iterator_next (_stmt_it)) {
			ValaCodeNode* stmt;
			ValaSourceReference* _tmp5_;
			ValaSourceReference* src;
			gboolean _tmp6_;
			stmt = (ValaCodeNode*) ((ValaStatement*) gee_iterator_get (_stmt_it));
			if (vala_code_node_get_error (stmt)) {
				(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
				continue;
			}
			_tmp5_ = NULL;
			src = (_tmp5_ = vala_code_node_get_source_reference (stmt), (_tmp5_ == NULL) ? NULL : vala_source_reference_ref (_tmp5_));
			_tmp6_ = FALSE;
			if (src != NULL) {
				_tmp6_ = vala_source_reference_get_comment (src) != NULL;
			} else {
				_tmp6_ = FALSE;
			}
			if (_tmp6_) {
				ValaCCodeComment* _tmp7_;
				_tmp7_ = NULL;
				vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp7_ = vala_ccode_comment_new (vala_source_reference_get_comment (src))));
				(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
			}
			if (VALA_IS_CCODE_FRAGMENT (vala_code_node_get_ccodenode (stmt))) {
				{
					GeeList* _tmp8_;
					GeeIterator* _tmp9_;
					GeeIterator* _cstmt_it;
					_tmp8_ = NULL;
					_tmp9_ = NULL;
					_cstmt_it = (_tmp9_ = gee_iterable_iterator ((GeeIterable*) (_tmp8_ = vala_ccode_fragment_get_children (VALA_CCODE_FRAGMENT (vala_code_node_get_ccodenode (stmt))))), (_tmp8_ == NULL) ? NULL : (_tmp8_ = (gee_collection_object_unref (_tmp8_), NULL)), _tmp9_);
					while (gee_iterator_next (_cstmt_it)) {
						ValaCCodeNode* cstmt;
						cstmt = (ValaCCodeNode*) gee_iterator_get (_cstmt_it);
						vala_ccode_block_add_statement (cblock, cstmt);
						(cstmt == NULL) ? NULL : (cstmt = (vala_ccode_node_unref (cstmt), NULL));
					}
					(_cstmt_it == NULL) ? NULL : (_cstmt_it = (gee_collection_object_unref (_cstmt_it), NULL));
				}
			} else {
				vala_ccode_block_add_statement (cblock, vala_code_node_get_ccodenode (stmt));
			}
			(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
			(src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL));
		}
		(_stmt_it == NULL) ? NULL : (_stmt_it = (gee_collection_object_unref (_stmt_it), NULL));
	}
	{
		GeeIterator* _local_it;
		_local_it = gee_iterable_iterator ((GeeIterable*) local_vars);
		while (gee_iterator_next (_local_it)) {
			ValaLocalVariable* local;
			gboolean _tmp10_;
			local = (ValaLocalVariable*) gee_iterator_get (_local_it);
			_tmp10_ = FALSE;
			if (!vala_local_variable_get_floating (local)) {
				_tmp10_ = vala_ccode_base_module_requires_destroy (self, vala_local_variable_get_variable_type (local));
			} else {
				_tmp10_ = FALSE;
			}
			if (_tmp10_) {
				ValaMemberAccess* ma;
				ValaCCodeExpressionStatement* _tmp13_;
				ValaCCodeExpression* _tmp12_;
				ValaCCodeExpression* _tmp11_;
				ma = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) local), NULL);
				vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) local);
				_tmp13_ = NULL;
				_tmp12_ = NULL;
				_tmp11_ = NULL;
				vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp13_ = vala_ccode_expression_statement_new (_tmp12_ = vala_ccode_base_module_get_unref_expression (self, _tmp11_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) local)), vala_local_variable_get_variable_type (local), (ValaExpression*) ma))));
				(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL));
				(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
				(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
				(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
			}
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
		}
		(_local_it == NULL) ? NULL : (_local_it = (gee_collection_object_unref (_local_it), NULL));
	}
	if (VALA_IS_METHOD (vala_symbol_get_parent_symbol ((ValaSymbol*) b))) {
		ValaMethod* _tmp14_;
		ValaMethod* m;
		_tmp14_ = NULL;
		m = (_tmp14_ = VALA_METHOD (vala_symbol_get_parent_symbol ((ValaSymbol*) b)), (_tmp14_ == NULL) ? NULL : vala_code_node_ref (_tmp14_));
		{
			GeeList* _tmp15_;
			GeeIterator* _tmp16_;
			GeeIterator* _param_it;
			_tmp15_ = NULL;
			_tmp16_ = NULL;
			_param_it = (_tmp16_ = gee_iterable_iterator ((GeeIterable*) (_tmp15_ = vala_method_get_parameters (m))), (_tmp15_ == NULL) ? NULL : (_tmp15_ = (gee_collection_object_unref (_tmp15_), NULL)), _tmp16_);
			while (gee_iterator_next (_param_it)) {
				ValaFormalParameter* param;
				gboolean _tmp17_;
				param = (ValaFormalParameter*) gee_iterator_get (_param_it);
				_tmp17_ = FALSE;
				if (vala_ccode_base_module_requires_destroy (self, vala_formal_parameter_get_parameter_type (param))) {
					_tmp17_ = vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_IN;
				} else {
					_tmp17_ = FALSE;
				}
				if (_tmp17_) {
					ValaMemberAccess* ma;
					ValaCCodeExpressionStatement* _tmp20_;
					ValaCCodeExpression* _tmp19_;
					ValaCCodeExpression* _tmp18_;
					ma = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) param), NULL);
					vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) param);
					_tmp20_ = NULL;
					_tmp19_ = NULL;
					_tmp18_ = NULL;
					vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp20_ = vala_ccode_expression_statement_new (_tmp19_ = vala_ccode_base_module_get_unref_expression (self, _tmp18_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) param)), vala_formal_parameter_get_parameter_type (param), (ValaExpression*) ma))));
					(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL));
					(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
					(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
					(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
				}
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
			}
			(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
		}
		(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
	}
	vala_code_node_set_ccodenode ((ValaCodeNode*) b, (ValaCCodeNode*) cblock);
	_tmp22_ = NULL;
	_tmp21_ = NULL;
	self->current_symbol = (_tmp22_ = (_tmp21_ = old_symbol, (_tmp21_ == NULL) ? NULL : vala_code_node_ref (_tmp21_)), (self->current_symbol == NULL) ? NULL : (self->current_symbol = (vala_code_node_unref (self->current_symbol), NULL)), _tmp22_);
	(old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL));
	(local_vars == NULL) ? NULL : (local_vars = (gee_collection_object_unref (local_vars), NULL));
	(cblock == NULL) ? NULL : (cblock = (vala_ccode_node_unref (cblock), NULL));
}


static void vala_ccode_base_module_real_visit_empty_statement (ValaCCodeModule* base, ValaEmptyStatement* stmt) {
	ValaCCodeBaseModule * self;
	ValaCCodeEmptyStatement* _tmp0_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (stmt != NULL);
	_tmp0_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) stmt, (ValaCCodeNode*) (_tmp0_ = vala_ccode_empty_statement_new ()));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
}


static void vala_ccode_base_module_real_visit_declaration_statement (ValaCCodeModule* base, ValaDeclarationStatement* stmt) {
	ValaCCodeBaseModule * self;
	ValaLocalVariable* _tmp1_;
	ValaSymbol* _tmp0_;
	ValaLocalVariable* local;
	gboolean _tmp2_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_declaration_statement_get_declaration (stmt), (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	vala_code_node_set_ccodenode ((ValaCodeNode*) stmt, vala_code_node_get_ccodenode ((ValaCodeNode*) vala_declaration_statement_get_declaration (stmt)));
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	local = (_tmp1_ = (_tmp0_ = vala_declaration_statement_get_declaration (stmt), VALA_IS_LOCAL_VARIABLE (_tmp0_) ? ((ValaLocalVariable*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp2_ = FALSE;
	if (local != NULL) {
		_tmp2_ = vala_local_variable_get_initializer (local) != NULL;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		vala_ccode_base_module_create_temp_decl (self, (ValaStatement*) stmt, (GeeList*) vala_local_variable_get_initializer (local)->temp_vars);
	}
	vala_ccode_base_module_create_temp_decl (self, (ValaStatement*) stmt, (GeeList*) self->temp_vars);
	gee_collection_clear ((GeeCollection*) self->temp_vars);
	(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
}


ValaCCodeExpression* vala_ccode_base_module_get_variable_cexpression (ValaCCodeBaseModule* self, const char* name) {
	gboolean _tmp0_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	_tmp0_ = FALSE;
	if (self->current_method != NULL) {
		_tmp0_ = vala_method_get_coroutine (self->current_method);
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		char* _tmp2_;
		ValaCCodeIdentifier* _tmp1_;
		ValaCCodeExpression* _tmp3_;
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		_tmp3_ = NULL;
		return (_tmp3_ = (ValaCCodeExpression*) vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp1_ = vala_ccode_identifier_new ("data")), _tmp2_ = vala_ccode_base_module_get_variable_cname (self, name)), _tmp2_ = (g_free (_tmp2_), NULL), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_ccode_node_unref (_tmp1_), NULL)), _tmp3_);
	} else {
		char* _tmp5_;
		ValaCCodeExpression* _tmp6_;
		_tmp5_ = NULL;
		_tmp6_ = NULL;
		return (_tmp6_ = (ValaCCodeExpression*) vala_ccode_identifier_new (_tmp5_ = vala_ccode_base_module_get_variable_cname (self, name)), _tmp5_ = (g_free (_tmp5_), NULL), _tmp6_);
	}
}


char* vala_ccode_base_module_get_variable_cname (ValaCCodeBaseModule* self, const char* name) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	if (g_utf8_get_char (g_utf8_offset_to_pointer (name, 0)) == '.') {
		/* compiler-internal variable*/
		if (!gee_map_contains (self->variable_name_map, name)) {
			char* _tmp0_;
			_tmp0_ = NULL;
			gee_map_set (self->variable_name_map, name, _tmp0_ = g_strdup_printf ("_tmp%d_", self->next_temp_var_id));
			_tmp0_ = (g_free (_tmp0_), NULL);
			self->next_temp_var_id++;
		}
		return (char*) gee_map_get (self->variable_name_map, name);
	} else {
		if (gee_collection_contains ((GeeCollection*) self->priv->reserved_identifiers, name)) {
			return g_strdup_printf ("_%s_", name);
		} else {
			const char* _tmp3_;
			_tmp3_ = NULL;
			return (_tmp3_ = name, (_tmp3_ == NULL) ? NULL : g_strdup (_tmp3_));
		}
	}
}


static void vala_ccode_base_module_real_visit_local_variable (ValaCCodeModule* base, ValaLocalVariable* local) {
	ValaCCodeBaseModule * self;
	ValaCCodeExpression* rhs;
	gboolean _tmp16_;
	ValaCCodeFragment* cfrag;
	gboolean _tmp59_;
	gboolean _tmp75_;
	gboolean _tmp88_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (local != NULL);
	vala_ccode_base_module_check_type (self, vala_local_variable_get_variable_type (local));
	vala_code_node_accept_children ((ValaCodeNode*) local, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	vala_ccode_base_module_generate_type_declaration (self, vala_local_variable_get_variable_type (local), self->source_declarations);
	if (VALA_IS_ARRAY_TYPE (vala_local_variable_get_variable_type (local))) {
		ValaArrayType* _tmp0_;
		ValaArrayType* array_type;
		/* create variables to store array dimensions*/
		_tmp0_ = NULL;
		array_type = (_tmp0_ = VALA_ARRAY_TYPE (vala_local_variable_get_variable_type (local)), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
		if (!vala_array_type_get_fixed_length (array_type)) {
			{
				gint dim;
				dim = 1;
				for (; dim <= vala_array_type_get_rank (array_type); dim++) {
					char* _tmp3_;
					char* _tmp2_;
					ValaDataType* _tmp1_;
					ValaLocalVariable* _tmp4_;
					ValaLocalVariable* len_var;
					_tmp3_ = NULL;
					_tmp2_ = NULL;
					_tmp1_ = NULL;
					_tmp4_ = NULL;
					len_var = (_tmp4_ = vala_local_variable_new (_tmp1_ = vala_data_type_copy (self->int_type), _tmp3_ = vala_ccode_module_get_array_length_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), _tmp2_ = vala_ccode_base_module_get_variable_cname (self, vala_symbol_get_name ((ValaSymbol*) local)), dim), NULL, NULL), _tmp3_ = (g_free (_tmp3_), NULL), _tmp2_ = (g_free (_tmp2_), NULL), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_code_node_unref (_tmp1_), NULL)), _tmp4_);
					gee_list_insert ((GeeList*) self->temp_vars, 0, len_var);
					(len_var == NULL) ? NULL : (len_var = (vala_code_node_unref (len_var), NULL));
				}
			}
			if (vala_array_type_get_rank (array_type) == 1) {
				char* _tmp7_;
				char* _tmp6_;
				ValaDataType* _tmp5_;
				ValaLocalVariable* _tmp8_;
				ValaLocalVariable* size_var;
				_tmp7_ = NULL;
				_tmp6_ = NULL;
				_tmp5_ = NULL;
				_tmp8_ = NULL;
				size_var = (_tmp8_ = vala_local_variable_new (_tmp5_ = vala_data_type_copy (self->int_type), _tmp7_ = vala_ccode_module_get_array_size_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), _tmp6_ = vala_ccode_base_module_get_variable_cname (self, vala_symbol_get_name ((ValaSymbol*) local))), NULL, NULL), _tmp7_ = (g_free (_tmp7_), NULL), _tmp6_ = (g_free (_tmp6_), NULL), (_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_code_node_unref (_tmp5_), NULL)), _tmp8_);
				gee_list_insert ((GeeList*) self->temp_vars, 0, size_var);
				(size_var == NULL) ? NULL : (size_var = (vala_code_node_unref (size_var), NULL));
			}
		}
		(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
	} else {
		if (VALA_IS_DELEGATE_TYPE (vala_local_variable_get_variable_type (local))) {
			ValaDelegateType* _tmp9_;
			ValaDelegateType* deleg_type;
			ValaDelegate* _tmp10_;
			ValaDelegate* d;
			_tmp9_ = NULL;
			deleg_type = (_tmp9_ = VALA_DELEGATE_TYPE (vala_local_variable_get_variable_type (local)), (_tmp9_ == NULL) ? NULL : vala_code_node_ref (_tmp9_));
			_tmp10_ = NULL;
			d = (_tmp10_ = vala_delegate_type_get_delegate_symbol (deleg_type), (_tmp10_ == NULL) ? NULL : vala_code_node_ref (_tmp10_));
			if (vala_delegate_get_has_target (d)) {
				char* _tmp14_;
				char* _tmp13_;
				ValaPointerType* _tmp12_;
				ValaVoidType* _tmp11_;
				ValaLocalVariable* _tmp15_;
				ValaLocalVariable* target_var;
				/* create variable to store delegate target*/
				_tmp14_ = NULL;
				_tmp13_ = NULL;
				_tmp12_ = NULL;
				_tmp11_ = NULL;
				_tmp15_ = NULL;
				target_var = (_tmp15_ = vala_local_variable_new ((ValaDataType*) (_tmp12_ = vala_pointer_type_new ((ValaDataType*) (_tmp11_ = vala_void_type_new (NULL)), NULL)), _tmp14_ = vala_ccode_base_module_get_delegate_target_cname (self, _tmp13_ = vala_ccode_base_module_get_variable_cname (self, vala_symbol_get_name ((ValaSymbol*) local))), NULL, NULL), _tmp14_ = (g_free (_tmp14_), NULL), _tmp13_ = (g_free (_tmp13_), NULL), (_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_code_node_unref (_tmp12_), NULL)), (_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_code_node_unref (_tmp11_), NULL)), _tmp15_);
				gee_list_insert ((GeeList*) self->temp_vars, 0, target_var);
				(target_var == NULL) ? NULL : (target_var = (vala_code_node_unref (target_var), NULL));
			}
			(deleg_type == NULL) ? NULL : (deleg_type = (vala_code_node_unref (deleg_type), NULL));
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
		}
	}
	rhs = NULL;
	_tmp16_ = FALSE;
	if (vala_local_variable_get_initializer (local) != NULL) {
		_tmp16_ = vala_code_node_get_ccodenode ((ValaCodeNode*) vala_local_variable_get_initializer (local)) != NULL;
	} else {
		_tmp16_ = FALSE;
	}
	if (_tmp16_) {
		ValaCCodeExpression* _tmp18_;
		ValaCCodeExpression* _tmp17_;
		_tmp18_ = NULL;
		_tmp17_ = NULL;
		rhs = (_tmp18_ = (_tmp17_ = VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_local_variable_get_initializer (local))), (_tmp17_ == NULL) ? NULL : vala_ccode_node_ref (_tmp17_)), (rhs == NULL) ? NULL : (rhs = (vala_ccode_node_unref (rhs), NULL)), _tmp18_);
		if (VALA_IS_ARRAY_TYPE (vala_local_variable_get_variable_type (local))) {
			ValaArrayType* _tmp19_;
			ValaArrayType* array_type;
			_tmp19_ = NULL;
			array_type = (_tmp19_ = VALA_ARRAY_TYPE (vala_local_variable_get_variable_type (local)), (_tmp19_ == NULL) ? NULL : vala_code_node_ref (_tmp19_));
			if (vala_array_type_get_fixed_length (array_type)) {
				ValaCCodeExpression* _tmp20_;
				_tmp20_ = NULL;
				rhs = (_tmp20_ = NULL, (rhs == NULL) ? NULL : (rhs = (vala_ccode_node_unref (rhs), NULL)), _tmp20_);
			} else {
				ValaCCodeCommaExpression* ccomma;
				ValaLocalVariable* temp_var;
				ValaCCodeAssignment* _tmp22_;
				ValaCCodeExpression* _tmp21_;
				ValaCCodeExpression* _tmp34_;
				ValaCCodeExpression* _tmp36_;
				ValaCCodeExpression* _tmp35_;
				ccomma = vala_ccode_comma_expression_new ();
				temp_var = vala_ccode_base_module_get_temp_variable (self, vala_local_variable_get_variable_type (local), TRUE, (ValaCodeNode*) local);
				gee_list_insert ((GeeList*) self->temp_vars, 0, temp_var);
				_tmp22_ = NULL;
				_tmp21_ = NULL;
				vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp22_ = vala_ccode_assignment_new (_tmp21_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) temp_var)), rhs, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
				(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL));
				(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL));
				{
					gint dim;
					dim = 1;
					for (; dim <= vala_array_type_get_rank (array_type); dim++) {
						char* _tmp24_;
						char* _tmp23_;
						ValaCCodeExpression* _tmp25_;
						ValaCCodeExpression* lhs_array_len;
						ValaCCodeExpression* rhs_array_len;
						ValaCCodeAssignment* _tmp26_;
						_tmp24_ = NULL;
						_tmp23_ = NULL;
						_tmp25_ = NULL;
						lhs_array_len = (_tmp25_ = vala_ccode_base_module_get_variable_cexpression (self, _tmp24_ = vala_ccode_module_get_array_length_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), _tmp23_ = vala_ccode_base_module_get_variable_cname (self, vala_symbol_get_name ((ValaSymbol*) local)), dim)), _tmp24_ = (g_free (_tmp24_), NULL), _tmp23_ = (g_free (_tmp23_), NULL), _tmp25_);
						rhs_array_len = vala_ccode_module_get_array_length_cexpression (vala_ccode_module_get_head ((ValaCCodeModule*) self), vala_local_variable_get_initializer (local), dim);
						_tmp26_ = NULL;
						vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp26_ = vala_ccode_assignment_new (lhs_array_len, rhs_array_len, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
						(_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_ccode_node_unref (_tmp26_), NULL));
						(lhs_array_len == NULL) ? NULL : (lhs_array_len = (vala_ccode_node_unref (lhs_array_len), NULL));
						(rhs_array_len == NULL) ? NULL : (rhs_array_len = (vala_ccode_node_unref (rhs_array_len), NULL));
					}
				}
				if (vala_array_type_get_rank (array_type) == 1) {
					char* _tmp28_;
					char* _tmp27_;
					ValaCCodeExpression* _tmp29_;
					ValaCCodeExpression* lhs_array_size;
					char* _tmp31_;
					char* _tmp30_;
					ValaCCodeExpression* _tmp32_;
					ValaCCodeExpression* rhs_array_len;
					ValaCCodeAssignment* _tmp33_;
					_tmp28_ = NULL;
					_tmp27_ = NULL;
					_tmp29_ = NULL;
					lhs_array_size = (_tmp29_ = vala_ccode_base_module_get_variable_cexpression (self, _tmp28_ = vala_ccode_module_get_array_size_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), _tmp27_ = vala_ccode_base_module_get_variable_cname (self, vala_symbol_get_name ((ValaSymbol*) local)))), _tmp28_ = (g_free (_tmp28_), NULL), _tmp27_ = (g_free (_tmp27_), NULL), _tmp29_);
					_tmp31_ = NULL;
					_tmp30_ = NULL;
					_tmp32_ = NULL;
					rhs_array_len = (_tmp32_ = vala_ccode_base_module_get_variable_cexpression (self, _tmp31_ = vala_ccode_module_get_array_length_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), _tmp30_ = vala_ccode_base_module_get_variable_cname (self, vala_symbol_get_name ((ValaSymbol*) local)), 1)), _tmp31_ = (g_free (_tmp31_), NULL), _tmp30_ = (g_free (_tmp30_), NULL), _tmp32_);
					_tmp33_ = NULL;
					vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp33_ = vala_ccode_assignment_new (lhs_array_size, rhs_array_len, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
					(_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_ccode_node_unref (_tmp33_), NULL));
					(lhs_array_size == NULL) ? NULL : (lhs_array_size = (vala_ccode_node_unref (lhs_array_size), NULL));
					(rhs_array_len == NULL) ? NULL : (rhs_array_len = (vala_ccode_node_unref (rhs_array_len), NULL));
				}
				_tmp34_ = NULL;
				vala_ccode_comma_expression_append_expression (ccomma, _tmp34_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) temp_var)));
				(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_ccode_node_unref (_tmp34_), NULL));
				_tmp36_ = NULL;
				_tmp35_ = NULL;
				rhs = (_tmp36_ = (_tmp35_ = (ValaCCodeExpression*) ccomma, (_tmp35_ == NULL) ? NULL : vala_ccode_node_ref (_tmp35_)), (rhs == NULL) ? NULL : (rhs = (vala_ccode_node_unref (rhs), NULL)), _tmp36_);
				(ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL));
				(temp_var == NULL) ? NULL : (temp_var = (vala_code_node_unref (temp_var), NULL));
			}
			(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
		} else {
			if (VALA_IS_DELEGATE_TYPE (vala_local_variable_get_variable_type (local))) {
				ValaDelegateType* _tmp37_;
				ValaDelegateType* deleg_type;
				ValaDelegate* _tmp38_;
				ValaDelegate* d;
				_tmp37_ = NULL;
				deleg_type = (_tmp37_ = VALA_DELEGATE_TYPE (vala_local_variable_get_variable_type (local)), (_tmp37_ == NULL) ? NULL : vala_code_node_ref (_tmp37_));
				_tmp38_ = NULL;
				d = (_tmp38_ = vala_delegate_type_get_delegate_symbol (deleg_type), (_tmp38_ == NULL) ? NULL : vala_code_node_ref (_tmp38_));
				if (vala_delegate_get_has_target (d)) {
					ValaCCodeCommaExpression* ccomma;
					ValaLocalVariable* temp_var;
					ValaCCodeAssignment* _tmp40_;
					ValaCCodeExpression* _tmp39_;
					char* _tmp42_;
					char* _tmp41_;
					ValaCCodeExpression* _tmp43_;
					ValaCCodeExpression* lhs_delegate_target;
					ValaCCodeExpression* rhs_delegate_target;
					ValaCCodeAssignment* _tmp44_;
					ValaCCodeExpression* _tmp45_;
					ValaCCodeExpression* _tmp47_;
					ValaCCodeExpression* _tmp46_;
					ccomma = vala_ccode_comma_expression_new ();
					temp_var = vala_ccode_base_module_get_temp_variable (self, vala_local_variable_get_variable_type (local), TRUE, (ValaCodeNode*) local);
					gee_list_insert ((GeeList*) self->temp_vars, 0, temp_var);
					_tmp40_ = NULL;
					_tmp39_ = NULL;
					vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp40_ = vala_ccode_assignment_new (_tmp39_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) temp_var)), rhs, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
					(_tmp40_ == NULL) ? NULL : (_tmp40_ = (vala_ccode_node_unref (_tmp40_), NULL));
					(_tmp39_ == NULL) ? NULL : (_tmp39_ = (vala_ccode_node_unref (_tmp39_), NULL));
					_tmp42_ = NULL;
					_tmp41_ = NULL;
					_tmp43_ = NULL;
					lhs_delegate_target = (_tmp43_ = vala_ccode_base_module_get_variable_cexpression (self, _tmp42_ = vala_ccode_base_module_get_delegate_target_cname (self, _tmp41_ = vala_ccode_base_module_get_variable_cname (self, vala_symbol_get_name ((ValaSymbol*) local)))), _tmp42_ = (g_free (_tmp42_), NULL), _tmp41_ = (g_free (_tmp41_), NULL), _tmp43_);
					rhs_delegate_target = vala_ccode_base_module_get_delegate_target_cexpression (self, vala_local_variable_get_initializer (local));
					_tmp44_ = NULL;
					vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp44_ = vala_ccode_assignment_new (lhs_delegate_target, rhs_delegate_target, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
					(_tmp44_ == NULL) ? NULL : (_tmp44_ = (vala_ccode_node_unref (_tmp44_), NULL));
					_tmp45_ = NULL;
					vala_ccode_comma_expression_append_expression (ccomma, _tmp45_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) temp_var)));
					(_tmp45_ == NULL) ? NULL : (_tmp45_ = (vala_ccode_node_unref (_tmp45_), NULL));
					_tmp47_ = NULL;
					_tmp46_ = NULL;
					rhs = (_tmp47_ = (_tmp46_ = (ValaCCodeExpression*) ccomma, (_tmp46_ == NULL) ? NULL : vala_ccode_node_ref (_tmp46_)), (rhs == NULL) ? NULL : (rhs = (vala_ccode_node_unref (rhs), NULL)), _tmp47_);
					(ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL));
					(temp_var == NULL) ? NULL : (temp_var = (vala_code_node_unref (temp_var), NULL));
					(lhs_delegate_target == NULL) ? NULL : (lhs_delegate_target = (vala_ccode_node_unref (lhs_delegate_target), NULL));
					(rhs_delegate_target == NULL) ? NULL : (rhs_delegate_target = (vala_ccode_node_unref (rhs_delegate_target), NULL));
				}
				(deleg_type == NULL) ? NULL : (deleg_type = (vala_code_node_unref (deleg_type), NULL));
				(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
			}
		}
	} else {
		if (vala_data_type_is_reference_type_or_type_parameter (vala_local_variable_get_variable_type (local))) {
			ValaCCodeExpression* _tmp48_;
			_tmp48_ = NULL;
			rhs = (_tmp48_ = (ValaCCodeExpression*) vala_ccode_constant_new ("NULL"), (rhs == NULL) ? NULL : (rhs = (vala_ccode_node_unref (rhs), NULL)), _tmp48_);
			if (VALA_IS_ARRAY_TYPE (vala_local_variable_get_variable_type (local))) {
				ValaArrayType* _tmp49_;
				ValaArrayType* array_type;
				/* initialize array length variables*/
				_tmp49_ = NULL;
				array_type = (_tmp49_ = VALA_ARRAY_TYPE (vala_local_variable_get_variable_type (local)), (_tmp49_ == NULL) ? NULL : vala_code_node_ref (_tmp49_));
				if (vala_array_type_get_fixed_length (array_type)) {
					ValaCCodeExpression* _tmp50_;
					_tmp50_ = NULL;
					rhs = (_tmp50_ = NULL, (rhs == NULL) ? NULL : (rhs = (vala_ccode_node_unref (rhs), NULL)), _tmp50_);
				} else {
					ValaCCodeCommaExpression* ccomma;
					ValaCCodeExpression* _tmp57_;
					ValaCCodeExpression* _tmp56_;
					ccomma = vala_ccode_comma_expression_new ();
					{
						gint dim;
						dim = 1;
						for (; dim <= vala_array_type_get_rank (array_type); dim++) {
							ValaCCodeAssignment* _tmp55_;
							ValaCCodeConstant* _tmp54_;
							ValaCCodeExpression* _tmp53_;
							char* _tmp52_;
							char* _tmp51_;
							_tmp55_ = NULL;
							_tmp54_ = NULL;
							_tmp53_ = NULL;
							_tmp52_ = NULL;
							_tmp51_ = NULL;
							vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp55_ = vala_ccode_assignment_new (_tmp53_ = vala_ccode_base_module_get_variable_cexpression (self, _tmp52_ = vala_ccode_module_get_array_length_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), _tmp51_ = vala_ccode_base_module_get_variable_cname (self, vala_symbol_get_name ((ValaSymbol*) local)), dim)), (ValaCCodeExpression*) (_tmp54_ = vala_ccode_constant_new ("0")), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
							(_tmp55_ == NULL) ? NULL : (_tmp55_ = (vala_ccode_node_unref (_tmp55_), NULL));
							(_tmp54_ == NULL) ? NULL : (_tmp54_ = (vala_ccode_node_unref (_tmp54_), NULL));
							(_tmp53_ == NULL) ? NULL : (_tmp53_ = (vala_ccode_node_unref (_tmp53_), NULL));
							_tmp52_ = (g_free (_tmp52_), NULL);
							_tmp51_ = (g_free (_tmp51_), NULL);
						}
					}
					vala_ccode_comma_expression_append_expression (ccomma, rhs);
					_tmp57_ = NULL;
					_tmp56_ = NULL;
					rhs = (_tmp57_ = (_tmp56_ = (ValaCCodeExpression*) ccomma, (_tmp56_ == NULL) ? NULL : vala_ccode_node_ref (_tmp56_)), (rhs == NULL) ? NULL : (rhs = (vala_ccode_node_unref (rhs), NULL)), _tmp57_);
					(ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL));
				}
				(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
			}
		}
	}
	cfrag = vala_ccode_fragment_new ();
	if (self->pre_statement_fragment != NULL) {
		ValaCCodeFragment* _tmp58_;
		vala_ccode_fragment_append (cfrag, (ValaCCodeNode*) self->pre_statement_fragment);
		_tmp58_ = NULL;
		self->pre_statement_fragment = (_tmp58_ = NULL, (self->pre_statement_fragment == NULL) ? NULL : (self->pre_statement_fragment = (vala_ccode_node_unref (self->pre_statement_fragment), NULL)), _tmp58_);
	}
	_tmp59_ = FALSE;
	if (self->current_method != NULL) {
		_tmp59_ = vala_method_get_coroutine (self->current_method);
	} else {
		_tmp59_ = FALSE;
	}
	if (_tmp59_) {
		char* _tmp63_;
		char* _tmp62_;
		char* _tmp61_;
		char* _tmp60_;
		_tmp63_ = NULL;
		_tmp62_ = NULL;
		_tmp61_ = NULL;
		_tmp60_ = NULL;
		vala_ccode_struct_add_field (self->closure_struct, _tmp60_ = vala_data_type_get_cname (vala_local_variable_get_variable_type (local)), _tmp63_ = g_strconcat (_tmp61_ = vala_ccode_base_module_get_variable_cname (self, vala_symbol_get_name ((ValaSymbol*) local)), _tmp62_ = vala_data_type_get_cdeclarator_suffix (vala_local_variable_get_variable_type (local)), NULL));
		_tmp63_ = (g_free (_tmp63_), NULL);
		_tmp62_ = (g_free (_tmp62_), NULL);
		_tmp61_ = (g_free (_tmp61_), NULL);
		_tmp60_ = (g_free (_tmp60_), NULL);
		if (vala_local_variable_get_initializer (local) != NULL) {
			ValaCCodeExpressionStatement* _tmp68_;
			ValaCCodeAssignment* _tmp67_;
			ValaCCodeMemberAccess* _tmp66_;
			char* _tmp65_;
			ValaCCodeIdentifier* _tmp64_;
			_tmp68_ = NULL;
			_tmp67_ = NULL;
			_tmp66_ = NULL;
			_tmp65_ = NULL;
			_tmp64_ = NULL;
			vala_ccode_fragment_append (cfrag, (ValaCCodeNode*) (_tmp68_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp67_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp66_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp64_ = vala_ccode_identifier_new ("data")), _tmp65_ = vala_ccode_base_module_get_variable_cname (self, vala_symbol_get_name ((ValaSymbol*) local)))), rhs, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
			(_tmp68_ == NULL) ? NULL : (_tmp68_ = (vala_ccode_node_unref (_tmp68_), NULL));
			(_tmp67_ == NULL) ? NULL : (_tmp67_ = (vala_ccode_node_unref (_tmp67_), NULL));
			(_tmp66_ == NULL) ? NULL : (_tmp66_ = (vala_ccode_node_unref (_tmp66_), NULL));
			_tmp65_ = (g_free (_tmp65_), NULL);
			(_tmp64_ == NULL) ? NULL : (_tmp64_ = (vala_ccode_node_unref (_tmp64_), NULL));
		}
	} else {
		char* _tmp70_;
		char* _tmp69_;
		ValaCCodeVariableDeclarator* _tmp71_;
		ValaCCodeVariableDeclarator* cvar;
		char* _tmp72_;
		ValaCCodeDeclaration* _tmp73_;
		ValaCCodeDeclaration* _cdecl_;
		_tmp70_ = NULL;
		_tmp69_ = NULL;
		_tmp71_ = NULL;
		cvar = (_tmp71_ = vala_ccode_variable_declarator_new (_tmp69_ = vala_ccode_base_module_get_variable_cname (self, vala_symbol_get_name ((ValaSymbol*) local)), rhs, _tmp70_ = vala_data_type_get_cdeclarator_suffix (vala_local_variable_get_variable_type (local))), _tmp70_ = (g_free (_tmp70_), NULL), _tmp69_ = (g_free (_tmp69_), NULL), _tmp71_);
		_tmp72_ = NULL;
		_tmp73_ = NULL;
		_cdecl_ = (_tmp73_ = vala_ccode_declaration_new (_tmp72_ = vala_data_type_get_cname (vala_local_variable_get_variable_type (local))), _tmp72_ = (g_free (_tmp72_), NULL), _tmp73_);
		vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) cvar);
		vala_ccode_fragment_append (cfrag, (ValaCCodeNode*) _cdecl_);
		/* try to initialize uninitialized variables
		 initialization not necessary for variables stored in closure*/
		if (vala_ccode_variable_declarator_get_initializer (cvar) == NULL) {
			ValaCCodeExpression* _tmp74_;
			_tmp74_ = NULL;
			vala_ccode_variable_declarator_set_initializer (cvar, _tmp74_ = vala_ccode_base_module_default_value_for_type (self, vala_local_variable_get_variable_type (local), TRUE));
			(_tmp74_ == NULL) ? NULL : (_tmp74_ = (vala_ccode_node_unref (_tmp74_), NULL));
		}
		(cvar == NULL) ? NULL : (cvar = (vala_ccode_node_unref (cvar), NULL));
		(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
	}
	_tmp75_ = FALSE;
	if (vala_local_variable_get_initializer (local) != NULL) {
		_tmp75_ = VALA_IS_ARRAY_TYPE (vala_local_variable_get_variable_type (local));
	} else {
		_tmp75_ = FALSE;
	}
	if (_tmp75_) {
		ValaArrayType* _tmp76_;
		ValaArrayType* array_type;
		_tmp76_ = NULL;
		array_type = (_tmp76_ = VALA_ARRAY_TYPE (vala_local_variable_get_variable_type (local)), (_tmp76_ == NULL) ? NULL : vala_code_node_ref (_tmp76_));
		if (vala_array_type_get_fixed_length (array_type)) {
			ValaCCodeIdentifier* _tmp77_;
			ValaCCodeFunctionCall* _tmp78_;
			ValaCCodeFunctionCall* sizeof_call;
			ValaCCodeIdentifier* _tmp80_;
			char* _tmp79_;
			ValaCCodeConstant* _tmp82_;
			char* _tmp81_;
			ValaCCodeBinaryExpression* _tmp83_;
			ValaCCodeBinaryExpression* size;
			ValaCCodeIdentifier* _tmp84_;
			ValaCCodeFunctionCall* _tmp85_;
			ValaCCodeFunctionCall* ccopy;
			ValaCCodeExpression* _tmp86_;
			ValaCCodeExpressionStatement* _tmp87_;
			vala_ccode_declaration_space_add_include (self->source_declarations, "string.h", FALSE);
			/* it is necessary to use memcpy for fixed-length (stack-allocated) arrays
			 simple assignments do not work in C*/
			_tmp77_ = NULL;
			_tmp78_ = NULL;
			sizeof_call = (_tmp78_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp77_ = vala_ccode_identifier_new ("sizeof"))), (_tmp77_ == NULL) ? NULL : (_tmp77_ = (vala_ccode_node_unref (_tmp77_), NULL)), _tmp78_);
			_tmp80_ = NULL;
			_tmp79_ = NULL;
			vala_ccode_function_call_add_argument (sizeof_call, (ValaCCodeExpression*) (_tmp80_ = vala_ccode_identifier_new (_tmp79_ = vala_data_type_get_cname (vala_array_type_get_element_type (array_type)))));
			(_tmp80_ == NULL) ? NULL : (_tmp80_ = (vala_ccode_node_unref (_tmp80_), NULL));
			_tmp79_ = (g_free (_tmp79_), NULL);
			_tmp82_ = NULL;
			_tmp81_ = NULL;
			_tmp83_ = NULL;
			size = (_tmp83_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_MUL, (ValaCCodeExpression*) (_tmp82_ = vala_ccode_constant_new (_tmp81_ = g_strdup_printf ("%d", vala_array_type_get_length (array_type)))), (ValaCCodeExpression*) sizeof_call), (_tmp82_ == NULL) ? NULL : (_tmp82_ = (vala_ccode_node_unref (_tmp82_), NULL)), _tmp81_ = (g_free (_tmp81_), NULL), _tmp83_);
			_tmp84_ = NULL;
			_tmp85_ = NULL;
			ccopy = (_tmp85_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp84_ = vala_ccode_identifier_new ("memcpy"))), (_tmp84_ == NULL) ? NULL : (_tmp84_ = (vala_ccode_node_unref (_tmp84_), NULL)), _tmp85_);
			_tmp86_ = NULL;
			vala_ccode_function_call_add_argument (ccopy, _tmp86_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) local)));
			(_tmp86_ == NULL) ? NULL : (_tmp86_ = (vala_ccode_node_unref (_tmp86_), NULL));
			vala_ccode_function_call_add_argument (ccopy, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_local_variable_get_initializer (local))));
			vala_ccode_function_call_add_argument (ccopy, (ValaCCodeExpression*) size);
			_tmp87_ = NULL;
			vala_ccode_fragment_append (cfrag, (ValaCCodeNode*) (_tmp87_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) ccopy)));
			(_tmp87_ == NULL) ? NULL : (_tmp87_ = (vala_ccode_node_unref (_tmp87_), NULL));
			(sizeof_call == NULL) ? NULL : (sizeof_call = (vala_ccode_node_unref (sizeof_call), NULL));
			(size == NULL) ? NULL : (size = (vala_ccode_node_unref (size), NULL));
			(ccopy == NULL) ? NULL : (ccopy = (vala_ccode_node_unref (ccopy), NULL));
		}
		(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
	}
	_tmp88_ = FALSE;
	if (vala_local_variable_get_initializer (local) != NULL) {
		_tmp88_ = vala_code_node_get_tree_can_fail ((ValaCodeNode*) vala_local_variable_get_initializer (local));
	} else {
		_tmp88_ = FALSE;
	}
	if (_tmp88_) {
		vala_ccode_module_add_simple_check (vala_ccode_module_get_head ((ValaCCodeModule*) self), (ValaCodeNode*) vala_local_variable_get_initializer (local), cfrag);
	}
	vala_code_node_set_ccodenode ((ValaCodeNode*) local, (ValaCCodeNode*) cfrag);
	vala_symbol_set_active ((ValaSymbol*) local, TRUE);
	(rhs == NULL) ? NULL : (rhs = (vala_ccode_node_unref (rhs), NULL));
	(cfrag == NULL) ? NULL : (cfrag = (vala_ccode_node_unref (cfrag), NULL));
}


static void vala_ccode_base_module_real_visit_initializer_list (ValaCCodeModule* base, ValaInitializerList* list) {
	ValaCCodeBaseModule * self;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (list != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) list, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	if (VALA_IS_STRUCT (vala_data_type_get_data_type (vala_expression_get_target_type ((ValaExpression*) list)))) {
		ValaStruct* _tmp0_;
		ValaStruct* st;
		ValaCCodeInitializerList* clist;
		GeeList* _tmp1_;
		GeeIterator* _tmp2_;
		GeeIterator* field_it;
		/* initializer is used as struct initializer */
		_tmp0_ = NULL;
		st = (_tmp0_ = VALA_STRUCT (vala_data_type_get_data_type (vala_expression_get_target_type ((ValaExpression*) list))), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
		clist = vala_ccode_initializer_list_new ();
		_tmp1_ = NULL;
		_tmp2_ = NULL;
		field_it = (_tmp2_ = gee_iterable_iterator ((GeeIterable*) (_tmp1_ = vala_struct_get_fields (st))), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (gee_collection_object_unref (_tmp1_), NULL)), _tmp2_);
		{
			GeeList* _tmp3_;
			GeeIterator* _tmp4_;
			GeeIterator* _expr_it;
			_tmp3_ = NULL;
			_tmp4_ = NULL;
			_expr_it = (_tmp4_ = gee_iterable_iterator ((GeeIterable*) (_tmp3_ = vala_initializer_list_get_initializers (list))), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (gee_collection_object_unref (_tmp3_), NULL)), _tmp4_);
			while (gee_iterator_next (_expr_it)) {
				ValaExpression* expr;
				ValaField* field;
				ValaCCodeExpression* _tmp7_;
				ValaCCodeExpression* cexpr;
				char* ctype;
				expr = (ValaExpression*) gee_iterator_get (_expr_it);
				field = NULL;
				while (field == NULL) {
					ValaField* _tmp5_;
					gee_iterator_next (field_it);
					_tmp5_ = NULL;
					field = (_tmp5_ = (ValaField*) gee_iterator_get (field_it), (field == NULL) ? NULL : (field = (vala_code_node_unref (field), NULL)), _tmp5_);
					if (vala_field_get_binding (field) != MEMBER_BINDING_INSTANCE) {
						ValaField* _tmp6_;
						/* we only initialize instance fields*/
						_tmp6_ = NULL;
						field = (_tmp6_ = NULL, (field == NULL) ? NULL : (field = (vala_code_node_unref (field), NULL)), _tmp6_);
					}
				}
				_tmp7_ = NULL;
				cexpr = (_tmp7_ = VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) expr)), (_tmp7_ == NULL) ? NULL : vala_ccode_node_ref (_tmp7_));
				ctype = vala_field_get_ctype (field);
				if (ctype != NULL) {
					ValaCCodeExpression* _tmp8_;
					_tmp8_ = NULL;
					cexpr = (_tmp8_ = (ValaCCodeExpression*) vala_ccode_cast_expression_new (cexpr, ctype), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), _tmp8_);
				}
				vala_ccode_initializer_list_append (clist, cexpr);
				(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
				(field == NULL) ? NULL : (field = (vala_code_node_unref (field), NULL));
				(cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL));
				ctype = (g_free (ctype), NULL);
			}
			(_expr_it == NULL) ? NULL : (_expr_it = (gee_collection_object_unref (_expr_it), NULL));
		}
		vala_code_node_set_ccodenode ((ValaCodeNode*) list, (ValaCCodeNode*) clist);
		(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		(clist == NULL) ? NULL : (clist = (vala_ccode_node_unref (clist), NULL));
		(field_it == NULL) ? NULL : (field_it = (gee_collection_object_unref (field_it), NULL));
	} else {
		ValaCCodeInitializerList* clist;
		clist = vala_ccode_initializer_list_new ();
		{
			GeeList* _tmp9_;
			GeeIterator* _tmp10_;
			GeeIterator* _expr_it;
			_tmp9_ = NULL;
			_tmp10_ = NULL;
			_expr_it = (_tmp10_ = gee_iterable_iterator ((GeeIterable*) (_tmp9_ = vala_initializer_list_get_initializers (list))), (_tmp9_ == NULL) ? NULL : (_tmp9_ = (gee_collection_object_unref (_tmp9_), NULL)), _tmp10_);
			while (gee_iterator_next (_expr_it)) {
				ValaExpression* expr;
				expr = (ValaExpression*) gee_iterator_get (_expr_it);
				vala_ccode_initializer_list_append (clist, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) expr)));
				(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
			}
			(_expr_it == NULL) ? NULL : (_expr_it = (gee_collection_object_unref (_expr_it), NULL));
		}
		vala_code_node_set_ccodenode ((ValaCodeNode*) list, (ValaCCodeNode*) clist);
		(clist == NULL) ? NULL : (clist = (vala_ccode_node_unref (clist), NULL));
	}
}


ValaLocalVariable* vala_ccode_base_module_get_temp_variable (ValaCCodeBaseModule* self, ValaDataType* type, gboolean value_owned, ValaCodeNode* node_reference) {
	ValaDataType* var_type;
	char* _tmp0_;
	ValaLocalVariable* _tmp1_;
	ValaLocalVariable* local;
	ValaLocalVariable* _tmp2_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type != NULL, NULL);
	var_type = vala_data_type_copy (type);
	vala_data_type_set_value_owned (var_type, value_owned);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	local = (_tmp1_ = vala_local_variable_new (var_type, _tmp0_ = g_strdup_printf ("_tmp%d_", self->next_temp_var_id), NULL, NULL), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	if (node_reference != NULL) {
		vala_code_node_set_source_reference ((ValaCodeNode*) local, vala_code_node_get_source_reference (node_reference));
	}
	self->next_temp_var_id++;
	_tmp2_ = NULL;
	return (_tmp2_ = local, (var_type == NULL) ? NULL : (var_type = (vala_code_node_unref (var_type), NULL)), _tmp2_);
}


static ValaCCodeExpression* vala_ccode_base_module_get_type_id_expression (ValaCCodeBaseModule* self, ValaDataType* type) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type != NULL, NULL);
	if (VALA_IS_GENERIC_TYPE (type)) {
		char* _tmp0_;
		char* _tmp1_;
		char* var_name;
		ValaCCodeMemberAccess* _tmp3_;
		ValaCCodeIdentifier* _tmp2_;
		ValaCCodeExpression* _tmp4_;
		ValaCCodeExpression* _tmp5_;
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		var_name = (_tmp1_ = g_strdup_printf ("%s_type", _tmp0_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) vala_data_type_get_type_parameter (type)), -1)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		_tmp4_ = NULL;
		_tmp5_ = NULL;
		return (_tmp5_ = (_tmp4_ = (ValaCCodeExpression*) vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp3_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp2_ = vala_ccode_identifier_new ("self")), "priv")), var_name), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL)), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_ccode_node_unref (_tmp2_), NULL)), _tmp4_), var_name = (g_free (var_name), NULL), _tmp5_);
	} else {
		char* type_id;
		ValaCCodeExpression* _tmp7_;
		type_id = vala_data_type_get_type_id (type);
		if (type_id == NULL) {
			char* _tmp6_;
			_tmp6_ = NULL;
			type_id = (_tmp6_ = g_strdup ("G_TYPE_INVALID"), type_id = (g_free (type_id), NULL), _tmp6_);
		} else {
			vala_ccode_base_module_generate_type_declaration (self, type, self->source_declarations);
		}
		_tmp7_ = NULL;
		return (_tmp7_ = (ValaCCodeExpression*) vala_ccode_identifier_new (type_id), type_id = (g_free (type_id), NULL), _tmp7_);
	}
}


static ValaCCodeExpression* vala_ccode_base_module_real_get_dup_func_expression (ValaCCodeBaseModule* self, ValaDataType* type, ValaSourceReference* source_reference) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type != NULL, NULL);
	if (VALA_IS_ERROR_TYPE (type)) {
		return (ValaCCodeExpression*) vala_ccode_identifier_new ("g_error_copy");
	} else {
		if (vala_data_type_get_data_type (type) != NULL) {
			char* dup_function;
			ValaClass* _tmp2_;
			ValaTypeSymbol* _tmp1_;
			ValaClass* cl;
			ValaCCodeExpression* _tmp17_;
			dup_function = NULL;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			cl = (_tmp2_ = (_tmp1_ = vala_data_type_get_data_type (type), VALA_IS_CLASS (_tmp1_) ? ((ValaClass*) _tmp1_) : NULL), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
			if (vala_typesymbol_is_reference_counting (vala_data_type_get_data_type (type))) {
				char* _tmp3_;
				gboolean _tmp4_;
				_tmp3_ = NULL;
				dup_function = (_tmp3_ = vala_typesymbol_get_ref_function (vala_data_type_get_data_type (type)), dup_function = (g_free (dup_function), NULL), _tmp3_);
				_tmp4_ = FALSE;
				if (VALA_IS_INTERFACE (vala_data_type_get_data_type (type))) {
					_tmp4_ = dup_function == NULL;
				} else {
					_tmp4_ = FALSE;
				}
				if (_tmp4_) {
					char* _tmp6_;
					char* _tmp5_;
					ValaCCodeExpression* _tmp7_;
					_tmp6_ = NULL;
					_tmp5_ = NULL;
					vala_report_error (source_reference, _tmp6_ = g_strdup_printf ("missing class prerequisite for interface `%s', add GLib.Object to interface declaration if unsure", _tmp5_ = vala_symbol_get_full_name ((ValaSymbol*) vala_data_type_get_data_type (type))));
					_tmp6_ = (g_free (_tmp6_), NULL);
					_tmp5_ = (g_free (_tmp5_), NULL);
					_tmp7_ = NULL;
					return (_tmp7_ = NULL, dup_function = (g_free (dup_function), NULL), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp7_);
				}
			} else {
				gboolean _tmp8_;
				_tmp8_ = FALSE;
				if (cl != NULL) {
					_tmp8_ = vala_class_get_is_immutable (cl);
				} else {
					_tmp8_ = FALSE;
				}
				if (_tmp8_) {
					char* _tmp9_;
					/* allow duplicates of immutable instances as for example strings*/
					_tmp9_ = NULL;
					dup_function = (_tmp9_ = vala_typesymbol_get_dup_function (vala_data_type_get_data_type (type)), dup_function = (g_free (dup_function), NULL), _tmp9_);
					if (dup_function == NULL) {
						char* _tmp10_;
						_tmp10_ = NULL;
						dup_function = (_tmp10_ = g_strdup (""), dup_function = (g_free (dup_function), NULL), _tmp10_);
					}
				} else {
					if (VALA_IS_VALUE_TYPE (type)) {
						char* _tmp11_;
						gboolean _tmp12_;
						_tmp11_ = NULL;
						dup_function = (_tmp11_ = vala_typesymbol_get_dup_function (vala_data_type_get_data_type (type)), dup_function = (g_free (dup_function), NULL), _tmp11_);
						_tmp12_ = FALSE;
						if (dup_function == NULL) {
							_tmp12_ = vala_data_type_get_nullable (type);
						} else {
							_tmp12_ = FALSE;
						}
						if (_tmp12_) {
							char* _tmp13_;
							_tmp13_ = NULL;
							dup_function = (_tmp13_ = vala_ccode_base_module_generate_struct_dup_wrapper (self, VALA_VALUE_TYPE (type)), dup_function = (g_free (dup_function), NULL), _tmp13_);
						} else {
							if (dup_function == NULL) {
								char* _tmp14_;
								_tmp14_ = NULL;
								dup_function = (_tmp14_ = g_strdup (""), dup_function = (g_free (dup_function), NULL), _tmp14_);
							}
						}
					} else {
						char* _tmp15_;
						ValaCCodeExpression* _tmp16_;
						/* duplicating non-reference counted objects may cause side-effects (and performance issues)*/
						_tmp15_ = NULL;
						vala_report_error (source_reference, _tmp15_ = g_strdup_printf ("duplicating %s instance, use unowned variable or explicitly invoke copy method", vala_symbol_get_name ((ValaSymbol*) vala_data_type_get_data_type (type))));
						_tmp15_ = (g_free (_tmp15_), NULL);
						_tmp16_ = NULL;
						return (_tmp16_ = NULL, dup_function = (g_free (dup_function), NULL), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp16_);
					}
				}
			}
			_tmp17_ = NULL;
			return (_tmp17_ = (ValaCCodeExpression*) vala_ccode_identifier_new (dup_function), dup_function = (g_free (dup_function), NULL), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp17_);
		} else {
			gboolean _tmp18_;
			_tmp18_ = FALSE;
			if (vala_data_type_get_type_parameter (type) != NULL) {
				_tmp18_ = VALA_IS_CLASS (self->current_type_symbol);
			} else {
				_tmp18_ = FALSE;
			}
			if (_tmp18_) {
				char* _tmp19_;
				char* _tmp20_;
				char* func_name;
				ValaCCodeMemberAccess* _tmp22_;
				ValaCCodeIdentifier* _tmp21_;
				ValaCCodeExpression* _tmp23_;
				ValaCCodeExpression* _tmp24_;
				_tmp19_ = NULL;
				_tmp20_ = NULL;
				func_name = (_tmp20_ = g_strdup_printf ("%s_dup_func", _tmp19_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) vala_data_type_get_type_parameter (type)), -1)), _tmp19_ = (g_free (_tmp19_), NULL), _tmp20_);
				_tmp22_ = NULL;
				_tmp21_ = NULL;
				_tmp23_ = NULL;
				_tmp24_ = NULL;
				return (_tmp24_ = (_tmp23_ = (ValaCCodeExpression*) vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp22_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp21_ = vala_ccode_identifier_new ("self")), "priv")), func_name), (_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL)), (_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL)), _tmp23_), func_name = (g_free (func_name), NULL), _tmp24_);
			} else {
				if (VALA_IS_POINTER_TYPE (type)) {
					ValaPointerType* _tmp25_;
					ValaPointerType* pointer_type;
					ValaCCodeExpression* _tmp26_;
					_tmp25_ = NULL;
					pointer_type = (_tmp25_ = VALA_POINTER_TYPE (type), (_tmp25_ == NULL) ? NULL : vala_code_node_ref (_tmp25_));
					_tmp26_ = NULL;
					return (_tmp26_ = vala_ccode_base_module_get_dup_func_expression (self, vala_pointer_type_get_base_type (pointer_type), source_reference), (pointer_type == NULL) ? NULL : (pointer_type = (vala_code_node_unref (pointer_type), NULL)), _tmp26_);
				} else {
					return (ValaCCodeExpression*) vala_ccode_constant_new ("NULL");
				}
			}
		}
	}
}


ValaCCodeExpression* vala_ccode_base_module_get_dup_func_expression (ValaCCodeBaseModule* self, ValaDataType* type, ValaSourceReference* source_reference) {
	return VALA_CCODE_BASE_MODULE_GET_CLASS (self)->get_dup_func_expression (self, type, source_reference);
}


static char* vala_ccode_base_module_generate_struct_dup_wrapper (ValaCCodeBaseModule* self, ValaValueType* value_type) {
	char* _tmp0_;
	char* _tmp1_;
	char* dup_func;
	char* _tmp3_;
	ValaCCodeFunction* _tmp4_;
	ValaCCodeFunction* function;
	ValaCCodeFormalParameter* _tmp6_;
	char* _tmp5_;
	ValaCCodeBlock* block;
	ValaCCodeFunction* _tmp43_;
	char* _tmp44_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (value_type != NULL, NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	dup_func = (_tmp1_ = g_strdup_printf ("_%sdup", _tmp0_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) vala_value_type_get_type_symbol (value_type))), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	if (!vala_ccode_base_module_add_wrapper (self, dup_func)) {
		/* wrapper already defined*/
		return dup_func;
	}
	/* declaration*/
	_tmp3_ = NULL;
	_tmp4_ = NULL;
	function = (_tmp4_ = vala_ccode_function_new (dup_func, _tmp3_ = vala_data_type_get_cname ((ValaDataType*) value_type)), _tmp3_ = (g_free (_tmp3_), NULL), _tmp4_);
	vala_ccode_function_set_modifiers (function, VALA_CCODE_MODIFIERS_STATIC);
	_tmp6_ = NULL;
	_tmp5_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp6_ = vala_ccode_formal_parameter_new ("self", _tmp5_ = vala_data_type_get_cname ((ValaDataType*) value_type)));
	(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
	_tmp5_ = (g_free (_tmp5_), NULL);
	/* definition*/
	block = vala_ccode_block_new ();
	if (vala_value_type_get_type_symbol (value_type) == VALA_TYPESYMBOL (self->gvalue_type)) {
		ValaCCodeIdentifier* _tmp7_;
		ValaCCodeFunctionCall* _tmp8_;
		ValaCCodeFunctionCall* dup_call;
		ValaCCodeIdentifier* _tmp9_;
		ValaCCodeIdentifier* _tmp10_;
		ValaCCodeReturnStatement* _tmp11_;
		_tmp7_ = NULL;
		_tmp8_ = NULL;
		dup_call = (_tmp8_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp7_ = vala_ccode_identifier_new ("g_boxed_copy"))), (_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL)), _tmp8_);
		_tmp9_ = NULL;
		vala_ccode_function_call_add_argument (dup_call, (ValaCCodeExpression*) (_tmp9_ = vala_ccode_identifier_new ("G_TYPE_VALUE")));
		(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
		_tmp10_ = NULL;
		vala_ccode_function_call_add_argument (dup_call, (ValaCCodeExpression*) (_tmp10_ = vala_ccode_identifier_new ("self")));
		(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
		_tmp11_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp11_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) dup_call)));
		(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
		(dup_call == NULL) ? NULL : (dup_call = (vala_ccode_node_unref (dup_call), NULL));
	} else {
		char* _tmp12_;
		ValaCCodeDeclaration* _tmp13_;
		ValaCCodeDeclaration* _cdecl_;
		ValaCCodeVariableDeclarator* _tmp14_;
		ValaCCodeIdentifier* _tmp15_;
		ValaCCodeFunctionCall* _tmp16_;
		ValaCCodeFunctionCall* creation_call;
		ValaCCodeConstant* _tmp18_;
		char* _tmp17_;
		ValaCCodeConstant* _tmp19_;
		ValaCCodeExpressionStatement* _tmp22_;
		ValaCCodeAssignment* _tmp21_;
		ValaCCodeIdentifier* _tmp20_;
		ValaStruct* _tmp24_;
		ValaTypeSymbol* _tmp23_;
		ValaStruct* st;
		gboolean _tmp25_;
		ValaCCodeReturnStatement* _tmp42_;
		ValaCCodeIdentifier* _tmp41_;
		_tmp12_ = NULL;
		_tmp13_ = NULL;
		_cdecl_ = (_tmp13_ = vala_ccode_declaration_new (_tmp12_ = vala_data_type_get_cname ((ValaDataType*) value_type)), _tmp12_ = (g_free (_tmp12_), NULL), _tmp13_);
		_tmp14_ = NULL;
		vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp14_ = vala_ccode_variable_declarator_new ("dup", NULL, NULL)));
		(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) _cdecl_);
		_tmp15_ = NULL;
		_tmp16_ = NULL;
		creation_call = (_tmp16_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp15_ = vala_ccode_identifier_new ("g_new0"))), (_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL)), _tmp16_);
		_tmp18_ = NULL;
		_tmp17_ = NULL;
		vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp18_ = vala_ccode_constant_new (_tmp17_ = vala_typesymbol_get_cname (vala_data_type_get_data_type ((ValaDataType*) value_type), FALSE))));
		(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
		_tmp17_ = (g_free (_tmp17_), NULL);
		_tmp19_ = NULL;
		vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp19_ = vala_ccode_constant_new ("1")));
		(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
		_tmp22_ = NULL;
		_tmp21_ = NULL;
		_tmp20_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp22_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp21_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp20_ = vala_ccode_identifier_new ("dup")), (ValaCCodeExpression*) creation_call, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
		(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL));
		(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL));
		(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL));
		_tmp24_ = NULL;
		_tmp23_ = NULL;
		st = (_tmp24_ = (_tmp23_ = vala_data_type_get_data_type ((ValaDataType*) value_type), VALA_IS_STRUCT (_tmp23_) ? ((ValaStruct*) _tmp23_) : NULL), (_tmp24_ == NULL) ? NULL : vala_code_node_ref (_tmp24_));
		_tmp25_ = FALSE;
		if (st != NULL) {
			_tmp25_ = vala_struct_is_disposable (st);
		} else {
			_tmp25_ = FALSE;
		}
		if (_tmp25_) {
			ValaCCodeIdentifier* _tmp27_;
			char* _tmp26_;
			ValaCCodeFunctionCall* _tmp28_;
			ValaCCodeFunctionCall* copy_call;
			ValaCCodeIdentifier* _tmp29_;
			ValaCCodeIdentifier* _tmp30_;
			ValaCCodeExpressionStatement* _tmp31_;
			_tmp27_ = NULL;
			_tmp26_ = NULL;
			_tmp28_ = NULL;
			copy_call = (_tmp28_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp27_ = vala_ccode_identifier_new (_tmp26_ = vala_typesymbol_get_copy_function ((ValaTypeSymbol*) st)))), (_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL)), _tmp26_ = (g_free (_tmp26_), NULL), _tmp28_);
			_tmp29_ = NULL;
			vala_ccode_function_call_add_argument (copy_call, (ValaCCodeExpression*) (_tmp29_ = vala_ccode_identifier_new ("self")));
			(_tmp29_ == NULL) ? NULL : (_tmp29_ = (vala_ccode_node_unref (_tmp29_), NULL));
			_tmp30_ = NULL;
			vala_ccode_function_call_add_argument (copy_call, (ValaCCodeExpression*) (_tmp30_ = vala_ccode_identifier_new ("dup")));
			(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL));
			_tmp31_ = NULL;
			vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp31_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) copy_call)));
			(_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_ccode_node_unref (_tmp31_), NULL));
			(copy_call == NULL) ? NULL : (copy_call = (vala_ccode_node_unref (copy_call), NULL));
		} else {
			ValaCCodeIdentifier* _tmp32_;
			ValaCCodeFunctionCall* _tmp33_;
			ValaCCodeFunctionCall* sizeof_call;
			ValaCCodeConstant* _tmp35_;
			char* _tmp34_;
			ValaCCodeIdentifier* _tmp36_;
			ValaCCodeFunctionCall* _tmp37_;
			ValaCCodeFunctionCall* copy_call;
			ValaCCodeIdentifier* _tmp38_;
			ValaCCodeIdentifier* _tmp39_;
			ValaCCodeExpressionStatement* _tmp40_;
			vala_ccode_declaration_space_add_include (self->source_declarations, "string.h", FALSE);
			_tmp32_ = NULL;
			_tmp33_ = NULL;
			sizeof_call = (_tmp33_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp32_ = vala_ccode_identifier_new ("sizeof"))), (_tmp32_ == NULL) ? NULL : (_tmp32_ = (vala_ccode_node_unref (_tmp32_), NULL)), _tmp33_);
			_tmp35_ = NULL;
			_tmp34_ = NULL;
			vala_ccode_function_call_add_argument (sizeof_call, (ValaCCodeExpression*) (_tmp35_ = vala_ccode_constant_new (_tmp34_ = vala_typesymbol_get_cname (vala_data_type_get_data_type ((ValaDataType*) value_type), FALSE))));
			(_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_ccode_node_unref (_tmp35_), NULL));
			_tmp34_ = (g_free (_tmp34_), NULL);
			_tmp36_ = NULL;
			_tmp37_ = NULL;
			copy_call = (_tmp37_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp36_ = vala_ccode_identifier_new ("memcpy"))), (_tmp36_ == NULL) ? NULL : (_tmp36_ = (vala_ccode_node_unref (_tmp36_), NULL)), _tmp37_);
			_tmp38_ = NULL;
			vala_ccode_function_call_add_argument (copy_call, (ValaCCodeExpression*) (_tmp38_ = vala_ccode_identifier_new ("dup")));
			(_tmp38_ == NULL) ? NULL : (_tmp38_ = (vala_ccode_node_unref (_tmp38_), NULL));
			_tmp39_ = NULL;
			vala_ccode_function_call_add_argument (copy_call, (ValaCCodeExpression*) (_tmp39_ = vala_ccode_identifier_new ("self")));
			(_tmp39_ == NULL) ? NULL : (_tmp39_ = (vala_ccode_node_unref (_tmp39_), NULL));
			vala_ccode_function_call_add_argument (copy_call, (ValaCCodeExpression*) sizeof_call);
			_tmp40_ = NULL;
			vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp40_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) copy_call)));
			(_tmp40_ == NULL) ? NULL : (_tmp40_ = (vala_ccode_node_unref (_tmp40_), NULL));
			(sizeof_call == NULL) ? NULL : (sizeof_call = (vala_ccode_node_unref (sizeof_call), NULL));
			(copy_call == NULL) ? NULL : (copy_call = (vala_ccode_node_unref (copy_call), NULL));
		}
		_tmp42_ = NULL;
		_tmp41_ = NULL;
		vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp42_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp41_ = vala_ccode_identifier_new ("dup")))));
		(_tmp42_ == NULL) ? NULL : (_tmp42_ = (vala_ccode_node_unref (_tmp42_), NULL));
		(_tmp41_ == NULL) ? NULL : (_tmp41_ = (vala_ccode_node_unref (_tmp41_), NULL));
		(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
		(creation_call == NULL) ? NULL : (creation_call = (vala_ccode_node_unref (creation_call), NULL));
		(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
	}
	/* append to file*/
	_tmp43_ = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (self->source_declarations, (ValaCCodeNode*) (_tmp43_ = vala_ccode_function_copy (function)));
	(_tmp43_ == NULL) ? NULL : (_tmp43_ = (vala_ccode_node_unref (_tmp43_), NULL));
	vala_ccode_function_set_block (function, block);
	vala_ccode_fragment_append (self->source_type_member_definition, (ValaCCodeNode*) function);
	_tmp44_ = NULL;
	return (_tmp44_ = dup_func, (function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL)), (block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL)), _tmp44_);
}


ValaCCodeExpression* vala_ccode_base_module_get_destroy_func_expression (ValaCCodeBaseModule* self, ValaDataType* type) {
	gboolean _tmp0_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type != NULL, NULL);
	_tmp0_ = FALSE;
	if (vala_data_type_get_data_type (type) == VALA_TYPESYMBOL (self->glist_type)) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = vala_data_type_get_data_type (type) == VALA_TYPESYMBOL (self->gslist_type);
	}
	if (_tmp0_) {
		gboolean elements_require_free;
		ValaCCodeExpression* element_destroy_func_expression;
		gboolean _tmp4_;
		/* create wrapper function to free list elements if necessary*/
		elements_require_free = FALSE;
		element_destroy_func_expression = NULL;
		{
			GeeList* _tmp1_;
			GeeIterator* _tmp2_;
			GeeIterator* _type_arg_it;
			_tmp1_ = NULL;
			_tmp2_ = NULL;
			_type_arg_it = (_tmp2_ = gee_iterable_iterator ((GeeIterable*) (_tmp1_ = vala_data_type_get_type_arguments (type))), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (gee_collection_object_unref (_tmp1_), NULL)), _tmp2_);
			while (gee_iterator_next (_type_arg_it)) {
				ValaDataType* type_arg;
				type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
				elements_require_free = vala_ccode_base_module_requires_destroy (self, type_arg);
				if (elements_require_free) {
					ValaCCodeExpression* _tmp3_;
					_tmp3_ = NULL;
					element_destroy_func_expression = (_tmp3_ = vala_ccode_base_module_get_destroy_func_expression (self, type_arg), (element_destroy_func_expression == NULL) ? NULL : (element_destroy_func_expression = (vala_ccode_node_unref (element_destroy_func_expression), NULL)), _tmp3_);
				}
				(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
			}
			(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
		}
		_tmp4_ = FALSE;
		if (elements_require_free) {
			_tmp4_ = VALA_IS_CCODE_IDENTIFIER (element_destroy_func_expression);
		} else {
			_tmp4_ = FALSE;
		}
		if (_tmp4_) {
			char* _tmp5_;
			ValaCCodeExpression* _tmp6_;
			ValaCCodeExpression* _tmp7_;
			_tmp5_ = NULL;
			_tmp6_ = NULL;
			_tmp7_ = NULL;
			return (_tmp7_ = (_tmp6_ = (ValaCCodeExpression*) vala_ccode_identifier_new (_tmp5_ = vala_ccode_base_module_generate_glist_free_wrapper (self, type, VALA_CCODE_IDENTIFIER (element_destroy_func_expression))), _tmp5_ = (g_free (_tmp5_), NULL), _tmp6_), (element_destroy_func_expression == NULL) ? NULL : (element_destroy_func_expression = (vala_ccode_node_unref (element_destroy_func_expression), NULL)), _tmp7_);
		} else {
			char* _tmp8_;
			ValaCCodeExpression* _tmp9_;
			ValaCCodeExpression* _tmp10_;
			_tmp8_ = NULL;
			_tmp9_ = NULL;
			_tmp10_ = NULL;
			return (_tmp10_ = (_tmp9_ = (ValaCCodeExpression*) vala_ccode_identifier_new (_tmp8_ = vala_typesymbol_get_free_function (vala_data_type_get_data_type (type))), _tmp8_ = (g_free (_tmp8_), NULL), _tmp9_), (element_destroy_func_expression == NULL) ? NULL : (element_destroy_func_expression = (vala_ccode_node_unref (element_destroy_func_expression), NULL)), _tmp10_);
		}
		(element_destroy_func_expression == NULL) ? NULL : (element_destroy_func_expression = (vala_ccode_node_unref (element_destroy_func_expression), NULL));
	} else {
		if (VALA_IS_ERROR_TYPE (type)) {
			return (ValaCCodeExpression*) vala_ccode_identifier_new ("g_error_free");
		} else {
			if (vala_data_type_get_data_type (type) != NULL) {
				char* unref_function;
				ValaCCodeExpression* _tmp23_;
				unref_function = NULL;
				if (VALA_IS_REFERENCE_TYPE (type)) {
					if (vala_typesymbol_is_reference_counting (vala_data_type_get_data_type (type))) {
						char* _tmp12_;
						gboolean _tmp13_;
						_tmp12_ = NULL;
						unref_function = (_tmp12_ = vala_typesymbol_get_unref_function (vala_data_type_get_data_type (type)), unref_function = (g_free (unref_function), NULL), _tmp12_);
						_tmp13_ = FALSE;
						if (VALA_IS_INTERFACE (vala_data_type_get_data_type (type))) {
							_tmp13_ = unref_function == NULL;
						} else {
							_tmp13_ = FALSE;
						}
						if (_tmp13_) {
							char* _tmp15_;
							char* _tmp14_;
							ValaCCodeExpression* _tmp16_;
							_tmp15_ = NULL;
							_tmp14_ = NULL;
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) type), _tmp15_ = g_strdup_printf ("missing class prerequisite for interface `%s', add GLib.Object to interface declaration if unsure", _tmp14_ = vala_symbol_get_full_name ((ValaSymbol*) vala_data_type_get_data_type (type))));
							_tmp15_ = (g_free (_tmp15_), NULL);
							_tmp14_ = (g_free (_tmp14_), NULL);
							_tmp16_ = NULL;
							return (_tmp16_ = NULL, unref_function = (g_free (unref_function), NULL), _tmp16_);
						}
					} else {
						char* _tmp17_;
						_tmp17_ = NULL;
						unref_function = (_tmp17_ = vala_typesymbol_get_free_function (vala_data_type_get_data_type (type)), unref_function = (g_free (unref_function), NULL), _tmp17_);
					}
				} else {
					if (vala_data_type_get_nullable (type)) {
						char* _tmp18_;
						_tmp18_ = NULL;
						unref_function = (_tmp18_ = vala_typesymbol_get_free_function (vala_data_type_get_data_type (type)), unref_function = (g_free (unref_function), NULL), _tmp18_);
						if (unref_function == NULL) {
							char* _tmp19_;
							_tmp19_ = NULL;
							unref_function = (_tmp19_ = g_strdup ("g_free"), unref_function = (g_free (unref_function), NULL), _tmp19_);
						}
					} else {
						ValaStruct* _tmp20_;
						ValaStruct* st;
						char* _tmp21_;
						_tmp20_ = NULL;
						st = (_tmp20_ = VALA_STRUCT (vala_data_type_get_data_type (type)), (_tmp20_ == NULL) ? NULL : vala_code_node_ref (_tmp20_));
						_tmp21_ = NULL;
						unref_function = (_tmp21_ = vala_typesymbol_get_destroy_function ((ValaTypeSymbol*) st), unref_function = (g_free (unref_function), NULL), _tmp21_);
						(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
					}
				}
				if (unref_function == NULL) {
					ValaCCodeExpression* _tmp22_;
					_tmp22_ = NULL;
					return (_tmp22_ = (ValaCCodeExpression*) vala_ccode_constant_new ("NULL"), unref_function = (g_free (unref_function), NULL), _tmp22_);
				}
				_tmp23_ = NULL;
				return (_tmp23_ = (ValaCCodeExpression*) vala_ccode_identifier_new (unref_function), unref_function = (g_free (unref_function), NULL), _tmp23_);
			} else {
				gboolean _tmp24_;
				_tmp24_ = FALSE;
				if (vala_data_type_get_type_parameter (type) != NULL) {
					_tmp24_ = VALA_IS_CLASS (self->current_type_symbol);
				} else {
					_tmp24_ = FALSE;
				}
				if (_tmp24_) {
					char* _tmp25_;
					char* _tmp26_;
					char* func_name;
					ValaCCodeMemberAccess* _tmp28_;
					ValaCCodeIdentifier* _tmp27_;
					ValaCCodeExpression* _tmp29_;
					ValaCCodeExpression* _tmp30_;
					_tmp25_ = NULL;
					_tmp26_ = NULL;
					func_name = (_tmp26_ = g_strdup_printf ("%s_destroy_func", _tmp25_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) vala_data_type_get_type_parameter (type)), -1)), _tmp25_ = (g_free (_tmp25_), NULL), _tmp26_);
					_tmp28_ = NULL;
					_tmp27_ = NULL;
					_tmp29_ = NULL;
					_tmp30_ = NULL;
					return (_tmp30_ = (_tmp29_ = (ValaCCodeExpression*) vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp28_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp27_ = vala_ccode_identifier_new ("self")), "priv")), func_name), (_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_ccode_node_unref (_tmp28_), NULL)), (_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL)), _tmp29_), func_name = (g_free (func_name), NULL), _tmp30_);
				} else {
					if (VALA_IS_ARRAY_TYPE (type)) {
						return (ValaCCodeExpression*) vala_ccode_identifier_new ("g_free");
					} else {
						if (VALA_IS_POINTER_TYPE (type)) {
							return (ValaCCodeExpression*) vala_ccode_identifier_new ("g_free");
						} else {
							return (ValaCCodeExpression*) vala_ccode_constant_new ("NULL");
						}
					}
				}
			}
		}
	}
}


static char* vala_ccode_base_module_generate_glist_free_wrapper (ValaCCodeBaseModule* self, ValaDataType* list_type, ValaCCodeIdentifier* element_destroy_func_expression) {
	char* _tmp0_;
	char* _tmp1_;
	char* destroy_func;
	ValaCCodeFunction* function;
	ValaCCodeFormalParameter* _tmp4_;
	char* _tmp3_;
	ValaCCodeBlock* block;
	ValaCCodeFunctionCall* element_free_call;
	ValaCCodeIdentifier* _tmp9_;
	ValaCCodeCastExpression* _tmp10_;
	ValaCCodeConstant* _tmp11_;
	ValaCCodeExpressionStatement* _tmp12_;
	ValaCCodeIdentifier* _tmp14_;
	char* _tmp13_;
	ValaCCodeFunctionCall* _tmp15_;
	ValaCCodeFunctionCall* cfreecall;
	ValaCCodeIdentifier* _tmp16_;
	ValaCCodeExpressionStatement* _tmp17_;
	ValaCCodeFunction* _tmp18_;
	char* _tmp19_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (list_type != NULL, NULL);
	g_return_val_if_fail (element_destroy_func_expression != NULL, NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	destroy_func = (_tmp1_ = g_strdup_printf ("_%s_%s", _tmp0_ = vala_typesymbol_get_free_function (vala_data_type_get_data_type (list_type)), vala_ccode_identifier_get_name (element_destroy_func_expression)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	if (!vala_ccode_base_module_add_wrapper (self, destroy_func)) {
		/* wrapper already defined*/
		return destroy_func;
	}
	/* declaration*/
	function = vala_ccode_function_new (destroy_func, "void");
	vala_ccode_function_set_modifiers (function, VALA_CCODE_MODIFIERS_STATIC);
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	vala_ccode_function_add_parameter (function, _tmp4_ = vala_ccode_formal_parameter_new ("self", _tmp3_ = vala_data_type_get_cname (list_type)));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	_tmp3_ = (g_free (_tmp3_), NULL);
	/* definition*/
	block = vala_ccode_block_new ();
	element_free_call = NULL;
	if (vala_data_type_get_data_type (list_type) == VALA_TYPESYMBOL (self->glist_type)) {
		ValaCCodeFunctionCall* _tmp6_;
		ValaCCodeIdentifier* _tmp5_;
		_tmp6_ = NULL;
		_tmp5_ = NULL;
		element_free_call = (_tmp6_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp5_ = vala_ccode_identifier_new ("g_list_foreach"))), (element_free_call == NULL) ? NULL : (element_free_call = (vala_ccode_node_unref (element_free_call), NULL)), _tmp6_);
		(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
	} else {
		ValaCCodeFunctionCall* _tmp8_;
		ValaCCodeIdentifier* _tmp7_;
		_tmp8_ = NULL;
		_tmp7_ = NULL;
		element_free_call = (_tmp8_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp7_ = vala_ccode_identifier_new ("g_slist_foreach"))), (element_free_call == NULL) ? NULL : (element_free_call = (vala_ccode_node_unref (element_free_call), NULL)), _tmp8_);
		(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
	}
	_tmp9_ = NULL;
	vala_ccode_function_call_add_argument (element_free_call, (ValaCCodeExpression*) (_tmp9_ = vala_ccode_identifier_new ("self")));
	(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
	_tmp10_ = NULL;
	vala_ccode_function_call_add_argument (element_free_call, (ValaCCodeExpression*) (_tmp10_ = vala_ccode_cast_expression_new ((ValaCCodeExpression*) element_destroy_func_expression, "GFunc")));
	(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
	_tmp11_ = NULL;
	vala_ccode_function_call_add_argument (element_free_call, (ValaCCodeExpression*) (_tmp11_ = vala_ccode_constant_new ("NULL")));
	(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
	_tmp12_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp12_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) element_free_call)));
	(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
	_tmp14_ = NULL;
	_tmp13_ = NULL;
	_tmp15_ = NULL;
	cfreecall = (_tmp15_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp14_ = vala_ccode_identifier_new (_tmp13_ = vala_typesymbol_get_free_function (vala_data_type_get_data_type (list_type))))), (_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL)), _tmp13_ = (g_free (_tmp13_), NULL), _tmp15_);
	_tmp16_ = NULL;
	vala_ccode_function_call_add_argument (cfreecall, (ValaCCodeExpression*) (_tmp16_ = vala_ccode_identifier_new ("self")));
	(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL));
	_tmp17_ = NULL;
	vala_ccode_block_add_statement (block, (ValaCCodeNode*) (_tmp17_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) cfreecall)));
	(_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_ccode_node_unref (_tmp17_), NULL));
	/* append to file*/
	_tmp18_ = NULL;
	vala_ccode_declaration_space_add_type_member_declaration (self->source_declarations, (ValaCCodeNode*) (_tmp18_ = vala_ccode_function_copy (function)));
	(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
	vala_ccode_function_set_block (function, block);
	vala_ccode_fragment_append (self->source_type_member_definition, (ValaCCodeNode*) function);
	_tmp19_ = NULL;
	return (_tmp19_ = destroy_func, (function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL)), (block == NULL) ? NULL : (block = (vala_ccode_node_unref (block), NULL)), (element_free_call == NULL) ? NULL : (element_free_call = (vala_ccode_node_unref (element_free_call), NULL)), (cfreecall == NULL) ? NULL : (cfreecall = (vala_ccode_node_unref (cfreecall), NULL)), _tmp19_);
}


static ValaCCodeExpression* vala_ccode_base_module_real_get_unref_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* cvar, ValaDataType* type, ValaExpression* expr) {
	ValaCCodeExpression* _tmp0_;
	ValaCCodeFunctionCall* _tmp1_;
	ValaCCodeFunctionCall* ccall;
	gboolean _tmp2_;
	ValaCCodeConstant* _tmp12_;
	ValaCCodeBinaryExpression* _tmp13_;
	ValaCCodeBinaryExpression* cisnull;
	ValaCCodeCommaExpression* ccomma;
	gboolean _tmp20_;
	gboolean _tmp21_;
	gboolean _tmp22_;
	ValaCCodeConstant* _tmp32_;
	ValaCCodeAssignment* cassign;
	gboolean _tmp33_;
	gboolean _tmp34_;
	gboolean uses_gfree;
	gboolean _tmp36_;
	ValaCCodeConstant* _tmp38_;
	ValaCCodeExpression* _tmp39_;
	ValaCCodeExpression* _tmp40_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (cvar != NULL, NULL);
	g_return_val_if_fail (type != NULL, NULL);
	g_return_val_if_fail (expr != NULL, NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	ccall = (_tmp1_ = vala_ccode_function_call_new (_tmp0_ = vala_ccode_base_module_get_destroy_func_expression (self, type)), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL)), _tmp1_);
	_tmp2_ = FALSE;
	if (VALA_IS_VALUE_TYPE (type)) {
		_tmp2_ = !vala_data_type_get_nullable (type);
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		ValaCCodeUnaryExpression* _tmp3_;
		/* normal value type, no null check*/
		_tmp3_ = NULL;
		vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp3_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, cvar)));
		(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
		if (vala_data_type_get_data_type (type) == VALA_TYPESYMBOL (self->gvalue_type)) {
			ValaCCodeIdentifier* _tmp4_;
			ValaCCodeFunctionCall* _tmp5_;
			ValaCCodeFunctionCall* cisvalid;
			ValaCCodeUnaryExpression* _tmp6_;
			ValaCCodeCommaExpression* ccomma;
			ValaCCodeConstant* _tmp7_;
			ValaCCodeConstant* _tmp8_;
			ValaCCodeExpression* _tmp9_;
			ValaCCodeExpression* _tmp10_;
			/* g_value_unset must not be called for already unset values*/
			_tmp4_ = NULL;
			_tmp5_ = NULL;
			cisvalid = (_tmp5_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp4_ = vala_ccode_identifier_new ("G_IS_VALUE"))), (_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL)), _tmp5_);
			_tmp6_ = NULL;
			vala_ccode_function_call_add_argument (cisvalid, (ValaCCodeExpression*) (_tmp6_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, cvar)));
			(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
			ccomma = vala_ccode_comma_expression_new ();
			vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) ccall);
			_tmp7_ = NULL;
			vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp7_ = vala_ccode_constant_new ("NULL")));
			(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
			_tmp8_ = NULL;
			_tmp9_ = NULL;
			_tmp10_ = NULL;
			return (_tmp10_ = (_tmp9_ = (ValaCCodeExpression*) vala_ccode_conditional_expression_new ((ValaCCodeExpression*) cisvalid, (ValaCCodeExpression*) ccomma, (ValaCCodeExpression*) (_tmp8_ = vala_ccode_constant_new ("NULL"))), (_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL)), _tmp9_), (cisvalid == NULL) ? NULL : (cisvalid = (vala_ccode_node_unref (cisvalid), NULL)), (ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL)), (ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL)), _tmp10_);
		} else {
			return (ValaCCodeExpression*) ccall;
		}
	}
	/* (foo == NULL ? NULL : foo = (unref (foo), NULL)) 
	 can be simplified to
	 * foo = (unref (foo), NULL)
	 * if foo is of static type non-null
	 */
	_tmp12_ = NULL;
	_tmp13_ = NULL;
	cisnull = (_tmp13_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, cvar, (ValaCCodeExpression*) (_tmp12_ = vala_ccode_constant_new ("NULL"))), (_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL)), _tmp13_);
	if (vala_data_type_get_type_parameter (type) != NULL) {
		gboolean _tmp14_;
		ValaCCodeConstant* _tmp17_;
		ValaCCodeExpression* _tmp16_;
		ValaCCodeBinaryExpression* _tmp18_;
		ValaCCodeBinaryExpression* cunrefisnull;
		ValaCCodeBinaryExpression* _tmp19_;
		_tmp14_ = FALSE;
		if (!VALA_IS_CLASS (self->current_type_symbol)) {
			_tmp14_ = TRUE;
		} else {
			_tmp14_ = vala_class_get_is_compact (self->current_class);
		}
		if (_tmp14_) {
			ValaCCodeExpression* _tmp15_;
			_tmp15_ = NULL;
			return (_tmp15_ = (ValaCCodeExpression*) vala_ccode_constant_new ("NULL"), (ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL)), (cisnull == NULL) ? NULL : (cisnull = (vala_ccode_node_unref (cisnull), NULL)), _tmp15_);
		}
		/* unref functions are optional for type parameters*/
		_tmp17_ = NULL;
		_tmp16_ = NULL;
		_tmp18_ = NULL;
		cunrefisnull = (_tmp18_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, _tmp16_ = vala_ccode_base_module_get_destroy_func_expression (self, type), (ValaCCodeExpression*) (_tmp17_ = vala_ccode_constant_new ("NULL"))), (_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_ccode_node_unref (_tmp17_), NULL)), (_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL)), _tmp18_);
		_tmp19_ = NULL;
		cisnull = (_tmp19_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_OR, (ValaCCodeExpression*) cisnull, (ValaCCodeExpression*) cunrefisnull), (cisnull == NULL) ? NULL : (cisnull = (vala_ccode_node_unref (cisnull), NULL)), _tmp19_);
		(cunrefisnull == NULL) ? NULL : (cunrefisnull = (vala_ccode_node_unref (cunrefisnull), NULL));
	}
	vala_ccode_function_call_add_argument (ccall, cvar);
	/* set freed references to NULL to prevent further use */
	ccomma = vala_ccode_comma_expression_new ();
	_tmp20_ = FALSE;
	_tmp21_ = FALSE;
	_tmp22_ = FALSE;
	if (vala_data_type_get_data_type (type) == self->gstringbuilder_type) {
		_tmp22_ = TRUE;
	} else {
		_tmp22_ = vala_data_type_get_data_type (type) == self->garray_type;
	}
	if (_tmp22_) {
		_tmp21_ = TRUE;
	} else {
		_tmp21_ = vala_data_type_get_data_type (type) == self->gbytearray_type;
	}
	if (_tmp21_) {
		_tmp20_ = TRUE;
	} else {
		_tmp20_ = vala_data_type_get_data_type (type) == self->gptrarray_type;
	}
	if (_tmp20_) {
		ValaCCodeConstant* _tmp23_;
		_tmp23_ = NULL;
		vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp23_ = vala_ccode_constant_new ("TRUE")));
		(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_ccode_node_unref (_tmp23_), NULL));
	} else {
		if (VALA_IS_ARRAY_TYPE (type)) {
			ValaArrayType* _tmp24_;
			ValaArrayType* array_type;
			gboolean _tmp25_;
			_tmp24_ = NULL;
			array_type = (_tmp24_ = VALA_ARRAY_TYPE (type), (_tmp24_ == NULL) ? NULL : vala_code_node_ref (_tmp24_));
			_tmp25_ = FALSE;
			if (vala_data_type_get_data_type (vala_array_type_get_element_type (array_type)) == NULL) {
				_tmp25_ = TRUE;
			} else {
				_tmp25_ = vala_typesymbol_is_reference_type (vala_data_type_get_data_type (vala_array_type_get_element_type (array_type)));
			}
			if (_tmp25_) {
				ValaCCodeExpression* csizeexpr;
				gboolean first;
				ValaCCodeIdentifier* _tmp29_;
				ValaCCodeCastExpression* _tmp31_;
				ValaCCodeExpression* _tmp30_;
				self->requires_array_free = TRUE;
				csizeexpr = NULL;
				first = TRUE;
				{
					gint dim;
					dim = 1;
					for (; dim <= vala_array_type_get_rank (array_type); dim++) {
						if (first) {
							ValaCCodeExpression* _tmp26_;
							_tmp26_ = NULL;
							csizeexpr = (_tmp26_ = vala_ccode_module_get_array_length_cexpression (vala_ccode_module_get_head ((ValaCCodeModule*) self), expr, dim), (csizeexpr == NULL) ? NULL : (csizeexpr = (vala_ccode_node_unref (csizeexpr), NULL)), _tmp26_);
							first = FALSE;
						} else {
							ValaCCodeExpression* _tmp28_;
							ValaCCodeExpression* _tmp27_;
							_tmp28_ = NULL;
							_tmp27_ = NULL;
							csizeexpr = (_tmp28_ = (ValaCCodeExpression*) vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_MUL, csizeexpr, _tmp27_ = vala_ccode_module_get_array_length_cexpression (vala_ccode_module_get_head ((ValaCCodeModule*) self), expr, dim)), (csizeexpr == NULL) ? NULL : (csizeexpr = (vala_ccode_node_unref (csizeexpr), NULL)), _tmp28_);
							(_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL));
						}
					}
				}
				_tmp29_ = NULL;
				vala_ccode_function_call_set_call (ccall, (ValaCCodeExpression*) (_tmp29_ = vala_ccode_identifier_new ("_vala_array_free")));
				(_tmp29_ == NULL) ? NULL : (_tmp29_ = (vala_ccode_node_unref (_tmp29_), NULL));
				vala_ccode_function_call_add_argument (ccall, csizeexpr);
				_tmp31_ = NULL;
				_tmp30_ = NULL;
				vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp31_ = vala_ccode_cast_expression_new (_tmp30_ = vala_ccode_base_module_get_destroy_func_expression (self, vala_array_type_get_element_type (array_type)), "GDestroyNotify")));
				(_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_ccode_node_unref (_tmp31_), NULL));
				(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL));
				(csizeexpr == NULL) ? NULL : (csizeexpr = (vala_ccode_node_unref (csizeexpr), NULL));
			}
			(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
		}
	}
	vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) ccall);
	_tmp32_ = NULL;
	vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp32_ = vala_ccode_constant_new ("NULL")));
	(_tmp32_ == NULL) ? NULL : (_tmp32_ = (vala_ccode_node_unref (_tmp32_), NULL));
	cassign = vala_ccode_assignment_new (cvar, (ValaCCodeExpression*) ccomma, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE);
	_tmp33_ = FALSE;
	_tmp34_ = FALSE;
	if (vala_data_type_get_data_type (type) != NULL) {
		_tmp34_ = !vala_typesymbol_is_reference_counting (vala_data_type_get_data_type (type));
	} else {
		_tmp34_ = FALSE;
	}
	if (_tmp34_) {
		char* _tmp35_;
		_tmp35_ = NULL;
		_tmp33_ = _vala_strcmp0 (_tmp35_ = vala_typesymbol_get_free_function (vala_data_type_get_data_type (type)), "g_free") == 0;
		_tmp35_ = (g_free (_tmp35_), NULL);
	} else {
		_tmp33_ = FALSE;
	}
	/* g_free (NULL) is allowed*/
	uses_gfree = _tmp33_;
	_tmp36_ = FALSE;
	if (uses_gfree) {
		_tmp36_ = TRUE;
	} else {
		_tmp36_ = VALA_IS_ARRAY_TYPE (type);
	}
	uses_gfree = _tmp36_;
	if (uses_gfree) {
		ValaCCodeExpression* _tmp37_;
		_tmp37_ = NULL;
		return (_tmp37_ = (ValaCCodeExpression*) cassign, (ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL)), (cisnull == NULL) ? NULL : (cisnull = (vala_ccode_node_unref (cisnull), NULL)), (ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL)), _tmp37_);
	}
	_tmp38_ = NULL;
	_tmp39_ = NULL;
	_tmp40_ = NULL;
	return (_tmp40_ = (_tmp39_ = (ValaCCodeExpression*) vala_ccode_conditional_expression_new ((ValaCCodeExpression*) cisnull, (ValaCCodeExpression*) (_tmp38_ = vala_ccode_constant_new ("NULL")), (ValaCCodeExpression*) cassign), (_tmp38_ == NULL) ? NULL : (_tmp38_ = (vala_ccode_node_unref (_tmp38_), NULL)), _tmp39_), (ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL)), (cisnull == NULL) ? NULL : (cisnull = (vala_ccode_node_unref (cisnull), NULL)), (ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL)), (cassign == NULL) ? NULL : (cassign = (vala_ccode_node_unref (cassign), NULL)), _tmp40_);
}


ValaCCodeExpression* vala_ccode_base_module_get_unref_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* cvar, ValaDataType* type, ValaExpression* expr) {
	return VALA_CCODE_BASE_MODULE_GET_CLASS (self)->get_unref_expression (self, cvar, type, expr);
}


static void vala_ccode_base_module_real_visit_end_full_expression (ValaCCodeModule* base, ValaExpression* expr) {
	ValaCCodeBaseModule * self;
	ValaDataType* _tmp0_;
	ValaDataType* expr_type;
	ValaLocalVariable* full_expr_var;
	ValaCCodeCommaExpression* expr_list;
	ValaCCodeAssignment* _tmp4_;
	ValaCCodeExpression* _tmp3_;
	ValaCCodeExpression* _tmp7_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	/* expr is a full expression, i.e. an initializer, the
	 * expression in an expression statement, the controlling
	 * expression in if, while, for, or foreach statements
	 *
	 * we unref temporary variables at the end of a full
	 * expression
	 
	 can't automatically deep copy lists yet, so do it
	 * manually for now
	 * replace with
	 * expr.temp_vars = temp_vars;
	 * when deep list copying works
	 */
	gee_collection_clear ((GeeCollection*) expr->temp_vars);
	{
		GeeIterator* _local_it;
		_local_it = gee_iterable_iterator ((GeeIterable*) self->temp_vars);
		while (gee_iterator_next (_local_it)) {
			ValaLocalVariable* local;
			local = (ValaLocalVariable*) gee_iterator_get (_local_it);
			gee_collection_add ((GeeCollection*) expr->temp_vars, local);
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
		}
		(_local_it == NULL) ? NULL : (_local_it = (gee_collection_object_unref (_local_it), NULL));
	}
	gee_collection_clear ((GeeCollection*) self->temp_vars);
	if (gee_collection_get_size ((GeeCollection*) GEE_LIST (self->temp_ref_vars)) == 0) {
		/* nothing to do without temporary variables */
		return;
	}
	_tmp0_ = NULL;
	expr_type = (_tmp0_ = vala_expression_get_value_type (expr), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	if (vala_expression_get_target_type (expr) != NULL) {
		ValaDataType* _tmp2_;
		ValaDataType* _tmp1_;
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		expr_type = (_tmp2_ = (_tmp1_ = vala_expression_get_target_type (expr), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (expr_type == NULL) ? NULL : (expr_type = (vala_code_node_unref (expr_type), NULL)), _tmp2_);
	}
	full_expr_var = vala_ccode_base_module_get_temp_variable (self, expr_type, TRUE, (ValaCodeNode*) expr);
	gee_collection_add ((GeeCollection*) expr->temp_vars, full_expr_var);
	expr_list = vala_ccode_comma_expression_new ();
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	vala_ccode_comma_expression_append_expression (expr_list, (ValaCCodeExpression*) (_tmp4_ = vala_ccode_assignment_new (_tmp3_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) full_expr_var)), VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) expr)), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
	{
		GeeIterator* _local_it;
		_local_it = gee_iterable_iterator ((GeeIterable*) self->temp_ref_vars);
		while (gee_iterator_next (_local_it)) {
			ValaLocalVariable* local;
			ValaMemberAccess* ma;
			ValaCCodeExpression* _tmp6_;
			ValaCCodeExpression* _tmp5_;
			local = (ValaLocalVariable*) gee_iterator_get (_local_it);
			ma = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) local), NULL);
			vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) local);
			_tmp6_ = NULL;
			_tmp5_ = NULL;
			vala_ccode_comma_expression_append_expression (expr_list, _tmp6_ = vala_ccode_base_module_get_unref_expression (self, _tmp5_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) local)), vala_local_variable_get_variable_type (local), (ValaExpression*) ma));
			(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
			(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
		}
		(_local_it == NULL) ? NULL : (_local_it = (gee_collection_object_unref (_local_it), NULL));
	}
	_tmp7_ = NULL;
	vala_ccode_comma_expression_append_expression (expr_list, _tmp7_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) full_expr_var)));
	(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) expr_list);
	gee_collection_clear ((GeeCollection*) self->temp_ref_vars);
	(expr_type == NULL) ? NULL : (expr_type = (vala_code_node_unref (expr_type), NULL));
	(full_expr_var == NULL) ? NULL : (full_expr_var = (vala_code_node_unref (full_expr_var), NULL));
	(expr_list == NULL) ? NULL : (expr_list = (vala_ccode_node_unref (expr_list), NULL));
}


void vala_ccode_base_module_append_temp_decl (ValaCCodeBaseModule* self, ValaCCodeFragment* cfrag, GeeList* temp_vars) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cfrag != NULL);
	g_return_if_fail (temp_vars != NULL);
	{
		GeeIterator* _local_it;
		_local_it = gee_iterable_iterator ((GeeIterable*) temp_vars);
		while (gee_iterator_next (_local_it)) {
			ValaLocalVariable* local;
			gboolean _tmp0_;
			local = (ValaLocalVariable*) gee_iterator_get (_local_it);
			_tmp0_ = FALSE;
			if (self->current_method != NULL) {
				_tmp0_ = vala_method_get_coroutine (self->current_method);
			} else {
				_tmp0_ = FALSE;
			}
			if (_tmp0_) {
				char* _tmp1_;
				_tmp1_ = NULL;
				vala_ccode_struct_add_field (self->closure_struct, _tmp1_ = vala_data_type_get_cname (vala_local_variable_get_variable_type (local)), vala_symbol_get_name ((ValaSymbol*) local));
				_tmp1_ = (g_free (_tmp1_), NULL);
			} else {
				char* _tmp2_;
				ValaCCodeDeclaration* _tmp3_;
				ValaCCodeDeclaration* _cdecl_;
				char* _tmp4_;
				ValaCCodeVariableDeclarator* _tmp5_;
				ValaCCodeVariableDeclarator* vardecl;
				ValaStruct* _tmp7_;
				ValaTypeSymbol* _tmp6_;
				ValaStruct* st;
				ValaArrayType* _tmp9_;
				ValaDataType* _tmp8_;
				ValaArrayType* array_type;
				/* no initialization necessary, closure struct is zeroed*/
				_tmp2_ = NULL;
				_tmp3_ = NULL;
				_cdecl_ = (_tmp3_ = vala_ccode_declaration_new (_tmp2_ = vala_data_type_get_cname (vala_local_variable_get_variable_type (local))), _tmp2_ = (g_free (_tmp2_), NULL), _tmp3_);
				_tmp4_ = NULL;
				_tmp5_ = NULL;
				vardecl = (_tmp5_ = vala_ccode_variable_declarator_new (vala_symbol_get_name ((ValaSymbol*) local), NULL, _tmp4_ = vala_data_type_get_cdeclarator_suffix (vala_local_variable_get_variable_type (local))), _tmp4_ = (g_free (_tmp4_), NULL), _tmp5_);
				/* sets #line*/
				vala_code_node_set_ccodenode ((ValaCodeNode*) local, (ValaCCodeNode*) vardecl);
				vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) vardecl);
				_tmp7_ = NULL;
				_tmp6_ = NULL;
				st = (_tmp7_ = (_tmp6_ = vala_data_type_get_data_type (vala_local_variable_get_variable_type (local)), VALA_IS_STRUCT (_tmp6_) ? ((ValaStruct*) _tmp6_) : NULL), (_tmp7_ == NULL) ? NULL : vala_code_node_ref (_tmp7_));
				_tmp9_ = NULL;
				_tmp8_ = NULL;
				array_type = (_tmp9_ = (_tmp8_ = vala_local_variable_get_variable_type (local), VALA_IS_ARRAY_TYPE (_tmp8_) ? ((ValaArrayType*) _tmp8_) : NULL), (_tmp9_ == NULL) ? NULL : vala_code_node_ref (_tmp9_));
				if (g_str_has_prefix (vala_symbol_get_name ((ValaSymbol*) local), "*")) {
				} else {
					gboolean _tmp10_;
					gboolean _tmp11_;
					_tmp10_ = FALSE;
					_tmp11_ = FALSE;
					if (!vala_data_type_get_nullable (vala_local_variable_get_variable_type (local))) {
						gboolean _tmp12_;
						_tmp12_ = FALSE;
						if (st != NULL) {
							_tmp12_ = !vala_struct_is_simple_type (st);
						} else {
							_tmp12_ = FALSE;
						}
						_tmp11_ = _tmp12_;
					} else {
						_tmp11_ = FALSE;
					}
					if (_tmp11_) {
						_tmp10_ = TRUE;
					} else {
						gboolean _tmp13_;
						_tmp13_ = FALSE;
						if (array_type != NULL) {
							_tmp13_ = vala_array_type_get_fixed_length (array_type);
						} else {
							_tmp13_ = FALSE;
						}
						_tmp10_ = _tmp13_;
					}
					if (_tmp10_) {
						ValaCCodeInitializerList* clist;
						ValaCCodeConstant* _tmp14_;
						/* 0-initialize struct with struct initializer { 0 }
						 necessary as they will be passed by reference*/
						clist = vala_ccode_initializer_list_new ();
						_tmp14_ = NULL;
						vala_ccode_initializer_list_append (clist, (ValaCCodeExpression*) (_tmp14_ = vala_ccode_constant_new ("0")));
						(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
						vala_ccode_variable_declarator_set_initializer (vardecl, (ValaCCodeExpression*) clist);
						(clist == NULL) ? NULL : (clist = (vala_ccode_node_unref (clist), NULL));
					} else {
						gboolean _tmp15_;
						_tmp15_ = FALSE;
						if (vala_data_type_is_reference_type_or_type_parameter (vala_local_variable_get_variable_type (local))) {
							_tmp15_ = TRUE;
						} else {
							_tmp15_ = vala_data_type_get_nullable (vala_local_variable_get_variable_type (local));
						}
						if (_tmp15_) {
							ValaCCodeConstant* _tmp16_;
							_tmp16_ = NULL;
							vala_ccode_variable_declarator_set_initializer (vardecl, (ValaCCodeExpression*) (_tmp16_ = vala_ccode_constant_new ("NULL")));
							(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL));
						}
					}
				}
				vala_ccode_fragment_append (cfrag, (ValaCCodeNode*) _cdecl_);
				(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
				(vardecl == NULL) ? NULL : (vardecl = (vala_ccode_node_unref (vardecl), NULL));
				(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
				(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
			}
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
		}
		(_local_it == NULL) ? NULL : (_local_it = (gee_collection_object_unref (_local_it), NULL));
	}
}


static void vala_ccode_base_module_real_visit_expression_statement (ValaCCodeModule* base, ValaExpressionStatement* stmt) {
	ValaCCodeBaseModule * self;
	ValaCCodeExpressionStatement* _tmp0_;
	gboolean _tmp1_;
	gboolean _tmp2_;
	ValaCCodeFragment* cfrag;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	if (vala_code_node_get_error ((ValaCodeNode*) vala_expression_statement_get_expression (stmt))) {
		vala_code_node_set_error ((ValaCodeNode*) stmt, TRUE);
		return;
	}
	_tmp0_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) stmt, (ValaCCodeNode*) (_tmp0_ = vala_ccode_expression_statement_new (VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_expression_statement_get_expression (stmt))))));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
	_tmp1_ = FALSE;
	if (vala_code_node_get_tree_can_fail ((ValaCodeNode*) stmt)) {
		_tmp1_ = vala_code_node_get_tree_can_fail ((ValaCodeNode*) vala_expression_statement_get_expression (stmt));
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		ValaCCodeFragment* cfrag;
		/* simple case, no node breakdown necessary*/
		cfrag = vala_ccode_fragment_new ();
		vala_ccode_fragment_append (cfrag, vala_code_node_get_ccodenode ((ValaCodeNode*) stmt));
		vala_ccode_module_add_simple_check (vala_ccode_module_get_head ((ValaCCodeModule*) self), (ValaCodeNode*) vala_expression_statement_get_expression (stmt), cfrag);
		vala_code_node_set_ccodenode ((ValaCodeNode*) stmt, (ValaCCodeNode*) cfrag);
		(cfrag == NULL) ? NULL : (cfrag = (vala_ccode_node_unref (cfrag), NULL));
	}
	_tmp2_ = FALSE;
	if (gee_collection_get_size ((GeeCollection*) GEE_LIST (self->temp_vars)) == 0) {
		_tmp2_ = self->pre_statement_fragment == NULL;
	} else {
		_tmp2_ = FALSE;
	}
	/* free temporary objects */
	if (_tmp2_) {
		/* nothing to do without temporary variables */
		return;
	}
	cfrag = vala_ccode_fragment_new ();
	vala_ccode_base_module_append_temp_decl (self, cfrag, (GeeList*) self->temp_vars);
	if (self->pre_statement_fragment != NULL) {
		ValaCCodeFragment* _tmp3_;
		vala_ccode_fragment_append (cfrag, (ValaCCodeNode*) self->pre_statement_fragment);
		_tmp3_ = NULL;
		self->pre_statement_fragment = (_tmp3_ = NULL, (self->pre_statement_fragment == NULL) ? NULL : (self->pre_statement_fragment = (vala_ccode_node_unref (self->pre_statement_fragment), NULL)), _tmp3_);
	}
	vala_ccode_fragment_append (cfrag, vala_code_node_get_ccodenode ((ValaCodeNode*) stmt));
	{
		GeeIterator* _local_it;
		_local_it = gee_iterable_iterator ((GeeIterable*) self->temp_ref_vars);
		while (gee_iterator_next (_local_it)) {
			ValaLocalVariable* local;
			ValaMemberAccess* ma;
			ValaCCodeExpressionStatement* _tmp6_;
			ValaCCodeExpression* _tmp5_;
			ValaCCodeIdentifier* _tmp4_;
			local = (ValaLocalVariable*) gee_iterator_get (_local_it);
			ma = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) local), NULL);
			vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) local);
			_tmp6_ = NULL;
			_tmp5_ = NULL;
			_tmp4_ = NULL;
			vala_ccode_fragment_append (cfrag, (ValaCCodeNode*) (_tmp6_ = vala_ccode_expression_statement_new (_tmp5_ = vala_ccode_base_module_get_unref_expression (self, (ValaCCodeExpression*) (_tmp4_ = vala_ccode_identifier_new (vala_symbol_get_name ((ValaSymbol*) local))), vala_local_variable_get_variable_type (local), (ValaExpression*) ma))));
			(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
			(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
			(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
		}
		(_local_it == NULL) ? NULL : (_local_it = (gee_collection_object_unref (_local_it), NULL));
	}
	vala_code_node_set_ccodenode ((ValaCodeNode*) stmt, (ValaCCodeNode*) cfrag);
	gee_collection_clear ((GeeCollection*) self->temp_vars);
	gee_collection_clear ((GeeCollection*) self->temp_ref_vars);
	(cfrag == NULL) ? NULL : (cfrag = (vala_ccode_node_unref (cfrag), NULL));
}


void vala_ccode_base_module_create_temp_decl (ValaCCodeBaseModule* self, ValaStatement* stmt, GeeList* temp_vars) {
	ValaCCodeFragment* cfrag;
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
	g_return_if_fail (temp_vars != NULL);
	/* declare temporary variables */
	if (gee_collection_get_size ((GeeCollection*) temp_vars) == 0) {
		/* nothing to do without temporary variables */
		return;
	}
	cfrag = vala_ccode_fragment_new ();
	vala_ccode_base_module_append_temp_decl (self, cfrag, temp_vars);
	vala_ccode_fragment_append (cfrag, vala_code_node_get_ccodenode ((ValaCodeNode*) stmt));
	vala_code_node_set_ccodenode ((ValaCodeNode*) stmt, (ValaCCodeNode*) cfrag);
	(cfrag == NULL) ? NULL : (cfrag = (vala_ccode_node_unref (cfrag), NULL));
}


void vala_ccode_base_module_append_local_free (ValaCCodeBaseModule* self, ValaSymbol* sym, ValaCCodeFragment* cfrag, gboolean stop_at_loop) {
	ValaBlock* _tmp0_;
	ValaBlock* b;
	GeeList* local_vars;
	g_return_if_fail (self != NULL);
	g_return_if_fail (sym != NULL);
	g_return_if_fail (cfrag != NULL);
	_tmp0_ = NULL;
	b = (_tmp0_ = VALA_BLOCK (sym), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	local_vars = vala_block_get_local_variables (b);
	{
		GeeIterator* _local_it;
		_local_it = gee_iterable_iterator ((GeeIterable*) local_vars);
		while (gee_iterator_next (_local_it)) {
			ValaLocalVariable* local;
			gboolean _tmp1_;
			gboolean _tmp2_;
			local = (ValaLocalVariable*) gee_iterator_get (_local_it);
			_tmp1_ = FALSE;
			_tmp2_ = FALSE;
			if (vala_symbol_get_active ((ValaSymbol*) local)) {
				_tmp2_ = !vala_local_variable_get_floating (local);
			} else {
				_tmp2_ = FALSE;
			}
			if (_tmp2_) {
				_tmp1_ = vala_ccode_base_module_requires_destroy (self, vala_local_variable_get_variable_type (local));
			} else {
				_tmp1_ = FALSE;
			}
			if (_tmp1_) {
				ValaMemberAccess* ma;
				ValaCCodeExpressionStatement* _tmp5_;
				ValaCCodeExpression* _tmp4_;
				ValaCCodeExpression* _tmp3_;
				ma = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) local), NULL);
				vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) local);
				_tmp5_ = NULL;
				_tmp4_ = NULL;
				_tmp3_ = NULL;
				vala_ccode_fragment_append (cfrag, (ValaCCodeNode*) (_tmp5_ = vala_ccode_expression_statement_new (_tmp4_ = vala_ccode_base_module_get_unref_expression (self, _tmp3_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) local)), vala_local_variable_get_variable_type (local), (ValaExpression*) ma))));
				(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
				(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
				(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
				(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
			}
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
		}
		(_local_it == NULL) ? NULL : (_local_it = (gee_collection_object_unref (_local_it), NULL));
	}
	if (stop_at_loop) {
		gboolean _tmp6_;
		gboolean _tmp7_;
		gboolean _tmp8_;
		gboolean _tmp9_;
		_tmp6_ = FALSE;
		_tmp7_ = FALSE;
		_tmp8_ = FALSE;
		_tmp9_ = FALSE;
		if (VALA_IS_DO_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) b))) {
			_tmp9_ = TRUE;
		} else {
			_tmp9_ = VALA_IS_WHILE_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) b));
		}
		if (_tmp9_) {
			_tmp8_ = TRUE;
		} else {
			_tmp8_ = VALA_IS_FOR_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) b));
		}
		if (_tmp8_) {
			_tmp7_ = TRUE;
		} else {
			_tmp7_ = VALA_IS_FOREACH_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) b));
		}
		if (_tmp7_) {
			_tmp6_ = TRUE;
		} else {
			_tmp6_ = VALA_IS_SWITCH_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) b));
		}
		if (_tmp6_) {
			(b == NULL) ? NULL : (b = (vala_code_node_unref (b), NULL));
			(local_vars == NULL) ? NULL : (local_vars = (gee_collection_object_unref (local_vars), NULL));
			return;
		}
	}
	if (VALA_IS_BLOCK (vala_symbol_get_parent_symbol (sym))) {
		vala_ccode_base_module_append_local_free (self, vala_symbol_get_parent_symbol (sym), cfrag, stop_at_loop);
	} else {
		if (VALA_IS_METHOD (vala_symbol_get_parent_symbol (sym))) {
			vala_ccode_base_module_append_param_free (self, VALA_METHOD (vala_symbol_get_parent_symbol (sym)), cfrag);
		}
	}
	(b == NULL) ? NULL : (b = (vala_code_node_unref (b), NULL));
	(local_vars == NULL) ? NULL : (local_vars = (gee_collection_object_unref (local_vars), NULL));
}


void vala_ccode_base_module_append_error_free (ValaCCodeBaseModule* self, ValaSymbol* sym, ValaCCodeFragment* cfrag, ValaTryStatement* current_try) {
	ValaBlock* _tmp0_;
	ValaBlock* b;
	GeeList* local_vars;
	g_return_if_fail (self != NULL);
	g_return_if_fail (sym != NULL);
	g_return_if_fail (cfrag != NULL);
	g_return_if_fail (current_try != NULL);
	_tmp0_ = NULL;
	b = (_tmp0_ = VALA_BLOCK (sym), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	local_vars = vala_block_get_local_variables (b);
	{
		GeeIterator* _local_it;
		_local_it = gee_iterable_iterator ((GeeIterable*) local_vars);
		while (gee_iterator_next (_local_it)) {
			ValaLocalVariable* local;
			gboolean _tmp1_;
			gboolean _tmp2_;
			local = (ValaLocalVariable*) gee_iterator_get (_local_it);
			_tmp1_ = FALSE;
			_tmp2_ = FALSE;
			if (vala_symbol_get_active ((ValaSymbol*) local)) {
				_tmp2_ = !vala_local_variable_get_floating (local);
			} else {
				_tmp2_ = FALSE;
			}
			if (_tmp2_) {
				_tmp1_ = vala_ccode_base_module_requires_destroy (self, vala_local_variable_get_variable_type (local));
			} else {
				_tmp1_ = FALSE;
			}
			if (_tmp1_) {
				ValaMemberAccess* ma;
				ValaCCodeExpressionStatement* _tmp5_;
				ValaCCodeExpression* _tmp4_;
				ValaCCodeExpression* _tmp3_;
				ma = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) local), NULL);
				vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) local);
				_tmp5_ = NULL;
				_tmp4_ = NULL;
				_tmp3_ = NULL;
				vala_ccode_fragment_append (cfrag, (ValaCCodeNode*) (_tmp5_ = vala_ccode_expression_statement_new (_tmp4_ = vala_ccode_base_module_get_unref_expression (self, _tmp3_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) local)), vala_local_variable_get_variable_type (local), (ValaExpression*) ma))));
				(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
				(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
				(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
				(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
			}
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
		}
		(_local_it == NULL) ? NULL : (_local_it = (gee_collection_object_unref (_local_it), NULL));
	}
	if (sym == VALA_SYMBOL (vala_try_statement_get_body (current_try))) {
		(b == NULL) ? NULL : (b = (vala_code_node_unref (b), NULL));
		(local_vars == NULL) ? NULL : (local_vars = (gee_collection_object_unref (local_vars), NULL));
		return;
	}
	if (VALA_IS_BLOCK (vala_symbol_get_parent_symbol (sym))) {
		vala_ccode_base_module_append_error_free (self, vala_symbol_get_parent_symbol (sym), cfrag, current_try);
	} else {
		if (VALA_IS_METHOD (vala_symbol_get_parent_symbol (sym))) {
			vala_ccode_base_module_append_param_free (self, VALA_METHOD (vala_symbol_get_parent_symbol (sym)), cfrag);
		}
	}
	(b == NULL) ? NULL : (b = (vala_code_node_unref (b), NULL));
	(local_vars == NULL) ? NULL : (local_vars = (gee_collection_object_unref (local_vars), NULL));
}


static void vala_ccode_base_module_append_param_free (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeFragment* cfrag) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	g_return_if_fail (cfrag != NULL);
	{
		GeeList* _tmp0_;
		GeeIterator* _tmp1_;
		GeeIterator* _param_it;
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		_param_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = vala_method_get_parameters (m))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (gee_collection_object_unref (_tmp0_), NULL)), _tmp1_);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			gboolean _tmp2_;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			_tmp2_ = FALSE;
			if (vala_ccode_base_module_requires_destroy (self, vala_formal_parameter_get_parameter_type (param))) {
				_tmp2_ = vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_IN;
			} else {
				_tmp2_ = FALSE;
			}
			if (_tmp2_) {
				ValaMemberAccess* ma;
				ValaCCodeExpressionStatement* _tmp5_;
				ValaCCodeExpression* _tmp4_;
				ValaCCodeExpression* _tmp3_;
				ma = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) param), NULL);
				vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) param);
				_tmp5_ = NULL;
				_tmp4_ = NULL;
				_tmp3_ = NULL;
				vala_ccode_fragment_append (cfrag, (ValaCCodeNode*) (_tmp5_ = vala_ccode_expression_statement_new (_tmp4_ = vala_ccode_base_module_get_unref_expression (self, _tmp3_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) param)), vala_formal_parameter_get_parameter_type (param), (ValaExpression*) ma))));
				(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
				(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
				(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
				(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
			}
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
}


void vala_ccode_base_module_create_local_free (ValaCCodeBaseModule* self, ValaCodeNode* stmt, gboolean stop_at_loop) {
	ValaCCodeFragment* cfrag;
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
	cfrag = vala_ccode_fragment_new ();
	vala_ccode_base_module_append_local_free (self, self->current_symbol, cfrag, stop_at_loop);
	vala_ccode_fragment_append (cfrag, vala_code_node_get_ccodenode (stmt));
	vala_code_node_set_ccodenode (stmt, (ValaCCodeNode*) cfrag);
	(cfrag == NULL) ? NULL : (cfrag = (vala_ccode_node_unref (cfrag), NULL));
}


static gboolean vala_ccode_base_module_append_local_free_expr (ValaCCodeBaseModule* self, ValaSymbol* sym, ValaCCodeCommaExpression* ccomma, gboolean stop_at_loop) {
	gboolean found;
	ValaBlock* _tmp0_;
	ValaBlock* b;
	GeeList* local_vars;
	gboolean _tmp7_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (sym != NULL, FALSE);
	g_return_val_if_fail (ccomma != NULL, FALSE);
	found = FALSE;
	_tmp0_ = NULL;
	b = (_tmp0_ = VALA_BLOCK (sym), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	local_vars = vala_block_get_local_variables (b);
	{
		GeeIterator* _local_it;
		_local_it = gee_iterable_iterator ((GeeIterable*) local_vars);
		while (gee_iterator_next (_local_it)) {
			ValaLocalVariable* local;
			gboolean _tmp1_;
			gboolean _tmp2_;
			local = (ValaLocalVariable*) gee_iterator_get (_local_it);
			_tmp1_ = FALSE;
			_tmp2_ = FALSE;
			if (vala_symbol_get_active ((ValaSymbol*) local)) {
				_tmp2_ = !vala_local_variable_get_floating (local);
			} else {
				_tmp2_ = FALSE;
			}
			if (_tmp2_) {
				_tmp1_ = vala_ccode_base_module_requires_destroy (self, vala_local_variable_get_variable_type (local));
			} else {
				_tmp1_ = FALSE;
			}
			if (_tmp1_) {
				ValaMemberAccess* ma;
				ValaCCodeExpression* _tmp4_;
				ValaCCodeExpression* _tmp3_;
				found = TRUE;
				ma = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) local), NULL);
				vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) local);
				_tmp4_ = NULL;
				_tmp3_ = NULL;
				vala_ccode_comma_expression_append_expression (ccomma, _tmp4_ = vala_ccode_base_module_get_unref_expression (self, _tmp3_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) local)), vala_local_variable_get_variable_type (local), (ValaExpression*) ma));
				(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
				(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
				(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
			}
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
		}
		(_local_it == NULL) ? NULL : (_local_it = (gee_collection_object_unref (_local_it), NULL));
	}
	if (VALA_IS_BLOCK (vala_symbol_get_parent_symbol (sym))) {
		gboolean _tmp5_;
		_tmp5_ = FALSE;
		if (vala_ccode_base_module_append_local_free_expr (self, vala_symbol_get_parent_symbol (sym), ccomma, stop_at_loop)) {
			_tmp5_ = TRUE;
		} else {
			_tmp5_ = found;
		}
		found = _tmp5_;
	} else {
		if (VALA_IS_METHOD (vala_symbol_get_parent_symbol (sym))) {
			gboolean _tmp6_;
			_tmp6_ = FALSE;
			if (vala_ccode_base_module_append_param_free_expr (self, VALA_METHOD (vala_symbol_get_parent_symbol (sym)), ccomma)) {
				_tmp6_ = TRUE;
			} else {
				_tmp6_ = found;
			}
			found = _tmp6_;
		}
	}
	return (_tmp7_ = found, (b == NULL) ? NULL : (b = (vala_code_node_unref (b), NULL)), (local_vars == NULL) ? NULL : (local_vars = (gee_collection_object_unref (local_vars), NULL)), _tmp7_);
}


static gboolean vala_ccode_base_module_append_param_free_expr (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeCommaExpression* ccomma) {
	gboolean found;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (m != NULL, FALSE);
	g_return_val_if_fail (ccomma != NULL, FALSE);
	found = FALSE;
	{
		GeeList* _tmp0_;
		GeeIterator* _tmp1_;
		GeeIterator* _param_it;
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		_param_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = vala_method_get_parameters (m))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (gee_collection_object_unref (_tmp0_), NULL)), _tmp1_);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			gboolean _tmp2_;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			_tmp2_ = FALSE;
			if (vala_ccode_base_module_requires_destroy (self, vala_formal_parameter_get_parameter_type (param))) {
				_tmp2_ = vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_IN;
			} else {
				_tmp2_ = FALSE;
			}
			if (_tmp2_) {
				ValaMemberAccess* ma;
				ValaCCodeExpression* _tmp4_;
				ValaCCodeExpression* _tmp3_;
				found = TRUE;
				ma = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) param), NULL);
				vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) param);
				_tmp4_ = NULL;
				_tmp3_ = NULL;
				vala_ccode_comma_expression_append_expression (ccomma, _tmp4_ = vala_ccode_base_module_get_unref_expression (self, _tmp3_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) param)), vala_formal_parameter_get_parameter_type (param), (ValaExpression*) ma));
				(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
				(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
				(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
			}
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	return found;
}


static void vala_ccode_base_module_create_local_free_expr (ValaCCodeBaseModule* self, ValaExpression* expr) {
	ValaDataType* _tmp0_;
	ValaDataType* expr_type;
	ValaLocalVariable* return_expr_decl;
	ValaCCodeCommaExpression* ccomma;
	ValaCCodeAssignment* _tmp4_;
	ValaCCodeExpression* _tmp3_;
	ValaCCodeExpression* _tmp5_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
	_tmp0_ = NULL;
	expr_type = (_tmp0_ = vala_expression_get_value_type (expr), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	if (vala_expression_get_target_type (expr) != NULL) {
		ValaDataType* _tmp2_;
		ValaDataType* _tmp1_;
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		expr_type = (_tmp2_ = (_tmp1_ = vala_expression_get_target_type (expr), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (expr_type == NULL) ? NULL : (expr_type = (vala_code_node_unref (expr_type), NULL)), _tmp2_);
	}
	return_expr_decl = vala_ccode_base_module_get_temp_variable (self, expr_type, TRUE, (ValaCodeNode*) expr);
	ccomma = vala_ccode_comma_expression_new ();
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp4_ = vala_ccode_assignment_new (_tmp3_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) return_expr_decl)), VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) expr)), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
	if (!vala_ccode_base_module_append_local_free_expr (self, self->current_symbol, ccomma, FALSE)) {
		/* no local variables need to be freed */
		(expr_type == NULL) ? NULL : (expr_type = (vala_code_node_unref (expr_type), NULL));
		(return_expr_decl == NULL) ? NULL : (return_expr_decl = (vala_code_node_unref (return_expr_decl), NULL));
		(ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL));
		return;
	}
	_tmp5_ = NULL;
	vala_ccode_comma_expression_append_expression (ccomma, _tmp5_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) return_expr_decl)));
	(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) ccomma);
	gee_collection_add ((GeeCollection*) expr->temp_vars, return_expr_decl);
	(expr_type == NULL) ? NULL : (expr_type = (vala_code_node_unref (expr_type), NULL));
	(return_expr_decl == NULL) ? NULL : (return_expr_decl = (vala_code_node_unref (return_expr_decl), NULL));
	(ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL));
}


static void vala_ccode_base_module_real_visit_return_statement (ValaCCodeModule* base, ValaReturnStatement* stmt) {
	ValaCCodeBaseModule * self;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (stmt != NULL);
	/* avoid unnecessary ref/unref pair*/
	if (vala_return_statement_get_return_expression (stmt) != NULL) {
		ValaLocalVariable* _tmp1_;
		ValaSymbol* _tmp0_;
		ValaLocalVariable* local;
		gboolean _tmp2_;
		gboolean _tmp3_;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		local = (_tmp1_ = (_tmp0_ = vala_expression_get_symbol_reference (vala_return_statement_get_return_expression (stmt)), VALA_IS_LOCAL_VARIABLE (_tmp0_) ? ((ValaLocalVariable*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
		_tmp2_ = FALSE;
		_tmp3_ = FALSE;
		if (vala_data_type_get_value_owned (self->current_return_type)) {
			_tmp3_ = local != NULL;
		} else {
			_tmp3_ = FALSE;
		}
		if (_tmp3_) {
			_tmp2_ = vala_data_type_get_value_owned (vala_local_variable_get_variable_type (local));
		} else {
			_tmp2_ = FALSE;
		}
		if (_tmp2_) {
			/* return expression is local variable taking ownership and
			 * current method is transferring ownership 
			 don't ref expression*/
			vala_data_type_set_value_owned (vala_expression_get_value_type (vala_return_statement_get_return_expression (stmt)), TRUE);
		}
		(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
	}
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	if (vala_return_statement_get_return_expression (stmt) == NULL) {
		ValaCCodeReturnStatement* _tmp4_;
		_tmp4_ = NULL;
		vala_code_node_set_ccodenode ((ValaCodeNode*) stmt, (ValaCCodeNode*) (_tmp4_ = vala_ccode_return_statement_new (NULL)));
		(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
		vala_ccode_base_module_create_local_free (self, (ValaCodeNode*) stmt, FALSE);
	} else {
		ValaSymbol* return_expression_symbol;
		ValaLocalVariable* _tmp6_;
		ValaSymbol* _tmp5_;
		ValaLocalVariable* local;
		gboolean _tmp7_;
		gboolean _tmp8_;
		gboolean _tmp11_;
		gboolean _tmp12_;
		gboolean _tmp21_;
		gboolean _tmp22_;
		return_expression_symbol = NULL;
		/* avoid unnecessary ref/unref pair*/
		_tmp6_ = NULL;
		_tmp5_ = NULL;
		local = (_tmp6_ = (_tmp5_ = vala_expression_get_symbol_reference (vala_return_statement_get_return_expression (stmt)), VALA_IS_LOCAL_VARIABLE (_tmp5_) ? ((ValaLocalVariable*) _tmp5_) : NULL), (_tmp6_ == NULL) ? NULL : vala_code_node_ref (_tmp6_));
		_tmp7_ = FALSE;
		_tmp8_ = FALSE;
		if (vala_data_type_get_value_owned (self->current_return_type)) {
			_tmp8_ = local != NULL;
		} else {
			_tmp8_ = FALSE;
		}
		if (_tmp8_) {
			_tmp7_ = vala_data_type_get_value_owned (vala_local_variable_get_variable_type (local));
		} else {
			_tmp7_ = FALSE;
		}
		if (_tmp7_) {
			ValaSymbol* _tmp10_;
			ValaSymbol* _tmp9_;
			/* return expression is local variable taking ownership and
			 * current method is transferring ownership 
			 don't unref variable*/
			_tmp10_ = NULL;
			_tmp9_ = NULL;
			return_expression_symbol = (_tmp10_ = (_tmp9_ = (ValaSymbol*) local, (_tmp9_ == NULL) ? NULL : vala_code_node_ref (_tmp9_)), (return_expression_symbol == NULL) ? NULL : (return_expression_symbol = (vala_code_node_unref (return_expression_symbol), NULL)), _tmp10_);
			vala_symbol_set_active (return_expression_symbol, FALSE);
		}
		_tmp11_ = FALSE;
		_tmp12_ = FALSE;
		if (self->current_method != NULL) {
			_tmp12_ = !vala_method_get_no_array_length (self->current_method);
		} else {
			_tmp12_ = FALSE;
		}
		if (_tmp12_) {
			_tmp11_ = VALA_IS_ARRAY_TYPE (self->current_return_type);
		} else {
			_tmp11_ = FALSE;
		}
		/* return array length if appropriate*/
		if (_tmp11_) {
			ValaLocalVariable* return_expr_decl;
			ValaCCodeCommaExpression* ccomma;
			ValaCCodeAssignment* _tmp14_;
			ValaCCodeExpression* _tmp13_;
			ValaArrayType* _tmp15_;
			ValaArrayType* array_type;
			ValaCCodeExpression* _tmp20_;
			return_expr_decl = vala_ccode_base_module_get_temp_variable (self, vala_expression_get_value_type (vala_return_statement_get_return_expression (stmt)), TRUE, (ValaCodeNode*) stmt);
			ccomma = vala_ccode_comma_expression_new ();
			_tmp14_ = NULL;
			_tmp13_ = NULL;
			vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp14_ = vala_ccode_assignment_new (_tmp13_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) return_expr_decl)), VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_return_statement_get_return_expression (stmt))), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
			(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
			(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL));
			_tmp15_ = NULL;
			array_type = (_tmp15_ = VALA_ARRAY_TYPE (self->current_return_type), (_tmp15_ == NULL) ? NULL : vala_code_node_ref (_tmp15_));
			{
				gint dim;
				dim = 1;
				for (; dim <= vala_array_type_get_rank (array_type); dim++) {
					ValaCCodeIdentifier* _tmp17_;
					char* _tmp16_;
					ValaCCodeUnaryExpression* _tmp18_;
					ValaCCodeUnaryExpression* len_l;
					ValaCCodeExpression* len_r;
					ValaCCodeAssignment* _tmp19_;
					_tmp17_ = NULL;
					_tmp16_ = NULL;
					_tmp18_ = NULL;
					len_l = (_tmp18_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_POINTER_INDIRECTION, (ValaCCodeExpression*) (_tmp17_ = vala_ccode_identifier_new (_tmp16_ = vala_ccode_module_get_array_length_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), "result", dim)))), (_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_ccode_node_unref (_tmp17_), NULL)), _tmp16_ = (g_free (_tmp16_), NULL), _tmp18_);
					len_r = vala_ccode_module_get_array_length_cexpression (vala_ccode_module_get_head ((ValaCCodeModule*) self), vala_return_statement_get_return_expression (stmt), dim);
					_tmp19_ = NULL;
					vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp19_ = vala_ccode_assignment_new ((ValaCCodeExpression*) len_l, len_r, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
					(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
					(len_l == NULL) ? NULL : (len_l = (vala_ccode_node_unref (len_l), NULL));
					(len_r == NULL) ? NULL : (len_r = (vala_ccode_node_unref (len_r), NULL));
				}
			}
			_tmp20_ = NULL;
			vala_ccode_comma_expression_append_expression (ccomma, _tmp20_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) return_expr_decl)));
			(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL));
			vala_code_node_set_ccodenode ((ValaCodeNode*) vala_return_statement_get_return_expression (stmt), (ValaCCodeNode*) ccomma);
			gee_collection_add ((GeeCollection*) vala_return_statement_get_return_expression (stmt)->temp_vars, return_expr_decl);
			(return_expr_decl == NULL) ? NULL : (return_expr_decl = (vala_code_node_unref (return_expr_decl), NULL));
			(ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL));
			(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
		}
		vala_ccode_base_module_create_local_free_expr (self, vala_return_statement_get_return_expression (stmt));
		_tmp21_ = FALSE;
		_tmp22_ = FALSE;
		if (self->current_property_accessor != NULL) {
			_tmp22_ = vala_property_accessor_get_readable (self->current_property_accessor);
		} else {
			_tmp22_ = FALSE;
		}
		if (_tmp22_) {
			_tmp21_ = vala_data_type_is_real_struct_type (vala_property_get_property_type (vala_property_accessor_get_prop (self->current_property_accessor)));
		} else {
			_tmp21_ = FALSE;
		}
		/* Property getters of non simple structs shall return the struct value as out parameter,
		 therefore replace any return statement with an assignment statement to the out formal
		 paramenter and insert an empty return statement afterwards.*/
		if (_tmp21_) {
			ValaCCodeFragment* cfragment;
			ValaCCodeExpressionStatement* _tmp25_;
			ValaCCodeAssignment* _tmp24_;
			ValaCCodeIdentifier* _tmp23_;
			ValaCCodeReturnStatement* _tmp26_;
			cfragment = vala_ccode_fragment_new ();
			_tmp25_ = NULL;
			_tmp24_ = NULL;
			_tmp23_ = NULL;
			vala_ccode_fragment_append (cfragment, (ValaCCodeNode*) (_tmp25_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp24_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp23_ = vala_ccode_identifier_new ("*value")), VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_return_statement_get_return_expression (stmt))), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)))));
			(_tmp25_ == NULL) ? NULL : (_tmp25_ = (vala_ccode_node_unref (_tmp25_), NULL));
			(_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_ccode_node_unref (_tmp24_), NULL));
			(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_ccode_node_unref (_tmp23_), NULL));
			_tmp26_ = NULL;
			vala_ccode_fragment_append (cfragment, (ValaCCodeNode*) (_tmp26_ = vala_ccode_return_statement_new (NULL)));
			(_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_ccode_node_unref (_tmp26_), NULL));
			vala_code_node_set_ccodenode ((ValaCodeNode*) stmt, (ValaCCodeNode*) cfragment);
			(cfragment == NULL) ? NULL : (cfragment = (vala_ccode_node_unref (cfragment), NULL));
		} else {
			ValaCCodeReturnStatement* _tmp27_;
			_tmp27_ = NULL;
			vala_code_node_set_ccodenode ((ValaCodeNode*) stmt, (ValaCCodeNode*) (_tmp27_ = vala_ccode_return_statement_new (VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_return_statement_get_return_expression (stmt))))));
			(_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL));
		}
		vala_ccode_base_module_create_temp_decl (self, (ValaStatement*) stmt, (GeeList*) vala_return_statement_get_return_expression (stmt)->temp_vars);
		if (return_expression_symbol != NULL) {
			vala_symbol_set_active (return_expression_symbol, TRUE);
		}
		(return_expression_symbol == NULL) ? NULL : (return_expression_symbol = (vala_code_node_unref (return_expression_symbol), NULL));
		(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
	}
}


char* vala_ccode_base_module_get_symbol_lock_name (ValaCCodeBaseModule* self, ValaSymbol* sym) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (sym != NULL, NULL);
	return g_strdup_printf ("__lock_%s", vala_symbol_get_name (sym));
}


static void vala_ccode_base_module_real_visit_lock_statement (ValaCCodeModule* base, ValaLockStatement* stmt) {
	ValaCCodeBaseModule * self;
	ValaCCodeFragment* cn;
	ValaCCodeExpression* l;
	ValaCCodeFunctionCall* fc;
	ValaExpression* _tmp0_;
	ValaExpression* inner_node;
	ValaCCodeExpression* _tmp7_;
	char* _tmp6_;
	ValaCCodeMemberAccess* _tmp5_;
	ValaCCodeFunctionCall* _tmp11_;
	ValaCCodeIdentifier* _tmp10_;
	char* _tmp9_;
	ValaMethod* _tmp8_;
	ValaCCodeUnaryExpression* _tmp12_;
	ValaCCodeExpressionStatement* _tmp13_;
	ValaCCodeFunctionCall* _tmp17_;
	ValaCCodeIdentifier* _tmp16_;
	char* _tmp15_;
	ValaMethod* _tmp14_;
	ValaCCodeUnaryExpression* _tmp18_;
	ValaCCodeExpressionStatement* _tmp19_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (stmt != NULL);
	cn = vala_ccode_fragment_new ();
	l = NULL;
	fc = NULL;
	_tmp0_ = NULL;
	inner_node = (_tmp0_ = vala_member_access_get_inner (VALA_MEMBER_ACCESS (vala_lock_statement_get_resource (stmt))), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	if (inner_node == NULL) {
		ValaCCodeExpression* _tmp1_;
		_tmp1_ = NULL;
		l = (_tmp1_ = (ValaCCodeExpression*) vala_ccode_identifier_new ("self"), (l == NULL) ? NULL : (l = (vala_ccode_node_unref (l), NULL)), _tmp1_);
	} else {
		if (vala_symbol_get_parent_symbol (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt))) != VALA_SYMBOL (self->current_type_symbol)) {
			ValaCCodeExpression* _tmp2_;
			_tmp2_ = NULL;
			l = (_tmp2_ = (ValaCCodeExpression*) vala_instance_cast_new (VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) inner_node)), VALA_TYPESYMBOL (vala_symbol_get_parent_symbol (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt))))), (l == NULL) ? NULL : (l = (vala_ccode_node_unref (l), NULL)), _tmp2_);
		} else {
			ValaCCodeExpression* _tmp4_;
			ValaCCodeExpression* _tmp3_;
			_tmp4_ = NULL;
			_tmp3_ = NULL;
			l = (_tmp4_ = (_tmp3_ = VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) inner_node)), (_tmp3_ == NULL) ? NULL : vala_ccode_node_ref (_tmp3_)), (l == NULL) ? NULL : (l = (vala_ccode_node_unref (l), NULL)), _tmp4_);
		}
	}
	_tmp7_ = NULL;
	_tmp6_ = NULL;
	_tmp5_ = NULL;
	l = (_tmp7_ = (ValaCCodeExpression*) vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp5_ = vala_ccode_member_access_new_pointer (l, "priv")), _tmp6_ = vala_ccode_base_module_get_symbol_lock_name (self, vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt)))), (l == NULL) ? NULL : (l = (vala_ccode_node_unref (l), NULL)), _tmp7_);
	_tmp6_ = (g_free (_tmp6_), NULL);
	(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
	_tmp11_ = NULL;
	_tmp10_ = NULL;
	_tmp9_ = NULL;
	_tmp8_ = NULL;
	fc = (_tmp11_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp10_ = vala_ccode_identifier_new (_tmp9_ = vala_method_get_cname (_tmp8_ = VALA_METHOD (vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) self->mutex_type), "lock")))))), (fc == NULL) ? NULL : (fc = (vala_ccode_node_unref (fc), NULL)), _tmp11_);
	(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
	_tmp9_ = (g_free (_tmp9_), NULL);
	(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_code_node_unref (_tmp8_), NULL));
	_tmp12_ = NULL;
	vala_ccode_function_call_add_argument (fc, (ValaCCodeExpression*) (_tmp12_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, l)));
	(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
	_tmp13_ = NULL;
	vala_ccode_fragment_append (cn, (ValaCCodeNode*) (_tmp13_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) fc)));
	(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL));
	vala_ccode_fragment_append (cn, vala_code_node_get_ccodenode ((ValaCodeNode*) vala_lock_statement_get_body (stmt)));
	_tmp17_ = NULL;
	_tmp16_ = NULL;
	_tmp15_ = NULL;
	_tmp14_ = NULL;
	fc = (_tmp17_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp16_ = vala_ccode_identifier_new (_tmp15_ = vala_method_get_cname (_tmp14_ = VALA_METHOD (vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) self->mutex_type), "unlock")))))), (fc == NULL) ? NULL : (fc = (vala_ccode_node_unref (fc), NULL)), _tmp17_);
	(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL));
	_tmp15_ = (g_free (_tmp15_), NULL);
	(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_code_node_unref (_tmp14_), NULL));
	_tmp18_ = NULL;
	vala_ccode_function_call_add_argument (fc, (ValaCCodeExpression*) (_tmp18_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, l)));
	(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
	_tmp19_ = NULL;
	vala_ccode_fragment_append (cn, (ValaCCodeNode*) (_tmp19_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) fc)));
	(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
	vala_code_node_set_ccodenode ((ValaCodeNode*) stmt, (ValaCCodeNode*) cn);
	(cn == NULL) ? NULL : (cn = (vala_ccode_node_unref (cn), NULL));
	(l == NULL) ? NULL : (l = (vala_ccode_node_unref (l), NULL));
	(fc == NULL) ? NULL : (fc = (vala_ccode_node_unref (fc), NULL));
	(inner_node == NULL) ? NULL : (inner_node = (vala_code_node_unref (inner_node), NULL));
}


static void vala_ccode_base_module_real_visit_delete_statement (ValaCCodeModule* base, ValaDeleteStatement* stmt) {
	ValaCCodeBaseModule * self;
	ValaPointerType* _tmp0_;
	ValaPointerType* pointer_type;
	ValaDataType* _tmp1_;
	ValaDataType* type;
	gboolean _tmp2_;
	ValaCCodeExpression* _tmp5_;
	ValaCCodeFunctionCall* _tmp6_;
	ValaCCodeFunctionCall* ccall;
	ValaCCodeExpressionStatement* _tmp7_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) stmt, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	_tmp0_ = NULL;
	pointer_type = (_tmp0_ = VALA_POINTER_TYPE (vala_expression_get_value_type (vala_delete_statement_get_expression (stmt))), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	_tmp1_ = NULL;
	type = (_tmp1_ = (ValaDataType*) pointer_type, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp2_ = FALSE;
	if (vala_data_type_get_data_type (vala_pointer_type_get_base_type (pointer_type)) != NULL) {
		_tmp2_ = vala_typesymbol_is_reference_type (vala_data_type_get_data_type (vala_pointer_type_get_base_type (pointer_type)));
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		ValaDataType* _tmp4_;
		ValaDataType* _tmp3_;
		_tmp4_ = NULL;
		_tmp3_ = NULL;
		type = (_tmp4_ = (_tmp3_ = vala_pointer_type_get_base_type (pointer_type), (_tmp3_ == NULL) ? NULL : vala_code_node_ref (_tmp3_)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp4_);
	}
	_tmp5_ = NULL;
	_tmp6_ = NULL;
	ccall = (_tmp6_ = vala_ccode_function_call_new (_tmp5_ = vala_ccode_base_module_get_destroy_func_expression (self, type)), (_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL)), _tmp6_);
	vala_ccode_function_call_add_argument (ccall, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_delete_statement_get_expression (stmt))));
	_tmp7_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) stmt, (ValaCCodeNode*) (_tmp7_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) ccall)));
	(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
	(pointer_type == NULL) ? NULL : (pointer_type = (vala_code_node_unref (pointer_type), NULL));
	(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
	(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
}


static void vala_ccode_base_module_real_visit_expression (ValaCCodeModule* base, ValaExpression* expr) {
	ValaCCodeBaseModule * self;
	gboolean _tmp0_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	_tmp0_ = FALSE;
	if (vala_code_node_get_ccodenode ((ValaCodeNode*) expr) != NULL) {
		_tmp0_ = !vala_expression_get_lvalue (expr);
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		gboolean _tmp1_;
		ValaCCodeExpression* _tmp3_;
		gboolean _tmp4_;
		_tmp1_ = FALSE;
		if (VALA_IS_GENERIC_TYPE (vala_expression_get_formal_value_type (expr))) {
			_tmp1_ = !VALA_IS_GENERIC_TYPE (vala_expression_get_value_type (expr));
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			if (vala_symbol_get_parent_symbol ((ValaSymbol*) vala_data_type_get_type_parameter (vala_expression_get_formal_value_type (expr))) != VALA_SYMBOL (self->garray_type)) {
				ValaCCodeExpression* _tmp2_;
				/* GArray doesn't use pointer-based generics*/
				_tmp2_ = NULL;
				vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp2_ = vala_ccode_base_module_convert_from_generic_pointer (self, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) expr)), vala_expression_get_value_type (expr))));
				(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_ccode_node_unref (_tmp2_), NULL));
			}
		}
		/* memory management, implicit casts, and boxing/unboxing*/
		_tmp3_ = NULL;
		vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp3_ = vala_ccode_base_module_transform_expression (self, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) expr)), vala_expression_get_value_type (expr), vala_expression_get_target_type (expr), expr)));
		(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
		_tmp4_ = FALSE;
		if (VALA_IS_GENERIC_TYPE (vala_expression_get_formal_target_type (expr))) {
			_tmp4_ = !VALA_IS_GENERIC_TYPE (vala_expression_get_target_type (expr));
		} else {
			_tmp4_ = FALSE;
		}
		if (_tmp4_) {
			if (vala_symbol_get_parent_symbol ((ValaSymbol*) vala_data_type_get_type_parameter (vala_expression_get_formal_target_type (expr))) != VALA_SYMBOL (self->garray_type)) {
				ValaCCodeExpression* _tmp5_;
				/* GArray doesn't use pointer-based generics*/
				_tmp5_ = NULL;
				vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp5_ = vala_ccode_base_module_convert_to_generic_pointer (self, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) expr)), vala_expression_get_target_type (expr))));
				(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
			}
		}
	}
}


static void vala_ccode_base_module_real_visit_boolean_literal (ValaCCodeModule* base, ValaBooleanLiteral* expr) {
	ValaCCodeBaseModule * self;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	if (vala_code_context_get_profile (self->priv->_context) == VALA_PROFILE_GOBJECT) {
		const char* _tmp0_;
		ValaCCodeConstant* _tmp1_;
		_tmp0_ = NULL;
		if (vala_boolean_literal_get_value (expr)) {
			_tmp0_ = "TRUE";
		} else {
			_tmp0_ = "FALSE";
		}
		_tmp1_ = NULL;
		vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp1_ = vala_ccode_constant_new (_tmp0_)));
		(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_ccode_node_unref (_tmp1_), NULL));
	} else {
		const char* _tmp2_;
		ValaCCodeConstant* _tmp3_;
		vala_ccode_declaration_space_add_include (self->source_declarations, "stdbool.h", FALSE);
		_tmp2_ = NULL;
		if (vala_boolean_literal_get_value (expr)) {
			_tmp2_ = "true";
		} else {
			_tmp2_ = "false";
		}
		_tmp3_ = NULL;
		vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp3_ = vala_ccode_constant_new (_tmp2_)));
		(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
	}
}


static void vala_ccode_base_module_real_visit_character_literal (ValaCCodeModule* base, ValaCharacterLiteral* expr) {
	ValaCCodeBaseModule * self;
	gboolean _tmp0_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	_tmp0_ = FALSE;
	if (vala_character_literal_get_char (expr) >= 0x20) {
		_tmp0_ = vala_character_literal_get_char (expr) < 0x80;
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		ValaCCodeConstant* _tmp1_;
		_tmp1_ = NULL;
		vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp1_ = vala_ccode_constant_new (vala_character_literal_get_value (expr))));
		(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_ccode_node_unref (_tmp1_), NULL));
	} else {
		ValaCCodeConstant* _tmp3_;
		char* _tmp2_;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp3_ = vala_ccode_constant_new (_tmp2_ = g_strdup_printf ("%uU", (guint) vala_character_literal_get_char (expr)))));
		(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
		_tmp2_ = (g_free (_tmp2_), NULL);
	}
}


static void vala_ccode_base_module_real_visit_integer_literal (ValaCCodeModule* base, ValaIntegerLiteral* expr) {
	ValaCCodeBaseModule * self;
	ValaCCodeConstant* _tmp0_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	_tmp0_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp0_ = vala_ccode_constant_new (vala_integer_literal_get_value (expr))));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
}


static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


static glong string_get_length (const char* self) {
	g_return_val_if_fail (self != NULL, 0L);
	return g_utf8_strlen (self, -1);
}


static gboolean string_contains (const char* self, const char* needle) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (needle != NULL, FALSE);
	return strstr (self, needle) != NULL;
}


static void vala_ccode_base_module_real_visit_real_literal (ValaCCodeModule* base, ValaRealLiteral* expr) {
	ValaCCodeBaseModule * self;
	const char* _tmp0_;
	char* c_literal;
	gboolean _tmp1_;
	gboolean _tmp3_;
	gboolean _tmp4_;
	ValaCCodeConstant* _tmp9_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	_tmp0_ = NULL;
	c_literal = (_tmp0_ = vala_real_literal_get_value (expr), (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_));
	_tmp1_ = FALSE;
	if (g_str_has_suffix (c_literal, "d")) {
		_tmp1_ = TRUE;
	} else {
		_tmp1_ = g_str_has_suffix (c_literal, "D");
	}
	if (_tmp1_) {
		char* _tmp2_;
		/* there is no suffix for double in C*/
		_tmp2_ = NULL;
		c_literal = (_tmp2_ = string_substring (c_literal, (glong) 0, string_get_length (c_literal) - 1), c_literal = (g_free (c_literal), NULL), _tmp2_);
	}
	_tmp3_ = FALSE;
	_tmp4_ = FALSE;
	if (string_contains (c_literal, ".")) {
		_tmp4_ = TRUE;
	} else {
		_tmp4_ = string_contains (c_literal, "e");
	}
	if (_tmp4_) {
		_tmp3_ = TRUE;
	} else {
		_tmp3_ = string_contains (c_literal, "E");
	}
	if (!_tmp3_) {
		gboolean _tmp5_;
		_tmp5_ = FALSE;
		if (string_contains (c_literal, "f")) {
			_tmp5_ = TRUE;
		} else {
			_tmp5_ = string_contains (c_literal, "F");
		}
		/* C requires period or exponent part for floating constants*/
		if (_tmp5_) {
			char* _tmp7_;
			char* _tmp6_;
			_tmp7_ = NULL;
			_tmp6_ = NULL;
			c_literal = (_tmp7_ = g_strconcat (_tmp6_ = string_substring (c_literal, (glong) 0, string_get_length (c_literal) - 1), ".f", NULL), c_literal = (g_free (c_literal), NULL), _tmp7_);
			_tmp6_ = (g_free (_tmp6_), NULL);
		} else {
			char* _tmp8_;
			_tmp8_ = NULL;
			c_literal = (_tmp8_ = g_strconcat (c_literal, ".", NULL), c_literal = (g_free (c_literal), NULL), _tmp8_);
		}
	}
	_tmp9_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp9_ = vala_ccode_constant_new (c_literal)));
	(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
	c_literal = (g_free (c_literal), NULL);
}


static void vala_ccode_base_module_real_visit_string_literal (ValaCCodeModule* base, ValaStringLiteral* expr) {
	ValaCCodeBaseModule * self;
	ValaCCodeConstant* _tmp0_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	_tmp0_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp0_ = vala_ccode_constant_new (vala_string_literal_get_value (expr))));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
}


static void vala_ccode_base_module_real_visit_null_literal (ValaCCodeModule* base, ValaNullLiteral* expr) {
	ValaCCodeBaseModule * self;
	ValaCCodeConstant* _tmp0_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	if (vala_code_context_get_profile (self->priv->_context) != VALA_PROFILE_GOBJECT) {
		vala_ccode_declaration_space_add_include (self->source_declarations, "stddef.h", FALSE);
	}
	_tmp0_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp0_ = vala_ccode_constant_new ("NULL")));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
}


static char* vala_ccode_base_module_real_get_delegate_target_cname (ValaCCodeBaseModule* self, const char* delegate_cname) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (delegate_cname != NULL, NULL);
	g_assert_not_reached ();
}


char* vala_ccode_base_module_get_delegate_target_cname (ValaCCodeBaseModule* self, const char* delegate_cname) {
	return VALA_CCODE_BASE_MODULE_GET_CLASS (self)->get_delegate_target_cname (self, delegate_cname);
}


static ValaCCodeExpression* vala_ccode_base_module_real_get_delegate_target_cexpression (ValaCCodeBaseModule* self, ValaExpression* delegate_expr) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (delegate_expr != NULL, NULL);
	g_assert_not_reached ();
}


ValaCCodeExpression* vala_ccode_base_module_get_delegate_target_cexpression (ValaCCodeBaseModule* self, ValaExpression* delegate_expr) {
	return VALA_CCODE_BASE_MODULE_GET_CLASS (self)->get_delegate_target_cexpression (self, delegate_expr);
}


static char* vala_ccode_base_module_real_get_delegate_target_destroy_notify_cname (ValaCCodeBaseModule* self, const char* delegate_cname) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (delegate_cname != NULL, NULL);
	g_assert_not_reached ();
}


char* vala_ccode_base_module_get_delegate_target_destroy_notify_cname (ValaCCodeBaseModule* self, const char* delegate_cname) {
	return VALA_CCODE_BASE_MODULE_GET_CLASS (self)->get_delegate_target_destroy_notify_cname (self, delegate_cname);
}


static void vala_ccode_base_module_real_visit_base_access (ValaCCodeModule* base, ValaBaseAccess* expr) {
	ValaCCodeBaseModule * self;
	ValaInstanceCast* _tmp1_;
	ValaCCodeIdentifier* _tmp0_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp1_ = vala_instance_cast_new ((ValaCCodeExpression*) (_tmp0_ = vala_ccode_identifier_new ("self")), vala_data_type_get_data_type (vala_expression_get_value_type ((ValaExpression*) expr)))));
	(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_ccode_node_unref (_tmp1_), NULL));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
}


static void vala_ccode_base_module_real_visit_postfix_expression (ValaCCodeModule* base, ValaPostfixExpression* expr) {
	ValaCCodeBaseModule * self;
	ValaMemberAccess* ma;
	ValaCCodeUnaryOperator _tmp8_;
	ValaCCodeUnaryOperator op;
	ValaCCodeUnaryExpression* _tmp9_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	ma = vala_ccode_base_module_find_property_access (self, vala_postfix_expression_get_inner (expr));
	if (ma != NULL) {
		ValaProperty* _tmp0_;
		ValaProperty* prop;
		ValaCCodeCommaExpression* ccomma;
		ValaLocalVariable* temp_decl;
		ValaCCodeAssignment* _tmp2_;
		ValaCCodeExpression* _tmp1_;
		ValaCCodeBinaryOperator _tmp3_;
		ValaCCodeBinaryOperator op;
		ValaCCodeConstant* _tmp5_;
		ValaCCodeExpression* _tmp4_;
		ValaCCodeBinaryExpression* _tmp6_;
		ValaCCodeBinaryExpression* cexpr;
		ValaCCodeFunctionCall* ccall;
		ValaCCodeIdentifier* _tmp7_;
		/* property postfix expression*/
		_tmp0_ = NULL;
		prop = (_tmp0_ = VALA_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
		ccomma = vala_ccode_comma_expression_new ();
		/* assign current value to temp variable*/
		temp_decl = vala_ccode_base_module_get_temp_variable (self, vala_property_get_property_type (prop), TRUE, (ValaCodeNode*) expr);
		gee_list_insert ((GeeList*) self->temp_vars, 0, temp_decl);
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp2_ = vala_ccode_assignment_new (_tmp1_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) temp_decl)), VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_postfix_expression_get_inner (expr))), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
		(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_ccode_node_unref (_tmp2_), NULL));
		(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_ccode_node_unref (_tmp1_), NULL));
		_tmp3_ = 0;
		if (vala_postfix_expression_get_increment (expr)) {
			_tmp3_ = VALA_CCODE_BINARY_OPERATOR_PLUS;
		} else {
			_tmp3_ = VALA_CCODE_BINARY_OPERATOR_MINUS;
		}
		/* increment/decrement property*/
		op = _tmp3_;
		_tmp5_ = NULL;
		_tmp4_ = NULL;
		_tmp6_ = NULL;
		cexpr = (_tmp6_ = vala_ccode_binary_expression_new (op, _tmp4_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) temp_decl)), (ValaCCodeExpression*) (_tmp5_ = vala_ccode_constant_new ("1"))), (_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL)), (_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL)), _tmp6_);
		ccall = vala_ccode_base_module_get_property_set_call (self, prop, ma, (ValaCCodeExpression*) cexpr);
		vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) ccall);
		/* return previous value*/
		_tmp7_ = NULL;
		vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp7_ = vala_ccode_identifier_new (vala_symbol_get_name ((ValaSymbol*) temp_decl))));
		(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
		vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) ccomma);
		(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
		(ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL));
		(temp_decl == NULL) ? NULL : (temp_decl = (vala_code_node_unref (temp_decl), NULL));
		(cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL));
		(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
		return;
	}
	_tmp8_ = 0;
	if (vala_postfix_expression_get_increment (expr)) {
		_tmp8_ = VALA_CCODE_UNARY_OPERATOR_POSTFIX_INCREMENT;
	} else {
		_tmp8_ = VALA_CCODE_UNARY_OPERATOR_POSTFIX_DECREMENT;
	}
	op = _tmp8_;
	_tmp9_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp9_ = vala_ccode_unary_expression_new (op, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_postfix_expression_get_inner (expr))))));
	(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
	(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
}


static ValaMemberAccess* vala_ccode_base_module_find_property_access (ValaCCodeBaseModule* self, ValaExpression* expr) {
	ValaMemberAccess* _tmp1_;
	ValaMemberAccess* ma;
	ValaMemberAccess* _tmp3_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (expr != NULL, NULL);
	if (!VALA_IS_MEMBER_ACCESS (expr)) {
		return NULL;
	}
	_tmp1_ = NULL;
	ma = (_tmp1_ = VALA_MEMBER_ACCESS (expr), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	if (VALA_IS_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
		return ma;
	}
	_tmp3_ = NULL;
	return (_tmp3_ = NULL, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp3_);
}


gboolean vala_ccode_base_module_requires_copy (ValaCCodeBaseModule* self, ValaDataType* type) {
	ValaClass* _tmp2_;
	ValaTypeSymbol* _tmp1_;
	ValaClass* cl;
	gboolean _tmp3_;
	gboolean _tmp4_;
	gboolean _tmp9_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type != NULL, FALSE);
	if (!vala_data_type_is_disposable (type)) {
		return FALSE;
	}
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	cl = (_tmp2_ = (_tmp1_ = vala_data_type_get_data_type (type), VALA_IS_CLASS (_tmp1_) ? ((ValaClass*) _tmp1_) : NULL), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
	_tmp3_ = FALSE;
	_tmp4_ = FALSE;
	if (cl != NULL) {
		_tmp4_ = vala_typesymbol_is_reference_counting ((ValaTypeSymbol*) cl);
	} else {
		_tmp4_ = FALSE;
	}
	if (_tmp4_) {
		char* _tmp5_;
		_tmp5_ = NULL;
		_tmp3_ = _vala_strcmp0 (_tmp5_ = vala_typesymbol_get_ref_function ((ValaTypeSymbol*) cl), "") == 0;
		_tmp5_ = (g_free (_tmp5_), NULL);
	} else {
		_tmp3_ = FALSE;
	}
	if (_tmp3_) {
		gboolean _tmp6_;
		/* empty ref_function => no ref necessary*/
		return (_tmp6_ = FALSE, (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp6_);
	}
	if (vala_data_type_get_type_parameter (type) != NULL) {
		gboolean _tmp7_;
		_tmp7_ = FALSE;
		if (!VALA_IS_CLASS (self->current_type_symbol)) {
			_tmp7_ = TRUE;
		} else {
			_tmp7_ = vala_class_get_is_compact (self->current_class);
		}
		if (_tmp7_) {
			gboolean _tmp8_;
			return (_tmp8_ = FALSE, (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp8_);
		}
	}
	return (_tmp9_ = TRUE, (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp9_);
}


gboolean vala_ccode_base_module_requires_destroy (ValaCCodeBaseModule* self, ValaDataType* type) {
	ValaArrayType* _tmp2_;
	ValaDataType* _tmp1_;
	ValaArrayType* array_type;
	gboolean _tmp3_;
	ValaClass* _tmp6_;
	ValaTypeSymbol* _tmp5_;
	ValaClass* cl;
	gboolean _tmp7_;
	gboolean _tmp8_;
	gboolean _tmp13_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type != NULL, FALSE);
	if (!vala_data_type_is_disposable (type)) {
		return FALSE;
	}
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	array_type = (_tmp2_ = (_tmp1_ = type, VALA_IS_ARRAY_TYPE (_tmp1_) ? ((ValaArrayType*) _tmp1_) : NULL), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
	_tmp3_ = FALSE;
	if (array_type != NULL) {
		_tmp3_ = vala_array_type_get_fixed_length (array_type);
	} else {
		_tmp3_ = FALSE;
	}
	if (_tmp3_) {
		gboolean _tmp4_;
		return (_tmp4_ = vala_ccode_base_module_requires_destroy (self, vala_array_type_get_element_type (array_type)), (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), _tmp4_);
	}
	_tmp6_ = NULL;
	_tmp5_ = NULL;
	cl = (_tmp6_ = (_tmp5_ = vala_data_type_get_data_type (type), VALA_IS_CLASS (_tmp5_) ? ((ValaClass*) _tmp5_) : NULL), (_tmp6_ == NULL) ? NULL : vala_code_node_ref (_tmp6_));
	_tmp7_ = FALSE;
	_tmp8_ = FALSE;
	if (cl != NULL) {
		_tmp8_ = vala_typesymbol_is_reference_counting ((ValaTypeSymbol*) cl);
	} else {
		_tmp8_ = FALSE;
	}
	if (_tmp8_) {
		char* _tmp9_;
		_tmp9_ = NULL;
		_tmp7_ = _vala_strcmp0 (_tmp9_ = vala_typesymbol_get_unref_function ((ValaTypeSymbol*) cl), "") == 0;
		_tmp9_ = (g_free (_tmp9_), NULL);
	} else {
		_tmp7_ = FALSE;
	}
	if (_tmp7_) {
		gboolean _tmp10_;
		/* empty unref_function => no unref necessary*/
		return (_tmp10_ = FALSE, (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp10_);
	}
	if (vala_data_type_get_type_parameter (type) != NULL) {
		gboolean _tmp11_;
		_tmp11_ = FALSE;
		if (!VALA_IS_CLASS (self->current_type_symbol)) {
			_tmp11_ = TRUE;
		} else {
			_tmp11_ = vala_class_get_is_compact (self->current_class);
		}
		if (_tmp11_) {
			gboolean _tmp12_;
			return (_tmp12_ = FALSE, (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp12_);
		}
	}
	return (_tmp13_ = TRUE, (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp13_);
}


static gboolean vala_ccode_base_module_is_ref_function_void (ValaCCodeBaseModule* self, ValaDataType* type) {
	ValaClass* _tmp1_;
	ValaTypeSymbol* _tmp0_;
	ValaClass* cl;
	gboolean _tmp2_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type != NULL, FALSE);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	cl = (_tmp1_ = (_tmp0_ = vala_data_type_get_data_type (type), VALA_IS_CLASS (_tmp0_) ? ((ValaClass*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp2_ = FALSE;
	if (cl != NULL) {
		_tmp2_ = vala_class_get_ref_function_void (cl);
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		gboolean _tmp3_;
		return (_tmp3_ = TRUE, (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp3_);
	} else {
		gboolean _tmp4_;
		return (_tmp4_ = FALSE, (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp4_);
	}
	(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
}


static ValaCCodeExpression* vala_ccode_base_module_real_get_ref_cexpression (ValaCCodeBaseModule* self, ValaDataType* expression_type, ValaCCodeExpression* cexpr, ValaExpression* expr, ValaCodeNode* node) {
	gboolean _tmp0_;
	ValaCCodeExpression* dupexpr;
	ValaCCodeFunctionCall* ccall;
	gboolean _tmp22_;
	gboolean _tmp23_;
	gboolean _tmp24_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (expression_type != NULL, NULL);
	g_return_val_if_fail (cexpr != NULL, NULL);
	g_return_val_if_fail (node != NULL, NULL);
	_tmp0_ = FALSE;
	if (VALA_IS_VALUE_TYPE (expression_type)) {
		_tmp0_ = !vala_data_type_get_nullable (expression_type);
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		ValaLocalVariable* decl;
		ValaCCodeExpression* ctemp;
		ValaValueType* _tmp1_;
		ValaValueType* vt;
		ValaStruct* _tmp2_;
		ValaStruct* st;
		ValaCCodeIdentifier* _tmp4_;
		char* _tmp3_;
		ValaCCodeFunctionCall* _tmp5_;
		ValaCCodeFunctionCall* copy_call;
		ValaCCodeUnaryExpression* _tmp6_;
		ValaCCodeUnaryExpression* _tmp7_;
		ValaCCodeCommaExpression* ccomma;
		char* _tmp8_;
		gboolean _tmp9_;
		/* normal value type, no null check
		 (copy (&expr, &temp), temp)*/
		decl = vala_ccode_base_module_get_temp_variable (self, expression_type, FALSE, node);
		gee_list_insert ((GeeList*) self->temp_vars, 0, decl);
		ctemp = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) decl));
		_tmp1_ = NULL;
		vt = (_tmp1_ = VALA_VALUE_TYPE (expression_type), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
		_tmp2_ = NULL;
		st = (_tmp2_ = VALA_STRUCT (vala_value_type_get_type_symbol (vt)), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
		_tmp4_ = NULL;
		_tmp3_ = NULL;
		_tmp5_ = NULL;
		copy_call = (_tmp5_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp4_ = vala_ccode_identifier_new (_tmp3_ = vala_typesymbol_get_copy_function ((ValaTypeSymbol*) st)))), (_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL)), _tmp3_ = (g_free (_tmp3_), NULL), _tmp5_);
		_tmp6_ = NULL;
		vala_ccode_function_call_add_argument (copy_call, (ValaCCodeExpression*) (_tmp6_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, cexpr)));
		(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
		_tmp7_ = NULL;
		vala_ccode_function_call_add_argument (copy_call, (ValaCCodeExpression*) (_tmp7_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, ctemp)));
		(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
		ccomma = vala_ccode_comma_expression_new ();
		_tmp8_ = NULL;
		if ((_tmp9_ = _vala_strcmp0 (_tmp8_ = vala_typesymbol_get_copy_function ((ValaTypeSymbol*) st), "g_value_copy") == 0, _tmp8_ = (g_free (_tmp8_), NULL), _tmp9_)) {
			ValaCCodeIdentifier* _tmp10_;
			ValaCCodeFunctionCall* _tmp11_;
			ValaCCodeFunctionCall* value_type_call;
			ValaCCodeUnaryExpression* _tmp12_;
			ValaCCodeIdentifier* _tmp13_;
			ValaCCodeFunctionCall* _tmp14_;
			ValaCCodeFunctionCall* init_call;
			ValaCCodeUnaryExpression* _tmp15_;
			/* GValue requires g_value_init in addition to g_value_copy*/
			_tmp10_ = NULL;
			_tmp11_ = NULL;
			value_type_call = (_tmp11_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp10_ = vala_ccode_identifier_new ("G_VALUE_TYPE"))), (_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL)), _tmp11_);
			_tmp12_ = NULL;
			vala_ccode_function_call_add_argument (value_type_call, (ValaCCodeExpression*) (_tmp12_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, cexpr)));
			(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
			_tmp13_ = NULL;
			_tmp14_ = NULL;
			init_call = (_tmp14_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp13_ = vala_ccode_identifier_new ("g_value_init"))), (_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL)), _tmp14_);
			_tmp15_ = NULL;
			vala_ccode_function_call_add_argument (init_call, (ValaCCodeExpression*) (_tmp15_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, ctemp)));
			(_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL));
			vala_ccode_function_call_add_argument (init_call, (ValaCCodeExpression*) value_type_call);
			vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) init_call);
			(value_type_call == NULL) ? NULL : (value_type_call = (vala_ccode_node_unref (value_type_call), NULL));
			(init_call == NULL) ? NULL : (init_call = (vala_ccode_node_unref (init_call), NULL));
		}
		vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) copy_call);
		vala_ccode_comma_expression_append_expression (ccomma, ctemp);
		if (vala_data_type_get_data_type (expression_type) == VALA_TYPESYMBOL (self->gvalue_type)) {
			ValaCCodeIdentifier* _tmp16_;
			ValaCCodeFunctionCall* _tmp17_;
			ValaCCodeFunctionCall* cisvalid;
			ValaCCodeUnaryExpression* _tmp18_;
			ValaCCodeExpression* _tmp19_;
			/* g_value_init/copy must not be called for uninitialized values*/
			_tmp16_ = NULL;
			_tmp17_ = NULL;
			cisvalid = (_tmp17_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp16_ = vala_ccode_identifier_new ("G_IS_VALUE"))), (_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL)), _tmp17_);
			_tmp18_ = NULL;
			vala_ccode_function_call_add_argument (cisvalid, (ValaCCodeExpression*) (_tmp18_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, cexpr)));
			(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
			_tmp19_ = NULL;
			return (_tmp19_ = (ValaCCodeExpression*) vala_ccode_conditional_expression_new ((ValaCCodeExpression*) cisvalid, (ValaCCodeExpression*) ccomma, cexpr), (cisvalid == NULL) ? NULL : (cisvalid = (vala_ccode_node_unref (cisvalid), NULL)), (decl == NULL) ? NULL : (decl = (vala_code_node_unref (decl), NULL)), (ctemp == NULL) ? NULL : (ctemp = (vala_ccode_node_unref (ctemp), NULL)), (vt == NULL) ? NULL : (vt = (vala_code_node_unref (vt), NULL)), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (copy_call == NULL) ? NULL : (copy_call = (vala_ccode_node_unref (copy_call), NULL)), (ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL)), _tmp19_);
		} else {
			ValaCCodeExpression* _tmp20_;
			_tmp20_ = NULL;
			return (_tmp20_ = (ValaCCodeExpression*) ccomma, (decl == NULL) ? NULL : (decl = (vala_code_node_unref (decl), NULL)), (ctemp == NULL) ? NULL : (ctemp = (vala_ccode_node_unref (ctemp), NULL)), (vt == NULL) ? NULL : (vt = (vala_code_node_unref (vt), NULL)), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (copy_call == NULL) ? NULL : (copy_call = (vala_ccode_node_unref (copy_call), NULL)), _tmp20_);
		}
		(decl == NULL) ? NULL : (decl = (vala_code_node_unref (decl), NULL));
		(ctemp == NULL) ? NULL : (ctemp = (vala_ccode_node_unref (ctemp), NULL));
		(vt == NULL) ? NULL : (vt = (vala_code_node_unref (vt), NULL));
		(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		(copy_call == NULL) ? NULL : (copy_call = (vala_ccode_node_unref (copy_call), NULL));
		(ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL));
	}
	/* (temp = expr, temp == NULL ? NULL : ref (temp))
	 *
	 * can be simplified to
	 * ref (expr)
	 * if static type of expr is non-null
	 */
	dupexpr = vala_ccode_base_module_get_dup_func_expression (self, expression_type, vala_code_node_get_source_reference (node));
	if (dupexpr == NULL) {
		ValaCCodeExpression* _tmp21_;
		vala_code_node_set_error (node, TRUE);
		_tmp21_ = NULL;
		return (_tmp21_ = NULL, (dupexpr == NULL) ? NULL : (dupexpr = (vala_ccode_node_unref (dupexpr), NULL)), _tmp21_);
	}
	ccall = vala_ccode_function_call_new (dupexpr);
	_tmp22_ = FALSE;
	_tmp23_ = FALSE;
	_tmp24_ = FALSE;
	if (!VALA_IS_ARRAY_TYPE (expression_type)) {
		_tmp24_ = expr != NULL;
	} else {
		_tmp24_ = FALSE;
	}
	if (_tmp24_) {
		_tmp23_ = vala_expression_is_non_null (expr);
	} else {
		_tmp23_ = FALSE;
	}
	if (_tmp23_) {
		_tmp22_ = !vala_ccode_base_module_is_ref_function_void (self, expression_type);
	} else {
		_tmp22_ = FALSE;
	}
	if (_tmp22_) {
		ValaCCodeExpression* _tmp25_;
		/* expression is non-null*/
		vala_ccode_function_call_add_argument (ccall, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) expr)));
		_tmp25_ = NULL;
		return (_tmp25_ = (ValaCCodeExpression*) ccall, (dupexpr == NULL) ? NULL : (dupexpr = (vala_ccode_node_unref (dupexpr), NULL)), _tmp25_);
	} else {
		ValaLocalVariable* decl;
		ValaCCodeExpression* ctemp;
		ValaCCodeConstant* _tmp26_;
		ValaCCodeBinaryExpression* _tmp27_;
		ValaCCodeBinaryExpression* cisnull;
		ValaCCodeCommaExpression* ccomma;
		ValaCCodeAssignment* _tmp39_;
		ValaCCodeExpression* cifnull;
		ValaCCodeConditionalExpression* _tmp42_;
		ValaCCodeExpression* _tmp43_;
		decl = vala_ccode_base_module_get_temp_variable (self, expression_type, FALSE, node);
		gee_list_insert ((GeeList*) self->temp_vars, 0, decl);
		ctemp = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) decl));
		_tmp26_ = NULL;
		_tmp27_ = NULL;
		cisnull = (_tmp27_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, ctemp, (ValaCCodeExpression*) (_tmp26_ = vala_ccode_constant_new ("NULL"))), (_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_ccode_node_unref (_tmp26_), NULL)), _tmp27_);
		if (vala_data_type_get_type_parameter (expression_type) != NULL) {
			ValaCCodeConstant* _tmp31_;
			ValaCCodeExpression* _tmp30_;
			ValaCCodeBinaryExpression* _tmp32_;
			ValaCCodeBinaryExpression* cdupisnull;
			ValaCCodeBinaryExpression* _tmp33_;
			if (!VALA_IS_CLASS (self->current_type_symbol)) {
				ValaCCodeExpression* _tmp28_;
				ValaCCodeExpression* _tmp29_;
				_tmp28_ = NULL;
				_tmp29_ = NULL;
				return (_tmp29_ = (_tmp28_ = cexpr, (_tmp28_ == NULL) ? NULL : vala_ccode_node_ref (_tmp28_)), (decl == NULL) ? NULL : (decl = (vala_code_node_unref (decl), NULL)), (ctemp == NULL) ? NULL : (ctemp = (vala_ccode_node_unref (ctemp), NULL)), (cisnull == NULL) ? NULL : (cisnull = (vala_ccode_node_unref (cisnull), NULL)), (dupexpr == NULL) ? NULL : (dupexpr = (vala_ccode_node_unref (dupexpr), NULL)), (ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL)), _tmp29_);
			}
			/* dup functions are optional for type parameters*/
			_tmp31_ = NULL;
			_tmp30_ = NULL;
			_tmp32_ = NULL;
			cdupisnull = (_tmp32_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, _tmp30_ = vala_ccode_base_module_get_dup_func_expression (self, expression_type, vala_code_node_get_source_reference (node)), (ValaCCodeExpression*) (_tmp31_ = vala_ccode_constant_new ("NULL"))), (_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_ccode_node_unref (_tmp31_), NULL)), (_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL)), _tmp32_);
			_tmp33_ = NULL;
			cisnull = (_tmp33_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_OR, (ValaCCodeExpression*) cisnull, (ValaCCodeExpression*) cdupisnull), (cisnull == NULL) ? NULL : (cisnull = (vala_ccode_node_unref (cisnull), NULL)), _tmp33_);
			(cdupisnull == NULL) ? NULL : (cdupisnull = (vala_ccode_node_unref (cdupisnull), NULL));
		}
		if (vala_data_type_get_type_parameter (expression_type) != NULL) {
			ValaCCodeCastExpression* _tmp34_;
			/* cast from gconstpointer to gpointer as GBoxedCopyFunc expects gpointer*/
			_tmp34_ = NULL;
			vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp34_ = vala_ccode_cast_expression_new (ctemp, "gpointer")));
			(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_ccode_node_unref (_tmp34_), NULL));
		} else {
			vala_ccode_function_call_add_argument (ccall, ctemp);
		}
		if (VALA_IS_ARRAY_TYPE (expression_type)) {
			ValaArrayType* _tmp35_;
			ValaArrayType* array_type;
			gboolean first;
			ValaCCodeExpression* csizeexpr;
			_tmp35_ = NULL;
			array_type = (_tmp35_ = VALA_ARRAY_TYPE (expression_type), (_tmp35_ == NULL) ? NULL : vala_code_node_ref (_tmp35_));
			first = TRUE;
			csizeexpr = NULL;
			{
				gint dim;
				dim = 1;
				for (; dim <= vala_array_type_get_rank (array_type); dim++) {
					if (first) {
						ValaCCodeExpression* _tmp36_;
						_tmp36_ = NULL;
						csizeexpr = (_tmp36_ = vala_ccode_module_get_array_length_cexpression (vala_ccode_module_get_head ((ValaCCodeModule*) self), expr, dim), (csizeexpr == NULL) ? NULL : (csizeexpr = (vala_ccode_node_unref (csizeexpr), NULL)), _tmp36_);
						first = FALSE;
					} else {
						ValaCCodeExpression* _tmp38_;
						ValaCCodeExpression* _tmp37_;
						_tmp38_ = NULL;
						_tmp37_ = NULL;
						csizeexpr = (_tmp38_ = (ValaCCodeExpression*) vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_MUL, csizeexpr, _tmp37_ = vala_ccode_module_get_array_length_cexpression (vala_ccode_module_get_head ((ValaCCodeModule*) self), expr, dim)), (csizeexpr == NULL) ? NULL : (csizeexpr = (vala_ccode_node_unref (csizeexpr), NULL)), _tmp38_);
						(_tmp37_ == NULL) ? NULL : (_tmp37_ = (vala_ccode_node_unref (_tmp37_), NULL));
					}
				}
			}
			vala_ccode_function_call_add_argument (ccall, csizeexpr);
			(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
			(csizeexpr == NULL) ? NULL : (csizeexpr = (vala_ccode_node_unref (csizeexpr), NULL));
		}
		ccomma = vala_ccode_comma_expression_new ();
		_tmp39_ = NULL;
		vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp39_ = vala_ccode_assignment_new (ctemp, cexpr, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
		(_tmp39_ == NULL) ? NULL : (_tmp39_ = (vala_ccode_node_unref (_tmp39_), NULL));
		cifnull = NULL;
		if (vala_data_type_get_data_type (expression_type) != NULL) {
			ValaCCodeExpression* _tmp40_;
			_tmp40_ = NULL;
			cifnull = (_tmp40_ = (ValaCCodeExpression*) vala_ccode_constant_new ("NULL"), (cifnull == NULL) ? NULL : (cifnull = (vala_ccode_node_unref (cifnull), NULL)), _tmp40_);
		} else {
			ValaCCodeExpression* _tmp41_;
			/* the value might be non-null even when the dup function is null,
			 so we may not just use NULL for type parameters
			 cast from gconstpointer to gpointer as methods in
			 generic classes may not return gconstpointer*/
			_tmp41_ = NULL;
			cifnull = (_tmp41_ = (ValaCCodeExpression*) vala_ccode_cast_expression_new (ctemp, "gpointer"), (cifnull == NULL) ? NULL : (cifnull = (vala_ccode_node_unref (cifnull), NULL)), _tmp41_);
		}
		_tmp42_ = NULL;
		vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp42_ = vala_ccode_conditional_expression_new ((ValaCCodeExpression*) cisnull, cifnull, (ValaCCodeExpression*) ccall)));
		(_tmp42_ == NULL) ? NULL : (_tmp42_ = (vala_ccode_node_unref (_tmp42_), NULL));
		/* repeat temp variable at the end of the comma expression
		 if the ref function returns void*/
		if (vala_ccode_base_module_is_ref_function_void (self, expression_type)) {
			vala_ccode_comma_expression_append_expression (ccomma, ctemp);
		}
		_tmp43_ = NULL;
		return (_tmp43_ = (ValaCCodeExpression*) ccomma, (decl == NULL) ? NULL : (decl = (vala_code_node_unref (decl), NULL)), (ctemp == NULL) ? NULL : (ctemp = (vala_ccode_node_unref (ctemp), NULL)), (cisnull == NULL) ? NULL : (cisnull = (vala_ccode_node_unref (cisnull), NULL)), (cifnull == NULL) ? NULL : (cifnull = (vala_ccode_node_unref (cifnull), NULL)), (dupexpr == NULL) ? NULL : (dupexpr = (vala_ccode_node_unref (dupexpr), NULL)), (ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL)), _tmp43_);
	}
	(dupexpr == NULL) ? NULL : (dupexpr = (vala_ccode_node_unref (dupexpr), NULL));
	(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
}


ValaCCodeExpression* vala_ccode_base_module_get_ref_cexpression (ValaCCodeBaseModule* self, ValaDataType* expression_type, ValaCCodeExpression* cexpr, ValaExpression* expr, ValaCodeNode* node) {
	return VALA_CCODE_BASE_MODULE_GET_CLASS (self)->get_ref_cexpression (self, expression_type, cexpr, expr, node);
}


static gboolean vala_ccode_base_module_is_reference_type_argument (ValaCCodeBaseModule* self, ValaDataType* type_arg) {
	gboolean _tmp0_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type_arg != NULL, FALSE);
	_tmp0_ = FALSE;
	if (vala_data_type_get_data_type (type_arg) != NULL) {
		_tmp0_ = vala_typesymbol_is_reference_type (vala_data_type_get_data_type (type_arg));
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		return TRUE;
	} else {
		return FALSE;
	}
}


static gboolean vala_ccode_base_module_is_nullable_value_type_argument (ValaCCodeBaseModule* self, ValaDataType* type_arg) {
	gboolean _tmp0_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type_arg != NULL, FALSE);
	_tmp0_ = FALSE;
	if (VALA_IS_VALUE_TYPE (type_arg)) {
		_tmp0_ = vala_data_type_get_nullable (type_arg);
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		return TRUE;
	} else {
		return FALSE;
	}
}


static gboolean vala_ccode_base_module_is_signed_integer_type_argument (ValaCCodeBaseModule* self, ValaDataType* type_arg) {
	ValaStruct* _tmp1_;
	ValaTypeSymbol* _tmp0_;
	ValaStruct* st;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type_arg != NULL, FALSE);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	st = (_tmp1_ = (_tmp0_ = vala_data_type_get_data_type (type_arg), VALA_IS_STRUCT (_tmp0_) ? ((ValaStruct*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	if (vala_data_type_get_nullable (type_arg)) {
		gboolean _tmp2_;
		return (_tmp2_ = FALSE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp2_);
	} else {
		if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->bool_type)) {
			gboolean _tmp3_;
			return (_tmp3_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp3_);
		} else {
			if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->char_type)) {
				gboolean _tmp4_;
				return (_tmp4_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp4_);
			} else {
				gboolean _tmp5_;
				_tmp5_ = FALSE;
				if (self->unichar_type != NULL) {
					_tmp5_ = VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->unichar_type);
				} else {
					_tmp5_ = FALSE;
				}
				if (_tmp5_) {
					gboolean _tmp6_;
					return (_tmp6_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp6_);
				} else {
					if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->short_type)) {
						gboolean _tmp7_;
						return (_tmp7_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp7_);
					} else {
						if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->int_type)) {
							gboolean _tmp8_;
							return (_tmp8_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp8_);
						} else {
							if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->long_type)) {
								gboolean _tmp9_;
								return (_tmp9_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp9_);
							} else {
								if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->int8_type)) {
									gboolean _tmp10_;
									return (_tmp10_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp10_);
								} else {
									if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->int16_type)) {
										gboolean _tmp11_;
										return (_tmp11_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp11_);
									} else {
										if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->int32_type)) {
											gboolean _tmp12_;
											return (_tmp12_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp12_);
										} else {
											if (VALA_TYPESYMBOL (st) == self->gtype_type) {
												gboolean _tmp13_;
												return (_tmp13_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp13_);
											} else {
												gboolean _tmp14_;
												return (_tmp14_ = FALSE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp14_);
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
}


static gboolean vala_ccode_base_module_is_unsigned_integer_type_argument (ValaCCodeBaseModule* self, ValaDataType* type_arg) {
	ValaStruct* _tmp1_;
	ValaTypeSymbol* _tmp0_;
	ValaStruct* st;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type_arg != NULL, FALSE);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	st = (_tmp1_ = (_tmp0_ = vala_data_type_get_data_type (type_arg), VALA_IS_STRUCT (_tmp0_) ? ((ValaStruct*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	if (vala_data_type_get_nullable (type_arg)) {
		gboolean _tmp2_;
		return (_tmp2_ = FALSE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp2_);
	} else {
		if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->uchar_type)) {
			gboolean _tmp3_;
			return (_tmp3_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp3_);
		} else {
			if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->ushort_type)) {
				gboolean _tmp4_;
				return (_tmp4_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp4_);
			} else {
				if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->uint_type)) {
					gboolean _tmp5_;
					return (_tmp5_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp5_);
				} else {
					if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->ulong_type)) {
						gboolean _tmp6_;
						return (_tmp6_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp6_);
					} else {
						if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->uint8_type)) {
							gboolean _tmp7_;
							return (_tmp7_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp7_);
						} else {
							if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->uint16_type)) {
								gboolean _tmp8_;
								return (_tmp8_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp8_);
							} else {
								if (VALA_TYPESYMBOL (st) == vala_data_type_get_data_type (self->uint32_type)) {
									gboolean _tmp9_;
									return (_tmp9_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp9_);
								} else {
									gboolean _tmp10_;
									return (_tmp10_ = FALSE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp10_);
								}
							}
						}
					}
				}
			}
		}
	}
	(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
}


void vala_ccode_base_module_check_type (ValaCCodeBaseModule* self, ValaDataType* type) {
	ValaArrayType* _tmp1_;
	ValaDataType* _tmp0_;
	ValaArrayType* array_type;
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	array_type = (_tmp1_ = (_tmp0_ = type, VALA_IS_ARRAY_TYPE (_tmp0_) ? ((ValaArrayType*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	if (array_type != NULL) {
		vala_ccode_base_module_check_type (self, vala_array_type_get_element_type (array_type));
	}
	{
		GeeList* _tmp2_;
		GeeIterator* _tmp3_;
		GeeIterator* _type_arg_it;
		_tmp2_ = NULL;
		_tmp3_ = NULL;
		_type_arg_it = (_tmp3_ = gee_iterable_iterator ((GeeIterable*) (_tmp2_ = vala_data_type_get_type_arguments (type))), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (gee_collection_object_unref (_tmp2_), NULL)), _tmp3_);
		while (gee_iterator_next (_type_arg_it)) {
			ValaDataType* type_arg;
			type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
			vala_ccode_base_module_check_type (self, type_arg);
			vala_ccode_base_module_check_type_argument (self, type_arg);
			(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
		}
		(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
	}
	(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
}


static void vala_ccode_base_module_check_type_argument (ValaCCodeBaseModule* self, ValaDataType* type_arg) {
	gboolean _tmp0_;
	gboolean _tmp1_;
	gboolean _tmp2_;
	gboolean _tmp3_;
	gboolean _tmp4_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (type_arg != NULL);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	_tmp2_ = FALSE;
	_tmp3_ = FALSE;
	_tmp4_ = FALSE;
	if (VALA_IS_GENERIC_TYPE (type_arg)) {
		_tmp4_ = TRUE;
	} else {
		_tmp4_ = VALA_IS_POINTER_TYPE (type_arg);
	}
	if (_tmp4_) {
		_tmp3_ = TRUE;
	} else {
		_tmp3_ = vala_ccode_base_module_is_reference_type_argument (self, type_arg);
	}
	if (_tmp3_) {
		_tmp2_ = TRUE;
	} else {
		_tmp2_ = vala_ccode_base_module_is_nullable_value_type_argument (self, type_arg);
	}
	if (_tmp2_) {
		_tmp1_ = TRUE;
	} else {
		_tmp1_ = vala_ccode_base_module_is_signed_integer_type_argument (self, type_arg);
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = vala_ccode_base_module_is_unsigned_integer_type_argument (self, type_arg);
	}
	if (_tmp0_) {
	} else {
		char* _tmp6_;
		char* _tmp5_;
		/* no error*/
		_tmp6_ = NULL;
		_tmp5_ = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) type_arg), _tmp6_ = g_strdup_printf ("`%s' is not a supported generic type argument, use `?' to box value types", _tmp5_ = vala_code_node_to_string ((ValaCodeNode*) type_arg)));
		_tmp6_ = (g_free (_tmp6_), NULL);
		_tmp5_ = (g_free (_tmp5_), NULL);
	}
}


static void vala_ccode_base_module_real_generate_class_declaration (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space) {
	char* _tmp0_;
	gboolean _tmp1_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	g_return_if_fail (decl_space != NULL);
	_tmp0_ = NULL;
	if ((_tmp1_ = vala_ccode_declaration_space_add_symbol_declaration (decl_space, (ValaSymbol*) cl, _tmp0_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) cl, FALSE)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_)) {
		return;
	}
}


void vala_ccode_base_module_generate_class_declaration (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space) {
	VALA_CCODE_BASE_MODULE_GET_CLASS (self)->generate_class_declaration (self, cl, decl_space);
}


static void vala_ccode_base_module_real_generate_interface_declaration (ValaCCodeBaseModule* self, ValaInterface* iface, ValaCCodeDeclarationSpace* decl_space) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (iface != NULL);
	g_return_if_fail (decl_space != NULL);
}


void vala_ccode_base_module_generate_interface_declaration (ValaCCodeBaseModule* self, ValaInterface* iface, ValaCCodeDeclarationSpace* decl_space) {
	VALA_CCODE_BASE_MODULE_GET_CLASS (self)->generate_interface_declaration (self, iface, decl_space);
}


static void vala_ccode_base_module_real_generate_method_declaration (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	g_return_if_fail (decl_space != NULL);
}


void vala_ccode_base_module_generate_method_declaration (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space) {
	VALA_CCODE_BASE_MODULE_GET_CLASS (self)->generate_method_declaration (self, m, decl_space);
}


static void vala_ccode_base_module_real_generate_error_domain_declaration (ValaCCodeBaseModule* self, ValaErrorDomain* edomain, ValaCCodeDeclarationSpace* decl_space) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (edomain != NULL);
	g_return_if_fail (decl_space != NULL);
}


void vala_ccode_base_module_generate_error_domain_declaration (ValaCCodeBaseModule* self, ValaErrorDomain* edomain, ValaCCodeDeclarationSpace* decl_space) {
	VALA_CCODE_BASE_MODULE_GET_CLASS (self)->generate_error_domain_declaration (self, edomain, decl_space);
}


static void vala_ccode_base_module_real_visit_object_creation_expression (ValaCCodeModule* base, ValaObjectCreationExpression* expr) {
	ValaCCodeBaseModule * self;
	ValaCCodeExpression* instance;
	ValaCCodeExpression* creation_expr;
	ValaStruct* _tmp1_;
	ValaTypeSymbol* _tmp0_;
	ValaStruct* st;
	gboolean _tmp2_;
	gboolean _tmp3_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	instance = NULL;
	creation_expr = NULL;
	vala_ccode_base_module_check_type (self, vala_object_creation_expression_get_type_reference (expr));
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	st = (_tmp1_ = (_tmp0_ = vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)), VALA_IS_STRUCT (_tmp0_) ? ((ValaStruct*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp2_ = FALSE;
	_tmp3_ = FALSE;
	if (st != NULL) {
		_tmp3_ = !vala_struct_is_simple_type (st);
	} else {
		_tmp3_ = FALSE;
	}
	if (_tmp3_) {
		_tmp2_ = TRUE;
	} else {
		GeeList* _tmp4_;
		_tmp4_ = NULL;
		_tmp2_ = gee_collection_get_size ((GeeCollection*) (_tmp4_ = vala_object_creation_expression_get_object_initializer (expr))) > 0;
		(_tmp4_ == NULL) ? NULL : (_tmp4_ = (gee_collection_object_unref (_tmp4_), NULL));
	}
	if (_tmp2_) {
		ValaLocalVariable* temp_decl;
		ValaCCodeExpression* _tmp6_;
		char* _tmp5_;
		/* value-type initialization or object creation expression with object initializer*/
		temp_decl = vala_ccode_base_module_get_temp_variable (self, vala_object_creation_expression_get_type_reference (expr), FALSE, (ValaCodeNode*) expr);
		gee_collection_add ((GeeCollection*) self->temp_vars, temp_decl);
		_tmp6_ = NULL;
		_tmp5_ = NULL;
		instance = (_tmp6_ = vala_ccode_base_module_get_variable_cexpression (self, _tmp5_ = vala_ccode_base_module_get_variable_cname (self, vala_symbol_get_name ((ValaSymbol*) temp_decl))), (instance == NULL) ? NULL : (instance = (vala_ccode_node_unref (instance), NULL)), _tmp6_);
		_tmp5_ = (g_free (_tmp5_), NULL);
		(temp_decl == NULL) ? NULL : (temp_decl = (vala_code_node_unref (temp_decl), NULL));
	}
	if (vala_expression_get_symbol_reference ((ValaExpression*) expr) == NULL) {
		ValaCCodeFunctionCall* creation_call;
		gboolean _tmp7_;
		ValaCCodeExpression* _tmp28_;
		ValaCCodeExpression* _tmp27_;
		creation_call = NULL;
		_tmp7_ = FALSE;
		if (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)) == VALA_TYPESYMBOL (self->glist_type)) {
			_tmp7_ = TRUE;
		} else {
			_tmp7_ = vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)) == VALA_TYPESYMBOL (self->gslist_type);
		}
		/* no creation method*/
		if (_tmp7_) {
			ValaCCodeConstant* _tmp8_;
			/* NULL is an empty list*/
			_tmp8_ = NULL;
			vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp8_ = vala_ccode_constant_new ("NULL")));
			(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL));
		} else {
			gboolean _tmp9_;
			_tmp9_ = FALSE;
			if (VALA_IS_CLASS (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)))) {
				_tmp9_ = vala_typesymbol_is_subtype_of (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)), self->gobject_type);
			} else {
				_tmp9_ = FALSE;
			}
			if (_tmp9_) {
				ValaCCodeFunctionCall* _tmp11_;
				ValaCCodeIdentifier* _tmp10_;
				ValaCCodeConstant* _tmp13_;
				char* _tmp12_;
				ValaCCodeConstant* _tmp14_;
				vala_ccode_base_module_generate_class_declaration (self, VALA_CLASS (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr))), self->source_declarations);
				_tmp11_ = NULL;
				_tmp10_ = NULL;
				creation_call = (_tmp11_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp10_ = vala_ccode_identifier_new ("g_object_new"))), (creation_call == NULL) ? NULL : (creation_call = (vala_ccode_node_unref (creation_call), NULL)), _tmp11_);
				(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
				_tmp13_ = NULL;
				_tmp12_ = NULL;
				vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp13_ = vala_ccode_constant_new (_tmp12_ = vala_typesymbol_get_type_id (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr))))));
				(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL));
				_tmp12_ = (g_free (_tmp12_), NULL);
				_tmp14_ = NULL;
				vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp14_ = vala_ccode_constant_new ("NULL")));
				(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
			} else {
				if (VALA_IS_CLASS (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)))) {
					ValaCCodeFunctionCall* _tmp16_;
					ValaCCodeIdentifier* _tmp15_;
					ValaCCodeConstant* _tmp18_;
					char* _tmp17_;
					ValaCCodeConstant* _tmp19_;
					_tmp16_ = NULL;
					_tmp15_ = NULL;
					creation_call = (_tmp16_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp15_ = vala_ccode_identifier_new ("g_new0"))), (creation_call == NULL) ? NULL : (creation_call = (vala_ccode_node_unref (creation_call), NULL)), _tmp16_);
					(_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL));
					_tmp18_ = NULL;
					_tmp17_ = NULL;
					vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp18_ = vala_ccode_constant_new (_tmp17_ = vala_typesymbol_get_cname (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)), FALSE))));
					(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
					_tmp17_ = (g_free (_tmp17_), NULL);
					_tmp19_ = NULL;
					vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp19_ = vala_ccode_constant_new ("1")));
					(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
				} else {
					if (VALA_IS_STRUCT (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)))) {
						ValaCCodeFunctionCall* _tmp21_;
						ValaCCodeIdentifier* _tmp20_;
						ValaCCodeUnaryExpression* _tmp22_;
						ValaCCodeConstant* _tmp23_;
						ValaCCodeIdentifier* _tmp26_;
						char* _tmp25_;
						char* _tmp24_;
						/* memset needs string.h*/
						vala_ccode_declaration_space_add_include (self->source_declarations, "string.h", FALSE);
						_tmp21_ = NULL;
						_tmp20_ = NULL;
						creation_call = (_tmp21_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp20_ = vala_ccode_identifier_new ("memset"))), (creation_call == NULL) ? NULL : (creation_call = (vala_ccode_node_unref (creation_call), NULL)), _tmp21_);
						(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL));
						_tmp22_ = NULL;
						vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp22_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, instance)));
						(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL));
						_tmp23_ = NULL;
						vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp23_ = vala_ccode_constant_new ("0")));
						(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_ccode_node_unref (_tmp23_), NULL));
						_tmp26_ = NULL;
						_tmp25_ = NULL;
						_tmp24_ = NULL;
						vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp26_ = vala_ccode_identifier_new (_tmp25_ = g_strdup_printf ("sizeof (%s)", _tmp24_ = vala_data_type_get_cname (vala_object_creation_expression_get_type_reference (expr))))));
						(_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_ccode_node_unref (_tmp26_), NULL));
						_tmp25_ = (g_free (_tmp25_), NULL);
						_tmp24_ = (g_free (_tmp24_), NULL);
					}
				}
			}
		}
		_tmp28_ = NULL;
		_tmp27_ = NULL;
		creation_expr = (_tmp28_ = (_tmp27_ = (ValaCCodeExpression*) creation_call, (_tmp27_ == NULL) ? NULL : vala_ccode_node_ref (_tmp27_)), (creation_expr == NULL) ? NULL : (creation_expr = (vala_ccode_node_unref (creation_expr), NULL)), _tmp28_);
		(creation_call == NULL) ? NULL : (creation_call = (vala_ccode_node_unref (creation_call), NULL));
	} else {
		if (VALA_IS_METHOD (vala_expression_get_symbol_reference ((ValaExpression*) expr))) {
			ValaMethod* _tmp29_;
			ValaMethod* m;
			GeeList* params;
			ValaCCodeFunctionCall* creation_call;
			ValaCCodeFunctionCall* _tmp32_;
			ValaCCodeIdentifier* _tmp31_;
			char* _tmp30_;
			gboolean _tmp33_;
			gboolean _tmp34_;
			ValaClass* _tmp37_;
			ValaTypeSymbol* _tmp36_;
			ValaClass* cl;
			gboolean _tmp38_;
			GeeHashMap* carg_map;
			gboolean ellipsis;
			gint i;
			gint arg_pos;
			GeeIterator* params_it;
			gint last_pos;
			gint min_pos;
			gboolean _tmp62_;
			gboolean _tmp63_;
			ValaCCodeExpression* _tmp70_;
			ValaCCodeExpression* _tmp69_;
			char* _tmp71_;
			gboolean _tmp72_;
			/* use creation method*/
			_tmp29_ = NULL;
			m = (_tmp29_ = VALA_METHOD (vala_expression_get_symbol_reference ((ValaExpression*) expr)), (_tmp29_ == NULL) ? NULL : vala_code_node_ref (_tmp29_));
			params = vala_method_get_parameters (m);
			creation_call = NULL;
			vala_ccode_base_module_generate_method_declaration (self, m, self->source_declarations);
			_tmp32_ = NULL;
			_tmp31_ = NULL;
			_tmp30_ = NULL;
			creation_call = (_tmp32_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp31_ = vala_ccode_identifier_new (_tmp30_ = vala_method_get_cname (m)))), (creation_call == NULL) ? NULL : (creation_call = (vala_ccode_node_unref (creation_call), NULL)), _tmp32_);
			(_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_ccode_node_unref (_tmp31_), NULL));
			_tmp30_ = (g_free (_tmp30_), NULL);
			_tmp33_ = FALSE;
			_tmp34_ = FALSE;
			if (st != NULL) {
				_tmp34_ = !vala_struct_is_simple_type (st);
			} else {
				_tmp34_ = FALSE;
			}
			if (_tmp34_) {
				_tmp33_ = !(vala_method_get_cinstance_parameter_position (m) < 0);
			} else {
				_tmp33_ = FALSE;
			}
			if (_tmp33_) {
				ValaCCodeUnaryExpression* _tmp35_;
				_tmp35_ = NULL;
				vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp35_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, instance)));
				(_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_ccode_node_unref (_tmp35_), NULL));
			}
			vala_ccode_base_module_generate_type_declaration (self, vala_object_creation_expression_get_type_reference (expr), self->source_declarations);
			_tmp37_ = NULL;
			_tmp36_ = NULL;
			cl = (_tmp37_ = (_tmp36_ = vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)), VALA_IS_CLASS (_tmp36_) ? ((ValaClass*) _tmp36_) : NULL), (_tmp37_ == NULL) ? NULL : vala_code_node_ref (_tmp37_));
			_tmp38_ = FALSE;
			if (cl != NULL) {
				_tmp38_ = !vala_class_get_is_compact (cl);
			} else {
				_tmp38_ = FALSE;
			}
			if (_tmp38_) {
				{
					GeeList* _tmp39_;
					GeeIterator* _tmp40_;
					GeeIterator* _type_arg_it;
					_tmp39_ = NULL;
					_tmp40_ = NULL;
					_type_arg_it = (_tmp40_ = gee_iterable_iterator ((GeeIterable*) (_tmp39_ = vala_data_type_get_type_arguments (vala_object_creation_expression_get_type_reference (expr)))), (_tmp39_ == NULL) ? NULL : (_tmp39_ = (gee_collection_object_unref (_tmp39_), NULL)), _tmp40_);
					while (gee_iterator_next (_type_arg_it)) {
						ValaDataType* type_arg;
						ValaCCodeExpression* _tmp41_;
						type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
						_tmp41_ = NULL;
						vala_ccode_function_call_add_argument (creation_call, _tmp41_ = vala_ccode_base_module_get_type_id_expression (self, type_arg));
						(_tmp41_ == NULL) ? NULL : (_tmp41_ = (vala_ccode_node_unref (_tmp41_), NULL));
						if (vala_ccode_base_module_requires_copy (self, type_arg)) {
							ValaCCodeExpression* dup_func;
							ValaCCodeCastExpression* _tmp42_;
							ValaCCodeExpression* _tmp43_;
							dup_func = vala_ccode_base_module_get_dup_func_expression (self, type_arg, vala_code_node_get_source_reference ((ValaCodeNode*) type_arg));
							if (dup_func == NULL) {
								/* type doesn't contain a copy function*/
								vala_code_node_set_error ((ValaCodeNode*) expr, TRUE);
								(dup_func == NULL) ? NULL : (dup_func = (vala_ccode_node_unref (dup_func), NULL));
								(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
								(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
								(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
								(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
								(creation_call == NULL) ? NULL : (creation_call = (vala_ccode_node_unref (creation_call), NULL));
								(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
								(instance == NULL) ? NULL : (instance = (vala_ccode_node_unref (instance), NULL));
								(creation_expr == NULL) ? NULL : (creation_expr = (vala_ccode_node_unref (creation_expr), NULL));
								(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
								return;
							}
							_tmp42_ = NULL;
							vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp42_ = vala_ccode_cast_expression_new (dup_func, "GBoxedCopyFunc")));
							(_tmp42_ == NULL) ? NULL : (_tmp42_ = (vala_ccode_node_unref (_tmp42_), NULL));
							_tmp43_ = NULL;
							vala_ccode_function_call_add_argument (creation_call, _tmp43_ = vala_ccode_base_module_get_destroy_func_expression (self, type_arg));
							(_tmp43_ == NULL) ? NULL : (_tmp43_ = (vala_ccode_node_unref (_tmp43_), NULL));
							(dup_func == NULL) ? NULL : (dup_func = (vala_ccode_node_unref (dup_func), NULL));
						} else {
							ValaCCodeConstant* _tmp44_;
							ValaCCodeConstant* _tmp45_;
							_tmp44_ = NULL;
							vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp44_ = vala_ccode_constant_new ("NULL")));
							(_tmp44_ == NULL) ? NULL : (_tmp44_ = (vala_ccode_node_unref (_tmp44_), NULL));
							_tmp45_ = NULL;
							vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp45_ = vala_ccode_constant_new ("NULL")));
							(_tmp45_ == NULL) ? NULL : (_tmp45_ = (vala_ccode_node_unref (_tmp45_), NULL));
						}
						(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
					}
					(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
				}
			}
			carg_map = gee_hash_map_new (G_TYPE_INT, NULL, NULL, VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_hash, g_direct_equal, g_direct_equal);
			ellipsis = FALSE;
			i = 1;
			arg_pos = 0;
			params_it = gee_iterable_iterator ((GeeIterable*) params);
			{
				GeeList* _tmp46_;
				GeeIterator* _tmp47_;
				GeeIterator* _arg_it;
				_tmp46_ = NULL;
				_tmp47_ = NULL;
				_arg_it = (_tmp47_ = gee_iterable_iterator ((GeeIterable*) (_tmp46_ = vala_object_creation_expression_get_argument_list (expr))), (_tmp46_ == NULL) ? NULL : (_tmp46_ = (gee_collection_object_unref (_tmp46_), NULL)), _tmp47_);
				while (gee_iterator_next (_arg_it)) {
					ValaExpression* arg;
					ValaCCodeExpression* _tmp48_;
					ValaCCodeExpression* cexpr;
					ValaFormalParameter* param;
					arg = (ValaExpression*) gee_iterator_get (_arg_it);
					_tmp48_ = NULL;
					cexpr = (_tmp48_ = VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) arg)), (_tmp48_ == NULL) ? NULL : vala_ccode_node_ref (_tmp48_));
					param = NULL;
					if (gee_iterator_next (params_it)) {
						ValaFormalParameter* _tmp49_;
						_tmp49_ = NULL;
						param = (_tmp49_ = (ValaFormalParameter*) gee_iterator_get (params_it), (param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL)), _tmp49_);
						ellipsis = vala_formal_parameter_get_ellipsis (param);
						if (!ellipsis) {
							gboolean _tmp50_;
							ValaCCodeExpression* _tmp55_;
							_tmp50_ = FALSE;
							if (!vala_formal_parameter_get_no_array_length (param)) {
								_tmp50_ = VALA_IS_ARRAY_TYPE (vala_formal_parameter_get_parameter_type (param));
							} else {
								_tmp50_ = FALSE;
							}
							if (_tmp50_) {
								ValaArrayType* _tmp51_;
								ValaArrayType* array_type;
								_tmp51_ = NULL;
								array_type = (_tmp51_ = VALA_ARRAY_TYPE (vala_formal_parameter_get_parameter_type (param)), (_tmp51_ == NULL) ? NULL : vala_code_node_ref (_tmp51_));
								{
									gint dim;
									dim = 1;
									for (; dim <= vala_array_type_get_rank (array_type); dim++) {
										ValaCCodeExpression* _tmp52_;
										_tmp52_ = NULL;
										gee_map_set ((GeeMap*) carg_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos (self, vala_formal_parameter_get_carray_length_parameter_position (param) + (0.01 * dim), FALSE)), _tmp52_ = vala_ccode_module_get_array_length_cexpression (vala_ccode_module_get_head ((ValaCCodeModule*) self), arg, dim));
										(_tmp52_ == NULL) ? NULL : (_tmp52_ = (vala_ccode_node_unref (_tmp52_), NULL));
									}
								}
								(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
							} else {
								if (VALA_IS_DELEGATE_TYPE (vala_formal_parameter_get_parameter_type (param))) {
									ValaDelegateType* _tmp53_;
									ValaDelegateType* deleg_type;
									ValaDelegate* _tmp54_;
									ValaDelegate* d;
									_tmp53_ = NULL;
									deleg_type = (_tmp53_ = VALA_DELEGATE_TYPE (vala_formal_parameter_get_parameter_type (param)), (_tmp53_ == NULL) ? NULL : vala_code_node_ref (_tmp53_));
									_tmp54_ = NULL;
									d = (_tmp54_ = vala_delegate_type_get_delegate_symbol (deleg_type), (_tmp54_ == NULL) ? NULL : vala_code_node_ref (_tmp54_));
									if (vala_delegate_get_has_target (d)) {
										ValaCCodeExpression* delegate_target;
										delegate_target = vala_ccode_base_module_get_delegate_target_cexpression (self, arg);
										gee_map_set ((GeeMap*) carg_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos (self, vala_formal_parameter_get_cdelegate_target_parameter_position (param), FALSE)), delegate_target);
										(delegate_target == NULL) ? NULL : (delegate_target = (vala_ccode_node_unref (delegate_target), NULL));
									}
									(deleg_type == NULL) ? NULL : (deleg_type = (vala_code_node_unref (deleg_type), NULL));
									(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
								}
							}
							_tmp55_ = NULL;
							cexpr = (_tmp55_ = vala_ccode_base_module_handle_struct_argument (self, param, arg, cexpr), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), _tmp55_);
						}
						arg_pos = vala_ccode_base_module_get_param_pos (self, vala_formal_parameter_get_cparameter_position (param), ellipsis);
					} else {
						/* default argument position*/
						arg_pos = vala_ccode_base_module_get_param_pos (self, (double) i, ellipsis);
					}
					gee_map_set ((GeeMap*) carg_map, GINT_TO_POINTER (arg_pos), cexpr);
					i++;
					(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
					(cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL));
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				}
				(_arg_it == NULL) ? NULL : (_arg_it = (gee_collection_object_unref (_arg_it), NULL));
			}
			while (gee_iterator_next (params_it)) {
				ValaFormalParameter* param;
				param = (ValaFormalParameter*) gee_iterator_get (params_it);
				if (vala_formal_parameter_get_ellipsis (param)) {
					ellipsis = TRUE;
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
					break;
				}
				if (vala_formal_parameter_get_default_expression (param) == NULL) {
					char* _tmp56_;
					_tmp56_ = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) expr), _tmp56_ = g_strdup_printf ("no default expression for argument %d", i));
					_tmp56_ = (g_free (_tmp56_), NULL);
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
					(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
					(creation_call == NULL) ? NULL : (creation_call = (vala_ccode_node_unref (creation_call), NULL));
					(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
					(carg_map == NULL) ? NULL : (carg_map = (gee_collection_object_unref (carg_map), NULL));
					(params_it == NULL) ? NULL : (params_it = (gee_collection_object_unref (params_it), NULL));
					(instance == NULL) ? NULL : (instance = (vala_ccode_node_unref (instance), NULL));
					(creation_expr == NULL) ? NULL : (creation_expr = (vala_ccode_node_unref (creation_expr), NULL));
					(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
					return;
				}
				/* evaluate default expression here as the code
				 * generator might not have visited the formal
				 * parameter yet */
				vala_code_node_accept ((ValaCodeNode*) vala_formal_parameter_get_default_expression (param), (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
				gee_map_set ((GeeMap*) carg_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos (self, vala_formal_parameter_get_cparameter_position (param), FALSE)), VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_formal_parameter_get_default_expression (param))));
				i++;
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
			}
			/* append C arguments in the right order*/
			last_pos = -1;
			min_pos = 0;
			while (TRUE) {
				ValaCCodeExpression* _tmp61_;
				min_pos = -1;
				{
					GeeSet* _tmp57_;
					GeeIterator* _tmp58_;
					GeeIterator* _pos_it;
					_tmp57_ = NULL;
					_tmp58_ = NULL;
					_pos_it = (_tmp58_ = gee_iterable_iterator ((GeeIterable*) (_tmp57_ = gee_map_get_keys ((GeeMap*) carg_map))), (_tmp57_ == NULL) ? NULL : (_tmp57_ = (gee_collection_object_unref (_tmp57_), NULL)), _tmp58_);
					while (gee_iterator_next (_pos_it)) {
						gint pos;
						gboolean _tmp59_;
						pos = GPOINTER_TO_INT (gee_iterator_get (_pos_it));
						_tmp59_ = FALSE;
						if (pos > last_pos) {
							gboolean _tmp60_;
							_tmp60_ = FALSE;
							if (min_pos == (-1)) {
								_tmp60_ = TRUE;
							} else {
								_tmp60_ = pos < min_pos;
							}
							_tmp59_ = _tmp60_;
						} else {
							_tmp59_ = FALSE;
						}
						if (_tmp59_) {
							min_pos = pos;
						}
					}
					(_pos_it == NULL) ? NULL : (_pos_it = (gee_collection_object_unref (_pos_it), NULL));
				}
				if (min_pos == (-1)) {
					break;
				}
				_tmp61_ = NULL;
				vala_ccode_function_call_add_argument (creation_call, _tmp61_ = (ValaCCodeExpression*) gee_map_get ((GeeMap*) carg_map, GINT_TO_POINTER (min_pos)));
				(_tmp61_ == NULL) ? NULL : (_tmp61_ = (vala_ccode_node_unref (_tmp61_), NULL));
				last_pos = min_pos;
			}
			_tmp62_ = FALSE;
			_tmp63_ = FALSE;
			if (st != NULL) {
				_tmp63_ = !vala_struct_is_simple_type (st);
			} else {
				_tmp63_ = FALSE;
			}
			if (_tmp63_) {
				_tmp62_ = vala_method_get_cinstance_parameter_position (m) < 0;
			} else {
				_tmp62_ = FALSE;
			}
			if (_tmp62_) {
				ValaCCodeUnaryExpression* _tmp64_;
				/* instance parameter is at the end in a struct creation method*/
				_tmp64_ = NULL;
				vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp64_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, instance)));
				(_tmp64_ == NULL) ? NULL : (_tmp64_ = (vala_ccode_node_unref (_tmp64_), NULL));
			}
			if (vala_code_node_get_tree_can_fail ((ValaCodeNode*) expr)) {
				ValaCCodeUnaryExpression* _tmp66_;
				ValaCCodeIdentifier* _tmp65_;
				/* method can fail*/
				self->current_method_inner_error = TRUE;
				_tmp66_ = NULL;
				_tmp65_ = NULL;
				vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp66_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp65_ = vala_ccode_identifier_new ("_inner_error_")))));
				(_tmp66_ == NULL) ? NULL : (_tmp66_ = (vala_ccode_node_unref (_tmp66_), NULL));
				(_tmp65_ == NULL) ? NULL : (_tmp65_ = (vala_ccode_node_unref (_tmp65_), NULL));
			}
			if (ellipsis) {
				gboolean _tmp67_;
				_tmp67_ = FALSE;
				if (!vala_method_get_printf_format (m)) {
					_tmp67_ = _vala_strcmp0 (vala_method_get_sentinel (m), "") != 0;
				} else {
					_tmp67_ = FALSE;
				}
				/* ensure variable argument list ends with NULL
				 * except when using printf-style arguments */
				if (_tmp67_) {
					ValaCCodeConstant* _tmp68_;
					_tmp68_ = NULL;
					vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp68_ = vala_ccode_constant_new (vala_method_get_sentinel (m))));
					(_tmp68_ == NULL) ? NULL : (_tmp68_ = (vala_ccode_node_unref (_tmp68_), NULL));
				}
			}
			_tmp70_ = NULL;
			_tmp69_ = NULL;
			creation_expr = (_tmp70_ = (_tmp69_ = (ValaCCodeExpression*) creation_call, (_tmp69_ == NULL) ? NULL : vala_ccode_node_ref (_tmp69_)), (creation_expr == NULL) ? NULL : (creation_expr = (vala_ccode_node_unref (creation_expr), NULL)), _tmp70_);
			/* cast the return value of the creation method back to the intended type if
			 it requested a special C return type*/
			_tmp71_ = NULL;
			if ((_tmp72_ = (_tmp71_ = vala_ccode_module_get_custom_creturn_type (vala_ccode_module_get_head ((ValaCCodeModule*) self), m)) != NULL, _tmp71_ = (g_free (_tmp71_), NULL), _tmp72_)) {
				ValaCCodeExpression* _tmp74_;
				char* _tmp73_;
				_tmp74_ = NULL;
				_tmp73_ = NULL;
				creation_expr = (_tmp74_ = (ValaCCodeExpression*) vala_ccode_cast_expression_new (creation_expr, _tmp73_ = vala_data_type_get_cname (vala_object_creation_expression_get_type_reference (expr))), (creation_expr == NULL) ? NULL : (creation_expr = (vala_ccode_node_unref (creation_expr), NULL)), _tmp74_);
				_tmp73_ = (g_free (_tmp73_), NULL);
			}
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			(params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL));
			(creation_call == NULL) ? NULL : (creation_call = (vala_ccode_node_unref (creation_call), NULL));
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
			(carg_map == NULL) ? NULL : (carg_map = (gee_collection_object_unref (carg_map), NULL));
			(params_it == NULL) ? NULL : (params_it = (gee_collection_object_unref (params_it), NULL));
		} else {
			if (VALA_IS_ERROR_CODE (vala_expression_get_symbol_reference ((ValaExpression*) expr))) {
				ValaErrorCode* _tmp75_;
				ValaErrorCode* ecode;
				ValaErrorDomain* _tmp76_;
				ValaErrorDomain* edomain;
				ValaCCodeFunctionCall* creation_call;
				GeeList* _tmp77_;
				gboolean _tmp78_;
				ValaCCodeIdentifier* _tmp84_;
				char* _tmp83_;
				ValaCCodeIdentifier* _tmp86_;
				char* _tmp85_;
				ValaCCodeExpression* _tmp90_;
				ValaCCodeExpression* _tmp89_;
				_tmp75_ = NULL;
				ecode = (_tmp75_ = VALA_ERROR_CODE (vala_expression_get_symbol_reference ((ValaExpression*) expr)), (_tmp75_ == NULL) ? NULL : vala_code_node_ref (_tmp75_));
				_tmp76_ = NULL;
				edomain = (_tmp76_ = VALA_ERROR_DOMAIN (vala_symbol_get_parent_symbol ((ValaSymbol*) ecode)), (_tmp76_ == NULL) ? NULL : vala_code_node_ref (_tmp76_));
				creation_call = NULL;
				vala_ccode_base_module_generate_error_domain_declaration (self, edomain, self->source_declarations);
				_tmp77_ = NULL;
				if ((_tmp78_ = gee_collection_get_size ((GeeCollection*) (_tmp77_ = vala_object_creation_expression_get_argument_list (expr))) == 1, (_tmp77_ == NULL) ? NULL : (_tmp77_ = (gee_collection_object_unref (_tmp77_), NULL)), _tmp78_)) {
					ValaCCodeFunctionCall* _tmp80_;
					ValaCCodeIdentifier* _tmp79_;
					/* must not be a format argument*/
					_tmp80_ = NULL;
					_tmp79_ = NULL;
					creation_call = (_tmp80_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp79_ = vala_ccode_identifier_new ("g_error_new_literal"))), (creation_call == NULL) ? NULL : (creation_call = (vala_ccode_node_unref (creation_call), NULL)), _tmp80_);
					(_tmp79_ == NULL) ? NULL : (_tmp79_ = (vala_ccode_node_unref (_tmp79_), NULL));
				} else {
					ValaCCodeFunctionCall* _tmp82_;
					ValaCCodeIdentifier* _tmp81_;
					_tmp82_ = NULL;
					_tmp81_ = NULL;
					creation_call = (_tmp82_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp81_ = vala_ccode_identifier_new ("g_error_new"))), (creation_call == NULL) ? NULL : (creation_call = (vala_ccode_node_unref (creation_call), NULL)), _tmp82_);
					(_tmp81_ == NULL) ? NULL : (_tmp81_ = (vala_ccode_node_unref (_tmp81_), NULL));
				}
				_tmp84_ = NULL;
				_tmp83_ = NULL;
				vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp84_ = vala_ccode_identifier_new (_tmp83_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) edomain, NULL))));
				(_tmp84_ == NULL) ? NULL : (_tmp84_ = (vala_ccode_node_unref (_tmp84_), NULL));
				_tmp83_ = (g_free (_tmp83_), NULL);
				_tmp86_ = NULL;
				_tmp85_ = NULL;
				vala_ccode_function_call_add_argument (creation_call, (ValaCCodeExpression*) (_tmp86_ = vala_ccode_identifier_new (_tmp85_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) ecode, FALSE))));
				(_tmp86_ == NULL) ? NULL : (_tmp86_ = (vala_ccode_node_unref (_tmp86_), NULL));
				_tmp85_ = (g_free (_tmp85_), NULL);
				{
					GeeList* _tmp87_;
					GeeIterator* _tmp88_;
					GeeIterator* _arg_it;
					_tmp87_ = NULL;
					_tmp88_ = NULL;
					_arg_it = (_tmp88_ = gee_iterable_iterator ((GeeIterable*) (_tmp87_ = vala_object_creation_expression_get_argument_list (expr))), (_tmp87_ == NULL) ? NULL : (_tmp87_ = (gee_collection_object_unref (_tmp87_), NULL)), _tmp88_);
					while (gee_iterator_next (_arg_it)) {
						ValaExpression* arg;
						arg = (ValaExpression*) gee_iterator_get (_arg_it);
						vala_ccode_function_call_add_argument (creation_call, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) arg)));
						(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
					}
					(_arg_it == NULL) ? NULL : (_arg_it = (gee_collection_object_unref (_arg_it), NULL));
				}
				_tmp90_ = NULL;
				_tmp89_ = NULL;
				creation_expr = (_tmp90_ = (_tmp89_ = (ValaCCodeExpression*) creation_call, (_tmp89_ == NULL) ? NULL : vala_ccode_node_ref (_tmp89_)), (creation_expr == NULL) ? NULL : (creation_expr = (vala_ccode_node_unref (creation_expr), NULL)), _tmp90_);
				(ecode == NULL) ? NULL : (ecode = (vala_code_node_unref (ecode), NULL));
				(edomain == NULL) ? NULL : (edomain = (vala_code_node_unref (edomain), NULL));
				(creation_call == NULL) ? NULL : (creation_call = (vala_ccode_node_unref (creation_call), NULL));
			} else {
				g_assert (FALSE);
			}
		}
	}
	if (instance != NULL) {
		ValaCCodeCommaExpression* ccomma;
		ccomma = vala_ccode_comma_expression_new ();
		if (VALA_IS_STRUCT (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)))) {
			vala_ccode_comma_expression_append_expression (ccomma, creation_expr);
		} else {
			ValaCCodeAssignment* _tmp91_;
			_tmp91_ = NULL;
			vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp91_ = vala_ccode_assignment_new (instance, creation_expr, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
			(_tmp91_ == NULL) ? NULL : (_tmp91_ = (vala_ccode_node_unref (_tmp91_), NULL));
		}
		{
			GeeList* _tmp92_;
			GeeIterator* _tmp93_;
			GeeIterator* _init_it;
			_tmp92_ = NULL;
			_tmp93_ = NULL;
			_init_it = (_tmp93_ = gee_iterable_iterator ((GeeIterable*) (_tmp92_ = vala_object_creation_expression_get_object_initializer (expr))), (_tmp92_ == NULL) ? NULL : (_tmp92_ = (gee_collection_object_unref (_tmp92_), NULL)), _tmp93_);
			while (gee_iterator_next (_init_it)) {
				ValaMemberInitializer* init;
				init = (ValaMemberInitializer*) gee_iterator_get (_init_it);
				if (VALA_IS_FIELD (vala_member_initializer_get_symbol_reference (init))) {
					ValaField* _tmp94_;
					ValaField* f;
					ValaDataType* instance_target_type;
					ValaCCodeExpression* typed_inst;
					ValaCCodeExpression* lhs;
					ValaCCodeAssignment* _tmp99_;
					gboolean _tmp100_;
					_tmp94_ = NULL;
					f = (_tmp94_ = VALA_FIELD (vala_member_initializer_get_symbol_reference (init)), (_tmp94_ == NULL) ? NULL : vala_code_node_ref (_tmp94_));
					instance_target_type = vala_ccode_base_module_get_data_type_for_symbol (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) f)));
					typed_inst = vala_ccode_base_module_transform_expression (self, instance, vala_object_creation_expression_get_type_reference (expr), instance_target_type, NULL);
					lhs = NULL;
					if (VALA_IS_STRUCT (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)))) {
						ValaCCodeExpression* _tmp96_;
						char* _tmp95_;
						_tmp96_ = NULL;
						_tmp95_ = NULL;
						lhs = (_tmp96_ = (ValaCCodeExpression*) vala_ccode_member_access_new (typed_inst, _tmp95_ = vala_field_get_cname (f), FALSE), (lhs == NULL) ? NULL : (lhs = (vala_ccode_node_unref (lhs), NULL)), _tmp96_);
						_tmp95_ = (g_free (_tmp95_), NULL);
					} else {
						ValaCCodeExpression* _tmp98_;
						char* _tmp97_;
						_tmp98_ = NULL;
						_tmp97_ = NULL;
						lhs = (_tmp98_ = (ValaCCodeExpression*) vala_ccode_member_access_new_pointer (typed_inst, _tmp97_ = vala_field_get_cname (f)), (lhs == NULL) ? NULL : (lhs = (vala_ccode_node_unref (lhs), NULL)), _tmp98_);
						_tmp97_ = (g_free (_tmp97_), NULL);
					}
					_tmp99_ = NULL;
					vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp99_ = vala_ccode_assignment_new (lhs, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_member_initializer_get_initializer (init))), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
					(_tmp99_ == NULL) ? NULL : (_tmp99_ = (vala_ccode_node_unref (_tmp99_), NULL));
					_tmp100_ = FALSE;
					if (VALA_IS_ARRAY_TYPE (vala_field_get_field_type (f))) {
						_tmp100_ = !vala_field_get_no_array_length (f);
					} else {
						_tmp100_ = FALSE;
					}
					if (_tmp100_) {
						ValaArrayType* _tmp101_;
						ValaArrayType* array_type;
						_tmp101_ = NULL;
						array_type = (_tmp101_ = VALA_ARRAY_TYPE (vala_field_get_field_type (f)), (_tmp101_ == NULL) ? NULL : vala_code_node_ref (_tmp101_));
						{
							gint dim;
							dim = 1;
							for (; dim <= vala_array_type_get_rank (array_type); dim++) {
								ValaCCodeExpression* rhs_array_len;
								ValaCCodeAssignment* _tmp108_;
								if (VALA_IS_STRUCT (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)))) {
									ValaCCodeExpression* _tmp104_;
									char* _tmp103_;
									char* _tmp102_;
									_tmp104_ = NULL;
									_tmp103_ = NULL;
									_tmp102_ = NULL;
									lhs = (_tmp104_ = (ValaCCodeExpression*) vala_ccode_member_access_new (typed_inst, _tmp103_ = vala_ccode_module_get_array_length_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), _tmp102_ = vala_field_get_cname (f), dim), FALSE), (lhs == NULL) ? NULL : (lhs = (vala_ccode_node_unref (lhs), NULL)), _tmp104_);
									_tmp103_ = (g_free (_tmp103_), NULL);
									_tmp102_ = (g_free (_tmp102_), NULL);
								} else {
									ValaCCodeExpression* _tmp107_;
									char* _tmp106_;
									char* _tmp105_;
									_tmp107_ = NULL;
									_tmp106_ = NULL;
									_tmp105_ = NULL;
									lhs = (_tmp107_ = (ValaCCodeExpression*) vala_ccode_member_access_new_pointer (typed_inst, _tmp106_ = vala_ccode_module_get_array_length_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), _tmp105_ = vala_field_get_cname (f), dim)), (lhs == NULL) ? NULL : (lhs = (vala_ccode_node_unref (lhs), NULL)), _tmp107_);
									_tmp106_ = (g_free (_tmp106_), NULL);
									_tmp105_ = (g_free (_tmp105_), NULL);
								}
								rhs_array_len = vala_ccode_module_get_array_length_cexpression (vala_ccode_module_get_head ((ValaCCodeModule*) self), vala_member_initializer_get_initializer (init), dim);
								_tmp108_ = NULL;
								vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp108_ = vala_ccode_assignment_new (lhs, rhs_array_len, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
								(_tmp108_ == NULL) ? NULL : (_tmp108_ = (vala_ccode_node_unref (_tmp108_), NULL));
								(rhs_array_len == NULL) ? NULL : (rhs_array_len = (vala_ccode_node_unref (rhs_array_len), NULL));
							}
						}
						(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
					} else {
						gboolean _tmp109_;
						_tmp109_ = FALSE;
						if (VALA_IS_DELEGATE_TYPE (vala_field_get_field_type (f))) {
							_tmp109_ = !vala_field_get_no_delegate_target (f);
						} else {
							_tmp109_ = FALSE;
						}
						if (_tmp109_) {
							ValaCCodeExpression* rhs_delegate_target;
							ValaCCodeAssignment* _tmp116_;
							if (VALA_IS_STRUCT (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)))) {
								ValaCCodeExpression* _tmp112_;
								char* _tmp111_;
								char* _tmp110_;
								_tmp112_ = NULL;
								_tmp111_ = NULL;
								_tmp110_ = NULL;
								lhs = (_tmp112_ = (ValaCCodeExpression*) vala_ccode_member_access_new (typed_inst, _tmp111_ = vala_ccode_base_module_get_delegate_target_cname (self, _tmp110_ = vala_field_get_cname (f)), FALSE), (lhs == NULL) ? NULL : (lhs = (vala_ccode_node_unref (lhs), NULL)), _tmp112_);
								_tmp111_ = (g_free (_tmp111_), NULL);
								_tmp110_ = (g_free (_tmp110_), NULL);
							} else {
								ValaCCodeExpression* _tmp115_;
								char* _tmp114_;
								char* _tmp113_;
								_tmp115_ = NULL;
								_tmp114_ = NULL;
								_tmp113_ = NULL;
								lhs = (_tmp115_ = (ValaCCodeExpression*) vala_ccode_member_access_new_pointer (typed_inst, _tmp114_ = vala_ccode_base_module_get_delegate_target_cname (self, _tmp113_ = vala_field_get_cname (f))), (lhs == NULL) ? NULL : (lhs = (vala_ccode_node_unref (lhs), NULL)), _tmp115_);
								_tmp114_ = (g_free (_tmp114_), NULL);
								_tmp113_ = (g_free (_tmp113_), NULL);
							}
							rhs_delegate_target = vala_ccode_base_module_get_delegate_target_cexpression (self, vala_member_initializer_get_initializer (init));
							_tmp116_ = NULL;
							vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp116_ = vala_ccode_assignment_new (lhs, rhs_delegate_target, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
							(_tmp116_ == NULL) ? NULL : (_tmp116_ = (vala_ccode_node_unref (_tmp116_), NULL));
							(rhs_delegate_target == NULL) ? NULL : (rhs_delegate_target = (vala_ccode_node_unref (rhs_delegate_target), NULL));
						}
					}
					(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
					(instance_target_type == NULL) ? NULL : (instance_target_type = (vala_code_node_unref (instance_target_type), NULL));
					(typed_inst == NULL) ? NULL : (typed_inst = (vala_ccode_node_unref (typed_inst), NULL));
					(lhs == NULL) ? NULL : (lhs = (vala_ccode_node_unref (lhs), NULL));
				} else {
					if (VALA_IS_PROPERTY (vala_member_initializer_get_symbol_reference (init))) {
						ValaMemberAccess* inst_ma;
						ValaMemberAccess* ma;
						ValaCCodeFunctionCall* _tmp117_;
						inst_ma = vala_member_access_new_simple ("new", NULL);
						vala_expression_set_value_type ((ValaExpression*) inst_ma, vala_object_creation_expression_get_type_reference (expr));
						vala_code_node_set_ccodenode ((ValaCodeNode*) inst_ma, (ValaCCodeNode*) instance);
						ma = vala_member_access_new ((ValaExpression*) inst_ma, vala_member_initializer_get_name (init), NULL);
						_tmp117_ = NULL;
						vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp117_ = vala_ccode_base_module_get_property_set_call (self, VALA_PROPERTY (vala_member_initializer_get_symbol_reference (init)), ma, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_member_initializer_get_initializer (init))))));
						(_tmp117_ == NULL) ? NULL : (_tmp117_ = (vala_ccode_node_unref (_tmp117_), NULL));
						(inst_ma == NULL) ? NULL : (inst_ma = (vala_code_node_unref (inst_ma), NULL));
						(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
					}
				}
				(init == NULL) ? NULL : (init = (vala_code_node_unref (init), NULL));
			}
			(_init_it == NULL) ? NULL : (_init_it = (gee_collection_object_unref (_init_it), NULL));
		}
		vala_ccode_comma_expression_append_expression (ccomma, instance);
		vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) ccomma);
		(ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL));
	} else {
		if (creation_expr != NULL) {
			vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) creation_expr);
		}
	}
	(instance == NULL) ? NULL : (instance = (vala_ccode_node_unref (instance), NULL));
	(creation_expr == NULL) ? NULL : (creation_expr = (vala_ccode_node_unref (creation_expr), NULL));
	(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
}


ValaCCodeExpression* vala_ccode_base_module_handle_struct_argument (ValaCCodeBaseModule* self, ValaFormalParameter* param, ValaExpression* arg, ValaCCodeExpression* cexpr) {
	gboolean _tmp0_;
	gboolean _tmp1_;
	ValaCCodeExpression* _tmp15_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (param != NULL, NULL);
	g_return_val_if_fail (arg != NULL, NULL);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (!VALA_IS_NULL_TYPE (vala_expression_get_value_type (arg))) {
		_tmp1_ = VALA_IS_STRUCT (vala_data_type_get_data_type (vala_formal_parameter_get_parameter_type (param)));
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		_tmp0_ = !vala_struct_is_simple_type (VALA_STRUCT (vala_data_type_get_data_type (vala_formal_parameter_get_parameter_type (param))));
	} else {
		_tmp0_ = FALSE;
	}
	/* pass non-simple struct instances always by reference*/
	if (_tmp0_) {
		gboolean _tmp2_;
		_tmp2_ = FALSE;
		if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_IN) {
			_tmp2_ = !vala_data_type_get_nullable (vala_formal_parameter_get_parameter_type (param));
		} else {
			_tmp2_ = FALSE;
		}
		/* we already use a reference for arguments of ref, out, and nullable parameters*/
		if (_tmp2_) {
			ValaCCodeUnaryExpression* _tmp4_;
			ValaCCodeExpression* _tmp3_;
			ValaCCodeUnaryExpression* unary;
			gboolean _tmp5_;
			_tmp4_ = NULL;
			_tmp3_ = NULL;
			unary = (_tmp4_ = (_tmp3_ = cexpr, VALA_IS_CCODE_UNARY_EXPRESSION (_tmp3_) ? ((ValaCCodeUnaryExpression*) _tmp3_) : NULL), (_tmp4_ == NULL) ? NULL : vala_ccode_node_ref (_tmp4_));
			_tmp5_ = FALSE;
			if (unary != NULL) {
				_tmp5_ = vala_ccode_unary_expression_get_operator (unary) == VALA_CCODE_UNARY_OPERATOR_POINTER_INDIRECTION;
			} else {
				_tmp5_ = FALSE;
			}
			if (_tmp5_) {
				ValaCCodeExpression* _tmp6_;
				ValaCCodeExpression* _tmp7_;
				/* *expr => expr*/
				_tmp6_ = NULL;
				_tmp7_ = NULL;
				return (_tmp7_ = (_tmp6_ = vala_ccode_unary_expression_get_inner (unary), (_tmp6_ == NULL) ? NULL : vala_ccode_node_ref (_tmp6_)), (unary == NULL) ? NULL : (unary = (vala_ccode_node_unref (unary), NULL)), _tmp7_);
			} else {
				gboolean _tmp8_;
				_tmp8_ = FALSE;
				if (VALA_IS_CCODE_IDENTIFIER (cexpr)) {
					_tmp8_ = TRUE;
				} else {
					_tmp8_ = VALA_IS_CCODE_MEMBER_ACCESS (cexpr);
				}
				if (_tmp8_) {
					ValaCCodeExpression* _tmp9_;
					_tmp9_ = NULL;
					return (_tmp9_ = (ValaCCodeExpression*) vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, cexpr), (unary == NULL) ? NULL : (unary = (vala_ccode_node_unref (unary), NULL)), _tmp9_);
				} else {
					ValaCCodeCommaExpression* ccomma;
					ValaLocalVariable* temp_var;
					ValaCCodeAssignment* _tmp11_;
					ValaCCodeExpression* _tmp10_;
					ValaCCodeUnaryExpression* _tmp13_;
					ValaCCodeIdentifier* _tmp12_;
					ValaCCodeExpression* _tmp14_;
					/* if cexpr is e.g. a function call, we can't take the address of the expression
					 (tmp = expr, &tmp)*/
					ccomma = vala_ccode_comma_expression_new ();
					temp_var = vala_ccode_base_module_get_temp_variable (self, vala_expression_get_value_type (arg), TRUE, NULL);
					gee_list_insert ((GeeList*) self->temp_vars, 0, temp_var);
					_tmp11_ = NULL;
					_tmp10_ = NULL;
					vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp11_ = vala_ccode_assignment_new (_tmp10_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) temp_var)), cexpr, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
					(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
					(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
					_tmp13_ = NULL;
					_tmp12_ = NULL;
					vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp13_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp12_ = vala_ccode_identifier_new (vala_symbol_get_name ((ValaSymbol*) temp_var))))));
					(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL));
					(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
					_tmp14_ = NULL;
					return (_tmp14_ = (ValaCCodeExpression*) ccomma, (temp_var == NULL) ? NULL : (temp_var = (vala_code_node_unref (temp_var), NULL)), (unary == NULL) ? NULL : (unary = (vala_ccode_node_unref (unary), NULL)), _tmp14_);
				}
			}
			(unary == NULL) ? NULL : (unary = (vala_ccode_node_unref (unary), NULL));
		}
	}
	_tmp15_ = NULL;
	return (_tmp15_ = cexpr, (_tmp15_ == NULL) ? NULL : vala_ccode_node_ref (_tmp15_));
}


static void vala_ccode_base_module_real_visit_sizeof_expression (ValaCCodeModule* base, ValaSizeofExpression* expr) {
	ValaCCodeBaseModule * self;
	ValaCCodeIdentifier* _tmp0_;
	ValaCCodeFunctionCall* _tmp1_;
	ValaCCodeFunctionCall* csizeof;
	ValaCCodeIdentifier* _tmp3_;
	char* _tmp2_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	csizeof = (_tmp1_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp0_ = vala_ccode_identifier_new ("sizeof"))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL)), _tmp1_);
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	vala_ccode_function_call_add_argument (csizeof, (ValaCCodeExpression*) (_tmp3_ = vala_ccode_identifier_new (_tmp2_ = vala_data_type_get_cname (vala_sizeof_expression_get_type_reference (expr)))));
	(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
	_tmp2_ = (g_free (_tmp2_), NULL);
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) csizeof);
	(csizeof == NULL) ? NULL : (csizeof = (vala_ccode_node_unref (csizeof), NULL));
}


static void vala_ccode_base_module_real_visit_typeof_expression (ValaCCodeModule* base, ValaTypeofExpression* expr) {
	ValaCCodeBaseModule * self;
	ValaCCodeExpression* _tmp0_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	_tmp0_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp0_ = vala_ccode_base_module_get_type_id_expression (self, vala_typeof_expression_get_type_reference (expr))));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
}


static void vala_ccode_base_module_real_visit_unary_expression (ValaCCodeModule* base, ValaUnaryExpression* expr) {
	ValaCCodeBaseModule * self;
	ValaCCodeUnaryOperator op;
	ValaCCodeUnaryExpression* _tmp0_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	op = 0;
	if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_PLUS) {
		op = VALA_CCODE_UNARY_OPERATOR_PLUS;
	} else {
		if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_MINUS) {
			op = VALA_CCODE_UNARY_OPERATOR_MINUS;
		} else {
			if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_LOGICAL_NEGATION) {
				op = VALA_CCODE_UNARY_OPERATOR_LOGICAL_NEGATION;
			} else {
				if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT) {
					op = VALA_CCODE_UNARY_OPERATOR_BITWISE_COMPLEMENT;
				} else {
					if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_INCREMENT) {
						op = VALA_CCODE_UNARY_OPERATOR_PREFIX_INCREMENT;
					} else {
						if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_DECREMENT) {
							op = VALA_CCODE_UNARY_OPERATOR_PREFIX_DECREMENT;
						} else {
							if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_REF) {
								op = VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF;
							} else {
								if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_OUT) {
									op = VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF;
								} else {
									g_assert_not_reached ();
								}
							}
						}
					}
				}
			}
		}
	}
	_tmp0_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp0_ = vala_ccode_unary_expression_new (op, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_unary_expression_get_inner (expr))))));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
}


static void vala_ccode_base_module_real_visit_cast_expression (ValaCCodeModule* base, ValaCastExpression* expr) {
	ValaCCodeBaseModule * self;
	ValaClass* _tmp1_;
	ValaTypeSymbol* _tmp0_;
	ValaClass* cl;
	ValaInterface* _tmp3_;
	ValaTypeSymbol* _tmp2_;
	ValaInterface* iface;
	gboolean _tmp4_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	cl = (_tmp1_ = (_tmp0_ = vala_data_type_get_data_type (vala_cast_expression_get_type_reference (expr)), VALA_IS_CLASS (_tmp0_) ? ((ValaClass*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	iface = (_tmp3_ = (_tmp2_ = vala_data_type_get_data_type (vala_cast_expression_get_type_reference (expr)), VALA_IS_INTERFACE (_tmp2_) ? ((ValaInterface*) _tmp2_) : NULL), (_tmp3_ == NULL) ? NULL : vala_code_node_ref (_tmp3_));
	_tmp4_ = FALSE;
	if (iface != NULL) {
		_tmp4_ = TRUE;
	} else {
		gboolean _tmp5_;
		_tmp5_ = FALSE;
		if (cl != NULL) {
			_tmp5_ = !vala_class_get_is_compact (cl);
		} else {
			_tmp5_ = FALSE;
		}
		_tmp4_ = _tmp5_;
	}
	if (_tmp4_) {
		/* checked cast for strict subtypes of GTypeInstance*/
		if (vala_cast_expression_get_is_silent_cast (expr)) {
			ValaCCodeCommaExpression* ccomma;
			ValaLocalVariable* temp_decl;
			ValaCCodeExpression* ctemp;
			ValaCCodeAssignment* cinit;
			ValaCCodeExpression* ccheck;
			char* _tmp6_;
			ValaCCodeCastExpression* _tmp7_;
			ValaCCodeCastExpression* ccast;
			ValaCCodeConstant* cnull;
			ValaCCodeConditionalExpression* _tmp8_;
			ccomma = vala_ccode_comma_expression_new ();
			temp_decl = vala_ccode_base_module_get_temp_variable (self, vala_expression_get_value_type (vala_cast_expression_get_inner (expr)), TRUE, (ValaCodeNode*) expr);
			gee_collection_add ((GeeCollection*) self->temp_vars, temp_decl);
			ctemp = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) temp_decl));
			cinit = vala_ccode_assignment_new (ctemp, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_cast_expression_get_inner (expr))), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE);
			ccheck = vala_ccode_base_module_create_type_check (self, (ValaCCodeNode*) ctemp, vala_cast_expression_get_type_reference (expr));
			_tmp6_ = NULL;
			_tmp7_ = NULL;
			ccast = (_tmp7_ = vala_ccode_cast_expression_new (ctemp, _tmp6_ = vala_data_type_get_cname (vala_cast_expression_get_type_reference (expr))), _tmp6_ = (g_free (_tmp6_), NULL), _tmp7_);
			cnull = vala_ccode_constant_new ("NULL");
			vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) cinit);
			_tmp8_ = NULL;
			vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp8_ = vala_ccode_conditional_expression_new (ccheck, (ValaCCodeExpression*) ccast, (ValaCCodeExpression*) cnull)));
			(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL));
			vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) ccomma);
			(ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL));
			(temp_decl == NULL) ? NULL : (temp_decl = (vala_code_node_unref (temp_decl), NULL));
			(ctemp == NULL) ? NULL : (ctemp = (vala_ccode_node_unref (ctemp), NULL));
			(cinit == NULL) ? NULL : (cinit = (vala_ccode_node_unref (cinit), NULL));
			(ccheck == NULL) ? NULL : (ccheck = (vala_ccode_node_unref (ccheck), NULL));
			(ccast == NULL) ? NULL : (ccast = (vala_ccode_node_unref (ccast), NULL));
			(cnull == NULL) ? NULL : (cnull = (vala_ccode_node_unref (cnull), NULL));
		} else {
			ValaInstanceCast* _tmp9_;
			_tmp9_ = NULL;
			vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp9_ = vala_instance_cast_new (VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_cast_expression_get_inner (expr))), vala_data_type_get_data_type (vala_cast_expression_get_type_reference (expr)))));
			(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
		}
	} else {
		ValaCCodeCastExpression* _tmp11_;
		char* _tmp10_;
		if (vala_cast_expression_get_is_silent_cast (expr)) {
			vala_code_node_set_error ((ValaCodeNode*) expr, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) expr), "Operation not supported for this type");
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
			(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
			return;
		}
		_tmp11_ = NULL;
		_tmp10_ = NULL;
		vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp11_ = vala_ccode_cast_expression_new (VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_cast_expression_get_inner (expr))), _tmp10_ = vala_data_type_get_cname (vala_cast_expression_get_type_reference (expr)))));
		(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
		_tmp10_ = (g_free (_tmp10_), NULL);
	}
	(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
	(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
}


static void vala_ccode_base_module_real_visit_pointer_indirection (ValaCCodeModule* base, ValaPointerIndirection* expr) {
	ValaCCodeBaseModule * self;
	ValaCCodeUnaryExpression* _tmp0_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	_tmp0_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp0_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_POINTER_INDIRECTION, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_pointer_indirection_get_inner (expr))))));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
}


static void vala_ccode_base_module_real_visit_addressof_expression (ValaCCodeModule* base, ValaAddressofExpression* expr) {
	ValaCCodeBaseModule * self;
	ValaCCodeUnaryExpression* _tmp0_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	_tmp0_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp0_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_addressof_expression_get_inner (expr))))));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
}


static void vala_ccode_base_module_real_visit_reference_transfer_expression (ValaCCodeModule* base, ValaReferenceTransferExpression* expr) {
	ValaCCodeBaseModule * self;
	ValaCCodeCommaExpression* ccomma;
	ValaLocalVariable* temp_decl;
	ValaCCodeExpression* cvar;
	ValaCCodeAssignment* _tmp0_;
	ValaCCodeAssignment* _tmp2_;
	ValaCCodeConstant* _tmp1_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	/* (tmp = var, var = null, tmp) */
	ccomma = vala_ccode_comma_expression_new ();
	temp_decl = vala_ccode_base_module_get_temp_variable (self, vala_expression_get_value_type ((ValaExpression*) expr), TRUE, (ValaCodeNode*) expr);
	gee_list_insert ((GeeList*) self->temp_vars, 0, temp_decl);
	cvar = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) temp_decl));
	_tmp0_ = NULL;
	vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp0_ = vala_ccode_assignment_new (cvar, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_reference_transfer_expression_get_inner (expr))), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp2_ = vala_ccode_assignment_new (VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_reference_transfer_expression_get_inner (expr))), (ValaCCodeExpression*) (_tmp1_ = vala_ccode_constant_new ("NULL")), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
	(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_ccode_node_unref (_tmp2_), NULL));
	(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_ccode_node_unref (_tmp1_), NULL));
	vala_ccode_comma_expression_append_expression (ccomma, cvar);
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) ccomma);
	(ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL));
	(temp_decl == NULL) ? NULL : (temp_decl = (vala_code_node_unref (temp_decl), NULL));
	(cvar == NULL) ? NULL : (cvar = (vala_ccode_node_unref (cvar), NULL));
}


static void vala_ccode_base_module_real_visit_binary_expression (ValaCCodeModule* base, ValaBinaryExpression* expr) {
	ValaCCodeBaseModule * self;
	ValaCCodeExpression* _tmp0_;
	ValaCCodeExpression* cleft;
	ValaCCodeExpression* _tmp1_;
	ValaCCodeExpression* cright;
	ValaCCodeBinaryOperator op;
	gboolean _tmp4_;
	gboolean _tmp20_;
	gboolean _tmp21_;
	gboolean _tmp22_;
	ValaCCodeBinaryExpression* _tmp47_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) expr, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	_tmp0_ = NULL;
	cleft = (_tmp0_ = VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_binary_expression_get_left (expr))), (_tmp0_ == NULL) ? NULL : vala_ccode_node_ref (_tmp0_));
	_tmp1_ = NULL;
	cright = (_tmp1_ = VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_binary_expression_get_right (expr))), (_tmp1_ == NULL) ? NULL : vala_ccode_node_ref (_tmp1_));
	op = 0;
	if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS) {
		op = VALA_CCODE_BINARY_OPERATOR_PLUS;
	} else {
		if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MINUS) {
			op = VALA_CCODE_BINARY_OPERATOR_MINUS;
		} else {
			if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MUL) {
				op = VALA_CCODE_BINARY_OPERATOR_MUL;
			} else {
				if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_DIV) {
					op = VALA_CCODE_BINARY_OPERATOR_DIV;
				} else {
					if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MOD) {
						op = VALA_CCODE_BINARY_OPERATOR_MOD;
					} else {
						if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_LEFT) {
							op = VALA_CCODE_BINARY_OPERATOR_SHIFT_LEFT;
						} else {
							if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_RIGHT) {
								op = VALA_CCODE_BINARY_OPERATOR_SHIFT_RIGHT;
							} else {
								if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN) {
									op = VALA_CCODE_BINARY_OPERATOR_LESS_THAN;
								} else {
									if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN) {
										op = VALA_CCODE_BINARY_OPERATOR_GREATER_THAN;
									} else {
										if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL) {
											op = VALA_CCODE_BINARY_OPERATOR_LESS_THAN_OR_EQUAL;
										} else {
											if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL) {
												op = VALA_CCODE_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL;
											} else {
												if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_EQUALITY) {
													op = VALA_CCODE_BINARY_OPERATOR_EQUALITY;
												} else {
													if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_INEQUALITY) {
														op = VALA_CCODE_BINARY_OPERATOR_INEQUALITY;
													} else {
														if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_AND) {
															op = VALA_CCODE_BINARY_OPERATOR_BITWISE_AND;
														} else {
															if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_OR) {
																op = VALA_CCODE_BINARY_OPERATOR_BITWISE_OR;
															} else {
																if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_XOR) {
																	op = VALA_CCODE_BINARY_OPERATOR_BITWISE_XOR;
																} else {
																	if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_AND) {
																		op = VALA_CCODE_BINARY_OPERATOR_AND;
																	} else {
																		if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_OR) {
																			op = VALA_CCODE_BINARY_OPERATOR_OR;
																		} else {
																			if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_IN) {
																				ValaCCodeBinaryExpression* _tmp3_;
																				ValaCCodeBinaryExpression* _tmp2_;
																				_tmp3_ = NULL;
																				_tmp2_ = NULL;
																				vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp3_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, (ValaCCodeExpression*) (_tmp2_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_BITWISE_AND, cright, cleft)), cleft)));
																				(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
																				(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_ccode_node_unref (_tmp2_), NULL));
																				(cleft == NULL) ? NULL : (cleft = (vala_ccode_node_unref (cleft), NULL));
																				(cright == NULL) ? NULL : (cright = (vala_ccode_node_unref (cright), NULL));
																				return;
																			} else {
																				g_assert_not_reached ();
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	_tmp4_ = FALSE;
	if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_EQUALITY) {
		_tmp4_ = TRUE;
	} else {
		_tmp4_ = vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_INEQUALITY;
	}
	if (_tmp4_) {
		ValaStruct* _tmp6_;
		ValaTypeSymbol* _tmp5_;
		ValaStruct* left_type_as_struct;
		ValaStruct* _tmp8_;
		ValaTypeSymbol* _tmp7_;
		ValaStruct* right_type_as_struct;
		gboolean _tmp9_;
		gboolean _tmp10_;
		gboolean _tmp11_;
		_tmp6_ = NULL;
		_tmp5_ = NULL;
		left_type_as_struct = (_tmp6_ = (_tmp5_ = vala_data_type_get_data_type (vala_expression_get_value_type (vala_binary_expression_get_left (expr))), VALA_IS_STRUCT (_tmp5_) ? ((ValaStruct*) _tmp5_) : NULL), (_tmp6_ == NULL) ? NULL : vala_code_node_ref (_tmp6_));
		_tmp8_ = NULL;
		_tmp7_ = NULL;
		right_type_as_struct = (_tmp8_ = (_tmp7_ = vala_data_type_get_data_type (vala_expression_get_value_type (vala_binary_expression_get_right (expr))), VALA_IS_STRUCT (_tmp7_) ? ((ValaStruct*) _tmp7_) : NULL), (_tmp8_ == NULL) ? NULL : vala_code_node_ref (_tmp8_));
		_tmp9_ = FALSE;
		_tmp10_ = FALSE;
		_tmp11_ = FALSE;
		if (VALA_IS_CLASS (vala_data_type_get_data_type (vala_expression_get_value_type (vala_binary_expression_get_left (expr))))) {
			_tmp11_ = !vala_class_get_is_compact (VALA_CLASS (vala_data_type_get_data_type (vala_expression_get_value_type (vala_binary_expression_get_left (expr)))));
		} else {
			_tmp11_ = FALSE;
		}
		if (_tmp11_) {
			_tmp10_ = VALA_IS_CLASS (vala_data_type_get_data_type (vala_expression_get_value_type (vala_binary_expression_get_right (expr))));
		} else {
			_tmp10_ = FALSE;
		}
		if (_tmp10_) {
			_tmp9_ = !vala_class_get_is_compact (VALA_CLASS (vala_data_type_get_data_type (vala_expression_get_value_type (vala_binary_expression_get_right (expr)))));
		} else {
			_tmp9_ = FALSE;
		}
		if (_tmp9_) {
			ValaClass* _tmp12_;
			ValaClass* left_cl;
			ValaClass* _tmp13_;
			ValaClass* right_cl;
			_tmp12_ = NULL;
			left_cl = (_tmp12_ = VALA_CLASS (vala_data_type_get_data_type (vala_expression_get_value_type (vala_binary_expression_get_left (expr)))), (_tmp12_ == NULL) ? NULL : vala_code_node_ref (_tmp12_));
			_tmp13_ = NULL;
			right_cl = (_tmp13_ = VALA_CLASS (vala_data_type_get_data_type (vala_expression_get_value_type (vala_binary_expression_get_right (expr)))), (_tmp13_ == NULL) ? NULL : vala_code_node_ref (_tmp13_));
			if (left_cl != right_cl) {
				if (vala_typesymbol_is_subtype_of ((ValaTypeSymbol*) left_cl, (ValaTypeSymbol*) right_cl)) {
					ValaCCodeExpression* _tmp14_;
					_tmp14_ = NULL;
					cleft = (_tmp14_ = (ValaCCodeExpression*) vala_instance_cast_new (cleft, (ValaTypeSymbol*) right_cl), (cleft == NULL) ? NULL : (cleft = (vala_ccode_node_unref (cleft), NULL)), _tmp14_);
				} else {
					if (vala_typesymbol_is_subtype_of ((ValaTypeSymbol*) right_cl, (ValaTypeSymbol*) left_cl)) {
						ValaCCodeExpression* _tmp15_;
						_tmp15_ = NULL;
						cright = (_tmp15_ = (ValaCCodeExpression*) vala_instance_cast_new (cright, (ValaTypeSymbol*) left_cl), (cright == NULL) ? NULL : (cright = (vala_ccode_node_unref (cright), NULL)), _tmp15_);
					}
				}
			}
			(left_cl == NULL) ? NULL : (left_cl = (vala_code_node_unref (left_cl), NULL));
			(right_cl == NULL) ? NULL : (right_cl = (vala_code_node_unref (right_cl), NULL));
		} else {
			gboolean _tmp16_;
			_tmp16_ = FALSE;
			if (left_type_as_struct != NULL) {
				_tmp16_ = right_type_as_struct != NULL;
			} else {
				_tmp16_ = FALSE;
			}
			if (_tmp16_) {
				gboolean _tmp17_;
				_tmp17_ = FALSE;
				if (vala_data_type_get_nullable (vala_expression_get_value_type (vala_binary_expression_get_left (expr)))) {
					_tmp17_ = vala_data_type_get_nullable (vala_expression_get_value_type (vala_binary_expression_get_right (expr)));
				} else {
					_tmp17_ = FALSE;
				}
				/* FIXME generate and use compare/equal function for real structs*/
				if (_tmp17_) {
				} else {
					if (vala_data_type_get_nullable (vala_expression_get_value_type (vala_binary_expression_get_left (expr)))) {
						ValaCCodeExpression* _tmp18_;
						/* FIXME check left value is not null*/
						_tmp18_ = NULL;
						cleft = (_tmp18_ = (ValaCCodeExpression*) vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_POINTER_INDIRECTION, cleft), (cleft == NULL) ? NULL : (cleft = (vala_ccode_node_unref (cleft), NULL)), _tmp18_);
					} else {
						if (vala_data_type_get_nullable (vala_expression_get_value_type (vala_binary_expression_get_right (expr)))) {
							ValaCCodeExpression* _tmp19_;
							/* FIXME check right value is not null*/
							_tmp19_ = NULL;
							cright = (_tmp19_ = (ValaCCodeExpression*) vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_POINTER_INDIRECTION, cright), (cright == NULL) ? NULL : (cright = (vala_ccode_node_unref (cright), NULL)), _tmp19_);
						}
					}
				}
			}
		}
		(left_type_as_struct == NULL) ? NULL : (left_type_as_struct = (vala_code_node_unref (left_type_as_struct), NULL));
		(right_type_as_struct == NULL) ? NULL : (right_type_as_struct = (vala_code_node_unref (right_type_as_struct), NULL));
	}
	_tmp20_ = FALSE;
	_tmp21_ = FALSE;
	_tmp22_ = FALSE;
	if (!VALA_IS_NULL_TYPE (vala_expression_get_value_type (vala_binary_expression_get_left (expr)))) {
		_tmp22_ = vala_data_type_compatible (vala_expression_get_value_type (vala_binary_expression_get_left (expr)), self->string_type);
	} else {
		_tmp22_ = FALSE;
	}
	if (_tmp22_) {
		_tmp21_ = !VALA_IS_NULL_TYPE (vala_expression_get_value_type (vala_binary_expression_get_right (expr)));
	} else {
		_tmp21_ = FALSE;
	}
	if (_tmp21_) {
		_tmp20_ = vala_data_type_compatible (vala_expression_get_value_type (vala_binary_expression_get_right (expr)), self->string_type);
	} else {
		_tmp20_ = FALSE;
	}
	if (_tmp20_) {
		if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS) {
			gboolean _tmp23_;
			_tmp23_ = FALSE;
			if (vala_expression_is_constant (vala_binary_expression_get_left (expr))) {
				_tmp23_ = vala_expression_is_constant (vala_binary_expression_get_right (expr));
			} else {
				_tmp23_ = FALSE;
			}
			/* string concatenation*/
			if (_tmp23_) {
				char* left;
				char* right;
				ValaCCodeConstant* _tmp33_;
				char* _tmp32_;
				left = NULL;
				right = NULL;
				if (VALA_IS_CCODE_IDENTIFIER (cleft)) {
					char* _tmp25_;
					const char* _tmp24_;
					_tmp25_ = NULL;
					_tmp24_ = NULL;
					left = (_tmp25_ = (_tmp24_ = vala_ccode_identifier_get_name (VALA_CCODE_IDENTIFIER (cleft)), (_tmp24_ == NULL) ? NULL : g_strdup (_tmp24_)), left = (g_free (left), NULL), _tmp25_);
				} else {
					if (VALA_IS_CCODE_CONSTANT (cleft)) {
						char* _tmp27_;
						const char* _tmp26_;
						_tmp27_ = NULL;
						_tmp26_ = NULL;
						left = (_tmp27_ = (_tmp26_ = vala_ccode_constant_get_name (VALA_CCODE_CONSTANT (cleft)), (_tmp26_ == NULL) ? NULL : g_strdup (_tmp26_)), left = (g_free (left), NULL), _tmp27_);
					} else {
						g_assert_not_reached ();
					}
				}
				if (VALA_IS_CCODE_IDENTIFIER (cright)) {
					char* _tmp29_;
					const char* _tmp28_;
					_tmp29_ = NULL;
					_tmp28_ = NULL;
					right = (_tmp29_ = (_tmp28_ = vala_ccode_identifier_get_name (VALA_CCODE_IDENTIFIER (cright)), (_tmp28_ == NULL) ? NULL : g_strdup (_tmp28_)), right = (g_free (right), NULL), _tmp29_);
				} else {
					if (VALA_IS_CCODE_CONSTANT (cright)) {
						char* _tmp31_;
						const char* _tmp30_;
						_tmp31_ = NULL;
						_tmp30_ = NULL;
						right = (_tmp31_ = (_tmp30_ = vala_ccode_constant_get_name (VALA_CCODE_CONSTANT (cright)), (_tmp30_ == NULL) ? NULL : g_strdup (_tmp30_)), right = (g_free (right), NULL), _tmp31_);
					} else {
						g_assert_not_reached ();
					}
				}
				_tmp33_ = NULL;
				_tmp32_ = NULL;
				vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp33_ = vala_ccode_constant_new (_tmp32_ = g_strdup_printf ("%s %s", left, right))));
				(_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_ccode_node_unref (_tmp33_), NULL));
				_tmp32_ = (g_free (_tmp32_), NULL);
				left = (g_free (left), NULL);
				right = (g_free (right), NULL);
				(cleft == NULL) ? NULL : (cleft = (vala_ccode_node_unref (cleft), NULL));
				(cright == NULL) ? NULL : (cright = (vala_ccode_node_unref (cright), NULL));
				return;
			} else {
				ValaCCodeIdentifier* _tmp34_;
				ValaCCodeFunctionCall* _tmp35_;
				ValaCCodeFunctionCall* ccall;
				ValaCCodeConstant* _tmp36_;
				/* convert to g_strconcat (a, b, NULL)*/
				_tmp34_ = NULL;
				_tmp35_ = NULL;
				ccall = (_tmp35_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp34_ = vala_ccode_identifier_new ("g_strconcat"))), (_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_ccode_node_unref (_tmp34_), NULL)), _tmp35_);
				vala_ccode_function_call_add_argument (ccall, cleft);
				vala_ccode_function_call_add_argument (ccall, cright);
				_tmp36_ = NULL;
				vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp36_ = vala_ccode_constant_new ("NULL")));
				(_tmp36_ == NULL) ? NULL : (_tmp36_ = (vala_ccode_node_unref (_tmp36_), NULL));
				vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) ccall);
				(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
				(cleft == NULL) ? NULL : (cleft = (vala_ccode_node_unref (cleft), NULL));
				(cright == NULL) ? NULL : (cright = (vala_ccode_node_unref (cright), NULL));
				return;
			}
		} else {
			gboolean _tmp37_;
			gboolean _tmp38_;
			gboolean _tmp39_;
			gboolean _tmp40_;
			gboolean _tmp41_;
			_tmp37_ = FALSE;
			_tmp38_ = FALSE;
			_tmp39_ = FALSE;
			_tmp40_ = FALSE;
			_tmp41_ = FALSE;
			if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_EQUALITY) {
				_tmp41_ = TRUE;
			} else {
				_tmp41_ = vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_INEQUALITY;
			}
			if (_tmp41_) {
				_tmp40_ = TRUE;
			} else {
				_tmp40_ = vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN;
			}
			if (_tmp40_) {
				_tmp39_ = TRUE;
			} else {
				_tmp39_ = vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN;
			}
			if (_tmp39_) {
				_tmp38_ = TRUE;
			} else {
				_tmp38_ = vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL;
			}
			if (_tmp38_) {
				_tmp37_ = TRUE;
			} else {
				_tmp37_ = vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL;
			}
			if (_tmp37_) {
				ValaCCodeIdentifier* _tmp42_;
				ValaCCodeFunctionCall* _tmp43_;
				ValaCCodeFunctionCall* ccall;
				ValaCCodeExpression* _tmp45_;
				ValaCCodeExpression* _tmp44_;
				ValaCCodeExpression* _tmp46_;
				self->requires_strcmp0 = TRUE;
				_tmp42_ = NULL;
				_tmp43_ = NULL;
				ccall = (_tmp43_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp42_ = vala_ccode_identifier_new ("_vala_strcmp0"))), (_tmp42_ == NULL) ? NULL : (_tmp42_ = (vala_ccode_node_unref (_tmp42_), NULL)), _tmp43_);
				vala_ccode_function_call_add_argument (ccall, cleft);
				vala_ccode_function_call_add_argument (ccall, cright);
				_tmp45_ = NULL;
				_tmp44_ = NULL;
				cleft = (_tmp45_ = (_tmp44_ = (ValaCCodeExpression*) ccall, (_tmp44_ == NULL) ? NULL : vala_ccode_node_ref (_tmp44_)), (cleft == NULL) ? NULL : (cleft = (vala_ccode_node_unref (cleft), NULL)), _tmp45_);
				_tmp46_ = NULL;
				cright = (_tmp46_ = (ValaCCodeExpression*) vala_ccode_constant_new ("0"), (cright == NULL) ? NULL : (cright = (vala_ccode_node_unref (cright), NULL)), _tmp46_);
				(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
			}
		}
	}
	_tmp47_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp47_ = vala_ccode_binary_expression_new (op, cleft, cright)));
	(_tmp47_ == NULL) ? NULL : (_tmp47_ = (vala_ccode_node_unref (_tmp47_), NULL));
	(cleft == NULL) ? NULL : (cleft = (vala_ccode_node_unref (cleft), NULL));
	(cright == NULL) ? NULL : (cright = (vala_ccode_node_unref (cright), NULL));
}


char* vala_ccode_base_module_get_type_check_function (ValaCCodeBaseModule* self, ValaTypeSymbol* type) {
	ValaClass* _tmp1_;
	ValaTypeSymbol* _tmp0_;
	ValaClass* cl;
	gboolean _tmp2_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type != NULL, NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	cl = (_tmp1_ = (_tmp0_ = type, VALA_IS_CLASS (_tmp0_) ? ((ValaClass*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp2_ = FALSE;
	if (cl != NULL) {
		_tmp2_ = vala_class_get_type_check_function (cl) != NULL;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		const char* _tmp3_;
		char* _tmp4_;
		_tmp3_ = NULL;
		_tmp4_ = NULL;
		return (_tmp4_ = (_tmp3_ = vala_class_get_type_check_function (cl), (_tmp3_ == NULL) ? NULL : g_strdup (_tmp3_)), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp4_);
	} else {
		char* _tmp5_;
		_tmp5_ = NULL;
		return (_tmp5_ = vala_typesymbol_get_upper_case_cname (type, "IS_"), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp5_);
	}
	(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
}


static ValaCCodeExpression* vala_ccode_base_module_create_type_check (ValaCCodeBaseModule* self, ValaCCodeNode* ccodenode, ValaDataType* type) {
	ValaErrorType* _tmp1_;
	ValaDataType* _tmp0_;
	ValaErrorType* et;
	gboolean _tmp2_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (ccodenode != NULL, NULL);
	g_return_val_if_fail (type != NULL, NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	et = (_tmp1_ = (_tmp0_ = type, VALA_IS_ERROR_TYPE (_tmp0_) ? ((ValaErrorType*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp2_ = FALSE;
	if (et != NULL) {
		_tmp2_ = vala_error_type_get_error_code (et) != NULL;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		ValaCCodeIdentifier* _tmp3_;
		ValaCCodeFunctionCall* _tmp4_;
		ValaCCodeFunctionCall* matches_call;
		ValaCCodeIdentifier* _tmp6_;
		char* _tmp5_;
		ValaCCodeIdentifier* _tmp8_;
		char* _tmp7_;
		ValaCCodeExpression* _tmp9_;
		_tmp3_ = NULL;
		_tmp4_ = NULL;
		matches_call = (_tmp4_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp3_ = vala_ccode_identifier_new ("g_error_matches"))), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL)), _tmp4_);
		vala_ccode_function_call_add_argument (matches_call, VALA_CCODE_EXPRESSION (ccodenode));
		_tmp6_ = NULL;
		_tmp5_ = NULL;
		vala_ccode_function_call_add_argument (matches_call, (ValaCCodeExpression*) (_tmp6_ = vala_ccode_identifier_new (_tmp5_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) vala_error_type_get_error_domain (et), NULL))));
		(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
		_tmp5_ = (g_free (_tmp5_), NULL);
		_tmp8_ = NULL;
		_tmp7_ = NULL;
		vala_ccode_function_call_add_argument (matches_call, (ValaCCodeExpression*) (_tmp8_ = vala_ccode_identifier_new (_tmp7_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) vala_error_type_get_error_code (et), FALSE))));
		(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL));
		_tmp7_ = (g_free (_tmp7_), NULL);
		_tmp9_ = NULL;
		return (_tmp9_ = (ValaCCodeExpression*) matches_call, (et == NULL) ? NULL : (et = (vala_code_node_unref (et), NULL)), _tmp9_);
	} else {
		gboolean _tmp10_;
		_tmp10_ = FALSE;
		if (et != NULL) {
			_tmp10_ = vala_error_type_get_error_domain (et) != NULL;
		} else {
			_tmp10_ = FALSE;
		}
		if (_tmp10_) {
			ValaCCodeMemberAccess* instance_domain;
			char* _tmp11_;
			ValaCCodeIdentifier* _tmp12_;
			ValaCCodeIdentifier* type_domain;
			ValaCCodeExpression* _tmp13_;
			instance_domain = vala_ccode_member_access_new_pointer (VALA_CCODE_EXPRESSION (ccodenode), "domain");
			_tmp11_ = NULL;
			_tmp12_ = NULL;
			type_domain = (_tmp12_ = vala_ccode_identifier_new (_tmp11_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) vala_error_type_get_error_domain (et), NULL)), _tmp11_ = (g_free (_tmp11_), NULL), _tmp12_);
			_tmp13_ = NULL;
			return (_tmp13_ = (ValaCCodeExpression*) vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, (ValaCCodeExpression*) instance_domain, (ValaCCodeExpression*) type_domain), (instance_domain == NULL) ? NULL : (instance_domain = (vala_ccode_node_unref (instance_domain), NULL)), (type_domain == NULL) ? NULL : (type_domain = (vala_ccode_node_unref (type_domain), NULL)), (et == NULL) ? NULL : (et = (vala_code_node_unref (et), NULL)), _tmp13_);
		} else {
			ValaCCodeIdentifier* _tmp15_;
			char* _tmp14_;
			ValaCCodeFunctionCall* _tmp16_;
			ValaCCodeFunctionCall* ccheck;
			ValaCCodeExpression* _tmp17_;
			_tmp15_ = NULL;
			_tmp14_ = NULL;
			_tmp16_ = NULL;
			ccheck = (_tmp16_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp15_ = vala_ccode_identifier_new (_tmp14_ = vala_ccode_base_module_get_type_check_function (self, vala_data_type_get_data_type (type))))), (_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL)), _tmp14_ = (g_free (_tmp14_), NULL), _tmp16_);
			vala_ccode_function_call_add_argument (ccheck, VALA_CCODE_EXPRESSION (ccodenode));
			_tmp17_ = NULL;
			return (_tmp17_ = (ValaCCodeExpression*) ccheck, (et == NULL) ? NULL : (et = (vala_code_node_unref (et), NULL)), _tmp17_);
		}
	}
	(et == NULL) ? NULL : (et = (vala_code_node_unref (et), NULL));
}


static void vala_ccode_base_module_real_visit_type_check (ValaCCodeModule* base, ValaTypeCheck* expr) {
	ValaCCodeBaseModule * self;
	ValaCCodeExpression* _tmp0_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (expr != NULL);
	vala_ccode_base_module_generate_type_declaration (self, vala_typecheck_get_type_reference (expr), self->source_declarations);
	_tmp0_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) expr, (ValaCCodeNode*) (_tmp0_ = vala_ccode_base_module_create_type_check (self, vala_code_node_get_ccodenode ((ValaCodeNode*) vala_typecheck_get_expression (expr)), vala_typecheck_get_type_reference (expr))));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
}


static void vala_ccode_base_module_real_visit_lambda_expression (ValaCCodeModule* base, ValaLambdaExpression* l) {
	ValaCCodeBaseModule * self;
	ValaDelegateType* _tmp0_;
	ValaDelegateType* dt;
	GeeArrayList* _tmp1_;
	GeeArrayList* old_temp_vars;
	GeeArrayList* _tmp2_;
	GeeArrayList* old_temp_ref_vars;
	GeeArrayList* _tmp3_;
	GeeArrayList* _tmp4_;
	GeeArrayList* _tmp6_;
	GeeArrayList* _tmp5_;
	GeeArrayList* _tmp8_;
	GeeArrayList* _tmp7_;
	ValaCCodeIdentifier* _tmp10_;
	char* _tmp9_;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (l != NULL);
	/* use instance position from delegate*/
	_tmp0_ = NULL;
	dt = (_tmp0_ = VALA_DELEGATE_TYPE (vala_expression_get_target_type ((ValaExpression*) l)), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	vala_method_set_cinstance_parameter_position (vala_lambda_expression_get_method (l), vala_delegate_get_cinstance_parameter_position (vala_delegate_type_get_delegate_symbol (dt)));
	_tmp1_ = NULL;
	old_temp_vars = (_tmp1_ = self->temp_vars, (_tmp1_ == NULL) ? NULL : gee_collection_object_ref (_tmp1_));
	_tmp2_ = NULL;
	old_temp_ref_vars = (_tmp2_ = self->temp_ref_vars, (_tmp2_ == NULL) ? NULL : gee_collection_object_ref (_tmp2_));
	_tmp3_ = NULL;
	self->temp_vars = (_tmp3_ = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal), (self->temp_vars == NULL) ? NULL : (self->temp_vars = (gee_collection_object_unref (self->temp_vars), NULL)), _tmp3_);
	_tmp4_ = NULL;
	self->temp_ref_vars = (_tmp4_ = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal), (self->temp_ref_vars == NULL) ? NULL : (self->temp_ref_vars = (gee_collection_object_unref (self->temp_ref_vars), NULL)), _tmp4_);
	vala_code_node_accept_children ((ValaCodeNode*) l, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	_tmp6_ = NULL;
	_tmp5_ = NULL;
	self->temp_vars = (_tmp6_ = (_tmp5_ = old_temp_vars, (_tmp5_ == NULL) ? NULL : gee_collection_object_ref (_tmp5_)), (self->temp_vars == NULL) ? NULL : (self->temp_vars = (gee_collection_object_unref (self->temp_vars), NULL)), _tmp6_);
	_tmp8_ = NULL;
	_tmp7_ = NULL;
	self->temp_ref_vars = (_tmp8_ = (_tmp7_ = old_temp_ref_vars, (_tmp7_ == NULL) ? NULL : gee_collection_object_ref (_tmp7_)), (self->temp_ref_vars == NULL) ? NULL : (self->temp_ref_vars = (gee_collection_object_unref (self->temp_ref_vars), NULL)), _tmp8_);
	_tmp10_ = NULL;
	_tmp9_ = NULL;
	vala_code_node_set_ccodenode ((ValaCodeNode*) l, (ValaCCodeNode*) (_tmp10_ = vala_ccode_identifier_new (_tmp9_ = vala_method_get_cname (vala_lambda_expression_get_method (l)))));
	(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
	_tmp9_ = (g_free (_tmp9_), NULL);
	(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
	(old_temp_vars == NULL) ? NULL : (old_temp_vars = (gee_collection_object_unref (old_temp_vars), NULL));
	(old_temp_ref_vars == NULL) ? NULL : (old_temp_ref_vars = (gee_collection_object_unref (old_temp_ref_vars), NULL));
}


ValaCCodeExpression* vala_ccode_base_module_convert_from_generic_pointer (ValaCCodeBaseModule* self, ValaCCodeExpression* cexpr, ValaDataType* actual_type) {
	ValaCCodeExpression* _tmp0_;
	ValaCCodeExpression* _result_;
	gboolean _tmp1_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (cexpr != NULL, NULL);
	g_return_val_if_fail (actual_type != NULL, NULL);
	_tmp0_ = NULL;
	_result_ = (_tmp0_ = cexpr, (_tmp0_ == NULL) ? NULL : vala_ccode_node_ref (_tmp0_));
	_tmp1_ = FALSE;
	if (vala_ccode_base_module_is_reference_type_argument (self, actual_type)) {
		_tmp1_ = TRUE;
	} else {
		_tmp1_ = vala_ccode_base_module_is_nullable_value_type_argument (self, actual_type);
	}
	if (_tmp1_) {
		ValaCCodeExpression* _tmp3_;
		char* _tmp2_;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		_result_ = (_tmp3_ = (ValaCCodeExpression*) vala_ccode_cast_expression_new (cexpr, _tmp2_ = vala_data_type_get_cname (actual_type)), (_result_ == NULL) ? NULL : (_result_ = (vala_ccode_node_unref (_result_), NULL)), _tmp3_);
		_tmp2_ = (g_free (_tmp2_), NULL);
	} else {
		if (vala_ccode_base_module_is_signed_integer_type_argument (self, actual_type)) {
			ValaCCodeIdentifier* _tmp4_;
			ValaCCodeFunctionCall* _tmp5_;
			ValaCCodeFunctionCall* cconv;
			ValaCCodeExpression* _tmp7_;
			ValaCCodeExpression* _tmp6_;
			_tmp4_ = NULL;
			_tmp5_ = NULL;
			cconv = (_tmp5_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp4_ = vala_ccode_identifier_new ("GPOINTER_TO_INT"))), (_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL)), _tmp5_);
			vala_ccode_function_call_add_argument (cconv, cexpr);
			_tmp7_ = NULL;
			_tmp6_ = NULL;
			_result_ = (_tmp7_ = (_tmp6_ = (ValaCCodeExpression*) cconv, (_tmp6_ == NULL) ? NULL : vala_ccode_node_ref (_tmp6_)), (_result_ == NULL) ? NULL : (_result_ = (vala_ccode_node_unref (_result_), NULL)), _tmp7_);
			(cconv == NULL) ? NULL : (cconv = (vala_ccode_node_unref (cconv), NULL));
		} else {
			if (vala_ccode_base_module_is_unsigned_integer_type_argument (self, actual_type)) {
				ValaCCodeIdentifier* _tmp8_;
				ValaCCodeFunctionCall* _tmp9_;
				ValaCCodeFunctionCall* cconv;
				ValaCCodeExpression* _tmp11_;
				ValaCCodeExpression* _tmp10_;
				_tmp8_ = NULL;
				_tmp9_ = NULL;
				cconv = (_tmp9_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp8_ = vala_ccode_identifier_new ("GPOINTER_TO_UINT"))), (_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL)), _tmp9_);
				vala_ccode_function_call_add_argument (cconv, cexpr);
				_tmp11_ = NULL;
				_tmp10_ = NULL;
				_result_ = (_tmp11_ = (_tmp10_ = (ValaCCodeExpression*) cconv, (_tmp10_ == NULL) ? NULL : vala_ccode_node_ref (_tmp10_)), (_result_ == NULL) ? NULL : (_result_ = (vala_ccode_node_unref (_result_), NULL)), _tmp11_);
				(cconv == NULL) ? NULL : (cconv = (vala_ccode_node_unref (cconv), NULL));
			}
		}
	}
	return _result_;
}


ValaCCodeExpression* vala_ccode_base_module_convert_to_generic_pointer (ValaCCodeBaseModule* self, ValaCCodeExpression* cexpr, ValaDataType* actual_type) {
	ValaCCodeExpression* _tmp0_;
	ValaCCodeExpression* _result_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (cexpr != NULL, NULL);
	g_return_val_if_fail (actual_type != NULL, NULL);
	_tmp0_ = NULL;
	_result_ = (_tmp0_ = cexpr, (_tmp0_ == NULL) ? NULL : vala_ccode_node_ref (_tmp0_));
	if (vala_ccode_base_module_is_signed_integer_type_argument (self, actual_type)) {
		ValaCCodeIdentifier* _tmp1_;
		ValaCCodeFunctionCall* _tmp2_;
		ValaCCodeFunctionCall* cconv;
		ValaCCodeExpression* _tmp4_;
		ValaCCodeExpression* _tmp3_;
		_tmp1_ = NULL;
		_tmp2_ = NULL;
		cconv = (_tmp2_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp1_ = vala_ccode_identifier_new ("GINT_TO_POINTER"))), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_ccode_node_unref (_tmp1_), NULL)), _tmp2_);
		vala_ccode_function_call_add_argument (cconv, cexpr);
		_tmp4_ = NULL;
		_tmp3_ = NULL;
		_result_ = (_tmp4_ = (_tmp3_ = (ValaCCodeExpression*) cconv, (_tmp3_ == NULL) ? NULL : vala_ccode_node_ref (_tmp3_)), (_result_ == NULL) ? NULL : (_result_ = (vala_ccode_node_unref (_result_), NULL)), _tmp4_);
		(cconv == NULL) ? NULL : (cconv = (vala_ccode_node_unref (cconv), NULL));
	} else {
		if (vala_ccode_base_module_is_unsigned_integer_type_argument (self, actual_type)) {
			ValaCCodeIdentifier* _tmp5_;
			ValaCCodeFunctionCall* _tmp6_;
			ValaCCodeFunctionCall* cconv;
			ValaCCodeExpression* _tmp8_;
			ValaCCodeExpression* _tmp7_;
			_tmp5_ = NULL;
			_tmp6_ = NULL;
			cconv = (_tmp6_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp5_ = vala_ccode_identifier_new ("GUINT_TO_POINTER"))), (_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL)), _tmp6_);
			vala_ccode_function_call_add_argument (cconv, cexpr);
			_tmp8_ = NULL;
			_tmp7_ = NULL;
			_result_ = (_tmp8_ = (_tmp7_ = (ValaCCodeExpression*) cconv, (_tmp7_ == NULL) ? NULL : vala_ccode_node_ref (_tmp7_)), (_result_ == NULL) ? NULL : (_result_ = (vala_ccode_node_unref (_result_), NULL)), _tmp8_);
			(cconv == NULL) ? NULL : (cconv = (vala_ccode_node_unref (cconv), NULL));
		}
	}
	return _result_;
}


/* manage memory and implicit casts*/
ValaCCodeExpression* vala_ccode_base_module_transform_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* source_cexpr, ValaDataType* expression_type, ValaDataType* target_type, ValaExpression* expr) {
	ValaCCodeExpression* _tmp0_;
	ValaCCodeExpression* cexpr;
	gboolean _tmp2_;
	gboolean _tmp14_;
	gboolean _tmp15_;
	gboolean _tmp16_;
	gboolean boxing;
	gboolean _tmp17_;
	gboolean _tmp18_;
	gboolean _tmp19_;
	gboolean unboxing;
	gboolean _tmp20_;
	gboolean _tmp56_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (source_cexpr != NULL, NULL);
	_tmp0_ = NULL;
	cexpr = (_tmp0_ = source_cexpr, (_tmp0_ == NULL) ? NULL : vala_ccode_node_ref (_tmp0_));
	if (expression_type == NULL) {
		return cexpr;
	}
	_tmp2_ = FALSE;
	if (vala_data_type_get_value_owned (expression_type)) {
		_tmp2_ = vala_data_type_get_floating_reference (expression_type);
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		ValaObjectTypeSymbol* _tmp4_;
		ValaTypeSymbol* _tmp3_;
		ValaObjectTypeSymbol* cl;
		char* _tmp5_;
		const char* _tmp8_;
		char* sink_func;
		/* floating reference, sink it.
		 */
		_tmp4_ = NULL;
		_tmp3_ = NULL;
		cl = (_tmp4_ = (_tmp3_ = vala_data_type_get_data_type (expression_type), VALA_IS_OBJECT_TYPE_SYMBOL (_tmp3_) ? ((ValaObjectTypeSymbol*) _tmp3_) : NULL), (_tmp4_ == NULL) ? NULL : vala_code_node_ref (_tmp4_));
		_tmp5_ = NULL;
		if (cl != NULL) {
			char* _tmp6_;
			_tmp6_ = NULL;
			_tmp5_ = (_tmp6_ = vala_typesymbol_get_ref_sink_function ((ValaTypeSymbol*) cl), _tmp5_ = (g_free (_tmp5_), NULL), _tmp6_);
		} else {
			char* _tmp7_;
			_tmp7_ = NULL;
			_tmp5_ = (_tmp7_ = NULL, _tmp5_ = (g_free (_tmp5_), NULL), _tmp7_);
		}
		_tmp8_ = NULL;
		sink_func = (_tmp8_ = _tmp5_, (_tmp8_ == NULL) ? NULL : g_strdup (_tmp8_));
		if (sink_func != NULL) {
			ValaCCodeIdentifier* _tmp9_;
			ValaCCodeFunctionCall* _tmp10_;
			ValaCCodeFunctionCall* csink;
			ValaCCodeExpression* _tmp12_;
			ValaCCodeExpression* _tmp11_;
			_tmp9_ = NULL;
			_tmp10_ = NULL;
			csink = (_tmp10_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp9_ = vala_ccode_identifier_new (sink_func))), (_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL)), _tmp10_);
			vala_ccode_function_call_add_argument (csink, cexpr);
			_tmp12_ = NULL;
			_tmp11_ = NULL;
			cexpr = (_tmp12_ = (_tmp11_ = (ValaCCodeExpression*) csink, (_tmp11_ == NULL) ? NULL : vala_ccode_node_ref (_tmp11_)), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), _tmp12_);
			(csink == NULL) ? NULL : (csink = (vala_ccode_node_unref (csink), NULL));
		} else {
			char* _tmp13_;
			_tmp13_ = NULL;
			vala_report_error (NULL, _tmp13_ = g_strdup_printf ("type `%s' does not support floating references", vala_symbol_get_name ((ValaSymbol*) vala_data_type_get_data_type (expression_type))));
			_tmp13_ = (g_free (_tmp13_), NULL);
		}
		(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		_tmp5_ = (g_free (_tmp5_), NULL);
		sink_func = (g_free (sink_func), NULL);
	}
	_tmp14_ = FALSE;
	_tmp15_ = FALSE;
	_tmp16_ = FALSE;
	if (VALA_IS_VALUE_TYPE (expression_type)) {
		_tmp16_ = !vala_data_type_get_nullable (expression_type);
	} else {
		_tmp16_ = FALSE;
	}
	if (_tmp16_) {
		_tmp15_ = VALA_IS_VALUE_TYPE (target_type);
	} else {
		_tmp15_ = FALSE;
	}
	if (_tmp15_) {
		_tmp14_ = vala_data_type_get_nullable (target_type);
	} else {
		_tmp14_ = FALSE;
	}
	boxing = _tmp14_;
	_tmp17_ = FALSE;
	_tmp18_ = FALSE;
	_tmp19_ = FALSE;
	if (VALA_IS_VALUE_TYPE (expression_type)) {
		_tmp19_ = vala_data_type_get_nullable (expression_type);
	} else {
		_tmp19_ = FALSE;
	}
	if (_tmp19_) {
		_tmp18_ = VALA_IS_VALUE_TYPE (target_type);
	} else {
		_tmp18_ = FALSE;
	}
	if (_tmp18_) {
		_tmp17_ = !vala_data_type_get_nullable (target_type);
	} else {
		_tmp17_ = FALSE;
	}
	unboxing = _tmp17_;
	_tmp20_ = FALSE;
	if (vala_data_type_get_value_owned (expression_type)) {
		gboolean _tmp21_;
		gboolean _tmp22_;
		gboolean _tmp23_;
		_tmp21_ = FALSE;
		_tmp22_ = FALSE;
		_tmp23_ = FALSE;
		if (target_type == NULL) {
			_tmp23_ = TRUE;
		} else {
			_tmp23_ = !vala_data_type_get_value_owned (target_type);
		}
		if (_tmp23_) {
			_tmp22_ = TRUE;
		} else {
			_tmp22_ = boxing;
		}
		if (_tmp22_) {
			_tmp21_ = TRUE;
		} else {
			_tmp21_ = unboxing;
		}
		_tmp20_ = _tmp21_;
	} else {
		_tmp20_ = FALSE;
	}
	if (_tmp20_) {
		ValaPointerType* _tmp25_;
		ValaDataType* _tmp24_;
		ValaPointerType* pointer_type;
		gboolean _tmp26_;
		/* value leaked, destroy it*/
		_tmp25_ = NULL;
		_tmp24_ = NULL;
		pointer_type = (_tmp25_ = (_tmp24_ = target_type, VALA_IS_POINTER_TYPE (_tmp24_) ? ((ValaPointerType*) _tmp24_) : NULL), (_tmp25_ == NULL) ? NULL : vala_code_node_ref (_tmp25_));
		_tmp26_ = FALSE;
		if (pointer_type != NULL) {
			_tmp26_ = !VALA_IS_VOID_TYPE (vala_pointer_type_get_base_type (pointer_type));
		} else {
			_tmp26_ = FALSE;
		}
		if (_tmp26_) {
		} else {
			if (vala_ccode_base_module_requires_destroy (self, expression_type)) {
				ValaLocalVariable* decl;
				ValaCCodeExpression* _tmp28_;
				ValaCCodeExpression* _tmp27_;
				gboolean _tmp29_;
				decl = vala_ccode_base_module_get_temp_variable (self, expression_type, TRUE, (ValaCodeNode*) expression_type);
				gee_list_insert ((GeeList*) self->temp_vars, 0, decl);
				gee_list_insert ((GeeList*) self->temp_ref_vars, 0, decl);
				_tmp28_ = NULL;
				_tmp27_ = NULL;
				cexpr = (_tmp28_ = (ValaCCodeExpression*) vala_ccode_assignment_new (_tmp27_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) decl)), cexpr, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), _tmp28_);
				(_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL));
				_tmp29_ = FALSE;
				if (VALA_IS_ARRAY_TYPE (expression_type)) {
					_tmp29_ = expr != NULL;
				} else {
					_tmp29_ = FALSE;
				}
				if (_tmp29_) {
					ValaArrayType* _tmp30_;
					ValaArrayType* array_type;
					ValaCCodeCommaExpression* ccomma;
					ValaCCodeExpression* _tmp37_;
					ValaCCodeExpression* _tmp39_;
					ValaCCodeExpression* _tmp38_;
					_tmp30_ = NULL;
					array_type = (_tmp30_ = VALA_ARRAY_TYPE (expression_type), (_tmp30_ == NULL) ? NULL : vala_code_node_ref (_tmp30_));
					ccomma = vala_ccode_comma_expression_new ();
					vala_ccode_comma_expression_append_expression (ccomma, cexpr);
					{
						gint dim;
						dim = 1;
						for (; dim <= vala_array_type_get_rank (array_type); dim++) {
							char* _tmp32_;
							ValaDataType* _tmp31_;
							ValaLocalVariable* _tmp33_;
							ValaLocalVariable* len_decl;
							ValaCCodeAssignment* _tmp36_;
							ValaCCodeExpression* _tmp35_;
							ValaCCodeExpression* _tmp34_;
							_tmp32_ = NULL;
							_tmp31_ = NULL;
							_tmp33_ = NULL;
							len_decl = (_tmp33_ = vala_local_variable_new (_tmp31_ = vala_data_type_copy (self->int_type), _tmp32_ = vala_ccode_module_get_array_length_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), vala_symbol_get_name ((ValaSymbol*) decl), dim), NULL, NULL), _tmp32_ = (g_free (_tmp32_), NULL), (_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_code_node_unref (_tmp31_), NULL)), _tmp33_);
							gee_list_insert ((GeeList*) self->temp_vars, 0, len_decl);
							_tmp36_ = NULL;
							_tmp35_ = NULL;
							_tmp34_ = NULL;
							vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp36_ = vala_ccode_assignment_new (_tmp34_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) len_decl)), _tmp35_ = vala_ccode_module_get_array_length_cexpression (vala_ccode_module_get_head ((ValaCCodeModule*) self), expr, dim), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
							(_tmp36_ == NULL) ? NULL : (_tmp36_ = (vala_ccode_node_unref (_tmp36_), NULL));
							(_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_ccode_node_unref (_tmp35_), NULL));
							(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_ccode_node_unref (_tmp34_), NULL));
							(len_decl == NULL) ? NULL : (len_decl = (vala_code_node_unref (len_decl), NULL));
						}
					}
					_tmp37_ = NULL;
					vala_ccode_comma_expression_append_expression (ccomma, _tmp37_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) decl)));
					(_tmp37_ == NULL) ? NULL : (_tmp37_ = (vala_ccode_node_unref (_tmp37_), NULL));
					_tmp39_ = NULL;
					_tmp38_ = NULL;
					cexpr = (_tmp39_ = (_tmp38_ = (ValaCCodeExpression*) ccomma, (_tmp38_ == NULL) ? NULL : vala_ccode_node_ref (_tmp38_)), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), _tmp39_);
					(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
					(ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL));
				}
				(decl == NULL) ? NULL : (decl = (vala_code_node_unref (decl), NULL));
			}
		}
		(pointer_type == NULL) ? NULL : (pointer_type = (vala_code_node_unref (pointer_type), NULL));
	}
	if (target_type == NULL) {
		/* value will be destroyed, no need for implicit casts*/
		return cexpr;
	}
	if (boxing) {
		ValaCCodeUnaryExpression* _tmp42_;
		ValaCCodeExpression* _tmp41_;
		ValaCCodeUnaryExpression* unary;
		gboolean _tmp43_;
		/* value needs to be boxed*/
		_tmp42_ = NULL;
		_tmp41_ = NULL;
		unary = (_tmp42_ = (_tmp41_ = cexpr, VALA_IS_CCODE_UNARY_EXPRESSION (_tmp41_) ? ((ValaCCodeUnaryExpression*) _tmp41_) : NULL), (_tmp42_ == NULL) ? NULL : vala_ccode_node_ref (_tmp42_));
		_tmp43_ = FALSE;
		if (unary != NULL) {
			_tmp43_ = vala_ccode_unary_expression_get_operator (unary) == VALA_CCODE_UNARY_OPERATOR_POINTER_INDIRECTION;
		} else {
			_tmp43_ = FALSE;
		}
		if (_tmp43_) {
			ValaCCodeExpression* _tmp45_;
			ValaCCodeExpression* _tmp44_;
			/* *expr => expr*/
			_tmp45_ = NULL;
			_tmp44_ = NULL;
			cexpr = (_tmp45_ = (_tmp44_ = vala_ccode_unary_expression_get_inner (unary), (_tmp44_ == NULL) ? NULL : vala_ccode_node_ref (_tmp44_)), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), _tmp45_);
		} else {
			gboolean _tmp46_;
			_tmp46_ = FALSE;
			if (VALA_IS_CCODE_IDENTIFIER (cexpr)) {
				_tmp46_ = TRUE;
			} else {
				_tmp46_ = VALA_IS_CCODE_MEMBER_ACCESS (cexpr);
			}
			if (_tmp46_) {
				ValaCCodeExpression* _tmp47_;
				_tmp47_ = NULL;
				cexpr = (_tmp47_ = (ValaCCodeExpression*) vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, cexpr), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), _tmp47_);
			} else {
				ValaLocalVariable* decl;
				ValaCCodeCommaExpression* ccomma;
				ValaCCodeAssignment* _tmp49_;
				ValaCCodeExpression* _tmp48_;
				ValaCCodeUnaryExpression* _tmp51_;
				ValaCCodeExpression* _tmp50_;
				ValaCCodeExpression* _tmp53_;
				ValaCCodeExpression* _tmp52_;
				decl = vala_ccode_base_module_get_temp_variable (self, expression_type, vala_data_type_get_value_owned (expression_type), (ValaCodeNode*) expression_type);
				gee_list_insert ((GeeList*) self->temp_vars, 0, decl);
				ccomma = vala_ccode_comma_expression_new ();
				_tmp49_ = NULL;
				_tmp48_ = NULL;
				vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp49_ = vala_ccode_assignment_new (_tmp48_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) decl)), cexpr, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
				(_tmp49_ == NULL) ? NULL : (_tmp49_ = (vala_ccode_node_unref (_tmp49_), NULL));
				(_tmp48_ == NULL) ? NULL : (_tmp48_ = (vala_ccode_node_unref (_tmp48_), NULL));
				_tmp51_ = NULL;
				_tmp50_ = NULL;
				vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp51_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, _tmp50_ = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) decl)))));
				(_tmp51_ == NULL) ? NULL : (_tmp51_ = (vala_ccode_node_unref (_tmp51_), NULL));
				(_tmp50_ == NULL) ? NULL : (_tmp50_ = (vala_ccode_node_unref (_tmp50_), NULL));
				_tmp53_ = NULL;
				_tmp52_ = NULL;
				cexpr = (_tmp53_ = (_tmp52_ = (ValaCCodeExpression*) ccomma, (_tmp52_ == NULL) ? NULL : vala_ccode_node_ref (_tmp52_)), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), _tmp53_);
				(decl == NULL) ? NULL : (decl = (vala_code_node_unref (decl), NULL));
				(ccomma == NULL) ? NULL : (ccomma = (vala_ccode_node_unref (ccomma), NULL));
			}
		}
		(unary == NULL) ? NULL : (unary = (vala_ccode_node_unref (unary), NULL));
	} else {
		if (unboxing) {
			ValaCCodeExpression* _tmp54_;
			/* unbox value*/
			_tmp54_ = NULL;
			cexpr = (_tmp54_ = (ValaCCodeExpression*) vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_POINTER_INDIRECTION, cexpr), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), _tmp54_);
		} else {
			ValaCCodeExpression* _tmp55_;
			_tmp55_ = NULL;
			cexpr = (_tmp55_ = vala_ccode_base_module_get_implicit_cast_expression (self, cexpr, expression_type, target_type, expr), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), _tmp55_);
		}
	}
	_tmp56_ = FALSE;
	if (vala_data_type_get_value_owned (target_type)) {
		gboolean _tmp57_;
		gboolean _tmp58_;
		_tmp57_ = FALSE;
		_tmp58_ = FALSE;
		if (!vala_data_type_get_value_owned (expression_type)) {
			_tmp58_ = TRUE;
		} else {
			_tmp58_ = boxing;
		}
		if (_tmp58_) {
			_tmp57_ = TRUE;
		} else {
			_tmp57_ = unboxing;
		}
		_tmp56_ = _tmp57_;
	} else {
		_tmp56_ = FALSE;
	}
	if (_tmp56_) {
		gboolean _tmp59_;
		_tmp59_ = FALSE;
		if (vala_ccode_base_module_requires_copy (self, target_type)) {
			_tmp59_ = !VALA_IS_NULL_TYPE (expression_type);
		} else {
			_tmp59_ = FALSE;
		}
		/* need to copy value*/
		if (_tmp59_) {
			ValaCodeNode* _tmp60_;
			ValaCodeNode* node;
			ValaCCodeExpression* _tmp63_;
			_tmp60_ = NULL;
			node = (_tmp60_ = (ValaCodeNode*) expr, (_tmp60_ == NULL) ? NULL : vala_code_node_ref (_tmp60_));
			if (node == NULL) {
				ValaCodeNode* _tmp62_;
				ValaCodeNode* _tmp61_;
				_tmp62_ = NULL;
				_tmp61_ = NULL;
				node = (_tmp62_ = (_tmp61_ = (ValaCodeNode*) expression_type, (_tmp61_ == NULL) ? NULL : vala_code_node_ref (_tmp61_)), (node == NULL) ? NULL : (node = (vala_code_node_unref (node), NULL)), _tmp62_);
			}
			_tmp63_ = NULL;
			cexpr = (_tmp63_ = vala_ccode_base_module_get_ref_cexpression (self, target_type, cexpr, expr, node), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), _tmp63_);
			(node == NULL) ? NULL : (node = (vala_code_node_unref (node), NULL));
		}
	}
	return cexpr;
}


static ValaCCodeExpression* vala_ccode_base_module_real_get_implicit_cast_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* source_cexpr, ValaDataType* expression_type, ValaDataType* target_type, ValaExpression* expr) {
	ValaCCodeExpression* _tmp0_;
	ValaCCodeExpression* cexpr;
	gboolean _tmp1_;
	ValaClass* _tmp5_;
	ValaTypeSymbol* _tmp4_;
	ValaClass* cl;
	ValaInterface* _tmp7_;
	ValaTypeSymbol* _tmp6_;
	ValaInterface* iface;
	gboolean _tmp8_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (source_cexpr != NULL, NULL);
	_tmp0_ = NULL;
	cexpr = (_tmp0_ = source_cexpr, (_tmp0_ == NULL) ? NULL : vala_ccode_node_ref (_tmp0_));
	_tmp1_ = FALSE;
	if (vala_data_type_get_data_type (expression_type) != NULL) {
		_tmp1_ = vala_data_type_get_data_type (expression_type) == vala_data_type_get_data_type (target_type);
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		/* same type, no cast required*/
		return cexpr;
	}
	if (VALA_IS_NULL_TYPE (expression_type)) {
		/* null literal, no cast required when not converting to generic type pointer*/
		return cexpr;
	}
	_tmp5_ = NULL;
	_tmp4_ = NULL;
	cl = (_tmp5_ = (_tmp4_ = vala_data_type_get_data_type (target_type), VALA_IS_CLASS (_tmp4_) ? ((ValaClass*) _tmp4_) : NULL), (_tmp5_ == NULL) ? NULL : vala_code_node_ref (_tmp5_));
	_tmp7_ = NULL;
	_tmp6_ = NULL;
	iface = (_tmp7_ = (_tmp6_ = vala_data_type_get_data_type (target_type), VALA_IS_INTERFACE (_tmp6_) ? ((ValaInterface*) _tmp6_) : NULL), (_tmp7_ == NULL) ? NULL : vala_code_node_ref (_tmp7_));
	_tmp8_ = FALSE;
	if (vala_code_context_get_checking (self->priv->_context)) {
		gboolean _tmp9_;
		_tmp9_ = FALSE;
		if (iface != NULL) {
			_tmp9_ = TRUE;
		} else {
			gboolean _tmp10_;
			_tmp10_ = FALSE;
			if (cl != NULL) {
				_tmp10_ = !vala_class_get_is_compact (cl);
			} else {
				_tmp10_ = FALSE;
			}
			_tmp9_ = _tmp10_;
		}
		_tmp8_ = _tmp9_;
	} else {
		_tmp8_ = FALSE;
	}
	if (_tmp8_) {
		ValaCCodeExpression* _tmp11_;
		/* checked cast for strict subtypes of GTypeInstance*/
		_tmp11_ = NULL;
		return (_tmp11_ = (ValaCCodeExpression*) vala_instance_cast_new (cexpr, vala_data_type_get_data_type (target_type)), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), (iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL)), _tmp11_);
	} else {
		gboolean _tmp12_;
		_tmp12_ = FALSE;
		if (vala_data_type_get_data_type (target_type) != NULL) {
			char* _tmp14_;
			char* _tmp13_;
			_tmp14_ = NULL;
			_tmp13_ = NULL;
			_tmp12_ = _vala_strcmp0 (_tmp13_ = vala_data_type_get_cname (expression_type), _tmp14_ = vala_data_type_get_cname (target_type)) != 0;
			_tmp14_ = (g_free (_tmp14_), NULL);
			_tmp13_ = (g_free (_tmp13_), NULL);
		} else {
			_tmp12_ = FALSE;
		}
		if (_tmp12_) {
			ValaStruct* _tmp16_;
			ValaTypeSymbol* _tmp15_;
			ValaStruct* st;
			gboolean _tmp17_;
			_tmp16_ = NULL;
			_tmp15_ = NULL;
			st = (_tmp16_ = (_tmp15_ = vala_data_type_get_data_type (target_type), VALA_IS_STRUCT (_tmp15_) ? ((ValaStruct*) _tmp15_) : NULL), (_tmp16_ == NULL) ? NULL : vala_code_node_ref (_tmp16_));
			_tmp17_ = FALSE;
			if (vala_typesymbol_is_reference_type (vala_data_type_get_data_type (target_type))) {
				_tmp17_ = TRUE;
			} else {
				gboolean _tmp18_;
				_tmp18_ = FALSE;
				if (st != NULL) {
					_tmp18_ = vala_struct_is_simple_type (st);
				} else {
					_tmp18_ = FALSE;
				}
				_tmp17_ = _tmp18_;
			}
			if (_tmp17_) {
				char* _tmp19_;
				ValaCCodeExpression* _tmp20_;
				ValaCCodeExpression* _tmp21_;
				/* don't cast non-simple structs*/
				_tmp19_ = NULL;
				_tmp20_ = NULL;
				_tmp21_ = NULL;
				return (_tmp21_ = (_tmp20_ = (ValaCCodeExpression*) vala_ccode_cast_expression_new (cexpr, _tmp19_ = vala_data_type_get_cname (target_type)), _tmp19_ = (g_free (_tmp19_), NULL), _tmp20_), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), (iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL)), _tmp21_);
			} else {
				ValaCCodeExpression* _tmp22_;
				_tmp22_ = NULL;
				return (_tmp22_ = cexpr, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), (iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL)), _tmp22_);
			}
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		} else {
			ValaCCodeExpression* _tmp23_;
			_tmp23_ = NULL;
			return (_tmp23_ = cexpr, (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), (iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL)), _tmp23_);
		}
	}
	(cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL));
	(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
	(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
}


ValaCCodeExpression* vala_ccode_base_module_get_implicit_cast_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* source_cexpr, ValaDataType* expression_type, ValaDataType* target_type, ValaExpression* expr) {
	return VALA_CCODE_BASE_MODULE_GET_CLASS (self)->get_implicit_cast_expression (self, source_cexpr, expression_type, target_type, expr);
}


ValaCCodeFunctionCall* vala_ccode_base_module_get_property_set_call (ValaCCodeBaseModule* self, ValaProperty* prop, ValaMemberAccess* ma, ValaCCodeExpression* cexpr) {
	char* set_func;
	ValaProperty* _tmp23_;
	ValaProperty* base_property;
	ValaCCodeIdentifier* _tmp30_;
	ValaCCodeFunctionCall* _tmp31_;
	ValaCCodeFunctionCall* ccall;
	ValaCCodeFunctionCall* _tmp35_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (prop != NULL, NULL);
	g_return_val_if_fail (ma != NULL, NULL);
	g_return_val_if_fail (cexpr != NULL, NULL);
	if (VALA_IS_BASE_ACCESS (vala_member_access_get_inner (ma))) {
		if (vala_property_get_base_property (prop) != NULL) {
			ValaClass* _tmp0_;
			ValaClass* base_class;
			ValaCCodeIdentifier* _tmp3_;
			char* _tmp2_;
			char* _tmp1_;
			ValaCCodeFunctionCall* _tmp4_;
			ValaCCodeFunctionCall* vcast;
			ValaCCodeIdentifier* _tmp7_;
			char* _tmp6_;
			char* _tmp5_;
			ValaCCodeMemberAccess* _tmp9_;
			char* _tmp8_;
			ValaCCodeFunctionCall* _tmp10_;
			ValaCCodeFunctionCall* ccall;
			ValaCCodeExpression* _tmp11_;
			ValaCCodeFunctionCall* _tmp12_;
			_tmp0_ = NULL;
			base_class = (_tmp0_ = VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) vala_property_get_base_property (prop))), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
			_tmp3_ = NULL;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			_tmp4_ = NULL;
			vcast = (_tmp4_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp3_ = vala_ccode_identifier_new (_tmp2_ = g_strdup_printf ("%s_CLASS", _tmp1_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) base_class, NULL))))), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL)), _tmp2_ = (g_free (_tmp2_), NULL), _tmp1_ = (g_free (_tmp1_), NULL), _tmp4_);
			_tmp7_ = NULL;
			_tmp6_ = NULL;
			_tmp5_ = NULL;
			vala_ccode_function_call_add_argument (vcast, (ValaCCodeExpression*) (_tmp7_ = vala_ccode_identifier_new (_tmp6_ = g_strdup_printf ("%s_parent_class", _tmp5_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) self->current_class, NULL)))));
			(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
			_tmp6_ = (g_free (_tmp6_), NULL);
			_tmp5_ = (g_free (_tmp5_), NULL);
			_tmp9_ = NULL;
			_tmp8_ = NULL;
			_tmp10_ = NULL;
			ccall = (_tmp10_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp9_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) vcast, _tmp8_ = g_strdup_printf ("set_%s", vala_symbol_get_name ((ValaSymbol*) prop))))), (_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL)), _tmp8_ = (g_free (_tmp8_), NULL), _tmp10_);
			_tmp11_ = NULL;
			vala_ccode_function_call_add_argument (ccall, _tmp11_ = VALA_CCODE_EXPRESSION (vala_ccode_base_module_get_ccodenode (self, (ValaCodeNode*) vala_member_access_get_inner (ma))));
			(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_ccode_node_unref (_tmp11_), NULL));
			vala_ccode_function_call_add_argument (ccall, cexpr);
			_tmp12_ = NULL;
			return (_tmp12_ = ccall, (base_class == NULL) ? NULL : (base_class = (vala_code_node_unref (base_class), NULL)), (vcast == NULL) ? NULL : (vcast = (vala_ccode_node_unref (vcast), NULL)), _tmp12_);
		} else {
			if (vala_property_get_base_interface_property (prop) != NULL) {
				ValaInterface* _tmp13_;
				ValaInterface* base_iface;
				char* _tmp15_;
				char* _tmp14_;
				char* _tmp16_;
				char* parent_iface_var;
				ValaCCodeMemberAccess* _tmp19_;
				char* _tmp18_;
				ValaCCodeIdentifier* _tmp17_;
				ValaCCodeFunctionCall* _tmp20_;
				ValaCCodeFunctionCall* ccall;
				ValaCCodeExpression* _tmp21_;
				ValaCCodeFunctionCall* _tmp22_;
				_tmp13_ = NULL;
				base_iface = (_tmp13_ = VALA_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) vala_property_get_base_interface_property (prop))), (_tmp13_ == NULL) ? NULL : vala_code_node_ref (_tmp13_));
				_tmp15_ = NULL;
				_tmp14_ = NULL;
				_tmp16_ = NULL;
				parent_iface_var = (_tmp16_ = g_strdup_printf ("%s_%s_parent_iface", _tmp14_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) self->current_class, NULL), _tmp15_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) base_iface, NULL)), _tmp15_ = (g_free (_tmp15_), NULL), _tmp14_ = (g_free (_tmp14_), NULL), _tmp16_);
				_tmp19_ = NULL;
				_tmp18_ = NULL;
				_tmp17_ = NULL;
				_tmp20_ = NULL;
				ccall = (_tmp20_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp19_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp17_ = vala_ccode_identifier_new (parent_iface_var)), _tmp18_ = g_strdup_printf ("set_%s", vala_symbol_get_name ((ValaSymbol*) prop))))), (_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL)), _tmp18_ = (g_free (_tmp18_), NULL), (_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_ccode_node_unref (_tmp17_), NULL)), _tmp20_);
				_tmp21_ = NULL;
				vala_ccode_function_call_add_argument (ccall, _tmp21_ = VALA_CCODE_EXPRESSION (vala_ccode_base_module_get_ccodenode (self, (ValaCodeNode*) vala_member_access_get_inner (ma))));
				(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL));
				vala_ccode_function_call_add_argument (ccall, cexpr);
				_tmp22_ = NULL;
				return (_tmp22_ = ccall, (base_iface == NULL) ? NULL : (base_iface = (vala_code_node_unref (base_iface), NULL)), parent_iface_var = (g_free (parent_iface_var), NULL), _tmp22_);
			}
		}
	}
	set_func = g_strdup ("g_object_set");
	_tmp23_ = NULL;
	base_property = (_tmp23_ = prop, (_tmp23_ == NULL) ? NULL : vala_code_node_ref (_tmp23_));
	if (!vala_property_get_no_accessor_method (prop)) {
		if (vala_property_get_base_property (prop) != NULL) {
			ValaProperty* _tmp25_;
			ValaProperty* _tmp24_;
			_tmp25_ = NULL;
			_tmp24_ = NULL;
			base_property = (_tmp25_ = (_tmp24_ = vala_property_get_base_property (prop), (_tmp24_ == NULL) ? NULL : vala_code_node_ref (_tmp24_)), (base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL)), _tmp25_);
		} else {
			if (vala_property_get_base_interface_property (prop) != NULL) {
				ValaProperty* _tmp27_;
				ValaProperty* _tmp26_;
				_tmp27_ = NULL;
				_tmp26_ = NULL;
				base_property = (_tmp27_ = (_tmp26_ = vala_property_get_base_interface_property (prop), (_tmp26_ == NULL) ? NULL : vala_code_node_ref (_tmp26_)), (base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL)), _tmp27_);
			}
		}
		if (VALA_IS_DYNAMIC_PROPERTY (prop)) {
			char* _tmp28_;
			_tmp28_ = NULL;
			set_func = (_tmp28_ = vala_ccode_module_get_dynamic_property_setter_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), VALA_DYNAMIC_PROPERTY (prop)), set_func = (g_free (set_func), NULL), _tmp28_);
		} else {
			char* _tmp29_;
			vala_ccode_base_module_generate_property_accessor_declaration (self, vala_property_get_set_accessor (base_property), self->source_declarations);
			_tmp29_ = NULL;
			set_func = (_tmp29_ = vala_property_accessor_get_cname (vala_property_get_set_accessor (base_property)), set_func = (g_free (set_func), NULL), _tmp29_);
		}
	}
	_tmp30_ = NULL;
	_tmp31_ = NULL;
	ccall = (_tmp31_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp30_ = vala_ccode_identifier_new (set_func))), (_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL)), _tmp31_);
	if (vala_property_get_binding (prop) == MEMBER_BINDING_INSTANCE) {
		ValaCCodeExpression* _tmp32_;
		/* target instance is first argument */
		_tmp32_ = NULL;
		vala_ccode_function_call_add_argument (ccall, _tmp32_ = VALA_CCODE_EXPRESSION (vala_ccode_base_module_get_ccodenode (self, (ValaCodeNode*) vala_member_access_get_inner (ma))));
		(_tmp32_ == NULL) ? NULL : (_tmp32_ = (vala_ccode_node_unref (_tmp32_), NULL));
	}
	if (vala_property_get_no_accessor_method (prop)) {
		ValaCCodeConstant* _tmp33_;
		/* property name is second argument of g_object_set */
		_tmp33_ = NULL;
		vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp33_ = vala_property_get_canonical_cconstant (prop)));
		(_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_ccode_node_unref (_tmp33_), NULL));
	}
	vala_ccode_function_call_add_argument (ccall, cexpr);
	if (vala_property_get_no_accessor_method (prop)) {
		ValaCCodeConstant* _tmp34_;
		_tmp34_ = NULL;
		vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp34_ = vala_ccode_constant_new ("NULL")));
		(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_ccode_node_unref (_tmp34_), NULL));
	}
	_tmp35_ = NULL;
	return (_tmp35_ = ccall, set_func = (g_free (set_func), NULL), (base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL)), _tmp35_);
}


/* indicates whether a given Expression eligable for an ADDRESS_OF operator
 * from a vala to C point of view all expressions denoting locals, fields and
 * parameters are eligable to an ADDRESS_OF operator */
gboolean vala_ccode_base_module_is_address_of_possible (ValaCCodeBaseModule* self, ValaExpression* e) {
	ValaMemberAccess* _tmp1_;
	ValaExpression* _tmp0_;
	ValaMemberAccess* ma;
	gboolean _tmp7_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (e != NULL, FALSE);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	ma = (_tmp1_ = (_tmp0_ = e, VALA_IS_MEMBER_ACCESS (_tmp0_) ? ((ValaMemberAccess*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	if (ma == NULL) {
		gboolean _tmp2_;
		return (_tmp2_ = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp2_);
	}
	if (vala_expression_get_symbol_reference ((ValaExpression*) ma) == NULL) {
		gboolean _tmp3_;
		return (_tmp3_ = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp3_);
	}
	if (VALA_IS_FORMAL_PARAMETER (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
		gboolean _tmp4_;
		return (_tmp4_ = TRUE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp4_);
	}
	if (VALA_IS_LOCAL_VARIABLE (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
		gboolean _tmp5_;
		return (_tmp5_ = TRUE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp5_);
	}
	if (VALA_IS_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
		gboolean _tmp6_;
		return (_tmp6_ = TRUE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp6_);
	}
	return (_tmp7_ = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp7_);
}


/* retrieve the correct address_of expression for a give expression, creates temporary variables
 * where necessary, ce is the corresponding ccode expression for e */
ValaCCodeExpression* vala_ccode_base_module_get_address_of_expression (ValaCCodeBaseModule* self, ValaExpression* e, ValaCCodeExpression* ce) {
	ValaCCodeCommaExpression* ccomma;
	ValaLocalVariable* temp_decl;
	ValaCCodeExpression* ctemp;
	ValaCCodeAssignment* _tmp1_;
	ValaCCodeUnaryExpression* _tmp2_;
	ValaCCodeExpression* _tmp3_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (e != NULL, NULL);
	g_return_val_if_fail (ce != NULL, NULL);
	/* is address of trivially possible?*/
	if (vala_ccode_base_module_is_address_of_possible (self, e)) {
		return (ValaCCodeExpression*) vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, ce);
	}
	ccomma = vala_ccode_comma_expression_new ();
	temp_decl = vala_ccode_base_module_get_temp_variable (self, vala_expression_get_value_type (e), TRUE, NULL);
	ctemp = vala_ccode_base_module_get_variable_cexpression (self, vala_symbol_get_name ((ValaSymbol*) temp_decl));
	gee_collection_add ((GeeCollection*) self->temp_vars, temp_decl);
	_tmp1_ = NULL;
	vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp1_ = vala_ccode_assignment_new (ctemp, ce, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE)));
	(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_ccode_node_unref (_tmp1_), NULL));
	_tmp2_ = NULL;
	vala_ccode_comma_expression_append_expression (ccomma, (ValaCCodeExpression*) (_tmp2_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, ctemp)));
	(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_ccode_node_unref (_tmp2_), NULL));
	_tmp3_ = NULL;
	return (_tmp3_ = (ValaCCodeExpression*) ccomma, (temp_decl == NULL) ? NULL : (temp_decl = (vala_code_node_unref (temp_decl), NULL)), (ctemp == NULL) ? NULL : (ctemp = (vala_ccode_node_unref (ctemp), NULL)), _tmp3_);
}


gboolean vala_ccode_base_module_add_wrapper (ValaCCodeBaseModule* self, const char* wrapper_name) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (wrapper_name != NULL, FALSE);
	return gee_collection_add ((GeeCollection*) self->wrappers, wrapper_name);
}


gboolean vala_ccode_base_module_add_generated_external_symbol (ValaCCodeBaseModule* self, ValaSymbol* external_symbol) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (external_symbol != NULL, FALSE);
	return gee_collection_add ((GeeCollection*) self->priv->generated_external_symbols, external_symbol);
}


ValaDataType* vala_ccode_base_module_get_data_type_for_symbol (ValaTypeSymbol* sym) {
	ValaDataType* type;
	g_return_val_if_fail (sym != NULL, NULL);
	type = NULL;
	if (VALA_IS_CLASS (sym)) {
		ValaDataType* _tmp0_;
		_tmp0_ = NULL;
		type = (_tmp0_ = (ValaDataType*) vala_object_type_new ((ValaObjectTypeSymbol*) VALA_CLASS (sym)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp0_);
	} else {
		if (VALA_IS_INTERFACE (sym)) {
			ValaDataType* _tmp1_;
			_tmp1_ = NULL;
			type = (_tmp1_ = (ValaDataType*) vala_object_type_new ((ValaObjectTypeSymbol*) VALA_INTERFACE (sym)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp1_);
		} else {
			if (VALA_IS_STRUCT (sym)) {
				ValaStruct* _tmp2_;
				ValaStruct* st;
				_tmp2_ = NULL;
				st = (_tmp2_ = VALA_STRUCT (sym), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
				if (vala_struct_is_boolean_type (st)) {
					ValaDataType* _tmp3_;
					_tmp3_ = NULL;
					type = (_tmp3_ = (ValaDataType*) vala_boolean_type_new (st), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp3_);
				} else {
					if (vala_struct_is_integer_type (st)) {
						ValaDataType* _tmp4_;
						_tmp4_ = NULL;
						type = (_tmp4_ = (ValaDataType*) vala_integer_type_new (st, NULL, NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp4_);
					} else {
						if (vala_struct_is_floating_type (st)) {
							ValaDataType* _tmp5_;
							_tmp5_ = NULL;
							type = (_tmp5_ = (ValaDataType*) vala_floating_type_new (st), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp5_);
						} else {
							ValaDataType* _tmp6_;
							_tmp6_ = NULL;
							type = (_tmp6_ = (ValaDataType*) vala_struct_value_type_new (st), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp6_);
						}
					}
				}
				(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
			} else {
				if (VALA_IS_ENUM (sym)) {
					ValaDataType* _tmp7_;
					_tmp7_ = NULL;
					type = (_tmp7_ = (ValaDataType*) vala_enum_value_type_new (VALA_ENUM (sym)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp7_);
				} else {
					if (VALA_IS_ERROR_DOMAIN (sym)) {
						ValaDataType* _tmp8_;
						_tmp8_ = NULL;
						type = (_tmp8_ = (ValaDataType*) vala_error_type_new (VALA_ERROR_DOMAIN (sym), NULL, NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp8_);
					} else {
						if (VALA_IS_ERROR_CODE (sym)) {
							ValaDataType* _tmp9_;
							_tmp9_ = NULL;
							type = (_tmp9_ = (ValaDataType*) vala_error_type_new (VALA_ERROR_DOMAIN (vala_symbol_get_parent_symbol ((ValaSymbol*) sym)), VALA_ERROR_CODE (sym), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp9_);
						} else {
							char* _tmp11_;
							char* _tmp10_;
							ValaDataType* _tmp12_;
							_tmp11_ = NULL;
							_tmp10_ = NULL;
							vala_report_error (NULL, _tmp11_ = g_strdup_printf ("internal error: `%s' is not a supported type", _tmp10_ = vala_symbol_get_full_name ((ValaSymbol*) sym)));
							_tmp11_ = (g_free (_tmp11_), NULL);
							_tmp10_ = (g_free (_tmp10_), NULL);
							_tmp12_ = NULL;
							return (_tmp12_ = (ValaDataType*) vala_invalid_type_new (), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp12_);
						}
					}
				}
			}
		}
	}
	return type;
}


ValaCCodeExpression* vala_ccode_base_module_default_value_for_type (ValaCCodeBaseModule* self, ValaDataType* type, gboolean initializer_expression) {
	ValaStruct* _tmp1_;
	ValaTypeSymbol* _tmp0_;
	ValaStruct* st;
	ValaArrayType* _tmp3_;
	ValaDataType* _tmp2_;
	ValaArrayType* array_type;
	gboolean _tmp4_;
	gboolean _tmp5_;
	ValaCCodeExpression* _tmp25_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type != NULL, NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	st = (_tmp1_ = (_tmp0_ = vala_data_type_get_data_type (type), VALA_IS_STRUCT (_tmp0_) ? ((ValaStruct*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	array_type = (_tmp3_ = (_tmp2_ = type, VALA_IS_ARRAY_TYPE (_tmp2_) ? ((ValaArrayType*) _tmp2_) : NULL), (_tmp3_ == NULL) ? NULL : vala_code_node_ref (_tmp3_));
	_tmp4_ = FALSE;
	_tmp5_ = FALSE;
	if (initializer_expression) {
		_tmp5_ = !vala_data_type_get_nullable (type);
	} else {
		_tmp5_ = FALSE;
	}
	if (_tmp5_) {
		gboolean _tmp6_;
		gboolean _tmp7_;
		_tmp6_ = FALSE;
		_tmp7_ = FALSE;
		if (st != NULL) {
			_tmp7_ = !vala_struct_is_simple_type (st);
		} else {
			_tmp7_ = FALSE;
		}
		if (_tmp7_) {
			_tmp6_ = TRUE;
		} else {
			gboolean _tmp8_;
			_tmp8_ = FALSE;
			if (array_type != NULL) {
				_tmp8_ = vala_array_type_get_fixed_length (array_type);
			} else {
				_tmp8_ = FALSE;
			}
			_tmp6_ = _tmp8_;
		}
		_tmp4_ = _tmp6_;
	} else {
		_tmp4_ = FALSE;
	}
	if (_tmp4_) {
		ValaCCodeInitializerList* clist;
		ValaCCodeConstant* _tmp9_;
		ValaCCodeExpression* _tmp10_;
		/* 0-initialize struct with struct initializer { 0 }
		 only allowed as initializer expression in C*/
		clist = vala_ccode_initializer_list_new ();
		_tmp9_ = NULL;
		vala_ccode_initializer_list_append (clist, (ValaCCodeExpression*) (_tmp9_ = vala_ccode_constant_new ("0")));
		(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
		_tmp10_ = NULL;
		return (_tmp10_ = (ValaCCodeExpression*) clist, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), _tmp10_);
	} else {
		gboolean _tmp11_;
		gboolean _tmp12_;
		gboolean _tmp13_;
		gboolean _tmp14_;
		gboolean _tmp15_;
		_tmp11_ = FALSE;
		_tmp12_ = FALSE;
		_tmp13_ = FALSE;
		_tmp14_ = FALSE;
		_tmp15_ = FALSE;
		if (vala_data_type_get_data_type (type) != NULL) {
			_tmp15_ = vala_typesymbol_is_reference_type (vala_data_type_get_data_type (type));
		} else {
			_tmp15_ = FALSE;
		}
		if (_tmp15_) {
			_tmp14_ = TRUE;
		} else {
			_tmp14_ = vala_data_type_get_nullable (type);
		}
		if (_tmp14_) {
			_tmp13_ = TRUE;
		} else {
			_tmp13_ = VALA_IS_POINTER_TYPE (type);
		}
		if (_tmp13_) {
			_tmp12_ = TRUE;
		} else {
			_tmp12_ = VALA_IS_DELEGATE_TYPE (type);
		}
		if (_tmp12_) {
			_tmp11_ = TRUE;
		} else {
			gboolean _tmp16_;
			_tmp16_ = FALSE;
			if (array_type != NULL) {
				_tmp16_ = !vala_array_type_get_fixed_length (array_type);
			} else {
				_tmp16_ = FALSE;
			}
			_tmp11_ = _tmp16_;
		}
		if (_tmp11_) {
			ValaCCodeExpression* _tmp17_;
			_tmp17_ = NULL;
			return (_tmp17_ = (ValaCCodeExpression*) vala_ccode_constant_new ("NULL"), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), _tmp17_);
		} else {
			gboolean _tmp18_;
			_tmp18_ = FALSE;
			if (vala_data_type_get_data_type (type) != NULL) {
				char* _tmp19_;
				_tmp19_ = NULL;
				_tmp18_ = (_tmp19_ = vala_typesymbol_get_default_value (vala_data_type_get_data_type (type))) != NULL;
				_tmp19_ = (g_free (_tmp19_), NULL);
			} else {
				_tmp18_ = FALSE;
			}
			if (_tmp18_) {
				char* _tmp20_;
				ValaCCodeExpression* _tmp21_;
				ValaCCodeExpression* _tmp22_;
				_tmp20_ = NULL;
				_tmp21_ = NULL;
				_tmp22_ = NULL;
				return (_tmp22_ = (_tmp21_ = (ValaCCodeExpression*) vala_ccode_constant_new (_tmp20_ = vala_typesymbol_get_default_value (vala_data_type_get_data_type (type))), _tmp20_ = (g_free (_tmp20_), NULL), _tmp21_), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), _tmp22_);
			} else {
				if (vala_data_type_get_type_parameter (type) != NULL) {
					ValaCCodeExpression* _tmp23_;
					_tmp23_ = NULL;
					return (_tmp23_ = (ValaCCodeExpression*) vala_ccode_constant_new ("NULL"), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), _tmp23_);
				} else {
					if (VALA_IS_ERROR_TYPE (type)) {
						ValaCCodeExpression* _tmp24_;
						_tmp24_ = NULL;
						return (_tmp24_ = (ValaCCodeExpression*) vala_ccode_constant_new ("NULL"), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), _tmp24_);
					}
				}
			}
		}
	}
	_tmp25_ = NULL;
	return (_tmp25_ = NULL, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), _tmp25_);
}


static ValaCCodeStatement* vala_ccode_base_module_create_property_type_check_statement (ValaCCodeBaseModule* self, ValaProperty* prop, gboolean check_return_type, ValaTypeSymbol* t, gboolean non_null, const char* var_name) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (prop != NULL, NULL);
	g_return_val_if_fail (t != NULL, NULL);
	g_return_val_if_fail (var_name != NULL, NULL);
	if (check_return_type) {
		return vala_ccode_base_module_create_type_check_statement (self, (ValaCodeNode*) prop, vala_property_get_property_type (prop), t, non_null, var_name);
	} else {
		ValaVoidType* _tmp1_;
		ValaCCodeStatement* _tmp2_;
		_tmp1_ = NULL;
		_tmp2_ = NULL;
		return (_tmp2_ = vala_ccode_base_module_create_type_check_statement (self, (ValaCodeNode*) prop, (ValaDataType*) (_tmp1_ = vala_void_type_new (NULL)), t, non_null, var_name), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_code_node_unref (_tmp1_), NULL)), _tmp2_);
	}
}


ValaCCodeStatement* vala_ccode_base_module_create_type_check_statement (ValaCCodeBaseModule* self, ValaCodeNode* method_node, ValaDataType* ret_type, ValaTypeSymbol* t, gboolean non_null, const char* var_name) {
	ValaCCodeFunctionCall* ccheck;
	ValaCreationMethod* _tmp20_;
	ValaCodeNode* _tmp19_;
	ValaCreationMethod* cm;
	gboolean _tmp21_;
	ValaCCodeStatement* _tmp27_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (method_node != NULL, NULL);
	g_return_val_if_fail (ret_type != NULL, NULL);
	g_return_val_if_fail (t != NULL, NULL);
	g_return_val_if_fail (var_name != NULL, NULL);
	ccheck = vala_ccode_function_call_new (NULL);
	if (!vala_code_context_get_assert (self->priv->_context)) {
		ValaCCodeStatement* _tmp0_;
		_tmp0_ = NULL;
		return (_tmp0_ = NULL, (ccheck == NULL) ? NULL : (ccheck = (vala_ccode_node_unref (ccheck), NULL)), _tmp0_);
	} else {
		gboolean _tmp1_;
		_tmp1_ = FALSE;
		if (vala_code_context_get_checking (self->priv->_context)) {
			gboolean _tmp2_;
			gboolean _tmp3_;
			_tmp2_ = FALSE;
			_tmp3_ = FALSE;
			if (VALA_IS_CLASS (t)) {
				_tmp3_ = !vala_class_get_is_compact (VALA_CLASS (t));
			} else {
				_tmp3_ = FALSE;
			}
			if (_tmp3_) {
				_tmp2_ = TRUE;
			} else {
				_tmp2_ = VALA_IS_INTERFACE (t);
			}
			_tmp1_ = _tmp2_;
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			ValaCCodeIdentifier* _tmp5_;
			char* _tmp4_;
			ValaCCodeFunctionCall* _tmp6_;
			ValaCCodeFunctionCall* ctype_check;
			ValaCCodeIdentifier* _tmp7_;
			ValaCCodeExpression* _tmp8_;
			ValaCCodeExpression* cexpr;
			_tmp5_ = NULL;
			_tmp4_ = NULL;
			_tmp6_ = NULL;
			ctype_check = (_tmp6_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp5_ = vala_ccode_identifier_new (_tmp4_ = vala_ccode_base_module_get_type_check_function (self, t)))), (_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL)), _tmp4_ = (g_free (_tmp4_), NULL), _tmp6_);
			_tmp7_ = NULL;
			vala_ccode_function_call_add_argument (ctype_check, (ValaCCodeExpression*) (_tmp7_ = vala_ccode_identifier_new (var_name)));
			(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
			_tmp8_ = NULL;
			cexpr = (_tmp8_ = (ValaCCodeExpression*) ctype_check, (_tmp8_ == NULL) ? NULL : vala_ccode_node_ref (_tmp8_));
			if (!non_null) {
				ValaCCodeConstant* _tmp10_;
				ValaCCodeIdentifier* _tmp9_;
				ValaCCodeBinaryExpression* _tmp11_;
				ValaCCodeBinaryExpression* cnull;
				ValaCCodeExpression* _tmp12_;
				_tmp10_ = NULL;
				_tmp9_ = NULL;
				_tmp11_ = NULL;
				cnull = (_tmp11_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, (ValaCCodeExpression*) (_tmp9_ = vala_ccode_identifier_new (var_name)), (ValaCCodeExpression*) (_tmp10_ = vala_ccode_constant_new ("NULL"))), (_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL)), (_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL)), _tmp11_);
				_tmp12_ = NULL;
				cexpr = (_tmp12_ = (ValaCCodeExpression*) vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_OR, (ValaCCodeExpression*) cnull, (ValaCCodeExpression*) ctype_check), (cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL)), _tmp12_);
				(cnull == NULL) ? NULL : (cnull = (vala_ccode_node_unref (cnull), NULL));
			}
			vala_ccode_function_call_add_argument (ccheck, cexpr);
			(ctype_check == NULL) ? NULL : (ctype_check = (vala_ccode_node_unref (ctype_check), NULL));
			(cexpr == NULL) ? NULL : (cexpr = (vala_ccode_node_unref (cexpr), NULL));
		} else {
			if (!non_null) {
				ValaCCodeStatement* _tmp13_;
				_tmp13_ = NULL;
				return (_tmp13_ = NULL, (ccheck == NULL) ? NULL : (ccheck = (vala_ccode_node_unref (ccheck), NULL)), _tmp13_);
			} else {
				gboolean _tmp14_;
				_tmp14_ = FALSE;
				if (t == VALA_TYPESYMBOL (self->glist_type)) {
					_tmp14_ = TRUE;
				} else {
					_tmp14_ = t == VALA_TYPESYMBOL (self->gslist_type);
				}
				if (_tmp14_) {
					ValaCCodeStatement* _tmp15_;
					/* NULL is empty list*/
					_tmp15_ = NULL;
					return (_tmp15_ = NULL, (ccheck == NULL) ? NULL : (ccheck = (vala_ccode_node_unref (ccheck), NULL)), _tmp15_);
				} else {
					ValaCCodeConstant* _tmp17_;
					ValaCCodeIdentifier* _tmp16_;
					ValaCCodeBinaryExpression* _tmp18_;
					ValaCCodeBinaryExpression* cnonnull;
					_tmp17_ = NULL;
					_tmp16_ = NULL;
					_tmp18_ = NULL;
					cnonnull = (_tmp18_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_INEQUALITY, (ValaCCodeExpression*) (_tmp16_ = vala_ccode_identifier_new (var_name)), (ValaCCodeExpression*) (_tmp17_ = vala_ccode_constant_new ("NULL"))), (_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_ccode_node_unref (_tmp17_), NULL)), (_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL)), _tmp18_);
					vala_ccode_function_call_add_argument (ccheck, (ValaCCodeExpression*) cnonnull);
					(cnonnull == NULL) ? NULL : (cnonnull = (vala_ccode_node_unref (cnonnull), NULL));
				}
			}
		}
	}
	_tmp20_ = NULL;
	_tmp19_ = NULL;
	cm = (_tmp20_ = (_tmp19_ = method_node, VALA_IS_CREATION_METHOD (_tmp19_) ? ((ValaCreationMethod*) _tmp19_) : NULL), (_tmp20_ == NULL) ? NULL : vala_code_node_ref (_tmp20_));
	_tmp21_ = FALSE;
	if (cm != NULL) {
		_tmp21_ = VALA_IS_OBJECT_TYPE_SYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) cm));
	} else {
		_tmp21_ = FALSE;
	}
	if (_tmp21_) {
		ValaCCodeIdentifier* _tmp22_;
		ValaCCodeConstant* _tmp23_;
		_tmp22_ = NULL;
		vala_ccode_function_call_set_call (ccheck, (ValaCCodeExpression*) (_tmp22_ = vala_ccode_identifier_new ("g_return_val_if_fail")));
		(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL));
		_tmp23_ = NULL;
		vala_ccode_function_call_add_argument (ccheck, (ValaCCodeExpression*) (_tmp23_ = vala_ccode_constant_new ("NULL")));
		(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_ccode_node_unref (_tmp23_), NULL));
	} else {
		if (VALA_IS_VOID_TYPE (ret_type)) {
			ValaCCodeIdentifier* _tmp24_;
			/* void function */
			_tmp24_ = NULL;
			vala_ccode_function_call_set_call (ccheck, (ValaCCodeExpression*) (_tmp24_ = vala_ccode_identifier_new ("g_return_if_fail")));
			(_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_ccode_node_unref (_tmp24_), NULL));
		} else {
			ValaCCodeIdentifier* _tmp25_;
			ValaCCodeExpression* cdefault;
			_tmp25_ = NULL;
			vala_ccode_function_call_set_call (ccheck, (ValaCCodeExpression*) (_tmp25_ = vala_ccode_identifier_new ("g_return_val_if_fail")));
			(_tmp25_ == NULL) ? NULL : (_tmp25_ = (vala_ccode_node_unref (_tmp25_), NULL));
			cdefault = vala_ccode_base_module_default_value_for_type (self, ret_type, FALSE);
			if (cdefault != NULL) {
				vala_ccode_function_call_add_argument (ccheck, cdefault);
			} else {
				ValaCCodeStatement* _tmp26_;
				_tmp26_ = NULL;
				return (_tmp26_ = NULL, (cdefault == NULL) ? NULL : (cdefault = (vala_ccode_node_unref (cdefault), NULL)), (ccheck == NULL) ? NULL : (ccheck = (vala_ccode_node_unref (ccheck), NULL)), (cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL)), _tmp26_);
			}
			(cdefault == NULL) ? NULL : (cdefault = (vala_ccode_node_unref (cdefault), NULL));
		}
	}
	_tmp27_ = NULL;
	return (_tmp27_ = (ValaCCodeStatement*) vala_ccode_expression_statement_new ((ValaCCodeExpression*) ccheck), (ccheck == NULL) ? NULL : (ccheck = (vala_ccode_node_unref (ccheck), NULL)), (cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL)), _tmp27_);
}


gint vala_ccode_base_module_get_param_pos (ValaCCodeBaseModule* self, double param_pos, gboolean ellipsis) {
	g_return_val_if_fail (self != NULL, 0);
	if (!ellipsis) {
		if (param_pos >= 0) {
			return (gint) (param_pos * 1000);
		} else {
			return (gint) ((100 + param_pos) * 1000);
		}
	} else {
		if (param_pos >= 0) {
			return (gint) ((100 + param_pos) * 1000);
		} else {
			return (gint) ((200 + param_pos) * 1000);
		}
	}
}


ValaCCodeNode* vala_ccode_base_module_get_ccodenode (ValaCCodeBaseModule* self, ValaCodeNode* node) {
	ValaCCodeNode* _tmp0_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (node != NULL, NULL);
	if (vala_code_node_get_ccodenode (node) == NULL) {
		vala_code_node_accept (node, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	}
	_tmp0_ = NULL;
	return (_tmp0_ = vala_code_node_get_ccodenode (node), (_tmp0_ == NULL) ? NULL : vala_ccode_node_ref (_tmp0_));
}


static void vala_ccode_base_module_real_visit_class (ValaCCodeModule* base, ValaClass* cl) {
	ValaCCodeBaseModule * self;
	self = (ValaCCodeBaseModule*) base;
	g_return_if_fail (cl != NULL);
}


ValaCodeContext* vala_ccode_base_module_get_context (ValaCCodeBaseModule* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_context;
}


void vala_ccode_base_module_set_context (ValaCCodeBaseModule* self, ValaCodeContext* value) {
	ValaCodeContext* _tmp2_;
	ValaCodeContext* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_context = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_context_ref (_tmp1_)), (self->priv->_context == NULL) ? NULL : (self->priv->_context = (vala_code_context_unref (self->priv->_context), NULL)), _tmp2_);
}


static void vala_ccode_base_module_class_init (ValaCCodeBaseModuleClass * klass) {
	vala_ccode_base_module_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_MODULE_CLASS (klass)->finalize = vala_ccode_base_module_finalize;
	g_type_class_add_private (klass, sizeof (ValaCCodeBaseModulePrivate));
	VALA_CCODE_MODULE_CLASS (klass)->emit = vala_ccode_base_module_real_emit;
	VALA_CCODE_MODULE_CLASS (klass)->get_value_setter_function = vala_ccode_base_module_real_get_value_setter_function;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->append_vala_array_free = vala_ccode_base_module_real_append_vala_array_free;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->append_vala_array_move = vala_ccode_base_module_real_append_vala_array_move;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->append_vala_array_length = vala_ccode_base_module_real_append_vala_array_length;
	VALA_CCODE_MODULE_CLASS (klass)->visit_source_file = vala_ccode_base_module_real_visit_source_file;
	VALA_CCODE_MODULE_CLASS (klass)->visit_enum = vala_ccode_base_module_real_visit_enum;
	VALA_CCODE_MODULE_CLASS (klass)->visit_member = vala_ccode_base_module_real_visit_member;
	VALA_CCODE_MODULE_CLASS (klass)->visit_constant = vala_ccode_base_module_real_visit_constant;
	VALA_CCODE_MODULE_CLASS (klass)->visit_field = vala_ccode_base_module_real_visit_field;
	VALA_CCODE_MODULE_CLASS (klass)->visit_formal_parameter = vala_ccode_base_module_real_visit_formal_parameter;
	VALA_CCODE_MODULE_CLASS (klass)->visit_property = vala_ccode_base_module_real_visit_property;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->generate_class_struct_declaration = vala_ccode_base_module_real_generate_class_struct_declaration;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->generate_struct_declaration = vala_ccode_base_module_real_generate_struct_declaration;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->generate_delegate_declaration = vala_ccode_base_module_real_generate_delegate_declaration;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->generate_cparameters = vala_ccode_base_module_real_generate_cparameters;
	VALA_CCODE_MODULE_CLASS (klass)->visit_property_accessor = vala_ccode_base_module_real_visit_property_accessor;
	VALA_CCODE_MODULE_CLASS (klass)->visit_destructor = vala_ccode_base_module_real_visit_destructor;
	VALA_CCODE_MODULE_CLASS (klass)->visit_block = vala_ccode_base_module_real_visit_block;
	VALA_CCODE_MODULE_CLASS (klass)->visit_empty_statement = vala_ccode_base_module_real_visit_empty_statement;
	VALA_CCODE_MODULE_CLASS (klass)->visit_declaration_statement = vala_ccode_base_module_real_visit_declaration_statement;
	VALA_CCODE_MODULE_CLASS (klass)->visit_local_variable = vala_ccode_base_module_real_visit_local_variable;
	VALA_CCODE_MODULE_CLASS (klass)->visit_initializer_list = vala_ccode_base_module_real_visit_initializer_list;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->get_dup_func_expression = vala_ccode_base_module_real_get_dup_func_expression;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->get_unref_expression = vala_ccode_base_module_real_get_unref_expression;
	VALA_CCODE_MODULE_CLASS (klass)->visit_end_full_expression = vala_ccode_base_module_real_visit_end_full_expression;
	VALA_CCODE_MODULE_CLASS (klass)->visit_expression_statement = vala_ccode_base_module_real_visit_expression_statement;
	VALA_CCODE_MODULE_CLASS (klass)->visit_return_statement = vala_ccode_base_module_real_visit_return_statement;
	VALA_CCODE_MODULE_CLASS (klass)->visit_lock_statement = vala_ccode_base_module_real_visit_lock_statement;
	VALA_CCODE_MODULE_CLASS (klass)->visit_delete_statement = vala_ccode_base_module_real_visit_delete_statement;
	VALA_CCODE_MODULE_CLASS (klass)->visit_expression = vala_ccode_base_module_real_visit_expression;
	VALA_CCODE_MODULE_CLASS (klass)->visit_boolean_literal = vala_ccode_base_module_real_visit_boolean_literal;
	VALA_CCODE_MODULE_CLASS (klass)->visit_character_literal = vala_ccode_base_module_real_visit_character_literal;
	VALA_CCODE_MODULE_CLASS (klass)->visit_integer_literal = vala_ccode_base_module_real_visit_integer_literal;
	VALA_CCODE_MODULE_CLASS (klass)->visit_real_literal = vala_ccode_base_module_real_visit_real_literal;
	VALA_CCODE_MODULE_CLASS (klass)->visit_string_literal = vala_ccode_base_module_real_visit_string_literal;
	VALA_CCODE_MODULE_CLASS (klass)->visit_null_literal = vala_ccode_base_module_real_visit_null_literal;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->get_delegate_target_cname = vala_ccode_base_module_real_get_delegate_target_cname;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->get_delegate_target_cexpression = vala_ccode_base_module_real_get_delegate_target_cexpression;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->get_delegate_target_destroy_notify_cname = vala_ccode_base_module_real_get_delegate_target_destroy_notify_cname;
	VALA_CCODE_MODULE_CLASS (klass)->visit_base_access = vala_ccode_base_module_real_visit_base_access;
	VALA_CCODE_MODULE_CLASS (klass)->visit_postfix_expression = vala_ccode_base_module_real_visit_postfix_expression;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->get_ref_cexpression = vala_ccode_base_module_real_get_ref_cexpression;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->generate_class_declaration = vala_ccode_base_module_real_generate_class_declaration;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->generate_interface_declaration = vala_ccode_base_module_real_generate_interface_declaration;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->generate_method_declaration = vala_ccode_base_module_real_generate_method_declaration;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->generate_error_domain_declaration = vala_ccode_base_module_real_generate_error_domain_declaration;
	VALA_CCODE_MODULE_CLASS (klass)->visit_object_creation_expression = vala_ccode_base_module_real_visit_object_creation_expression;
	VALA_CCODE_MODULE_CLASS (klass)->visit_sizeof_expression = vala_ccode_base_module_real_visit_sizeof_expression;
	VALA_CCODE_MODULE_CLASS (klass)->visit_typeof_expression = vala_ccode_base_module_real_visit_typeof_expression;
	VALA_CCODE_MODULE_CLASS (klass)->visit_unary_expression = vala_ccode_base_module_real_visit_unary_expression;
	VALA_CCODE_MODULE_CLASS (klass)->visit_cast_expression = vala_ccode_base_module_real_visit_cast_expression;
	VALA_CCODE_MODULE_CLASS (klass)->visit_pointer_indirection = vala_ccode_base_module_real_visit_pointer_indirection;
	VALA_CCODE_MODULE_CLASS (klass)->visit_addressof_expression = vala_ccode_base_module_real_visit_addressof_expression;
	VALA_CCODE_MODULE_CLASS (klass)->visit_reference_transfer_expression = vala_ccode_base_module_real_visit_reference_transfer_expression;
	VALA_CCODE_MODULE_CLASS (klass)->visit_binary_expression = vala_ccode_base_module_real_visit_binary_expression;
	VALA_CCODE_MODULE_CLASS (klass)->visit_type_check = vala_ccode_base_module_real_visit_type_check;
	VALA_CCODE_MODULE_CLASS (klass)->visit_lambda_expression = vala_ccode_base_module_real_visit_lambda_expression;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->get_implicit_cast_expression = vala_ccode_base_module_real_get_implicit_cast_expression;
	VALA_CCODE_MODULE_CLASS (klass)->visit_class = vala_ccode_base_module_real_visit_class;
}


static void vala_ccode_base_module_instance_init (ValaCCodeBaseModule * self) {
	self->priv = VALA_CCODE_BASE_MODULE_GET_PRIVATE (self);
	self->temp_vars = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->temp_ref_vars = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->next_temp_var_id = 0;
	self->in_creation_method = FALSE;
	self->in_constructor = FALSE;
	self->in_static_or_class_ctor = FALSE;
	self->current_method_inner_error = FALSE;
	self->next_coroutine_state = 1;
	self->in_plugin = FALSE;
	self->variable_name_map = (GeeMap*) gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal);
}


static void vala_ccode_base_module_finalize (ValaCCodeModule* obj) {
	ValaCCodeBaseModule * self;
	self = VALA_CCODE_BASE_MODULE (obj);
	(self->priv->_context == NULL) ? NULL : (self->priv->_context = (vala_code_context_unref (self->priv->_context), NULL));
	(self->root_symbol == NULL) ? NULL : (self->root_symbol = (vala_code_node_unref (self->root_symbol), NULL));
	(self->current_symbol == NULL) ? NULL : (self->current_symbol = (vala_code_node_unref (self->current_symbol), NULL));
	(self->current_type_symbol == NULL) ? NULL : (self->current_type_symbol = (vala_code_node_unref (self->current_type_symbol), NULL));
	(self->current_class == NULL) ? NULL : (self->current_class = (vala_code_node_unref (self->current_class), NULL));
	(self->current_method == NULL) ? NULL : (self->current_method = (vala_code_node_unref (self->current_method), NULL));
	(self->current_return_type == NULL) ? NULL : (self->current_return_type = (vala_code_node_unref (self->current_return_type), NULL));
	(self->current_try == NULL) ? NULL : (self->current_try = (vala_code_node_unref (self->current_try), NULL));
	(self->current_property_accessor == NULL) ? NULL : (self->current_property_accessor = (vala_code_node_unref (self->current_property_accessor), NULL));
	(self->header_declarations == NULL) ? NULL : (self->header_declarations = (vala_ccode_declaration_space_unref (self->header_declarations), NULL));
	(self->internal_header_declarations == NULL) ? NULL : (self->internal_header_declarations = (vala_ccode_declaration_space_unref (self->internal_header_declarations), NULL));
	(self->source_declarations == NULL) ? NULL : (self->source_declarations = (vala_ccode_declaration_space_unref (self->source_declarations), NULL));
	(self->source_signal_marshaller_declaration == NULL) ? NULL : (self->source_signal_marshaller_declaration = (vala_ccode_node_unref (self->source_signal_marshaller_declaration), NULL));
	(self->source_type_member_definition == NULL) ? NULL : (self->source_type_member_definition = (vala_ccode_node_unref (self->source_type_member_definition), NULL));
	(self->class_init_fragment == NULL) ? NULL : (self->class_init_fragment = (vala_ccode_node_unref (self->class_init_fragment), NULL));
	(self->base_init_fragment == NULL) ? NULL : (self->base_init_fragment = (vala_ccode_node_unref (self->base_init_fragment), NULL));
	(self->class_finalize_fragment == NULL) ? NULL : (self->class_finalize_fragment = (vala_ccode_node_unref (self->class_finalize_fragment), NULL));
	(self->base_finalize_fragment == NULL) ? NULL : (self->base_finalize_fragment = (vala_ccode_node_unref (self->base_finalize_fragment), NULL));
	(self->instance_init_fragment == NULL) ? NULL : (self->instance_init_fragment = (vala_ccode_node_unref (self->instance_init_fragment), NULL));
	(self->instance_finalize_fragment == NULL) ? NULL : (self->instance_finalize_fragment = (vala_ccode_node_unref (self->instance_finalize_fragment), NULL));
	(self->source_signal_marshaller_definition == NULL) ? NULL : (self->source_signal_marshaller_definition = (vala_ccode_node_unref (self->source_signal_marshaller_definition), NULL));
	(self->module_init_fragment == NULL) ? NULL : (self->module_init_fragment = (vala_ccode_node_unref (self->module_init_fragment), NULL));
	(self->param_spec_struct == NULL) ? NULL : (self->param_spec_struct = (vala_ccode_node_unref (self->param_spec_struct), NULL));
	(self->closure_struct == NULL) ? NULL : (self->closure_struct = (vala_ccode_node_unref (self->closure_struct), NULL));
	(self->prop_enum == NULL) ? NULL : (self->prop_enum = (vala_ccode_node_unref (self->prop_enum), NULL));
	(self->function == NULL) ? NULL : (self->function = (vala_ccode_node_unref (self->function), NULL));
	(self->pre_statement_fragment == NULL) ? NULL : (self->pre_statement_fragment = (vala_ccode_node_unref (self->pre_statement_fragment), NULL));
	(self->temp_vars == NULL) ? NULL : (self->temp_vars = (gee_collection_object_unref (self->temp_vars), NULL));
	(self->temp_ref_vars == NULL) ? NULL : (self->temp_ref_vars = (gee_collection_object_unref (self->temp_ref_vars), NULL));
	(self->user_marshal_set == NULL) ? NULL : (self->user_marshal_set = (gee_collection_object_unref (self->user_marshal_set), NULL));
	(self->predefined_marshal_set == NULL) ? NULL : (self->predefined_marshal_set = (gee_collection_object_unref (self->predefined_marshal_set), NULL));
	(self->priv->reserved_identifiers == NULL) ? NULL : (self->priv->reserved_identifiers = (gee_collection_object_unref (self->priv->reserved_identifiers), NULL));
	(self->bool_type == NULL) ? NULL : (self->bool_type = (vala_code_node_unref (self->bool_type), NULL));
	(self->char_type == NULL) ? NULL : (self->char_type = (vala_code_node_unref (self->char_type), NULL));
	(self->uchar_type == NULL) ? NULL : (self->uchar_type = (vala_code_node_unref (self->uchar_type), NULL));
	(self->unichar_type == NULL) ? NULL : (self->unichar_type = (vala_code_node_unref (self->unichar_type), NULL));
	(self->short_type == NULL) ? NULL : (self->short_type = (vala_code_node_unref (self->short_type), NULL));
	(self->ushort_type == NULL) ? NULL : (self->ushort_type = (vala_code_node_unref (self->ushort_type), NULL));
	(self->int_type == NULL) ? NULL : (self->int_type = (vala_code_node_unref (self->int_type), NULL));
	(self->uint_type == NULL) ? NULL : (self->uint_type = (vala_code_node_unref (self->uint_type), NULL));
	(self->long_type == NULL) ? NULL : (self->long_type = (vala_code_node_unref (self->long_type), NULL));
	(self->ulong_type == NULL) ? NULL : (self->ulong_type = (vala_code_node_unref (self->ulong_type), NULL));
	(self->int8_type == NULL) ? NULL : (self->int8_type = (vala_code_node_unref (self->int8_type), NULL));
	(self->uint8_type == NULL) ? NULL : (self->uint8_type = (vala_code_node_unref (self->uint8_type), NULL));
	(self->int16_type == NULL) ? NULL : (self->int16_type = (vala_code_node_unref (self->int16_type), NULL));
	(self->uint16_type == NULL) ? NULL : (self->uint16_type = (vala_code_node_unref (self->uint16_type), NULL));
	(self->int32_type == NULL) ? NULL : (self->int32_type = (vala_code_node_unref (self->int32_type), NULL));
	(self->uint32_type == NULL) ? NULL : (self->uint32_type = (vala_code_node_unref (self->uint32_type), NULL));
	(self->int64_type == NULL) ? NULL : (self->int64_type = (vala_code_node_unref (self->int64_type), NULL));
	(self->uint64_type == NULL) ? NULL : (self->uint64_type = (vala_code_node_unref (self->uint64_type), NULL));
	(self->string_type == NULL) ? NULL : (self->string_type = (vala_code_node_unref (self->string_type), NULL));
	(self->float_type == NULL) ? NULL : (self->float_type = (vala_code_node_unref (self->float_type), NULL));
	(self->double_type == NULL) ? NULL : (self->double_type = (vala_code_node_unref (self->double_type), NULL));
	(self->gtype_type == NULL) ? NULL : (self->gtype_type = (vala_code_node_unref (self->gtype_type), NULL));
	(self->gobject_type == NULL) ? NULL : (self->gobject_type = (vala_code_node_unref (self->gobject_type), NULL));
	(self->gerror_type == NULL) ? NULL : (self->gerror_type = (vala_code_node_unref (self->gerror_type), NULL));
	(self->glist_type == NULL) ? NULL : (self->glist_type = (vala_code_node_unref (self->glist_type), NULL));
	(self->gslist_type == NULL) ? NULL : (self->gslist_type = (vala_code_node_unref (self->gslist_type), NULL));
	(self->gstringbuilder_type == NULL) ? NULL : (self->gstringbuilder_type = (vala_code_node_unref (self->gstringbuilder_type), NULL));
	(self->garray_type == NULL) ? NULL : (self->garray_type = (vala_code_node_unref (self->garray_type), NULL));
	(self->gbytearray_type == NULL) ? NULL : (self->gbytearray_type = (vala_code_node_unref (self->gbytearray_type), NULL));
	(self->gptrarray_type == NULL) ? NULL : (self->gptrarray_type = (vala_code_node_unref (self->gptrarray_type), NULL));
	(self->gquark_type == NULL) ? NULL : (self->gquark_type = (vala_code_node_unref (self->gquark_type), NULL));
	(self->gvalue_type == NULL) ? NULL : (self->gvalue_type = (vala_code_node_unref (self->gvalue_type), NULL));
	(self->mutex_type == NULL) ? NULL : (self->mutex_type = (vala_code_node_unref (self->mutex_type), NULL));
	(self->type_module_type == NULL) ? NULL : (self->type_module_type = (vala_code_node_unref (self->type_module_type), NULL));
	(self->dbus_object_type == NULL) ? NULL : (self->dbus_object_type = (vala_code_node_unref (self->dbus_object_type), NULL));
	self->module_init_param_name = (g_free (self->module_init_param_name), NULL);
	(self->wrappers == NULL) ? NULL : (self->wrappers = (gee_collection_object_unref (self->wrappers), NULL));
	(self->priv->generated_external_symbols == NULL) ? NULL : (self->priv->generated_external_symbols = (gee_collection_object_unref (self->priv->generated_external_symbols), NULL));
	(self->variable_name_map == NULL) ? NULL : (self->variable_name_map = (gee_collection_object_unref (self->variable_name_map), NULL));
	VALA_CCODE_MODULE_CLASS (vala_ccode_base_module_parent_class)->finalize (obj);
}


GType vala_ccode_base_module_get_type (void) {
	static GType vala_ccode_base_module_type_id = 0;
	if (vala_ccode_base_module_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeBaseModuleClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_base_module_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeBaseModule), 0, (GInstanceInitFunc) vala_ccode_base_module_instance_init, NULL };
		vala_ccode_base_module_type_id = g_type_register_static (VALA_TYPE_CCODE_MODULE, "ValaCCodeBaseModule", &g_define_type_info, 0);
	}
	return vala_ccode_base_module_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




