/* valaformalparameter.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 * Copyright (C) 2006-2008  Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>
#include <float.h>
#include <math.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;

#define VALA_TYPE_FORMAL_PARAMETER (vala_formal_parameter_get_type ())
#define VALA_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameter))
#define VALA_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))
#define VALA_IS_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_IS_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_FORMAL_PARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))

typedef struct _ValaFormalParameter ValaFormalParameter;
typedef struct _ValaFormalParameterClass ValaFormalParameterClass;
typedef struct _ValaFormalParameterPrivate ValaFormalParameterPrivate;

#define VALA_TYPE_PARAMETER_DIRECTION (vala_parameter_direction_get_type ())

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_SYMBOL_ACCESSIBILITY (vala_symbol_accessibility_get_type ())

#define VALA_TYPE_SOURCE_FILE (vala_source_file_get_type ())
#define VALA_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFile))
#define VALA_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))
#define VALA_IS_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_FILE))
#define VALA_IS_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_FILE))
#define VALA_SOURCE_FILE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))

typedef struct _ValaSourceFile ValaSourceFile;
typedef struct _ValaSourceFileClass ValaSourceFileClass;

#define VALA_TYPE_REFERENCE_TYPE (vala_reference_type_get_type ())
#define VALA_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceType))
#define VALA_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))
#define VALA_IS_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_REFERENCE_TYPE))
#define VALA_IS_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_REFERENCE_TYPE))
#define VALA_REFERENCE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))

typedef struct _ValaReferenceType ValaReferenceType;
typedef struct _ValaReferenceTypeClass ValaReferenceTypeClass;

#define VALA_TYPE_ARRAY_TYPE (vala_array_type_get_type ())
#define VALA_ARRAY_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ARRAY_TYPE, ValaArrayType))
#define VALA_ARRAY_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ARRAY_TYPE, ValaArrayTypeClass))
#define VALA_IS_ARRAY_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ARRAY_TYPE))
#define VALA_IS_ARRAY_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ARRAY_TYPE))
#define VALA_ARRAY_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ARRAY_TYPE, ValaArrayTypeClass))

typedef struct _ValaArrayType ValaArrayType;
typedef struct _ValaArrayTypeClass ValaArrayTypeClass;

#define VALA_TYPE_LITERAL (vala_literal_get_type ())
#define VALA_LITERAL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LITERAL, ValaLiteral))
#define VALA_LITERAL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LITERAL, ValaLiteralClass))
#define VALA_IS_LITERAL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LITERAL))
#define VALA_IS_LITERAL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LITERAL))
#define VALA_LITERAL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LITERAL, ValaLiteralClass))

typedef struct _ValaLiteral ValaLiteral;
typedef struct _ValaLiteralClass ValaLiteralClass;

#define VALA_TYPE_NULL_LITERAL (vala_null_literal_get_type ())
#define VALA_NULL_LITERAL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_NULL_LITERAL, ValaNullLiteral))
#define VALA_NULL_LITERAL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_NULL_LITERAL, ValaNullLiteralClass))
#define VALA_IS_NULL_LITERAL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_NULL_LITERAL))
#define VALA_IS_NULL_LITERAL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_NULL_LITERAL))
#define VALA_NULL_LITERAL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_NULL_LITERAL, ValaNullLiteralClass))

typedef struct _ValaNullLiteral ValaNullLiteral;
typedef struct _ValaNullLiteralClass ValaNullLiteralClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Represents a node in the symbol tree.
 */
struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
};

/**
 * Represents a formal parameter in method and callback signatures.
 */
struct _ValaFormalParameter {
	ValaSymbol parent_instance;
	ValaFormalParameterPrivate * priv;
};

struct _ValaFormalParameterClass {
	ValaSymbolClass parent_class;
};

typedef enum  {
	VALA_PARAMETER_DIRECTION_IN,
	VALA_PARAMETER_DIRECTION_OUT,
	VALA_PARAMETER_DIRECTION_REF
} ValaParameterDirection;

struct _ValaFormalParameterPrivate {
	ValaParameterDirection _direction;
	gboolean _ellipsis;
	gboolean _params_array;
	ValaExpression* _default_expression;
	gboolean _no_array_length;
	gboolean _array_null_terminated;
	double _cparameter_position;
	double _carray_length_parameter_position;
	double _cdelegate_target_parameter_position;
	gboolean _async_only;
	char* _ctype;
	ValaDataType* _data_type;
};

typedef enum  {
	VALA_SYMBOL_ACCESSIBILITY_PRIVATE,
	VALA_SYMBOL_ACCESSIBILITY_INTERNAL,
	VALA_SYMBOL_ACCESSIBILITY_PROTECTED,
	VALA_SYMBOL_ACCESSIBILITY_PUBLIC
} ValaSymbolAccessibility;



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_formal_parameter_get_type (void);
GType vala_parameter_direction_get_type (void);
#define VALA_FORMAL_PARAMETER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterPrivate))
enum  {
	VALA_FORMAL_PARAMETER_DUMMY_PROPERTY
};
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
ValaSymbol* vala_symbol_new (const char* name, ValaSourceReference* source_reference);
ValaSymbol* vala_symbol_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
void vala_formal_parameter_set_parameter_type (ValaFormalParameter* self, ValaDataType* value);
GType vala_symbol_accessibility_get_type (void);
void vala_symbol_set_access (ValaSymbol* self, ValaSymbolAccessibility value);
ValaFormalParameter* vala_formal_parameter_new (const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference);
ValaFormalParameter* vala_formal_parameter_construct (GType object_type, const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference);
ValaFormalParameter* vala_formal_parameter_new (const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference);
void vala_formal_parameter_set_ellipsis (ValaFormalParameter* self, gboolean value);
ValaFormalParameter* vala_formal_parameter_new_with_ellipsis (ValaSourceReference* source_reference);
ValaFormalParameter* vala_formal_parameter_construct_with_ellipsis (GType object_type, ValaSourceReference* source_reference);
ValaFormalParameter* vala_formal_parameter_new_with_ellipsis (ValaSourceReference* source_reference);
void vala_code_visitor_visit_formal_parameter (ValaCodeVisitor* self, ValaFormalParameter* p);
static void vala_formal_parameter_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
gboolean vala_formal_parameter_get_ellipsis (ValaFormalParameter* self);
ValaDataType* vala_formal_parameter_get_parameter_type (ValaFormalParameter* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
ValaExpression* vala_formal_parameter_get_default_expression (ValaFormalParameter* self);
static void vala_formal_parameter_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_formal_parameter_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
gboolean vala_attribute_has_argument (ValaAttribute* self, const char* name);
char* vala_attribute_get_string (ValaAttribute* self, const char* name);
void vala_formal_parameter_set_ctype (ValaFormalParameter* self, const char* value);
gboolean vala_attribute_get_bool (ValaAttribute* self, const char* name);
void vala_formal_parameter_set_no_array_length (ValaFormalParameter* self, gboolean value);
void vala_formal_parameter_set_array_null_terminated (ValaFormalParameter* self, gboolean value);
double vala_attribute_get_double (ValaAttribute* self, const char* name);
void vala_formal_parameter_set_carray_length_parameter_position (ValaFormalParameter* self, double value);
void vala_formal_parameter_set_cdelegate_target_parameter_position (ValaFormalParameter* self, double value);
void vala_formal_parameter_set_async_only (ValaFormalParameter* self, gboolean value);
static void vala_formal_parameter_process_ccode_attribute (ValaFormalParameter* self, ValaAttribute* a);
const char* vala_attribute_get_name (ValaAttribute* self);
void vala_formal_parameter_process_attributes (ValaFormalParameter* self);
const char* vala_symbol_get_name (ValaSymbol* self);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
gboolean vala_formal_parameter_get_params_array (ValaFormalParameter* self);
void vala_formal_parameter_set_params_array (ValaFormalParameter* self, gboolean value);
ValaParameterDirection vala_formal_parameter_get_direction (ValaFormalParameter* self);
void vala_formal_parameter_set_direction (ValaFormalParameter* self, ValaParameterDirection value);
ValaFormalParameter* vala_formal_parameter_copy (ValaFormalParameter* self);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gpointer vala_source_file_ref (gpointer instance);
void vala_source_file_unref (gpointer instance);
GParamSpec* vala_param_spec_source_file (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_file (GValue* value, gpointer v_object);
gpointer vala_value_get_source_file (const GValue* value);
GType vala_source_file_get_type (void);
ValaSourceFile* vala_semantic_analyzer_get_current_source_file (ValaSemanticAnalyzer* self);
ValaSymbol* vala_semantic_analyzer_get_current_symbol (ValaSemanticAnalyzer* self);
ValaSourceFile* vala_source_reference_get_file (ValaSourceReference* self);
void vala_semantic_analyzer_set_current_source_file (ValaSemanticAnalyzer* self, ValaSourceFile* value);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
void vala_semantic_analyzer_set_current_symbol (ValaSemanticAnalyzer* self, ValaSymbol* value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
GType vala_reference_type_get_type (void);
GType vala_array_type_get_type (void);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
void vala_report_error (ValaSourceReference* source, const char* message);
GType vala_literal_get_type (void);
GType vala_null_literal_get_type (void);
gboolean vala_data_type_get_nullable (ValaDataType* self);
void vala_report_warning (ValaSourceReference* source, const char* message);
char* vala_code_node_to_string (ValaCodeNode* self);
gboolean vala_semantic_analyzer_is_type_accessible (ValaSemanticAnalyzer* self, ValaSymbol* sym, ValaDataType* type);
char* vala_symbol_get_full_name (ValaSymbol* self);
static gboolean vala_formal_parameter_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
void vala_formal_parameter_set_default_expression (ValaFormalParameter* self, ValaExpression* value);
gboolean vala_formal_parameter_get_no_array_length (ValaFormalParameter* self);
gboolean vala_formal_parameter_get_array_null_terminated (ValaFormalParameter* self);
double vala_formal_parameter_get_cparameter_position (ValaFormalParameter* self);
void vala_formal_parameter_set_cparameter_position (ValaFormalParameter* self, double value);
double vala_formal_parameter_get_carray_length_parameter_position (ValaFormalParameter* self);
double vala_formal_parameter_get_cdelegate_target_parameter_position (ValaFormalParameter* self);
gboolean vala_formal_parameter_get_async_only (ValaFormalParameter* self);
const char* vala_formal_parameter_get_ctype (ValaFormalParameter* self);
static gpointer vala_formal_parameter_parent_class = NULL;
static void vala_formal_parameter_finalize (ValaCodeNode* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new formal parameter.
 *
 * @param name   parameter name
 * @param type   parameter type
 * @param source reference to source code
 * @return       newly created formal parameter
 */
ValaFormalParameter* vala_formal_parameter_construct (GType object_type, const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference) {
	ValaFormalParameter* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (parameter_type != NULL, NULL);
	self = (ValaFormalParameter*) vala_symbol_construct (object_type, name, source_reference);
	vala_formal_parameter_set_parameter_type (self, parameter_type);
	vala_symbol_set_access ((ValaSymbol*) self, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	return self;
}


ValaFormalParameter* vala_formal_parameter_new (const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference) {
	return vala_formal_parameter_construct (VALA_TYPE_FORMAL_PARAMETER, name, parameter_type, source_reference);
}


/**
 * Creates a new ellipsis parameter representing an indefinite number of
 * parameters.
 */
ValaFormalParameter* vala_formal_parameter_construct_with_ellipsis (GType object_type, ValaSourceReference* source_reference) {
	ValaFormalParameter* self;
	self = (ValaFormalParameter*) vala_symbol_construct (object_type, NULL, source_reference);
	vala_formal_parameter_set_ellipsis (self, TRUE);
	vala_symbol_set_access ((ValaSymbol*) self, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	return self;
}


ValaFormalParameter* vala_formal_parameter_new_with_ellipsis (ValaSourceReference* source_reference) {
	return vala_formal_parameter_construct_with_ellipsis (VALA_TYPE_FORMAL_PARAMETER, source_reference);
}


static void vala_formal_parameter_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaFormalParameter * self;
	self = (ValaFormalParameter*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_formal_parameter (visitor, self);
}


static void vala_formal_parameter_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaFormalParameter * self;
	self = (ValaFormalParameter*) base;
	g_return_if_fail (visitor != NULL);
	if (!self->priv->_ellipsis) {
		vala_code_node_accept ((ValaCodeNode*) vala_formal_parameter_get_parameter_type (self), visitor);
		if (self->priv->_default_expression != NULL) {
			vala_code_node_accept ((ValaCodeNode*) self->priv->_default_expression, visitor);
		}
	}
}


static void vala_formal_parameter_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaFormalParameter * self;
	self = (ValaFormalParameter*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_formal_parameter_get_parameter_type (self) == old_type) {
		vala_formal_parameter_set_parameter_type (self, new_type);
	}
}


static void vala_formal_parameter_process_ccode_attribute (ValaFormalParameter* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "type")) {
		char* _tmp0_;
		_tmp0_ = NULL;
		vala_formal_parameter_set_ctype (self, _tmp0_ = vala_attribute_get_string (a, "type"));
		_tmp0_ = (g_free (_tmp0_), NULL);
	}
	if (vala_attribute_has_argument (a, "array_length")) {
		vala_formal_parameter_set_no_array_length (self, !vala_attribute_get_bool (a, "array_length"));
	}
	if (vala_attribute_has_argument (a, "array_null_terminated")) {
		vala_formal_parameter_set_array_null_terminated (self, vala_attribute_get_bool (a, "array_null_terminated"));
	}
	if (vala_attribute_has_argument (a, "array_length_pos")) {
		vala_formal_parameter_set_carray_length_parameter_position (self, vala_attribute_get_double (a, "array_length_pos"));
	}
	if (vala_attribute_has_argument (a, "delegate_target_pos")) {
		vala_formal_parameter_set_cdelegate_target_parameter_position (self, vala_attribute_get_double (a, "delegate_target_pos"));
	}
	if (vala_attribute_has_argument (a, "async_only")) {
		vala_formal_parameter_set_async_only (self, vala_attribute_get_bool (a, "async_only"));
	}
}


/**
 * Process all associated attributes.
 */
void vala_formal_parameter_process_attributes (ValaFormalParameter* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0_;
			ValaAttribute* a;
			_tmp0_ = NULL;
			a = (_tmp0_ = (ValaAttribute*) a_it->data, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_formal_parameter_process_ccode_attribute (self, a);
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


ValaFormalParameter* vala_formal_parameter_copy (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (!self->priv->_ellipsis) {
		ValaFormalParameter* _result_;
		_result_ = vala_formal_parameter_new (vala_symbol_get_name ((ValaSymbol*) self), vala_formal_parameter_get_parameter_type (self), vala_code_node_get_source_reference ((ValaCodeNode*) self));
		vala_formal_parameter_set_params_array (_result_, self->priv->_params_array);
		vala_formal_parameter_set_direction (_result_, self->priv->_direction);
		return _result_;
	} else {
		return vala_formal_parameter_new_with_ellipsis (NULL);
	}
}


static gboolean vala_formal_parameter_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaFormalParameter * self;
	ValaSourceFile* _tmp1_;
	ValaSourceFile* old_source_file;
	ValaSymbol* _tmp2_;
	ValaSymbol* old_symbol;
	gboolean _tmp12_;
	self = (ValaFormalParameter*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_formal_parameter_process_attributes (self);
	_tmp1_ = NULL;
	old_source_file = (_tmp1_ = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp1_ == NULL) ? NULL : vala_source_file_ref (_tmp1_));
	_tmp2_ = NULL;
	old_symbol = (_tmp2_ = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, vala_symbol_get_parent_symbol ((ValaSymbol*) self));
	if (vala_formal_parameter_get_parameter_type (self) != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_formal_parameter_get_parameter_type (self), analyzer);
	}
	if (!self->priv->_ellipsis) {
		gboolean _tmp3_;
		vala_code_node_check ((ValaCodeNode*) vala_formal_parameter_get_parameter_type (self), analyzer);
		_tmp3_ = FALSE;
		if (self->priv->_params_array) {
			_tmp3_ = !VALA_IS_ARRAY_TYPE (vala_formal_parameter_get_parameter_type (self));
		} else {
			_tmp3_ = FALSE;
		}
		if (_tmp3_) {
			gboolean _tmp4_;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "parameter array expected");
			return (_tmp4_ = FALSE, (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), _tmp4_);
		}
		if (self->priv->_default_expression != NULL) {
			vala_code_node_check ((ValaCodeNode*) self->priv->_default_expression, analyzer);
		}
	}
	if (self->priv->_default_expression != NULL) {
		gboolean _tmp5_;
		gboolean _tmp6_;
		_tmp5_ = FALSE;
		_tmp6_ = FALSE;
		if (VALA_IS_NULL_LITERAL (self->priv->_default_expression)) {
			_tmp6_ = !vala_data_type_get_nullable (vala_formal_parameter_get_parameter_type (self));
		} else {
			_tmp6_ = FALSE;
		}
		if (_tmp6_) {
			_tmp5_ = self->priv->_direction != VALA_PARAMETER_DIRECTION_OUT;
		} else {
			_tmp5_ = FALSE;
		}
		if (_tmp5_) {
			char* _tmp8_;
			char* _tmp7_;
			_tmp8_ = NULL;
			_tmp7_ = NULL;
			vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp8_ = g_strdup_printf ("`null' incompatible with parameter type `%s`", _tmp7_ = vala_code_node_to_string ((ValaCodeNode*) vala_formal_parameter_get_parameter_type (self))));
			_tmp8_ = (g_free (_tmp8_), NULL);
			_tmp7_ = (g_free (_tmp7_), NULL);
		}
	}
	if (!self->priv->_ellipsis) {
		/* check whether parameter type is at least as accessible as the method*/
		if (!vala_semantic_analyzer_is_type_accessible (analyzer, (ValaSymbol*) self, vala_formal_parameter_get_parameter_type (self))) {
			char* _tmp11_;
			char* _tmp10_;
			char* _tmp9_;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp11_ = NULL;
			_tmp10_ = NULL;
			_tmp9_ = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp11_ = g_strdup_printf ("parameter type `%s` is less accessible than method `%s`", _tmp9_ = vala_code_node_to_string ((ValaCodeNode*) vala_formal_parameter_get_parameter_type (self)), _tmp10_ = vala_symbol_get_full_name (vala_symbol_get_parent_symbol ((ValaSymbol*) self))));
			_tmp11_ = (g_free (_tmp11_), NULL);
			_tmp10_ = (g_free (_tmp10_), NULL);
			_tmp9_ = (g_free (_tmp9_), NULL);
		}
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	vala_semantic_analyzer_set_current_symbol (analyzer, old_symbol);
	return (_tmp12_ = !vala_code_node_get_error ((ValaCodeNode*) self), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), _tmp12_);
}


ValaDataType* vala_formal_parameter_get_parameter_type (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_data_type;
}


void vala_formal_parameter_set_parameter_type (ValaFormalParameter* self, ValaDataType* value) {
	ValaDataType* _tmp2_;
	ValaDataType* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_data_type = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL)), _tmp2_);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_data_type, (ValaCodeNode*) self);
}


ValaParameterDirection vala_formal_parameter_get_direction (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_direction;
}


void vala_formal_parameter_set_direction (ValaFormalParameter* self, ValaParameterDirection value) {
	g_return_if_fail (self != NULL);
	self->priv->_direction = value;
}


gboolean vala_formal_parameter_get_ellipsis (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_ellipsis;
}


void vala_formal_parameter_set_ellipsis (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_ellipsis = value;
}


gboolean vala_formal_parameter_get_params_array (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_params_array;
}


void vala_formal_parameter_set_params_array (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_params_array = value;
}


ValaExpression* vala_formal_parameter_get_default_expression (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_default_expression;
}


void vala_formal_parameter_set_default_expression (ValaFormalParameter* self, ValaExpression* value) {
	ValaExpression* _tmp2_;
	ValaExpression* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_default_expression = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->priv->_default_expression == NULL) ? NULL : (self->priv->_default_expression = (vala_code_node_unref (self->priv->_default_expression), NULL)), _tmp2_);
}


gboolean vala_formal_parameter_get_no_array_length (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_no_array_length;
}


void vala_formal_parameter_set_no_array_length (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_no_array_length = value;
}


gboolean vala_formal_parameter_get_array_null_terminated (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_array_null_terminated;
}


void vala_formal_parameter_set_array_null_terminated (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_array_null_terminated = value;
}


double vala_formal_parameter_get_cparameter_position (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, 0.0);
	return self->priv->_cparameter_position;
}


void vala_formal_parameter_set_cparameter_position (ValaFormalParameter* self, double value) {
	g_return_if_fail (self != NULL);
	self->priv->_cparameter_position = value;
}


double vala_formal_parameter_get_carray_length_parameter_position (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, 0.0);
	return self->priv->_carray_length_parameter_position;
}


void vala_formal_parameter_set_carray_length_parameter_position (ValaFormalParameter* self, double value) {
	g_return_if_fail (self != NULL);
	self->priv->_carray_length_parameter_position = value;
}


double vala_formal_parameter_get_cdelegate_target_parameter_position (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, 0.0);
	return self->priv->_cdelegate_target_parameter_position;
}


void vala_formal_parameter_set_cdelegate_target_parameter_position (ValaFormalParameter* self, double value) {
	g_return_if_fail (self != NULL);
	self->priv->_cdelegate_target_parameter_position = value;
}


gboolean vala_formal_parameter_get_async_only (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_async_only;
}


void vala_formal_parameter_set_async_only (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_async_only = value;
}


const char* vala_formal_parameter_get_ctype (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_ctype;
}


void vala_formal_parameter_set_ctype (ValaFormalParameter* self, const char* value) {
	char* _tmp2_;
	const char* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_ctype = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), self->priv->_ctype = (g_free (self->priv->_ctype), NULL), _tmp2_);
}


static void vala_formal_parameter_class_init (ValaFormalParameterClass * klass) {
	vala_formal_parameter_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_formal_parameter_finalize;
	g_type_class_add_private (klass, sizeof (ValaFormalParameterPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_formal_parameter_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_formal_parameter_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_formal_parameter_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_formal_parameter_real_check;
}


static void vala_formal_parameter_instance_init (ValaFormalParameter * self) {
	self->priv = VALA_FORMAL_PARAMETER_GET_PRIVATE (self);
	self->priv->_direction = VALA_PARAMETER_DIRECTION_IN;
}


static void vala_formal_parameter_finalize (ValaCodeNode* obj) {
	ValaFormalParameter * self;
	self = VALA_FORMAL_PARAMETER (obj);
	(self->priv->_default_expression == NULL) ? NULL : (self->priv->_default_expression = (vala_code_node_unref (self->priv->_default_expression), NULL));
	self->priv->_ctype = (g_free (self->priv->_ctype), NULL);
	(self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL));
	VALA_CODE_NODE_CLASS (vala_formal_parameter_parent_class)->finalize (obj);
}


GType vala_formal_parameter_get_type (void) {
	static GType vala_formal_parameter_type_id = 0;
	if (vala_formal_parameter_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaFormalParameterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_formal_parameter_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaFormalParameter), 0, (GInstanceInitFunc) vala_formal_parameter_instance_init, NULL };
		vala_formal_parameter_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaFormalParameter", &g_define_type_info, 0);
	}
	return vala_formal_parameter_type_id;
}



GType vala_parameter_direction_get_type (void) {
	static GType vala_parameter_direction_type_id = 0;
	if (G_UNLIKELY (vala_parameter_direction_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_PARAMETER_DIRECTION_IN, "VALA_PARAMETER_DIRECTION_IN", "in"}, {VALA_PARAMETER_DIRECTION_OUT, "VALA_PARAMETER_DIRECTION_OUT", "out"}, {VALA_PARAMETER_DIRECTION_REF, "VALA_PARAMETER_DIRECTION_REF", "ref"}, {0, NULL, NULL}};
		vala_parameter_direction_type_id = g_enum_register_static ("ValaParameterDirection", values);
	}
	return vala_parameter_direction_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




