/* valaccodecommaexpression.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <gee.h>
#include <stdlib.h>
#include <string.h>


#define VALA_TYPE_CCODE_NODE (vala_ccode_node_get_type ())
#define VALA_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNode))
#define VALA_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))
#define VALA_IS_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_NODE))
#define VALA_IS_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_NODE))
#define VALA_CCODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))

typedef struct _ValaCCodeNode ValaCCodeNode;
typedef struct _ValaCCodeNodeClass ValaCCodeNodeClass;
typedef struct _ValaCCodeNodePrivate ValaCCodeNodePrivate;

#define VALA_TYPE_CCODE_WRITER (vala_ccode_writer_get_type ())
#define VALA_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriter))
#define VALA_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))
#define VALA_IS_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_WRITER))
#define VALA_IS_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_WRITER))
#define VALA_CCODE_WRITER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))

typedef struct _ValaCCodeWriter ValaCCodeWriter;
typedef struct _ValaCCodeWriterClass ValaCCodeWriterClass;

#define VALA_TYPE_CCODE_EXPRESSION (vala_ccode_expression_get_type ())
#define VALA_CCODE_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpression))
#define VALA_CCODE_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpressionClass))
#define VALA_IS_CCODE_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_EXPRESSION))
#define VALA_IS_CCODE_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_EXPRESSION))
#define VALA_CCODE_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpressionClass))

typedef struct _ValaCCodeExpression ValaCCodeExpression;
typedef struct _ValaCCodeExpressionClass ValaCCodeExpressionClass;
typedef struct _ValaCCodeExpressionPrivate ValaCCodeExpressionPrivate;

#define VALA_TYPE_CCODE_COMMA_EXPRESSION (vala_ccode_comma_expression_get_type ())
#define VALA_CCODE_COMMA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_COMMA_EXPRESSION, ValaCCodeCommaExpression))
#define VALA_CCODE_COMMA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_COMMA_EXPRESSION, ValaCCodeCommaExpressionClass))
#define VALA_IS_CCODE_COMMA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_COMMA_EXPRESSION))
#define VALA_IS_CCODE_COMMA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_COMMA_EXPRESSION))
#define VALA_CCODE_COMMA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_COMMA_EXPRESSION, ValaCCodeCommaExpressionClass))

typedef struct _ValaCCodeCommaExpression ValaCCodeCommaExpression;
typedef struct _ValaCCodeCommaExpressionClass ValaCCodeCommaExpressionClass;
typedef struct _ValaCCodeCommaExpressionPrivate ValaCCodeCommaExpressionPrivate;

/**
 * Represents a node in the C code tree.
 */
struct _ValaCCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeNodePrivate * priv;
};

struct _ValaCCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeNode *self);
	void (*write) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_declaration) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_combined) (ValaCCodeNode* self, ValaCCodeWriter* writer);
};

/**
 * Represents an expression node in the C code tree.
 */
struct _ValaCCodeExpression {
	ValaCCodeNode parent_instance;
	ValaCCodeExpressionPrivate * priv;
};

struct _ValaCCodeExpressionClass {
	ValaCCodeNodeClass parent_class;
	void (*write_inner) (ValaCCodeExpression* self, ValaCCodeWriter* writer);
};

/**
 * Represents a comma separated expression list in the C code.
 */
struct _ValaCCodeCommaExpression {
	ValaCCodeExpression parent_instance;
	ValaCCodeCommaExpressionPrivate * priv;
};

struct _ValaCCodeCommaExpressionClass {
	ValaCCodeExpressionClass parent_class;
};

struct _ValaCCodeCommaExpressionPrivate {
	GeeList* inner;
};


static gpointer vala_ccode_comma_expression_parent_class = NULL;

gpointer vala_ccode_node_ref (gpointer instance);
void vala_ccode_node_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_node (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_node (const GValue* value);
GType vala_ccode_node_get_type (void);
gpointer vala_ccode_writer_ref (gpointer instance);
void vala_ccode_writer_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_writer (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_writer (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_writer (const GValue* value);
GType vala_ccode_writer_get_type (void);
GType vala_ccode_expression_get_type (void);
GType vala_ccode_comma_expression_get_type (void);
#define VALA_CCODE_COMMA_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_COMMA_EXPRESSION, ValaCCodeCommaExpressionPrivate))
enum  {
	VALA_CCODE_COMMA_EXPRESSION_DUMMY_PROPERTY
};
void vala_ccode_comma_expression_append_expression (ValaCCodeCommaExpression* self, ValaCCodeExpression* expr);
GeeList* vala_ccode_comma_expression_get_inner (ValaCCodeCommaExpression* self);
void vala_ccode_writer_write_string (ValaCCodeWriter* self, const char* s);
void vala_ccode_node_write (ValaCCodeNode* self, ValaCCodeWriter* writer);
static void vala_ccode_comma_expression_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer);
ValaCCodeCommaExpression* vala_ccode_comma_expression_new (void);
ValaCCodeCommaExpression* vala_ccode_comma_expression_construct (GType object_type);
ValaCCodeCommaExpression* vala_ccode_comma_expression_new (void);
static void vala_ccode_comma_expression_finalize (ValaCCodeNode* obj);



/**
 * Appends the specified expression to the expression list.
 *
 * @param expr a C code expression
 */
void vala_ccode_comma_expression_append_expression (ValaCCodeCommaExpression* self, ValaCCodeExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
	gee_collection_add ((GeeCollection*) self->priv->inner, expr);
}


GeeList* vala_ccode_comma_expression_get_inner (ValaCCodeCommaExpression* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, self->priv->inner);
	return result;
}


static void vala_ccode_comma_expression_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer) {
	ValaCCodeCommaExpression * self;
	gboolean first;
	self = (ValaCCodeCommaExpression*) base;
	g_return_if_fail (writer != NULL);
	first = TRUE;
	vala_ccode_writer_write_string (writer, "(");
	{
		GeeIterator* _expr_it;
		_expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->inner);
		while (TRUE) {
			ValaCCodeExpression* expr;
			if (!gee_iterator_next (_expr_it)) {
				break;
			}
			expr = (ValaCCodeExpression*) gee_iterator_get (_expr_it);
			if (!first) {
				vala_ccode_writer_write_string (writer, ", ");
			} else {
				first = FALSE;
			}
			vala_ccode_node_write ((ValaCCodeNode*) expr, writer);
			(expr == NULL) ? NULL : (expr = (vala_ccode_node_unref (expr), NULL));
		}
		(_expr_it == NULL) ? NULL : (_expr_it = (gee_collection_object_unref (_expr_it), NULL));
	}
	vala_ccode_writer_write_string (writer, ")");
}


/**
 * Represents a comma separated expression list in the C code.
 */
ValaCCodeCommaExpression* vala_ccode_comma_expression_construct (GType object_type) {
	ValaCCodeCommaExpression* self;
	self = (ValaCCodeCommaExpression*) g_type_create_instance (object_type);
	return self;
}


ValaCCodeCommaExpression* vala_ccode_comma_expression_new (void) {
	return vala_ccode_comma_expression_construct (VALA_TYPE_CCODE_COMMA_EXPRESSION);
}


static void vala_ccode_comma_expression_class_init (ValaCCodeCommaExpressionClass * klass) {
	vala_ccode_comma_expression_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_NODE_CLASS (klass)->finalize = vala_ccode_comma_expression_finalize;
	g_type_class_add_private (klass, sizeof (ValaCCodeCommaExpressionPrivate));
	VALA_CCODE_NODE_CLASS (klass)->write = vala_ccode_comma_expression_real_write;
}


static void vala_ccode_comma_expression_instance_init (ValaCCodeCommaExpression * self) {
	self->priv = VALA_CCODE_COMMA_EXPRESSION_GET_PRIVATE (self);
	self->priv->inner = (GeeList*) gee_array_list_new (VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_equal);
}


static void vala_ccode_comma_expression_finalize (ValaCCodeNode* obj) {
	ValaCCodeCommaExpression * self;
	self = VALA_CCODE_COMMA_EXPRESSION (obj);
	(self->priv->inner == NULL) ? NULL : (self->priv->inner = (gee_collection_object_unref (self->priv->inner), NULL));
	VALA_CCODE_NODE_CLASS (vala_ccode_comma_expression_parent_class)->finalize (obj);
}


GType vala_ccode_comma_expression_get_type (void) {
	static GType vala_ccode_comma_expression_type_id = 0;
	if (vala_ccode_comma_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeCommaExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_comma_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeCommaExpression), 0, (GInstanceInitFunc) vala_ccode_comma_expression_instance_init, NULL };
		vala_ccode_comma_expression_type_id = g_type_register_static (VALA_TYPE_CCODE_EXPRESSION, "ValaCCodeCommaExpression", &g_define_type_info, 0);
	}
	return vala_ccode_comma_expression_type_id;
}




