/* valaccodebinaryexpression.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>


#define VALA_TYPE_CCODE_NODE (vala_ccode_node_get_type ())
#define VALA_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNode))
#define VALA_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))
#define VALA_IS_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_NODE))
#define VALA_IS_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_NODE))
#define VALA_CCODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))

typedef struct _ValaCCodeNode ValaCCodeNode;
typedef struct _ValaCCodeNodeClass ValaCCodeNodeClass;
typedef struct _ValaCCodeNodePrivate ValaCCodeNodePrivate;

#define VALA_TYPE_CCODE_WRITER (vala_ccode_writer_get_type ())
#define VALA_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriter))
#define VALA_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))
#define VALA_IS_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_WRITER))
#define VALA_IS_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_WRITER))
#define VALA_CCODE_WRITER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))

typedef struct _ValaCCodeWriter ValaCCodeWriter;
typedef struct _ValaCCodeWriterClass ValaCCodeWriterClass;

#define VALA_TYPE_CCODE_EXPRESSION (vala_ccode_expression_get_type ())
#define VALA_CCODE_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpression))
#define VALA_CCODE_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpressionClass))
#define VALA_IS_CCODE_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_EXPRESSION))
#define VALA_IS_CCODE_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_EXPRESSION))
#define VALA_CCODE_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpressionClass))

typedef struct _ValaCCodeExpression ValaCCodeExpression;
typedef struct _ValaCCodeExpressionClass ValaCCodeExpressionClass;
typedef struct _ValaCCodeExpressionPrivate ValaCCodeExpressionPrivate;

#define VALA_TYPE_CCODE_BINARY_EXPRESSION (vala_ccode_binary_expression_get_type ())
#define VALA_CCODE_BINARY_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_BINARY_EXPRESSION, ValaCCodeBinaryExpression))
#define VALA_CCODE_BINARY_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_BINARY_EXPRESSION, ValaCCodeBinaryExpressionClass))
#define VALA_IS_CCODE_BINARY_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_BINARY_EXPRESSION))
#define VALA_IS_CCODE_BINARY_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_BINARY_EXPRESSION))
#define VALA_CCODE_BINARY_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_BINARY_EXPRESSION, ValaCCodeBinaryExpressionClass))

typedef struct _ValaCCodeBinaryExpression ValaCCodeBinaryExpression;
typedef struct _ValaCCodeBinaryExpressionClass ValaCCodeBinaryExpressionClass;
typedef struct _ValaCCodeBinaryExpressionPrivate ValaCCodeBinaryExpressionPrivate;

#define VALA_TYPE_CCODE_BINARY_OPERATOR (vala_ccode_binary_operator_get_type ())

/**
 * Represents a node in the C code tree.
 */
struct _ValaCCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeNodePrivate * priv;
};

struct _ValaCCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeNode *self);
	void (*write) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_declaration) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_combined) (ValaCCodeNode* self, ValaCCodeWriter* writer);
};

/**
 * Represents an expression node in the C code tree.
 */
struct _ValaCCodeExpression {
	ValaCCodeNode parent_instance;
	ValaCCodeExpressionPrivate * priv;
};

struct _ValaCCodeExpressionClass {
	ValaCCodeNodeClass parent_class;
	void (*write_inner) (ValaCCodeExpression* self, ValaCCodeWriter* writer);
};

/**
 * Represents an expression with two operands in C code.
 */
struct _ValaCCodeBinaryExpression {
	ValaCCodeExpression parent_instance;
	ValaCCodeBinaryExpressionPrivate * priv;
};

struct _ValaCCodeBinaryExpressionClass {
	ValaCCodeExpressionClass parent_class;
};

typedef enum  {
	VALA_CCODE_BINARY_OPERATOR_PLUS,
	VALA_CCODE_BINARY_OPERATOR_MINUS,
	VALA_CCODE_BINARY_OPERATOR_MUL,
	VALA_CCODE_BINARY_OPERATOR_DIV,
	VALA_CCODE_BINARY_OPERATOR_MOD,
	VALA_CCODE_BINARY_OPERATOR_SHIFT_LEFT,
	VALA_CCODE_BINARY_OPERATOR_SHIFT_RIGHT,
	VALA_CCODE_BINARY_OPERATOR_LESS_THAN,
	VALA_CCODE_BINARY_OPERATOR_GREATER_THAN,
	VALA_CCODE_BINARY_OPERATOR_LESS_THAN_OR_EQUAL,
	VALA_CCODE_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL,
	VALA_CCODE_BINARY_OPERATOR_EQUALITY,
	VALA_CCODE_BINARY_OPERATOR_INEQUALITY,
	VALA_CCODE_BINARY_OPERATOR_BITWISE_AND,
	VALA_CCODE_BINARY_OPERATOR_BITWISE_OR,
	VALA_CCODE_BINARY_OPERATOR_BITWISE_XOR,
	VALA_CCODE_BINARY_OPERATOR_AND,
	VALA_CCODE_BINARY_OPERATOR_OR
} ValaCCodeBinaryOperator;

struct _ValaCCodeBinaryExpressionPrivate {
	ValaCCodeBinaryOperator _operator;
	ValaCCodeExpression* _left;
	ValaCCodeExpression* _right;
};


static gpointer vala_ccode_binary_expression_parent_class = NULL;

gpointer vala_ccode_node_ref (gpointer instance);
void vala_ccode_node_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_node (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_node (const GValue* value);
GType vala_ccode_node_get_type (void);
gpointer vala_ccode_writer_ref (gpointer instance);
void vala_ccode_writer_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_writer (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_writer (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_writer (const GValue* value);
GType vala_ccode_writer_get_type (void);
GType vala_ccode_expression_get_type (void);
GType vala_ccode_binary_expression_get_type (void);
GType vala_ccode_binary_operator_get_type (void);
#define VALA_CCODE_BINARY_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_BINARY_EXPRESSION, ValaCCodeBinaryExpressionPrivate))
enum  {
	VALA_CCODE_BINARY_EXPRESSION_DUMMY_PROPERTY
};
void vala_ccode_binary_expression_set_operator (ValaCCodeBinaryExpression* self, ValaCCodeBinaryOperator value);
void vala_ccode_binary_expression_set_left (ValaCCodeBinaryExpression* self, ValaCCodeExpression* value);
void vala_ccode_binary_expression_set_right (ValaCCodeBinaryExpression* self, ValaCCodeExpression* value);
ValaCCodeBinaryExpression* vala_ccode_binary_expression_new (ValaCCodeBinaryOperator op, ValaCCodeExpression* l, ValaCCodeExpression* r);
ValaCCodeBinaryExpression* vala_ccode_binary_expression_construct (GType object_type, ValaCCodeBinaryOperator op, ValaCCodeExpression* l, ValaCCodeExpression* r);
ValaCCodeExpression* vala_ccode_binary_expression_get_left (ValaCCodeBinaryExpression* self);
void vala_ccode_expression_write_inner (ValaCCodeExpression* self, ValaCCodeWriter* writer);
void vala_ccode_writer_write_string (ValaCCodeWriter* self, const char* s);
ValaCCodeBinaryOperator vala_ccode_binary_expression_get_operator (ValaCCodeBinaryExpression* self);
ValaCCodeExpression* vala_ccode_binary_expression_get_right (ValaCCodeBinaryExpression* self);
static void vala_ccode_binary_expression_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer);
void vala_ccode_node_write (ValaCCodeNode* self, ValaCCodeWriter* writer);
static void vala_ccode_binary_expression_real_write_inner (ValaCCodeExpression* base, ValaCCodeWriter* writer);
static void vala_ccode_binary_expression_finalize (ValaCCodeNode* obj);



ValaCCodeBinaryExpression* vala_ccode_binary_expression_construct (GType object_type, ValaCCodeBinaryOperator op, ValaCCodeExpression* l, ValaCCodeExpression* r) {
	ValaCCodeBinaryExpression* self;
	g_return_val_if_fail (l != NULL, NULL);
	g_return_val_if_fail (r != NULL, NULL);
	self = (ValaCCodeBinaryExpression*) g_type_create_instance (object_type);
	vala_ccode_binary_expression_set_operator (self, op);
	vala_ccode_binary_expression_set_left (self, l);
	vala_ccode_binary_expression_set_right (self, r);
	return self;
}


ValaCCodeBinaryExpression* vala_ccode_binary_expression_new (ValaCCodeBinaryOperator op, ValaCCodeExpression* l, ValaCCodeExpression* r) {
	return vala_ccode_binary_expression_construct (VALA_TYPE_CCODE_BINARY_EXPRESSION, op, l, r);
}


static void vala_ccode_binary_expression_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer) {
	ValaCCodeBinaryExpression * self;
	self = (ValaCCodeBinaryExpression*) base;
	g_return_if_fail (writer != NULL);
	vala_ccode_expression_write_inner (self->priv->_left, writer);
	vala_ccode_writer_write_string (writer, " ");
	if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_PLUS) {
		vala_ccode_writer_write_string (writer, "+");
	} else {
		if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_MINUS) {
			vala_ccode_writer_write_string (writer, "-");
		} else {
			if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_MUL) {
				vala_ccode_writer_write_string (writer, "*");
			} else {
				if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_DIV) {
					vala_ccode_writer_write_string (writer, "/");
				} else {
					if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_MOD) {
						vala_ccode_writer_write_string (writer, "%");
					} else {
						if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_SHIFT_LEFT) {
							vala_ccode_writer_write_string (writer, "<<");
						} else {
							if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_SHIFT_RIGHT) {
								vala_ccode_writer_write_string (writer, ">>");
							} else {
								if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_LESS_THAN) {
									vala_ccode_writer_write_string (writer, "<");
								} else {
									if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_GREATER_THAN) {
										vala_ccode_writer_write_string (writer, ">");
									} else {
										if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_LESS_THAN_OR_EQUAL) {
											vala_ccode_writer_write_string (writer, "<=");
										} else {
											if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL) {
												vala_ccode_writer_write_string (writer, ">=");
											} else {
												if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_EQUALITY) {
													vala_ccode_writer_write_string (writer, "==");
												} else {
													if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_INEQUALITY) {
														vala_ccode_writer_write_string (writer, "!=");
													} else {
														if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_BITWISE_AND) {
															vala_ccode_writer_write_string (writer, "&");
														} else {
															if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_BITWISE_OR) {
																vala_ccode_writer_write_string (writer, "|");
															} else {
																if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_BITWISE_XOR) {
																	vala_ccode_writer_write_string (writer, "^");
																} else {
																	if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_AND) {
																		vala_ccode_writer_write_string (writer, "&&");
																	} else {
																		if (self->priv->_operator == VALA_CCODE_BINARY_OPERATOR_OR) {
																			vala_ccode_writer_write_string (writer, "||");
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	vala_ccode_writer_write_string (writer, " ");
	vala_ccode_expression_write_inner (self->priv->_right, writer);
}


static void vala_ccode_binary_expression_real_write_inner (ValaCCodeExpression* base, ValaCCodeWriter* writer) {
	ValaCCodeBinaryExpression * self;
	self = (ValaCCodeBinaryExpression*) base;
	g_return_if_fail (writer != NULL);
	vala_ccode_writer_write_string (writer, "(");
	vala_ccode_node_write ((ValaCCodeNode*) self, writer);
	vala_ccode_writer_write_string (writer, ")");
}


ValaCCodeBinaryOperator vala_ccode_binary_expression_get_operator (ValaCCodeBinaryExpression* self) {
	ValaCCodeBinaryOperator result;
	g_return_val_if_fail (self != NULL, 0);
	result = self->priv->_operator;
	return result;
}


void vala_ccode_binary_expression_set_operator (ValaCCodeBinaryExpression* self, ValaCCodeBinaryOperator value) {
	g_return_if_fail (self != NULL);
	self->priv->_operator = value;
}


ValaCCodeExpression* vala_ccode_binary_expression_get_left (ValaCCodeBinaryExpression* self) {
	ValaCCodeExpression* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_left;
	return result;
}


void vala_ccode_binary_expression_set_left (ValaCCodeBinaryExpression* self, ValaCCodeExpression* value) {
	ValaCCodeExpression* _tmp1_;
	ValaCCodeExpression* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_left = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : vala_ccode_node_ref (_tmp0_)), (self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_ccode_node_unref (self->priv->_left), NULL)), _tmp1_);
}


ValaCCodeExpression* vala_ccode_binary_expression_get_right (ValaCCodeBinaryExpression* self) {
	ValaCCodeExpression* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_right;
	return result;
}


void vala_ccode_binary_expression_set_right (ValaCCodeBinaryExpression* self, ValaCCodeExpression* value) {
	ValaCCodeExpression* _tmp1_;
	ValaCCodeExpression* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_right = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : vala_ccode_node_ref (_tmp0_)), (self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_ccode_node_unref (self->priv->_right), NULL)), _tmp1_);
}


static void vala_ccode_binary_expression_class_init (ValaCCodeBinaryExpressionClass * klass) {
	vala_ccode_binary_expression_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_NODE_CLASS (klass)->finalize = vala_ccode_binary_expression_finalize;
	g_type_class_add_private (klass, sizeof (ValaCCodeBinaryExpressionPrivate));
	VALA_CCODE_NODE_CLASS (klass)->write = vala_ccode_binary_expression_real_write;
	VALA_CCODE_EXPRESSION_CLASS (klass)->write_inner = vala_ccode_binary_expression_real_write_inner;
}


static void vala_ccode_binary_expression_instance_init (ValaCCodeBinaryExpression * self) {
	self->priv = VALA_CCODE_BINARY_EXPRESSION_GET_PRIVATE (self);
}


static void vala_ccode_binary_expression_finalize (ValaCCodeNode* obj) {
	ValaCCodeBinaryExpression * self;
	self = VALA_CCODE_BINARY_EXPRESSION (obj);
	(self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_ccode_node_unref (self->priv->_left), NULL));
	(self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_ccode_node_unref (self->priv->_right), NULL));
	VALA_CCODE_NODE_CLASS (vala_ccode_binary_expression_parent_class)->finalize (obj);
}


GType vala_ccode_binary_expression_get_type (void) {
	static GType vala_ccode_binary_expression_type_id = 0;
	if (vala_ccode_binary_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeBinaryExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_binary_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeBinaryExpression), 0, (GInstanceInitFunc) vala_ccode_binary_expression_instance_init, NULL };
		vala_ccode_binary_expression_type_id = g_type_register_static (VALA_TYPE_CCODE_EXPRESSION, "ValaCCodeBinaryExpression", &g_define_type_info, 0);
	}
	return vala_ccode_binary_expression_type_id;
}



GType vala_ccode_binary_operator_get_type (void) {
	static GType vala_ccode_binary_operator_type_id = 0;
	if (G_UNLIKELY (vala_ccode_binary_operator_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_CCODE_BINARY_OPERATOR_PLUS, "VALA_CCODE_BINARY_OPERATOR_PLUS", "plus"}, {VALA_CCODE_BINARY_OPERATOR_MINUS, "VALA_CCODE_BINARY_OPERATOR_MINUS", "minus"}, {VALA_CCODE_BINARY_OPERATOR_MUL, "VALA_CCODE_BINARY_OPERATOR_MUL", "mul"}, {VALA_CCODE_BINARY_OPERATOR_DIV, "VALA_CCODE_BINARY_OPERATOR_DIV", "div"}, {VALA_CCODE_BINARY_OPERATOR_MOD, "VALA_CCODE_BINARY_OPERATOR_MOD", "mod"}, {VALA_CCODE_BINARY_OPERATOR_SHIFT_LEFT, "VALA_CCODE_BINARY_OPERATOR_SHIFT_LEFT", "shift-left"}, {VALA_CCODE_BINARY_OPERATOR_SHIFT_RIGHT, "VALA_CCODE_BINARY_OPERATOR_SHIFT_RIGHT", "shift-right"}, {VALA_CCODE_BINARY_OPERATOR_LESS_THAN, "VALA_CCODE_BINARY_OPERATOR_LESS_THAN", "less-than"}, {VALA_CCODE_BINARY_OPERATOR_GREATER_THAN, "VALA_CCODE_BINARY_OPERATOR_GREATER_THAN", "greater-than"}, {VALA_CCODE_BINARY_OPERATOR_LESS_THAN_OR_EQUAL, "VALA_CCODE_BINARY_OPERATOR_LESS_THAN_OR_EQUAL", "less-than-or-equal"}, {VALA_CCODE_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL, "VALA_CCODE_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL", "greater-than-or-equal"}, {VALA_CCODE_BINARY_OPERATOR_EQUALITY, "VALA_CCODE_BINARY_OPERATOR_EQUALITY", "equality"}, {VALA_CCODE_BINARY_OPERATOR_INEQUALITY, "VALA_CCODE_BINARY_OPERATOR_INEQUALITY", "inequality"}, {VALA_CCODE_BINARY_OPERATOR_BITWISE_AND, "VALA_CCODE_BINARY_OPERATOR_BITWISE_AND", "bitwise-and"}, {VALA_CCODE_BINARY_OPERATOR_BITWISE_OR, "VALA_CCODE_BINARY_OPERATOR_BITWISE_OR", "bitwise-or"}, {VALA_CCODE_BINARY_OPERATOR_BITWISE_XOR, "VALA_CCODE_BINARY_OPERATOR_BITWISE_XOR", "bitwise-xor"}, {VALA_CCODE_BINARY_OPERATOR_AND, "VALA_CCODE_BINARY_OPERATOR_AND", "and"}, {VALA_CCODE_BINARY_OPERATOR_OR, "VALA_CCODE_BINARY_OPERATOR_OR", "or"}, {0, NULL, NULL}};
		vala_ccode_binary_operator_type_id = g_enum_register_static ("ValaCCodeBinaryOperator", values);
	}
	return vala_ccode_binary_operator_type_id;
}




