// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "server.h"
#include <pwd.h>
#include <grp.h>
#include <math.h>
#include <dlfcn.h>

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

#define	LIB_VERSION	"/" VERPATH "/"

KeyLocal::KeyLocal() :
Keydata("/bayonne/localize")
{
	static Keydata::Define keydefs[] = {
	{"primarycurrency", "dollar"},
	{"primarychange", "cent"},
	{"convertcurrency", "1.00"},
	{NULL, NULL}};

	load(keydefs);
}

#ifdef	XML_SCRIPTS

KeyProxy::KeyProxy() :
Keydata("/bayonne/proxy")
{
	static Keydata::Define keydefs[] = {
	{"timeout", "0"},
	{NULL, NULL}};

	load("~bayonne/proxy");
	load(keydefs);
}

const char *KeyProxy::getHTTPServer(void)
{
	const char *cp = getLast("httpserver");
	if(!cp)
		cp = getLast("server");
	return cp;
}

tpport_t KeyProxy::getHTTPPort(void)
{
	const char *cp;

	if(!getHTTPServer())
		return 0;

	cp = getLast("httpport");
	if(!cp)
		cp = getLast("port");
	if(cp)
		return atoi(cp);

	return 80;
}

timeout_t KeyProxy::getTimeout(void)
{
	return 1000l * atoi(getLast("timeout"));
}	

#endif

KeyImports::KeyImports() :
Keydata("/bayonne/imports")
{
	static Keydata::Define defimports[] = {
	{"perl", ""},
	{"python", ""},
	{NULL, NULL}};

	load("~bayonne/imports");
	load(defimports);
}

KeyPaths::KeyPaths() :
Keydata("/bayonne/paths")
{
	static Keydata::Define defpaths[] = {
//	{"libpath", "/usr/lib/bayonne"},
//	{"libexec", "/usr/libexec"},
//	{"tgipath", "/usr/libexec/bayonne:/bin:/usr/bin"},
//	{"datafiles", "/var/bayonne"},
//	{"scripts", "/usr/share/aascripts"},
//	{"prompts", "/usr/share/aaprompts"},
	{"spool", "/var/spool/bayonne"},
	{"runfiles", "/var/run/bayonne"},
	{"precache", "/var/bayonne/cache"},
	{"logpath", "/var/log/bayonne"},
	{"sox", "/usr/bin/sox"},
	{NULL, NULL}};

	load("~bayonne/paths");
	load(defpaths);
}

void KeyPaths::setPrefix(char *prefix)
{
	char tmpfs[65];
	char *pp = prefix + strlen(prefix);
	const char *cp;

	cp = getLast("libpath");
	if(!cp)
	{
		strcpy(pp, "/../lib/bayonne");
		setValue("libpath", prefix);
	}

	cp = getLast("libexec");
	if(!cp)
	{
		strcpy(pp, "/../libexec");
		setValue("libexec", prefix);
	}
	cp = getLast("tgipath");
	if(!cp)
	{
		strcpy(pp, "/../libexec/bayonne:/bin:/usr/bin");
		setValue("tgipath", prefix);
	}

	cp = getLast("scripts");
	if(!cp)
	{
		strcpy(pp, "/../share/aascripts");
		setValue("scripts", prefix);
	}

	cp = getLast("prompts");
	if(!cp)
	{
		strcpy(pp, "/../share/aaprompts");
		setValue("prompts", prefix);
	}

	cp = getLast("tmpfs");
	if(!cp)
		cp = "/dev/shm";

	if(!isDir(cp))
		cp = "/tmp";

	snprintf(tmpfs, sizeof(tmpfs), "%s/.bayonne", cp);
	setValue("tmpfs", tmpfs);
	setValue("tmp", "/tmp/.bayonne");

	cp = getenv("CONFIG_KEYDATA");
	if(!cp)
		cp = ETC_PREFIX;
	setValue("etc", cp);

	*pp = 0;
}
	
KeyHandlers::KeyHandlers() :
Keydata("/bayonne/handlers")
{
	static Keydata::Define defpaths[] = {
	{"tts", "festival"},
	{"url", "wget"},
	{"timeout", "90"},
	{NULL, NULL}};

	load("~bayonne/handlers");
	load(defpaths);
}

KeyServer::KeyServer() :
Keydata("/bayonne/server")
{
	static Keydata::Define defkeys[] = {
	{"user", "bayonne"},
	{"group", "bayonne"},
	{"default", "default"},
	{"nodes", "1"},
	{"token", "&"},
	{"password", "fts"},
	{NULL, NULL}};

	struct passwd *pwd;
	struct group *grp;
	char namebuf[128];
	char prefix[256];
	char *cp;

	strcpy(prefix, getenv("HOME"));
	strcat(prefix, "/.bayonne");
	altdir = strdup(prefix);
	phrdir = "~.bayonne/phrases/";

	load("~bayonne/server");

	if(!getLast("node"))
	{
		gethostname(namebuf, sizeof(namebuf) - 1);
		cp = strchr(namebuf, '.');
		if(cp)
			*cp = 0;
		setValue("node", namebuf);
	}
	load(defkeys);

	uid = getuid();
	gid = getgid();

	pwd = getpwnam(getLast("user"));
	if(pwd)
		gid = pwd->pw_gid;

	if(pwd && !uid)
	{
		uid = pwd->pw_uid;
		strcpy(prefix, pwd->pw_dir);
		strcat(prefix, "/apps");
		altdir = strdup(prefix);
		phrdir = "~phrases/";
	}
	if(!pwd && !uid)
	{
		uid = 0xffff;
		gid = 0xffff;
	}

#ifdef	USER_HOSTING
	if(getuid())
		grp = NULL;
	else
	{
		grp = getgrnam(getLast("group"));
		if(!grp)
			endgrent();
	}
	if(grp)
	{
		while(*grp->gr_mem)
		{
			cp = *(grp->gr_mem++);
			pwd = getpwnam(cp);
			if(pwd)
			{
				snprintf(prefix, sizeof(prefix), "%s/.bayonne", pwd->pw_dir);
				if(isDir(prefix))
					keyusers.setValue(cp, prefix);
			}
		}
	}
	endpwent();
	if(grp)
		endgrent();
#endif
}

void KeyServer::loadGroups(bool test)
{
	char *group, *tok;
	char buffer[256];

	new TrunkGroup();
	slog(Slog::levelDebug) << "loading default trunk group" << endl;
	
	group = (char *)TrunkGroup::first->getLast("groups");
	if(!group || test)
		return;

	strcpy(buffer, group);
	group = strtok_r(buffer, " \t\n", &tok);
	while(group)
	{
		slog(Slog::levelDebug) << "loading " << group << " trunk group" << endl;
		new TrunkGroup(group);
		group = strtok_r(NULL, " \t\n", &tok);
	}
}

void KeyServer::setUid(void)
{
	setuid(uid);
	uid = getuid();
}

void KeyServer::setGid(void)
{
	setgid(gid);
	gid = getgid();
}

KeyThreads::KeyThreads() :
Keydata("/bayonne/threads")
{
	static Keydata::Define defpaths[] = {
	{"audit", "0"},
	{"audio", "0"},
	{"priority", "0"},
	{"gateways", "0,1"},
	{"services", "1,1"},
	{"database", "0"},
	{"network", "0"},
#ifdef	XML_SCRIPTS
	{"xml", "0"},
#endif
	{"managers", "0"},
	{"scheduler", "0"},
	{"gui", "0"},
	{"rtp", "0"},
	{"resetdelay", "18"},
	{"stepdelay", "36"},
	{"stepinterval", "18"},
	{"interval", "15"},
	{"refresh", "5"},
	{"policy", "other"},
	{"pages", "0"},
#ifdef	__FreeBSD__
	{"stack", "8"},
#endif
	{NULL, NULL}};

	const char *cp = getLast("pri");
	
	if(cp)
		setValue("priority", cp);

	load(defpaths);
}

int KeyThreads::getPolicy()
{
	const char *cp = getLast("policy");
#ifdef	SCHED_RR
	if(!stricmp(cp, "rr"))
		return SCHED_RR;
#endif
#ifdef	SCHED_FIFO
	if(!stricmp(cp, "fifo"))
		return SCHED_FIFO;
#endif
	return 0;
}

#ifdef	NODE_SERVICES

KeyNetwork::KeyNetwork() :
Keydata("/bayonne/network")
{
	static Keydata::Define defkeys[] = {
	{"refresh", "5"},
	{"live", "25"},
	{"elect", "120"},
	{"expire", "300"},
	{"database", "127.0.0.1:7002"},
	{"address", "127.0.0.1:7001"},
	{"broadcast", "127.255.255.255"},
	{"monitor", "127.0.0.1:7070"},
	{NULL, NULL}};

	load("~bayonne/network");

	load(defkeys);
}

InetAddress KeyNetwork::getAddress(void)
{
	char buffer[65];
	char *cp;

	strcpy(buffer, getLast("address"));
	cp = strchr(buffer, ':');
	if(*cp)
		*cp = 0;
	return InetAddress(buffer);
}

InetAddress KeyNetwork::getMonitorAddress(void)
{
	char buffer[65];
	char *cp;

	strcpy(buffer, getLast("monitor"));
	cp = strchr(buffer, ':');
	if(*cp)
		*cp = 0;
	return InetAddress(buffer);
}

InetHostAddress KeyNetwork::getDBHost(void)
{
	char buffer[65];
	char *cp;

	strcpy(buffer, getLast("database"));
	cp = strchr(buffer, ':');
	if(*cp)
		*cp = 0;
	return InetHostAddress(buffer);
}

InetHostAddress KeyNetwork::getBroadcast(void)
{
	return InetHostAddress(getLast("broadcast"));
}

tpport_t KeyNetwork::getPort(void)
{
	char buffer[65];
	char *cp;

	strcpy(buffer, getLast("address"));
	cp = strchr(buffer, ':');
	if(cp)
		return atoi(++cp);
	return 0;
}

tpport_t KeyNetwork::getMonitorPort(void)
{
	char buffer[65];
	char *cp;

	strcpy(buffer, getLast("monitor"));
	cp = strchr(buffer, ':');
	if(cp)
		return atoi(++cp);
	return 0;
}

tpport_t KeyNetwork::getDBPort(void)
{
	char buffer[65];
	char *cp;

	strcpy(buffer, getLast("database"));
	cp = strchr(buffer, ':');
	if(cp)
		return atoi(++cp);
	return 0;
}

#endif

KeyMemory::KeyMemory() :
Keydata("/bayonne/memory")
{
	static Keydata::Define defpaths[] = {
	{"symbols", "64"},
	{"page", "1024"},
	{NULL, NULL}};

	load(defpaths);
}

size_t KeyThreads::getStack(void)
{
	const char *cp = getLast("stack");

	if(cp)
		return atoi(cp) * 1024;
	
	return 0;
}

int KeyThreads::priResolver(void)
{
	const char *cp = getLast("resolver");
	if(cp)
		return atoi(cp);

	return 0;
}

int KeyThreads::getResolver(void)
{
	char buf[32];
	int cnt;
	const char *cp = getLast("resolver");
	
	if(!cp)
		return 0;

	strcpy(buf, cp);
	cp = strchr(buf, ',');
	if(cp)
	{
		++cp;
		while(*cp == ' ' || *cp == '\t')
			++cp;
		return atoi(cp);
	}
	return 15;
}

int KeyThreads::getServices(void)
{
	char buf[32];
	int cnt;

	strcpy(buf, getLast("services"));
	char *cp = strchr(buf, ',');
	if(cp)
	{
		++cp;
		while(*cp == ' ' || *cp == '\t')
			++cp;
		cnt = atoi(cp);
		if(cnt > 0)
			return cnt;
	}
	return 1;
}

int KeyThreads::getGateways(void)
{
	char buf[32];

	strcpy(buf, getLast("gateways"));
	char *cp = strchr(buf, ',');
	if(cp)
	{
		++cp;
		if(*cp == ' ' || *cp == '\t')
			++cp;
		return atoi(cp);
	}
	else
		return 1;
}

Plugins::Plugins() :
Keydata("/bayonne/plugins")
{
	if(getLast("driver"))
		return;

#if defined(HAVE_LINUX_TELEPHONY_H)
	setValue("driver", "phonedev");
#elif defined(have_montecarlo_h) || defined(HAVE_MONTECARLO_H)
	setValue("driver", "pika");
#else
	setValue("driver", "dummy");
#endif
	pidcount = 0;	
}

Plugins::~Plugins()
{
//	dynunload();
	while(pidcount)
		kill(pids[--pidcount], SIGTERM);
}

char *Plugins::getDriverName(void)
{
	static char name[33];

	const char *drv = getLast("driver");
	char *cp = strrchr(drv, '/');
	if(cp)
		++cp;
	else
		cp = (char *)drv;
	strncpy(name, cp, 32);
	name[33] = 0;
	cp = strrchr(name, '.');
	if(!cp)
		return name;

	if(!strcmp(cp, ".ivr"))
		*cp = 0;
	return name;
}

void Plugins::loadManagers(void)
{
	char path[256];
	char list[412];
	char *argv[3];
	char *cp;

	cp = (char *)getLast("managers");
	if(!cp)
		return;

	strcpy(list, cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(*cp == '/')
			path[0] = 0;
		else
		{
			strcpy(path, keypaths.getLibexec());
			strcat(path, "/");
		}
		strcat(path, "bayonne.");
		strcat(path, cp);
		pids[pidcount] = fork();
		if(pids[pidcount] == 0)
		{
			argv[0] = path;
			argv[1] = (char *)keypaths.getRunfiles();
			argv[2] = NULL; 
			execv(path, argv);
			slog(Slog::levelError) << "libexec: " << path << ": unable to run" << endl;
			exit(-1);
		}
		else if(pids[pidcount] > 0)
			++pidcount;
		else
			slog(Slog::levelError) << "libexec: " << path << ": unable to fork" << endl;
	}
}

void Plugins::loadAuditing(void)
{
	char path[256];
	char list[412];
	char *cp;

	cp = (char *)getLast("auditing");
	if(!cp)
		return;

	strcpy(list, cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(*cp == '/')
			path[0] = 0;
		else
		{
			strcpy(path, keypaths.getLibpath());
			strcat(path, LIB_VERSION);
		}
		strcat(path, cp);
		if(*cp != '/')
			strcat(path, ".aud");

		new DSO(path);
		cp = strtok(NULL, " \t\n;,");
	}
}

void Plugins::loadPreload(void)
{
	char path[256];
	char list[512];
	char *cp;

	cp = (char *)getLast("preload");
	if(!cp)
		return;

	if(!stricmp(cp, "none"))
		return;

	if(!stricmp(cp, "no"))
		return;

	strcpy(list, cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		Script::use(cp);
		cp = strtok(NULL, " \t\n;,");
	}
}

void Plugins::loadModules(void)
{
	char path[256];
	char list[512];
	char *cp;

	cp = (char *)getLast("modules");

	if(!cp)
		return;

	if(!stricmp(cp, "no"))
		return;

	if(!stricmp(cp, "none"))
		return;

	if(!stricmp(cp, "*") || !stricmp(cp, "all"))
	{
		snprintf(path, sizeof(path), "%s/%s",
			keypaths.getLibpath(), LIB_VERSION);
		loader(path, ".mod");
		return;
	}

	strcpy(list, cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(*cp == '/')
			path[0] = 0;
		else
			snprintf(path, sizeof(path), "%s/%s",
				keypaths.getLibpath(), LIB_VERSION);

		strcat(path, cp);
		if(*cp != '/')
			strcat(path, ".mod");
		new DSO(path);
		cp = strtok(NULL, " \t\n;,");
	}
}

#ifdef	XML_SCRIPTS
void Plugins::loadXML(void)
{
	char path[256];
	char list[512];
	char *cp;

	cp = (char *)getLast("xml");
	if(!cp)
		return;

	if(!stricmp(cp, "no") || !stricmp(cp, "none"))
		return;

	if(!stricmp(cp, "*") || !stricmp(cp, "all"))
	{
		snprintf(path, sizeof(path), "%s/%s",
			keypaths.getLibpath(), LIB_VERSION);
		loader(path, ".xml");
		return;
	}

	strcpy(list, cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(*cp == '/')
			path[0] = 0;
		else
			snprintf(path, sizeof(path), "%s/%s",
				keypaths.getLibpath(), LIB_VERSION);

		strcat(path, cp);
		if(*cp != '/')
			strcat(path, ".xml");
		new DSO(path);
		cp = strtok(NULL, " \t\n;,");
	}
}
#endif

void Plugins::loadTGI(void)
{
	char path[256];
	char list[512];
	char *cp;

	cp = (char *)getLast("tgi");
	if(!cp)
		return;

	strcpy(list, cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(*cp == '/')
			path[0] = 0;
		else
		{
			strcpy(path, keypaths.getLibpath());
			strcat(path, LIB_VERSION);
		}
		strcat(path, cp);
		if(*cp != '/')
			strcat(path, ".tgi");
		dlopen(path, RTLD_LAZY | RTLD_GLOBAL);
//		new DSO(path);
		cp = strtok(NULL, " \t\n;,");
	}
}


void Plugins::loadTranslators(const char *lcp)
{
	char path[256];
	char list[512];
	char *cp;

	if(!lcp)
		lcp = getLast("languages");

	if(!lcp)
		return;

	strcpy(list, lcp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(*cp == '/')
			path[0] = 0;
		else
		{
			strcpy(path, keypaths.getLibpath());
			strcat(path, LIB_VERSION);
		}

		strcat(path, cp);
		if(*cp != '/')
			strcat(path, ".tts");
		new DSO(path);
		cp = strtok(NULL, " \t\n;,");
	}
}

void Plugins::loadExtensions(void)
{
	char path[256];
	strcpy(path, keypaths.getLibpath());
	strcat(path, LIB_VERSION);
	strcat(path, "bayonne.ext");
	if(canAccess(path))
		new DSO(path);
}

void Plugins::loadDebug(void)
{
	char path[256];
	char *d = (char *)getLast("debug");

	if(!d)
	{
		new Debug();
		return;
	}

	if(*d != '/')
	{
		strcpy(path, keypaths.getLibpath());
		strcat(path, LIB_VERSION);
		strcat(path, d);
		strcat(path, ".dbg");			
		d = path;
	}
	new DSO(d);
	if(!debug)
	{
		slog(Slog::levelNotice) << "no debug handler installed" << endl;
		new Debug();
	}
}

void Plugins::loadMonitor(void)
{
	char path[256];
	char *d = (char *)getLast("monitors");

	if(!d)
	{
		new Monitor();
		return;
	}

	if(*d != '/')
	{
		strcpy(path, keypaths.getLibpath());
		strcat(path, LIB_VERSION);
		strcat(path, d);
		strcat(path, ".mon");
		d = path;
	}
	new DSO(d);
	if(!monitor)
	{
		slog(Slog::levelNotice) << "no monitor handler installed" << endl;
		new Monitor();
	}
}

void Plugins::loadSwitch(void)
{
	char path[256];
	const char *sim = getLast("switch");

	if(!sim)
		return;

	if(*sim)
	{
		snprintf(path, sizeof(path), "%s/%s/%s.sim",
			keypaths.getLibpath(), LIB_VERSION, sim);
		new DSO(path);
	}
}

DSO *Plugins::loadDriver(void)
{
	char path[256];
	char *drv = (char *)getLast("driver");
	
	if(*drv != '/')
	{
		strcpy(path, keypaths.getLibpath());
		strcat(path, LIB_VERSION);
		strcat(path, drv);
		strcat(path, ".ivr");
		drv = path;
	}
	
	return new DSO(drv);
}

KeyTones::KeyTones() :
Keydata("/bayonne/tones")
{
	char *tones = (char *)getLast("tones");
	int v1, v2, v3;
	char *tone;

	tones = strtok(tones, " \t\n");
	while(tones)
	{
		v1 = v2 = v3 = 0;
		tone = (char *)getLast(tones);
		if(!tone)
		{
			tones = strtok(NULL, " \t\n");
			continue;
		}
		sscanf(tone, "%d %d %d", &v1, &v2, &v3);
		if(v3)
			new phTone(tones, v3, v1, v2);
		else
			new phTone(tones, v2, v1);
		tones = strtok(NULL, " \t\n");
	}
}

TrunkGroup *TrunkGroup::first = NULL;

TrunkGroup::TrunkGroup(char *name) :
Keydata("/bayonne/trunks"), CallStat()
{
	Keydata::Define keys[] = {
		{"answer", "1"},
		{"accept", "1"},
		{"hangup", "100"},
		{"siezetime", "12"},
		{"ringtime", "7"},
		{"flash", "200"},
		{"dialtone", "2400"},
		{"dialspeed", "160"},
		{"volume", "80"},
		{"callerid", "1500"},
		{"pickup", "500"},
		{"requests", "hangup"},
		{"select", "last"},
		{"threashold", "0"},
		{"ready", "1000"},
		{"idletime", "600"},
		{"analysis", "16"},
		{"dialmode", "dtmf"},
		{NULL, NULL}};

	int count = driver->getTrunkCount();
	int i;
	char *cp;
	char namebuf[65];
	
	schedule[0] = 0;
	load("~bayonne/trunks");

	if(name)
	{
		strcpy(namebuf, "/bayonne/");
		strcat(namebuf, name);
		strcat(namebuf, "-trunks");
		load(namebuf);
		*namebuf = '~';
		load(namebuf);
		cp = (char *)getLast("trunks");
		if(cp)
			cp = strtok(cp, " ,;\t\n");
		while(cp)
		{
			i = atoi(cp);
			if(i >= 0 && i < count)
				driver->groups[i] = this;
			cp = strtok(NULL, " ,;\t\n");
		}

		cp = (char *)getLast("spans");
		if(cp)
			cp = strtok(cp, " ,;\t\n");
		while(cp)
		{
			i = atoi(cp);
			if(i > 0 && i < MAX_SPANS)
				driver->spans[i] = this;
			cp = strtok(NULL, " ,;\t\n");
		}

                cp = (char *)getLast("cards");
                if(cp)
                        cp = strtok(cp, " ,;\t\n");
                while(cp)
                {
                        i = atoi(cp);
                        if(i > 0 && i < MAX_CARDS)
                                driver->cards[i] = this;
                        cp = strtok(NULL, " ,;\t\n");
                }
	}
	else
	{
		for(i = 0; i < count; ++i) {
			driver->groups[i] = this;
		}
		name = "*";
	}

	setValue("name", name);

	load(keys);
	next = NULL;
	reqfirst = reqlast = NULL;
	polFirst = NULL;
	members = 0;

	if(!first)
		first = this;
	else
	{
		next = first;
		while(next->next)
			next = next->next;
		next->next = this;
	}
	next = NULL;
}	

bool TrunkGroup::getAccept(void)
{
	const char *cp = getLast("accept");
	if(!cp)
		return false;

	switch(*cp)
	{
	case '0':
	case 'f':
	case 'F':
	case 'n':
	case 'N':
		return false;
	}
	return true;
}

bool TrunkGroup::getDetect(void)
{
	const char *cp = getLast("detect");
	if(!cp)
		return false;

	switch(*cp)
	{
	case '0':
	case 'f':
	case 'F':
	case 'n':
	case 'N':
		return false;
	}

	return true;
}

seltype_t TrunkGroup::getSelect(void)
{
	const char *cp = getLast("select");

	if(!stricmp(cp, "last"))
		return SELECT_LAST;

	return SELECT_FIRST;
}

Request *TrunkGroup::getRequest(void)
{
	Request *req = NULL;
	Policy *pol;
	Module *reply = Module::reqFirst;
	Trunk *trunk;
	TrunkEvent event;

	enterMutex();
	pol = polFirst;
	while(pol)
	{
		req = pol->hiPriority();
		if(req)
		{
			leaveMutex();
			return req;
		}
		pol = pol->next;
	}
	while(reqfirst)
	{
		reply = Module::reqFirst;
		if(reqfirst->isExpired())
		{
			slog(Slog::levelWarning) << "request: " << getName() << "(" << reqfirst->id << ") expired...";
			trunk = driver->getTrunkId(reqfirst->parent);
			if(trunk)
			{
				event.id = TRUNK_CHILD_FAIL;
				trunk->postEvent(&event);
			}
			while(reply)
			{
				reply->expires(reqfirst);
				reply = reply->reqNext;
			}
			delete reqfirst;
			continue;
		}
	}
	if(reqfirst)
	{
		reply = Module::reqFirst;
		while(reply)
		{
			reply->running(reqfirst);
			reply = reply->reqNext;
		}
		req = reqfirst;
		req->detach();
	}
	pol = polFirst;
	while(!req && pol)
	{
		req = pol->loPriority();
		pol = pol->next;
	}
	leaveMutex();
	return req;
}

const char *TrunkGroup::getRedirect(const char *redirect, char *buf)
{
	const char *cp = getLast(redirect);
	if(cp)
	{
		strcpy(buf, cp);
		return buf;
	}
	return getSchedule(buf);
}

const char *TrunkGroup::getSchedule(char *buf)
{
	const char *scr = getLast("script");	// check for .conf override
	if(scr)
	{
		strcpy(buf, scr);
		return buf;
	}

	scheduler.readLock();
	if(schedule[0])
		strcpy(buf, schedule);
	else if(this == first)
		strcpy(buf, keyserver.getDefault());
	else
		strcpy(buf, getName());
	scheduler.unlock();
	return buf;
}

const char *TrunkGroup::getNumber(void)
{
	const char *num = getLast("number");

	if(num)
		return num;

	return "UNKNOWN";
}

void TrunkGroup::setSchedule(const char *str)
{
	if(!str)
		str = "";

	scheduler.writeLock();
	strncpy(schedule, str, sizeof(schedule) - 1);
	schedule[sizeof(schedule) - 1] = 0;
	scheduler.unlock();
}

phTone *phTone::first = NULL;

int phTone::ulaw[256] = { 
	0,0,1,1,2,2,2,2,3,3,3,3,3,3,3,3,
        4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
        5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
        5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7};

unsigned char phTone::linear2ulaw(int sample)
{
	int sign, exponent, mantissa, retval;

	sign = (sample >> 8) & 0x80;
	if(sign != 0) sample = -sample;
	sample += 0x84;
	exponent = ulaw[(sample >> 7) & 0xff];
	mantissa = (sample >> (exponent + 3)) & 0x0f;
	retval = ~(sign | (exponent << 4) | mantissa);
	if(!retval)
		retval = 0x02;

	return retval;
}

phTone::phTone(const char *n, timeout_t dur, unsigned f) 
{
	int i, sign, exponent, mantissa;
	int sample;

	next = first;
	first = this;

        duration = dur;
        samples = new unsigned char[dur * 8];
        freq1 = f;
        freq2 = 0;
        strncpy(name, n, 32);
        name[32] = 0; 

	double freq = (f * M_PI * 2) / 8000.;
	double pos = 0;

	for(i = 0; i < dur * 8; ++i)
	{
		sample = (int)(sin(pos) * 20000.0);
		pos += freq;
		samples[i] = linear2ulaw(sample);
	}
}

phTone::phTone(const char *n, timeout_t dur, unsigned f1, unsigned f2) 
{
	int i, sign, exponent, mantissa;
	int sample;

	next = first;
	first = this;

        duration = dur;
        samples = new unsigned char[dur * 8];
        freq1 = f1;
        freq2 = f2;
        strncpy(name, n, 32);
        name[32] = 0; 

	double fa1 = (f1 * M_PI * 2) / 8000.;
	double fa2 = (f2 * M_PI * 2) / 8000.;
	double pos1 = 0, pos2 = 0;

	for(i = 0; i < dur * 8; ++i)
	{
		sample = (int)((sin(pos1) + sin(pos2)) * 10000.0);
		pos1 += fa1;
		pos2 += fa2;
		samples[i] = linear2ulaw(sample);
	}
}

 
phTone::~phTone()
{
	if(samples)
		delete samples;
}


void phTone::clear(void)
{
	if(samples)
		delete samples;

	samples = NULL;
}

phTone *getphTone(const char *name) 
{
	phTone *tone = phTone::first;
	while(tone)
	{
		if(!stricmp(tone->name, name))
			break;
		tone = tone->next;
	}
	return tone;
}

TrunkGroup *getGroup(const char *name)
{
	TrunkGroup *group = TrunkGroup::first;

	if(!name)
		return group;

	while(group)
	{
		if(!stricmp(name, group->getName()))
			return group;
		group = group->next;
	}
	return NULL;
}		

Keydata application("/bayonne/application");
#ifdef	USER_HOSTING
Keydata keyusers;
#endif
KeyServer keyserver;
KeyPaths keypaths;
KeyLocal keylocal;
KeyHandlers keyhandlers;
#ifdef	NODE_SERVICES
KeyNetwork keynetwork;
#endif
KeyTones keytones;
KeyThreads keythreads;
KeyMemory keymemory;
#ifdef	XML_SCRIPTS
KeyProxy keyproxy;
#endif
Plugins plugins;
Scheduler scheduler;
#ifdef	NODE_SERVICES
Network network;
#endif

ScriptSymbol Trunk::globals(keymemory.getSymbolSize(), keymemory.getPageSize());

#ifdef	CCXX_NAMESPACES
};
#endif
