# Copyright (C) 2005 David Sugar, Tycho Softworks    
#
# This file is free software; as a special exception the author gives
# unlimited permission to copy and/or distribute it, with or without
# modifications, as long as this notice is preserved.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY, to the extent permitted by law; without even the
# implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
#
# Bayonne perl libexec interface module.

package Bayonne::Libexec;

require 5.004;

# disable buffering
$|=1;

sub new {
	my ($class, %args) = @_;
	my $self = {};
	my ($buffer);

	# default voice
	$self->{voice} = "";

	# digits buffer
	$self->{digits} = "";

	# query buffer
	$self->{query} = "";

	# audio position
	$self->{position} = "00:00:00.000";

	# last header reply id number
	$self->{reply} = 0;

	# last result code from a transaction.
	$self->{result} = 0;

	# exit code if terminated by server, 0 if active
	$self->{exitcode} = 0;

	# version of our interface
	$self->{version} = "4.0";

	$self->{tsession} = $ENV{'PORT_TSESSION'} if $ENV{'PORT_TSESSION'};
	
	if(!$self->{tsession}) {
		$self->{exitcode} = 1;
	        bless $self, ref $class || $class;
        	return $self;   
	}

	# issue libexec HEAD request to get headers...

	print STDOUT "$self->{tsession} HEAD\n";
	while(<STDIN>)
	{
		$buffer = $_;
		if($buffer > 900) {
			$self->{reply} = $buffer - 0;
			$self->{exitcode} = $buffer - 900;
			last;
		}
		if($buffer > 0) {
			$self->{reply} = $buffer - 0;
			next;
		}
		if($buffer eq "\n") {
			last;
		}
		$_ =~ /(.*?)[:][ ](.*\n)/;
		my($keyword, $value) = ($1, $2);
		$value =~ s/\s+$//;
		if($keyword eq "DIGITS") {
			$self->{digits} = $value;
		}
		$self->{head}{$keyword}=$value;
	}

	# issue libexec ARGS request to get command arguments...

	print STDOUT "$self->{tsession} ARGS\n";
	while(<STDIN>)
	{
		$buffer = $_;
		if($buffer > 900) {
			$self->{reply} = $buffer - 0;
			$self->{exitcode} = $buffer - 900;
			last;
		}
		if($buffer > 0) {
			$self->{reply} = $buffer - 0;
			next;
		}
		if($buffer eq "\n") {
			last;
		}
		$_ =~ /(.*?)[:][ ](.*\n)/;
		my($keyword, $value) = ($1, $2);
		$value =~ s/\s+$//;
		$self->{args}{$keyword}=$value;
	}
	
	bless $self, ref $class || $class;
	return $self;
};

# hangup

sub hangup($) {
	my($self) = @_;
	my($tsid) = $self->{tsession};
	if($tsid) {
		print STDOUT "$tsid hangup\n";
		$self->{tsession} = undef;
	}
}

# disconnect (server resumes...)

sub resume($$) {
	my($self,$code) = @_;
	my($tsid) = $self->{tsession};

	if($tsid) {
		print STDOUT "$tsid exit $code\n";
		$self->{tsession} = undef;	
	}
}
	
# replay audio

sub replay {
	my $self = shift;    
        my $file = shift;
        my $offset = undef;  

	if(!$file) {
		return "255";
	}

	if($offset) {
		return $self->command("replay $file $offset");
	} else {
		return $self->command("replay $file");
	}
}

# record audio

sub record {
	my $self = shift;
	my $file = shift;
	my $timeout = shift;
	my $silence = undef;
	my $offset = undef;

	if(!$file) {
		return "255";
	}

	if($timeout) {
		$silence = shift;
		if($silence) {
			$offset = shift;
		}
	}

	if(!$timeout) {
		$timeout = 60;
	}

	if(!$silence) {
		$silence = 0;
	}

	if($offset) {
		return $self->command("record $file $timeout $silence $offset");
	} else {
		return $self->command("record $file $timeout $silence");
	}
}	

# set voice to use, undef to reset...

sub voice {
	my $self = shift;
	my $voice = shift;

	$self->{voice} = $voice;
}

# process input line

sub input($$$) {
	my ($self, $count, $timeout) = @_;

	if(!$count) {
		$count = 1;
	}

	if(!$timeout) {
		$timeout = 0;
	}

	my $result = $self->command("READ $timeout $count");
	if($result != 0) {
		return "";
	}

	return $self->{digits};
}

# clear pending input

sub clear($) {
	my($self) = @_;
	return $self->command("FLUSH");
}

# wait for a key event

sub wait($$) {
	my ($self, $timeout) = @_;

	if(!$timeout) {
		$timeout = 0;
	}
	my $result = $self->command("WAIT $timeout");
	if($result == 3) {
		return true;
	}
	return false;
}

# process single key input

sub inkey($$) {
	my ($self, $timeout) = @_;

	if(!$timeout) {
		$timeout = 0;
	}

	my $result = $self->command("READ $timeout");
	if($result != 0) {
		return "";
	}
	return substr($self->{digits}, 0, 1);
}

# send results back to server.

sub result($$) {
	my($self, $buf) = @_;
	$buf =~ s/\%/\%\%/g;
        $buf =~ s/(.)/ord $1 < 32 ?
                sprintf "%%%s", chr(ord($1) + 64) : sprintf "%s",$1/eg; 

	return $self->command("result $buf");
}

# get symbol value

sub get($$) {
	my($self, $buf) = @_;
	$self->command("get $value");
	return $self->{query};
}

# set symbol value

sub set($$) {
	my($self, $value) = @_;
	return $self->command("set $value");
}

sub add($$) {
        my($self, $value) = @_;
        return $self->command("add $value");
} 

# size a symbol

sub size($$) {
	my($self, $buf) = @_;
	my($size) = $buf - 0;
	return $self->command("new $size");
}
	
# build prompt

sub prompt($$) {
        my($self, $buf) = @_;
	my($voice) = $self->{voice};

	if(!$voice) {
		$voice = "prompt";
	}

	if($voice eq "") {
		$voice = "prompt";
	}

        return $self->command("$voice $buf");
}

# issue a libexec command and parse the transaction results.

sub command($$) {
	my($self,$buf) = @_;
        my($hid) = 0;
        my($result) = 255;      # no result value   
	my($tsession) = $self->{tsession};
	my($exitcode) = $self->{exitcode};
	if(!$tsession || $exitcode > 0) {
		return -$exitcode;
	}
        $buf =~ s/\%/\%\%/g;
        $buf =~ s/(.)/ord $1 < 32 ?
                sprintf "%%%s", chr(ord($1) + 64) : sprintf "%s",$1/eg; 

	$self->{query} = "";
	print STDOUT "$tsession $buf\n";

	while(<STDIN>)
        {
                $buffer = $_;
                if($buffer > 900) {
                        $self->{reply} = $buffer - 0;
                        $self->{exitcode} = $buffer - 900;
			$result = -$self->{exitcode};
                        last;      
                }      	
                if($buffer > 0) {
                        $self->{reply} = $buffer - 0;
			$hid = $buffer - 0;
                        next;
                } 
		if($buffer eq "\n") {
                        last;
                }
		if($hid != 100 && $hid != 400) {
			next;
		}
                $_ =~ /(.*?)[:][ ](.*\n)/;
                my($keyword, $value) = ($1, $2);
                $value =~ s/\s+$//; 
		$keyword = lc($keyword);
		if($hid == 400) {
			$keyword = "query";
		}
		if($keyword eq "result") {
			$result = $value - 0;
		}
		$self->{$keyword}=$value;
	}
	return $result;  
}	

# generic print function, works whether in TGI or direct execute mode

sub print($$) {
	my($self,$buf) = @_;
  	$buf =~ s/\%/\%\%/g; 
  	$buf =~ s/(.)/ord $1 < 32 ? 
		sprintf "%%%s", chr(ord($1) + 64) : sprintf "%s",$1/eg; 
	if($self->{tsession}) {
		print STDERR $buf;
	} else {
		print STDOUT $buf;
	}
}
1;

