/* 
 *    Programmed By: Mohammed Isam Mohammed [mohammed_isam1984@yahoo.com]
 *    Copyright 2013, 2014, 2015, 2016, 2017, 2018 (c)
 * 
 *    file: file.c
 *    This file is part of Prime.
 *
 *    Prime is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    Prime is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with Prime.  If not, see <http://www.gnu.org/licenses/>.
 */    

#include <string.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <pwd.h>
#include "defs.h"
#include "file.h"

FILE *file;

void purgeLogFile(FILE *logfile, char *logfilename)
{
    fseek(logfile, 0, SEEK_END);
    int i = ftell(logfile);
    if(i > 0)
    {
        show_readme(logfilename, " Error log ", GNU_DOS_LEVEL);
    }
    fclose(logfile);
}


/***********************************************
 * This function exports the tree of directory
 * passed as argv[1] into file passed as argv[2]
 * from the command line.
 * *********************************************/
void exportTreeFromCommandLine(char *d, char *f) 
{
    if(!(file = fopen(f, "w"))) 
    {
        fprintf(stderr, "Error opening export file: '%s'.\nAborting.\n", f);
        exit(1);
    }

    fprintf(stdout, "Reading directory tree. Please wait..\n");
    fprintf(file, "\nDirectory tree of '%s/':", d);
    scanThisDir(d, 0, 0);
  
    fprintf(stdout, "Finished writing directory tree of '%s' to '%s'.\n", d, f);
    fclose(file);
    return;
}

/***********************************************
 * This function exports the tree under the
 * current working directory into an external
 * file as input by the user.
 * *********************************************/
void exportTree(int showFileNameDialogBox) 
{
    if(showFileNameDialogBox == YES) 
    {	//show dialog box
        char *f = getUserInput("The directory tree of the current dir will be exported.\n"
                               "Enter export file name:", "Export");
        if(f == NULL) return;
        if(!(file = fopen(f, "w"))) 
        {
            msgBoxH("Error opening export file. Aborting.", OK, ERROR);
            free(f);
            refreshWindows();
            return;
        }
        free(f);
    }
  
    fprintf(file, "Directory tree of '%s/':", cwd);
    scanThisDir(cwd, 0, 1);
    fclose(file);

    int x = chdir(cwd);
    if(x == -1 && showFileNameDialogBox)
    {
        showErrorMsgBox("Error changing directory: ", strerror(errno));
    }
}

/***********************************************
 * This function scans the directory passed to
 * it as tmp[] for exporting. Not to be called
 * directly. Instead, called from exportTree().
 * *********************************************/
void scanThisDir(char *tmp, int level, int showProgress) 
{
    static int nf = 0;
    static int nd = 0;	//static: total num of files and dirs
    int n;
    struct dirent **eps;	//structure used in exportTree() function
    struct stat statbuf;
    int h = SCREEN_H/2;
    int w = SCREEN_W/2;
  
    //variables used in displaying progress message to the user
    static char exportingString[] = "Exporting %d of %d";
  
    n = scandir(tmp, &eps, one, alphasort);
    if(n >= 0) 
    {
        int z = chdir(tmp);
        if(z == -1)
        {
            fprintf(file, "\n%*s (Error: %s)", level, " ", strerror(errno));
            return;
        }
        int cnt;
        for(cnt = 0; cnt < n; ++cnt) 
        {
            char *d = eps[cnt]->d_name;
            if(level == 0 && showProgress) 
            {
                drawBox(h-2, w-12, h+1, w+12, NULL, YES);
                fprintf(stdout, "\e[%d;%dH", h-1, w-11);
                fprintf(stdout, exportingString, cnt+1, n);
            }

            z = lstat(d, &statbuf);
            if(z == -1)
            {
                if(showProgress) showErrorMsgBox(strerror(errno), d);
                if(chdir("..")) return;
                return;
            }
      
            if(S_ISDIR(statbuf.st_mode)) 
            {
                if(strcmp(d, ".") == 0 || strcmp(d, "..") == 0)	//ignore "." & ".."
                    continue;
                fprintf(file, "\n%*s|---- %s", level, " ", eps[cnt]->d_name);
                
                nd++;
                scanThisDir(d, level+4, showProgress);
            } 
            else 
            {
                fprintf(file, "\n%*s|-[f] %s", level, " ", eps[cnt]->d_name);
                nf++;
            }
        }
        z = chdir("..");
        if(z == -1)
        {
            fprintf(file, "\n%*s (Error: %s)", level, " ", strerror(errno));
        }
    }
    else 
    {
        fprintf(file, " (Failed to open directory)");
    }

    if(level == 0) 
    {
        fprintf(file, "\n---------------------------------\n");
        fprintf(file, "Total dirs: %d, Total files: %d\n", nd, nf);
    }
}

char *file_open_location()
{
    char *dir = getUserInput("Enter directory path to open:", "Open Location");
    hideCursor();
    if(!dir) return NULL;
    scanDir(dir);
    refreshWindows();
    return dir;
}

void fileMenu_CreateDir()
{
    char *dir = getUserInput("Enter directory name to create:", "New Directory");
    hideCursor();
    if(!dir) return;
    struct stat st;
    if(stat(dir, &st) == -1) 
    {
        mkdir(dir, 0775);
    } 
    else 
    {
        msgBoxH("Directory already exists!", OK, ERROR);
    }
    free(dir);
    scanDir(cwd);
    refreshWindows();
}

void fileMenu_Open()
{
    file_open_location();
}

void fileMenu_ExportTree()
{
    exportTree(YES);
    setScreenColorsI(COLOR_WINDOW);
    scanDir(cwd);
}

void fileMenu_Print()
{
    msgBoxH("Oops! This function is currently not implemented.", OK, INFO);
    //showPrintDialogBox();
    refreshFileView();
    refreshDirView();
}

void fileMenu_Exit()
{
    int i = msgBoxH("Are you sure you want to exit?", YES | NO, CONFIRM);
    if(i == YES) 
    {
        exit_gracefully();
    }
}
