/* serverfuncs.c: Implements TCP/IP server technology directly in Meta-HTML. */

/*  Copyright (c) 1997 Brian J. Fox
    Author: Brian J. Fox (bfox@ai.mit.edu) Tue Sep 30 09:02:13 1997.  */
#include "language.h"

#if defined (__cplusplus)
extern "C"
{
#endif
static void pf_make_server (PFunArgs);

static PFunDesc func_table[] =
{
  { "SERVER::MAKE-SERVER", 0, 0, pf_make_server },
  { (char *)NULL,	0, 0, (PFunHandler *)NULL }
};

PACKAGE_INITIALIZER (initialize_server_functions)
DOC_SECTION (PROCESS-OPERATORS)

DEFUNX (pf_server::make_server, start-fun port ... &key hostname,
"Create a server process which will listen on <var port> for incoming
TCP/IP connections, and return a <i>server indentifier</i> which can
be used to crudely control that process.

When a connection is received on that port, the standard streams are
bound to the <Meta-HTML> variables <code>*standard-input*</code> and
<code>*standard-output*</code>, and <var start-fun> is invoked.  Only the
functions which have been defined before the invocation of
<code>server::make-server</code> are available to the server process.

A number of variables are bound at connection time.  These are:

<ul>
<li> <b>SERVER::REMOTE-ADDR</b><br>The IP address of the connecting machine.
<li> <b>SERVER::REMOTE-PORT</b><br>The port number which the remote machine connected to.
</ul>")

static void
pf_make_server (PFunArgs)
{
  int i = 0, high_sock = 0;
  pid_t child;
  int done = 0, servers = 0;
  Package *server_info = symbol_get_package ((Package *)NULL);

  /* Make the server process right away.  The child is a single process,
     perhaps listening on several ports. */
  child = fork ();

  /* In both the parent and the child, parse the arguments, and add the
     information to a package which describes what this server does. */
  while (!done)
    {
      char *func = mhtml_evaluate_string (get_positional_arg (vars, i++));
      char *port = mhtml_evaluate_string (get_positional_arg (vars, i++));

      if ((empty_string_p (func)) || (empty_string_p (port)))
	done = 1;
      else
	{
	  forms_set_tag_value_in_package (server_info, port, func);

	  if (child == (pid_t) 0)
	    {
	      int sock = -1, one = 1;
	      int sock_size = sizeof (struct sockaddr_in);
	      struct in_addr host_address;
	      struct sockaddr_in socket_address;
	      int portnum = atoi (port);

	      host_address.s_addr = htonl (INADDR_ANY);

	      /* Set up the socket address. */
	      memset (&socket_address, 0, sizeof (socket_address));
	      socket_address.sin_addr = host_address;
	      socket_address.sin_family = AF_INET;
	      socket_address.sin_port = htons ((short) portnum);

	      /* Set up the socket. */
	      sock = socket (AF_INET, SOCK_STREAM, 0);
	      if (sock < 0)
		{
		  /* fatal ("Cannot create socket!"); */
		}
	      else if (sock > high_sock)
		high_sock = sock;
		
	      setsockopt (sock, SOL_SOCKET, SO_REUSEADDR,
			  (char *)&one, sizeof (one));

	      /* Bind the socket to our address. */
	      if (bind (sock, (struct sockaddr *)&socket_address,
			sizeof (socket_address)) == -1)
		{
		  /* fatal ("Cannot bind socket to socket address"); */
		}

	      /* Start listening on this socket. */
	      listen (sock, 5);
	    }
	}
    }

  if (child != (pid_t) 0)
    {
      while (1)
	{
	  register int j;
	  int connection = -1;
	  pid_t child = (pid_t)0;
	  fd_set read_fds;
	  int ready;

	  FD_ZERO (&read_fds);
	  for (j = 0; j != high_sock; j++)
	    FD_SET (j, &read_fds);

	  ready = select (high_fd + 1, &read_fds, (fd_set *)0, (fd_set *)0,
			  (struct timeval *)NULL);

	  if (ready > -1)
	    {
	      for (j = 0; j < high_sock; j++)
		if (FD_ISSET (j, &read_fds))
		  {
		    int connection = accept
		      (j, (struct sockaddr *)&socket_address, &sock_size);

		    if (connection > -1)
		      {
			/* Fork a server process to handle this request. */
			pid_t instance = fork ();

			if (instance == 0)
			  {
			    struct sockaddr_in client;

			    if (getpeername
				(connection,
				 (struct sockaddr *)&client, &sock_size) != -1)
			      {
				struct hostent *client_info =
				  (struct hostent *)NULL;
				long addr =  client.sin_addr.s_addr;
				const char *x = (const char *)&addr;
				char addr_rep[24];
				BPRINTF_BUFFER exec_buffer;

				exec_buffer = bprintf_create_buffer ();

				sprintf (addr_rep, "%d.%d.%d.%d",
					 (unsigned char)x[0],
					 (unsigned char)x[1],
					 (unsigned char)x[2],
					 (unsigned char)x[3]);

				signal (SIGCHLD, SIG_DFL);
				signal (SIGHUP,  SIG_IGN);

				/* Bind variables in this instance. */
				mhtml_stdout_fileno = connection;
				mhtml_stdin_fileno = connection;
				mhtml_stderr_fileno = connection;

				pagefunc_set_variable
				  ("server::server-addr", addr_rep);

				sprintf (addr_rep, "%d", j);
				pagefunc_set_variable
				  ("server::server-port", addr_rep);

				bprintf (exec_buffer, "<%s>",
					 forms_get_tag_value_in_package
					 (server_info, addr_rep));
				mhtml_evaluate_string (exec_buffer->buffer);
				_exit (0);
			      }
			  }
		      }
		  }
	    }
	}
    }

  {
    char *result = package_to_alist (server_info);

    bprintf_insert (page, start, "%s", result);
    *newstart += strlen (result);
  }
}

#if defined (__cplusplus)
}
#endif
