/* wnt_slow.cpp  NT specific slow seeder implementation
 *	Copyright (C) 1999 Martin Grap
 *
 * This file is part of winseed.
 *
 * winseed is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * winseed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include<windows.h>
#include<string.h>
#include<util.h>
#include<md5_internal.h>
#include<stdio.h>
#include<wnt_slow.h>

static const UINT32 WINNT_SLOW_SEEDER_EXPECTED_SIZE = (128 * 1024);
static const DWORD LOG_ENTRY_SIZE = 4096;

nt_slow_seeder::nt_slow_seeder(netapi_loader *na)
	: seeder(WINNT_SLOW_SEEDER_EXPECTED_SIZE)
{
	netapi = na;
}

nt_slow_seeder::nt_slow_seeder(UINT32 internal_buffer_size, netapi_loader *na)
	: seeder(internal_buffer_size)
{
	netapi = na;
}

nt_slow_seeder::~nt_slow_seeder()
{
	if (netapi != NULL)
	{
		delete netapi;
	}
}

UINT32 nt_slow_seeder::get_expected_seed_size()
{
	return WINNT_SLOW_SEEDER_EXPECTED_SIZE;
}

UINT32 nt_slow_seeder::query_network_statistics(mem_writer &w, LPWSTR service)
{
	UINT32 result = PCP_SUCCESS;
	DWORD buffer_size;
	bool free_buffer = false;
	LPBYTE buffer;

	do 
	{
		/*
			Now get network statistics either for the server or the workstation
			service.
		*/

		if (netapi->NaNetStatisticsGet2(NULL, service, 0, 0, &buffer) != 0)
		{
			result = PCP_SEEDER_FAILED;
			break;
		}

		free_buffer = true;

		if (netapi->NaNetApiBufferSize(buffer, &buffer_size) != 0)
		{
			result = PCP_SEEDER_FAILED;
			break;
		}

		w.add(buffer, buffer_size);

		pcp_util::zero_memory(buffer, buffer_size);

	} while(0);


	if (free_buffer)
	{
		/*
			The documentation does not say what the value of
			the buffer pointer is if NetStatisticsGet2 fails.
			I assume it is NULL but the boolean flag free_buffer
			is used to make sure that we free the buffer only
			if NetStatisticsGet succeeded.
		*/
		
		netapi->NaNetApiBufferFree(buffer);
	}

	return result;
}


UINT32 nt_slow_seeder::query_performance_data(UINT8 *buffer, UINT32 &buffer_length)
{
	UINT32 result = PCP_SUCCESS;
	DWORD n_buffer = buffer_length;
	LONG error;

	if ((error = RegQueryValueEx(HKEY_PERFORMANCE_DATA, "Global", NULL, NULL, buffer, &n_buffer ))
		 != ERROR_SUCCESS)
	{
		buffer_length = 0;

		if (error == ERROR_MORE_DATA)
		{
			/*
				The current size of the performance data is bigger than
				the internal seed. Tell caller about it. A call with an
				increased buffer size will probably succeed.
			*/
			result = PCP_SEEDER_TOO_SMALL;
		}
		else
		{
			result = PCP_SEEDER_FAILED;
		}
	}
	else
	{
		buffer_length = n_buffer;

		/*
			Microsoft says:
			The first call opens the key; you do not need to explicitly open 
			the key first. However, be sure to use the RegCloseKey function to 
			close the handle to the key when you are finished obtaining 
			performance data. The user cannot install or remove a software 
			component while its performance data is in use.
		*/
		RegCloseKey(HKEY_PERFORMANCE_DATA);
	}

	return result;
}

UINT32 nt_slow_seeder::get_log_entries(PCP_CHAR *log_name, mem_writer& w)
{
	UINT32 result = PCP_SUCCESS;
	DWORD bytes_read, bytes_needed_next;
	UINT8 buffer[LOG_ENTRY_SIZE];
	HANDLE log;

	if ((log = OpenEventLog(NULL, log_name)) == NULL)
	{
		result = PCP_SEEDER_FAILED;
	}
	else
	{
		/*
			Read the latest log entries which do fit into 4 K. 
		*/
		if (ReadEventLog(log, 
			             EVENTLOG_BACKWARDS_READ | EVENTLOG_SEQUENTIAL_READ, 
						 0,
						 buffer, 
						 LOG_ENTRY_SIZE,
						 &bytes_read,
						 &bytes_needed_next) == 0)
		{
			result = PCP_SEEDER_FAILED;
		}
		else
		{
			w.add(buffer, bytes_read);
		}
		
		CloseEventLog(log);
	}

	pcp_util::zero_memory(buffer, LOG_ENTRY_SIZE);

	return result;
}


UINT32 nt_slow_seeder::get_seed(UINT8 *seed, UINT32& desired_length)
{
	UINT8 *Buffer = NULL;
	mem_writer w;
	UINT32 result = PCP_SUCCESS, bytes_to_copy, size_perf_data;
	
	CMD5Internal md5;
	

	do
	{
		if (seed == NULL)
		{
			result = PCP_NULL_POINTER;
			break;
		}


		if ((Buffer = new UINT8[get_internal_seed_size()]) == NULL)
		{
			result = PCP_SEEDER_NO_MEM;
			break;
		}

		size_perf_data = get_internal_seed_size();

		if ((result = query_performance_data(Buffer, size_perf_data)) != PCP_SUCCESS )
		{
			break;
		}

		w.reset(Buffer, get_internal_seed_size());
		w.set_pos(size_perf_data);

		if (w.is_full())
		{
			result = PCP_SEEDER_TOO_SMALL;
			break;
		}

		/*
			Get the latest 4 K of log entries from the application log. This does not
			contain heaps of entropy, but it can't do any damage either. The application
			log on my workstation does not contain many entries. For the system log
			the situation is a little bit better. At least one entry is made each day:
			The EventLog service logs that it has been started ... . One word about
			access permissions: Normal users should be able to read the  application 
			and the system log.
		*/
		if ((result = get_log_entries("Application", w)) != PCP_SUCCESS)
		{
			break;
		}

		if (w.is_full())
		{
			result = PCP_SEEDER_TOO_SMALL;
			break;
		}

		/*
			Get the latest 4 K of log entries from the system log. This does not
			contain heaps of entropy, but it can't do any damage either.
		*/
		if ((result = get_log_entries("System", w)) != PCP_SUCCESS)
		{
			break;
		}

		if (w.is_full())
		{
			result = PCP_SEEDER_TOO_SMALL;
			break;
		}

		if ((result = query_network_statistics(w, L"LanmanWorkstation")) != PCP_SUCCESS)
		{
			break;
		}

		if (w.is_full())
		{
			result = PCP_SEEDER_TOO_SMALL;
			break;
		}

		if ((result = query_network_statistics(w, L"LanmanServer")) != PCP_SUCCESS)
		{
			break;
		}

		if (w.is_full())
		{
			result = PCP_SEEDER_TOO_SMALL;
			break;
		}

	} while(0);

	if (result == PCP_SUCCESS)
	{
		bytes_to_copy = min(w.get_pos(), desired_length);
		memcpy(seed, Buffer, bytes_to_copy);

		/* 
			In case some people do not want to retrieve the full size seed 
			we xor the hash (which depends on the full seed) to the first  
			16 bytes of the returned buffer. This (hopefully) makes sure   
			that no entropy is lost even if some seed bytes are not        
			returned to the caller.                                        
		*/
		md5.md5_mem(Buffer, get_internal_seed_size());
		pcp_util::mem_xor(seed, bytes_to_copy, md5.ctx_digest, 16);

		desired_length = bytes_to_copy;
	}
	else
	{
		desired_length = 0;
	}
	
	if (Buffer != NULL)
	{
		pcp_util::zero_memory(Buffer, get_internal_seed_size());
		delete[] Buffer; 
	}

	return result;
}