
#include <stdio.h>
#include <stdlib.h>
#include <gnutls/gnutls.h>
#include <gnutls/x509.h>

/* This function will print information about this session's peer
 * certificate. 
 */
static void print_x509_certificate_info(gnutls_session session)
{
   char digest[20];
   char serial[40];
   int digest_size, i;
   int serial_size;
   char printable[120];
   int printable_size;
   char *print;
   int algo, bits;
   time_t expiration_time, activation_time;
   const gnutls_datum *cert_list;
   int cert_list_size = 0;
   gnutls_x509_crt cert;

   cert_list = gnutls_certificate_get_peers(session, &cert_list_size);

   if (cert_list_size > 0
       && gnutls_certificate_type_get(session) == GNUTLS_CRT_X509) {

      /* no error checking
       */
      gnutls_x509_crt_init( &cert);

      gnutls_x509_crt_import( cert, &cert_list[0]);

      printf(" - Certificate info:\n");

      expiration_time = gnutls_x509_crt_get_expiration_time( cert);
      activation_time = gnutls_x509_crt_get_activation_time( cert);

      printf(" - Certificate is valid since: %s", ctime(&activation_time));
      printf(" - Certificate expires: %s", ctime(&expiration_time));

      /* Print the fingerprint of the certificate
       */
      digest_size = sizeof(digest);
      if (gnutls_x509_fingerprint
          (GNUTLS_DIG_MD5, &cert_list[0], digest, &digest_size) >= 0) {
         print = printable;
         for (i = 0; i < digest_size; i++) {
            sprintf(print, "%.2x ", (unsigned char) digest[i]);
            print += 3;
         }
         printf(" - Certificate fingerprint: %s\n", printable);
      }

      /* Print the serial number of the certificate.
       */
      serial_size = sizeof(serial);
      if (gnutls_x509_crt_get_serial(cert, serial, &serial_size) >= 0) 
      {
         print = printable;
         for (i = 0; i < serial_size; i++) {
            sprintf(print, "%.2x ", (unsigned char) serial[i]);
            print += 3;
         }
         printf(" - Certificate serial number: %s\n", printable);
      }

      /* Extract some of the public key algorithm's parameters
       */
      algo =
          gnutls_x509_crt_get_pk_algorithm(cert, &bits);

      printf("Certificate public key: ");

      if (algo == GNUTLS_PK_RSA) {
         printf("RSA\n");
         printf(" Modulus: %d bits\n", bits);
      } else if (algo == GNUTLS_PK_DSA) {
         printf("DSA\n");
         printf(" Exponent: %d bits\n", bits);
      } else {
         printf("UNKNOWN\n");
      }

      /* Print the version of the X.509 
       * certificate.
       */
      printf(" - Certificate version: #%d\n",
             gnutls_x509_crt_get_version( cert));

      printable_size = sizeof(printable);
      gnutls_x509_crt_get_dn( cert, printable, &printable_size);
      printf(" - DN: %s\n", printable);

      printable_size = sizeof(printable);
      gnutls_x509_crt_get_issuer_dn( cert, printable, &printable_size);
      printf(" - Certificate Issuer's DN: %s\n", printable);

      gnutls_x509_crt_deinit( cert);

   }
}

