/* getkey-openpgp.c - Kludge to use OpenPGP to emulate GPGME.
 *	Copyright (C) 2002, 2003 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <sys/types.h>
#include <string.h>
#include <stdlib.h>
#include <windows.h>

#include "gpgme-config.h"
#ifdef WITH_OPENPGP
#include "context.h"
#include "ops.h"
#include "util.h"
#include "key.h"
#include "openpgp.h"


static void
set_subkey( struct subkey_s * s, PKT_public_key * pk )
{
    s->key_algo = pk->pubkey_algo;
    s->timestamp = pk->timestamp;
    s->flags.can_encrypt = gpgme_key_cability_from_algo( s->key_algo ) & 0x02;
    s->flags.can_certify = gpgme_key_cability_from_algo( s->key_algo ) & 0x01;
    s->flags.can_sign = gpgme_key_cability_from_algo( s->key_algo ) & 0x01;
    if( pk->expiredate )
	s->flags.expired = pk->expiredate> pk->timestamp? 1 : 0;
    s->flags.revoked = pk->is_revoked? 1 : 0;
}


static void
set_subkey_fprints( struct subkey_s * s, byte *fpr, size_t fprlen )
{
    char skeyid[16+1], sfpr[40+1];
    int i;

    for( i = 0; i < 8; i++ )
	sprintf( skeyid + 2*i, "%02X", fpr[i+12] );
    skeyid[16] = '\0';
    memcpy( s->keyid, skeyid, 16 );
    s->keyid[16] = '\0';
    for( i = 0; i < 20; i++ )
	sprintf( sfpr + 2*i, "%02X", fpr[i] );
    sfpr[40] = '\0';
    s->fingerprint = malloc( 41 );
    if( s->fingerprint ) {
	memcpy( s->fingerprint, sfpr, 40 );
	s->fingerprint[40] = '\0';
    }
}


int
gpgme_getkey_bykeyid( void *fd_handle, const char * keyid, gpgme_key_t *r_key )
{
    PACKET * pkt = NULL;
    gpg_kbnode_t key = NULL, ctx = NULL, n;
    gpgme_key_t _key = NULL;
    gpg_iobuf_t inp = fd_handle;
    struct subkey_s * s;
    u32 kid[2], pk_keyid;
    byte fprint[21] = {0};
    char buf[4];
    size_t i;

    if( !fd_handle )
	return mk_error( Invalid_Value );
    
    if( !strncmp( keyid, "0x", 2 ) )
	keyid += 2;
    for( i = 0; i < (strlen( keyid )+1) / 8; i++ ) {
	strncpy( buf, keyid + 8*i, 8 );	
	kid[i] = strtoul( buf, NULL, 16 );
    }
    if( strlen( keyid ) == 8 ) {
	kid[1] = kid[0];
	kid[0] = 0;
    }

    gpg_iobuf_seek( inp, 0 );
    while( !gpg_read_keyblock( inp, &pkt, &key ) ) {
	while( (n = gpg_walk_kbnode( key, &ctx, 1 ) ) ) {
	    switch( n->pkt->pkttype ) {
	    case PKT_PUBLIC_KEY:
	    case PKT_PUBLIC_SUBKEY:
		pk_keyid = gpg_keyid_from_pk( n->pkt->pkt.public_key, fprint );
		if( n->pkt->pkttype == PKT_PUBLIC_SUBKEY && _key )
		    ;
		else if( pk_keyid != kid[1] )
		    break;
		if( !_key )
		    _gpgme_key_new( &_key );		
		s = _gpgme_key_add_subkey( _key );
		if( s ) {
		    set_subkey( s, n->pkt->pkt.public_key );
		    set_subkey_fprints( s, fprint, 20 );
		}
		break;

	    case PKT_USER_ID:
		if( !_key )
		    continue;
		_gpgme_key_append_name( _key, n->pkt->pkt.user_id->name, NULL );
		break;
	    }
	}
	gpg_release_kbnode( key );
	key = NULL;
	ctx = NULL;
	if( _key )
	    break;
    }
    if( r_key )
	*r_key = _key;
    return 0;
}


#endif /*WITH_OPENPGP*/
