/* keycache.c - Caching for the pub- and the secring
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <string.h>

#include "context.h"
#include "ops.h"
#include "key.h"
#include "util.h"


gpgme_error_t
gpgme_keycache_new( gpgme_keycache_t *r_ctx  )
{
    gpgme_keycache_t ctx;
    
    if( !r_ctx )
	return mk_error( Invalid_Value );
    ctx = calloc( 1, sizeof *ctx );
    if( !ctx )
        return mk_error( Out_Of_Core );
    ctx->secret = 0;
    ctx->pos = 0;
    *r_ctx = ctx;
    return 0;
} /* gpgme_keycache_new */


void
gpgme_keycache_release (gpgme_keycache_t ctx)
{
    struct keycache_s * c, * c2;
    
    if (!ctx)
	return;

    for( c = ctx->item; c; c = c2 ) {
        c2 = c->next;
	gpgme_key_release( c->key );
	c->key = NULL;
	safe_free( c );   
    }
    safe_free( ctx );
} /* gpgme_keycache_release */


void
gpgme_keycache_set_cb (gpgme_keycache_t ctx,
		       void (*cb)(void *, const char *, int, unsigned, unsigned),
		       void * cb_value1, int cb_value2)
{
    if (!ctx)
	return;
    ctx->cb = cb;
    ctx->cb_value = cb_value1;
    ctx->cb_value2 = cb_value2;
}


int
gpgme_keycache_add_key( gpgme_keycache_t ctx, gpgme_key_t key )
{
    struct keycache_s * c, * n1;
    
    if( !ctx )
        return mk_error(Invalid_Value);
    
    c = calloc( 1, sizeof *c );
    if( !c )
        return mk_error( Out_Of_Core );
    c->key = key;
    if( !ctx->item )
	ctx->item = c;
    else {
	for( n1 = ctx->item; n1 && n1->next; n1 = n1->next )
	    ;
	n1->next = c;
    }
    return 0;
} /* gpgme_keycache_add_key */


#define is_deleted_item(c) (((c)->flags & 0x01))
#define has_keyid_len(pattern) (\
    strlen (pattern) == 8  || strlen (pattern) == 10 || \
    strlen (pattern) == 16 || strlen (pattern) == 18)


static gpgme_error_t
keycache_find_key (gpgme_keycache_t ctx, const char * pattern, int flags,
		   gpgme_key_t * r_key, struct keycache_s ** r_item)
{
    struct keycache_s * c;
    struct subkey_s * s;
    struct user_id_s * u;
    const char *kid;    
    
    if (!ctx || !r_key)
        return mk_error (Invalid_Value);
    
    if (strstr (pattern, "0x"))
        pattern += 2;
    /* fixme: this is VERY slow, se we should use the GPGME original code */
    for (c = ctx->item; c; c = c->next) {
	if (is_deleted_item (c))
	    continue;
        for (s = &c->key->keys; s; s = s->next) {
            for (u = c->key->uids; u; u = u->next) {
                if (u->name && memistr (u->name, strlen (u->name), pattern)) {
		    if (r_item)
			*r_item = c;
		    *r_key = flags? c->pubpart : c->key;
                    return 0;
                }
            }
	    if (has_keyid_len (pattern))
                kid = s->keyid;
            else
                kid = s->fingerprint;
            
            if (kid && memistr (kid, strlen (kid), pattern)) {
		if (r_item)
		    *r_item = c;
		*r_key = flags? c->pubpart : c->key;
                return 0;
            }
        }
    }
    *r_key = NULL;
    return mk_error (General_Error);
} /* keycache_find_key */


gpgme_error_t
gpgme_keycache_find_key (gpgme_keycache_t ctx, const char * pattern,
			 int flags, gpgme_key_t * r_key)
{
    return keycache_find_key (ctx, pattern, flags, r_key, NULL);
} /* gpgme_keycache_find_key */


gpgme_error_t
gpgme_keycache_delete_key (gpgme_keycache_t ctx, const char * pattern)
{
    struct keycache_s * c = NULL;
    gpgme_key_t key;
    gpgme_error_t rc;

    if (!ctx)
	return mk_error (Invalid_Value);
    rc = keycache_find_key (ctx, pattern, 0, &key, &c);
    if (!rc)	
	c->flags |= 1;
    return rc;
} /* gpgme_keycache_delete_key */


gpgme_error_t
gpgme_keycache_init( gpgme_keycache_t ctx, const char *pattern, int secret )
{
    gpgme_error_t err;
    gpgme_ctx_t c;
    gpgme_key_t key;
    
    if (!ctx)
        return mk_error( Invalid_Value );
    
    err = gpgme_new (&c);
    if (err)
	return err;
    if (ctx->cb)
    {
	gpgme_control (c, GPGME_CTRL_CB_VAL, ctx->cb_value2);
	gpgme_set_progress_cb (c, ctx->cb, ctx->cb_value);
    }
    gpgme_control (c, GPGME_CTRL_LOGGING, 1);
    err = gpgme_op_keylist_start( c, pattern, secret );    
    while(!err)
    {
	err = gpgme_op_keylist_next (c, &key);
	if (!err)
	    err = gpgme_keycache_add_key (ctx, key);
    }
    if (err == GPGME_EOF)
	err = 0;
    if (gpgme_get_process_rc (c))
	err = gpgme_check_logging (c);
    gpgme_release (c);
    return err;
} /* gpgme_keycache_init */


gpgme_error_t
gpgme_keycache_sync (gpgme_keycache_t pub, gpgme_keycache_t sec)
{
    struct keycache_s * c;
    const char * s;
    gpgme_key_t key;

    if (!pub || !sec)
	return mk_error (Invalid_Value);
    /* The GPG secret key listing does not contain much information
       so we add some information to the public key cache. */
    for (c=sec->item; c; c=c->next) {
	s = gpgme_key_get_string_attr (c->key, GPGME_ATTR_KEYID, NULL, 0);
	if (!gpgme_keycache_find_key (pub, s, 0, &key)) {
	    key->gloflags.is_protected = c->key->gloflags.is_protected;
	    key->gloflags.divert_to_card = c->key->gloflags.divert_to_card;
	    c->pubpart = key;
	}
    }
    return 0;
}


void
gpgme_keycache_rewind (gpgme_keycache_t ctx)
{
    if (ctx)
	ctx->pos = 0;
} /* gpgme_keycache_rewind */


int
gpgme_keycache_count (gpgme_keycache_t ctx)
{
    struct keycache_s * c;
    int count = 0;
    
    if (!ctx)
        return 0;
    for (c = ctx->item; c; c = c->next) {
	if (is_deleted_item (c))
	    continue;
        count++;
    }
    return count;
} /* gpgme_keycache_count */


gpgme_error_t
gpgme_keycache_next_key (gpgme_keycache_t ctx, int flags, gpgme_key_t * r_key)
{       
    if (!ctx || !r_key)
        return mk_error (Invalid_Value);

    if (!ctx->pos)
        ctx->tmp = ctx->item;
    
    if (!ctx->tmp || !ctx->tmp->key) {
        ctx->pos = 0;
	*r_key = NULL;
        return mk_error (Invalid_Value);
    }
    
    if (r_key)
	*r_key = flags? ctx->tmp->pubpart : ctx->tmp->key;
    
    ctx->tmp = ctx->tmp->next;
    ctx->pos++;

    return 0;
} /* gpgme_keycache_next_key */
