/* wptMAPI.cpp
 *	Copyright (C) 2003, 2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <stdio.h>
#include <mapi.h>

extern HINSTANCE glob_hinst;

#include "../resource.h"
#include "wptTypes.h"
#include "wptErrors.h"
#include "wptW32API.h"
#include "wptGPG.h"


static LPMAPILOGON	    mapi_logon = NULL; 
static LPMAPILOGOFF	    mapi_logoff = NULL;     
static LPMAPISENDDOCUMENTS  mapi_send_documents = NULL; 
static LPMAPISENDMAIL	    mapi_send_mail = NULL;
static HINSTANCE	    hlib = NULL;
static int		    init = 0;

#define load_one_fnc(cast, hlib, name) (cast)GetProcAddress ((hlib), name)



int
mapi_init (void)
{
    if (init)
	return 0;

    hlib = LoadLibrary ("MAPI32.DLL");
    if (!hlib)
	return -1;

    mapi_logon = load_one_fnc (LPMAPILOGON, hlib, "MAPILogon");
    mapi_logoff = load_one_fnc (LPMAPILOGOFF, hlib, "MAPILogoff");
    mapi_send_documents = load_one_fnc (LPMAPISENDDOCUMENTS, hlib, "MAPISendDocuments");
    mapi_send_mail = load_one_fnc (LPMAPISENDMAIL, hlib, "MAPISendMail");
    if (!mapi_logon || !mapi_logoff || !mapi_send_documents || !mapi_send_mail)
	return -1;
    init = 1;

    return 0;
} /* mapi_init */


void
mapi_deinit (void)
{
    if (hlib) {
	FreeLibrary (hlib);
	hlib = NULL;
	init = 0;
    }
} /* mapi_deinit */


int
mapi_send_ascfile (char * ascfile)
{
    LHANDLE hd;
    int rc;

    if (!init)
	return 0;

    rc = mapi_logon (0, NULL, NULL, MAPI_LOGON_UI, 0, &hd);
    if (rc != SUCCESS_SUCCESS) {
	MessageBox (NULL, _("MAPI Login failed."), "MAPI", MB_ICONWARNING|MB_OK);
	goto fail;
    }
    rc = mapi_send_documents (0, ";", ascfile, NULL, 0);
    if (rc == MAPI_E_USER_ABORT)
	rc = SUCCESS_SUCCESS;
    if (rc != SUCCESS_SUCCESS)
	MessageBox (NULL, _("Could not sent mail."), "MAPI", MB_ICONERROR|MB_OK);

fail:
    mapi_logoff (hd, 0, 0, 0);
    return rc;
}


int
mapi_send_pubkey (const char * keyid, char * keyfile)
{
    LHANDLE hd;
    const char * fmt;
    char * keyinf = NULL;
    int rc;

    if (!init)
	return 0;

    fmt = _("GPG Public Key of %s");
    keyinf = new char[strlen (fmt) + strlen (keyid) + 2];
    if (!keyinf)
	BUG (0);
    sprintf (keyinf, fmt, keyid);
    rc = mapi_logon (0, NULL, NULL, MAPI_LOGON_UI, 0, &hd);
    if (rc != SUCCESS_SUCCESS) {
	MessageBox (NULL, _("MAPI Login failed."), "MAPI", MB_ICONWARNING|MB_OK);
	goto fail;
    }
    rc = mapi_send_documents (0, ";", keyfile, keyinf, 0);
    if (rc == MAPI_E_USER_ABORT)
	rc = SUCCESS_SUCCESS;
    if (rc != SUCCESS_SUCCESS)
	MessageBox (NULL, _("Could not sent mail."), "MAPI", MB_ICONERROR|MB_OK);

fail:
    mapi_logoff (hd, 0, 0, 0);
    free_if_alloc (keyinf);
    return rc;
} /* mapi_send_pubkey */


static void
free_mapi_msg (MapiMessage * msg)
{
    if (!msg)
	return;
    safe_free (msg->lpszSubject);
    safe_free (msg->lpszNoteText);
    safe_free (msg);
} /* free_mapi_msg */


static void
free_recip_tab (MapiRecipDesc * recip, size_t n)
{
    size_t i;

    if (!recip)
	return;
    if (!n)
	return;
    for (i=0; i < n; i++)
	safe_free (recip[i].lpszName);
    safe_free (recip);
} /* free_recip_tab */


static void
free_files_tab (MapiFileDesc * files, size_t n)
{
    size_t i;

    if (!files)
	return;
    if (!n)
	return;
    for (i=0; i < n; i++) {
	safe_free (files[i].lpszFileName);
	safe_free (files[i].lpszPathName);
    }
    safe_free (files);
} /* free_files_tab */



static gpgme_recipients_t
conv_recipients (gpgme_recipients_t rset)
{
    gpgme_recipients_t r;
    gpgme_error_t rc;
    void * ctx=NULL;
    const char * s;

    /* we need to convert the recipients to email addresses so 
       GPG can handle them. */
    rc = gpgme_recipients_new (&r);
    if (rc)
	return NULL;
    gpgme_recipients_enum_open (rset, &ctx);

    while ((s=gpgme_recipients_enum_read (rset, &ctx))) {
	char * p, * q, * buf;
	if (!(p = strchr (s, '<')) || !(q = strchr (s, '>')))
	    continue;
	buf = (char * )calloc (1, (q-s)-(p-s)+2);
	strncpy (buf, s+(p-s)+1, (q-s)-(p-s)-1);
	gpgme_recipients_add_name (r, buf);
	safe_free (buf);
    }
    return r;
} /* conv_recipients */


static char *
secure_attachment (gpgme_recipients_t rset, const char * fname)
{
    char tmpdir[512+32], * p;
    gpgme_recipients_t addrs;
    gpgme_ctx_t ctx;
    gpgme_error_t rc;

    if (strlen (fname) > 200)
	BUG (0);
    GetTempPath (sizeof tmpdir-200, tmpdir);
    p = strrchr (fname, '\\');
    if (!p)
	strcat (tmpdir, fname);
    else
	strcat (tmpdir, fname+(p-fname)+1);
    strcat (tmpdir, ".asc");

    rc = gpgme_new (&ctx);
    if (rc)
	return NULL;
    gpgme_control (ctx, GPGME_CTRL_ARMOR, 1);
    gpgme_control (ctx, GPGME_CTRL_FORCETRUST, 1);
    addrs = conv_recipients (rset);
    if (!addrs) {
	msg_box (NULL, _("No valid mail addresses found."), _("Secure Attachment"), MB_ERR);
	gpgme_release (NULL);
	return NULL;
    }
    rc = gpgme_op_file_encrypt (ctx, addrs, fname, tmpdir);
    if (rc)
	log_box (_("Secure Attachment"), MB_ERR, _("Could not encrypt '%s'"), fname);
    gpgme_recipients_release (addrs);
    gpgme_release (ctx);
    return strdup (tmpdir);
} /* secure_attachment */


static char *
secure_message (gpgme_recipients_t rset, const char * data)
{
    gpgme_recipients_t addrs;
    gpgme_error_t rc;
    gpgme_data_t in, out;
    gpgme_ctx_t ctx;
    char * p;
    size_t n=0;

    rc = gpgme_new (&ctx);
    if (rc)
	return NULL;
    gpgme_control (ctx, GPGME_CTRL_ARMOR, 1);
    gpgme_control (ctx, GPGME_CTRL_FORCETRUST, 1);

    addrs = conv_recipients (rset);
    rc = gpgme_data_new_from_mem (&in, data, strlen (data), 1);
    if (rc) {
	gpgme_release (ctx);
	return NULL;
    }
    gpgme_data_new (&out);
    
    rc = gpgme_op_encrypt (ctx, addrs, in, out);
    if (rc)
	log_box (_("Secure Message"), MB_ERR, "Could not encrypt the data");

    p = gpgme_data_release_and_return_string (out);

    gpgme_data_release (in);
    gpgme_release (ctx);
    gpgme_recipients_release (addrs);

    return p;
} /* secure_message */


int
mapi_send_message (gpgme_recipients_t rset, const char * msgtxt,
		   const char * subject, const char ** files, size_t nfiles)
{
    LHANDLE hd;
    MapiMessage * msg;
    MapiRecipDesc * recip;
    MapiFileDesc * attch;
    char * p;
    const char * s;
    void * ctx=NULL;
    size_t n, i=0;
    int rc;

    if (!init)
	return 0;

    rc = mapi_logon (0, NULL, NULL, MAPI_LOGON_UI, 0, &hd);
    if (rc != SUCCESS_SUCCESS) {
	MessageBox (NULL, "MAPI Login failed.", "MAPI", MB_ICONWARNING|MB_OK);
	return rc;
    }

    msg = (MapiMessage *)calloc (1, sizeof * msg);
    p = msg->lpszSubject = strdup (subject);
    if (!p)
	BUG (0);
    p = msg->lpszNoteText = secure_message (rset, msgtxt);
    if (!p)
	BUG (0);
    n = msg->nRecipCount = gpgme_recipients_count (rset);
    recip = (MapiRecipDesc *)calloc (n+1, sizeof * recip);
    
    gpgme_recipients_enum_open (rset, &ctx);
    while ((s = gpgme_recipients_enum_read (rset, &ctx))) {
	if (!i)
	    recip[i].ulRecipClass = MAPI_TO;
	else
	    recip[i].ulRecipClass = MAPI_CC;	
	p = recip[i].lpszName = strdup (s);
	if (!p)
	    BUG (0);
	i++;
    }
    msg->lpRecips = recip;

    if (nfiles) {
	msg->nFileCount = nfiles;
	attch = (MapiFileDesc *)calloc (nfiles+1, sizeof * attch);
	for (i=0; i < nfiles; i++) {
	    char * p = secure_attachment (rset, *files);
	    if (!p)
		continue;
	    attch[i].lpszFileName = strdup (*files);	    
	    attch[i].lpszPathName = strdup (p);
	    files++;
	    safe_free (p);
	}
	msg->lpFiles = attch;
    }

    rc = mapi_send_mail (hd, 0, msg, 0, 0);
    if (rc == MAPI_E_USER_ABORT)
	rc = SUCCESS_SUCCESS;
    if (rc != SUCCESS_SUCCESS)
	MessageBox (NULL, _("Could not sent mail."), "MAPI", MB_ERR);

    free_recip_tab (recip, n);
    free_files_tab (attch, nfiles);
    free_mapi_msg (msg);
    mapi_logoff (hd, 0, 0, 0);

    return 0;
} /* mapi_send_message */


static int
add_recipient (gpgme_recipients_t * r_rset, const char * addr)
{
    gpgme_keycache_t pub = keycache_get_ctx (1);
    gpgme_key_t key;
    gpgme_error_t rc;
    const char * s;

    if (!*r_rset)
	gpgme_recipients_new (&(*r_rset));
    rc = gpgme_keycache_find_key (pub, addr, 0, &key);
    if (rc) {
	log_box (_("Add Recipient"), MB_ERR, _("Could not find key for '%s'"), addr);
	return -1;
    }
    s = gpgme_key_get_string_attr (key, GPGME_ATTR_USERID, NULL, 0);
    if (s)
	gpgme_recipients_add_name (*r_rset, s);
    return 0;
} /* add_recipient */


static int
add_all_recipients (HWND dlg, int itemid, gpgme_recipients_t * r_rset)
{    
    char buf[1024], * p;
    int n=0;

    n = GetDlgItemText (dlg, itemid, buf, sizeof buf-10);
    if (!n)
	return -1;
    p = strtok (buf, ";,");
    while (p != NULL) {
	add_recipient (&*r_rset, p);
	p = strtok (NULL, ";,");
    }
    return 0;
} /* add_all_recipients */


BOOL CALLBACK
winpt_mail_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static gpgme_recipients_t rset=NULL;
    char subject[128];
    char * msgbuf;
    int n;

    switch (msg) {
    case WM_INITDIALOG:
	center_window (dlg);
	SetForegroundWindow (dlg);
	break;

    case WM_COMMAND:
	switch (LOWORD (wparam)) {
	case IDOK:
	    add_all_recipients (dlg, IDC_PMAIL_TO, &rset);
	    add_all_recipients (dlg, IDC_PMAIL_CC, &rset);
	    if (!gpgme_recipients_count (rset)) {
		msg_box (dlg, _("Please enter a recipient."), _("Mail"), MB_ERR);
		return FALSE;
	    }
	    n=GetDlgItemText (dlg, IDC_PMAIL_SUBJECT, subject, sizeof subject-1);
	    if (!n)
		strcpy (subject, "");
	    n = SendDlgItemMessage (dlg, IDC_PMAIL_MSG, WM_GETTEXTLENGTH, 0, 0);
	    if (!n) {
		msg_box (dlg, _("Please enter a message."), _("Mail"), MB_ERR);
		return FALSE;
	    }
	    msgbuf = (char * )calloc (1, n+2);
	    GetDlgItemText (dlg, IDC_PMAIL_MSG, msgbuf, n+1);
	    mapi_send_message (rset, msgbuf, subject, NULL, 0);
	    safe_free (msgbuf);
	    EndDialog (dlg, TRUE);
	    break;

	case IDCANCEL:
	    EndDialog (dlg, FALSE);
	    break;
	}
	break;
    }

    return FALSE;
} /* winpt_mail_proc */
