class Finance::Api::InvoicesController < Finance::Api::BaseController

  representer ::Invoice

  paginate only: :index

  ##~ sapi = source2swagger.namespace("Billing API")
  #
  ##~ @base_path = ""
  #
  ##~ sapi.basePath     = @base_path
  ##~ sapi.swaggerVersion = "0.1a"
  ##~ sapi.apiVersion   = "1.0"
  #
  ##~ e = sapi.apis.add
  ##~ e.path = "/api/invoices.xml"
  ##~ e.responseClass = "List[invoice]"
  #
  ##~ op            = e.operations.add
  ##~ op.httpMethod = "GET"
  ##~ op.summary    = "Invoice List"
  ##~ op.description = "Returns the list of all invoices. Note that results can be paginated and you can apply filters by month and state."
  ##~ op.group = "finance"
  ##~
  ##~ @parameter_access_token = { :name => "access_token", :description => "A personal Access Token", :dataType => "string", :required => true, :paramType => "query", :threescale_name => "access_token" }
  ##~ @parameter_invoice_id_by_id = { :name => "id", :description => "ID of the invoice.", :dataType => "int", :allowMultiple => false, :required => true, :paramType => "path" }
  ##~ @parameter_page = {:name => "page", :description => "Page in the paginated list. Defaults to 1.", :dataType => "int", :paramType => "query", :defaultValue => "1"}
  ##~ @parameter_per_page = {:name => "per_page", :description => "Number of results per page. Default and max is 20.", :dataType => "int", :paramType => "query", :defaultValue => "20"}
  ##~ @parameter_invoice_state = {:name => "state", :description => "Filter invoices by state. 'open': line items can be added (via web interface). 'pending': the invoice has been issued, no items can be added, the PDF has been generated and the invoice is waiting to be charged. 'paid': sucessfully paid. 'unpaid': charging failed at least once. 'cancelled': the invoice was explicitly cancelled and is out of the normal life-cycle.", :dataType => "string", :paramType => "query", :required => false, :defaultValue => "", :allowableValues => { :values => ["open", "pending", "paid", "unpaid", "cancelled"], :valueType => "LIST" }}
  ##~ @parameter_invoice_month = {:name => "month", :description => "Filter invoices by month. Format YYYY-MM, e.g. '2012-03'.", :dataType => "string", :paramType => "query"}
  ##~ @parameter_account_id = {:name => "account_id", :description => "ID of the account.", :dataType => "int", :required => true, :paramType => "path", :threescale_name => "account_ids"}
  #
  ##~ op.parameters.add @parameter_access_token
  ##~ op.parameters.add @parameter_invoice_state
  ##~ op.parameters.add @parameter_invoice_month
  ##~ op.parameters.add @parameter_page
  ##~ op.parameters.add @parameter_per_page
  ##~
  #
  def index
    search = ThreeScale::Search.new(params[:search] || params)
    results = invoices.scope_search(search)
                .order_by(params[:sort], params[:direction])
                .paginate(pagination_params)

    respond_with(results)
  end

  ##~ e = sapi.apis.add
  ##~ e.path = "/api/invoices/{id}.xml"
  ##~ e.responseClass = "invoice"
  #
  ##~ op            = e.operations.add
  ##~ op.httpMethod = "GET"
  ##~ op.summary    = "Invoice"
  ##~ op.description = "Returns an invoice by ID."
  ##~ op.group = "finance"
  #
  ##
  ##~ op.parameters.add @parameter_access_token
  ##~ op.parameters.add @parameter_invoice_id_by_id
  #
  def show
    respond_with(invoice) do |format|
      format.pdf {  redirect_to invoice.pdf.url }
    end
  end


  ##~ e = sapi.apis.add
  ##~ e.path = "/api/invoices/{id}/state.xml"
  ##~ e.responseClass = "invoice"
  #
  ##~ op            = e.operations.add
  ##~ op.httpMethod = "PUT"
  ##~ op.summary    = "Invoice"
  ##~ op.description = "Modifies the state of the invoice."
  ##~ op.group = "finance"
  #
  ##~ @parameter_state = { :name => "state", :description => "State of the invoice to set. Only 'cancel' value is allowed", :dataType => "string", :required => true, :paramType => "query", :threescale_name => "state"}
  ##~ op.parameters.add @parameter_access_token
  ##~ op.parameters.add @parameter_invoice_id_by_id
  ##~ op.parameters.add @parameter_state
  #
  def state
    if invoice && params[:state] == "cancel" && invoice.cancel
      respond_with(invoice)
    else
      render_error("Action forbidden. Only cancel state allowed as destination",
                   status: 409)
    end
  end

  private

  def invoices
    @invoices ||= current_account.buyer_invoices.includes(:line_items, {:buyer_account => [:country]}, :provider_account)
  end

  def invoice
    @invoice ||= invoices.find(params[:id])
  end

end
