require 'test_helper'

class DeveloperPortal::LoginControllerTest < DeveloperPortal::ActionController::TestCase

  include TestHelpers::FakeWeb

  test 'recognizes alternative create route' do
    assert_recognizes({:controller => 'developer_portal/login', :action => 'create'},
                      {:method => :get, :path => 'session/create'})
  end

  # {{{ CAS

  test 'cas is not displayed on login page' do
    provider_account = Factory :provider_account
    provider_settings = provider_account.settings
    provider_settings.authentication_strategy = 'internal'
    provider_settings.save!
    @request.host = provider_account.domain

    get :new

    assert_response 200
    assert !@response.body.include?("CAS")
  end

  test 'cas is displayed on login page' do
    provider_account = Factory :provider_account

    provider_settings = provider_account.settings
    provider_settings.authentication_strategy = 'cas'
    provider_settings.cas_server_url = "http://mamacit.as"
    provider_settings.save!
    @request.host = provider_account.domain

    get :new

    assert_response 200
    assert @response.body.include?("CAS")
  end

  test 'cas successful auth' do
    provider_account = Factory :provider_account
    provider_settings = provider_account.settings
    provider_settings.authentication_strategy = 'cas'
    provider_settings.cas_server_url = "http://mamacit.as"
    provider_settings.save!

    buyer_account = Factory :buyer_account, :provider_account => provider_account
    user = Factory :user, :account  => buyer_account, :cas_identifier => "laurie"
    user.activate!
    user.save!

    @request.host = provider_account.domain

    res = stub :body => "yes\nlaurie", :code => 200
    HTTPClient.expects(:get).with(anything).returns(res)

    get :create, :ticket => "made-up"

    assert_redirected_to '/'

    assert_nil session[:user_id]
    assert_equal UserSession.authenticate(cookies.signed[:user_session]).user.id, user.id
  end

  test 'oauth2 successful authenticate' do
    provider_account = Factory(:provider_account)
    provider_settings = provider_account.settings
    provider_settings.authentication_strategy = 'oauth2'
    provider_settings.save!

    buyer_account = Factory(:buyer_account, :provider_account => provider_account)
    user = Factory(:user, :account  => buyer_account, :password => 'kangaroo')
    user.activate!
    user.authentication_id = "oauth|1234"
    user.save!

    authentication_provider = Factory(:authentication_provider, account: provider_account, kind: 'base')
    @request.host = provider_account.domain

    mock_oauth2(user.authentication_id, 'C6789', authentication_provider.user_info_url)

    post :create, system_name: authentication_provider.system_name, code: 'C6789'
    assert_equal 'Signed in successfully', flash[:notice]
  end

  test 'oauth2 redirect to signup' do
    provider_account = Factory(:provider_account)
    provider_settings = provider_account.settings
    provider_settings.authentication_strategy = 'oauth2'
    provider_settings.save!

    authentication_provider = Factory(:authentication_provider, account: provider_account, kind: 'base')
    @request.host = provider_account.domain

    mock_oauth2('foo', 'C6789', authentication_provider.user_info_url)

    post :create, system_name: authentication_provider.system_name, code: 'C6789', plan_id: 42

    assert_redirected_to signup_path(plan_id: 42)
    assert_equal 'Succesfully authenticated, please complete the signup form', flash[:notice]

  end

  test 'disabled when account is suspended' do
    host! Factory(:simple_provider, state: 'suspended').domain

    get :new

    assert_response :not_found
  end

  test 'ssl certificate error' do
    provider_account = FactoryGirl.create(:provider_account)
    provider_settings = provider_account.settings
    provider_settings.authentication_strategy = 'oauth2'
    provider_settings.save!

    buyer_account = FactoryGirl.create(:buyer_account, provider_account: provider_account)
    user = FactoryGirl.create(:user, account: buyer_account, password: 'kangaroo')
    user.activate!
    user.authentication_id = "oauth|1234"
    user.save!

    authentication_provider = FactoryGirl.create(:authentication_provider, account: provider_account, kind: 'base')
    host! provider_account.domain

    client = mock
    client.stubs(:authenticate!).raises(Faraday::SSLError.new('Faraday::SSLError: hostname "example.com" does not match the server certificate'))
    ThreeScale::OAuth2::Client.expects(build: client)
    post :create, system_name: authentication_provider.system_name, code: 'abcdefg1234567'
    assert_equal I18n.t('errors.messages.oauth_invalid_certificate'), flash[:error]
  end
end
