require File.expand_path(File.dirname(__FILE__) + '/../../test_helper')

class Account::DomainsTest < ActiveSupport::TestCase
  def setup
  end

  test '#domain must be downcase' do
    account_one = Factory(:simple_provider)
    account_one.subdomain = 'FOO'
    assert !account_one.valid?
  end

  test '#domain must be unique' do
    account_one = Factory(:simple_provider)

    account_two = Factory.build(:provider_account)
    account_two.domain = account_one.domain

    assert !account_two.valid?
    assert account_two.errors[:domain].include? 'has already been taken'
  end

  test '#domain uniqueness ignores deleted' do
    account_one = Factory(:simple_provider)

    account_two = Factory.build(:provider_account)
    account_two.domain = account_one.domain

    assert !account_two.valid?
    assert account_two.errors[:domain].include? 'has already been taken'

    account_one.destroy

    assert account_two.valid?
  end

  test '#dedicated_domain returns nil if domain is set to a subdomain of the master domain' do
    account = Account.new(:provider_account => master_account,
                          :domain => "lol.#{master_account.superdomain}")
    assert_nil account.dedicated_domain
  end

  test 'domain is excluded from mass assignment ' do
    account = Account.new(:domain => 'api.lolcatz.net')
    assert_nil account.dedicated_domain
  end

  test '#superdomain returns the domain without www striped' do
    account = Account.new

    account.domain = 'www.foo.com'
    assert_equal '3scale.net', account.superdomain

    account.domain = 'foo.com'
    assert_equal '3scale.net', account.superdomain
  end

  test '#subdomain returns nil if domain is not set to a subdomain of the provider domain' do
    account = Account.new(:provider_account => master_account,
                          :domain => 'api.lolcatz.net')

    assert_nil account.subdomain
  end


  test '#generate_subdomain normalizes the input' do
    assert_equal 'foo-bar', master_account.generate_subdomain('Foo Bar')
  end

  test '#generate_subdomain returns unique subdomain' do
    Factory(:simple_account, :provider_account => master_account,
                      :domain => "foo.#{master_account.superdomain}")
    assert_equal 'foo-2', master_account.generate_subdomain('Foo')

    Factory(:simple_account, :provider_account => master_account,
                      :domain => "foo-2.#{master_account.superdomain}")
    assert_equal 'foo-3', master_account.generate_subdomain('Foo')

    Factory(:simple_account, :provider_account => master_account,
                      :domain => "foo-42.#{master_account.superdomain}")
    assert_equal 'foo-3', master_account.generate_subdomain('Foo')
  end

  test '#generate_subdomain returns empty string on empty input' do
    assert_equal '', master_account.generate_subdomain('')
  end

  test '#generate_subdomain returns empty string on empty input even if there exists an account with empty subdomain' do
    Factory(:simple_account, :provider_account => master_account,
                      :domain => ".#{master_account.superdomain}")
    assert_equal '', master_account.generate_subdomain('')
  end

  test '#generate_subdomain handles nils' do
    assert_equal '', master_account.generate_subdomain(nil)
  end

  test 'find_by_domain! raises an exception if domain is blank' do
    assert_raise ActiveRecord::RecordNotFound do
      Account.find_by_domain!(nil)
    end
  end

  test 'find_by_domain! raises an exception if domain is blank even if there is an account with blank domain' do
    account = Account.create!(:org_name => 'foo', :domain => '')

    assert_raise ActiveRecord::RecordNotFound do
      Account.find_by_domain!(nil)
    end
  end

  test 'find_by_domain! finds account by domain' do
    account = Factory(:simple_account, :domain => 'foo.com')
    assert_equal account, Account.find_by_domain!('foo.com')
  end
end
