require 'test_helper'

class ApiDocs::AccountDataTest < ActiveSupport::TestCase

  def setup
    @provided_apps, @bought_apps, @metrics, @buyer_accounts, @admins, @plans = [], [], [], [], [], []

    @services = [service = stub('service', name: 'service-foo', id: 1)]
    @metrics.stubs(:top_level => [
        stub('top level metric', name: 'hits', friendly_name: 'Hits', id: '1', service: service)
    ])

    @provided_apps.stubs(:latest => stub('latest provided app', :live => [
        stub('live provided app', id: 1, service: service, application_id: '1', user_key: 'prov-123', name: 'Provided Foo', keys: [])
    ]))
    @bought_apps.stubs(:latest => stub('latest bought app', :live => [stub('live bought app', :id => 1, :application_id => '1', :user_key => 'bought-123', :name => 'Bought Bar', :keys => ['bought-123-app-key'])]))

    @buyer_accounts.stubs(:latest => [stub('latest buyer account', :name => 'Foo Account', :id => '1', :admins => [stub(:username => 'Foo User', :id => '1')])])
    @admins.stubs(:latest => [stub('latest admin', :username => "Foo Admin", :id => '1')])

    @plans.stubs(:latest => [stub('latest plan', name: 'Foo Plan', id: '1', service: service)])

    @provider = stub('provider',
            :account_plans => @plans,
            :service_plans => @plans,
            :application_plans => @plans,
            :buyer_accounts => @buyer_accounts,
            :admins => @admins,
            :buyer? => false,
            :services => @services,
            :provided_cinstances => @provided_apps,
            :api_key => 'prov-api-key',
            :metrics => @metrics)
    @buyer = stub('buyer', :buyer? => true, :bought_cinstances => @bought_apps, :api_key => 'buyer-api-key')
  end

  def test_status_with_account
    data = ApiDocs::BuyerData.new(@buyer).as_json
    assert_equal 200, data[:status]
  end

  def test_status_without_user
    data = ApiDocs::ProviderData.new(nil).as_json
    assert_equal 401, data[:status]
  end

  def test_returns_metrics_for_provider
    data = ApiDocs::ProviderData.new(@provider).as_json[:results]
    assert_equal [{:name => 'Hits | service-foo', :value => 'hits'}], data[:metric_names]
    assert_equal [{:name => 'Hits | service-foo', :value => '1'}], data[:metric_ids]
  end

  def test_returns_user_ids_for_provider
    data = ApiDocs::ProviderData.new(@provider).as_json[:results]
    assert_equal [{:name => 'Foo User', :value => '1'}], data[:user_ids]
  end

  def test_returns_admin_ids_for_provider
    data = ApiDocs::ProviderData.new(@provider).as_json[:results]
    assert_equal [{:name => 'Foo Admin', :value => '1'}], data[:admin_ids]
  end

  def test_returns_provided_applications_for_provider
    data = ApiDocs::ProviderData.new(@provider).as_json[:results]
    assert_equal [{:name => 'Provided Foo', :value => 'prov-123'}], data[:user_keys]
    assert_equal [{:name => 'Provided Foo', :value => '1'}], data[:app_ids]
    assert_equal [{:name => 'Provided Foo', :value => ''}], data[:app_keys]
  end

  def test_returns_bought_applications_for_buyer
    data = ApiDocs::BuyerData.new(@buyer).as_json[:results]
    assert_equal [{:name => 'Bought Bar', :value => 'bought-123'}], data[:user_keys]
    assert_equal [{:name => 'Bought Bar', :value => '1'}], data[:app_ids]
    assert_equal [{:name => 'Bought Bar', :value => 'bought-123-app-key'}], data[:app_keys]
  end

  def test_returns_correct_data_for_buyer
    instance = ApiDocs::BuyerData.new(@buyer)
    data = instance.as_json[:results]
    keys = instance.data_items

    keys.each do |key|
      assert data.has_key?(key.to_sym)
    end
  end

  def test_returns_correct_data_for_provider
    instance = ApiDocs::ProviderData.new(@provider)
    data = instance.as_json[:results]
    keys = instance.data_items

    keys.each do |key|
      assert data.has_key?(key.to_sym)
    end
  end

end
