require 'test_helper'

module PaymentGateways
  class Adyen12CryptTest < ActiveSupport::TestCase
    include ActiveMerchantTestHelpers
    include ActiveMerchantTestHelpers::Adyen12

    def setup
      user = mock
      @account = mock
      attributes = {
        payment_gateway_type: :adyen12,
        payment_gateway_options: {
          merchantAccount: '12345',
          login: 'hello',
          password: 'world'
        }
      }
      @provider_account = FactoryGirl.build_stubbed(:simple_provider, attributes)
      @payment_gateway = @provider_account.payment_gateway

      @account.stubs(provider_account: @provider_account, id: 'account-id')
      user.stubs(account: @account, email: 'email@example.com')

      @adyen = PaymentGateways::Adyen12Crypt.new(user)
    end

    test '#test? inherits from Active Merchant gateway_mode' do
      ActiveMerchant::Billing::Base.stubs(:gateway_mode).returns(:test)
      assert @adyen.test?
      assert @adyen.gateway_client.test?

      ActiveMerchant::Billing::Base.stubs(:gateway_mode).returns(:production)
      refute @adyen.test?
      refute @adyen.gateway_client.test?
    end

    def test_authorize_with_encrypted_card
      expected_options = {
        shopperEmail: 'email@example.com',
        shopperReference: PaymentGateways::BuyerReferences.buyer_reference(@account, @provider_account),
        shopperIP: nil,
        recurring: 'RECURRING',
        reference: PaymentGateways::BuyerReferences.recurring_authorization_reference(@account, @provider_account),
        shopperIP: '123.124.125.126'
      }

      @payment_gateway.expects(:authorize_recurring).with(0, 'encrypted_data', expected_options)
      @adyen.authorize_with_encrypted_card('encrypted_data', ip: '123.124.125.126')
    end

    def test_retrieve_card_details
      expected_details = {
        'expiryMonth' => '2',
        'expiryYear' => '2017',
        'number' => '0380'
      }
      @payment_gateway.stubs(list_recurring_details: successful_adyen_response)
      assert_equal expected_details, @adyen.retrieve_card_details
    end

    def test_store_credit_card_details_successful
      details = {
        'expiryMonth' => '2',
        'expiryYear' => '2017',
        'number' => '0380'
      }
      reference = PaymentGateways::BuyerReferences.buyer_reference(@account, @provider_account)
      @adyen.stubs(retrieve_card_details: details)
      @account.expects(:credit_card_expires_on_month=).with '2'
      @account.expects(:credit_card_expires_on_year=).with '2017'
      @account.expects(:credit_card_partial_number=).with '0380'
      @account.expects(:credit_card_auth_code=).with reference
      @account.expects(:save)
      @adyen.store_credit_card_details
    end

    def test_store_credit_card_details_failing
      details = {
      }
      @adyen.stubs(retrieve_card_details: details)
      @account.expects(:credit_card_expires_on_month=).never
      @account.expects(:credit_card_expires_on_year=).never
      @account.expects(:credit_card_partial_number=).never
      @account.expects(:credit_card_auth_code=).never
      @account.expects(:save).never
      @adyen.store_credit_card_details
    end

  end
end
