require 'test_helper'

class ProxyConfigTest < ActiveSupport::TestCase

  def test_clone_to
    ProxyConfig::ENVIRONMENTS.each do |environment|
      source_env    = (ProxyConfig::ENVIRONMENTS - [environment]).sample
      source_config = FactoryGirl.create(:proxy_config, environment: source_env)

      assert_difference(ProxyConfig.where(environment: environment).method(:count)) do
        cloned_config = source_config.clone_to(environment: environment)
        assert_equal cloned_config.version, source_config.version
        assert_equal cloned_config.environment, environment

        refute source_config.clone_to(environment: environment).persisted?
      end
    end
  end

  def test_differs_from
    config_1 = FactoryGirl.build_stubbed(:proxy_config, content: json_content(hosts: []))
    assert config_1.differs_from?(nil)

    config_2 = FactoryGirl.build_stubbed(:proxy_config, content: json_content(hosts: []))
    refute config_1.differs_from?(config_2)

    config_3 = FactoryGirl.build_stubbed(:proxy_config, content: json_content(hosts: ['alaska']))
    assert config_1.differs_from?(config_3)
  end

  def test_denormalize_hosts
    config_1 = FactoryGirl.create(:proxy_config, content: json_content(hosts: []))
    assert_nil config_1.read_attribute(:hosts)

    config_2 = FactoryGirl.create(:proxy_config, content: json_content(hosts: ['alaska.com']), proxy: config_1.proxy)
    assert_equal '|alaska.com|', config_2.read_attribute(:hosts)

    config_3 = FactoryGirl.create(:proxy_config, content: json_content(hosts: ['alaska.com', 'wild.com']), proxy: config_1.proxy)
    assert_equal '|alaska.com|wild.com|', config_3.read_attribute(:hosts)
  end

  def test_by_host
    FactoryGirl.create(:proxy_config, content: json_content(hosts: ['alaska.com']))
    assert ProxyConfig.by_host('alaska.com').present?
    refute ProxyConfig.by_host('alaska.co').present?

    FactoryGirl.create(:proxy_config, content: json_content(hosts: ['wild.com', 'alex.com']))
    assert ProxyConfig.by_host('wild.com').present?
    assert ProxyConfig.by_host('alex.com').present?
  end

  def test_hosts
    config = FactoryGirl.build(:proxy_config)
    config.hosts = '|alaska|wild|'
    assert_equal ['alaska', 'wild'], config.hosts

    config.hosts = nil
    assert_equal [], config.hosts
  end

  def test_current_versions
    proxy = FactoryGirl.create(:proxy)
    FactoryGirl.create(:proxy_config, version: 1, proxy: proxy)
    assert_equal [1], ProxyConfig.current_versions.pluck(:version)

    FactoryGirl.create(:proxy_config, version: 2, proxy: proxy)
    assert_equal [2], ProxyConfig.current_versions.pluck(:version)
  end

  def test_filename
    proxy = FactoryGirl.create(:proxy)
    proxy_config = FactoryGirl.create(:proxy_config, version: 1, proxy: proxy)
    proxy.service.name = 'bla ñé bla'
    assert_equal 'apicast-config-bla-ne-bla-sandbox-1.json', proxy_config.filename
  end

  private

  def json_content(hosts: [])
    { proxy: { hosts: hosts }}.to_json
  end
end
