require 'test_helper'

class ThreeScale::RequestLogsSearchTest < ActiveSupport::TestCase
  def setup
    @response_code  = 200
    @application_id = 1000
    @timestamp      = Time.now
  end

  def search_params(params = {})
    { code: @response_code, application_id: @application_id, timestamp: @timestamp }
      .merge(params)
  end

  test 'to_hash returns an empty hash if there are no search params' do
    search = ThreeScale::RequestLogsSearch.new({})

    assert_equal ({}), search.to_hash
  end

  test 'initialize raises an exception for unknown paramso' do
    assert_raise(ArgumentError) do
      ThreeScale::RequestLogsSearch.new({ foo: "bar" })
    end
  end

  test 'when code is a string returns a "0" if it doesn\'t represent a digit' do
    search = ThreeScale::RequestLogsSearch.new(search_params(code: "200"))

    expected = { "log" => { "code" => @response_code.to_s },
                 "application_id" => @application_id,
                 "timestamp" => @timestamp }
    assert_equal expected, search.to_hash

    search = ThreeScale::RequestLogsSearch.new(search_params(code: "foo"))
    expected = { "log" => { "code" => "0" },
                 "application_id" => @application_id,
                 "timestamp" => @timestamp }
    assert_equal expected, search.to_hash
  end

  test 'when code is an integer to_hash returns a hash with a single value for code as string' do
    search = ThreeScale::RequestLogsSearch.new(search_params)

    expected = { "log" => { "code" => @response_code.to_s },
                 "application_id" => @application_id,
                 "timestamp" => @timestamp }
    assert_equal expected, search.to_hash
  end

  test 'when code has wildcard (2XX) format to_hash returns a hash with code with gte and lte keys' do
    search = ThreeScale::RequestLogsSearch.new(search_params(code: "2XX"))

    expected = { "log" => { "code" => { "$gte" => "200", "$lt" => "300" } },
                 "application_id" => @application_id,
                 "timestamp" => @timestamp }

    assert_equal expected, search.to_hash
    assert_equal expected, ThreeScale::RequestLogsSearch.new(search_params.merge(code: "2xx")).to_hash
  end

  test 'when code has different values separated by commas, to_hash returns an array with codes' do
    search = ThreeScale::RequestLogsSearch.new(search_params(code: "202, 403, 5XX"))

    expected = { "log" => { "code" => ["202", "403", { "$gte" => "500", "$lt" => "600" }] },
                 "application_id" => @application_id,
                 "timestamp" => @timestamp }

    assert_equal expected, search.to_hash
  end

  test 'when code has different values separated by commas and invalid codes, to_hash returns an array with codes' do
    search = ThreeScale::RequestLogsSearch.new(search_params(code: "202, foo, bar"))

    expected = { "log" => { "code" => ["202", "0", "0"] },
                 "application_id" => @application_id,
                 "timestamp" => @timestamp }

    assert_equal expected, search.to_hash
  end

end
