/* Perform post-reload loop optimizations.
   Copyright (C) 2001, 2003 Free Software Foundation, Inc.

This file is part of GCC.

GCC is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

GCC is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.  */

/* This pass performs some simple post-reload loop hoisting.  It is intended
   to be run between if-conversion and scheduling, since if-conversion may
   introduce loop-invariant instructions in some cases.

   The pass only tries to optimize a loop if:

   (1) it has no call or volatile instructions;
   (2) it has no inner loops;
   (3) it has at least one exit; and
   (4) its latch jumps rather than falls through to its header.

   Assuming those conditions are met, the pass will hoist instruction X if:

   (a) X does not refer to memory;
   (b) X contains at least one SET;
   (c) all the registers that X reads are loop invariants;
   (d) no other loop instruction changes the registers that X does; and
   (e) none of the registers that X changes are live at the start of the loop.

   These conditions are rather strict and few loop instructions will meet
   them without modification.  If X meets (a), (b) and (c), but doesn't meet
   (d) and (e), the pass will attempt to rename the left-hand side of X.
   Renaming will only be successful if:

   (f) X contains a single, unconditional SET, which sets a register R;
   (g) R dies in X's basic block, and all of it dies in one place;
   (h) no (conditional) instruction modifies R before it dies;
   (i) the pass can substitute register R with some other register R'.  This
       substitution must be possible on the left-hand side of X and on the
       right hand side of every instruction that uses X's definition of R.
   (j) the new instruction meets (d) and (e) above.  */

#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "tm.h"
#include "rtl.h"
#include "tm_p.h"
#include "basic-block.h"
#include "cfgloop.h"
#include "insn-config.h"
#include "recog.h"
#include "output.h"
#include "regs.h"

/* The loop we are currently processing.  */
static struct loop *this_loop;

/* An array of the basic blocks that make up THIS_LOOP, in DFS order.  */
static basic_block *this_loop_body;

/* True if the loop meets condition (1) from the comments at the top
   of this file.  */
static bool loop_insns_ok_p;

/* def_count[R] counts the number of times R appears in a
   left-hand context in THIS_LOOP.  */
static int def_count[FIRST_PSEUDO_REGISTER];

/* Likewise a right-hand context.  */
static int use_count[FIRST_PSEUDO_REGISTER];

/* The set of registers that are live on entry to the loop.  */
static regset live_on_entry;

/* The set of registers that are live on at least one exit from the loop.  */
static regset live_on_exit;

/* The loop's back edge.  ??? Probably should be part of struct loop.  */
static edge back_edge;

/* The set of registers that are set by hoisted instructions.  This
   information will eventually be propagated to the loop's basic block
   structures.  */
static regset hoisted_defs;

/* The number of instructions moved so far.  */
static int hoist_count;

/* The instruction we are currently processing.  */
static rtx this_insn;

/* True while the current instruction is a candidate for hoisting.  */
static bool this_insn_can_be_hoisted;

/* Utility functions.  */
static int note_all_regs (rtx *, void *);
static void note_all_source_regs_1 (rtx *, void *);
static void note_all_source_regs (void (*) (unsigned int));
static void note_all_dest_regs_1 (rtx, rtx, void *);
static void note_all_dest_regs (void (*) (unsigned int));
static void for_each_insn (void (*) (void));

/* Debugging functions.  */
static void dump_reg_count (const char *, int *);
static void dump_loop_info (const char *);

static void check_movable_source_reg (unsigned int);
static void check_movable_dest_reg (unsigned int);
static bool insn_source_is_movable (void);
static bool insn_dest_is_movable (void);
static bool register_available_p (unsigned int);
static bool registers_alike_p (unsigned int, rtx);
static rtx find_replacement_reg (rtx);
static void modify_reg_count (rtx, int *, int);
static bool replacement_can_be_made (rtx, rtx);
static bool try_rename (void);
static void prepare_move_reg_use (unsigned int);
static void prepare_move_reg_def (unsigned int);
static void maybe_hoist_insn (void);
static void hoist_removed_instructions (void);
static void optimize_loop (void);
static void update_basic_blocks (void);
static void record_reg_def (unsigned int);
static void record_reg_use (unsigned int);
static void analyze_insn (void);
static void analyze_loop (void);
static bool candidate_loop_p (void);

/* Print a summary of THIS_LOOP's definition or use information.
   COUNTS points to the definition or use array and NAME describes it.  */

static void
dump_reg_count (const char *name, int *counts)
{
  unsigned int regno;

  fprintf (rtl_dump_file, "%s:", name);
  for (regno = 0; regno < FIRST_PSEUDO_REGISTER; regno++)
    if (counts[regno] > 0)
      {
	fprintf (rtl_dump_file, " %s", reg_names[regno]);
	if (counts[regno] > 1)
	  fprintf (rtl_dump_file, " (%d times)", counts[regno]);
      }
  fputc ('\n', rtl_dump_file);
}

/* Print a summary of THIS_LOOP.  STAGE describes the stage we're at.  */

static void
dump_loop_info (const char *stage)
{
  fprintf (rtl_dump_file,
	   "\nLoop beginning at insn %d and ending at insn %d, %s\n",
	   INSN_UID (this_loop->header->head),
	   INSN_UID (this_loop->latch->end),
	   stage);

  dump_reg_count ("Registers defined in loop", def_count);
  dump_reg_count ("Registers used in loop", use_count);

  fprintf (rtl_dump_file, "Registers live at loop exit:");
  dump_regset (live_on_exit, rtl_dump_file);
  fputc ('\n', rtl_dump_file);

  if (hoist_count == 0)
    fprintf (rtl_dump_file, "No hoisted instructions\n");
  else
    {
      fprintf (rtl_dump_file,
	       "Number of hoisted instructions: %d\n", hoist_count);

      fprintf (rtl_dump_file, "Registers whose definitions were hoisted:");
      dump_regset (hoisted_defs, rtl_dump_file);
      fputc ('\n', rtl_dump_file);
    }
}

/* A for_each_rtx callback.  DATA points to a void (*) (int) function.
   Call it for each register in *X.  */

static int
note_all_regs (rtx *x, void *data)
{
  if (GET_CODE (*x) == REG)
    {
      void (*fn) (unsigned int);
      unsigned int regno, end_regno;

      fn = *(void (**) (unsigned int)) data;
      end_regno = REGNO (*x) + HARD_REGNO_NREGS (REGNO (*x), GET_MODE (*x));
      for (regno = REGNO (*x); regno < end_regno; regno++)
	fn (regno);
    }
  return 0;
}

/* A note_uses callback used by note_all_source_regs.  */

static void
note_all_source_regs_1 (rtx *source, void *data)
{
  for_each_rtx (source, note_all_regs, data);
}

/* Call FN (REGNO) for every REGNO that is used by the current instruction.  */

static void
note_all_source_regs (void (*fn) (unsigned int))
{
  note_uses (&PATTERN (this_insn), note_all_source_regs_1, &fn);
}

/* A note_stores callback used by note_all_dest_regs.  */

static void
note_all_dest_regs_1 (rtx x, rtx pat ATTRIBUTE_UNUSED, void *data)
{
  note_all_regs (&x, data);
}

/* Call FN (REGNO) for every REGNO that is set by the current instruction.  */

static void
note_all_dest_regs (void (*fn) (unsigned int))
{
  note_stores (PATTERN (this_insn), note_all_dest_regs_1, &fn);
}

/* Call FN with THIS_INSN set to each instruction in THIS_LOOP.
   It is OK for FN to remove the instruction.  */

static void
for_each_insn (void (*fn) (void))
{
  unsigned int block;
  rtx next_insn;

  for (block = 0; block < this_loop->num_nodes; block++)
    for (this_insn = this_loop_body[block]->head;
	 this_insn != NEXT_INSN (this_loop_body[block]->end);
	 this_insn = next_insn)
      {
	next_insn = NEXT_INSN (this_insn);
	if (INSN_P (this_insn))
	  fn ();
      }
}

/* Clear THIS_INSN_CAN_BE_HOISTED if register REGNO is not a loop
   invariant.  */

static void
check_movable_source_reg (unsigned int regno)
{
  if (def_count[regno] > 0)
    {
      this_insn_can_be_hoisted = false;
      if (rtl_dump_file)
	fprintf (rtl_dump_file, "insn %d: uses non-invariant register %s\n",
		 INSN_UID (this_insn), reg_names[regno]);
    }
}

/* Return true if the every register used by THIS_INSN meets the
   requirements of check_movable_source_reg.  */

static bool
insn_source_is_movable (void)
{
  this_insn_can_be_hoisted = true;
  note_all_source_regs (check_movable_source_reg);
  return this_insn_can_be_hoisted;
}

/* Check whether an instruction that clobbers REGNO can be moved; clear
   THIS_INSN_CAN_BE_HOISTED if it can't.  The instruction can only be
   moved if it is the only one in the loop that assigns to REGNO and if
   REGNO is not live at the start of the loop.  */

static void
check_movable_dest_reg (unsigned int regno)
{
  if (def_count[regno] != 1)
    {
      this_insn_can_be_hoisted = false;
      if (rtl_dump_file)
	fprintf (rtl_dump_file, "insn %d: register %s defined %d times\n",
		 INSN_UID (this_insn), reg_names[regno], def_count[regno]);
    }
  else if (REGNO_REG_SET_P (live_on_entry, regno))
    {
      this_insn_can_be_hoisted = false;
      if (rtl_dump_file)
	fprintf (rtl_dump_file, "insn %d: register %s live at start\n",
		 INSN_UID (this_insn), reg_names[regno]);
    }
}

/* Return true if every register set by THIS_INSN meets the requirements
   of check_movable_dest_reg.  */

static bool
insn_dest_is_movable (void)
{
  this_insn_can_be_hoisted = true;
  note_all_dest_regs (check_movable_dest_reg);
  return this_insn_can_be_hoisted;
}

/* True if register REGNO can act as a replacement register.  Such a
   register cannot be set or used during the loop, nor can it be live
   on entry.  It can't be a fixed register.  If isn't call-saved, it
   must already be used in this function.  */

static bool
register_available_p (unsigned int regno)
{
  return (def_count[regno] == 0
	  && use_count[regno] == 0
	  && !REGNO_REG_SET_P (live_on_entry, regno)
	  && !fixed_regs[regno]
	  && (call_used_regs[regno] || regs_ever_live[regno]));
}

/* Return true if it is likely that we can change ORIGINAL's register
   number to REGNO and still have a valid instruction.  This function
   makes no guarantee about whether such a replacement is valid, it only
   provides a simple heuristic.

   A basic requirement is that REGNO be valid for ORIGINAL's mode.  For now,
   also require that REGNO be a member of the smallest class containing
   ORIGINAL.  We might get better results if we relax this condition.  */

static bool
registers_alike_p (unsigned int regno, rtx original)
{
  return (HARD_REGNO_MODE_OK (regno, GET_MODE (original))
	  && REGNO_REG_CLASS (regno) == REGNO_REG_CLASS (REGNO (original)));
}

/* Suggest a replacement register for REG.  Return NULL if no suitable
   replacement is available.  Any suggested register will have the same mode
   and class as the original, but it is up to the caller to decide whether
   the replacement is valid.  */

static rtx
find_replacement_reg (rtx reg)
{
  unsigned int regno, nregs, count;

  /* We can't replace fixed registers.  */
  nregs = HARD_REGNO_NREGS (REGNO (reg), GET_MODE (reg));
  for (regno = REGNO (reg); regno < REGNO (reg) + nregs; regno++)
    if (fixed_regs[regno])
      return NULL;

  /* The loop invariant is that (1) registers REGNO-COUNT to REGNO-1
     (inclusive) are available as replacements and (2) the first register
     in this range is "similar" to REG, as decided by registers_alike_p.  */
  count = 0;
  for (regno = 0; regno < FIRST_PSEUDO_REGISTER; regno++)
    if (!register_available_p (regno))
      count = 0;
    else if (count > 0 || registers_alike_p (regno, reg))
      if (++count == nregs)
	return gen_rtx_REG (GET_MODE (reg), regno - count + 1);

  return 0;
}

/* For each register in REG, add AMOUNT to the register's entry in ARRAY.  */

static void
modify_reg_count (rtx reg, int *array, int amount)
{
  unsigned int regno, end_regno;

  end_regno = REGNO (reg) + HARD_REGNO_NREGS (REGNO (reg), GET_MODE (reg));
  for (regno = REGNO (reg); regno < end_regno; regno++)
    array[regno] += amount;
}

/* Go through the rest of the THIS_INSN's basic block and try to replace
   all uses of FROM by TO.  Queue all changes using recog's validate_*
   machinery.  Return true if we reached a point where FROM died
   completely.  */

static bool
replacement_can_be_made (rtx from, rtx to)
{
  rtx insn, note;

  for (insn = NEXT_INSN (this_insn);
       insn != NULL && BLOCK_FOR_INSN (insn) == BLOCK_FOR_INSN (this_insn);
       insn = NEXT_INSN (insn))
    if (INSN_P (insn))
      {
	/* Replace every right-hand occurence of FROM with TO.  Fail if
	   there are still some lingering uses of FROM, perhaps due to
	   overlapping multi-word registers.  */
	validate_replace_src_group (from, to, insn);
	if (reg_overlap_mentioned_p (from, PATTERN (insn)))
	  return false;

	/* Check for a REG_DEAD note that mentions FROM.  Stop now if
	   there is one; succeed if the killed register matches exactly.  */
	for (note = REG_NOTES (insn); note; note = XEXP (note, 1))
	  if (REG_NOTE_KIND (note) == REG_DEAD
	      && reg_overlap_mentioned_p (XEXP (note, 0), from))
	    return rtx_equal_p (XEXP (note, 0), from);

	/* If an instruction changes some part of FROM, we can't predict
	   what value it will have in subsequent instructions.  If there's
	   no REG_DEAD note for this register, the instruction is likely
	   to be a cond_exec.  */
	if (set_of (from, insn) != NULL)
	  return false;
      }
  return false;
}

/* Try to rename the destination of THIS_INSN, which is known to meet
   every requirement for hoisting except insn_dest_is_movable.  Return
   true if the renaming was successful and the instruction can now be
   hoisted.  */

static bool
try_rename (void)
{
  rtx set, from, to;
  int num_changes;

  /* The instruction must be a single SET.  */
  set = PATTERN (this_insn);
  if (GET_CODE (set) != SET)
    return false;

  /* The instruction must set a register.  */
  from = SET_DEST (set);
  if (GET_CODE (from) != REG)
    return false;

  /* There must be a credible replacement register.  */
  to = find_replacement_reg (from);
  if (to == 0)
    {
      if (rtl_dump_file)
	fprintf (rtl_dump_file,
		 "insn %d: cannot find a replacement for %s\n",
		 INSN_UID (this_insn), reg_names[REGNO (from)]);
      return false;
    }

  if (rtl_dump_file)
    fprintf (rtl_dump_file,
	     "insn %d: trying to replace register %s with %s\n",
	     INSN_UID (this_insn),
	     reg_names[REGNO (from)],
	     reg_names[REGNO (to)]);

  /* Check whether all the replacement conditions are met.  */
  if (replacement_can_be_made (from, to)
      && (num_changes = num_changes_pending (), apply_change_group ()))
    {
      /* Change the destination of the SET and update the loop information
	 accordingly.  */
      SET_DEST (set) = to;
      modify_reg_count (from, def_count, -1);
      modify_reg_count (to, def_count, 1);

      /* Update the loop's use_count array in light of the changes made by
	 replacement_can_be_made.  */
      modify_reg_count (from, use_count, -num_changes);
      modify_reg_count (to, use_count, num_changes);
      return true;
    }
  else
    {
      /* Undo any replacements that we tried to make, in case we never
	 got to the point of verifying them.  */
      cancel_changes (0);
      return false;
    }
}

/* A for_each_rtx callback.  Return 1 if *X refers to memory.  */

static int
rtx_is_mem (rtx *x, void *data ATTRIBUTE_UNUSED)
{
  return GET_CODE (*x) == MEM;
}

/* Update THIS_LOOP given that a use of REGNO is about to removed from
   the loop.  Called for each register on the right-hand side of an
   instruction that we're about to hoist.  */

static void
prepare_move_reg_use (unsigned int regno)
{
  use_count[regno]--;
}

/* Update THIS_LOOP given that we're about to hoist the (only) loop
   instruction that clobbers REGNO.  */

static void
prepare_move_reg_def (unsigned int regno)
{
  def_count[regno] = 0;
  SET_REGNO_REG_SET (hoisted_defs, regno);
}

/* Try to hoist THIS_INSN out of THIS_LOOP.  */

static void
maybe_hoist_insn (void)
{
  if (GET_CODE (this_insn) == INSN
      && (single_set (this_insn) || multiple_sets (this_insn))
      && !for_each_rtx (&PATTERN (this_insn), rtx_is_mem, 0)
      && insn_source_is_movable ()
      && (insn_dest_is_movable () || try_rename ()))
    {
      if (rtl_dump_file)
	fprintf (rtl_dump_file, "insn %d: hoisted\n", INSN_UID (this_insn));

      /* We group all the instructions we hoist in a sequence.  Start the
	 sequence now if we haven't added any instructions yet.  */
      if (hoist_count == 0)
	start_sequence ();

      /* Remove the instruction from its current position and add it to
	 the end of the hoisted sequence.  */
      remove_insn (this_insn);
      add_insn (this_insn);

      /* Update the number and set of hoisted intructions.  */
      hoist_count++;

      /* Update the loop's definition/use information.  Update the set of
	 registers whose definitions we've hoisted.  */
      note_all_source_regs (prepare_move_reg_use);
      note_all_dest_regs (prepare_move_reg_def);
    }
}

/* maybe_hoist_insn will remove each hoistable instruction and emit it
   into an open sequence.  This function completes the sequence and
   emits it at the start of the loop header.  It creates a new label
   after the hoisted instructions and redirects the latch to this
   new label.  */

static void
hoist_removed_instructions (void)
{
  regset_head used_after_hoist_head;
  regset used_after_hoist;
  rtx insn, new_header_label, hoisted_insns;
  unsigned int block, regno;
  edge fallthru;

  /* Create and emit a new label, which will be the new iteration point.  */
  new_header_label = gen_label_rtx ();
  emit_label (new_header_label);

  /* Collect all the hoisted instructions, including the new label.  */
  hoisted_insns = get_insns ();
  end_sequence ();

  /* Take the union of the registers that are used within the loop
     and those that are live on exit.  */
  used_after_hoist = INIT_REG_SET (&used_after_hoist_head);
  COPY_REG_SET (used_after_hoist, live_on_exit);
  for (regno = 0; regno < FIRST_PSEUDO_REGISTER; regno++)
    if (use_count[regno] > 0)
      SET_REGNO_REG_SET (used_after_hoist, regno);

  /* Make HOISTED_REGS live throughout the loop, then restrict all live
     sets to USED_AFTER_HOIST.  Thus all registers defined by hoisted
     instructions and used by non-hoisted instructions will become live
     throughout the loop.  Those that are defined by hoisted instructions
     and used only by other hoisted instructions will become dead throughout
     the loop. */
  for (block = 0; block < this_loop->num_nodes; block++)
    {
      basic_block bb = this_loop_body[block];

      IOR_REG_SET (bb->global_live_at_start, hoisted_defs);
      AND_REG_SET (bb->global_live_at_start, used_after_hoist);
      IOR_REG_SET (bb->global_live_at_end, hoisted_defs);
      AND_REG_SET (bb->global_live_at_end, used_after_hoist);
    }

  /* The header should start with a code label ... */
  insn = this_loop->header->head;
  if (!LABEL_P (insn))
    abort ();

  /* ... followed by a basic block note.  */
  insn = NEXT_INSN (insn);
  if (!NOTE_INSN_BASIC_BLOCK_P (insn))
    abort ();

  /* Insert the hoisted instructions after the header's basic block note.  */
  emit_insn_after (hoisted_insns, insn);

  /* Start a new basic block at the header label.  The old block will only
     contain the hoisted instructions and the original label.  FALLTHRU is
     the edge that connects the old block with the new one.  */
  fallthru = split_block (this_loop->header, PREV_INSN (new_header_label));

  /* Redirect the back edge to the new header.  */
  redirect_edge_succ (back_edge, fallthru->dest);
  redirect_jump (back_edge->src->end, new_header_label, 0);

  /* The same set of registers are live on entry to the loop as before.  */
  COPY_REG_SET (fallthru->src->global_live_at_start, live_on_entry);
}

/* Optimize THIS_LOOP, which is known to meet the conditions described
   at the top of this file.  */

static void
optimize_loop (void)
{
  /* Clear hoisting information.  */
  CLEAR_REG_SET (hoisted_defs);
  hoist_count = 0;

  /* Seach for and remove all hoistable instructions.  */
  if (rtl_dump_file)
    dump_loop_info ("before hoist");
  for_each_insn (maybe_hoist_insn);
  if (rtl_dump_file)
    dump_loop_info ("after hoist");

  /* Collect the hoisted instructions and update the life information.  */
  if (hoist_count > 0)
    hoist_removed_instructions ();
}

/* Update the intra-block life information of the blocks that we've
   modified.  */

static void
update_basic_blocks (void)
{
  basic_block bb;
  sbitmap modified;

  modified = sbitmap_alloc (last_basic_block);

  sbitmap_zero (modified);
  FOR_EACH_BB (bb)
    if (bb->flags & BB_DIRTY)
      SET_BIT (modified, bb->index);

  count_or_remove_death_notes (modified, 1);
  update_life_info (modified, UPDATE_LIFE_LOCAL,
		    PROP_DEATH_NOTES | PROP_REG_INFO);
  sbitmap_free (modified);
}

/* Record that the current instruction clobbers REGNO.  */

static void
record_reg_def (unsigned int regno)
{
  def_count[regno]++;
}

/* Record that the current instruction uses REGNO.  */

static void
record_reg_use (unsigned int regno)
{
  use_count[regno]++;
}

/* Record the definition/use information for the current instruction.
   Clear LOOP_INSNS_OK_P if the instruction is volatile or contains
   a call.  */

static void
analyze_insn (void)
{
  note_all_source_regs (record_reg_use);
  note_all_dest_regs (record_reg_def);

  if (volatile_insn_p (PATTERN (this_insn))
      || GET_CODE (this_insn) == CALL_INSN)
    loop_insns_ok_p = false;
}


/* Initialize global variables for THIS_LOOP.  */

static void
analyze_loop (void)
{
  int i;

  /* Take a copy of the registers that are live on entry.  */
  COPY_REG_SET (live_on_entry, this_loop->header->global_live_at_start);

  /* Find the set of registers that are live at at least one exit point.  */
  CLEAR_REG_SET (live_on_exit);
  for (i = 0; i < this_loop->num_exits; i++)
    IOR_REG_SET (live_on_exit,
		 this_loop->exit_edges[i]->dest->global_live_at_start);

  /* Find the edge from the latch to the header.  */
  for (back_edge = this_loop->latch->succ;
       back_edge->dest != this_loop->header;
       back_edge = back_edge->succ_next)
    ;

  /* Initialize register usage information.  Look for calls and
     volatile insns.  */
  loop_insns_ok_p = true;
  memset (&use_count, 0, sizeof (use_count));
  memset (&def_count, 0, sizeof (def_count));
  for_each_insn (analyze_insn);
}

/* True if the current loop meets conditions (1)-(4) from the comments
   at the top of this file.  */

static bool
candidate_loop_p (void)
{
  return (loop_insns_ok_p
	  && this_loop->level == 1
	  && this_loop->num_exits > 0
	  && (back_edge->flags & EDGE_FALLTHRU) == 0);
}

/* Main entry point for pass.  */

void
post_reload_loop_optimize (void)
{
  regset_head hoisted_defs_head;
  regset_head live_on_entry_head;
  regset_head live_on_exit_head;
  struct loops loops;
  unsigned int i;

  /* Initialize variables.  */
  hoisted_defs = INIT_REG_SET (&hoisted_defs_head);
  live_on_entry = INIT_REG_SET (&live_on_entry_head);
  live_on_exit = INIT_REG_SET (&live_on_exit_head);
  flow_loops_find (&loops, LOOP_TREE | LOOP_EXIT_EDGES);

  /* Process each loop.  Ignore the root element, which describes
     the whole function.  */
  for (i = 0; i < loops.num; i++)
    if (loops.parray[i] != loops.tree_root)
      {
	this_loop = loops.parray[i];
	this_loop_body = get_loop_body (this_loop);
	analyze_loop ();
	if (candidate_loop_p ())
	  optimize_loop ();
	free (this_loop_body);
      }

  /* Update the local life information of all the blocks we've modified.  */
  update_basic_blocks ();

  /* Free memory.  */
  flow_loops_free (&loops);

#ifdef ENABLE_CHECKING
  verify_flow_info ();
#endif
}
