package com.redhat.installer.postinstall;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.util.AbstractUIProcessHandler;

import java.io.*;

/**
 * Class for generating a keystore for use with password vaults
 * Since we know that this class is only going to be used within ProcessPanel.spec, we can make assumptions about the format /
 * contents of the arguements. Thus we can delimit using \t, among other shortcuts
 *
 * 
 * @author thauser
 * 
 */
public class KeystoreGenerator {
	
	private static final String KEYSTORE = "keystore";
	private static final String KEY_PASS = "keypass";
	private static final String KEY_ALG = "keyalg";
	private static final String KEY_SIZE = "keysize";
	private static final String ALIAS = "alias";
	private static final String STORE_TYPE = "storetype";
	private static final String STORE_PASS = "storepass";
	private static final String DNAME = "dname";
    private static final String VALIDITY = "validity";

	private static BufferedReader in;
	private static BufferedWriter out;

	public static boolean run(AbstractUIProcessHandler handler, String[] args) throws FileNotFoundException {

		AutomatedInstallData idata = AutomatedInstallData.getInstance();
		String workingDirectory = idata.getInstallPath()+File.separator+idata.getVariable("INSTALL_SUBPATH");
		ArgumentParser parser = new ArgumentParser();
		parser.parse(args);
		
		if (!parser.hasProperty(ALIAS) || !parser.hasProperty(STORE_PASS)){
			// fail
			//ProcessPanelHelper.printToPanel(handler,  "KeystoreGenerator called with insufficient parameters.", true);
            ProcessPanelHelper.printToPanel(handler,  idata.langpack.getString("KeystoreGenerator.paramerror"), true);
			return false;
		}
		StringBuilder sb = new StringBuilder();
		// construct the command to pass to keytool.
		sb.append("keytool\t-debug\t-genkey\t");
		appendProperty(sb, parser, ALIAS);
		appendProperty(sb, parser, DNAME);
		appendProperty(sb, parser, KEYSTORE);
		appendProperty(sb, parser, STORE_PASS);
		appendProperty(sb, parser, STORE_TYPE);
        appendProperty(sb, parser, VALIDITY);
		
		if (parser.hasProperty(KEY_PASS)){
			// we desire a unique keystore password
			appendProperty(sb, parser, KEY_PASS);
		} else {
			// use the same password for both of the passwords
			sb.append("-keypass\t");
			sb.append(parser.getProperty(STORE_PASS));
			sb.append("\t");
		}
		
		appendProperty(sb, parser, KEY_ALG);
		appendProperty(sb, parser, KEY_SIZE);		
		
		ProcessPanelHelper.printToPanel(handler, String.format(idata.langpack.getString("KeystoreGenerator.start"),parser.getProperty(KEYSTORE)), false);
		try {
			ProcessBuilder pb = new ProcessBuilder(sb.toString().split("\t"));
			pb.redirectErrorStream(true);
			Process p = pb.start();
			in = new BufferedReader(new InputStreamReader(p.getInputStream()));
			out = new BufferedWriter(new OutputStreamWriter(p.getOutputStream()));
			
			Thread eater = new Thread(new Runnable() {
				
				@Override
				public void run() {
					String line;
					try {
						while ((line = in.readLine()) != null){
							out.flush();
						}
					} catch (IOException e) {
						e.printStackTrace();
					}
				}
			});
			
			int retVal = p.waitFor();
			if (retVal != 0){
				ProcessPanelHelper.printToPanel(handler, String.format(idata.langpack.getString("KeystoreGenerator.error"),parser.getProperty(KEYSTORE)), true);
				return false;
			} else {
				ProcessPanelHelper.printToPanel(handler, String.format(idata.langpack.getString("KeystoreGenerator.success"),parser.getProperty(KEYSTORE)), false);
				return true;
			}
			//KeyTool.main(sb.toString().split("\t"));

		} catch (Exception e){
			ProcessPanelHelper.printToPanel(handler, String.format(idata.langpack.getString("KeystoreGenerator.error"),parser.getProperty(KEYSTORE)), true);
			e.printStackTrace();
			return false;
		}
	}
	private static void appendProperty(StringBuilder sb, ArgumentParser parser, String propertyName){
		sb.append("-"+propertyName+"\t");
		sb.append(parser.getProperty(propertyName));
		sb.append("\t");
	}

}
