package com.redhat.installer.postinstall;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.StringWriter;
import java.lang.reflect.Field;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.logging.Handler;
import java.util.logging.Logger;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import com.redhat.installer.util.JBossJDBCConstants;
import org.apache.tools.ant.util.TeeOutputStream;
import org.jboss.as.cli.CliInitializationException;
import org.jboss.as.cli.CommandContext;
import org.jboss.as.cli.CommandContextFactory;
import org.jboss.as.cli.CommandLineException;
import org.jboss.as.security.vault.VaultSession;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.izforge.izpack.installer.AutomatedInstallData;

//import java.util.ArrayList;

public class ServerCommands {

	// TODO: Redesign the Datasource addition method to use some kind of
	// datastructure for the properties. This will allow more
	// flexibility in the re-use and the method signature won't be ridiculous
	// however, it is also much harder to check that the properties are valid :(

	// TODO: Rethink or take a look at better return values. Perhaps a layer
	// with
	// more information rather than just returning the context integer value
	// from
	// the operation

	/** Useful commands made final, change in one place if needed */
	private static final String RUN_BATCH_CMD = "run-batch";
	private static final String RELOAD_CMD = "reload --admin-only=true";
	private static final String DOMAIN_CMD_PREFIX = "/host=master";
	private static final String SHUTDOWN_CMD = ":shutdown";

	/** Vault related variables. **/
	private static VaultSession vault = null; // the vault session, if it is chosen to be
	private String keystore, keystorePwd, alias, encrDir, salt;
	private int iterations;


	/** Command Context related variables **/
	private CommandContext context;
	private boolean isDomain; // commands need to be modified if we're a domain
	private String[] domainProfiles; // the profile the domain commands should be
									// added to

	/** This class needs to know about these. */
	//TODO: use JBossJDBCConstants? 

	/** Security-domain validation values **/
	final String[] validAuthenticationCodes = { "Client", "org.jboss.security.ClientLoginModule", "Certificate", "org.jbos.security.auth.spi.BaseCertLoginModule", "CertificateUsers",
			"org.jboss.security.auth.spi.BaseCertLoginModule", "CertificateRoles", "org.jboss.security.auth.spi.CertRolesLoginModule", "Database",
			"org.jboss.security.auth.spi.DatabaseServerLoginModule", "DatabaseCertificate", "or.jboss.securityauth.spi.DatabaseCertLoginModule", "DatabaseUsers",
			"org.jboss.security.auth.spi.DatabaseServerLoginModule", "Identity", "org.jboss.security.auth.spi.IndentityLoginModule", "Ldap", "org.jboss.security.auth.spi.LdapLoginModule",
			"LdapExtended", "org.jboss.security.auth.spi.LdapExtLoginModule", "RoleMapping", "org.jboss.security.auth.spi.RoleMappingLoginModule", "RunAs",
			"org.jboss.security.auth.spi.RunAsLoginModule", "Simple", "org.jboss.security.auth.spi.SimpleServerLoginModule", "ConfiguredIdentity",
			"org.picketbox.datasource.security.ConfiguredIdentityLoginModule", "SecureIdentity", "org.picketbox.datasource.security.SecureIdentityLoginModule", "PropertiesUsers",
			"org.jboss.security.auth.spi.PropertiesUsersLoginModule", "SimpleUsers", "org.jboss.security.auth.spi.SimpleUsersLoginModule", "LdapUsers",
			"org.jboss.security.auth.spi.LdapUsersLoginModule", "Kerberos", "com.sun.security.auth.module.K5b5LoginModule", "SPNEGOUsers", "org.jboss.security.negotiation.spnego.SPNEGOLoginModule",
			"AdvancedLdap", "org.jboss.security.negotiation.AdvancedLdapLoginModule", "AdvancedADLdap", "org.jboss.security.negotiation.AdvancedADLoginModule", "UsersRoles",
			"org.jboss.security.auth.spi.UsersRolesLoginModule" };

	final String[] validAuthorizationCodes = { "DenyAll", "org.jboss.security.authorization.modules.AllDenyAuthorizationModule", "PermitAll",
			"org.jboss.security.authorization.modules.AllPermitAuthorizationModule", "Delegating", "org.jboss.security.authorization.modules.DelegatingAuthorizationModule", "Web",
			"org.jboss.security.authorization.modules.WebAuthorizationModule", "JACC", "org.jboss.security.authorization.modules.JACCAuthorizationModule", "XACML",
			"org.jboss.security.authorization.modules.XACMLAuthorizationModule" };

	final String[] validMappingCodes = { "PropertiesRoles", "org.jboss.security.mapping.providers.role.PropertiesRolesMappingProvider", "SimpleRoles",
			"org.jboss.security.mapping.providers.role.SimpleRolesMappingProvider", "DeploymentRoles", "org.jboss.security.mapping.providers.DeploymentRolesMappingProvider", "DatabaseRoles",
			"org.jboss.security.mapping.providers.role.DatabaseRolesMappingProvider", "LdapRoles", "org.jboss.security.mapping.providers.role.LdapRolesMappingProvider" };

	final String[] validFlags = { "required", "requisite", "sufficient", "optional" };

	final String[] validTypes = { "principal", "credential", "role", "attribute" };

	/**
	 * These fields are logger related. For our uses, we simply use a List of
	 * the commands we have entered on the context, unless we're in Batch Mode;
	 * then we log the batch commands
	 */
	private String logFilePath = null;
	private List<String> naiveLogger;
	private static int exitCode = 0;
	private static int ERROR = -1;

	/**
	 * Create an instance to connect using the given username / password If the
	 * host parameter is null, the context will connect to the localhost.
	 */
	private ServerCommands(String user, char[] pwd, String host, int port, String ... profile) throws CliInitializationException {
		if (profile != null) {
			domainProfiles = profile;
			isDomain = true;
		}

		
		Map<String, String> newEnv = new HashMap<String, String>();
		newEnv.put("JBOSS_HOME", AutomatedInstallData.getInstance().getInstallPath() + File.separator + AutomatedInstallData.getInstance().getVariable("INSTALL_SUBPATH") + File.separator);
		addEnv(newEnv);

		context = CommandContextFactory.getInstance().newCommandContext(host, port, user, pwd);
		context.setSilent(false);
		context.setResolveParameterValues(false);
		naiveLogger = new ArrayList<String>(1);
	}

	/**
	 * Method to return an instance of ServerCommands that connects to a domain
	 * controller using a username and password
	 * 
	 * @param user
	 *            The username for authenticating with the management interface
	 *            of the server in question
	 * @param pwd
	 *            The password associated with the given username
	 * @param port
	 *            The port upon which the management interface is listening
	 * @param profile
	 *            The profile(s) that the Context should connect to on the host
	 *            controller
	 */

	public static ServerCommands createLocalDomainUsernameSession(String user, char[] pwd, int port, String ... profile) throws CliInitializationException {
		return new ServerCommands(user, pwd, null, port, profile);
	}

	/**
	 * Method to return an instance of ServerCommands that connects to a
	 * standalone server using a username and password
	 * 
	 * @param user
	 *            The username for authenticating with the management interface
	 *            of the server in question
	 * @param pwd
	 *            The password associated with the given username
	 * @param port
	 *            The port upon which the management interface is listening
	 */
	public static ServerCommands createLocalStandaloneUsernameSession(String user, char[] pwd, int port) throws CliInitializationException {
		return new ServerCommands(user, pwd, null, port, (String[])null);
	}

	/**
	 * Method to return an instance of ServerCommands that connects to a remote
	 * management interface, using username and password authentication.
	 * 
	 * @param user
	 *            The username for authenticating with the management interface
	 *            of the server in question
	 * @param pwd
	 *            The password associated with the given username
	 * @param port
	 *            The port upon which the management interface is listening
	 * @param profile
	 *            The profile that the Context should connect to. Only
	 *            applicable if the server is running in domain mode. null value
	 *            means remote server is in standalone mode
	 * @param host
	 *            The host upon which the management interface is running
	 */
	public static ServerCommands createRemoteUsernameSession(String user, char[] pwd, int port, String profile, String host) throws CliInitializationException {
		return new ServerCommands(user, pwd, host, port, profile);
	}

	
	/**
	 * Returns a general session that makes no assumptions about what kind of server that it's on.
	 * @throws CliInitializationException 
	 * 
	 */
	
	public static ServerCommands createSession(String user, char[] pwd, int port) throws CliInitializationException{
		return new ServerCommands(user, pwd, null, port, (String[])null);
	}
	/**
	 * Calling this method creates a new batch. A batch is a construct that
	 * takes all subsequent commands into a list, and upon calling the
	 * executeBatch() method, all of the commands in the currently active batch
	 * will be run consecutively as an atomic unit. This means that any failures
	 * in the commands in the batch will cause all of the other commands to be
	 * rolled back. If the commands to be run include a :reload command, it is
	 * recommended to use the Batch functionality TODO: get more of the batch
	 * manager functionality in the ServerCommands code.
	 */
	public void createNewBatch() {
		context.getBatchManager().activateNewBatch();
	}
	
	/**
	 * Stores the current batch with a given name to be executed later. 
	 * @param name
	 */
	public void storeCurrentBatch(String name){
		context.getBatchManager().holdbackActiveBatch(name);
	}

	/**
	 * Get the list of domain profiles that this context is modifying.
	 * 
	 * @return
	 */
	public String[] getDomainProfiles() {
		return domainProfiles;
	}

	/**
	 * Set the domain profile that this context is modifying. Do not use this in
	 * between issuing commands.
	 * 
	 * @param domainProfile
	 */
	public void setDomainProfiles(String ... domainProfile) {
		this.domainProfiles = domainProfile;
	}
	
	/**
	 * Set the flag for the context to evaluate properties within the commands. Defaults to false.
	 * @param resolve
	 */
	
	public void setResolveParameterValues(boolean resolve){
		context.setResolveParameterValues(resolve);
	}

	/**
	 * If true, the context is currently using a password vault to mask
	 * passwords. This value is set only by the createVaultSession() method.
	 * 
	 * @return
	 */
	public boolean hasVault() {
		return vault != null;
	}

	/**
	 * Sets the location to output the logfile. If this location is not set (it
	 * is not by default), no log will be output.
	 * 
	 * @param string
	 */
	public void setLogFilePath(String loc) {
		this.logFilePath = loc;
	}

	/**
	 * Gets the current logfile location
	 * 
	 * @return
	 */
	public String getLogFilePath() {
		return logFilePath;
	}

	/**
	 * Disconnects the context from the host. The hostname and port are not
	 * retained.
	 */
	public void terminateSession() {
		context.terminateSession();
	}

	/**
	 * Connects the controller to the server that the commandContext was
	 * instantiated with. also loads in all commands that were accumulated in
	 * disconnected batch mode.
	 */
	public void connectContext() throws CommandLineException {
		if (context.getControllerHost() == null && context.isBatchMode()) {
			// we were in disconnected batch mode.
			context.connectController();
			// add all of the commands in the log to the real batch.
			// these should all be safe already
			for (String command : naiveLogger) {
				try {
					context.handle(command);
				} catch (CommandLineException e){
                    e.printStackTrace();
					exitCode = ERROR;
				}
			}
		} else {
			// nothing fancy needed
			context.connectController();
		}
	}

	/**
	 * Connects the context to the specified host and port. Uses the same
	 * authentication as when the instance was created
	 * 
	 * @throws CommandLineException
	 */
	public void connectContext(String host, int port) throws CommandLineException {
		context.connectController(host, port);
	}

	/**
	 * Creates a VaultSession object which allows the ServerCommands instance to
	 * mask plaintext passwords. Exception handling is deferred to the caller. We made this static because of changes in the PostInstall that 
	 * require changes to all profiles / server xml descriptors.
	 * 
	 * @param keystore
	 *            The location of the keystore on the filesystem that is to be
	 *            used to store the vault
	 * @param keystorePwd
	 *            The password for the given keystore
	 * @param encrDir
	 *            The directory into which the vault should store files
	 * @param salt
	 *            The salt for the Vault. This must be 8 characters long and can
	 *            be alphanumeric
	 * @param iterations
	 *            The iteration count.
	 * @param alias
	 *            The alias of the keystore.
	 **/

	public void createVaultSession(String keystore, String keystorePwd, String encrDir, String salt, int iterations, String alias) throws Throwable {
		this.keystore = keystore;
		this.keystorePwd = keystorePwd;
		this.encrDir = encrDir;
		this.salt = salt;
		this.iterations = iterations;
		this.alias = alias;

		if (!encrDir.endsWith(File.separator))
			encrDir += File.separator;
		if (!hasVault()){
			vault = new VaultSession(keystore, keystorePwd, encrDir, salt, iterations);	
			vault.startVaultSession(alias);
		}
	}

	/**
	 * Install the password vault onto the server our context is connected to.
	 * We use the same details as the vault that has been initialized, if any.
	 * Error handling is deferred to the caller via the exit code. It is
	 * recommended to use this method only while using Batch mode. If this
	 * method is called, any subsequent calls to methods that contain password
	 * fields (Datasources, SSL, LDAP) will attempt to use the vault to mask the
	 * password. However, if the target host has not been issued a :reload
	 * command in between adding the vault and submitting the next command with
	 * a password, the command may fail because the Vault properties are not yet
	 * visible to the host.
	 * 
	 * @return -1 if there is no vault connection and hence no vault information
	 */

	public int installVault() {
		if (!hasVault()) {
			return -1;
		}
		String masked = vault.getKeystoreMaskedPassword();
        String addVaultCmd = "/core-service=vault:add(vault-options=[(\"KEYSTORE_URL\" => \""
                + keystore.replaceAll("\\\\",
                        "\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\")
                + "\"), "
                + "(\"KEYSTORE_PASSWORD\" => \""
                + masked
                + "\"), "
                + "(\"KEYSTORE_ALIAS\" => \""
                + alias
                + "\"), "
                + "(\"SALT\" => \""
                + salt
                + "\"), "
                + "(\"ITERATION_COUNT\" => \""
                + iterations
                + "\"), "
                + "(\"ENC_FILE_DIR\" => \""
                + encrDir
                        .replaceAll("\\\\", "\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\")
                + "\")])";

		submitCommand(addVaultCmd);
		// we probably shouldn't force this
		// submitCommand(RELOAD_CMD);

		return exitCode;
	}

    /**
     * This allows us to install the vault with different locations for the keystore / encrDir. Useful when the location of the keystore / encrDir contain
     * server variables.
     * @param keystore
     * @param encrDir
     * @return
     */

    public int installVault(String keystore, String encrDir){
        if (!hasVault()){
            return -1;
        }

        String masked = vault.getKeystoreMaskedPassword();
        String addVaultCmd = "/core-service=vault:add(vault-options=[(\"KEYSTORE_URL\" => \""
                + keystore.replaceAll("\\\\",
                "\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\")
                + "\"), "
                + "(\"KEYSTORE_PASSWORD\" => \""
                + masked
                + "\"), "
                + "(\"KEYSTORE_ALIAS\" => \""
                + alias
                + "\"), "
                + "(\"SALT\" => \""
                + salt
                + "\"), "
                + "(\"ITERATION_COUNT\" => \""
                + iterations
                + "\"), "
                + "(\"ENC_FILE_DIR\" => \""
                + encrDir
                .replaceAll("\\\\", "\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\")
                + "\")])";
        submitCommand(addVaultCmd);
        // we probably shouldn't force this
        // submitCommand(RELOAD_CMD);

        return exitCode;
    }

	/**
	 * Create a module.xml at a given basePath+modulePath location.
	 * 
	 * @param moduleName
	 *            the name of the module inside the xml
	 * @param resourceNames
	 *            an array of Strings that represent paths to files that are to
	 *            be placed within basePath+modulePath.
	 * @param dependencyNames
	 *            an array of Strings that name other modules as dependencies
	 *            for this one
	 * @throws ParserConfigurationException
	 *             a DocumentBuilder cannot be provided for the given settings
	 * @throws TransformerException
	 *             uncrecoverable error while transforming XML source into
	 *             Result
	 * @throws IOException
	 *             error occured while writing Result
	 */
	public void createModuleXml(String basePath, String modulePath, String moduleName, List<String> resourceNames, List<String> dependencyNames) throws ParserConfigurationException,
			TransformerException, IOException {
		File module = new File(basePath + modulePath + File.separator + "module.xml");

		DocumentBuilderFactory docFactory = DocumentBuilderFactory.newInstance();
		DocumentBuilder docBuilder = docFactory.newDocumentBuilder();

		// module element and attributes
		Document doc = docBuilder.newDocument();
		Element root = doc.createElement("module");
		Attr xmlns = doc.createAttribute("xmlns");
		Attr modName = doc.createAttribute("name");
		xmlns.setValue("urn:jboss:module:1.0");
		modName.setValue(moduleName);
		root.setAttributeNode(xmlns);
		root.setAttributeNode(modName);
		doc.appendChild(root);

		// resources element and attributes (child of root)
		Element resources = doc.createElement("resources");
		root.appendChild(resources);

		// resource-root element and attributes (child of resources)
		for (String res : resourceNames) {
			Element resourceRoot = doc.createElement("resource-root");
			Attr resourcePath = doc.createAttribute("path");
			resourcePath.setValue(res);
			resourceRoot.setAttributeNode(resourcePath);
			resources.appendChild(resourceRoot);
		}

		// dependencies element and attributes (child of root)
		Element dependencies = doc.createElement("dependencies");
		root.appendChild(dependencies);

		// module element and attributes (child of dependencies)
		for (String dep : dependencyNames) {
			Element dependency = doc.createElement("module");
			Attr depName = doc.createAttribute("name");
			depName.setValue(dep);
			dependency.setAttributeNode(depName);
			dependencies.appendChild(dependency);
		}

		TransformerFactory tFactory = TransformerFactory.newInstance();
		tFactory.setAttribute("indent-number", 4);
		Transformer trans = tFactory.newTransformer();
		trans.setOutputProperty(OutputKeys.INDENT, "yes");
		DOMSource source = new DOMSource(doc);

		// Write to file from a String to preserve indentation
		StreamResult result = new StreamResult(new StringWriter());
		trans.transform(source, result);

		String outputString = result.getWriter().toString();
		BufferedWriter writeOut = null;
		try {
			writeOut = new BufferedWriter(new FileWriter(module));
			writeOut.write(outputString);
		} finally {
			if (writeOut != null) {
				writeOut.close();
			}
		}
	}

	/**
	 * Helper method that attempts to return a masked password from the vault.
	 * 
	 * @param vaultBlock
	 *            the vaultBlock for this password (used for identification)
	 * @param vaultAttribute
	 *            the vaultAttribute for this password (more specific
	 *            identification)
	 * @param password
	 *            the password to be masked
	 *            @return the full VAULT value, or the empty string if no vault is defined
	 */

	public String maskPassword(String vaultBlock, String vaultAttribute, String password) {
		String ret = "";
		if (vault != null) {
			try {
				ret = "${" + vault.addSecuredAttribute(vaultBlock, vaultAttribute, password.toCharArray()) + "}";
				// }
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return ret;
	}

    public String maskPasswordPlain(String vaultBlock, String vaultAttribute, String password){
        String ret = "";
        if (vault != null){
            try {
                ret = vault.addSecuredAttribute(vaultBlock, vaultAttribute, password.toCharArray());
            } catch (Exception e){
                e.printStackTrace();
            }
        }
        return ret;
    }


	/**
	 * Configure the management interfaces to use an SSL cert If a vault is
	 * configured in this instance of ServerCommands, the given password will be
	 * passed through the vault before the SSL element is configured. TODO:
	 * perhaps paramaterize the vaultBlock and/or AttributeName
	 * 
	 * @param keystore
	 *            Location of the keystore
	 * @param keystorePwd
	 *            Password to the keystore
     * @param securityRealm
     *            the name of the security realm to install the ssl cert on
	 */
	public int installSsl(String keystore, String keystorePwd, String securityRealm) {
		String password = keystorePwd;
		if (hasVault()) {
			password = maskPassword("ssl", "password", password);
		}

        String addSslCmd = "/core-service=management/security-realm="+securityRealm+"/server-identity=ssl:add(keystore-path=\""
                + keystore.replaceAll("\\\\",
                        "\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\")
                + "\",keystore-password=\"" + password + "\")";

		return submitCommand(addSslCmd);
	}


    /**
     * Installs the SSL cert to the ManagementRealm security realm
     * @param keystore
     * @param keystorePassword
     */
    public int installSsl(String keystore, String keystorePassword) {
        return installSsl(keystore, keystorePassword, "ManagementRealm");
    }

    /**
     * Removes the http-interface and adds a secure http-interface.
     * Used during configuration of SSL cert.
     */
    public int addHttps() {
        String removeCommand = "/core-service=management/management-interface=http-interface:remove()";
        String addCommand = "/core-service=management/management-interface=http-interface:add(secure-socket-binding=management-https,security-realm=\"ManagementRealm\")";
        submitCommand(removeCommand);
        return submitCommand(addCommand);
    }

	/**
	 * Helper method to submit a command to the context. This method exists to
	 * help with logging all the commands, instead of explicitly adding log
	 * statements to each command.
	 * 
	 * @param string
	 */
	private int submitCommand(String cmd) {
		// modify the command for domain usage if it needs it
		if (isDomain) {
			if (cmd.contains("subsystem")) {
				String cmdReplace = "/profile=%s"+cmd;
				for (String profile : domainProfiles) {
					cmd = String.format(cmdReplace,profile);
					// run the command for each profile
					addToLoggerAndHandle(cmd);
/*					naiveLogger.add(cmd);
					if (context.getControllerHost() == null && context.isBatchMode()) {
						// we're in disconnected batch mode. we don't add the
						// commands here.
						// instead,
						// they are added in the connectContext() call, from the
						// naiveLogger's list
					} else {
						context.handleSafe(cmd);
					}*/
				}
			} else if (cmd.contains(RELOAD_CMD)) {
				cmd = RELOAD_CMD + " --host=master";
				addToLoggerAndHandle(cmd);
			} else if (cmd.contains("core-service") || cmd.contains(SHUTDOWN_CMD)) {
				cmd = DOMAIN_CMD_PREFIX + cmd;
				addToLoggerAndHandle(cmd);
			} else {
				// the logic is destroyed because of the new "apply domain commands to every profile in domainProfiles"
				addToLoggerAndHandle(cmd);
			}
			
		} else{
			// simply add the command to the logger as is; no modification required
			addToLoggerAndHandle(cmd);
/*			naiveLogger.add(cmd);
			if (context.getControllerHost() == null && context.isBatchMode()) {
				// we're in disconnected batch mode. we don't add the commands here.
				// instead,
				// they are added in the connectContext() call, from the
				// naiveLogger's list
			} else {
				context.handleSafe(cmd);
			}*/
		}
		// if in batched mode, getExitCode will indicate validity of the
		// command. if
		// not in batched mode, the it will indicate success of the command.
		// additionally, a run-batch command will execute the batch, at which
		// time getExitCode() will indicate
		// the success or failure of that batch.

		return exitCode;
	}
	
	/**
	 * Helper method to make the submitCommand method less cluttered.
	 * Adds the given command to the logger, and also calls handleSafe with it, 
	 * as long as it is safe to do so.
	 * @param command
	 */
	private void addToLoggerAndHandle(String command){
		naiveLogger.add(command);
		if (context.getControllerHost() == null && context.isBatchMode()) {
			// we're in disconnected batch mode. we don't add the
			// commands here.
			// instead,
			// they are added in the connectContext() call, from the
			// naiveLogger's list
		} else {
			context.setSilent(false);
			Logger system = Logger.getLogger("");
			for (Handler handle :  system.getHandlers())
				system.removeHandler(handle);
			
			try {
				context.handle(command);
			} catch (CommandLineException e) {
                e.printStackTrace();
				exitCode = ERROR;
			}
		}
	}
	
	/**
	 * This method takes a List of commands and attempts to submit them all in order, treating
	 * each element in the List as a command to be run. If the ServerCommands instance is in 
	 * batch mode, all commands in the List will be added to the batch.
	 * @param commands
	 */
	public void runCommandsInList(List<String> commands){
		for (String cmd : commands){
			submitCommand(cmd);
		}
	}
	
	/**
	 * Treats the given file as a list of commands. Each line is interpreted as a new command. The expected
	 * file format follows the spec of the jboss-cli.sh itself, with some improvements (comments can be placed 
	 * into the file even in batch mode, for instance.)<br/>
	 * Typical file format caveats are:<br/>
	 * 1. # denotes a comment. <br/>
	 * 2. Empty lines are ignored. <br/> 
	 * 3. Freespace before the first character or after the last character is ignored. <br/>
	 * 
	 * @param commandFile
	 */
	public void runCommandsInFile(File file){
		ArrayList<String> commands = new ArrayList<String>();
		String line = "";
		try {
		BufferedReader br = new BufferedReader(new FileReader(file));
			while ((line = br.readLine()) != null){
				if (line.trim().startsWith("#") || line.trim().isEmpty()){
					continue; // ignore comments and empty lines.
				}
				commands.add(line.trim());
			}
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		// all lines in the file read. time to run them.
		runCommandsInList(commands);
	}

	/**
	 * 
	 * Treats the given file as a list of commands. Each line is interpreted as a new command. The expected
	 * file format follows the spec of the jboss-cli.sh itself, with some improvements (comments can be placed 
	 * into the file even in batch mode, for instance.)<br/>
	 * Typical file format caveats are:<br/>
	 * 1. # denotes a comment. <br/>
	 * 2. Empty lines are ignored. <br/> 
	 * 3. Freespace before the first character or after the last character is ignored. <br/>
	 * Identical to runCommandsInFile(File), but using a String to point to a path instead of a File object
	 * @param path
	 */
	public void runCommandsInFile(String path){
		runCommandsInFile(new File(path));
	}
	/**
	 * Install a JDBC Driver using the supplied values
	 * 
	 */

	public int installJdbcDriver(String jdbcName, String jdbcModuleName, String jdbcXaDsName)
    {
		String addJdbcCmd
                = "/subsystem=datasources/jdbc-driver=" + jdbcName +
                  ":add(driver-name=\"" + jdbcName + "\",driver-module-name=\"" + jdbcModuleName + "\",driver-xa-datasource-class-name=\"" + jdbcXaDsName + "\")";
		return submitCommand(addJdbcCmd);
	}

	/**
	 * Install an xa datasource using a username and password instead of
	 * Security Domain
	 * 
	 * @param dsName
	 *            name of the datasource
	 * @param dsJndiName
	 *            JNDI name for the datasource
	 * @param driverName
	 *            the name of the driver this datasource should use
	 * @param dsMinPool
	 *            the minimum pool size
	 * @param dsMaxPool
	 *            the maximum pool size
	 * @param dsUsername
	 *            the username for the datasource to be secured by
	 * @param dsPassword
	 *            the password for the username the datasource is to be secured
	 *            by
	 * @param xaProps
	 *            a map of XA properties where the key is the property name, and
	 *            the value associated with the key is the property value
	 * @param dsXaRecoveryUser
	 *            the recovery user for the XA datasource
	 * @param dsXaRecoveryPass
	 *            the recovery password for the recovery user
	 */
	public int installXaDatasourceUsernamePwd(String dsName,
			String dsJndiName,
			String driverName,
			String dsMinPool,
			String dsMaxPool,
			String dsUsername,
			String dsPassword,
			Map<String, String> xaProps,
			String dsXaRecoveryUser,
			String dsXaRecoveryPass) {
		return installDatasource(dsName, dsJndiName, driverName, null, dsMinPool, dsMaxPool, null, dsUsername, dsPassword, xaProps, dsXaRecoveryUser, dsXaRecoveryPass);
	}

	/**
	 * Install an xa datasource using a Security Domain instead of username and
	 * password
	 * 
	 * @param dsName
	 *            name of the datasource
	 * @param dsJndiName
	 *            JNDI name for the datasource
	 * @param driverName
	 *            the name of the driver this datasource should use
	 * @param dsMinPool
	 *            the minimum pool size
	 * @param dsMaxPool
	 *            the maximum pool size
	 * @param dsSecurityDomain
	 *            the security domain to use to secure the datasource
	 * @param xaProps
	 *            a map of XA properties where the key is the property name, and
	 *            the value associated with the key is the property value
	 * @param dsXaRecoveryUser
	 *            the recovery user for the XA datasource
	 * @param dsXaRecoveryPass
	 *            the recovery password for the recovery user
	 */
	public int installXaDatasourceSecurityDomain(String dsName,
			String dsJndiName,
			String driverName,
			String dsMinPool,
			String dsMaxPool,
			String dsSecurityDomain,
			Map<String, String> xaProps,
			String dsXaRecoveryUser,
			String dsXaRecoveryPass) {
		return installDatasource(dsName, dsJndiName, driverName, null, dsMinPool, dsMaxPool, dsSecurityDomain, null, null, xaProps, dsXaRecoveryUser, dsXaRecoveryPass);
	}

	/**
	 * Install a datasource using a username and password instead of a security
	 * domain. If the ServerCommands instance has a Vault initialized and
	 * installed, the given password will be masked using the Vault
	 * 
	 * @param dsName
	 *            name of the datasource
	 * @param dsJndiName
	 *            JNDI name for the datasource
	 * @param driverName
	 *            the name of the driver this datasource should use
	 * @param connectionUrl
	 *            the URL for the database
	 * @param dsMinPool
	 *            the minimum pool size
	 * @param dsMaxPool
	 *            the maximum pool size
	 * @param dsUsername
	 *            the username for the datasource to be secured by
	 * @param dsPassword
	 *            the password for the username the datasource is to be secured
	 *            by
	 * 
	 */
	public int installDatasourceUsernamePwd(String dsName, String dsJndiName, String driverName, String connectionUrl, String dsMinPool, String dsMaxPool, String dsUsername, String dsPassword) {
		return installDatasource(dsName, dsJndiName, driverName, connectionUrl, dsMinPool, dsMaxPool, null, dsUsername, dsPassword, null, null, null);
	}

	/**
	 * Install a datasource using a security domain.
	 * 
	 * @param dsName
	 *            name of the datasource
	 * @param dsJndiName
	 *            JNDI name for the datasource
	 * @param driverName
	 *            the name of the driver this datasource should use
	 * @param connectionUrl
	 *            the URL for the database
	 * @param dsMinPool
	 *            the minimum pool size
	 * @param dsMaxPool
	 *            the maximum pool size
	 * @param dsSecurityDomain
	 *            the security domain to use to secure the datasource
	 * 
	 */
	public int installDatasourceSecurityDomain(String dsName, String dsJndiName, String driverName, String connectionUrl, String dsMinPool, String dsMaxPool, String dsSecurityDomain) {
		return installDatasource(dsName, dsJndiName, driverName, connectionUrl, dsMinPool, dsMaxPool, dsSecurityDomain, null, null, null, null, null);
	}
	
	/**
	 * This method is used to add a datasource with minimal required information from the user.
	 * For datasources that need to use a security domain for authentication
	 * @param dsName
	 * @param dsJndiName
	 * @param driverName
	 * @param connectionUrl
	 * @param dsSecurityDomain
	 * @return
	 */
	public int installDatasourceSecurityDomainMinimal(String dsName, String dsJndiName, String driverName, String connectionUrl, String dsSecurityDomain){
		return installDatasource(dsName, dsJndiName, driverName, connectionUrl, null, null, dsSecurityDomain, null, null, null, null, null); 			
	}
	
	/**
	 * This method is used to add a datasource with minimal required information from the user. For datasources
	 * that need to use a username + password authentication mechanism
	 * @param dsName
	 * @param dsJndiName
	 * @param driverName
	 * @param connectionUrl
	 * @param dsUsername
	 * @param dsPassword
	 * @return
	 */
	public int installDatasourceUserPwdMinimal(String dsName, String dsJndiName, String driverName, String connectionUrl, String dsUsername, String dsPassword){
		return installDatasource(dsName, dsJndiName, driverName, connectionUrl, null, null, null, dsUsername, dsPassword, null, null, null); 			
	}

	/**
	 * This method should only be called by the more specific public methods, in
	 * order to more closely control the values being passed into this method.
	 * Install a datasource using the supplied values. If xaProps is null, the
	 * datasource will be added as a non-XA one. Additionally, if the
	 * ServerCommands instance has an associated VaultSession, and a dsPassword
	 * is specified, this password will be masked using the vault with the
	 * vaultBlock = "ds_" and the vaultAttribute = "password"<br>
	 * Special return values:<br>
	 * -1 - One or all of dsName, dsJndiName, or driverName are null.<br>
	 * -2 - Both connectionUrl and xaProps are null.<br>
	 * -3 - Both dsUsername and dsPassword, as well as dsSecurityDomain are null<br>
	 * 
	 * @param dsName
	 *            name of the datasource
	 * @param dsJndiName
	 *            jndiname of the datasource
	 * @param driverName
	 *            the driver name this datasource should use
	 * @param connectionUrl
	 *            URL of the database that this datasource should use
	 * @param dsSecurityDomain
	 *            Name of the security-domain this datasource should be secured
	 *            by. dsSecurityDomain == null iff dsUsername != null &&
	 *            dsPassword != null
	 * @param dsUsername
	 *            username for the user to secure the datasource. dsUsername ==
	 *            null iff dsSecurityDomain != null
	 * @param dsPassword
	 *            password for the user to secure the datasource. dsPassword ==
	 *            null iff dsSecurityDomain != null
	 * @param dsMinPool
	 *            the minimum pool size for this datasource
	 * @param dsMaxPool
	 *            the maximum pool size
	 * @param dsSecurityDomain
	 *            the security domain to use to secure the datasource. Can be
	 *            null iff dsUsername and dsPassword are not null.
	 * @param dsUsername
	 *            the username. can be null iff dsSecurityDomain is not null
	 * @param dsPassword
	 *            the password. can be null iff dsSecurityDomain is not null
	 * @param xaProps
	 *            a map of XA properties where the key is the property name, and
	 *            the value associated with the key is the property value
	 * @param dsXaRecoveryUser
	 *            the user to be used for recovery in an XA datasource
	 * @param dsXaRecoveryPass
	 *            the password to be used with the recovery user for an XA
	 *            datasource
	 * 
	 * @return
	 */
	private int installDatasource(String dsName,
			String dsJndiName,
			String driverName,
			String connectionUrl,
			String dsMinPool,
			String dsMaxPool,
			String dsSecurityDomain,
			String dsUsername,
			String dsPassword,
			Map<String, String> xaProps,
			String dsXaRecoveryUser,
			String dsXaRecoveryPass) {

		boolean dsIsXa = false;
		String addDsCmd = "";

		// these situations can't be allowed to continue.
		if (dsName == null || dsJndiName == null || driverName == null) {
			return -1;
		}
		if (connectionUrl == null && xaProps == null) {
			return -2;
		}
		if (dsSecurityDomain == null && dsUsername == null && dsPassword == null) {
			return -3;
		}

		// if xaProps is not null, it's safe to assume the datasource we're
		// adding is an XA datasource
		if (xaProps != null) {
			dsIsXa = true;
		}

		addDsCmd += "add --name=" + dsName + " --jndi-name=\"" + dsJndiName + "\" --driver-name=" + driverName;

		if (dsMinPool != null) {
			addDsCmd += " --min-pool-size=" + dsMinPool;
		}
		if (dsMaxPool != null) {
			addDsCmd += " --max-pool-size=" + dsMaxPool;
		}
		if (dsSecurityDomain != null) {
			addDsCmd += " --security-domain=" + dsSecurityDomain;
		} else {
			if (hasVault()) {
				// only attempt this if the dsPassword is actually being used
				if (dsPassword != null) {
					dsPassword = maskPassword("ds_" + dsName, "password", dsPassword);
				}
			}
			addDsCmd += " --user-name=" + dsUsername + " --password=\"" + dsPassword + "\"";
		}

		addDsCmd = setUniqueDsElements(addDsCmd, driverName);

		if (dsIsXa) {
			addDsCmd += " --xa-datasource-properties=";
			for (String property : xaProps.keySet()) {
				if (xaProps.get(property) != null) {
                    addDsCmd += property
                            + ","
 + xaProps.get(property) + ",";
				}
			}
			addDsCmd = addDsCmd.substring(0, addDsCmd.length() - 1); // to
																		// remove
																		// the
																		// last
																		// ","
																		// from
																		// the
																		// xa
																		// properties

			if (driverName.equals(JBossJDBCConstants.ibmJdbcName)) { // ibm needs special recovery
													// plugin elements
				addDsCmd += " --recovery-plugin-class-name=org.jboss.jca.core.recovery.ConfigurableRecoveryPlugin --recovery-plugin-properties=EnabledIsValid=false,IsValidOverride=false,EnableClose=false";
			} // recovery password stuff
            // If we are using a vault, vault the password:
            if (hasVault()) {
                if (dsXaRecoveryPass != null) {
                    dsXaRecoveryPass = maskPassword("ds_" + dsName,
                            "recoveryPassword", dsXaRecoveryPass);
				}
            }

            // If null, we just ignore these.
            if (dsXaRecoveryUser != null && dsXaRecoveryPass != null) { // if

                addDsCmd += " --recovery-username=" + dsXaRecoveryUser
                        + " --recovery-password=" + dsXaRecoveryPass;
            }
		} else {
			addDsCmd += " --connection-url=\"" + connectionUrl + "\"";
		}
		// submit this crazy command
		if (isDomain) {
			addDsCmd = " --profile=%s " + addDsCmd;
		}
		if (dsIsXa) {
			addDsCmd = "xa-data-source " + addDsCmd;
		} else {
			addDsCmd = "data-source " + addDsCmd;
		}
		// special case command, cannot use the submitCommand method, because
		// the submitCommand method appends things based upon isDomain; this
		// command instead
		// has the --profile= component.
		if (isDomain) {
			for (String profile : domainProfiles) {
				addToLoggerAndHandle(String.format(addDsCmd, profile));
			}
		} else {
			addToLoggerAndHandle(addDsCmd);
		}
/*		naiveLogger.add(addDsCmd);
		// if we are connected, this will not return null
		if (context.getControllerHost() == null && context.isBatchMode()) {
			// we're disconnected and in batch mode. Thus, we do nothing, to
			// avoid an NPE
			// in the CommandContext code which cannot be fixed for this
			// version. see https://issues.jboss.org/browse/WFLY-1512
		} else {
			context.handleSafe(addDsCmd);
		}*/

		if (exitCode == 0) { // our installation succeeded OR the
											// command was successfully added to
											// batch (properly formed)
            String enableCmd = "";
            if (dsIsXa) { // enable the datasource
                enableCmd = ("/subsystem=datasources/xa-data-source=" + dsName + ":enable");
            } else {
                enableCmd = ("/subsystem=datasources/data-source=" + dsName + ":enable");
            }
            return submitCommand(enableCmd);
		}
		return exitCode;
	}

	/**
	 * Method adds all of the special classnames for different vendors.
	 * Unfortunate but necessary special case logic.
	 * 
	 * @param addDsCmd
	 * @param driverName
	 * @return the command string with added information
	 */
	private String setUniqueDsElements(String addDsCmd, String driverName) {
		// --same-rm-override doesn't seem to exist anymore in jboss-cli.sh
		String dsExceptionSorter, dsValidChecker, dsStaleChecker;
		if (driverName.equals(JBossJDBCConstants.ibmJdbcName)) {
			// addDsCmd += " --same-rm-override=false";
			dsExceptionSorter = "org.jboss.jca.adapters.jdbc.extensions.db2.DB2ExceptionSorter";
			dsValidChecker = "org.jboss.jca.adapters.jdbc.extensions.db2.DB2ValidConnectionChecker";
			dsStaleChecker = "org.jboss.jca.adapters.jdbc.extensions.db2.DB2StaleConnectionChecker";

		} else if (driverName.equals(JBossJDBCConstants.sybaseJdbcName)) {
			dsExceptionSorter = "org.jboss.jca.adapters.jdbc.extensions.sybase.SybaseExceptionSorter";
			dsValidChecker = "org.jboss.jca.adapters.jdbc.extensions.sybase.SybaseValidConnectionChecker";
			dsStaleChecker = null;

		} else if (driverName.equals(JBossJDBCConstants.mysqlJdbcName)) {
			dsExceptionSorter = "org.jboss.jca.adapters.jdbc.extensions.mysql.MySQLExceptionSorter";
			dsValidChecker = "org.jboss.jca.adapters.jdbc.extensions.mysql.MySQLValidConnectionChecker";
			dsStaleChecker = null;

		} else if (driverName.equals(JBossJDBCConstants.postgresqlJdbcName)) {
			dsExceptionSorter = "org.jboss.jca.adapters.jdbc.extensions.postgres.PostgreSQLExceptionSorter";
			dsValidChecker = "org.jboss.jca.adapters.jdbc.extensions.postgres.PostgreSQLValidConnectionChecker";
			dsStaleChecker = null;

		} else if (driverName.equals(JBossJDBCConstants.microsoftJdbcName)) {
			// addDsCmd += " --same-rm-override=false";
			dsExceptionSorter = null;
			dsValidChecker = "org.jboss.jca.adapters.jdbc.extensions.mssql.MSSQLValidConnectionChecker";
			dsStaleChecker = null;

		} else if (driverName.equals(JBossJDBCConstants.oracleJdbcName)) {
			// addDsCmd += " --same-rm-override=false";
			dsExceptionSorter = "org.jboss.jca.adapters.jdbc.extensions.oracle.OracleExceptionSorter";
			dsValidChecker = "org.jboss.jca.adapters.jdbc.extensions.oracle.OracleValidConnectionChecker";
			dsStaleChecker = "org.jboss.jca.adapters.jdbc.extensions.oracle.OracleStaleConnectionChecker";
		} else {
			dsExceptionSorter = null;
			dsValidChecker = null;
			dsStaleChecker = null;
		}

		if (dsExceptionSorter != null) {
			addDsCmd += " --exception-sorter-class-name=" + dsExceptionSorter;
		}
		if (dsValidChecker != null) {
			addDsCmd += " --valid-connection-checker-class-name=" + dsValidChecker;
		}
		if (dsStaleChecker != null) {
			addDsCmd += " --stale-connection-checker-class-name=" + dsStaleChecker;
		}
		return addDsCmd;
	}

	/**
	 * This method performs three tasks. 1. Creates a new LDAP connection using
	 * the supplied ldapName, ldapPassword, ldapUrl, and ldapAuthDn
	 * (Authentication Distinguished Name) 2. Creates a new Security Realm,
	 * named ldapRealmName, which uses the newly created LDAP connection, along
	 * with ldapBaseDn, ldapFilter, and ldapRecursive. 3. Applies this new
	 * Security Realm to the management interface.
	 * 
	 * @param ldapName
	 * @param ldapPassword
	 * @param ldapUrl
	 * @param ldapAuthDn
	 * @param ldapRealmName
	 * @param ldapBaseDn
	 * @param ldapRecursive
	 * @param ldapFilter
	 * @return
	 */

	public int installLdap(String ldapName,
			String ldapPassword,
			String ldapUrl,
			String ldapAuthDn,
			String ldapRealmName,
			String ldapBaseDn,
			String ldapFilter,
			String ldapRecursive,
			boolean isAdvancedFilter) {
		int retVal = 0;
		retVal = createLdapConn(ldapName, ldapPassword, ldapUrl, ldapAuthDn);
		if (retVal != 0)
			return retVal;
		retVal = createLdapSecurityRealm(ldapName, ldapRealmName, ldapBaseDn, ldapFilter, ldapRecursive, isAdvancedFilter);
		if (retVal != 0)
			return retVal;
		retVal = installLdapToHttpInterface(ldapRealmName);
		if (retVal != 0)
			return retVal;
		retVal = installLdapToNativeInterface(ldapRealmName);
		return retVal;
	}

	/**
	 * Creates the connection that will be referenced by the new security realm.
	 * Contains the credentials for connecting to the ldap server to perform
	 * searches
	 * 
	 * @param ldapName
	 *            name of the connection. is arbitrary
	 * @param ldapPwd
	 *            credential used for authentication with the LDAP server
	 * @param ldapUrl
	 *            location of the LDAP server, including port
	 * @param ldapDn
	 *            the user to authenticate to the LDAP server with
	 * 
	 */

	private int createLdapConn(String ldapName, String ldapPwd, String ldapUrl, String ldapDn) {
		String password = "";

		if (hasVault()) {
			password = maskPassword("ldap_" + ldapName, "password", ldapPwd);
		} else {
			password = ldapPwd;
		}
		String addLdapConnCmd = "/core-service=management/ldap-connection=" + ldapName + "/:add(search-credential=\"" + password + "\",url=\"" + ldapUrl + "\",search-dn=\"" + ldapDn + "\")";

		return submitCommand(addLdapConnCmd);
	}

	/**
	 * Method to create a new ldap security realm, and add the required
	 * characteristics to it. This method performs two steps: The creation of an
	 * LDAP security realm, and addition of the desired characteristics to the
	 * newly created LDAP realm.
	 * 
	 * @param ldapName
	 *            name of the ldap connection this security realm should use
	 * @param ldapRealmName
	 *            the name of the security realm
	 * @param ldapBaseDn
	 *            the distinguished name that the LDAP realm should begin it's
	 *            search for users at
	 * @param ldapRecursive
	 *            indicate whether or not recursive search should be used
	 * @param ldapFilter
	 *            indicates, using either an advanced filter or a simple
	 *            username-attribute, which value in the ldap server to use as a
	 *            username
	 * 
	 * @return 0: success of both operations (create and add) 1: failure of
	 *         create operation 2: success of create operation, failure of add
	 *         operation
	 * 
	 **/

	private int createLdapSecurityRealm(String ldapName, String ldapRealmName, String ldapBaseDn, String ldapFilter, String ldapRecursive, boolean isAdvancedFilter) {
		String createLdapSecRealmCmd = "/core-service=management/security-realm=\"" + ldapRealmName + "\":add";
		String addLdapSecRealmCmd = "/core-service=management/security-realm=\"" + ldapRealmName + "\"/authentication=ldap:add(base-dn=\"" + ldapBaseDn + "\", recursive=\"" + ldapRecursive
				+ "\", connection=\"" + ldapName + "\"";
		
		if (isAdvancedFilter) {
			addLdapSecRealmCmd += ",advanced-filter=\"" + ldapFilter + "\")";
		} else {
			addLdapSecRealmCmd += ",username-attribute=\"" + ldapFilter + "\")";
		}

		submitCommand(createLdapSecRealmCmd);
		if (exitCode != 0)
			return 1;

		submitCommand(addLdapSecRealmCmd);
		if (exitCode != 0)
			return 2;

		return 0; // if we got here, we're successful
	}

	/**
	 * TODO: this can be generalized into a specific instance of a general
	 * method (associate any security realm with a given interface) Associates
	 * the given security realm to the management interfaces
	 * 
	 * @param ldapRealmName
	 *            the realm to add to the management interfaces
	 * 
	 * @return the exit code of the operation on the command context
	 */
	private int installLdapToHttpInterface(String ldapRealmName) {
		String installLdapToIntCmd = "/core-service=management/management-interface=http-interface/:" + writeAttribute("security-realm", "\"" + ldapRealmName + "\"");
		return submitCommand(installLdapToIntCmd);
	}
	
	/**
	 * 
	 * @param ldapRealmName
	 *            the realm to add to the management interfaces
	 * 
	 * @return the exit code of the operation on the command context
	 */
	private int installLdapToNativeInterface(String ldapRealmName) {
		String installLdapToIntCmd = "/core-service=management/management-interface=native-interface/:" + writeAttribute("security-realm", "\"" + ldapRealmName + "\"");
		return submitCommand(installLdapToIntCmd);
	}

	/**
	 * Allows the user to configure logging on the CommandContext.
	 */
	public void setContextLoggingConfig(String file) {
		System.setProperty("logging.configuration", file);
	}

	/**
	 * Writes out a Log of the raw commands used by the CommandContext. This
	 * allows users to see what commands were executed on their host. If
	 * setLogFileLocation already exists, this method will overwrite it.
	 * 
	 * @throws IOException
	 *             either the file location has not been specified, or writing
	 *             to the given location failed
	 */

	public void writeLogFile() throws IOException {
		// if setLogFileLocation has been called
		if (logFilePath != null) {
			File logFile = new File(logFilePath);
			BufferedWriter bw = new BufferedWriter(new FileWriter(logFile));
			for (String line : naiveLogger) {
				bw.write(line + "\n");
			}
			bw.close();
		} else {
			throw new FileNotFoundException("writeLogFile: the logFileLocation has not been set");
		}
	}

	/**
	 * Issues a reload command to the currently connected host.
	 */
	public int reloadHost() {
		return submitCommand(RELOAD_CMD);
	}

	/**
	 * Issues a shutdown command to the currently connected host.
	 */

	public int shutdownHost() {
		return submitCommand(SHUTDOWN_CMD);
	}
	
	/**
	 * Utility method for use when the ServerCommands instance is connected to a domain host, but the 
	 * ServerCommands instance may not be aware of this (not created through the *DomainSession* factory methods. <br/>
	 * Explicitly calls shutdown on a domain host. This will result in failure if used on a standalone host.
	 * @return
	 */
	public int shutdownDomainHost(){
		return submitCommand(DOMAIN_CMD_PREFIX+SHUTDOWN_CMD);
	}

	/**
	 * Submits the current batch.
	 * 
	 * @throws BatchNotActiveException
	 *             if there is no active batch
	 * @throws CommandLineException
	 */
	public int runBatch() throws BatchNotActiveException, CommandLineException, BatchIsEmptyException {
		if (naiveLogger.isEmpty()) {
			throw new BatchIsEmptyException("ServerCommands: The batch is empty.");
		}
		if (context.getControllerHost() == null) {
			throw new CommandLineException("ServerCommands: You are not connected to any host. runBatch() can only be run while connected");
		}
		if (context.isBatchMode()) {
			return submitCommand(RUN_BATCH_CMD);
		} else {
			throw new BatchNotActiveException("ServerCommands: there is no currently active batch");
		}
	}
	
	/**
	 * Submits a batch with the given name. This will run the currently active batch if there is no heldback batch with
	 * the same name
	 * @param name
	 * @return
	 * @throws BatchIsEmptyException 
	 * @throws CommandLineException 
	 * @throws BatchNotActiveException 
	 */
	public int runBatch(String name) throws BatchNotActiveException, CommandLineException, BatchIsEmptyException{
		context.getBatchManager().activateHeldbackBatch(name);
		return runBatch();
	}
	
	/** 
	 * Checks if a given batch exists as either the active or heldback batch 
	 */
	public boolean isStoredBatch(String name){
		return context.getBatchManager().isHeldback(name);
	}

	/**
	 * Sets the transaction manager's default timeout to the given value TODO:
	 * complete this command
	 * 
	 * @param timeout
	 *            the desired new timeout
	 */

	public int setTransactionManagerTimeout(int timeout) {
		String setTimeoutCmd = "";
		return submitCommand(setTimeoutCmd);
	}

	/**
	 * Helper method to create a logger level with only parentHandlers set to
	 * either false or true
	 * 
	 * @param name
	 *            name of the logger level
	 * @param level
	 *            name of the logger level
	 * @param useParentHandlers
	 *            true or false; useParentHandlers or not
	 * @return
	 */

	public int createLoggerLevel(String name, String level, String useParentHandlers) {
		boolean useParents = (useParentHandlers != null) ? useParentHandlers.equals("true") : false;
		return createLoggerLevel(name, level, null, null, null, null, useParents);
	}

	// TODO: add more helpers

	/**
	 * Adds a new logger level to the connected server. All parameters can be
	 * null except for the name, which must be supplied. Non-zero exit status
	 * indicates an error has occurred.
	 * 
	 * @param name
	 *            Name of the new logger level
	 * @param level
	 *            level of the new logger level (DEBUG, TRACE etc)
	 * @param category
	 *            the category of the logger
	 * @param filter
	 *            filter for the logger
	 * @param filterSpec
	 *            specification of the filter for the logger
	 * @param handlers
	 *            handlers for the logger
	 * @param useParent
	 *            true if the parent handlers should be used for the new logger;
	 *            false otherwise TODO: perhaps make this like the add
	 *            datasource command
	 */

	public int createLoggerLevel(String name, String level, String category, String filter, String filterSpec, String handlers, boolean useParentHandlers) {
		if (name == null) {
			return 1; // fail immediately if the name is null.
		}
		String createLoggerCmd = "/subsystem=logging/logger=" + name + ":add(";

		if (level != null) {
			createLoggerCmd += "level=" + level + ",";
		}
		if (category != null) {
			createLoggerCmd += "category=" + category + ",";
		}
		if (filter != null) {
			createLoggerCmd += "filter=" + filter + ",";
		}
		if (filterSpec != null) {
			createLoggerCmd += "filter-spec=" + filterSpec + ",";
		}
		if (handlers != null) {
			createLoggerCmd += "handlers=" + handlers + ",";
		}
		if (useParentHandlers) {
			createLoggerCmd += "use-parent-handlers=true";
		} else {
			createLoggerCmd += "use-parent-handlers=false";
		}

		submitCommand(createLoggerCmd);
		return exitCode;
	}

	// this method is designed to enable use without knowledge of the underlying
	// CLI API
	/**
	 * Adds a security-domain with the given options to the server this context
	 * is connected to.TODO: add the rest of the jsse parameters, hopefully
	 * using some kind of datastructure to mitigate the large amount of
	 * parameters this would add.
	 * 
	 * @param domainName
	 *            required. Name of the security-domain
	 * @param cacheType
	 *            optional. Can take two possible values: default or infinispan.
	 *            Other values will result in no cache-type attribute being
	 *            defined.
	 * @param authenCode
	 *            optional. Required to define the authentication element
	 * @param authenFlag
	 *            optional, required if authenCode is provided. Contains one of
	 *            the following values: required, requisite, sufficient,
	 *            optional
	 * @param authenOptions
	 *            optional. Contains a map of (name,value)s for filling the
	 *            module-options element of the authentication element
	 * @param authorCode
	 *            optional. Required to define the authorization element
	 * @param authorFlag
	 *            optional, required if authorCode is provided. Contains one of
	 *            the following values: required, requisite, sufficient,
	 *            optional
	 * @param authorOptions
	 *            optional. Contains a map of (name,values)s for filling the
	 *            module-options element of the authorization element
	 * @param mappingCode
	 *            optional. Required to define the mapping element
	 * @param mappingFlag
	 *            optional, required if mappingCode is provided. Contains one of
	 *            the following values: required, requisite, sufficient,
	 *            optional
	 * @param mappingOptions
	 *            optional. Contains a map of (name,value)s for filling the
	 *            module-options element of the mapping element
	 * @return
	 */
	public int addSecurityDomain(String domainName,
			String cacheType,
			String authenCode,
			String authenFlag,
			Map<String, String> authenOptions,
			String authorCode,
			String authorFlag,
			Map<String, String> authorOptions,
			String mappingCode,
			String mappingFlag,
			Map<String, String> mappingOptions,
			Map<String, String> jsseAttrs,
			Map<String, String> jsseKeystoreAttrs,
			Map<String, String> jsseKeystoreManagerAttrs,
			Map<String, String> jsseTruststoreAttrs,
			Map<String, String> jsseTruststoreManagerAttrs,
			Map<String, String> jsseAdditionalProps) {
		int retval = createSecurityDomain(domainName, cacheType);
		if (retval != 0) {
			return retval;
		}

		if (authenCode != null && authenFlag != null) {
			retval = addSecurityDomainAuthentication(domainName, authenCode, authenFlag, authenOptions);
			if (retval != 0) {
				return retval;
			}
		}

		if (authorCode != null && authorFlag != null) {
			retval = addSecurityDomainAuthorization(domainName, authorCode, authorFlag, authorOptions);
			if (retval != 0) {
				return retval;
			}
		}

		if (mappingCode != null && mappingFlag != null) {
			retval = addSecurityDomainMapping(domainName, mappingCode, mappingFlag, mappingOptions);
			if (retval != 0) {
				return retval;
			}
		}

		// seems sketchy. needs improvement
		if (jsseAttrs != null || jsseKeystoreAttrs != null || jsseKeystoreManagerAttrs != null || jsseTruststoreAttrs != null || jsseTruststoreManagerAttrs != null) {
			retval = addSecurityDomainJsse(domainName, jsseAttrs, jsseKeystoreAttrs, jsseKeystoreManagerAttrs, jsseTruststoreAttrs, jsseTruststoreManagerAttrs, jsseAdditionalProps);
			if (retval != 0) {
				return retval;
			}
		}

		// TODO: decide if we need to be able to add jsse element. it's a doozy
		// if we do.

		return retval;
	}

	/**
	 * This method takes a bunch of maps. Unfortunately, given the size of the
	 * jsse element, I can't come up with a better solution. However, I have not
	 * given it much though, perhaps it will be reworked at a later stage. The
	 * maps *must* be formatted like this: each Key is a valid attribute within
	 * the CLI; each Value is a valid value for it's associated attribute. For
	 * the installer, this will be checked in the panel and guaranteed. For
	 * others utilizing this class, you will have to sanitize the input.
	 * TODO: perhaps I (tom) should abandon my idea of this class being used
	 * outside the installer, or a few pet projects. It would make things easier
	 * :D
	 * 
	 * @param jsseAttrs
	 * @param jsseKeystoreAttrs
	 * @param jsseKeystoreManagerAttrs
	 * @param jsseTruststoreAttrs
	 * @param jsseTruststoreManagerAttrs
	 * @return
	 */
	private int addSecurityDomainJsse(String domainName,
			Map<String, String> jsseAttrs,
			Map<String, String> jsseKeystoreAttrs,
			Map<String, String> jsseKeystoreManagerAttrs,
			Map<String, String> jsseTruststoreAttrs,
			Map<String, String> jsseTruststoreManagerAttrs,
			Map<String, String> jsseAdditionalProps) {

		String addSecurityDomainJsseCmd = "/subsystem=security/security-domain=" + domainName + "/jsse=classic:add("; // initial
																														// command

		if (jsseAttrs != null) {
			for (Entry<String, String> entry : jsseAttrs.entrySet()) {
				// assume that this will result in a valid command. in the
				// installer's case, it is guaranteed to
				if (entry.getValue() != null && !entry.getValue().isEmpty()) {
					addSecurityDomainJsseCmd += entry.getKey() + "=\"" + entry.getValue() + "\",";
				}
			}
		}

		if (jsseKeystoreAttrs != null) {
			addSecurityDomainJsseCmd += listString(domainName, "keystore", jsseKeystoreAttrs);

			if (jsseKeystoreManagerAttrs != null) {
				addSecurityDomainJsseCmd += listString(domainName, "key-manager", jsseKeystoreManagerAttrs);

			}
		}

		if (jsseTruststoreAttrs != null) {
			addSecurityDomainJsseCmd += listString(domainName, "truststore", jsseTruststoreAttrs);
			
			if (jsseTruststoreManagerAttrs != null) {
				addSecurityDomainJsseCmd += listString(domainName, "trust-manager", jsseTruststoreManagerAttrs);
			}
		}

		submitCommand(addSecurityDomainJsseCmd);

		return exitCode;

	}
	
	/**
	 * Creates a properly formatted LIST type parameter for CLI commands.
	 * @param listName
	 * @param attributes
	 * @return
	 */

	private String listString(String domainName, String listName, Map<String, String> attributes) {
		String retval = listName + "={";
		for (Entry<String, String> entry : attributes.entrySet()) {
			if (entry.getValue() != null && !entry.getValue().isEmpty()) {
				String value = entry.getValue();
				//TODO: not a very good long term solution. passable for now.
				if (entry.getKey().contains("password")){
					// attempt to vault this value
					if (hasVault()){
						value = maskPassword(domainName+"_"+listName, "password", value);
					}
				}
                retval += entry.getKey()
 + "=\""
                        + value.replaceAll("\\\\",
                                "\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\") + "\",";
			}
		}
		retval += "},";
		if (retval.equals(listName + "={}")) {
			return "";
		} else {// the list is empty
			return retval;
		}
	}

	/**
	 * Creates a new security domain security domain with the given attributes
	 * TODO: Investigate why the extends attribute is not available for addition
	 * with the jboss-cli
	 */

	public int createSecurityDomain(String domainName, String cacheType) {
		String addSecurityDomainCmd;
		if (cacheType.equals("infinispan") || cacheType.equals("default")) {
			addSecurityDomainCmd = "/subsystem=security/security-domain=" + domainName + ":add(cache-type=" + cacheType + ")";
		} else {
			addSecurityDomainCmd = "/subsystem=security/security-domain=" + domainName + ":add()";
		}

		submitCommand(addSecurityDomainCmd);
		return exitCode;
	}

	/**
	 * Add the authentication element to a given security domain
	 * 
	 * @param code
	 *            contains a shortname or fully classified class name from a
	 *            long list here:
	 * @param flag
	 *            contains the value required, requisite, sufficient, or
	 *            optional
	 * @param moduleOptions
	 *            contains a map of name,value pairs for the module-option
	 *            elements, if any TODO: find out why the authentication-jaspi
	 *            element is not available from jboss-cli.sh
	 * @return
	 */
	public int addSecurityDomainAuthentication(String domainName, String code, String flag, Map<String, String> moduleOptions) {
		String addSecurityDomainAuthCmd;

		if (!validateCodes(code, validAuthenticationCodes)) {
			return -1; // invalid contents for code element
		}

		if (!validateCodes(flag, validFlags)) {
			return -2; // invalid flag contents
		}

		addSecurityDomainAuthCmd = "/subsystem=security/security-domain=" + domainName + "/authentication=classic:add(login-modules=[{\"code\"=>\"" + code + "\",\"flag\"=>\"" + flag + "\"";

		if (moduleOptions != null && !moduleOptions.isEmpty()) {
			addSecurityDomainAuthCmd += ",\"module-options\"=>[";
			for (String name : moduleOptions.keySet()) {
				addSecurityDomainAuthCmd += "(\"" + name + "\"=>\"" + moduleOptions.get(name) + "\"),";
			}
			addSecurityDomainAuthCmd += "]}])"; // first ] closes module-options
												// list; } closes the the
												// login-modules block, second ]
												// closes login-modules list, )
												// closes the add command
		} else {
			addSecurityDomainAuthCmd += "}])";
		}

		submitCommand(addSecurityDomainAuthCmd);
		return exitCode;
	}

	/**
	 * Adds the authorization element to the security domain
	 * 
	 * @param domainName
	 *            name of the security domain to add authorization to.
	 * @param code
	 *            code for the authorization element. See documentation for
	 *            valid values.
	 * @param flag
	 *            flag for the authorization element (one of required,
	 *            requisite, sufficient, or optional)
	 * @param moduleOptions
	 *            a map of (name, value)s for the module-options elements, if
	 *            desired. This can be null.
	 * @return
	 */
	public int addSecurityDomainAuthorization(String domainName, String code, String flag, Map<String, String> moduleOptions) {
		String addSecurityDomainAuthCmd;

		if (!validateCodes(code, validAuthorizationCodes)) {
			return -1;
		}

		if (!validateCodes(flag, validFlags)) {
			return -2;
		}

		addSecurityDomainAuthCmd = "/subsystem=security/security-domain=" + domainName + "/authorization=classic:add(policy-modules=[{\"code\"=>\"" + code + "\",\"flag\"=>\"" + flag + "\"";

		if (moduleOptions != null && !moduleOptions.isEmpty()) {
			addSecurityDomainAuthCmd += ",\"module-options\"=>[";
			for (String name : moduleOptions.keySet()) {
				addSecurityDomainAuthCmd += "(\"" + name + "\"=>\"" + moduleOptions.get(name) + "\"),";
			}
			addSecurityDomainAuthCmd += "]}])"; // first ] closes module-options
												// list; } closes the the
												// policy-modules block, second
												// ] closes login-modules list,
												// ) closes the add command
		} else {
			addSecurityDomainAuthCmd += "}])";
		}

		submitCommand(addSecurityDomainAuthCmd);
		return exitCode;
	}

	/**
	 * Adds the mapping element to the given security domain
	 * 
	 * @param domainName
	 *            name of the domain to add the mapping element to.
	 * @param code
	 *            code for the mapping-module. See documentation for valid
	 *            values.
	 * @param flag
	 *            flag for the mapping-module (one of required, requisite,
	 *            sufficient, optional)
	 * @param moduleOptions
	 *            Map of (name,value)s for module-options entries.
	 * @return
	 */
	public int addSecurityDomainMapping(String domainName, String code, String type, Map<String, String> moduleOptions) {
		String addSecurityDomainMappingCmd;

		if (!validateCodes(code, validMappingCodes)) {
			return -1;
		}

		if (!validateCodes(type, validTypes)) {
			return -2;
		}

		addSecurityDomainMappingCmd = "/subsystem=security/security-domain=" + domainName + "/mapping=classic:add(mapping-modules=[{\"code\"=>\"" + code + "\",\"type\"=>\"" + type + "\"";

		if (moduleOptions != null && !moduleOptions.isEmpty()) {
			addSecurityDomainMappingCmd += ",\"module-options\"=>[";
			for (String name : moduleOptions.keySet()) {
				addSecurityDomainMappingCmd += "(\"" + name + "\"=>\"" + moduleOptions.get(name) + "\"),";
			}
			addSecurityDomainMappingCmd += "]}])";
		} else {
			addSecurityDomainMappingCmd += "}])";
		}

		submitCommand(addSecurityDomainMappingCmd);
		return exitCode;
	}

	/**
	 * Helper method to help validate the code attribute of the authentication
	 * element of security-domains For the list of valid values, see <a href=
	 * "https://docs.jboss.org/author/display/AS71/Security+subsystem+configuration#Securitysubsystemconfiguration-securitydomains"
	 * >here</a>
	 * 
	 * @param code
	 *            the string value in question
	 * @param codes
	 *            the array containing the valid values for what we're checking
	 * @return true upon valid result, false otherwise
	 */
	private boolean validateCodes(String code, String[] codes) {
		for (String a : codes) {
			if (code.equals(a)) {
				return true;
			}
		}
		return false; // if we get here, the check fails

	}

	/**
	 * Modifies the redirect port of the http web connector TODO: generalize
	 * this for any attribute. probably a very complex task
	 */

	public int modifyHttpRedirectPort(String port) {
		String modifyPortCmd = "/subsystem=web/connector=http:" + writeAttribute("redirect-port", port);

		submitCommand(modifyPortCmd);
		return exitCode;
	}

	/**
	 * Disables the standard welcome screen of the AS
	 */
	public int disableWelcomeScreen() {
		String disableWelcomeCmd = "/subsystem=web/virtual-server=default-host:" + writeAttribute("enable-welcome-root", "false");

		submitCommand(disableWelcomeCmd);
		return exitCode;
	}

	/**
	 * RHQ specific modification of smpt port. TODO: generalize
	 */

	public int writeSmtpPort(String port) {
		String writeSmtpPortCmd = "/socket-binding-group=full-sockets/remote-destination-outbound-socket-binding=mail-smtp:" + writeAttribute(port, "${rhq.server.email.smtp-port:25}");

		submitCommand(writeSmtpPortCmd);
		return exitCode;
	}

	/**
	 * Helper method to add the required string to modify an attribute. Used to
	 * reduce code duplication
	 * 
	 */
	private String writeAttribute(String name, String value) {
		return "write-attribute(name=" + name + ", value=" + value + ")";
	}

	/**
	 * Add a JMS queue with all options available.
	 * 
	 * @param address
	 *            the address of this queue
	 * @param durable
	 *            should the queue be durable or not
	 * @param entries
	 *            a list of entries for the queue
	 * @param selector
	 *            the selector for the queue.
	 */

	public int addJmsQueue(String address, List<String> entries, boolean durable, List<String> headers, String selector) {
		if (!isDomain) {
			return -1; // we cannot add a JMS queue to a non-domain server
		}

		if (address == null) {
			return -2; // can't add a queue with a null address
		}

		if (entries == null || entries.size() == 0) {
			return -3; // can't add a queue with no entries.
		}

		String addJmsQueueCmd = "jms-queue --profile=" + domainProfiles + " add --queue-address=" + address + " --entries=";

		// add entry list
		for (String entry : entries) {
			addJmsQueueCmd += entry + ",";
		}

		if (durable) { // we add this explicitly. it may not be needed, but it's
						// more verbose than to have no element for durable =
						// false
			addJmsQueueCmd += " --durable=true";
		} else {
			addJmsQueueCmd += " --durable=false";
		}

		// headers
		if (headers != null) {
			for (String header : headers) {
				// add the headers. not sure how to do this yet
				// TODO: figure it out
			}
		}

		// TODO: ask someone about what format these selectors should be in
		if (selector != null) {
			addJmsQueueCmd += " --selector=" + selector;
		}

		// special case; cannot use submitCommand method
		naiveLogger.add(addJmsQueueCmd);
		try {
			context.handle(addJmsQueueCmd);
		} catch (CommandLineException e){
            e.printStackTrace();
			exitCode = ERROR;
		}
		return exitCode;
	}

	/**
	 * Adds an infinispan queue with the given values
	 */
	public int addInfinispanCache(String name, String jndiName, String localCacheName, String transactionMode, String evictionStrategy, String evictionMaxEntries, String expirationMaxIdle) {
		int retVal = 0;
		retVal = createInfinispanContainer(name, jndiName);
		if (retVal != 0)
			return retVal;
		retVal = addInfinispanLocalCache(name, localCacheName);
		if (retVal != 0)
			return retVal;
		retVal = addInfinispanEviction(name, localCacheName, evictionStrategy, evictionMaxEntries);
		if (retVal != 0)
			return retVal;
		retVal = addInfinispanTransaction(name, localCacheName, transactionMode);
		if (retVal != 0)
			return retVal;
		retVal = addInfinispanExpiration(name, localCacheName, expirationMaxIdle);

		return retVal;

	}

	private int createInfinispanContainer(String name, String jndiName) {
		if (name == null || jndiName == null) {
			return -1; // fail immediately
		}
		String addInfinispanContainerCmd = "/subsystem=infinispan/cache-container=" + name + ":add(jndi-name=\"" + jndiName + "\")";

		submitCommand(addInfinispanContainerCmd);
		return exitCode;
	}

	private int addInfinispanLocalCache(String name, String localCacheName) {
		if (localCacheName == null) {
			return 0; // not an error; assume that a local cache is not wanted
		}
		String addInfinispanLocalCacheCmd = "/subsystem=infinispan/cache-container=" + name + "/local-cache=" + localCacheName + ":add()";

		submitCommand(addInfinispanLocalCacheCmd);
		return exitCode;
	}

	// TODO: add validation on the evictionStrategy string.
	// TODO: add validation for all possible options
	private int addInfinispanEviction(String name, String localCacheName, String evictionStrategy, String evictionMaxEntries) {
		if (evictionMaxEntries != null) {
			try {
				Integer.parseInt(evictionMaxEntries);
			} catch (NumberFormatException e) {
				return -1; // fail out of any strings that need to be ints are
							// not, in fact, ints.
			}
		}

		String addInfinispanEvictionCmd = "/subsystem=infinispan/cache-container=" + name + "/local-cache=" + localCacheName + "/eviction=EVICTION:add(strategy=" + evictionStrategy + ",max-entries="
				+ evictionMaxEntries + ")";

		return submitCommand(addInfinispanEvictionCmd);
	}

	// TODO: add validations on the transaction mode string
	// TODO: add parameters for all possible options
	private int addInfinispanTransaction(String name, String localCacheName, String transactionMode) {
		String addInfinispanTransactionCmd = "/subsystem=infinispan/cache-container=" + name + "/local-cache=" + localCacheName + "/transaction=TRANSACTION:add(mode=" + transactionMode + ")";

		return submitCommand(addInfinispanTransactionCmd);
	}

	// TODO: add parameters for all possible options
	private int addInfinispanExpiration(String name, String localCacheName, String expirationMaxIdle) {
		if (expirationMaxIdle != null) {
			try {
				Integer.parseInt(expirationMaxIdle);
			} catch (NumberFormatException e) {
				return -1;
			}
		}

		String addInfinispanExpirationCmd = "/subsystem=infinispan/cache-container=" + name + "/local-cache=" + localCacheName + "/expiration=EXPIRATION:add(max-idle=" + expirationMaxIdle + ")";

		return submitCommand(addInfinispanExpirationCmd);
	}


    /**
	 * simple class to indicate that a Batch is not active!
	 * 
	 * @author thauser
	 * 
	 */
	class BatchNotActiveException extends Exception {
		private static final long serialVersionUID = -5463741947772468327L;

		public BatchNotActiveException(String m) {
			super(m);
		}

		public BatchNotActiveException() {
			super();
		}
	}

	class BatchIsEmptyException extends Exception {
		/**
         *
         */
		private static final long serialVersionUID = 8465278229646760882L;

		public BatchIsEmptyException() {
			super();
		}

		public BatchIsEmptyException(String s) {
			super(s);
		}
	}
	
	/**
	 * Reads java properties from the given file, for use in substituting values into commands from an external source.
	 * @param file
	 * @throws IOException
	 */
	public void readPropertiesFile(File file) throws IOException{
		InputStream inStream = new FileInputStream(file);
		System.getProperties().load(inStream);
	}
	
	/**
	 * Simple wrapper on readPropertiesFile(File) for use with a String path rather than a File object. 
	 * @param path
	 * @throws IOException
	 */
	public void readPropertiesFile(String path) throws IOException {
		readPropertiesFile(new File(path));
	}
	
	public void disconnect(){
		context.disconnectController();
	}
	
	/**
	 * Set environment variables into memory
	 */
	/**
	 * Append or overwrite environment variables to system variables in memory.
	 * 
	 * @param newenv Map of additional variables you want to add
	 * @param append Set to false if you'd like to overwrite the system variables in memory
	 */
	public static void addEnv(Map<String, String> newenv){ setEnv(newenv, true); }
	public static void setEnv(Map<String, String> newenv, final boolean append) {
		
		Map<String, String> env = System.getenv();
		if (append) {
			for (String key : env.keySet())
				if (newenv.get(key) == null) newenv.put(key, env.get(key));
		}
		
	    Class[] classes = Collections.class.getDeclaredClasses();
	    try {
		    for(Class cl : classes) {
		        if("java.util.Collections$UnmodifiableMap".equals(cl.getName())) {
		            Field field = cl.getDeclaredField("m");
		            field.setAccessible(true);
		            Object obj = field.get(env);
		            Map<String, String> map = (Map<String, String>) obj;
		            map.clear();
		            map.putAll(newenv);
		        }
		    }
	    } catch (Exception e) {
	    	e.printStackTrace();
	    }
	}
	
	/**
	 * Method that gets database properties from the idata
	 * @param properties
	 */
	
	public void addProperties(String[] properties){
		AutomatedInstallData idata = AutomatedInstallData.getInstance();
		for (String prop : properties ) {
			System.getProperties().setProperty(prop, idata.getVariable(prop));
		}		
	}
	
}
