package com.redhat.installer.tests.validator;

import com.redhat.installer.tests.DataValidatorTester;
import com.redhat.installer.tests.TestUtils;
import com.redhat.installer.validator.DuplicateUserValidator;
import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.redhat.installer.tests.TestUtils.createNewFileWithContent;

/**
 * Created by thauser on 2/19/14.
 */
public class DuplicateUserValidatorTest extends DataValidatorTester {
    static final String CONFIG_VAR = "current.test";
    static final String MGMT_USER = "adminUser";
    static final String FSW_USER = "fsw.user";
    static final String BPMS_USER = "bpms.user";
    static final String BRMS_USER = "brms.user";
    static final String TEIID_ADMIN = "Teiid.admin.user";
    static final String TEIID_NORM = "Teiid.norm.user";
    static final String MODESHAPE = "Modeshape.user";
    static final String MGMT_FILE = "mgmt-users.properties";
    static final String APP_FILE = "application-users.properties";
    static final String TEIID_FILE = "teiid-security-users.properties";
    static final String MODESHAPE_FILE = "modeshape-users.properties";

    static final Map<String,String> filenameMap = new HashMap<String,String>();
    static {
        filenameMap.put(MGMT_USER, MGMT_FILE);
        filenameMap.put(FSW_USER, APP_FILE);
        filenameMap.put(BPMS_USER, APP_FILE);
        filenameMap.put(BRMS_USER, APP_FILE);
        filenameMap.put(TEIID_ADMIN, TEIID_FILE);
        filenameMap.put(TEIID_NORM, TEIID_FILE);
        filenameMap.put(MODESHAPE, MODESHAPE_FILE);
    }

    static final List<String> configList = new ArrayList<String>();
    static {
        configList.add(MGMT_USER);
        configList.add(FSW_USER);
        configList.add(BPMS_USER);
        configList.add(BRMS_USER);
        configList.add(TEIID_ADMIN);
        configList.add(TEIID_NORM);
        configList.add(MODESHAPE);
    }

    @BeforeClass
    public static void specificInit() throws Exception {
        idata.langpack = TestUtils.createMockLangpack("DuplicateUsername.errorMsg","DuplicateUsername.warningMsg");
    }

    @Before
    public void setUp() throws Exception {
         // similar to the PortCollisionValidatorTest, we can
         // use an anonymous class to test all of the other classes.
        dv = new DuplicateUserValidator() {
            @Override
            protected String getUserVar() {
                return idata.getVariable(CONFIG_VAR);
            }

            @Override
            protected String getCondVar() {
                String conf = idata.getVariable(CONFIG_VAR);
                if (conf.equals(TEIID_ADMIN) || conf.equals(TEIID_NORM) || conf.equals(MODESHAPE)){
                    return conf+".add";
                } else if (conf.equals(MGMT_USER)){
                    return "addUser";
                } else {
                    return null;
                }
            }

            @Override
            protected String getFileName() {
                String conf = idata.getVariable(CONFIG_VAR);
                if (conf.equals(TEIID_ADMIN) || conf.equals(TEIID_NORM)){
                    return TEIID_FILE;
                } else if (conf.equals(MODESHAPE)){
                    return MODESHAPE_FILE;
                } else if (conf.equals(MGMT_USER)){
                    return MGMT_FILE;
                } else {
                    return APP_FILE;
                }
            }

            @Override
            protected Status getConflictStatus() {
                String conf = idata.getVariable(CONFIG_VAR);
                if (conf.equals(MGMT_USER)){
                    return Status.SKIP;
                } else {
                    return Status.ERROR;
                }
            }
        };
        idata.setVariable("INSTALL_PATH", TestUtils.INSTALL_PATH);
        idata.setVariable("INSTALL_SUBPATH", TestUtils.INSTALL_SUBPATH);
    }

    // only valid for mgmt, since it's the only skippable one
    @Test
    public void testMgmtNoAddUser(){
        idata.setVariable(CONFIG_VAR, MGMT_USER);
        idata.setVariable("addUser", "false");
        assertOk();
    }

    @Test
    public void testNoDuplicateUser() throws Exception{
        for (String config : configList){
            createTestEnvironment(config, "newuser", "testuser=testpassword");
            assertOk();
            FileUtils.deleteDirectory(TestUtils.INSTALL_PATH_FILE);
        }
    }

    @Test
    public void testDuplicateUser() throws Exception{
        for (String config : configList) {
            createTestEnvironment(config, "testuser", "testuser=testpassword");
            if (config.equals(MGMT_USER)){
                assertSkip();
            } else {
                assertError();
            }
            FileUtils.deleteDirectory(TestUtils.INSTALL_PATH_FILE);
        }
    }

    @Test
    public void testFileParsingDuplicateUserAddingNewUser() throws Exception{
        for (String config : configList){
            createTestEnvironment(config, "newuser", "testuser=testpassword", "testuser=testpassword");
            assertOk();
            FileUtils.deleteDirectory(TestUtils.INSTALL_PATH_FILE);
        }
    }

    @Test
    public void testFileParsingDuplicateUserAddingDuplicateUser() throws Exception{
        for (String config : configList){
            createTestEnvironment(config, "testuser", "testuser=testpassword", "testuser=testpassword");
            if (config.equals(MGMT_USER)){
                assertSkip();
            } else {
                assertError();
            }
            FileUtils.deleteDirectory(TestUtils.INSTALL_PATH_FILE);
        }
    }

    @Test
    public void testFileParsingMissingPasswordAddingNewUser() throws Exception{
        for (String config : configList){
            createTestEnvironment(config, "newuser", "testuser=", "anotheruser=hihi");
            assertOk();
            FileUtils.deleteDirectory(TestUtils.INSTALL_PATH_FILE);
        }
    }

    @Test
    public void testFileParsingMissingPasswordAddingDuplicateUser() throws Exception{
        for (String config : configList){
            createTestEnvironment(config, "testuser", "testuser=", "anotheruser=hihi");
            if (config.equals(MGMT_USER)){
                assertSkip();
            } else {
                assertError();
            }
            FileUtils.deleteDirectory(TestUtils.INSTALL_PATH_FILE);
        }
    }

    @Test
    public void testFileParsingMissingUsernameAddingNewUser() throws Exception{
        for (String config : configList){
            createTestEnvironment(config, "newuser", "testuser=", "=testpassword");
            assertOk();
            FileUtils.deleteDirectory(TestUtils.INSTALL_PATH_FILE);
        }
    }

    @Test
    public void testFileParsingMissingUsernameAddingDuplicateUser() throws Exception{
        for (String config : configList){
            createTestEnvironment(config, "","=testpassword");
            if (config.equals(MGMT_USER)){
                assertSkip();
            } else {
                assertError();
            }
            FileUtils.deleteDirectory(TestUtils.INSTALL_PATH_FILE);
        }
    }

   @Test
    public void testFileParsingMismatchedFilesAddingNewUser() throws Exception{
       for (String config : configList){
           createNewFileWithContent("/standalone/configuration/"+filenameMap.get(config), "testuser1=testpassword1");
           createNewFileWithContent("/domain/configuration/"+filenameMap.get(config),"testuser2=testpassword2");
           idata.setVariable(CONFIG_VAR, config);
           idata.setVariable(config, "newuser");
           assertOk();
           FileUtils.deleteDirectory(TestUtils.INSTALL_PATH_FILE);
       }
    }

    @Test
    public void testFileParsingMismatchedFilesAddingDuplicateUser() throws Exception{
        for (String config : configList){
            createNewFileWithContent("/standalone/configuration/"+filenameMap.get(config), "testuser1=testpassword1");
            createNewFileWithContent("/domain/configuration/"+filenameMap.get(config),"testuser2=testpassword2");
            idata.setVariable(CONFIG_VAR, config);
            idata.setVariable(config, "testuser1");
            if (config.equals(MGMT_USER)){
                assertSkip();
            } else {
                assertError();
            }
            idata.setVariable(config, "testuser2");
            if (config.equals(MGMT_USER)){
                assertSkip();
            } else {
                assertError();
            }
            FileUtils.deleteDirectory(TestUtils.INSTALL_PATH_FILE);
        }
    }


    private void createUserFiles(String filename, String ... args) {
        createNewFileWithContent("/standalone/configuration/"+filename, args);
        createNewFileWithContent("/domain/configuration/"+filename, args);
    }

    private void createTestEnvironment(String config, String username, String ... fileContents){
        idata.setVariable(CONFIG_VAR, config);
        idata.setVariable(config, username);
        createUserFiles(filenameMap.get(config), fileContents);
    }


}
