package com.redhat.installer.tests.validator;

import com.redhat.installer.tests.DataValidatorTester;
import com.redhat.installer.tests.TestUtils;
import com.redhat.installer.validator.HornetQValidator;
import org.junit.Before;
import org.junit.Test;

import static junit.framework.TestCase.assertEquals;

/**
 * Unit tests for the HornetQValidator
 * Created by thauser on 2/27/14.
 */
public class HornetQValidatorTest extends DataValidatorTester {
    private static final String resourcePath = "/hornetq-serverTest/";
    private static final String errorMessage = "The following descriptors contain a hornetq-server element with incompatible configuration for this product: %s. Please remove the configuration or choose a new installation path.";
    private static final String warningMessage = "The following descriptors already contain a hornetq-server element: %s. The configuration appears to be compatible with the product, but errors could occur. Would you like to continue?";

    @Before
    public void setUp(){
        idata.setVariable("INSTALL_PATH", TestUtils.INSTALL_PATH);
        idata.setVariable("INSTALL_SUBPATH", TestUtils.INSTALL_SUBPATH);
        dv = new HornetQValidator();
    }

    @Test
    public void testNoConflicts() throws Exception {
        TestUtils.copyFileToTestDir(resourcePath + "no-conflicting.xml", "/standalone/configuration/standalone.xml");
        assertOk();
    }

    @Test
    public void testNoConfig() throws Exception {
        TestUtils.copyFileToTestDir(resourcePath + "no-config.xml", "/standalone/configuration/standalone.xml");
        assertOk();
    }

    @Test
    public void testWarnings() throws Exception {
        TestUtils.copyFileToTestDir(resourcePath + "compatible.xml", "/standalone/configuration/standalone.xml");
        assertWarning();
        assertEquals(String.format(warningMessage, "standalone.xml"), dv.getWarningMessageId());
        assertEquals("true",idata.getVariable("standalone.xml.hornetq.exists"));
    }

    @Test
    public void testMultiWarnings() throws Exception {
        TestUtils.copyFileToTestDir(resourcePath + "compatible.xml", "/standalone/configuration/standalone.xml");
        TestUtils.copyFileToTestDir(resourcePath + "compatible.xml", "/standalone/configuration/standalone-ha.xml");
        TestUtils.copyFileToTestDir(resourcePath + "compatible.xml", "/standalone/configuration/standalone-full.xml");
        assertWarning();
        assertEquals(String.format(warningMessage, "standalone.xml standalone-ha.xml standalone-full.xml"), dv.getWarningMessageId());
        assertEquals("true",idata.getVariable("standalone.xml.hornetq.exists"));
        assertEquals("true",idata.getVariable("standalone-ha.xml.hornetq.exists"));
        assertEquals("true",idata.getVariable("standalone-full.xml.hornetq.exists"));
    }

    @Test
    public void testErrors() throws Exception {
        TestUtils.copyFileToTestDir(resourcePath + "incompatible.xml", "/standalone/configuration/standalone.xml");
        assertError();
        assertEquals(String.format(errorMessage, "standalone.xml"), dv.getErrorMessageId());
    }

    @Test
    public void testMultiErrors() throws Exception {
        TestUtils.copyFileToTestDir(resourcePath + "incompatible.xml", "/standalone/configuration/standalone.xml");
        TestUtils.copyFileToTestDir(resourcePath + "incompatible.xml", "/standalone/configuration/standalone-ha.xml");
        TestUtils.copyFileToTestDir(resourcePath + "incompatible.xml", "/standalone/configuration/standalone-full.xml");
        assertError();
        assertEquals(String.format(errorMessage, "standalone.xml standalone-ha.xml standalone-full.xml"), dv.getErrorMessageId());
    }

    @Test
    public void testMoreThanOneConfig() throws Exception {
        TestUtils.copyFileToTestDir(resourcePath + "compatible-gt1.xml", "/standalone/configuration/standalone.xml");
        assertWarning();
        assertEquals(String.format(warningMessage, "standalone.xml"), dv.getWarningMessageId());
        assertEquals( "true", idata.getVariable("standalone.xml.hornetq.exists"));
    }

    @Test public void testDomainNoConfig() throws Exception {
        TestUtils.copyFileToTestDir(resourcePath + "no-config.xml", "/domain/configuration/domain.xml");
        assertOk();
    }
    @Test 
    public void testDomainWarnings() throws Exception {
        TestUtils.copyFileToTestDir(resourcePath + "compatible.xml", "/domain/configuration/domain.xml");
        assertWarning();
        assertEquals("qwe",String.format(warningMessage, "domain.xml"), dv.getWarningMessageId());
        assertEquals("asd","true", idata.getVariable("domain.xml.hornetq.exists"));
    }

    @Test
    public void testDomainNoConflicts() throws Exception {
        TestUtils.copyFileToTestDir(resourcePath + "no-conflicting.xml", "/domain/configuration/domain.xml");
        assertOk();
    }

    @Test
    public void testAdditionalConfig () throws Exception {
        TestUtils.copyFileToTestDir(resourcePath + "additional.xml", "/standalone/configuration/standalone.xml");
        assertWarning();
        assertEquals(String.format(warningMessage, "standalone.xml"), dv.getWarningMessageId());
        assertEquals("true",idata.getVariable("standalone.xml.hornetq.exists"));
    }
}
