import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.Properties;

/*
 * RHQ Management Platform
 * Copyright (C) 2005-2008 Red Hat, Inc.
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * Load two property files and output a resulting one.
 * Idea is to read in a base properties file and an additional one. Then depending on mode
 * <ul>
 * <li>Output those from additional that override base</li>
 * <li>Output the merging of the two files</li>
 * </ul>
 * @author Heiko W. Rupp
 *
 */
public class JoinProperties {

    // Number of header lines in the 
    private static final int TWO = 2;

    /**
     * @param args
     */
    public static void main(String[] args) {
        boolean diffMode = false;

        if (args.length < 3) {
            System.err.println("Usage: JoinProperties [-d] base additional output");
            System.exit(1);
        }
        int start = 0;
        if (args.length == 4) {
            if ("-d".equals(args[0])) {
                start++;
                diffMode = true;
            }
        }

        Properties[] props = new Properties[2];
        try {
            for (int i = 0; i < 2; i++) {
                InputStream in = new FileInputStream(new File(args[i + start]));
                props[i] = new Properties();
                props[i].load(in);
                in.close();
            }

            JoinProperties jp = new JoinProperties();
            Properties out;
            String task;
            if (diffMode == false) {
                task = " Merged";
                out = jp.joinProps(props[0], props[1]);
            } else {
                task = " Diffed";
                out = jp.diffProps(props[0], props[1]);
            }
            FileOutputStream fos = new FileOutputStream(new File(args[2 + start]));
            out.store(fos, task + " Properties created on ");
            fos.flush();
            fos.close();
        } catch (Exception e) {
            e.printStackTrace();
            System.exit(2);
        }
    }

    /**
     * Join two property files and output a resulting one.
     * Idea is to read in a base properties file and an additional one:
     * <ul>
     * <li>All entries that are in base and not in additional are copied to the output</li>
     * <li>All entries in additional, that are in base will over write the version from base to the output</li>
     * <li>All entries only in additional will go to the output</li>
     * </ul> 
     * 
     */
    public Properties joinProps(Properties base, Properties additional) {

        int baseCount = 0; // Properties in base only
        int mergedCount = 0; // Properties in base and additional 
        int additionalCount = 0; // Properties only in additional
        int equalsCount = 0; // Properties in base in additional with same value ( so always <= mc)

        Properties result = new Properties();

        for (Object baseKey : base.keySet()) {
            String key = (String) baseKey;
            if (additional.containsKey(key)) {
                result.put(key, additional.getProperty(key));
                mergedCount++;
                if (base.getProperty(key).equals(additional.getProperty(key)))
                    equalsCount++;
            } else {
                result.put(key, base.getProperty(key));
                baseCount++;
            }
        }

        // Now lets see what keys are in additional and not in base and copy them over.
        for (Object addKey : additional.keySet()) {
            String key = (String) addKey;
            if (!base.containsKey(key)) {
                result.put(key, additional.getProperty(key));
                additionalCount++;
            }
        }

        int outLines = mergedCount + baseCount + additionalCount + TWO;
        System.out.println("bc= " + baseCount + ", ec= " + equalsCount + ", mc= " + mergedCount + ", ac= "
            + additionalCount + ", ol= " + outLines);
        return result;
    }

    /**
     * Only add properties to output that are either
     * <ul>
     * <li>Only in additional</li>
     * <li>Different in additional than in base</li>
     * </ul>
     * @param base Base version of the properties
     * @param additional Additional properties
     * @return properties in additional only or changed
     */
    public Properties diffProps(Properties base, Properties additional) {

        int additionalCount = 0; // new properties from additional 
        int diffCount = 0; // properties in base and additional, with different values
        int baseCount = 0; // properties from base

        Properties result = new Properties();

        for (Object addKey : additional.keySet()) {
            String key = (String) addKey;
            String addValue = additional.getProperty(key);
            if (!base.containsKey(key)) {
                result.put(key, addValue);
                additionalCount++;
            } else { // key is in base and additional, so only add if they differ
                String property = base.getProperty(key);
                if (!(addValue.equals(property))) {
                    result.put(key, addValue);
                    diffCount++;
                } else
                    baseCount++;
            }
        }
        int outLines = additionalCount + diffCount + TWO;
        System.out
            .println("ac= " + additionalCount + ", dc= " + diffCount + ", bc= " + baseCount + ", ol= " + outLines);
        return result;
    }

}
