/*
 * 
 */
package org.jboss.maven.plugins.zip;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectHelper;
import org.codehaus.plexus.archiver.zip.ZipArchiver;

import java.io.File;

/**
 * Packages a zipfile containing project resources.
 *
 * @goal zip
 * @phase package
 * @requiresProject
 * @requiresDependencyResolution
 */
public class ZipMojo extends AbstractMojo
{

    private static final String[] DEFAULT_EXCLUDES = new String[0];

    private static final String[] DEFAULT_INCLUDES = new String[]{"**/**"};

    /**
     * Directory containing the generated zipfile.
     *
     * @parameter expression="${project.build.directory}"
     * @required
     */
    private File outputDirectory;

    /**
     * Directory containing the content to be zipped.
     *
     * @parameter expression="${project.build.outputDirectory}"
     * @required
     */
    private File sourceDirectory;
    
    /**
     * Basename of the generated zipfile.
     *
     * @parameter alias="zipName" expression="${project.build.finalName}"
     * @required
     */
    private String finalName;

    /**
     * The zip archiver.
     *
     * @parameter expression="${component.org.codehaus.plexus.archiver.Archiver#zip}"
     * @required
     */
    private ZipArchiver zipArchiver;

    /**
     * The maven project.
     *
     * @parameter expression="${project}"
     * @required
     * @readonly
     */
    private MavenProject project;

    /**
     * @component
     */
    private MavenProjectHelper projectHelper;

    /**
     * Whether creating the archive should be forced.
     * @parameter expression="${zip.forceCreation}" default-value="false"
     */
    private boolean forceCreation;

    /**
     * Classifier to add to the artifact generated. If given, the artifact will be an attachment instead.
     *
     * @parameter
     */
    private String classifier;

    /**
     * Whether the source directory itself should be included in the archive.
     * @parameter default-value="false"
     */
    private boolean includeSourceDirectory;
    
    protected final MavenProject getProject()
    {
        return project;
    }

    protected String getClassifier()
    {
        return classifier;
    }

    /**
     * Return the main source directory, so it's used as the root of the zipfile.
     */
    protected File getSourceDirectory()
    {
        return sourceDirectory;
    }

    protected static File getZipFile( File basedir, String finalName, String classifier )
    {
        if ( classifier == null )
        {
            classifier = "";
        }
        else if ( classifier.trim().length() > 0 && !classifier.startsWith( "-" ) )
        {
            classifier = "-" + classifier;
        }

        return new File( basedir, finalName + classifier + ".zip" );
    }

    /**
     * Generates the zipfile.
     */
    public File createArchive()
        throws MojoExecutionException
    {
        File zipFile = getZipFile( outputDirectory, finalName, getClassifier() );

        zipArchiver.setDestFile( zipFile );
        zipArchiver.setForced( forceCreation);

        try
        {
            File contentDirectory = getSourceDirectory();
            if ( !contentDirectory.exists() )
            {
                getLog().warn( "Zipfile will be empty - no content was marked for inclusion!" );
            }
            else
            {
                if (includeSourceDirectory)
                {
                   String[] includes = new String[]{contentDirectory.getName() + "/**"};
                   String[] excludes = new String[0];
                   zipArchiver.addDirectory( contentDirectory.getParentFile(), includes, excludes );
                }
                else
                {
                   zipArchiver.addDirectory( contentDirectory );
                }
            }

            zipArchiver.createArchive();

            return zipFile;
        }
        catch ( Exception e )
        {
            // TODO: improve error handling
            throw new MojoExecutionException( "Error assembling zipfile", e );
        }
    }

    /**
     * Generates the zipfile.
     */
    public void execute()
        throws MojoExecutionException
    {
        File zipFile = createArchive();

        String classifier = getClassifier();
        if ( classifier != null )
        {
            projectHelper.attachArtifact( getProject(), "zip", classifier, zipFile );
        }
        else
        {
            getProject().getArtifact().setFile( zipFile );
        }
    }
    
}
