/*
 * Copyright (c) 2011-2022 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
 */

package io.vertx.mssqlclient.impl.codec;

import io.netty.util.collection.IntObjectHashMap;
import io.netty.util.collection.IntObjectMap;

public enum WindowsLocale {
  ar_SA(0x0401, Encoding.CP1256),
  bg_BG(0x0402, Encoding.CP1251),
  ca_ES(0x0403, Encoding.CP1252),
  zh_TW(0x0404, Encoding.CP950),
  cs_CZ(0x0405, Encoding.CP1250),
  da_DK(0x0406, Encoding.CP1252),
  de_DE(0x0407, Encoding.CP1252),
  el_GR(0x0408, Encoding.CP1253),
  en_US(0x0409, Encoding.CP1252),
  es_ES_tradnl(0x040a, Encoding.CP1252),
  fi_FI(0x040b, Encoding.CP1252),
  fr_FR(0x040c, Encoding.CP1252),
  he_IL(0x040d, Encoding.CP1255),
  hu_HU(0x040e, Encoding.CP1250),
  is_IS(0x040f, Encoding.CP1252),
  it_IT(0x0410, Encoding.CP1252),
  ja_JP(0x0411, Encoding.CP932),
  ko_KR(0x0412, Encoding.CP949),
  nl_NL(0x0413, Encoding.CP1252),
  nb_NO(0x0414, Encoding.CP1252),
  pl_PL(0x0415, Encoding.CP1250),
  pt_BR(0x0416, Encoding.CP1252),
  rm_CH(0x0417, Encoding.CP1252),
  ro_RO(0x0418, Encoding.CP1250),
  ru_RU(0x0419, Encoding.CP1251),
  hr_HR(0x041a, Encoding.CP1250),
  sk_SK(0x041b, Encoding.CP1250),
  sq_AL(0x041c, Encoding.CP1250),
  sv_SE(0x041d, Encoding.CP1252),
  th_TH(0x041e, Encoding.CP874),
  tr_TR(0x041f, Encoding.CP1254),
  ur_PK(0x0420, Encoding.CP1256),
  id_ID(0x0421, Encoding.CP1252),
  uk_UA(0x0422, Encoding.CP1251),
  be_BY(0x0423, Encoding.CP1251),
  sl_SI(0x0424, Encoding.CP1250),
  et_EE(0x0425, Encoding.CP1257),
  lv_LV(0x0426, Encoding.CP1257),
  lt_LT(0x0427, Encoding.CP1257),
  tg_Cyrl_TJ(0x0428, Encoding.CP1251),
  fa_IR(0x0429, Encoding.CP1256),
  vi_VN(0x042a, Encoding.CP1258),
  hy_AM(0x042b, Encoding.CP1252),
  az_Latn_AZ(0x042c, Encoding.CP1254),
  eu_ES(0x042d, Encoding.CP1252),
  wen_DE(0x042e, Encoding.CP1252),
  mk_MK(0x042f, Encoding.CP1251),
  tn_ZA(0x0432, Encoding.CP1252),
  xh_ZA(0x0434, Encoding.CP1252),
  zu_ZA(0x0435, Encoding.CP1252),
  Af_ZA(0x0436, Encoding.CP1252),
  ka_GE(0x0437, Encoding.CP1252),
  fo_FO(0x0438, Encoding.CP1252),
  hi_IN(0x0439, Encoding.UNICODE),
  mt_MT(0x043a, Encoding.UNICODE),
  se_NO(0x043b, Encoding.CP1252),
  ms_MY(0x043e, Encoding.CP1252),
  kk_KZ(0x043f, Encoding.CP1251),
  ky_KG(0x0440, Encoding.CP1251),
  sw_KE(0x0441, Encoding.CP1252),
  tk_TM(0x0442, Encoding.CP1250),
  uz_Latn_UZ(0x0443, Encoding.CP1254),
  tt_RU(0x0444, Encoding.CP1251),
  bn_IN(0x0445, Encoding.UNICODE),
  pa_IN(0x0446, Encoding.UNICODE),
  gu_IN(0x0447, Encoding.UNICODE),
  or_IN(0x0448, Encoding.UNICODE),
  ta_IN(0x0449, Encoding.UNICODE),
  te_IN(0x044a, Encoding.UNICODE),
  kn_IN(0x044b, Encoding.UNICODE),
  ml_IN(0x044c, Encoding.UNICODE),
  as_IN(0x044d, Encoding.UNICODE),
  mr_IN(0x044e, Encoding.UNICODE),
  sa_IN(0x044f, Encoding.UNICODE),
  mn_MN(0x0450, Encoding.CP1251),
  bo_CN(0x0451, Encoding.UNICODE),
  cy_GB(0x0452, Encoding.CP1252),
  km_KH(0x0453, Encoding.UNICODE),
  lo_LA(0x0454, Encoding.UNICODE),
  gl_ES(0x0456, Encoding.CP1252),
  kok_IN(0x0457, Encoding.UNICODE),
  syr_SY(0x045a, Encoding.UNICODE),
  si_LK(0x045b, Encoding.UNICODE),
  iu_Cans_CA(0x045d, Encoding.CP1252),
  am_ET(0x045e, Encoding.CP1252),
  ne_NP(0x0461, Encoding.UNICODE),
  fy_NL(0x0462, Encoding.CP1252),
  ps_AF(0x0463, Encoding.UNICODE),
  fil_PH(0x0464, Encoding.CP1252),
  dv_MV(0x0465, Encoding.UNICODE),
  ha_Latn_NG(0x0468, Encoding.CP1252),
  yo_NG(0x046a, Encoding.CP1252),
  quz_BO(0x046b, Encoding.CP1252),
  nso_ZA(0x046c, Encoding.CP1252),
  ba_RU(0x046d, Encoding.CP1251),
  lb_LU(0x046e, Encoding.CP1252),
  kl_GL(0x046f, Encoding.CP1252),
  ig_NG(0x0470, Encoding.CP1252),
  ii_CN(0x0478, Encoding.CP1252),
  arn_CL(0x047a, Encoding.CP1252),
  moh_CA(0x047c, Encoding.CP1252),
  br_FR(0x047e, Encoding.CP1252),
  ug_CN(0x0480, Encoding.CP1256),
  mi_NZ(0x0481, Encoding.UNICODE),
  oc_FR(0x0482, Encoding.CP1252),
  co_FR(0x0483, Encoding.CP1252),
  gsw_FR(0x0484, Encoding.CP1252),
  sah_RU(0x0485, Encoding.CP1251),
  qut_GT(0x0486, Encoding.CP1252),
  rw_RW(0x0487, Encoding.CP1252),
  wo_SN(0x0488, Encoding.CP1252),
  prs_AF(0x048c, Encoding.CP1256),
  ar_IQ(0x0801, Encoding.CP1256),
  zh_CN(0x0804, Encoding.CP936),
  de_CH(0x0807, Encoding.CP1252),
  en_GB(0x0809, Encoding.CP1252),
  es_MX(0x080a, Encoding.CP1252),
  fr_BE(0x080c, Encoding.CP1252),
  it_CH(0x0810, Encoding.CP1252),
  nl_BE(0x0813, Encoding.CP1252),
  nn_NO(0x0814, Encoding.CP1252),
  pt_PT(0x0816, Encoding.CP1252),
  sr_Latn_CS(0x081a, Encoding.CP1250),
  sv_FI(0x081d, Encoding.CP1252),
  Lithuanian_Classic(0x0827, Encoding.CP1257),
  az_Cyrl_AZ(0x082c, Encoding.CP1251),
  dsb_DE(0x082e, Encoding.CP1252),
  se_SE(0x083b, Encoding.CP1252),
  ga_IE(0x083c, Encoding.CP1252),
  ms_BN(0x083e, Encoding.CP1252),
  uz_Cyrl_UZ(0x0843, Encoding.CP1251),
  bn_BD(0x0845, Encoding.UNICODE),
  mn_Mong_CN(0x0850, Encoding.CP1251),
  iu_Latn_CA(0x085d, Encoding.CP1252),
  tzm_Latn_DZ(0x085f, Encoding.CP1252),
  quz_EC(0x086b, Encoding.CP1252),
  ar_EG(0x0c01, Encoding.CP1256),
  zh_HK(0x0c04, Encoding.CP950),
  de_AT(0x0c07, Encoding.CP1252),
  en_AU(0x0c09, Encoding.CP1252),
  es_ES(0x0c0a, Encoding.CP1252),
  fr_CA(0x0c0c, Encoding.CP1252),
  sr_Cyrl_CS(0x0c1a, Encoding.CP1251),
  se_FI(0x0c3b, Encoding.CP1252),
  quz_PE(0x0c6b, Encoding.CP1252),
  ar_LY(0x1001, Encoding.CP1256),
  zh_SG(0x1004, Encoding.CP936),
  de_LU(0x1007, Encoding.CP1252),
  en_CA(0x1009, Encoding.CP1252),
  es_GT(0x100a, Encoding.CP1252),
  fr_CH(0x100c, Encoding.CP1252),
  hr_BA(0x101a, Encoding.CP1250),
  smj_NO(0x103b, Encoding.CP1252),
  ar_DZ(0x1401, Encoding.CP1256),
  zh_MO(0x1404, Encoding.CP950),
  de_LI(0x1407, Encoding.CP1252),
  en_NZ(0x1409, Encoding.CP1252),
  es_CR(0x140a, Encoding.CP1252),
  fr_LU(0x140c, Encoding.CP1252),
  bs_Latn_BA(0x141a, Encoding.CP1250),
  smj_SE(0x143b, Encoding.CP1252),
  ar_MA(0x1801, Encoding.CP1256),
  en_IE(0x1809, Encoding.CP1252),
  es_PA(0x180a, Encoding.CP1252),
  fr_MC(0x180c, Encoding.CP1252),
  sr_Latn_BA(0x181a, Encoding.CP1250),
  sma_NO(0x183b, Encoding.CP1252),
  ar_TN(0x1c01, Encoding.CP1256),
  en_ZA(0x1c09, Encoding.CP1252),
  es_DO(0x1c0a, Encoding.CP1252),
  sr_Cyrl_BA(0x1c1a, Encoding.CP1251),
  sma_SB(0x1c3b, Encoding.CP1252),
  ar_OM(0x2001, Encoding.CP1256),
  en_JM(0x2009, Encoding.CP1252),
  es_VE(0x200a, Encoding.CP1252),
  bs_Cyrl_BA(0x201a, Encoding.CP1251),
  sms_FI(0x203b, Encoding.CP1252),
  ar_YE(0x2401, Encoding.CP1256),
  en_CB(0x2409, Encoding.CP1252),
  es_CO(0x240a, Encoding.CP1252),
  smn_FI(0x243b, Encoding.CP1252),
  ar_SY(0x2801, Encoding.CP1256),
  en_BZ(0x2809, Encoding.CP1252),
  es_PE(0x280a, Encoding.CP1252),
  ar_JO(0x2c01, Encoding.CP1256),
  en_TT(0x2c09, Encoding.CP1252),
  es_AR(0x2c0a, Encoding.CP1252),
  ar_LB(0x3001, Encoding.CP1256),
  en_ZW(0x3009, Encoding.CP1252),
  es_EC(0x300a, Encoding.CP1252),
  ar_KW(0x3401, Encoding.CP1256),
  en_PH(0x3409, Encoding.CP1252),
  es_CL(0x340a, Encoding.CP1252),
  ar_AE(0x3801, Encoding.CP1256),
  es_UY(0x380a, Encoding.CP1252),
  ar_BH(0x3c01, Encoding.CP1256),
  es_PY(0x3c0a, Encoding.CP1252),
  ar_QA(0x4001, Encoding.CP1256),
  en_IN(0x4009, Encoding.CP1252),
  es_BO(0x400a, Encoding.CP1252),
  en_MY(0x4409, Encoding.CP1252),
  es_SV(0x440a, Encoding.CP1252),
  en_SG(0x4809, Encoding.CP1252),
  es_HN(0x480a, Encoding.CP1252),
  es_NI(0x4c0a, Encoding.CP1252),
  es_PR(0x500a, Encoding.CP1252),
  es_US(0x540a, Encoding.CP1252);

  private final int langId;
  public final Encoding encoding;

  WindowsLocale(int langId, Encoding encoding) {
    this.langId = langId;
    this.encoding = encoding;
  }

  private static final IntObjectMap<WindowsLocale> windowsLocalesByLangId;

  static {
    WindowsLocale[] values = values();
    windowsLocalesByLangId = new IntObjectHashMap<>(values.length);
    for (WindowsLocale windowsLocale : values) {
      windowsLocalesByLangId.put(windowsLocale.langId, windowsLocale);
    }
  }

  public static WindowsLocale forLangId(int langId) {
    WindowsLocale windowsLocale = windowsLocalesByLangId.get(langId);
    if (windowsLocale == null) {
      throw new IllegalArgumentException("Unknown Windows locale: " + Integer.toHexString(langId));
    }
    return windowsLocale;
  }

}
