/*
 * Copyright 2010 Red Hat, Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.jboss.soa.dsp.deployer.jbossas6;

import java.io.File;
import java.io.IOException;
import java.net.URL;

import org.jboss.dependency.spi.ControllerContext;
import org.jboss.deployers.client.spi.DeployerClient;
import org.jboss.deployers.client.spi.Deployment;
import org.jboss.deployers.spi.attachments.MutableAttachments;
import org.jboss.deployers.structure.spi.ClassLoaderFactory;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.structure.spi.DeploymentUnitFilter;
import org.jboss.deployers.vfs.spi.client.VFSDeploymentFactory;
import org.jboss.kernel.Kernel;
import org.jboss.metadata.web.jboss.JBossWebMetaData;
import org.jboss.soa.dsp.EndpointMetaData;
import org.jboss.soa.dsp.ServiceEndpointReference;
import org.jboss.soa.dsp.deployer.ServiceDeployer;
import org.jboss.soa.dsp.server.ServerConfig;
import org.jboss.soa.dsp.server.jbossas6.KernelLocator;
import org.jboss.soa.dsp.ws.BuildProcessor;
import org.jboss.vfs.VFS;
import org.jboss.vfs.VirtualFile;

public class JBossServiceDeployer implements ServiceDeployer {

	public ServiceEndpointReference deploy(EndpointMetaData metaData, Class<?> providerImpl, URL serviceUrl,
							final ClassLoader classLoader, File warArchive, ServerConfig serverConfig) throws Exception {
	      Deployment deployment = createVFSDeployment(warArchive);

	      // Classloading
	      ClassLoaderFactory clf = new DelegatingClassLoaderFactory(classLoader);

	      // WebMetaData
	      String[] webContext = deriveWebContextFromServiceUrl(serviceUrl);

	      WebMetaDataFactory wmdFactory = new WebMetaDataFactory(
	          metaData.getEndpointId(), webContext[0], webContext[1], providerImpl.getName()
	      );

	      MutableAttachments mutableAttachments =
	          (MutableAttachments)deployment.getPredeterminedManagedObjects();

	      // Applies to in memory only. Not used with VFS underneath
	      //mutableAttachments.addAttachment(StructureMetaData.class, new StructureMetaDataImpl());
	      mutableAttachments.addAttachment(ClassLoaderFactory.class, clf);
	      mutableAttachments.addAttachment(JBossWebMetaData.class, wmdFactory.createWebMetaData(classLoader));
	      mutableAttachments.addAttachment(DeploymentUnitFilter.class, new RiftsawWSDeploymentUnitFilter());

	      getMainDeployer().deploy(deployment);

	      ServiceEndpointReference ref = new ServiceEndpointReference(
	          metaData.getEndpointId(), serviceUrl, deployment.getName(), warArchive.getAbsolutePath()
	      );

	      //ref.setArchiveLocation(warArchive.getAbsolutePath());

	      return(ref);
	}

	public void undeploy(ServiceEndpointReference ref, ServerConfig serverConfig) throws Exception {
		getMainDeployer().undeploy(ref.getDeploymentName());
	}

	private String[] deriveWebContextFromServiceUrl(URL serviceUrl) {
		// metadata based on soapAddress
		String urlPathInfo = serviceUrl.getPath(); // has always leading slash
	    urlPathInfo = urlPathInfo.substring(1, urlPathInfo.length());

	    String actualWebContext;
	    String actualUrlPattern;
	    actualWebContext = urlPathInfo;
	    actualUrlPattern = "/*";

	    return new String[] {actualWebContext, actualUrlPattern};
	}
	  
	private Deployment createVFSDeployment(File war) throws IOException {
		VirtualFile webAppVFS = VFS.getChild(war.toURI());
		return VFSDeploymentFactory.getInstance().createVFSDeployment(webAppVFS);
	}

	private DeployerClient getMainDeployer() {
		Kernel mc = KernelLocator.getKernel();
		ControllerContext context = mc.getController().getInstalledContext("MainDeployer");
		DeployerClient mainDeployer = (DeployerClient)context.getTarget();
		return mainDeployer;
	}

	/**
	 * This method returns the optional build processor associated with the
	 * service deployer. This build processor can be used to performed any
	 * platform specific deployment archive modifications prior to it being
	 * deployed to the target platform.
	 * 
	 * @return The optional build procesor
	 */
	public BuildProcessor getBuildProcessor() {
		return(null);
	}
	
	public class DelegatingClassLoaderFactory implements ClassLoaderFactory {
		private ClassLoader delegate;

		public DelegatingClassLoaderFactory(final ClassLoader delegate) {
			this.delegate = delegate;
		}

		public ClassLoader createClassLoader(DeploymentUnit unit) throws Exception {
			return delegate;
		}

		public void removeClassLoader(DeploymentUnit unit) throws Exception {
			// cleanup?
		}
	}

	public class RiftsawWSDeploymentUnitFilter implements DeploymentUnitFilter {
		public boolean accepts(DeploymentUnit unit) {
			// When I am attached, it's always a BPEL deployment
			return false;
		}
	}

}
