/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.couriers.transport;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.jboss.internal.soa.esb.couriers.tx.InVMXAResource;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;

/**
 * Provide support for the transactional InVM operations.
 * 
 * @author kevin
 */
public class InVMResourceManager
{
    /**
     * Mapping of transactions to transactional resource entries.
     */
    private final Map<Object, InVMXAResource> transactionToXAResource = new HashMap<Object, InVMXAResource>() ;
    /**
     * The lock guarding access and modification to the structures.
     */
    private ReadWriteLock lock = new ReentrantReadWriteLock() ;

    /**
     * Factory singleton instance.
     */
    private static InVMResourceManager instance = new InVMResourceManager() ;

    public InVMXAResource getXAResource()
        throws InVMException
    {
        final TransactionStrategy txStrategy = TransactionStrategy.getTransactionStrategy(true) ;
        final Object txHandle ;
        final boolean isActive ;
        try
        {
            txHandle = txStrategy.getTransaction() ;
            isActive = txStrategy.isActive() ;
        }
        catch (final TransactionStrategyException tse)
        {
            throw new InVMException("Failed to obtain details about current transaction", tse) ;
        }

        if (!isActive)
        {
            throw new InVMException("Associated transaction is no longer active!") ;
        }

        acquireReadLock() ;
        try
        {
            final InVMXAResource current = transactionToXAResource.get(txHandle) ;
            if (current != null)
            {
                return current ;
            }
        }
        finally
        {
            releaseReadLock() ;
        }
        
        boolean enlist = false ;
        final InVMXAResource result ;
        
        acquireWriteLock() ;
        try
        {
            final InVMXAResource newResource = new InVMXAResource() ;
            final InVMXAResource oldResource = transactionToXAResource.put(txHandle, newResource) ;
            if (oldResource != null)
            {
                transactionToXAResource.put(txHandle,  oldResource) ;
                result = oldResource ;
            }
            else
            {
                result = newResource ;
                enlist = true ;
            }
        }
        finally
        {
            releaseWriteLock() ;
        }
        
        if (enlist)
        {
            try
            {
                txStrategy.enlistResource(result) ;
            }
            catch (final TransactionStrategyException tse)
            {
                throw new InVMException("Failed to enlist XAResource in current transaction", tse) ;
            }
        }
        
        return result ;
    }

    public void removeXAResource()
        throws InVMException
    {
        final TransactionStrategy txStrategy = TransactionStrategy.getTransactionStrategy(true) ;
        final Object txHandle ;
        try
        {
            txHandle = txStrategy.getTransaction() ;
        }
        catch (final TransactionStrategyException tse)
        {
            throw new InVMException("Failed to obtain details about current transaction", tse) ;
        }
        
        acquireWriteLock() ;
        try
        {
            transactionToXAResource.remove(txHandle) ;
        }
        finally
        {
            releaseWriteLock() ;
        }
    }

    /**
     * Get the InVM Resource Manager.
     * @return The InVM Resource Manager instance.
     */
    public static InVMResourceManager getInstance()
    {
        return instance;
    }

    /**
     * Acquire a read lock for accessing the data.
     */
    private void acquireReadLock()
    {
        lock.readLock().lock() ;
    }
    
    /**
     * Release a read lock for accessing the data.
     */
    private void releaseReadLock()
    {
        lock.readLock().unlock() ;
    }
    
    /**
     * Acquire a write lock for accessing the data.
     */
    private void acquireWriteLock()
    {
        lock.writeLock().lock() ;
    }
    
    /**
     * Release a write lock for accessing the data.
     */
    private void releaseWriteLock()
    {
        lock.writeLock().unlock() ;
    }
}
