/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.couriers.tx;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.transaction.xa.XAException;
import javax.transaction.xa.XAResource;
import javax.transaction.xa.Xid;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.transport.InVMResourceManager;
import org.jboss.internal.soa.esb.couriers.transport.InVMTemporaryTransport;
import org.jboss.internal.soa.esb.couriers.transport.InVMTransport;
import org.jboss.internal.soa.esb.couriers.transport.InVMException;
import org.jboss.soa.esb.addressing.eprs.InVMEpr;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;

/**
 * This XAResource instance controls the InVM queue manipulation under the control
 * of the transaction manager. Since InVM is inherently volatile, a crash failure
 * will take the entire queue with it, along with the client and server, we have much
 * less to worry about here in terms of recovery. Therefore we need to ensure that
 * the transaction manager does not save this instance to the log, since it won't
 * make sense upon recovery: the entire queue will have been destroyed.
 * 
 * This does mean that we could have non-atomic outcomes in the event of a crash where
 * durable participants (e.g., database) were involved in the same transaction. But
 * there is nothing we can do about that, without affecting the performance benefits of
 * InVM. We will call this out explicitly in the documentation and the user needs to
 * be aware of the possible consequences.
 * 
 * Could be a Synchronization if we could guarantee that afterCompletion will be called
 * (where we would do the compensation). But since that's only a 'best effort' we need
 * to make this an XAResource.
 */

public class InVMXAResource implements XAResource, Serializable
{

    public static final String INVM_RETRY_COUNT = "org.jboss.soa.esb.invm.retry.count";

    /**
     * Serial version UID for this class.
     */
    private static final long serialVersionUID = 77430212548543969L;
    /**
     * Redelivery retry limit.
     */
    private static int retryLimit;
    /**
     * Dead letter channel ServiceInvoker. Messages are delivered to the DLQ after the retry limit for a
     * failed message has been exceeded.
     */
    private static ServiceInvoker dlQueueInvoker;

    public enum Operation { INSERT, REMOVE };

    static {
        String retryLimitConfig = ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.INVM_RETRY_LIMIT, "5").trim();
        try {
            retryLimit = Integer.parseInt(retryLimitConfig);
        } catch (NumberFormatException e) {
            retryLimit = 5;
        }
    }
    
    public void addEntry(final InVMEpr inVMEpr, final Object msgObject, final Operation op)
    {
        entries.add(new InVMXAResourceEntry(inVMEpr, msgObject, op)) ;
    }
    
    /*
     * During commit, we deliver the message on to the queue if we were sending a message.
     * If we were receiving, then this is a no-op since we already changed the queue by
     * removing the message.
     *
     * @see javax.transaction.xa.XAResource#commit(javax.transaction.xa.Xid, boolean)
     */
    
    public void commit (Xid xid, boolean onePhase) throws XAException
    {
        removeXAResource() ;
        if (entries != null)
        {
            for(InVMXAResourceEntry entry: entries)
            {
                if (entry.getOpcode() == Operation.INSERT)
                {
                    boolean problem = false;
                    
                    try
                    {
                        deliverTx(entry.getInVMEpr(), entry.getMsgObject()) ;
                    }
                    catch (final Exception ex)
                    {
                        _logger.debug("Unexpected exception received when delivering to the courier", ex) ;
                        
                        problem = true;  // oops!
                    }
                    
                    if (problem)  // not a lot we can do at this stage
                    {
                        _logger.warn("InVMXAResource failed to commit to the InVM queue!");
                        
                        throw new XAException(XAException.XA_HEURHAZ);
                    }
                }
            }
        }
    }

    public void end (Xid xid, int flags) throws XAException
    {
    }

    public void forget (Xid xid) throws XAException
    {
    }
    
    public int getTransactionTimeout () throws XAException
    {
        return _timeout;
    }
    
    public int prepare (Xid xid) throws XAException
    {
        return XAResource.XA_OK;
    }

    /*
     * There is nothing to recover.
     * 
     * @see javax.transaction.xa.XAResource#recover(int)
     */
    
    public Xid[] recover (int flag) throws XAException
    {
        return null;
    }

    /*
     * During rollback we put the message back on the queue (tail) if we were receiving
     * a message. If we were delivering then there is nothing to do because we did not
     * update the queue directly, but rely on this instance to do it if the transaction
     * commits.
     */
    
    public void rollback (Xid xid) throws XAException
    {
        removeXAResource() ;
        if (entries != null)
        {
            for(InVMXAResourceEntry entry: entries)
            {
                if (entry.getOpcode() == Operation.REMOVE) // put the message back on the queue
                {
                    /*
                     * The message goes back on the queue. This may not be the same
                     * location as it had previously, but any attempt to do a truly
                     * transactional queue will affect performance adversely, for relatively
                     * little benefit. In an asynchronous world, applications should not
                     * be written assuming that a queue (or any transport) guarantees ordering.
                     * The ordering (or lack thereof) should be dealt with at the application
                     * level, where possible.
                     * 
                     * TODO we could add a queue-insertion policy that allows the developer to override
                     * how the message gets placed back into the queue.
                     */
                    final InVMEpr inVMEpr = entry.getInVMEpr() ;
                    final Object origMsgObject = entry.getMsgObject() ;
                    final Object msgObject = assertRedeliver(inVMEpr, origMsgObject) ;
                    if(msgObject != null) {
                        boolean problem = false;
                        try
                        {
                            deliverTx(inVMEpr, msgObject) ;
                        }
                        catch (final Exception ex)
                        {
                            _logger.debug("Unexpected exception received when delivering to the courier", ex) ;
        
                            problem = true;
                        }
        
                        if (problem)  // shouldn't get here, but ...
                        {
                            _logger.warn("InVMXAResource could not rollback and put Message on to InVM queue!");
        
                            throw new XAException(XAException.XA_HEURHAZ);
                        }
                    } else {
                        // Send to the DLQ...
                        try {
                            deliverToDLQ(getMessage(inVMEpr, origMsgObject));
                        } catch (MessageDeliverException e) {
                            _logger.debug("Unexpected exception received when delivering to the courier", e) ;
                            throw new XAException(XAException.XA_HEURHAZ);
                        }
                    }
                }
            }
        }
    }

    protected void deliverTx(final InVMEpr inVMEpr, final Object msgObject)
        throws Exception
    {
        if (inVMEpr.isTemporaryEPR())
        {
            InVMTemporaryTransport.getInstance().deliverTx(inVMEpr, msgObject) ;
        }
        else
        {
            InVMTransport.getInstance().deliverTx(inVMEpr, msgObject) ;
        }
    }

    public boolean setTransactionTimeout (int seconds) throws XAException
    {
        _timeout = seconds;
        
        return true;
    }

    public void start (Xid xid, int flags) throws XAException
    {
    }

    private Object assertRedeliver(final InVMEpr inVMEpr, final Object msgObject) throws XAException {
        Message message = getMessage(inVMEpr, msgObject);
        Integer retryCount = (Integer) message.getContext().getContext(INVM_RETRY_COUNT);

        if(retryCount == null || retryCount < retryLimit) {
            // Increment the retry count...
            if(retryCount == null) {
                message.getContext().setContext(INVM_RETRY_COUNT, 1);
            } else {
                message.getContext().setContext(INVM_RETRY_COUNT, retryCount + 1);
            }

            // Need to recreate the delivery object with the incremented
            // redelivery count...
            final Object result ;
            try {
                result = InVMTransport.toDeliveryObject(message, inVMEpr.getPassByValue());
            } catch (InVMException e) {
                _logger.debug("Unexpected exception received when delivering to the courier", e) ;
                throw new XAException(XAException.XA_HEURHAZ);
            }
            
            return result;
        } else {
            return null;
        }
    }

    private Message getMessage(final InVMEpr inVMEpr, final Object msgObject) throws XAException {
        Message message;
        try {
            message = InVMTransport.fromDeliveryObject(msgObject, inVMEpr.getPassByValue());
        } catch (InVMException e) {
            _logger.debug("Unexpected exception received when delivering to the courier", e) ;
            throw new XAException(XAException.XA_HEURHAZ);
        }
        return message;
    }

    /**
     * Deliver a message to the Dead Letter Channel Service.
     *
     * @param message The message to be delivered to the dead letter chennel.
     * @throws org.jboss.soa.esb.listeners.message.MessageDeliverException Message delivery failure.
     */
    protected void deliverToDLQ(Message message) throws MessageDeliverException {
        if (!"true".equalsIgnoreCase(Configuration.getRedeliveryDlsOn())) {
            _logger.debug("org.jboss.soa.esb.dls.redeliver is turned off");
        } else {
            synchronized (ServiceInvoker.dlqService) {
                if (dlQueueInvoker == null) {
                    dlQueueInvoker = new ServiceInvoker(ServiceInvoker.dlqService);
                }
            }

            dlQueueInvoker.deliverAsync(message);
        }
    }
    
    private void removeXAResource()
        throws XAException
    {
        try
        {
            InVMResourceManager.getInstance().removeXAResource() ;
        }
        catch (final InVMException invme)
        {
            _logger.warn("InVMXAResource could not be disassociated from Resource Manager") ;
            throw new XAException(XAException.XA_HEURHAZ) ;
        }
    }

    public boolean isSameRM (XAResource xares) throws XAException
   {
       return (xares == this);
   }

    private transient List<InVMXAResourceEntry> entries = new ArrayList<InVMXAResourceEntry>() ;

    private transient int _timeout;
    
    protected static final Logger _logger = Logger.getLogger(InVMXAResource.class);
    
    private static final class InVMXAResourceEntry
    {
        private final InVMEpr inVMEpr ;
        private final Object msgObject ;
        private final Operation opcode ;
        
        InVMXAResourceEntry(final InVMEpr inVMEpr, final Object msgObject, final Operation opcode)
        {
            this.inVMEpr = inVMEpr ;
            this.msgObject = msgObject ;
            this.opcode = opcode ;
        }
        
        InVMEpr getInVMEpr()
        {
            return inVMEpr ;
        }
        
        Object getMsgObject()
        {
            return msgObject ;
        }
        
        Operation getOpcode()
        {
            return opcode ;
        }
    }

}