/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

package org.jboss.internal.soa.esb.message.format.serialized;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Hashtable;

import org.jboss.soa.esb.message.Attachment;

/**
 * Messages may contain attachments that do not appear in the main payload body.
 * For example, binary document formats, zip files etc.
 * 
 * @author Mark Little
 */

public class AttachmentImpl implements Attachment, java.io.Serializable
{
	private static final long serialVersionUID = 0x0;

	public Object get(String name)
	{
		synchronized(_table)
		{
			return SerializedValueImpl.unwrap(_table.get(name));
		}
	}

	public Object put(String name, Object value) 
	{
		if (value instanceof Serializable)
		{
                    final Serializable wrapped = SerializedValueImpl.wrap((Serializable)value) ;
                    synchronized(_table)
                    {
                        final Serializable previous = _table.put(name, wrapped) ;
                        return SerializedValueImpl.unwrap(previous);
                    }
		}
		throw new IllegalArgumentException("value must be Serializable");
	}

	public Object remove(String name) 
	{
		synchronized(_table)
		{
			return SerializedValueImpl.unwrap(_table.remove(name));
		}
	}

	public String[] getNames() 
	{
		synchronized(_table)
		{
			return _table.keySet().toArray(new String[_table.size()]);
		}
	}

	public Object itemAt(int index) throws IndexOutOfBoundsException 
	{
		synchronized(_list)
		{
			return SerializedValueImpl.unwrap(_list.get(index));
		}
	}

	public Object removeItemAt(int index) throws IndexOutOfBoundsException 
	{
		synchronized(_list)
		{
			return SerializedValueImpl.unwrap(_list.remove(index));
		}
	}

	public Object replaceItemAt(int index, Object value) throws IndexOutOfBoundsException 
	{
		if (value instanceof Serializable)
		{
                    final Serializable wrapped = SerializedValueImpl.wrap((Serializable)value) ;
                    synchronized(_list)
                    {
                        final Serializable previous = _list.set(index, wrapped) ;
                        return SerializedValueImpl.unwrap(previous);
                    }
		}
		throw new IllegalArgumentException("value must be Serializable");
	}

	public void addItem(Object value) 
	{
		if (value instanceof Serializable)
			synchronized(_list)
			{
				_list.add(SerializedValueImpl.wrap((Serializable)value)) ;
			}
		else
			throw new IllegalArgumentException("value must be Serializable");
	}

	public void addItemAt(int index, Object value) throws IndexOutOfBoundsException 
	{
		if (value instanceof Serializable)
			synchronized(_list)
			{
				_list.add(index, SerializedValueImpl.wrap((Serializable)value));
			}
		else
			throw new IllegalArgumentException("value must be Serializable");
	}

	public int getNamedCount() 
	{
		synchronized(_table)
		{
			return _table.size();
		}
	}
	
	public int getUnnamedCount() 
	{
		synchronized(_list)
		{
			return _list.size();
		}
	}
	
	public String toString() 
	{ 
		final String tableVal ;
		synchronized(_table)
		{
			tableVal = _table.toString() ;
		}
		final String listVal ;
		synchronized(_list)
		{
			listVal = _list.toString() ;
		}
		return "attachments: [ Named:" + tableVal + ", Unnamed:" + listVal +" ]";
	}
	
	private void readObject(final ObjectInputStream stream)
	    throws IOException, ClassNotFoundException
        {
	    stream.defaultReadObject() ;
	    SerializedValueImpl.wrapList(_list) ;
	    SerializedValueImpl.wrapMap(_table) ;
        }

	private final ArrayList<Serializable> _list = new ArrayList<Serializable>();
	private final Hashtable<String, Serializable> _table = new Hashtable<String,Serializable>();
}
