/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.jboss.internal.soa.esb.rosetta.pooling;

import java.util.Enumeration;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.jms.JMSException;
import javax.jms.Queue;
import javax.jms.QueueBrowser;

/**
 * Wrapped queue browser class.
 */
class WrappedQueueBrowser implements QueueBrowser
{
    /**
     * The connection pool associated with the session.
     */
    private final JmsConnectionPool connectionPool ;
    /**
     * The session associated with the browser.
     */
    private final JmsSession jmsSession ;
    /**
     * The queue browser.
     */
    private final QueueBrowser queueBrowser ;
    /**
     * valid flag for xa associations.
     */
    private final AtomicBoolean wrapperValid ;

    /**
     * Construct the wrapped queue browser.
     * @param connectionPool The connection pool associated with the session.
     * @param jmsSession The session associated with the browser.
     * @param queueBrowser The queue browser.
     */
    public WrappedQueueBrowser(final JmsConnectionPool connectionPool,
        final JmsSession jmsSession, final QueueBrowser queueBrowser)
    {
        this(connectionPool, jmsSession, queueBrowser, null) ;
    }

    /**
     * Construct the wrapped queue browser.
     * @param connectionPool The connection pool associated with the session.
     * @param jmsSession The session associated with the browser.
     * @param queueBrowser The queue browser.
     * @param wrapperValid valid flag for xa associations.
     */
    public WrappedQueueBrowser(final JmsConnectionPool connectionPool,
        final JmsSession jmsSession, final QueueBrowser queueBrowser,
        final AtomicBoolean wrapperValid)
    {
        this.connectionPool = connectionPool ;
        this.jmsSession = jmsSession ;
        this.queueBrowser = queueBrowser ;
        this.wrapperValid = wrapperValid ;
    }

    public void close()
        throws JMSException
    {
        validate() ;
        try
        {
            queueBrowser.close() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public Enumeration getEnumeration()
        throws JMSException
    {
        associate() ;
        try
        {
            return queueBrowser.getEnumeration() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public String getMessageSelector()
        throws JMSException
    {
        associate() ;
        try
        {
            return queueBrowser.getMessageSelector() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public Queue getQueue()
        throws JMSException
    {
        associate() ;
        try
        {
            return queueBrowser.getQueue() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    private void validate()
        throws JMSException
    {
        if ((wrapperValid != null) && !wrapperValid.get())
        {
            throw new JMSException("Instance is no longer valid") ;
        }
    }

    private void associate()
        throws JMSException
    {
        validate() ;
        jmsSession.associate() ;
    }
}
