/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006
 */
package org.jboss.soa.esb.actions;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringReader;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.ActionProcessingPipeline;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;

/**
 * Simple action that checks to see if the message should be stored into a file.
 * This action is primarily used for testing the quickstarts.
 * 
 * @author Kevin Conner
 */
public class StoreMessageToFile extends AbstractActionPipelineProcessor
{
    /**
     * The logger for this class.
     */
    private static final Logger LOG =  Logger.getLogger(StoreMessageToFile.class) ;
    
    /**
     * The newline string.
     */
    private static final String NEWLINE =  System.getProperty("line.separator") ;
    
    /**
     * The lock protecting file write access.
     */
    private static final Lock globalLock = new ReentrantLock() ;
    
    /**
     * The filename parameter.
     */
    public static final String FILENAME = "filename" ;
    /**
     * The message content location.
     */
    public static final String LOCATION = "location" ;
    /**
     * Whether it is raw or not.
     */
    public static final String RAW = "raw" ;
    
    /**
     * The name of the filename when stored in a message property.
     */
    public static final String PROPERTY_JBESB_FILENAME = "jbesbfilename" ;
    /**
     * The start of the filename when it is embedded in the contents.
     */
    public static final String MESSAGE_CONTENT_FILENAME_START = "[jbesb-filename]" ;
    /**
     * The end of the filename when it is embedded in the contents.
     */
    public static final String MESSAGE_CONTENT_FILENAME_END = "[/jbesb-filename]" ;

    private MessagePayloadProxy payloadProxy;

    /**
     * Create the action with the specified configuration.
     * 
     * @param config The action configuration.
     */
    public StoreMessageToFile(final ConfigTree config)
    {
        filename = config.getAttribute(FILENAME) ;
        String location = config.getAttribute(LOCATION) ;
        if(location != null) {
            config.setAttribute(MessagePayloadProxy.GET_PAYLOAD_LOCATION, location);
        }
        payloadProxy = new MessagePayloadProxy(config);
        payloadProxy.setNullGetPayloadHandling(MessagePayloadProxy.NullPayloadHandling.LOG);

        raw = Boolean.parseBoolean(config.getAttribute(RAW, "false")) ;
    }

    /**
     * Process the message.
     * 
     * @param message The current message.
     * @throws ActionProcessingPipeline for errors in processing.
     */
    public Message process(final Message message)
        throws ActionProcessingException
    {
        final Object contents;

        try {
            contents = payloadProxy.getPayload(message);
        } catch (MessageDeliverException e) {
            throw new ActionProcessingException(e);
        }

        if (contents != null)
        {
            final String messageVal ;
            if (contents instanceof byte[])
            {
                messageVal = new String((byte[])contents) ;
            }
            else
            {
                messageVal = contents.toString() ;
            }

            if (LOG.isDebugEnabled())
            {
                LOG.debug("Message value: " + messageVal) ;
            }
            
            final String storeFilename ;
            if (filename != null)
            {
                storeFilename = filename ;
            }
            else
            {
                final Object jbesbfilename = message.getProperties().getProperty(PROPERTY_JBESB_FILENAME) ;
                if (jbesbfilename != null)
                {
                    storeFilename = jbesbfilename.toString() ;
                }
                else
                {
                    final int startIndex = messageVal.indexOf(MESSAGE_CONTENT_FILENAME_START) ;
                    if (startIndex != -1)
                    {
                        final int endIndex = messageVal.indexOf(MESSAGE_CONTENT_FILENAME_END) ;
                        if ((endIndex != -1) && (endIndex > startIndex))
                        {
                            storeFilename = messageVal.substring(startIndex + MESSAGE_CONTENT_FILENAME_START.length(), endIndex) ;
                        }
                        else
                        {
                            storeFilename = null ;
                        }
                    }
                    else
                    {
                        storeFilename = null ;
                    }
                }
            }
            
            if (storeFilename == null)
            {
                LOG.debug("Store filename is null") ;
            }
            else
            {
                final File outputFile ;
                final File storeFile = new File(storeFilename) ;
                if (storeFile.isAbsolute())
                {
                    outputFile = storeFile ;
                }
                else
                {
                    final String tmpDir = System.getProperty("java.io.tmpdir") ;
                    outputFile = new File(tmpDir, storeFilename) ;
                }
                try
                {
                    if (LOG.isDebugEnabled())
                    {
                        LOG.debug("Storing message in filename: " + outputFile.getCanonicalPath()) ;
                    }
                    
                    globalLock.lock() ;
                    try
                    {
                        final FileWriter fw = new FileWriter(outputFile, true) ;
                        try
                        {
                            if (raw)
                            {
                                fw.write(messageVal) ;
                            }
                            else
                            {
                                // We put everything on one line due to the limitations of the test helper class
                                final StringReader sr = new StringReader(messageVal) ;
                                final BufferedReader br = new BufferedReader(sr) ;
                                while(true)
                                {
                                    final String line = br.readLine();
                                    if (line == null)
                                    {
                                        break ;
                                    }
                                    fw.write(line) ;
                                }
                                fw.write(NEWLINE) ;
                            }
                            fw.flush();
                        }
                        finally
                        {
                            try
                            {
                                fw.close() ;
                            }
                            catch(final Throwable th) {} // ignore
                        }
                    }
                    finally
                    {
                        globalLock.unlock() ;
                    }
                }
                catch (final IOException ioe)
                {
                    LOG.warn("Error accessing file", ioe) ;
                }
            }
        }
        
        return message;
    }

    /**
     * The filename to create.
     */
    private final String filename ;
    /**
     * True if we output the raw contents.
     */
    private final boolean raw ;
}
